/* valaunresolvedtype.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaDataTypePrivate ValaDataTypePrivate;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_UNRESOLVED_TYPE (vala_unresolved_type_get_type ())
#define VALA_UNRESOLVED_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_UNRESOLVED_TYPE, ValaUnresolvedType))
#define VALA_UNRESOLVED_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_UNRESOLVED_TYPE, ValaUnresolvedTypeClass))
#define VALA_IS_UNRESOLVED_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_UNRESOLVED_TYPE))
#define VALA_IS_UNRESOLVED_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_UNRESOLVED_TYPE))
#define VALA_UNRESOLVED_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_UNRESOLVED_TYPE, ValaUnresolvedTypeClass))

typedef struct _ValaUnresolvedType ValaUnresolvedType;
typedef struct _ValaUnresolvedTypeClass ValaUnresolvedTypeClass;
typedef struct _ValaUnresolvedTypePrivate ValaUnresolvedTypePrivate;

#define VALA_TYPE_UNRESOLVED_SYMBOL (vala_unresolved_symbol_get_type ())
#define VALA_UNRESOLVED_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_UNRESOLVED_SYMBOL, ValaUnresolvedSymbol))
#define VALA_UNRESOLVED_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_UNRESOLVED_SYMBOL, ValaUnresolvedSymbolClass))
#define VALA_IS_UNRESOLVED_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_UNRESOLVED_SYMBOL))
#define VALA_IS_UNRESOLVED_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_UNRESOLVED_SYMBOL))
#define VALA_UNRESOLVED_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_UNRESOLVED_SYMBOL, ValaUnresolvedSymbolClass))

typedef struct _ValaUnresolvedSymbol ValaUnresolvedSymbol;
typedef struct _ValaUnresolvedSymbolClass ValaUnresolvedSymbolClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_MEMBER_ACCESS (vala_member_access_get_type ())
#define VALA_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccess))
#define VALA_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))
#define VALA_IS_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER_ACCESS))
#define VALA_IS_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER_ACCESS))
#define VALA_MEMBER_ACCESS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))

typedef struct _ValaMemberAccess ValaMemberAccess;
typedef struct _ValaMemberAccessClass ValaMemberAccessClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * A reference to a data type. This is used to specify static types of
 * expressions.
 */
struct _ValaDataType {
	ValaCodeNode parent_instance;
	ValaDataTypePrivate * priv;
};

struct _ValaDataTypeClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cname) (ValaDataType* self);
	char* (*get_cdeclarator_suffix) (ValaDataType* self);
	char* (*get_lower_case_cname) (ValaDataType* self, const char* infix);
	char* (*to_qualified_string) (ValaDataType* self, ValaScope* scope);
	ValaDataType* (*copy) (ValaDataType* self);
	gboolean (*equals) (ValaDataType* self, ValaDataType* type2);
	gboolean (*stricter) (ValaDataType* self, ValaDataType* type2);
	gboolean (*compatible) (ValaDataType* self, ValaDataType* target_type);
	gboolean (*is_invokable) (ValaDataType* self);
	ValaDataType* (*get_return_type) (ValaDataType* self);
	GeeList* (*get_parameters) (ValaDataType* self);
	gboolean (*is_reference_type_or_type_parameter) (ValaDataType* self);
	gboolean (*is_array) (ValaDataType* self);
	GeeList* (*get_symbols) (ValaDataType* self);
	ValaSymbol* (*get_member) (ValaDataType* self, const char* member_name);
	ValaSymbol* (*get_pointer_member) (ValaDataType* self, const char* member_name);
	gboolean (*is_real_struct_type) (ValaDataType* self);
	char* (*get_type_id) (ValaDataType* self);
	char* (*get_type_signature) (ValaDataType* self);
	gboolean (*is_disposable) (ValaDataType* self);
};

/**
 * An unresolved reference to a data type.
 */
struct _ValaUnresolvedType {
	ValaDataType parent_instance;
	ValaUnresolvedTypePrivate * priv;
};

struct _ValaUnresolvedTypeClass {
	ValaDataTypeClass parent_class;
};

struct _ValaUnresolvedTypePrivate {
	ValaUnresolvedSymbol* _unresolved_symbol;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
GType vala_formal_parameter_get_type (void);
GType vala_unresolved_type_get_type (void);
GType vala_unresolved_symbol_get_type (void);
#define VALA_UNRESOLVED_TYPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_UNRESOLVED_TYPE, ValaUnresolvedTypePrivate))
enum  {
	VALA_UNRESOLVED_TYPE_DUMMY_PROPERTY
};
ValaUnresolvedType* vala_unresolved_type_new (void);
ValaUnresolvedType* vala_unresolved_type_construct (GType object_type);
ValaUnresolvedType* vala_unresolved_type_new (void);
void vala_unresolved_type_set_unresolved_symbol (ValaUnresolvedType* self, ValaUnresolvedSymbol* value);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
ValaUnresolvedType* vala_unresolved_type_new_from_symbol (ValaUnresolvedSymbol* symbol, ValaSourceReference* source);
ValaUnresolvedType* vala_unresolved_type_construct_from_symbol (GType object_type, ValaUnresolvedSymbol* symbol, ValaSourceReference* source);
ValaUnresolvedType* vala_unresolved_type_new_from_symbol (ValaUnresolvedSymbol* symbol, ValaSourceReference* source);
GType vala_member_access_get_type (void);
ValaExpression* vala_member_access_get_inner (ValaMemberAccess* self);
const char* vala_member_access_get_member_name (ValaMemberAccess* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
ValaUnresolvedSymbol* vala_unresolved_symbol_new (ValaUnresolvedSymbol* inner, const char* name, ValaSourceReference* source_reference);
ValaUnresolvedSymbol* vala_unresolved_symbol_construct (GType object_type, ValaUnresolvedSymbol* inner, const char* name, ValaSourceReference* source_reference);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
GeeList* vala_member_access_get_type_arguments (ValaMemberAccess* self);
void vala_data_type_add_type_argument (ValaDataType* self, ValaDataType* arg);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaUnresolvedType* vala_unresolved_type_new_from_expression (ValaExpression* expr);
gboolean vala_data_type_get_value_owned (ValaDataType* self);
gboolean vala_data_type_get_nullable (ValaDataType* self);
void vala_data_type_set_nullable (ValaDataType* self, gboolean value);
gboolean vala_data_type_get_is_dynamic (ValaDataType* self);
void vala_data_type_set_is_dynamic (ValaDataType* self, gboolean value);
ValaUnresolvedSymbol* vala_unresolved_type_get_unresolved_symbol (ValaUnresolvedType* self);
ValaUnresolvedSymbol* vala_unresolved_symbol_copy (ValaUnresolvedSymbol* self);
GeeList* vala_data_type_get_type_arguments (ValaDataType* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
static ValaDataType* vala_unresolved_type_real_copy (ValaDataType* base);
char* vala_code_node_to_string (ValaCodeNode* self);
static char* vala_unresolved_type_real_to_qualified_string (ValaDataType* base, ValaScope* scope);
static gpointer vala_unresolved_type_parent_class = NULL;
static void vala_unresolved_type_finalize (ValaCodeNode* obj);



ValaUnresolvedType* vala_unresolved_type_construct (GType object_type) {
	ValaUnresolvedType* self;
	self = (ValaUnresolvedType*) g_type_create_instance (object_type);
	return self;
}


ValaUnresolvedType* vala_unresolved_type_new (void) {
	return vala_unresolved_type_construct (VALA_TYPE_UNRESOLVED_TYPE);
}


/**
 * Creates a new type reference.
 *
 * @param symbol    unresolved type symbol
 * @param source    reference to source code
 * @return          newly created type reference
 */
ValaUnresolvedType* vala_unresolved_type_construct_from_symbol (GType object_type, ValaUnresolvedSymbol* symbol, ValaSourceReference* source) {
	ValaUnresolvedType* self;
	g_return_val_if_fail (symbol != NULL, NULL);
	self = (ValaUnresolvedType*) g_type_create_instance (object_type);
	vala_unresolved_type_set_unresolved_symbol (self, symbol);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source);
	return self;
}


ValaUnresolvedType* vala_unresolved_type_new_from_symbol (ValaUnresolvedSymbol* symbol, ValaSourceReference* source) {
	return vala_unresolved_type_construct_from_symbol (VALA_TYPE_UNRESOLVED_TYPE, symbol, source);
}


/**
 * Creates a new type reference from a code expression.
 *
 * @param expr   member access expression
 * @param source reference to source code
 * @return       newly created type reference
 */
ValaUnresolvedType* vala_unresolved_type_new_from_expression (ValaExpression* expr) {
	g_return_val_if_fail (expr != NULL, NULL);
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaUnresolvedType* type_ref;
		ValaMemberAccess* _tmp0_;
		ValaMemberAccess* ma;
		type_ref = NULL;
		_tmp0_ = NULL;
		ma = (_tmp0_ = VALA_MEMBER_ACCESS (expr), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		if (vala_member_access_get_inner (ma) != NULL) {
			if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (ma))) {
				ValaMemberAccess* _tmp1_;
				ValaMemberAccess* simple;
				ValaUnresolvedType* _tmp4_;
				ValaUnresolvedSymbol* _tmp3_;
				ValaUnresolvedSymbol* _tmp2_;
				_tmp1_ = NULL;
				simple = (_tmp1_ = VALA_MEMBER_ACCESS (vala_member_access_get_inner (ma)), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				_tmp2_ = NULL;
				type_ref = (_tmp4_ = vala_unresolved_type_new_from_symbol (_tmp3_ = vala_unresolved_symbol_new (_tmp2_ = vala_unresolved_symbol_new (NULL, vala_member_access_get_member_name (simple), vala_code_node_get_source_reference ((ValaCodeNode*) ma)), vala_member_access_get_member_name (ma), vala_code_node_get_source_reference ((ValaCodeNode*) ma)), vala_code_node_get_source_reference ((ValaCodeNode*) ma)), (type_ref == NULL) ? NULL : (type_ref = (vala_code_node_unref (type_ref), NULL)), _tmp4_);
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
				(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_code_node_unref (_tmp2_), NULL));
				(simple == NULL) ? NULL : (simple = (vala_code_node_unref (simple), NULL));
			}
		} else {
			ValaUnresolvedType* _tmp6_;
			ValaUnresolvedSymbol* _tmp5_;
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			type_ref = (_tmp6_ = vala_unresolved_type_new_from_symbol (_tmp5_ = vala_unresolved_symbol_new (NULL, vala_member_access_get_member_name (ma), vala_code_node_get_source_reference ((ValaCodeNode*) ma)), vala_code_node_get_source_reference ((ValaCodeNode*) ma)), (type_ref == NULL) ? NULL : (type_ref = (vala_code_node_unref (type_ref), NULL)), _tmp6_);
			(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
		}
		if (type_ref != NULL) {
			GeeList* type_args;
			ValaUnresolvedType* _tmp7_;
			vala_data_type_set_value_owned ((ValaDataType*) type_ref, TRUE);
			type_args = vala_member_access_get_type_arguments (ma);
			{
				GeeIterator* _arg_it;
				_arg_it = gee_iterable_iterator ((GeeIterable*) type_args);
				while (gee_iterator_next (_arg_it)) {
					ValaDataType* arg;
					arg = (ValaDataType*) gee_iterator_get (_arg_it);
					vala_data_type_add_type_argument ((ValaDataType*) type_ref, arg);
					(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
				}
				(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
			}
			_tmp7_ = NULL;
			return (_tmp7_ = type_ref, (type_args == NULL) ? NULL : (type_args = (gee_collection_object_unref (type_args), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp7_);
		}
		(type_ref == NULL) ? NULL : (type_ref = (vala_code_node_unref (type_ref), NULL));
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) expr), "Type reference must be simple name or member access expression");
	return NULL;
}


static ValaDataType* vala_unresolved_type_real_copy (ValaDataType* base) {
	ValaUnresolvedType * self;
	ValaUnresolvedType* _result_;
	ValaUnresolvedSymbol* _tmp0_;
	self = (ValaUnresolvedType*) base;
	_result_ = vala_unresolved_type_new ();
	vala_code_node_set_source_reference ((ValaCodeNode*) _result_, vala_code_node_get_source_reference ((ValaCodeNode*) self));
	vala_data_type_set_value_owned ((ValaDataType*) _result_, vala_data_type_get_value_owned ((ValaDataType*) self));
	vala_data_type_set_nullable ((ValaDataType*) _result_, vala_data_type_get_nullable ((ValaDataType*) self));
	vala_data_type_set_is_dynamic ((ValaDataType*) _result_, vala_data_type_get_is_dynamic ((ValaDataType*) self));
	_tmp0_ = NULL;
	vala_unresolved_type_set_unresolved_symbol (_result_, _tmp0_ = vala_unresolved_symbol_copy (self->priv->_unresolved_symbol));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL));
	{
		GeeList* _tmp1_;
		GeeIterator* _tmp2_;
		GeeIterator* _arg_it;
		_tmp1_ = NULL;
		_tmp2_ = NULL;
		_arg_it = (_tmp2_ = gee_iterable_iterator ((GeeIterable*) (_tmp1_ = vala_data_type_get_type_arguments ((ValaDataType*) self))), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (gee_collection_object_unref (_tmp1_), NULL)), _tmp2_);
		while (gee_iterator_next (_arg_it)) {
			ValaDataType* arg;
			ValaDataType* _tmp3_;
			arg = (ValaDataType*) gee_iterator_get (_arg_it);
			_tmp3_ = NULL;
			vala_data_type_add_type_argument ((ValaDataType*) _result_, _tmp3_ = vala_data_type_copy (arg));
			(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(_arg_it == NULL) ? NULL : (_arg_it = (gee_collection_object_unref (_arg_it), NULL));
	}
	return (ValaDataType*) _result_;
}


static char* vala_unresolved_type_real_to_qualified_string (ValaDataType* base, ValaScope* scope) {
	ValaUnresolvedType * self;
	self = (ValaUnresolvedType*) base;
	return vala_code_node_to_string ((ValaCodeNode*) self->priv->_unresolved_symbol);
}


ValaUnresolvedSymbol* vala_unresolved_type_get_unresolved_symbol (ValaUnresolvedType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_unresolved_symbol;
}


void vala_unresolved_type_set_unresolved_symbol (ValaUnresolvedType* self, ValaUnresolvedSymbol* value) {
	ValaUnresolvedSymbol* _tmp2_;
	ValaUnresolvedSymbol* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_unresolved_symbol = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_unresolved_symbol == NULL) ? NULL : (self->priv->_unresolved_symbol = (vala_code_node_unref (self->priv->_unresolved_symbol), NULL)), _tmp2_);
}


static void vala_unresolved_type_class_init (ValaUnresolvedTypeClass * klass) {
	vala_unresolved_type_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_unresolved_type_finalize;
	g_type_class_add_private (klass, sizeof (ValaUnresolvedTypePrivate));
	VALA_DATA_TYPE_CLASS (klass)->copy = vala_unresolved_type_real_copy;
	VALA_DATA_TYPE_CLASS (klass)->to_qualified_string = vala_unresolved_type_real_to_qualified_string;
}


static void vala_unresolved_type_instance_init (ValaUnresolvedType * self) {
	self->priv = VALA_UNRESOLVED_TYPE_GET_PRIVATE (self);
}


static void vala_unresolved_type_finalize (ValaCodeNode* obj) {
	ValaUnresolvedType * self;
	self = VALA_UNRESOLVED_TYPE (obj);
	(self->priv->_unresolved_symbol == NULL) ? NULL : (self->priv->_unresolved_symbol = (vala_code_node_unref (self->priv->_unresolved_symbol), NULL));
	VALA_CODE_NODE_CLASS (vala_unresolved_type_parent_class)->finalize (obj);
}


GType vala_unresolved_type_get_type (void) {
	static GType vala_unresolved_type_type_id = 0;
	if (vala_unresolved_type_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaUnresolvedTypeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_unresolved_type_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaUnresolvedType), 0, (GInstanceInitFunc) vala_unresolved_type_instance_init, NULL };
		vala_unresolved_type_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaUnresolvedType", &g_define_type_info, 0);
	}
	return vala_unresolved_type_type_id;
}




