/* collectionobject.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gobject/gvaluecollector.h>


#define GEE_TYPE_COLLECTION_OBJECT (gee_collection_object_get_type ())
#define GEE_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObject))
#define GEE_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))
#define GEE_IS_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_IS_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_COLLECTION_OBJECT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))

typedef struct _GeeCollectionObject GeeCollectionObject;
typedef struct _GeeCollectionObjectClass GeeCollectionObjectClass;
typedef struct _GeeCollectionObjectPrivate GeeCollectionObjectPrivate;
typedef struct _GeeParamSpecCollectionObject GeeParamSpecCollectionObject;

/**
 * Base class for all collections.
 */
struct _GeeCollectionObject {
	GTypeInstance parent_instance;
	volatile int ref_count;
	GeeCollectionObjectPrivate * priv;
};

struct _GeeCollectionObjectClass {
	GTypeClass parent_class;
	void (*finalize) (GeeCollectionObject *self);
};

struct _GeeParamSpecCollectionObject {
	GParamSpec parent_instance;
};



gpointer gee_collection_object_ref (gpointer instance);
void gee_collection_object_unref (gpointer instance);
GParamSpec* gee_param_spec_collection_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void gee_value_set_collection_object (GValue* value, gpointer v_object);
gpointer gee_value_get_collection_object (const GValue* value);
GType gee_collection_object_get_type (void);
enum  {
	GEE_COLLECTION_OBJECT_DUMMY_PROPERTY
};
GeeCollectionObject* gee_collection_object_new (void);
GeeCollectionObject* gee_collection_object_construct (GType object_type);
GeeCollectionObject* gee_collection_object_new (void);
static gpointer gee_collection_object_parent_class = NULL;
static void gee_collection_object_finalize (GeeCollectionObject* obj);



/**
 * Base class for all collections.
 */
GeeCollectionObject* gee_collection_object_construct (GType object_type) {
	GeeCollectionObject* self;
	self = (GeeCollectionObject*) g_type_create_instance (object_type);
	return self;
}


GeeCollectionObject* gee_collection_object_new (void) {
	return gee_collection_object_construct (GEE_TYPE_COLLECTION_OBJECT);
}


static void gee_value_collection_object_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void gee_value_collection_object_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		gee_collection_object_unref (value->data[0].v_pointer);
	}
}


static void gee_value_collection_object_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = gee_collection_object_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer gee_value_collection_object_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* gee_value_collection_object_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		GeeCollectionObject* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = gee_collection_object_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* gee_value_collection_object_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	GeeCollectionObject** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = gee_collection_object_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* gee_param_spec_collection_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	GeeParamSpecCollectionObject* spec;
	g_return_val_if_fail (g_type_is_a (object_type, GEE_TYPE_COLLECTION_OBJECT), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer gee_value_get_collection_object (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, GEE_TYPE_COLLECTION_OBJECT), NULL);
	return value->data[0].v_pointer;
}


void gee_value_set_collection_object (GValue* value, gpointer v_object) {
	GeeCollectionObject* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, GEE_TYPE_COLLECTION_OBJECT));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, GEE_TYPE_COLLECTION_OBJECT));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		gee_collection_object_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		gee_collection_object_unref (old);
	}
}


static void gee_collection_object_class_init (GeeCollectionObjectClass * klass) {
	gee_collection_object_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_collection_object_finalize;
}


static void gee_collection_object_instance_init (GeeCollectionObject * self) {
	self->ref_count = 1;
}


static void gee_collection_object_finalize (GeeCollectionObject* obj) {
	GeeCollectionObject * self;
	self = GEE_COLLECTION_OBJECT (obj);
}


GType gee_collection_object_get_type (void) {
	static GType gee_collection_object_type_id = 0;
	if (gee_collection_object_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { gee_value_collection_object_init, gee_value_collection_object_free_value, gee_value_collection_object_copy_value, gee_value_collection_object_peek_pointer, "p", gee_value_collection_object_collect_value, "p", gee_value_collection_object_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (GeeCollectionObjectClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_collection_object_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeCollectionObject), 0, (GInstanceInitFunc) gee_collection_object_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		gee_collection_object_type_id = g_type_register_fundamental (g_type_fundamental_next (), "GeeCollectionObject", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return gee_collection_object_type_id;
}


gpointer gee_collection_object_ref (gpointer instance) {
	GeeCollectionObject* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void gee_collection_object_unref (gpointer instance) {
	GeeCollectionObject* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		GEE_COLLECTION_OBJECT_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




