/* valaccodeassignment.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>


#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;
typedef struct _ValaCCodeNodePrivate ValaCCodeNodePrivate;

#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;

#define VALA_TYPE_CCODE_EXPRESSION (vala_ccode_expression_get_type ())
#define VALA_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpression))
#define VALA_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))
#define VALA_IS_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_IS_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_CCODE_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))

typedef struct _ValaCCodeExpression ValaCCodeExpression;
typedef struct _ValaCCodeExpressionClass ValaCCodeExpressionClass;
typedef struct _ValaCCodeExpressionPrivate ValaCCodeExpressionPrivate;

#define VALA_TYPE_CCODE_ASSIGNMENT (vala_ccode_assignment_get_type ())
#define VALA_CCODE_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_ASSIGNMENT, ValaCCodeAssignment))
#define VALA_CCODE_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_ASSIGNMENT, ValaCCodeAssignmentClass))
#define VALA_IS_CCODE_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_ASSIGNMENT))
#define VALA_IS_CCODE_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_ASSIGNMENT))
#define VALA_CCODE_ASSIGNMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_ASSIGNMENT, ValaCCodeAssignmentClass))

typedef struct _ValaCCodeAssignment ValaCCodeAssignment;
typedef struct _ValaCCodeAssignmentClass ValaCCodeAssignmentClass;
typedef struct _ValaCCodeAssignmentPrivate ValaCCodeAssignmentPrivate;

#define VALA_TYPE_CCODE_ASSIGNMENT_OPERATOR (vala_ccode_assignment_operator_get_type ())

/**
 * Represents a node in the C code tree.
 */
struct _ValaCCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeNodePrivate * priv;
};

struct _ValaCCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeNode *self);
	void (*write) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_declaration) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_combined) (ValaCCodeNode* self, ValaCCodeWriter* writer);
};

/**
 * Represents an expression node in the C code tree.
 */
struct _ValaCCodeExpression {
	ValaCCodeNode parent_instance;
	ValaCCodeExpressionPrivate * priv;
};

struct _ValaCCodeExpressionClass {
	ValaCCodeNodeClass parent_class;
	void (*write_inner) (ValaCCodeExpression* self, ValaCCodeWriter* writer);
};

/**
 * Represents an assignment expression in the C code.
 */
struct _ValaCCodeAssignment {
	ValaCCodeExpression parent_instance;
	ValaCCodeAssignmentPrivate * priv;
};

struct _ValaCCodeAssignmentClass {
	ValaCCodeExpressionClass parent_class;
};

typedef enum  {
	VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE,
	VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_OR,
	VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_AND,
	VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_XOR,
	VALA_CCODE_ASSIGNMENT_OPERATOR_ADD,
	VALA_CCODE_ASSIGNMENT_OPERATOR_SUB,
	VALA_CCODE_ASSIGNMENT_OPERATOR_MUL,
	VALA_CCODE_ASSIGNMENT_OPERATOR_DIV,
	VALA_CCODE_ASSIGNMENT_OPERATOR_PERCENT,
	VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_LEFT,
	VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_RIGHT
} ValaCCodeAssignmentOperator;

struct _ValaCCodeAssignmentPrivate {
	ValaCCodeExpression* _left;
	ValaCCodeAssignmentOperator _operator;
	ValaCCodeExpression* _right;
};



gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
GType vala_ccode_expression_get_type (void);
GType vala_ccode_assignment_get_type (void);
GType vala_ccode_assignment_operator_get_type (void);
#define VALA_CCODE_ASSIGNMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_ASSIGNMENT, ValaCCodeAssignmentPrivate))
enum  {
	VALA_CCODE_ASSIGNMENT_DUMMY_PROPERTY
};
void vala_ccode_assignment_set_left (ValaCCodeAssignment* self, ValaCCodeExpression* value);
void vala_ccode_assignment_set_operator (ValaCCodeAssignment* self, ValaCCodeAssignmentOperator value);
void vala_ccode_assignment_set_right (ValaCCodeAssignment* self, ValaCCodeExpression* value);
ValaCCodeAssignment* vala_ccode_assignment_new (ValaCCodeExpression* l, ValaCCodeExpression* r, ValaCCodeAssignmentOperator op);
ValaCCodeAssignment* vala_ccode_assignment_construct (GType object_type, ValaCCodeExpression* l, ValaCCodeExpression* r, ValaCCodeAssignmentOperator op);
ValaCCodeAssignment* vala_ccode_assignment_new (ValaCCodeExpression* l, ValaCCodeExpression* r, ValaCCodeAssignmentOperator op);
ValaCCodeExpression* vala_ccode_assignment_get_left (ValaCCodeAssignment* self);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
ValaCCodeAssignmentOperator vala_ccode_assignment_get_operator (ValaCCodeAssignment* self);
ValaCCodeExpression* vala_ccode_assignment_get_right (ValaCCodeAssignment* self);
static void vala_ccode_assignment_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static void vala_ccode_assignment_real_write_inner (ValaCCodeExpression* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_assignment_parent_class = NULL;
static void vala_ccode_assignment_finalize (ValaCCodeNode* obj);



ValaCCodeAssignment* vala_ccode_assignment_construct (GType object_type, ValaCCodeExpression* l, ValaCCodeExpression* r, ValaCCodeAssignmentOperator op) {
	ValaCCodeAssignment* self;
	g_return_val_if_fail (l != NULL, NULL);
	g_return_val_if_fail (r != NULL, NULL);
	self = (ValaCCodeAssignment*) g_type_create_instance (object_type);
	vala_ccode_assignment_set_left (self, l);
	vala_ccode_assignment_set_operator (self, op);
	vala_ccode_assignment_set_right (self, r);
	return self;
}


ValaCCodeAssignment* vala_ccode_assignment_new (ValaCCodeExpression* l, ValaCCodeExpression* r, ValaCCodeAssignmentOperator op) {
	return vala_ccode_assignment_construct (VALA_TYPE_CCODE_ASSIGNMENT, l, r, op);
}


static void vala_ccode_assignment_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeAssignment * self;
	self = (ValaCCodeAssignment*) base;
	g_return_if_fail (writer != NULL);
	vala_ccode_node_write ((ValaCCodeNode*) self->priv->_left, writer);
	vala_ccode_writer_write_string (writer, " ");
	if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_OR) {
		vala_ccode_writer_write_string (writer, "|");
	} else {
		if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_AND) {
			vala_ccode_writer_write_string (writer, "&");
		} else {
			if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
				vala_ccode_writer_write_string (writer, "^");
			} else {
				if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_ADD) {
					vala_ccode_writer_write_string (writer, "+");
				} else {
					if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_SUB) {
						vala_ccode_writer_write_string (writer, "-");
					} else {
						if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_MUL) {
							vala_ccode_writer_write_string (writer, "*");
						} else {
							if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_DIV) {
								vala_ccode_writer_write_string (writer, "/");
							} else {
								if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_PERCENT) {
									vala_ccode_writer_write_string (writer, "%");
								} else {
									if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
										vala_ccode_writer_write_string (writer, "<<");
									} else {
										if (self->priv->_operator == VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
											vala_ccode_writer_write_string (writer, ">>");
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	vala_ccode_writer_write_string (writer, "= ");
	vala_ccode_node_write ((ValaCCodeNode*) self->priv->_right, writer);
}


static void vala_ccode_assignment_real_write_inner (ValaCCodeExpression* base, ValaCCodeWriter* writer) {
	ValaCCodeAssignment * self;
	self = (ValaCCodeAssignment*) base;
	g_return_if_fail (writer != NULL);
	vala_ccode_writer_write_string (writer, "(");
	vala_ccode_node_write ((ValaCCodeNode*) self, writer);
	vala_ccode_writer_write_string (writer, ")");
}


ValaCCodeExpression* vala_ccode_assignment_get_left (ValaCCodeAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_left;
}


void vala_ccode_assignment_set_left (ValaCCodeAssignment* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2_;
	ValaCCodeExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_left = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_ccode_node_unref (self->priv->_left), NULL)), _tmp2_);
}


ValaCCodeAssignmentOperator vala_ccode_assignment_get_operator (ValaCCodeAssignment* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_operator;
}


void vala_ccode_assignment_set_operator (ValaCCodeAssignment* self, ValaCCodeAssignmentOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaCCodeExpression* vala_ccode_assignment_get_right (ValaCCodeAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_right;
}


void vala_ccode_assignment_set_right (ValaCCodeAssignment* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2_;
	ValaCCodeExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_right = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_ccode_node_unref (self->priv->_right), NULL)), _tmp2_);
}


static void vala_ccode_assignment_class_init (ValaCCodeAssignmentClass * klass) {
	vala_ccode_assignment_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_assignment_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeAssignmentPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_assignment_real_write;
	VALA_CCODE_EXPRESSION_CLASS (klass)->write_inner = vala_ccode_assignment_real_write_inner;
}


static void vala_ccode_assignment_instance_init (ValaCCodeAssignment * self) {
	self->priv = VALA_CCODE_ASSIGNMENT_GET_PRIVATE (self);
}


static void vala_ccode_assignment_finalize (ValaCCodeNode* obj) {
	ValaCCodeAssignment * self;
	self = VALA_CCODE_ASSIGNMENT (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_ccode_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_ccode_node_unref (self->priv->_right), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_assignment_parent_class)->finalize (obj);
}


GType vala_ccode_assignment_get_type (void) {
	static GType vala_ccode_assignment_type_id = 0;
	if (vala_ccode_assignment_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeAssignmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_assignment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeAssignment), 0, (GInstanceInitFunc) vala_ccode_assignment_instance_init, NULL };
		vala_ccode_assignment_type_id = g_type_register_static (VALA_TYPE_CCODE_EXPRESSION, "ValaCCodeAssignment", &g_define_type_info, 0);
	}
	return vala_ccode_assignment_type_id;
}



GType vala_ccode_assignment_operator_get_type (void) {
	static GType vala_ccode_assignment_operator_type_id = 0;
	if (G_UNLIKELY (vala_ccode_assignment_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE, "VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE", "simple"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_OR, "VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_AND, "VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_XOR, "VALA_CCODE_ASSIGNMENT_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_ADD, "VALA_CCODE_ASSIGNMENT_OPERATOR_ADD", "add"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_SUB, "VALA_CCODE_ASSIGNMENT_OPERATOR_SUB", "sub"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_MUL, "VALA_CCODE_ASSIGNMENT_OPERATOR_MUL", "mul"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_DIV, "VALA_CCODE_ASSIGNMENT_OPERATOR_DIV", "div"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_PERCENT, "VALA_CCODE_ASSIGNMENT_OPERATOR_PERCENT", "percent"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_LEFT, "VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, "VALA_CCODE_ASSIGNMENT_OPERATOR_SHIFT_RIGHT", "shift-right"}, {0, NULL, NULL}};
		vala_ccode_assignment_operator_type_id = g_enum_register_static ("ValaCCodeAssignmentOperator", values);
	}
	return vala_ccode_assignment_operator_type_id;
}




