/* valasourcefile.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <glib/gstdio.h>
#include <gobject/gvaluecollector.h>


#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;
typedef struct _ValaSourceFilePrivate ValaSourceFilePrivate;

#define VALA_TYPE_CODE_CONTEXT (vala_code_context_get_type ())
#define VALA_CODE_CONTEXT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_CONTEXT, ValaCodeContext))
#define VALA_CODE_CONTEXT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_CONTEXT, ValaCodeContextClass))
#define VALA_IS_CODE_CONTEXT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_CONTEXT))
#define VALA_IS_CODE_CONTEXT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_CONTEXT))
#define VALA_CODE_CONTEXT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_CONTEXT, ValaCodeContextClass))

typedef struct _ValaCodeContext ValaCodeContext;
typedef struct _ValaCodeContextClass ValaCodeContextClass;

#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;

#define VALA_TYPE_USING_DIRECTIVE (vala_using_directive_get_type ())
#define VALA_USING_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_USING_DIRECTIVE, ValaUsingDirective))
#define VALA_USING_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_USING_DIRECTIVE, ValaUsingDirectiveClass))
#define VALA_IS_USING_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_USING_DIRECTIVE))
#define VALA_IS_USING_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_USING_DIRECTIVE))
#define VALA_USING_DIRECTIVE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_USING_DIRECTIVE, ValaUsingDirectiveClass))

typedef struct _ValaUsingDirective ValaUsingDirective;
typedef struct _ValaUsingDirectiveClass ValaUsingDirectiveClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_UNRESOLVED_SYMBOL (vala_unresolved_symbol_get_type ())
#define VALA_UNRESOLVED_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_UNRESOLVED_SYMBOL, ValaUnresolvedSymbol))
#define VALA_UNRESOLVED_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_UNRESOLVED_SYMBOL, ValaUnresolvedSymbolClass))
#define VALA_IS_UNRESOLVED_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_UNRESOLVED_SYMBOL))
#define VALA_IS_UNRESOLVED_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_UNRESOLVED_SYMBOL))
#define VALA_UNRESOLVED_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_UNRESOLVED_SYMBOL, ValaUnresolvedSymbolClass))

typedef struct _ValaUnresolvedSymbol ValaUnresolvedSymbol;
typedef struct _ValaUnresolvedSymbolClass ValaUnresolvedSymbolClass;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;
typedef struct _ValaParamSpecSourceFile ValaParamSpecSourceFile;

/**
 * Represents a Vala source or VAPI package file.
 */
struct _ValaSourceFile {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaSourceFilePrivate * priv;
};

struct _ValaSourceFileClass {
	GTypeClass parent_class;
	void (*finalize) (ValaSourceFile *self);
};

struct _ValaSourceFilePrivate {
	char* _filename;
	char* _comment;
	gboolean _external_package;
	ValaCodeContext* _context;
	GeeList* using_directives;
	GeeList* nodes;
	char* csource_filename;
	char* cinclude_filename;
	GeeArrayList* source_array;
	GMappedFile* mapped_file;
	char* _content;
};

struct _ValaParamSpecSourceFile {
	GParamSpec parent_instance;
};



gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
gpointer vala_code_context_ref (gpointer instance);
void vala_code_context_unref (gpointer instance);
GParamSpec* vala_param_spec_code_context (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_context (GValue* value, gpointer v_object);
gpointer vala_value_get_code_context (const GValue* value);
GType vala_code_context_get_type (void);
gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
GType vala_using_directive_get_type (void);
#define VALA_SOURCE_FILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SOURCE_FILE, ValaSourceFilePrivate))
enum  {
	VALA_SOURCE_FILE_DUMMY_PROPERTY
};
void vala_source_file_set_filename (ValaSourceFile* self, const char* value);
void vala_source_file_set_external_package (ValaSourceFile* self, gboolean value);
void vala_source_file_set_context (ValaSourceFile* self, ValaCodeContext* value);
void vala_source_file_set_content (ValaSourceFile* self, const char* value);
ValaSourceFile* vala_source_file_new (ValaCodeContext* context, const char* filename, gboolean pkg, const char* content);
ValaSourceFile* vala_source_file_construct (GType object_type, ValaCodeContext* context, const char* filename, gboolean pkg, const char* content);
ValaSourceFile* vala_source_file_new (ValaCodeContext* context, const char* filename, gboolean pkg, const char* content);
GType vala_symbol_get_type (void);
static gboolean vala_source_file_same_symbol (ValaSourceFile* self, ValaSymbol* sym1, ValaSymbol* sym2);
ValaSymbol* vala_using_directive_get_namespace_symbol (ValaUsingDirective* self);
void vala_source_file_add_using_directive (ValaSourceFile* self, ValaUsingDirective* ns);
void vala_source_file_clear_using_directives (ValaSourceFile* self);
GType vala_unresolved_symbol_get_type (void);
ValaUnresolvedSymbol* vala_unresolved_symbol_get_inner (ValaUnresolvedSymbol* self);
const char* vala_symbol_get_name (ValaSymbol* self);
GeeList* vala_source_file_get_using_directives (ValaSourceFile* self);
void vala_source_file_add_node (ValaSourceFile* self, ValaCodeNode* node);
void vala_source_file_remove_node (ValaSourceFile* self, ValaCodeNode* node);
GeeList* vala_source_file_get_nodes (ValaSourceFile* self);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
void vala_code_visitor_visit_source_file (ValaCodeVisitor* self, ValaSourceFile* source_file);
void vala_source_file_accept (ValaSourceFile* self, ValaCodeVisitor* visitor);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
void vala_source_file_accept_children (ValaSourceFile* self, ValaCodeVisitor* visitor);
ValaCodeContext* vala_source_file_get_context (ValaSourceFile* self);
const char* vala_code_context_get_basedir (ValaCodeContext* self);
const char* vala_source_file_get_filename (ValaSourceFile* self);
static char* vala_source_file_get_subdir (ValaSourceFile* self);
const char* vala_code_context_get_directory (ValaCodeContext* self);
static char* vala_source_file_get_destination_directory (ValaSourceFile* self);
static char* vala_source_file_get_basename (ValaSourceFile* self);
char* vala_source_file_get_relative_filename (ValaSourceFile* self);
gboolean vala_code_context_get_ccode_only (ValaCodeContext* self);
gboolean vala_code_context_get_save_csources (ValaCodeContext* self);
char* vala_source_file_get_csource_filename (ValaSourceFile* self);
const char* vala_code_context_get_header_filename (ValaCodeContext* self);
char* vala_source_file_get_cinclude_filename (ValaSourceFile* self);
const char* vala_source_file_get_content (ValaSourceFile* self);
static void vala_source_file_read_source_lines (ValaSourceFile* self, const char* cont);
static void vala_source_file_read_source_file (ValaSourceFile* self);
char* vala_source_file_get_source_line (ValaSourceFile* self, gint lineno);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_report_error (ValaSourceReference* source, const char* message);
gchar* vala_source_file_get_mapped_contents (ValaSourceFile* self);
gsize vala_source_file_get_mapped_length (ValaSourceFile* self);
GType vala_semantic_analyzer_get_type (void);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
gboolean vala_source_file_check (ValaSourceFile* self, ValaSemanticAnalyzer* analyzer);
const char* vala_source_file_get_comment (ValaSourceFile* self);
void vala_source_file_set_comment (ValaSourceFile* self, const char* value);
gboolean vala_source_file_get_external_package (ValaSourceFile* self);
static gpointer vala_source_file_parent_class = NULL;
static void vala_source_file_finalize (ValaSourceFile* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new source file.
 *
 * @param filename source file name
 * @param pkg      true if this is a VAPI package file
 * @return         newly created source file
 */
ValaSourceFile* vala_source_file_construct (GType object_type, ValaCodeContext* context, const char* filename, gboolean pkg, const char* content) {
	ValaSourceFile* self;
	g_return_val_if_fail (context != NULL, NULL);
	g_return_val_if_fail (filename != NULL, NULL);
	self = (ValaSourceFile*) g_type_create_instance (object_type);
	vala_source_file_set_filename (self, filename);
	vala_source_file_set_external_package (self, pkg);
	vala_source_file_set_context (self, context);
	vala_source_file_set_content (self, content);
	return self;
}


ValaSourceFile* vala_source_file_new (ValaCodeContext* context, const char* filename, gboolean pkg, const char* content) {
	return vala_source_file_construct (VALA_TYPE_SOURCE_FILE, context, filename, pkg, content);
}


/**
 * Adds a new using directive with the specified namespace.
 *
 * @param ns reference to namespace
 */
void vala_source_file_add_using_directive (ValaSourceFile* self, ValaUsingDirective* ns) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
	{
		GeeIterator* _using_directive_it;
		_using_directive_it = gee_iterable_iterator ((GeeIterable*) self->priv->using_directives);
		while (gee_iterator_next (_using_directive_it)) {
			ValaUsingDirective* using_directive;
			using_directive = (ValaUsingDirective*) gee_iterator_get (_using_directive_it);
			if (vala_source_file_same_symbol (self, vala_using_directive_get_namespace_symbol (using_directive), vala_using_directive_get_namespace_symbol (ns))) {
				/* ignore duplicates*/
				(using_directive == NULL) ? NULL : (using_directive = (vala_code_node_unref (using_directive), NULL));
				(_using_directive_it == NULL) ? NULL : (_using_directive_it = (gee_collection_object_unref (_using_directive_it), NULL));
				return;
			}
			(using_directive == NULL) ? NULL : (using_directive = (vala_code_node_unref (using_directive), NULL));
		}
		(_using_directive_it == NULL) ? NULL : (_using_directive_it = (gee_collection_object_unref (_using_directive_it), NULL));
	}
	gee_collection_add ((GeeCollection*) self->priv->using_directives, ns);
}


void vala_source_file_clear_using_directives (ValaSourceFile* self) {
	g_return_if_fail (self != NULL);
	gee_collection_clear ((GeeCollection*) self->priv->using_directives);
}


static gboolean vala_source_file_same_symbol (ValaSourceFile* self, ValaSymbol* sym1, ValaSymbol* sym2) {
	ValaUnresolvedSymbol* _tmp2_;
	ValaSymbol* _tmp1_;
	ValaUnresolvedSymbol* unresolved_symbol1;
	ValaUnresolvedSymbol* _tmp4_;
	ValaSymbol* _tmp3_;
	ValaUnresolvedSymbol* unresolved_symbol2;
	gboolean _tmp5_;
	gboolean _tmp7_;
	g_return_val_if_fail (self != NULL, FALSE);
	if (sym1 == sym2) {
		return TRUE;
	}
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	unresolved_symbol1 = (_tmp2_ = (_tmp1_ = sym1, VALA_IS_UNRESOLVED_SYMBOL (_tmp1_) ? ((ValaUnresolvedSymbol*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	unresolved_symbol2 = (_tmp4_ = (_tmp3_ = sym2, VALA_IS_UNRESOLVED_SYMBOL (_tmp3_) ? ((ValaUnresolvedSymbol*) _tmp3_) : NULL), (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
	_tmp5_ = FALSE;
	if (unresolved_symbol1 != NULL) {
		_tmp5_ = unresolved_symbol2 != NULL;
	} else {
		_tmp5_ = FALSE;
	}
	if (_tmp5_) {
		if (vala_source_file_same_symbol (self, (ValaSymbol*) vala_unresolved_symbol_get_inner (unresolved_symbol1), (ValaSymbol*) vala_unresolved_symbol_get_inner (unresolved_symbol2))) {
			gboolean _tmp6_;
			return (_tmp6_ = _vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) unresolved_symbol1), vala_symbol_get_name ((ValaSymbol*) unresolved_symbol2)) == 0, (unresolved_symbol1 == NULL) ? NULL : (unresolved_symbol1 = (vala_code_node_unref (unresolved_symbol1), NULL)), (unresolved_symbol2 == NULL) ? NULL : (unresolved_symbol2 = (vala_code_node_unref (unresolved_symbol2), NULL)), _tmp6_);
		}
	}
	return (_tmp7_ = FALSE, (unresolved_symbol1 == NULL) ? NULL : (unresolved_symbol1 = (vala_code_node_unref (unresolved_symbol1), NULL)), (unresolved_symbol2 == NULL) ? NULL : (unresolved_symbol2 = (vala_code_node_unref (unresolved_symbol2), NULL)), _tmp7_);
}


/**
 * Returns a copy of the list of using directives.
 *
 * @return using directive list
 */
GeeList* vala_source_file_get_using_directives (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_USING_DIRECTIVE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->using_directives);
}


/**
 * Adds the specified code node to this source file.
 *
 * @param node a code node
 */
void vala_source_file_add_node (ValaSourceFile* self, ValaCodeNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	gee_collection_add ((GeeCollection*) self->priv->nodes, node);
}


void vala_source_file_remove_node (ValaSourceFile* self, ValaCodeNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	gee_collection_remove ((GeeCollection*) self->priv->nodes, node);
}


/**
 * Returns a copy of the list of code nodes.
 *
 * @return code node list
 */
GeeList* vala_source_file_get_nodes (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CODE_NODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->nodes);
}


void vala_source_file_accept (ValaSourceFile* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_source_file (visitor, self);
}


void vala_source_file_accept_children (ValaSourceFile* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _ns_ref_it;
		_ns_ref_it = gee_iterable_iterator ((GeeIterable*) self->priv->using_directives);
		while (gee_iterator_next (_ns_ref_it)) {
			ValaUsingDirective* ns_ref;
			ns_ref = (ValaUsingDirective*) gee_iterator_get (_ns_ref_it);
			vala_code_node_accept ((ValaCodeNode*) ns_ref, visitor);
			(ns_ref == NULL) ? NULL : (ns_ref = (vala_code_node_unref (ns_ref), NULL));
		}
		(_ns_ref_it == NULL) ? NULL : (_ns_ref_it = (gee_collection_object_unref (_ns_ref_it), NULL));
	}
	{
		GeeIterator* _node_it;
		_node_it = gee_iterable_iterator ((GeeIterable*) self->priv->nodes);
		while (gee_iterator_next (_node_it)) {
			ValaCodeNode* node;
			node = (ValaCodeNode*) gee_iterator_get (_node_it);
			vala_code_node_accept (node, visitor);
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(_node_it == NULL) ? NULL : (_node_it = (gee_collection_object_unref (_node_it), NULL));
	}
}


static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static char* vala_source_file_get_subdir (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_code_context_get_basedir (self->priv->_context) == NULL) {
		return g_strdup ("");
	}
	/* filename and basedir are already canonicalized*/
	if (g_str_has_prefix (self->priv->_filename, vala_code_context_get_basedir (self->priv->_context))) {
		char* basename;
		char* subdir;
		char* _tmp3_;
		basename = g_path_get_basename (self->priv->_filename);
		subdir = string_substring (self->priv->_filename, g_utf8_strlen (vala_code_context_get_basedir (self->priv->_context), -1), (g_utf8_strlen (self->priv->_filename, -1) - g_utf8_strlen (vala_code_context_get_basedir (self->priv->_context), -1)) - g_utf8_strlen (basename, -1));
		while (g_utf8_get_char (g_utf8_offset_to_pointer (subdir, 0)) == '/') {
			char* _tmp2_;
			const char* _tmp1_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			subdir = (_tmp2_ = (_tmp1_ = g_utf8_offset_to_pointer (subdir, (glong) 1), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), subdir = (g_free (subdir), NULL), _tmp2_);
		}
		_tmp3_ = NULL;
		return (_tmp3_ = subdir, basename = (g_free (basename), NULL), _tmp3_);
	}
	return g_strdup ("");
}


static char* vala_source_file_get_destination_directory (ValaSourceFile* self) {
	char* _tmp1_;
	char* _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_code_context_get_directory (self->priv->_context) == NULL) {
		return vala_source_file_get_subdir (self);
	}
	_tmp1_ = NULL;
	_tmp2_ = NULL;
	return (_tmp2_ = g_strdup_printf ("%s/%s", vala_code_context_get_directory (self->priv->_context), _tmp1_ = vala_source_file_get_subdir (self)), _tmp1_ = (g_free (_tmp1_), NULL), _tmp2_);
}


static char* vala_source_file_get_basename (ValaSourceFile* self) {
	glong dot;
	char* _tmp0_;
	char* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	dot = g_utf8_pointer_to_offset (self->priv->_filename, g_utf8_strrchr (self->priv->_filename, (glong) (-1), (gunichar) '.'));
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	return (_tmp1_ = g_path_get_basename (_tmp0_ = string_substring (self->priv->_filename, (glong) 0, dot)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
}


char* vala_source_file_get_relative_filename (ValaSourceFile* self) {
	char* _tmp1_;
	char* _tmp0_;
	char* _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	return (_tmp2_ = g_strconcat (_tmp0_ = vala_source_file_get_subdir (self), _tmp1_ = g_path_get_basename (self->priv->_filename), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
}


/**
 * Returns the filename to use when generating C source files.
 *
 * @return generated C source filename
 */
char* vala_source_file_get_csource_filename (ValaSourceFile* self) {
	const char* _tmp7_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->csource_filename == NULL) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (vala_code_context_get_ccode_only (self->priv->_context)) {
			_tmp0_ = TRUE;
		} else {
			_tmp0_ = vala_code_context_get_save_csources (self->priv->_context);
		}
		if (_tmp0_) {
			char* _tmp3_;
			char* _tmp2_;
			char* _tmp1_;
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			self->priv->csource_filename = (_tmp3_ = g_strdup_printf ("%s%s.c", _tmp1_ = vala_source_file_get_destination_directory (self), _tmp2_ = vala_source_file_get_basename (self)), self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL), _tmp3_);
			_tmp2_ = (g_free (_tmp2_), NULL);
			_tmp1_ = (g_free (_tmp1_), NULL);
		} else {
			char* _tmp6_;
			char* _tmp5_;
			char* _tmp4_;
			/* temporary file*/
			_tmp6_ = NULL;
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			self->priv->csource_filename = (_tmp6_ = g_strdup_printf ("%s%s.vala.c", _tmp4_ = vala_source_file_get_destination_directory (self), _tmp5_ = vala_source_file_get_basename (self)), self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL), _tmp6_);
			_tmp5_ = (g_free (_tmp5_), NULL);
			_tmp4_ = (g_free (_tmp4_), NULL);
		}
	}
	_tmp7_ = NULL;
	return (_tmp7_ = self->priv->csource_filename, (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_));
}


/**
 * Returns the filename to use when including the generated C header
 * file.
 *
 * @return C header filename to include
 */
char* vala_source_file_get_cinclude_filename (ValaSourceFile* self) {
	const char* _tmp4_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cinclude_filename == NULL) {
		if (vala_code_context_get_header_filename (self->priv->_context) != NULL) {
			char* _tmp0_;
			_tmp0_ = NULL;
			self->priv->cinclude_filename = (_tmp0_ = g_path_get_basename (vala_code_context_get_header_filename (self->priv->_context)), self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL), _tmp0_);
		} else {
			char* _tmp3_;
			char* _tmp2_;
			char* _tmp1_;
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			self->priv->cinclude_filename = (_tmp3_ = g_strdup_printf ("%s%s.h", _tmp1_ = vala_source_file_get_subdir (self), _tmp2_ = vala_source_file_get_basename (self)), self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL), _tmp3_);
			_tmp2_ = (g_free (_tmp2_), NULL);
			_tmp1_ = (g_free (_tmp1_), NULL);
		}
	}
	_tmp4_ = NULL;
	return (_tmp4_ = self->priv->cinclude_filename, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_));
}


/**
 * Returns the requested line from this file, loading it if needed.
 *
 * @param lineno 1-based line number
 * @return       the specified source line
 */
char* vala_source_file_get_source_line (ValaSourceFile* self, gint lineno) {
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->source_array == NULL) {
		if (vala_source_file_get_content (self) != NULL) {
			vala_source_file_read_source_lines (self, vala_source_file_get_content (self));
		} else {
			vala_source_file_read_source_file (self);
		}
	}
	_tmp0_ = FALSE;
	if (lineno < 1) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = lineno > gee_collection_get_size ((GeeCollection*) self->priv->source_array);
	}
	if (_tmp0_) {
		return NULL;
	}
	return (char*) gee_list_get ((GeeList*) self->priv->source_array, lineno - 1);
}


/**
 * Parses the input file into ::source_array.
 */
static void vala_source_file_read_source_file (ValaSourceFile* self) {
	GError * _inner_error_;
	char* cont;
	g_return_if_fail (self != NULL);
	_inner_error_ = NULL;
	cont = NULL;
	{
		char* _tmp2_;
		gboolean _tmp1_;
		char* _tmp0_;
		_tmp2_ = NULL;
		_tmp0_ = NULL;
		_tmp1_ = g_file_get_contents (self->priv->_filename, &_tmp0_, NULL, &_inner_error_);
		cont = (_tmp2_ = _tmp0_, cont = (g_free (cont), NULL), _tmp2_);
		_tmp1_;
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch6_g_file_error;
			}
			goto __finally6;
		}
	}
	goto __finally6;
	__catch6_g_file_error:
	{
		GError * fe;
		fe = _inner_error_;
		_inner_error_ = NULL;
		{
			(fe == NULL) ? NULL : (fe = (g_error_free (fe), NULL));
			cont = (g_free (cont), NULL);
			return;
		}
	}
	__finally6:
	if (_inner_error_ != NULL) {
		cont = (g_free (cont), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	vala_source_file_read_source_lines (self, cont);
	cont = (g_free (cont), NULL);
}


static void vala_source_file_read_source_lines (ValaSourceFile* self, const char* cont) {
	GeeArrayList* _tmp0_;
	char** _tmp2_;
	gint lines_size;
	gint lines_length1;
	char** _tmp1_;
	char** lines;
	gint idx;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cont != NULL);
	_tmp0_ = NULL;
	self->priv->source_array = (_tmp0_ = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal), (self->priv->source_array == NULL) ? NULL : (self->priv->source_array = (gee_collection_object_unref (self->priv->source_array), NULL)), _tmp0_);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	lines = (_tmp2_ = _tmp1_ = g_strsplit (cont, "\n", 0), lines_length1 = _vala_array_length (_tmp1_), lines_size = lines_length1, _tmp2_);
	idx = 0;
	for (idx = 0; lines[idx] != NULL; idx = idx + 1) {
		gee_collection_add ((GeeCollection*) self->priv->source_array, lines[idx]);
	}
	lines = (_vala_array_free (lines, lines_length1, (GDestroyNotify) g_free), NULL);
}


gchar* vala_source_file_get_mapped_contents (ValaSourceFile* self) {
	GError * _inner_error_;
	g_return_val_if_fail (self != NULL, NULL);
	_inner_error_ = NULL;
	if (vala_source_file_get_content (self) != NULL) {
		return vala_source_file_get_content (self);
	}
	if (self->priv->mapped_file == NULL) {
		{
			GMappedFile* _tmp1_;
			_tmp1_ = NULL;
			self->priv->mapped_file = (_tmp1_ = g_mapped_file_new (self->priv->_filename, FALSE, &_inner_error_), (self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL)), _tmp1_);
			if (_inner_error_ != NULL) {
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch7_g_file_error;
				}
				goto __finally7;
			}
		}
		goto __finally7;
		__catch7_g_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				char* _tmp2_;
				gchar* _tmp3_;
				_tmp2_ = NULL;
				vala_report_error (NULL, _tmp2_ = g_strdup_printf ("Unable to map file `%s': %s", self->priv->_filename, e->message));
				_tmp2_ = (g_free (_tmp2_), NULL);
				return (_tmp3_ = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), _tmp3_);
			}
		}
		__finally7:
		if (_inner_error_ != NULL) {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return NULL;
		}
	}
	return g_mapped_file_get_contents (self->priv->mapped_file);
}


static glong string_get_length (const char* self) {
	g_return_val_if_fail (self != NULL, 0L);
	return g_utf8_strlen (self, -1);
}


gsize vala_source_file_get_mapped_length (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, 0UL);
	if (vala_source_file_get_content (self) != NULL) {
		return (gsize) string_get_length (vala_source_file_get_content (self));
	}
	return (gsize) g_mapped_file_get_length (self->priv->mapped_file);
}


gboolean vala_source_file_check (ValaSourceFile* self, ValaSemanticAnalyzer* analyzer) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (analyzer != NULL, FALSE);
	{
		GeeIterator* _node_it;
		_node_it = gee_iterable_iterator ((GeeIterable*) self->priv->nodes);
		while (gee_iterator_next (_node_it)) {
			ValaCodeNode* node;
			node = (ValaCodeNode*) gee_iterator_get (_node_it);
			vala_code_node_check (node, analyzer);
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(_node_it == NULL) ? NULL : (_node_it = (gee_collection_object_unref (_node_it), NULL));
	}
	return TRUE;
}


const char* vala_source_file_get_filename (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_filename;
}


void vala_source_file_set_filename (ValaSourceFile* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_filename = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_filename = (g_free (self->priv->_filename), NULL), _tmp2_);
}


const char* vala_source_file_get_comment (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_comment;
}


void vala_source_file_set_comment (ValaSourceFile* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_comment = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_comment = (g_free (self->priv->_comment), NULL), _tmp2_);
}


gboolean vala_source_file_get_external_package (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_external_package;
}


void vala_source_file_set_external_package (ValaSourceFile* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_external_package = value;
}


ValaCodeContext* vala_source_file_get_context (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_context;
}


void vala_source_file_set_context (ValaSourceFile* self, ValaCodeContext* value) {
	g_return_if_fail (self != NULL);
	self->priv->_context = value;
}


const char* vala_source_file_get_content (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_content;
}


void vala_source_file_set_content (ValaSourceFile* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	GeeArrayList* _tmp3_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_content = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_content = (g_free (self->priv->_content), NULL), _tmp2_);
	_tmp3_ = NULL;
	self->priv->source_array = (_tmp3_ = NULL, (self->priv->source_array == NULL) ? NULL : (self->priv->source_array = (gee_collection_object_unref (self->priv->source_array), NULL)), _tmp3_);
}


static void vala_value_source_file_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_source_file_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_source_file_unref (value->data[0].v_pointer);
	}
}


static void vala_value_source_file_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_source_file_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_source_file_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_source_file_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaSourceFile* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_source_file_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_source_file_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaSourceFile** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_source_file_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecSourceFile* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_SOURCE_FILE), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_source_file (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SOURCE_FILE), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_source_file (GValue* value, gpointer v_object) {
	ValaSourceFile* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SOURCE_FILE));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_SOURCE_FILE));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_source_file_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_source_file_unref (old);
	}
}


static void vala_source_file_class_init (ValaSourceFileClass * klass) {
	vala_source_file_parent_class = g_type_class_peek_parent (klass);
	VALA_SOURCE_FILE_CLASS (klass)->finalize = vala_source_file_finalize;
	g_type_class_add_private (klass, sizeof (ValaSourceFilePrivate));
}


static void vala_source_file_instance_init (ValaSourceFile * self) {
	self->priv = VALA_SOURCE_FILE_GET_PRIVATE (self);
	self->priv->using_directives = (GeeList*) gee_array_list_new (VALA_TYPE_USING_DIRECTIVE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->nodes = (GeeList*) gee_array_list_new (VALA_TYPE_CODE_NODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->csource_filename = NULL;
	self->priv->cinclude_filename = NULL;
	self->priv->source_array = NULL;
	self->priv->mapped_file = NULL;
	self->priv->_content = NULL;
	self->ref_count = 1;
}


static void vala_source_file_finalize (ValaSourceFile* obj) {
	ValaSourceFile * self;
	self = VALA_SOURCE_FILE (obj);
	self->priv->_filename = (g_free (self->priv->_filename), NULL);
	self->priv->_comment = (g_free (self->priv->_comment), NULL);
	(self->priv->using_directives == NULL) ? NULL : (self->priv->using_directives = (gee_collection_object_unref (self->priv->using_directives), NULL));
	(self->priv->nodes == NULL) ? NULL : (self->priv->nodes = (gee_collection_object_unref (self->priv->nodes), NULL));
	self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL);
	self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL);
	(self->priv->source_array == NULL) ? NULL : (self->priv->source_array = (gee_collection_object_unref (self->priv->source_array), NULL));
	(self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL));
	self->priv->_content = (g_free (self->priv->_content), NULL);
}


GType vala_source_file_get_type (void) {
	static GType vala_source_file_type_id = 0;
	if (vala_source_file_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_source_file_init, vala_value_source_file_free_value, vala_value_source_file_copy_value, vala_value_source_file_peek_pointer, "p", vala_value_source_file_collect_value, "p", vala_value_source_file_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaSourceFileClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_source_file_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSourceFile), 0, (GInstanceInitFunc) vala_source_file_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_source_file_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaSourceFile", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_source_file_type_id;
}


gpointer vala_source_file_ref (gpointer instance) {
	ValaSourceFile* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_source_file_unref (gpointer instance) {
	ValaSourceFile* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_SOURCE_FILE_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




