/* valaccodemethodmodule.vala
 *
 * Copyright (C) 2007-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdlib.h>
#include <string.h>
#include <valaccode.h>
#include <gee.h>
#include <float.h>
#include <math.h>


#define VALA_TYPE_CCODE_MODULE (vala_ccode_module_get_type ())
#define VALA_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModule))
#define VALA_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))
#define VALA_IS_CCODE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_MODULE))
#define VALA_IS_CCODE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_MODULE))
#define VALA_CCODE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_MODULE, ValaCCodeModuleClass))

typedef struct _ValaCCodeModule ValaCCodeModule;
typedef struct _ValaCCodeModuleClass ValaCCodeModuleClass;
typedef struct _ValaCCodeModulePrivate ValaCCodeModulePrivate;

#define VALA_TYPE_CCODE_BASE_MODULE (vala_ccode_base_module_get_type ())
#define VALA_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModule))
#define VALA_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))
#define VALA_IS_CCODE_BASE_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_IS_CCODE_BASE_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_BASE_MODULE))
#define VALA_CCODE_BASE_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_BASE_MODULE, ValaCCodeBaseModuleClass))

typedef struct _ValaCCodeBaseModule ValaCCodeBaseModule;
typedef struct _ValaCCodeBaseModuleClass ValaCCodeBaseModuleClass;
typedef struct _ValaCCodeBaseModulePrivate ValaCCodeBaseModulePrivate;

#define VALA_TYPE_CCODE_DECLARATION_SPACE (vala_ccode_declaration_space_get_type ())
#define VALA_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpace))
#define VALA_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))
#define VALA_IS_CCODE_DECLARATION_SPACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_IS_CCODE_DECLARATION_SPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_DECLARATION_SPACE))
#define VALA_CCODE_DECLARATION_SPACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_DECLARATION_SPACE, ValaCCodeDeclarationSpaceClass))

typedef struct _ValaCCodeDeclarationSpace ValaCCodeDeclarationSpace;
typedef struct _ValaCCodeDeclarationSpaceClass ValaCCodeDeclarationSpaceClass;

#define VALA_TYPE_CCODE_STRUCT_MODULE (vala_ccode_struct_module_get_type ())
#define VALA_CCODE_STRUCT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModule))
#define VALA_CCODE_STRUCT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModuleClass))
#define VALA_IS_CCODE_STRUCT_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_STRUCT_MODULE))
#define VALA_IS_CCODE_STRUCT_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_STRUCT_MODULE))
#define VALA_CCODE_STRUCT_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_STRUCT_MODULE, ValaCCodeStructModuleClass))

typedef struct _ValaCCodeStructModule ValaCCodeStructModule;
typedef struct _ValaCCodeStructModuleClass ValaCCodeStructModuleClass;
typedef struct _ValaCCodeStructModulePrivate ValaCCodeStructModulePrivate;

#define VALA_TYPE_CCODE_METHOD_MODULE (vala_ccode_method_module_get_type ())
#define VALA_CCODE_METHOD_MODULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_METHOD_MODULE, ValaCCodeMethodModule))
#define VALA_CCODE_METHOD_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_METHOD_MODULE, ValaCCodeMethodModuleClass))
#define VALA_IS_CCODE_METHOD_MODULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_METHOD_MODULE))
#define VALA_IS_CCODE_METHOD_MODULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_METHOD_MODULE))
#define VALA_CCODE_METHOD_MODULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_METHOD_MODULE, ValaCCodeMethodModuleClass))

typedef struct _ValaCCodeMethodModule ValaCCodeMethodModule;
typedef struct _ValaCCodeMethodModuleClass ValaCCodeMethodModuleClass;
typedef struct _ValaCCodeMethodModulePrivate ValaCCodeMethodModulePrivate;

#define VALA_TYPE_CCODE_GENERATOR (vala_ccode_generator_get_type ())
#define VALA_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGenerator))
#define VALA_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))
#define VALA_IS_CCODE_GENERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_GENERATOR))
#define VALA_IS_CCODE_GENERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_GENERATOR))
#define VALA_CCODE_GENERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_GENERATOR, ValaCCodeGeneratorClass))

typedef struct _ValaCCodeGenerator ValaCCodeGenerator;
typedef struct _ValaCCodeGeneratorClass ValaCCodeGeneratorClass;

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeModule {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeModulePrivate * priv;
};

struct _ValaCCodeModuleClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeModule *self);
	void (*emit) (ValaCCodeModule* self, ValaCodeContext* context);
	void (*visit_source_file) (ValaCCodeModule* self, ValaSourceFile* source_file);
	void (*visit_class) (ValaCCodeModule* self, ValaClass* cl);
	void (*visit_interface) (ValaCCodeModule* self, ValaInterface* iface);
	void (*visit_struct) (ValaCCodeModule* self, ValaStruct* st);
	void (*visit_enum) (ValaCCodeModule* self, ValaEnum* en);
	void (*visit_error_domain) (ValaCCodeModule* self, ValaErrorDomain* edomain);
	void (*visit_delegate) (ValaCCodeModule* self, ValaDelegate* d);
	void (*visit_member) (ValaCCodeModule* self, ValaMember* m);
	void (*visit_constant) (ValaCCodeModule* self, ValaConstant* c);
	void (*visit_field) (ValaCCodeModule* self, ValaField* f);
	void (*visit_method) (ValaCCodeModule* self, ValaMethod* m);
	void (*visit_creation_method) (ValaCCodeModule* self, ValaCreationMethod* m);
	void (*visit_formal_parameter) (ValaCCodeModule* self, ValaFormalParameter* p);
	void (*visit_property) (ValaCCodeModule* self, ValaProperty* prop);
	void (*visit_property_accessor) (ValaCCodeModule* self, ValaPropertyAccessor* acc);
	void (*visit_signal) (ValaCCodeModule* self, ValaSignal* sig);
	void (*visit_constructor) (ValaCCodeModule* self, ValaConstructor* c);
	void (*visit_destructor) (ValaCCodeModule* self, ValaDestructor* d);
	void (*visit_block) (ValaCCodeModule* self, ValaBlock* b);
	void (*visit_empty_statement) (ValaCCodeModule* self, ValaEmptyStatement* stmt);
	void (*visit_declaration_statement) (ValaCCodeModule* self, ValaDeclarationStatement* stmt);
	void (*visit_local_variable) (ValaCCodeModule* self, ValaLocalVariable* local);
	void (*visit_initializer_list) (ValaCCodeModule* self, ValaInitializerList* list);
	void (*visit_end_full_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_expression_statement) (ValaCCodeModule* self, ValaExpressionStatement* stmt);
	void (*visit_if_statement) (ValaCCodeModule* self, ValaIfStatement* stmt);
	void (*visit_switch_statement) (ValaCCodeModule* self, ValaSwitchStatement* stmt);
	void (*visit_switch_section) (ValaCCodeModule* self, ValaSwitchSection* section);
	void (*visit_switch_label) (ValaCCodeModule* self, ValaSwitchLabel* label);
	void (*visit_while_statement) (ValaCCodeModule* self, ValaWhileStatement* stmt);
	void (*visit_do_statement) (ValaCCodeModule* self, ValaDoStatement* stmt);
	void (*visit_for_statement) (ValaCCodeModule* self, ValaForStatement* stmt);
	void (*visit_foreach_statement) (ValaCCodeModule* self, ValaForeachStatement* stmt);
	void (*visit_break_statement) (ValaCCodeModule* self, ValaBreakStatement* stmt);
	void (*visit_continue_statement) (ValaCCodeModule* self, ValaContinueStatement* stmt);
	void (*visit_return_statement) (ValaCCodeModule* self, ValaReturnStatement* stmt);
	void (*visit_yield_statement) (ValaCCodeModule* self, ValaYieldStatement* stmt);
	void (*visit_throw_statement) (ValaCCodeModule* self, ValaThrowStatement* stmt);
	void (*visit_try_statement) (ValaCCodeModule* self, ValaTryStatement* stmt);
	void (*visit_catch_clause) (ValaCCodeModule* self, ValaCatchClause* clause);
	void (*visit_lock_statement) (ValaCCodeModule* self, ValaLockStatement* stmt);
	void (*visit_delete_statement) (ValaCCodeModule* self, ValaDeleteStatement* stmt);
	void (*visit_expression) (ValaCCodeModule* self, ValaExpression* expr);
	void (*visit_array_creation_expression) (ValaCCodeModule* self, ValaArrayCreationExpression* expr);
	void (*visit_boolean_literal) (ValaCCodeModule* self, ValaBooleanLiteral* expr);
	void (*visit_character_literal) (ValaCCodeModule* self, ValaCharacterLiteral* expr);
	void (*visit_integer_literal) (ValaCCodeModule* self, ValaIntegerLiteral* expr);
	void (*visit_real_literal) (ValaCCodeModule* self, ValaRealLiteral* expr);
	void (*visit_string_literal) (ValaCCodeModule* self, ValaStringLiteral* expr);
	void (*visit_null_literal) (ValaCCodeModule* self, ValaNullLiteral* expr);
	void (*visit_member_access) (ValaCCodeModule* self, ValaMemberAccess* expr);
	void (*visit_method_call) (ValaCCodeModule* self, ValaMethodCall* expr);
	void (*visit_element_access) (ValaCCodeModule* self, ValaElementAccess* expr);
	void (*visit_base_access) (ValaCCodeModule* self, ValaBaseAccess* expr);
	void (*visit_postfix_expression) (ValaCCodeModule* self, ValaPostfixExpression* expr);
	void (*visit_object_creation_expression) (ValaCCodeModule* self, ValaObjectCreationExpression* expr);
	void (*visit_sizeof_expression) (ValaCCodeModule* self, ValaSizeofExpression* expr);
	void (*visit_typeof_expression) (ValaCCodeModule* self, ValaTypeofExpression* expr);
	void (*visit_unary_expression) (ValaCCodeModule* self, ValaUnaryExpression* expr);
	void (*visit_cast_expression) (ValaCCodeModule* self, ValaCastExpression* expr);
	void (*visit_pointer_indirection) (ValaCCodeModule* self, ValaPointerIndirection* expr);
	void (*visit_addressof_expression) (ValaCCodeModule* self, ValaAddressofExpression* expr);
	void (*visit_reference_transfer_expression) (ValaCCodeModule* self, ValaReferenceTransferExpression* expr);
	void (*visit_binary_expression) (ValaCCodeModule* self, ValaBinaryExpression* expr);
	void (*visit_type_check) (ValaCCodeModule* self, ValaTypeCheck* expr);
	void (*visit_lambda_expression) (ValaCCodeModule* self, ValaLambdaExpression* l);
	void (*visit_assignment) (ValaCCodeModule* self, ValaAssignment* a);
	char* (*get_custom_creturn_type) (ValaCCodeModule* self, ValaMethod* m);
	void (*generate_dynamic_method_wrapper) (ValaCCodeModule* self, ValaDynamicMethod* method);
	gboolean (*method_has_wrapper) (ValaCCodeModule* self, ValaMethod* method);
	ValaCCodeIdentifier* (*get_value_setter_function) (ValaCCodeModule* self, ValaDataType* type_reference);
	ValaCCodeExpression* (*get_construct_property_assignment) (ValaCCodeModule* self, ValaCCodeConstant* canonical_cconstant, ValaDataType* property_type, ValaCCodeExpression* value);
	ValaCCodeFunctionCall* (*get_param_spec) (ValaCCodeModule* self, ValaProperty* prop);
	ValaCCodeFunctionCall* (*get_signal_creation) (ValaCCodeModule* self, ValaSignal* sig, ValaTypeSymbol* type);
	ValaCCodeFragment* (*register_dbus_info) (ValaCCodeModule* self, ValaObjectTypeSymbol* bindable);
	char* (*get_dynamic_property_getter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_property_setter_cname) (ValaCCodeModule* self, ValaDynamicProperty* node);
	char* (*get_dynamic_signal_cname) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_connect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	char* (*get_dynamic_signal_disconnect_wrapper_name) (ValaCCodeModule* self, ValaDynamicSignal* node);
	void (*generate_marshaller) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, gboolean dbus);
	char* (*get_marshaller_function) (ValaCCodeModule* self, GeeList* params, ValaDataType* return_type, const char* prefix, gboolean dbus);
	char* (*get_array_length_cname) (ValaCCodeModule* self, const char* array_cname, gint dim);
	ValaCCodeExpression* (*get_array_length_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr, gint dim);
	char* (*get_array_size_cname) (ValaCCodeModule* self, const char* array_cname);
	ValaCCodeExpression* (*get_array_size_cexpression) (ValaCCodeModule* self, ValaExpression* array_expr);
	void (*add_simple_check) (ValaCCodeModule* self, ValaCodeNode* node, ValaCCodeFragment* cfrag);
};

/**
 * Code visitor generating C Code.
 */
struct _ValaCCodeBaseModule {
	ValaCCodeModule parent_instance;
	ValaCCodeBaseModulePrivate * priv;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaTypeSymbol* current_type_symbol;
	ValaClass* current_class;
	ValaMethod* current_method;
	ValaDataType* current_return_type;
	ValaTryStatement* current_try;
	ValaPropertyAccessor* current_property_accessor;
	ValaCCodeDeclarationSpace* header_declarations;
	ValaCCodeDeclarationSpace* internal_header_declarations;
	ValaCCodeDeclarationSpace* source_declarations;
	ValaCCodeFragment* source_signal_marshaller_declaration;
	ValaCCodeFragment* source_type_member_definition;
	ValaCCodeFragment* class_init_fragment;
	ValaCCodeFragment* base_init_fragment;
	ValaCCodeFragment* class_finalize_fragment;
	ValaCCodeFragment* base_finalize_fragment;
	ValaCCodeFragment* instance_init_fragment;
	ValaCCodeFragment* instance_finalize_fragment;
	ValaCCodeFragment* source_signal_marshaller_definition;
	ValaCCodeFragment* module_init_fragment;
	ValaCCodeStruct* param_spec_struct;
	ValaCCodeStruct* closure_struct;
	ValaCCodeEnum* prop_enum;
	ValaCCodeFunction* function;
	ValaCCodeFragment* pre_statement_fragment;
	GeeArrayList* temp_vars;
	GeeArrayList* temp_ref_vars;
	GeeSet* user_marshal_set;
	GeeSet* predefined_marshal_set;
	gint next_temp_var_id;
	gboolean in_creation_method;
	gboolean in_constructor;
	gboolean in_static_or_class_ctor;
	gboolean current_method_inner_error;
	gint next_coroutine_state;
	ValaDataType* bool_type;
	ValaDataType* char_type;
	ValaDataType* uchar_type;
	ValaDataType* unichar_type;
	ValaDataType* short_type;
	ValaDataType* ushort_type;
	ValaDataType* int_type;
	ValaDataType* uint_type;
	ValaDataType* long_type;
	ValaDataType* ulong_type;
	ValaDataType* int8_type;
	ValaDataType* uint8_type;
	ValaDataType* int16_type;
	ValaDataType* uint16_type;
	ValaDataType* int32_type;
	ValaDataType* uint32_type;
	ValaDataType* int64_type;
	ValaDataType* uint64_type;
	ValaDataType* string_type;
	ValaDataType* float_type;
	ValaDataType* double_type;
	ValaTypeSymbol* gtype_type;
	ValaTypeSymbol* gobject_type;
	ValaErrorType* gerror_type;
	ValaClass* glist_type;
	ValaClass* gslist_type;
	ValaTypeSymbol* gstringbuilder_type;
	ValaTypeSymbol* garray_type;
	ValaTypeSymbol* gbytearray_type;
	ValaTypeSymbol* gptrarray_type;
	ValaDataType* gquark_type;
	ValaStruct* gvalue_type;
	ValaStruct* mutex_type;
	ValaTypeSymbol* type_module_type;
	ValaTypeSymbol* dbus_object_type;
	gboolean in_plugin;
	char* module_init_param_name;
	gboolean gvaluecollector_h_needed;
	gboolean gio_h_needed;
	gboolean requires_array_free;
	gboolean requires_array_move;
	gboolean requires_array_length;
	gboolean requires_strcmp0;
	gboolean dbus_glib_h_needed;
	gboolean dbus_glib_h_needed_in_header;
	GeeSet* wrappers;
	GeeMap* variable_name_map;
};

struct _ValaCCodeBaseModuleClass {
	ValaCCodeModuleClass parent_class;
	void (*append_vala_array_free) (ValaCCodeBaseModule* self);
	void (*append_vala_array_move) (ValaCCodeBaseModule* self);
	void (*append_vala_array_length) (ValaCCodeBaseModule* self);
	void (*generate_class_struct_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_struct_declaration) (ValaCCodeBaseModule* self, ValaStruct* st, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_delegate_declaration) (ValaCCodeBaseModule* self, ValaDelegate* d, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_cparameters) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
	ValaCCodeExpression* (*get_dup_func_expression) (ValaCCodeBaseModule* self, ValaDataType* type, ValaSourceReference* source_reference);
	ValaCCodeExpression* (*get_unref_expression) (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr);
	char* (*get_delegate_target_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	ValaCCodeExpression* (*get_delegate_target_cexpression) (ValaCCodeBaseModule* self, ValaExpression* delegate_expr);
	char* (*get_delegate_target_destroy_notify_cname) (ValaCCodeBaseModule* self, const char* delegate_cname);
	ValaCCodeExpression* (*get_ref_cexpression) (ValaCCodeBaseModule* self, ValaDataType* expression_type, ValaCCodeExpression* cexpr, ValaExpression* expr, ValaCodeNode* node);
	void (*generate_class_declaration) (ValaCCodeBaseModule* self, ValaClass* cl, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_interface_declaration) (ValaCCodeBaseModule* self, ValaInterface* iface, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_method_declaration) (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
	void (*generate_error_domain_declaration) (ValaCCodeBaseModule* self, ValaErrorDomain* edomain, ValaCCodeDeclarationSpace* decl_space);
	ValaCCodeExpression* (*get_implicit_cast_expression) (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
};

struct _ValaCCodeStructModule {
	ValaCCodeBaseModule parent_instance;
	ValaCCodeStructModulePrivate * priv;
};

struct _ValaCCodeStructModuleClass {
	ValaCCodeBaseModuleClass parent_class;
};

/**
 * The link between a method and generated code.
 */
struct _ValaCCodeMethodModule {
	ValaCCodeStructModule parent_instance;
	ValaCCodeMethodModulePrivate * priv;
};

struct _ValaCCodeMethodModuleClass {
	ValaCCodeStructModuleClass parent_class;
	void (*generate_method_result_declaration) (ValaCCodeMethodModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, ValaCCodeFunction* cfunc, GeeMap* cparam_map, GeeMap* carg_map);
	void (*generate_parameter) (ValaCCodeMethodModule* self, ValaFormalParameter* param, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, GeeMap* carg_map);
};



gpointer vala_ccode_module_ref (gpointer instance);
void vala_ccode_module_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_module (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_module (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_module (const GValue* value);
GType vala_ccode_module_get_type (void);
GType vala_ccode_base_module_get_type (void);
gpointer vala_ccode_declaration_space_ref (gpointer instance);
void vala_ccode_declaration_space_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_declaration_space (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_declaration_space (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_declaration_space (const GValue* value);
GType vala_ccode_declaration_space_get_type (void);
GType vala_ccode_struct_module_get_type (void);
GType vala_ccode_method_module_get_type (void);
enum  {
	VALA_CCODE_METHOD_MODULE_DUMMY_PROPERTY
};
GType vala_ccode_generator_get_type (void);
ValaCCodeStructModule* vala_ccode_struct_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeStructModule* vala_ccode_struct_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeMethodModule* vala_ccode_method_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeMethodModule* vala_ccode_method_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next);
ValaCCodeMethodModule* vala_ccode_method_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next);
static gboolean vala_ccode_method_module_real_method_has_wrapper (ValaCCodeModule* base, ValaMethod* method);
static char* vala_ccode_method_module_real_get_custom_creturn_type (ValaCCodeModule* base, ValaMethod* m);
char* vala_ccode_module_get_custom_creturn_type (ValaCCodeModule* self, ValaMethod* m);
static char* vala_ccode_method_module_get_creturn_type (ValaCCodeMethodModule* self, ValaMethod* m, const char* default_value);
static gboolean vala_ccode_method_module_is_gtypeinstance_creation_method (ValaCCodeMethodModule* self, ValaMethod* m);
void vala_ccode_base_module_generate_type_declaration (ValaCCodeBaseModule* self, ValaDataType* type, ValaCCodeDeclarationSpace* decl_space);
ValaCCodeModule* vala_ccode_module_get_head (ValaCCodeModule* self);
char* vala_ccode_module_get_array_length_cname (ValaCCodeModule* self, const char* array_cname, gint dim);
gint vala_ccode_base_module_get_param_pos (ValaCCodeBaseModule* self, double param_pos, gboolean ellipsis);
ValaCCodeExpression* vala_ccode_base_module_get_variable_cexpression (ValaCCodeBaseModule* self, const char* name);
char* vala_ccode_base_module_get_delegate_target_cname (ValaCCodeBaseModule* self, const char* delegate_cname);
void vala_ccode_method_module_generate_method_result_declaration (ValaCCodeMethodModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, ValaCCodeFunction* cfunc, GeeMap* cparam_map, GeeMap* carg_map);
static void vala_ccode_method_module_real_generate_method_result_declaration (ValaCCodeMethodModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, ValaCCodeFunction* cfunc, GeeMap* cparam_map, GeeMap* carg_map);
ValaCCodeStatement* vala_ccode_method_module_complete_async (ValaCCodeMethodModule* self);
gboolean vala_ccode_declaration_space_add_symbol_declaration (ValaCCodeDeclarationSpace* self, ValaSymbol* sym, const char* name);
void vala_ccode_base_module_generate_cparameters (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
void vala_ccode_declaration_space_add_type_member_declaration (ValaCCodeDeclarationSpace* self, ValaCCodeNode* node);
static void vala_ccode_method_module_real_generate_method_declaration (ValaCCodeBaseModule* base, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
void vala_ccode_base_module_check_type (ValaCCodeBaseModule* self, ValaDataType* type);
ValaCCodeGenerator* vala_ccode_module_get_codegen (ValaCCodeModule* self);
static void vala_ccode_method_module_add_object_creation (ValaCCodeMethodModule* self, ValaCCodeBlock* b, gboolean has_params);
gboolean vala_ccode_base_module_requires_destroy (ValaCCodeBaseModule* self, ValaDataType* type);
ValaCCodeExpression* vala_ccode_base_module_get_unref_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* cvar, ValaDataType* type, ValaExpression* expr);
void vala_ccode_base_module_generate_method_declaration (ValaCCodeBaseModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space);
ValaCCodeExpression* vala_ccode_base_module_transform_expression (ValaCCodeBaseModule* self, ValaCCodeExpression* source_cexpr, ValaDataType* expression_type, ValaDataType* target_type, ValaExpression* expr);
static ValaCCodeStatement* vala_ccode_method_module_create_method_type_check_statement (ValaCCodeMethodModule* self, ValaMethod* m, ValaDataType* return_type, ValaTypeSymbol* t, gboolean non_null, const char* var_name);
char* vala_ccode_base_module_get_variable_cname (ValaCCodeBaseModule* self, const char* name);
ValaCCodeExpression* vala_ccode_module_get_construct_property_assignment (ValaCCodeModule* self, ValaCCodeConstant* canonical_cconstant, ValaDataType* property_type, ValaCCodeExpression* value);
void vala_ccode_declaration_space_add_include (ValaCCodeDeclarationSpace* self, const char* filename, gboolean local);
ValaCodeContext* vala_ccode_base_module_get_context (ValaCCodeBaseModule* self);
static ValaCCodeStatement* vala_ccode_method_module_create_precondition_statement (ValaCCodeMethodModule* self, ValaCodeNode* method_node, ValaDataType* ret_type, ValaExpression* precondition);
ValaCCodeExpression* vala_ccode_base_module_default_value_for_type (ValaCCodeBaseModule* self, ValaDataType* type, gboolean initializer_expression);
void vala_ccode_method_module_generate_vfunc (ValaCCodeMethodModule* self, ValaMethod* m, ValaDataType* return_type, GeeMap* cparam_map, GeeMap* carg_map, const char* suffix, gint direction);
static void vala_ccode_method_module_real_visit_method (ValaCCodeModule* base, ValaMethod* m);
void vala_ccode_method_module_generate_parameter (ValaCCodeMethodModule* self, ValaFormalParameter* param, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, GeeMap* carg_map);
static void vala_ccode_method_module_real_generate_parameter (ValaCCodeMethodModule* self, ValaFormalParameter* param, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, GeeMap* carg_map);
static ValaTypeSymbol* vala_ccode_method_module_find_parent_type (ValaCCodeMethodModule* self, ValaSymbol* sym);
static void vala_ccode_method_module_real_generate_cparameters (ValaCCodeBaseModule* base, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction);
static ValaCCodeStatement* vala_ccode_method_module_create_postcondition_statement (ValaCCodeMethodModule* self, ValaExpression* postcondition);
ValaCCodeStatement* vala_ccode_base_module_create_type_check_statement (ValaCCodeBaseModule* self, ValaCodeNode* method_node, ValaDataType* ret_type, ValaTypeSymbol* t, gboolean non_null, const char* var_name);
void vala_ccode_module_visit_method (ValaCCodeModule* self, ValaMethod* m);
static void vala_ccode_method_module_real_visit_creation_method (ValaCCodeModule* base, ValaCreationMethod* m);
static gpointer vala_ccode_method_module_parent_class = NULL;



ValaCCodeMethodModule* vala_ccode_method_module_construct (GType object_type, ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	ValaCCodeMethodModule* self;
	g_return_val_if_fail (codegen != NULL, NULL);
	self = (ValaCCodeMethodModule*) vala_ccode_struct_module_construct (object_type, codegen, next);
	return self;
}


ValaCCodeMethodModule* vala_ccode_method_module_new (ValaCCodeGenerator* codegen, ValaCCodeModule* next) {
	return vala_ccode_method_module_construct (VALA_TYPE_CCODE_METHOD_MODULE, codegen, next);
}


static gboolean vala_ccode_method_module_real_method_has_wrapper (ValaCCodeModule* base, ValaMethod* method) {
	ValaCCodeMethodModule * self;
	ValaAttribute* _tmp0_;
	gboolean _tmp1_;
	self = (ValaCCodeMethodModule*) base;
	g_return_val_if_fail (method != NULL, FALSE);
	_tmp0_ = NULL;
	return (_tmp1_ = (_tmp0_ = vala_code_node_get_attribute ((ValaCodeNode*) method, "NoWrapper")) == NULL, (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp1_);
}


static char* vala_ccode_method_module_real_get_custom_creturn_type (ValaCCodeModule* base, ValaMethod* m) {
	ValaCCodeMethodModule * self;
	ValaAttribute* attr;
	char* _tmp1_;
	self = (ValaCCodeMethodModule*) base;
	g_return_val_if_fail (m != NULL, NULL);
	attr = vala_code_node_get_attribute ((ValaCodeNode*) m, "CCode");
	if (attr != NULL) {
		char* type;
		type = vala_attribute_get_string (attr, "type");
		if (type != NULL) {
			char* _tmp0_;
			_tmp0_ = NULL;
			return (_tmp0_ = type, (attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL)), _tmp0_);
		}
		type = (g_free (type), NULL);
	}
	_tmp1_ = NULL;
	return (_tmp1_ = NULL, (attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL)), _tmp1_);
}


static char* vala_ccode_method_module_get_creturn_type (ValaCCodeMethodModule* self, ValaMethod* m, const char* default_value) {
	char* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (m != NULL, NULL);
	g_return_val_if_fail (default_value != NULL, NULL);
	type = vala_ccode_module_get_custom_creturn_type ((ValaCCodeModule*) self, m);
	if (type == NULL) {
		const char* _tmp0_;
		char* _tmp1_;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		return (_tmp1_ = (_tmp0_ = default_value, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), type = (g_free (type), NULL), _tmp1_);
	}
	return type;
}


static gboolean vala_ccode_method_module_is_gtypeinstance_creation_method (ValaCCodeMethodModule* self, ValaMethod* m) {
	gboolean _result_;
	ValaClass* _tmp1_;
	ValaSymbol* _tmp0_;
	ValaClass* cl;
	gboolean _tmp2_;
	gboolean _tmp3_;
	gboolean _tmp4_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (m != NULL, FALSE);
	_result_ = FALSE;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	cl = (_tmp1_ = (_tmp0_ = vala_symbol_get_parent_symbol ((ValaSymbol*) m), VALA_IS_CLASS (_tmp0_) ? ((ValaClass*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	if (VALA_IS_CREATION_METHOD (m)) {
		_tmp3_ = cl != NULL;
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		_tmp2_ = !vala_class_get_is_compact (cl);
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_result_ = TRUE;
	}
	return (_tmp4_ = _result_, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp4_);
}


static void vala_ccode_method_module_real_generate_method_result_declaration (ValaCCodeMethodModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, ValaCCodeFunction* cfunc, GeeMap* cparam_map, GeeMap* carg_map) {
	ValaDataType* _tmp0_;
	ValaDataType* creturn_type;
	char* _tmp5_;
	char* _tmp4_;
	gboolean _tmp6_;
	GeeList* _tmp16_;
	gboolean _tmp17_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (decl_space != NULL);
	g_return_if_fail (cfunc != NULL);
	g_return_if_fail (cparam_map != NULL);
	_tmp0_ = NULL;
	creturn_type = (_tmp0_ = vala_method_get_return_type (m), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaClass* _tmp2_;
		ValaSymbol* _tmp1_;
		ValaClass* cl;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		cl = (_tmp2_ = (_tmp1_ = vala_symbol_get_parent_symbol ((ValaSymbol*) m), VALA_IS_CLASS (_tmp1_) ? ((ValaClass*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
		if (cl != NULL) {
			ValaDataType* _tmp3_;
			/* object creation methods return the new object in C
			 in Vala they have no return type*/
			_tmp3_ = NULL;
			creturn_type = (_tmp3_ = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) cl), (creturn_type == NULL) ? NULL : (creturn_type = (vala_code_node_unref (creturn_type), NULL)), _tmp3_);
		}
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	}
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	vala_ccode_function_set_return_type (cfunc, _tmp5_ = vala_ccode_method_module_get_creturn_type (self, m, _tmp4_ = vala_data_type_get_cname (creturn_type)));
	_tmp5_ = (g_free (_tmp5_), NULL);
	_tmp4_ = (g_free (_tmp4_), NULL);
	vala_ccode_base_module_generate_type_declaration ((ValaCCodeBaseModule*) self, vala_method_get_return_type (m), decl_space);
	_tmp6_ = FALSE;
	if (!vala_method_get_no_array_length (m)) {
		_tmp6_ = VALA_IS_ARRAY_TYPE (vala_method_get_return_type (m));
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		ValaArrayType* _tmp7_;
		ValaArrayType* array_type;
		/* return array length if appropriate*/
		_tmp7_ = NULL;
		array_type = (_tmp7_ = VALA_ARRAY_TYPE (vala_method_get_return_type (m)), (_tmp7_ == NULL) ? NULL : vala_code_node_ref (_tmp7_));
		{
			gint dim;
			dim = 1;
			for (; dim <= vala_array_type_get_rank (array_type); dim++) {
				char* _tmp8_;
				ValaCCodeFormalParameter* _tmp9_;
				ValaCCodeFormalParameter* cparam;
				_tmp8_ = NULL;
				_tmp9_ = NULL;
				cparam = (_tmp9_ = vala_ccode_formal_parameter_new (_tmp8_ = vala_ccode_module_get_array_length_cname (vala_ccode_module_get_head ((ValaCCodeModule*) self), "result", dim), "int*"), _tmp8_ = (g_free (_tmp8_), NULL), _tmp9_);
				gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_carray_length_parameter_position (m) + (0.01 * dim), FALSE)), cparam);
				if (carg_map != NULL) {
					ValaCCodeExpression* _tmp10_;
					_tmp10_ = NULL;
					gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_carray_length_parameter_position (m) + (0.01 * dim), FALSE)), _tmp10_ = vala_ccode_base_module_get_variable_cexpression ((ValaCCodeBaseModule*) self, vala_ccode_formal_parameter_get_name (cparam)));
					(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL));
				}
				(cparam == NULL) ? NULL : (cparam = (vala_ccode_node_unref (cparam), NULL));
			}
		}
		(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
	} else {
		if (VALA_IS_DELEGATE_TYPE (vala_method_get_return_type (m))) {
			ValaDelegateType* _tmp11_;
			ValaDelegateType* deleg_type;
			ValaDelegate* _tmp12_;
			ValaDelegate* d;
			/* return delegate target if appropriate*/
			_tmp11_ = NULL;
			deleg_type = (_tmp11_ = VALA_DELEGATE_TYPE (vala_method_get_return_type (m)), (_tmp11_ == NULL) ? NULL : vala_code_node_ref (_tmp11_));
			_tmp12_ = NULL;
			d = (_tmp12_ = vala_delegate_type_get_delegate_symbol (deleg_type), (_tmp12_ == NULL) ? NULL : vala_code_node_ref (_tmp12_));
			if (vala_delegate_get_has_target (d)) {
				char* _tmp13_;
				ValaCCodeFormalParameter* _tmp14_;
				ValaCCodeFormalParameter* cparam;
				_tmp13_ = NULL;
				_tmp14_ = NULL;
				cparam = (_tmp14_ = vala_ccode_formal_parameter_new (_tmp13_ = vala_ccode_base_module_get_delegate_target_cname ((ValaCCodeBaseModule*) self, "result"), "void*"), _tmp13_ = (g_free (_tmp13_), NULL), _tmp14_);
				gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_cdelegate_target_parameter_position (m), FALSE)), cparam);
				if (carg_map != NULL) {
					ValaCCodeExpression* _tmp15_;
					_tmp15_ = NULL;
					gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_cdelegate_target_parameter_position (m), FALSE)), _tmp15_ = vala_ccode_base_module_get_variable_cexpression ((ValaCCodeBaseModule*) self, vala_ccode_formal_parameter_get_name (cparam)));
					(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
				}
				(cparam == NULL) ? NULL : (cparam = (vala_ccode_node_unref (cparam), NULL));
			}
			(deleg_type == NULL) ? NULL : (deleg_type = (vala_code_node_unref (deleg_type), NULL));
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
	}
	_tmp16_ = NULL;
	if ((_tmp17_ = gee_collection_get_size ((GeeCollection*) (_tmp16_ = vala_code_node_get_error_types ((ValaCodeNode*) m))) > 0, (_tmp16_ == NULL) ? NULL : (_tmp16_ = (gee_collection_object_unref (_tmp16_), NULL)), _tmp17_)) {
		ValaCCodeFormalParameter* cparam;
		{
			GeeList* _tmp18_;
			GeeIterator* _tmp19_;
			GeeIterator* _error_type_it;
			_tmp18_ = NULL;
			_tmp19_ = NULL;
			_error_type_it = (_tmp19_ = gee_iterable_iterator ((GeeIterable*) (_tmp18_ = vala_code_node_get_error_types ((ValaCodeNode*) m))), (_tmp18_ == NULL) ? NULL : (_tmp18_ = (gee_collection_object_unref (_tmp18_), NULL)), _tmp19_);
			while (gee_iterator_next (_error_type_it)) {
				ValaDataType* error_type;
				error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
				vala_ccode_base_module_generate_type_declaration ((ValaCCodeBaseModule*) self, error_type, decl_space);
				(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
			}
			(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
		}
		cparam = vala_ccode_formal_parameter_new ("error", "GError**");
		gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (double) (-1), FALSE)), cparam);
		if (carg_map != NULL) {
			ValaCCodeIdentifier* _tmp20_;
			_tmp20_ = NULL;
			gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (double) (-1), FALSE)), (ValaCCodeExpression*) (_tmp20_ = vala_ccode_identifier_new (vala_ccode_formal_parameter_get_name (cparam))));
			(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
		}
		(cparam == NULL) ? NULL : (cparam = (vala_ccode_node_unref (cparam), NULL));
	}
	(creturn_type == NULL) ? NULL : (creturn_type = (vala_code_node_unref (creturn_type), NULL));
}


void vala_ccode_method_module_generate_method_result_declaration (ValaCCodeMethodModule* self, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, ValaCCodeFunction* cfunc, GeeMap* cparam_map, GeeMap* carg_map) {
	VALA_CCODE_METHOD_MODULE_GET_CLASS (self)->generate_method_result_declaration (self, m, decl_space, cfunc, cparam_map, carg_map);
}


ValaCCodeStatement* vala_ccode_method_module_complete_async (ValaCCodeMethodModule* self) {
	ValaCCodeBlock* complete_block;
	ValaCCodeBlock* direct_block;
	ValaCCodeIdentifier* _tmp0_;
	ValaCCodeFunctionCall* _tmp1_;
	ValaCCodeFunctionCall* direct_call;
	ValaCCodeIdentifier* _tmp2_;
	ValaCCodeMemberAccess* _tmp3_;
	ValaCCodeMemberAccess* async_result_expr;
	ValaCCodeExpressionStatement* _tmp4_;
	ValaCCodeBlock* idle_block;
	ValaCCodeIdentifier* _tmp5_;
	ValaCCodeFunctionCall* _tmp6_;
	ValaCCodeFunctionCall* idle_call;
	ValaCCodeExpressionStatement* _tmp7_;
	ValaCCodeIdentifier* _tmp8_;
	ValaCCodeMemberAccess* _tmp9_;
	ValaCCodeMemberAccess* state;
	ValaCCodeConstant* zero;
	ValaCCodeBinaryExpression* state_is_zero;
	ValaCCodeIfStatement* dispatch;
	ValaCCodeIdentifier* _tmp10_;
	ValaCCodeFunctionCall* _tmp11_;
	ValaCCodeFunctionCall* unref;
	ValaCCodeExpressionStatement* _tmp12_;
	ValaCCodeReturnStatement* _tmp14_;
	ValaCCodeConstant* _tmp13_;
	ValaCCodeStatement* _tmp15_;
	g_return_val_if_fail (self != NULL, NULL);
	complete_block = vala_ccode_block_new ();
	direct_block = vala_ccode_block_new ();
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	direct_call = (_tmp1_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp0_ = vala_ccode_identifier_new ("g_simple_async_result_complete"))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL)), _tmp1_);
	_tmp2_ = NULL;
	_tmp3_ = NULL;
	async_result_expr = (_tmp3_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp2_ = vala_ccode_identifier_new ("data")), "_async_result"), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_ccode_node_unref (_tmp2_), NULL)), _tmp3_);
	vala_ccode_function_call_add_argument (direct_call, (ValaCCodeExpression*) async_result_expr);
	_tmp4_ = NULL;
	vala_ccode_block_add_statement (direct_block, (ValaCCodeNode*) (_tmp4_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) direct_call)));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	idle_block = vala_ccode_block_new ();
	_tmp5_ = NULL;
	_tmp6_ = NULL;
	idle_call = (_tmp6_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp5_ = vala_ccode_identifier_new ("g_simple_async_result_complete_in_idle"))), (_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL)), _tmp6_);
	vala_ccode_function_call_add_argument (idle_call, (ValaCCodeExpression*) async_result_expr);
	_tmp7_ = NULL;
	vala_ccode_block_add_statement (idle_block, (ValaCCodeNode*) (_tmp7_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) idle_call)));
	(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	_tmp8_ = NULL;
	_tmp9_ = NULL;
	state = (_tmp9_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp8_ = vala_ccode_identifier_new ("data")), "state"), (_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL)), _tmp9_);
	zero = vala_ccode_constant_new ("0");
	state_is_zero = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_EQUALITY, (ValaCCodeExpression*) state, (ValaCCodeExpression*) zero);
	dispatch = vala_ccode_if_statement_new ((ValaCCodeExpression*) state_is_zero, (ValaCCodeStatement*) idle_block, (ValaCCodeStatement*) direct_block);
	vala_ccode_block_add_statement (complete_block, (ValaCCodeNode*) dispatch);
	_tmp10_ = NULL;
	_tmp11_ = NULL;
	unref = (_tmp11_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new ("g_object_unref"))), (_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL)), _tmp11_);
	vala_ccode_function_call_add_argument (unref, (ValaCCodeExpression*) async_result_expr);
	_tmp12_ = NULL;
	vala_ccode_block_add_statement (complete_block, (ValaCCodeNode*) (_tmp12_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) unref)));
	(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
	_tmp14_ = NULL;
	_tmp13_ = NULL;
	vala_ccode_block_add_statement (complete_block, (ValaCCodeNode*) (_tmp14_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_constant_new ("FALSE")))));
	(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_ccode_node_unref (_tmp14_), NULL));
	(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
	_tmp15_ = NULL;
	return (_tmp15_ = (ValaCCodeStatement*) complete_block, (direct_block == NULL) ? NULL : (direct_block = (vala_ccode_node_unref (direct_block), NULL)), (direct_call == NULL) ? NULL : (direct_call = (vala_ccode_node_unref (direct_call), NULL)), (async_result_expr == NULL) ? NULL : (async_result_expr = (vala_ccode_node_unref (async_result_expr), NULL)), (idle_block == NULL) ? NULL : (idle_block = (vala_ccode_node_unref (idle_block), NULL)), (idle_call == NULL) ? NULL : (idle_call = (vala_ccode_node_unref (idle_call), NULL)), (state == NULL) ? NULL : (state = (vala_ccode_node_unref (state), NULL)), (zero == NULL) ? NULL : (zero = (vala_ccode_node_unref (zero), NULL)), (state_is_zero == NULL) ? NULL : (state_is_zero = (vala_ccode_node_unref (state_is_zero), NULL)), (dispatch == NULL) ? NULL : (dispatch = (vala_ccode_node_unref (dispatch), NULL)), (unref == NULL) ? NULL : (unref = (vala_ccode_node_unref (unref), NULL)), _tmp15_);
}


static void vala_ccode_method_module_real_generate_method_declaration (ValaCCodeBaseModule* base, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space) {
	ValaCCodeMethodModule * self;
	char* _tmp0_;
	gboolean _tmp1_;
	char* _tmp2_;
	ValaCCodeFunction* _tmp3_;
	ValaCCodeFunction* function;
	GeeHashMap* cparam_map;
	GeeHashMap* carg_map;
	ValaCCodeFunctionCall* _tmp5_;
	ValaCCodeIdentifier* _tmp4_;
	gboolean _tmp6_;
	self = (ValaCCodeMethodModule*) base;
	g_return_if_fail (m != NULL);
	g_return_if_fail (decl_space != NULL);
	_tmp0_ = NULL;
	if ((_tmp1_ = vala_ccode_declaration_space_add_symbol_declaration (decl_space, (ValaSymbol*) m, _tmp0_ = vala_method_get_cname (m)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_)) {
		return;
	}
	_tmp2_ = NULL;
	_tmp3_ = NULL;
	function = (_tmp3_ = vala_ccode_function_new (_tmp2_ = vala_method_get_cname (m), "void"), _tmp2_ = (g_free (_tmp2_), NULL), _tmp3_);
	if (vala_symbol_is_private_symbol ((ValaSymbol*) m)) {
		vala_ccode_function_set_modifiers (function, vala_ccode_function_get_modifiers (function) | VALA_CCODE_MODIFIERS_STATIC);
		if (vala_method_get_is_inline (m)) {
			vala_ccode_function_set_modifiers (function, vala_ccode_function_get_modifiers (function) | VALA_CCODE_MODIFIERS_INLINE);
		}
	}
	cparam_map = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal);
	carg_map = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal);
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	vala_ccode_base_module_generate_cparameters ((ValaCCodeBaseModule*) self, m, decl_space, (GeeMap*) cparam_map, function, NULL, (GeeMap*) carg_map, _tmp5_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp4_ = vala_ccode_identifier_new ("fake"))), 3);
	(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_ccode_node_unref (_tmp5_), NULL));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_ccode_node_unref (_tmp4_), NULL));
	vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) function);
	_tmp6_ = FALSE;
	if (VALA_IS_CREATION_METHOD (m)) {
		_tmp6_ = VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m));
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		ValaCCodeFunction* _tmp8_;
		char* _tmp7_;
		GeeHashMap* _tmp9_;
		/* _construct function*/
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		function = (_tmp8_ = vala_ccode_function_new (_tmp7_ = vala_method_get_real_cname (m), "void"), (function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL)), _tmp8_);
		_tmp7_ = (g_free (_tmp7_), NULL);
		if (vala_symbol_is_private_symbol ((ValaSymbol*) m)) {
			vala_ccode_function_set_modifiers (function, vala_ccode_function_get_modifiers (function) | VALA_CCODE_MODIFIERS_STATIC);
		}
		_tmp9_ = NULL;
		cparam_map = (_tmp9_ = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal), (cparam_map == NULL) ? NULL : (cparam_map = (gee_collection_object_unref (cparam_map), NULL)), _tmp9_);
		vala_ccode_base_module_generate_cparameters ((ValaCCodeBaseModule*) self, m, decl_space, (GeeMap*) cparam_map, function, NULL, NULL, NULL, 3);
		vala_ccode_declaration_space_add_type_member_declaration (decl_space, (ValaCCodeNode*) function);
	}
	(function == NULL) ? NULL : (function = (vala_ccode_node_unref (function), NULL));
	(cparam_map == NULL) ? NULL : (cparam_map = (gee_collection_object_unref (cparam_map), NULL));
	(carg_map == NULL) ? NULL : (carg_map = (gee_collection_object_unref (carg_map), NULL));
}


static void vala_ccode_method_module_real_visit_method (ValaCCodeModule* base, ValaMethod* m) {
	ValaCCodeMethodModule * self;
	ValaTypeSymbol* _tmp0_;
	ValaTypeSymbol* old_type_symbol;
	ValaSymbol* _tmp1_;
	ValaSymbol* old_symbol;
	ValaMethod* _tmp2_;
	ValaMethod* old_method;
	ValaDataType* _tmp3_;
	ValaDataType* old_return_type;
	gboolean old_method_inner_error;
	gboolean old_in_creation_method;
	gint old_next_temp_var_id;
	GeeArrayList* _tmp4_;
	GeeArrayList* old_temp_vars;
	GeeArrayList* _tmp5_;
	GeeArrayList* old_temp_ref_vars;
	GeeMap* _tmp6_;
	GeeMap* old_variable_name_map;
	ValaTryStatement* _tmp7_;
	ValaTryStatement* old_try;
	ValaSymbol* _tmp11_;
	ValaSymbol* _tmp10_;
	ValaMethod* _tmp13_;
	ValaMethod* _tmp12_;
	ValaDataType* _tmp15_;
	ValaDataType* _tmp14_;
	GeeArrayList* _tmp16_;
	GeeArrayList* _tmp17_;
	GeeMap* _tmp18_;
	ValaTryStatement* _tmp19_;
	gboolean in_gobject_creation_method;
	gboolean in_fundamental_creation_method;
	ValaDataType* _tmp23_;
	ValaDataType* creturn_type;
	gboolean inner_error;
	ValaTypeSymbol* _tmp52_;
	ValaTypeSymbol* _tmp51_;
	ValaSymbol* _tmp54_;
	ValaSymbol* _tmp53_;
	ValaMethod* _tmp56_;
	ValaMethod* _tmp55_;
	ValaDataType* _tmp58_;
	ValaDataType* _tmp57_;
	GeeArrayList* _tmp60_;
	GeeArrayList* _tmp59_;
	GeeArrayList* _tmp62_;
	GeeArrayList* _tmp61_;
	GeeMap* _tmp64_;
	GeeMap* _tmp63_;
	ValaTryStatement* _tmp66_;
	ValaTryStatement* _tmp65_;
	gboolean _tmp67_;
	gboolean _tmp68_;
	ValaCCodeFunction* _tmp71_;
	char* _tmp70_;
	GeeHashMap* cparam_map;
	gboolean _tmp72_;
	gboolean _tmp243_;
	gboolean _tmp244_;
	self = (ValaCCodeMethodModule*) base;
	g_return_if_fail (m != NULL);
	_tmp0_ = NULL;
	old_type_symbol = (_tmp0_ = ((ValaCCodeBaseModule*) self)->current_type_symbol, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	_tmp1_ = NULL;
	old_symbol = (_tmp1_ = ((ValaCCodeBaseModule*) self)->current_symbol, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = NULL;
	old_method = (_tmp2_ = ((ValaCCodeBaseModule*) self)->current_method, (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	_tmp3_ = NULL;
	old_return_type = (_tmp3_ = ((ValaCCodeBaseModule*) self)->current_return_type, (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
	old_method_inner_error = ((ValaCCodeBaseModule*) self)->current_method_inner_error;
	old_in_creation_method = ((ValaCCodeBaseModule*) self)->in_creation_method;
	old_next_temp_var_id = ((ValaCCodeBaseModule*) self)->next_temp_var_id;
	_tmp4_ = NULL;
	old_temp_vars = (_tmp4_ = ((ValaCCodeBaseModule*) self)->temp_vars, (_tmp4_ == NULL) ? NULL : gee_collection_object_ref (_tmp4_));
	_tmp5_ = NULL;
	old_temp_ref_vars = (_tmp5_ = ((ValaCCodeBaseModule*) self)->temp_ref_vars, (_tmp5_ == NULL) ? NULL : gee_collection_object_ref (_tmp5_));
	_tmp6_ = NULL;
	old_variable_name_map = (_tmp6_ = ((ValaCCodeBaseModule*) self)->variable_name_map, (_tmp6_ == NULL) ? NULL : gee_collection_object_ref (_tmp6_));
	_tmp7_ = NULL;
	old_try = (_tmp7_ = ((ValaCCodeBaseModule*) self)->current_try, (_tmp7_ == NULL) ? NULL : vala_code_node_ref (_tmp7_));
	if (VALA_IS_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) m))) {
		ValaTypeSymbol* _tmp9_;
		ValaTypeSymbol* _tmp8_;
		_tmp9_ = NULL;
		_tmp8_ = NULL;
		((ValaCCodeBaseModule*) self)->current_type_symbol = (_tmp9_ = (_tmp8_ = VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp8_ == NULL) ? NULL : vala_code_node_ref (_tmp8_)), (((ValaCCodeBaseModule*) self)->current_type_symbol == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_type_symbol = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_type_symbol), NULL)), _tmp9_);
	}
	_tmp11_ = NULL;
	_tmp10_ = NULL;
	((ValaCCodeBaseModule*) self)->current_symbol = (_tmp11_ = (_tmp10_ = (ValaSymbol*) m, (_tmp10_ == NULL) ? NULL : vala_code_node_ref (_tmp10_)), (((ValaCCodeBaseModule*) self)->current_symbol == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_symbol = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_symbol), NULL)), _tmp11_);
	_tmp13_ = NULL;
	_tmp12_ = NULL;
	((ValaCCodeBaseModule*) self)->current_method = (_tmp13_ = (_tmp12_ = m, (_tmp12_ == NULL) ? NULL : vala_code_node_ref (_tmp12_)), (((ValaCCodeBaseModule*) self)->current_method == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_method = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_method), NULL)), _tmp13_);
	_tmp15_ = NULL;
	_tmp14_ = NULL;
	((ValaCCodeBaseModule*) self)->current_return_type = (_tmp15_ = (_tmp14_ = vala_method_get_return_type (m), (_tmp14_ == NULL) ? NULL : vala_code_node_ref (_tmp14_)), (((ValaCCodeBaseModule*) self)->current_return_type == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_return_type = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_return_type), NULL)), _tmp15_);
	((ValaCCodeBaseModule*) self)->current_method_inner_error = FALSE;
	((ValaCCodeBaseModule*) self)->next_temp_var_id = 0;
	_tmp16_ = NULL;
	((ValaCCodeBaseModule*) self)->temp_vars = (_tmp16_ = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (((ValaCCodeBaseModule*) self)->temp_vars == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->temp_vars = (gee_collection_object_unref (((ValaCCodeBaseModule*) self)->temp_vars), NULL)), _tmp16_);
	_tmp17_ = NULL;
	((ValaCCodeBaseModule*) self)->temp_ref_vars = (_tmp17_ = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (((ValaCCodeBaseModule*) self)->temp_ref_vars == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->temp_ref_vars = (gee_collection_object_unref (((ValaCCodeBaseModule*) self)->temp_ref_vars), NULL)), _tmp17_);
	_tmp18_ = NULL;
	((ValaCCodeBaseModule*) self)->variable_name_map = (_tmp18_ = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal), (((ValaCCodeBaseModule*) self)->variable_name_map == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->variable_name_map = (gee_collection_object_unref (((ValaCCodeBaseModule*) self)->variable_name_map), NULL)), _tmp18_);
	_tmp19_ = NULL;
	((ValaCCodeBaseModule*) self)->current_try = (_tmp19_ = NULL, (((ValaCCodeBaseModule*) self)->current_try == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_try = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_try), NULL)), _tmp19_);
	in_gobject_creation_method = FALSE;
	in_fundamental_creation_method = FALSE;
	vala_ccode_base_module_check_type ((ValaCCodeBaseModule*) self, vala_method_get_return_type (m));
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaClass* _tmp21_;
		ValaTypeSymbol* _tmp20_;
		ValaClass* cl;
		gboolean _tmp22_;
		((ValaCCodeBaseModule*) self)->in_creation_method = TRUE;
		_tmp21_ = NULL;
		_tmp20_ = NULL;
		cl = (_tmp21_ = (_tmp20_ = ((ValaCCodeBaseModule*) self)->current_type_symbol, VALA_IS_CLASS (_tmp20_) ? ((ValaClass*) _tmp20_) : NULL), (_tmp21_ == NULL) ? NULL : vala_code_node_ref (_tmp21_));
		_tmp22_ = FALSE;
		if (cl != NULL) {
			_tmp22_ = !vala_class_get_is_compact (cl);
		} else {
			_tmp22_ = FALSE;
		}
		if (_tmp22_) {
			if (vala_class_get_base_class (cl) == NULL) {
				in_fundamental_creation_method = TRUE;
			} else {
				if (vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) cl, ((ValaCCodeBaseModule*) self)->gobject_type)) {
					in_gobject_creation_method = TRUE;
				}
			}
		}
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	} else {
		((ValaCCodeBaseModule*) self)->in_creation_method = FALSE;
	}
	_tmp23_ = NULL;
	creturn_type = (_tmp23_ = ((ValaCCodeBaseModule*) self)->current_return_type, (_tmp23_ == NULL) ? NULL : vala_code_node_ref (_tmp23_));
	vala_code_node_accept_children ((ValaCodeNode*) m, (ValaCodeVisitor*) vala_ccode_module_get_codegen ((ValaCCodeModule*) self));
	if (VALA_IS_CREATION_METHOD (m)) {
		gboolean _tmp24_;
		_tmp24_ = FALSE;
		if (in_gobject_creation_method) {
			_tmp24_ = vala_method_get_body (m) != NULL;
		} else {
			_tmp24_ = FALSE;
		}
		if (_tmp24_) {
			ValaCCodeBlock* cblock;
			cblock = vala_ccode_block_new ();
			if (!vala_creation_method_get_chain_up (VALA_CREATION_METHOD (m))) {
				gboolean _tmp32_;
				/* set construct properties*/
				{
					GeeList* _tmp25_;
					GeeIterator* _tmp26_;
					GeeIterator* _stmt_it;
					/* set construct properties*/
					_tmp25_ = NULL;
					_tmp26_ = NULL;
					_stmt_it = (_tmp26_ = gee_iterable_iterator ((GeeIterable*) (_tmp25_ = vala_block_get_statements (vala_method_get_body (m)))), (_tmp25_ == NULL) ? NULL : (_tmp25_ = (gee_collection_object_unref (_tmp25_), NULL)), _tmp26_);
					/* set construct properties*/
					while (gee_iterator_next (_stmt_it)) {
						ValaCodeNode* stmt;
						ValaExpressionStatement* _tmp28_;
						ValaCodeNode* _tmp27_;
						ValaExpressionStatement* expr_stmt;
						/* set construct properties*/
						stmt = (ValaCodeNode*) ((ValaStatement*) gee_iterator_get (_stmt_it));
						_tmp28_ = NULL;
						_tmp27_ = NULL;
						expr_stmt = (_tmp28_ = (_tmp27_ = stmt, VALA_IS_EXPRESSION_STATEMENT (_tmp27_) ? ((ValaExpressionStatement*) _tmp27_) : NULL), (_tmp28_ == NULL) ? NULL : vala_code_node_ref (_tmp28_));
						if (expr_stmt != NULL) {
							ValaProperty* prop;
							gboolean _tmp29_;
							prop = vala_expression_statement_assigned_property (expr_stmt);
							_tmp29_ = FALSE;
							if (prop != NULL) {
								_tmp29_ = vala_property_accessor_get_construction (vala_property_get_set_accessor (prop));
							} else {
								_tmp29_ = FALSE;
							}
							if (_tmp29_) {
								if (VALA_IS_CCODE_FRAGMENT (vala_code_node_get_ccodenode (stmt))) {
									{
										GeeList* _tmp30_;
										GeeIterator* _tmp31_;
										GeeIterator* _cstmt_it;
										_tmp30_ = NULL;
										_tmp31_ = NULL;
										_cstmt_it = (_tmp31_ = gee_iterable_iterator ((GeeIterable*) (_tmp30_ = vala_ccode_fragment_get_children (VALA_CCODE_FRAGMENT (vala_code_node_get_ccodenode (stmt))))), (_tmp30_ == NULL) ? NULL : (_tmp30_ = (gee_collection_object_unref (_tmp30_), NULL)), _tmp31_);
										while (gee_iterator_next (_cstmt_it)) {
											ValaCCodeNode* cstmt;
											cstmt = (ValaCCodeNode*) gee_iterator_get (_cstmt_it);
											vala_ccode_block_add_statement (cblock, cstmt);
											(cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL));
										}
										(_cstmt_it == NULL) ? NULL : (_cstmt_it = (gee_collection_object_unref (_cstmt_it), NULL));
									}
								} else {
									vala_ccode_block_add_statement (cblock, vala_code_node_get_ccodenode (stmt));
								}
							}
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						}
						(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
						(expr_stmt == NULL) ? NULL : (expr_stmt = (vala_code_node_unref (expr_stmt), NULL));
					}
					(_stmt_it == NULL) ? NULL : (_stmt_it = (gee_collection_object_unref (_stmt_it), NULL));
				}
				_tmp32_ = FALSE;
				if (vala_creation_method_get_n_construction_params (VALA_CREATION_METHOD (m)) > 0) {
					_tmp32_ = TRUE;
				} else {
					GeeList* _tmp33_;
					_tmp33_ = NULL;
					_tmp32_ = gee_collection_get_size ((GeeCollection*) (_tmp33_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class))) > 0;
					(_tmp33_ == NULL) ? NULL : (_tmp33_ = (gee_collection_object_unref (_tmp33_), NULL));
				}
				vala_ccode_method_module_add_object_creation (self, cblock, _tmp32_);
			} else {
				char* _tmp35_;
				char* _tmp34_;
				ValaCCodeDeclaration* _tmp36_;
				ValaCCodeDeclaration* cdeclaration;
				ValaCCodeVariableDeclarator* _tmp37_;
				_tmp35_ = NULL;
				_tmp34_ = NULL;
				_tmp36_ = NULL;
				cdeclaration = (_tmp36_ = vala_ccode_declaration_new (_tmp35_ = g_strdup_printf ("%s *", _tmp34_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) VALA_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol), FALSE))), _tmp35_ = (g_free (_tmp35_), NULL), _tmp34_ = (g_free (_tmp34_), NULL), _tmp36_);
				_tmp37_ = NULL;
				vala_ccode_declaration_add_declarator (cdeclaration, (ValaCCodeDeclarator*) (_tmp37_ = vala_ccode_variable_declarator_new ("self", NULL, NULL)));
				(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
				vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) cdeclaration);
				(cdeclaration == NULL) ? NULL : (cdeclaration = (vala_ccode_node_unref (cdeclaration), NULL));
			}
			/* other initialization code*/
			{
				GeeList* _tmp38_;
				GeeIterator* _tmp39_;
				GeeIterator* _stmt_it;
				/* other initialization code*/
				_tmp38_ = NULL;
				_tmp39_ = NULL;
				_stmt_it = (_tmp39_ = gee_iterable_iterator ((GeeIterable*) (_tmp38_ = vala_block_get_statements (vala_method_get_body (m)))), (_tmp38_ == NULL) ? NULL : (_tmp38_ = (gee_collection_object_unref (_tmp38_), NULL)), _tmp39_);
				/* other initialization code*/
				while (gee_iterator_next (_stmt_it)) {
					ValaCodeNode* stmt;
					ValaExpressionStatement* _tmp41_;
					ValaCodeNode* _tmp40_;
					ValaExpressionStatement* expr_stmt;
					/* other initialization code*/
					stmt = (ValaCodeNode*) ((ValaStatement*) gee_iterator_get (_stmt_it));
					_tmp41_ = NULL;
					_tmp40_ = NULL;
					expr_stmt = (_tmp41_ = (_tmp40_ = stmt, VALA_IS_EXPRESSION_STATEMENT (_tmp40_) ? ((ValaExpressionStatement*) _tmp40_) : NULL), (_tmp41_ == NULL) ? NULL : vala_code_node_ref (_tmp41_));
					if (expr_stmt != NULL) {
						ValaProperty* prop;
						gboolean _tmp42_;
						prop = vala_expression_statement_assigned_property (expr_stmt);
						_tmp42_ = FALSE;
						if (prop != NULL) {
							_tmp42_ = vala_property_accessor_get_construction (vala_property_get_set_accessor (prop));
						} else {
							_tmp42_ = FALSE;
						}
						if (_tmp42_) {
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
							(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
							(expr_stmt == NULL) ? NULL : (expr_stmt = (vala_code_node_unref (expr_stmt), NULL));
							continue;
						}
						(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
					}
					if (VALA_IS_CCODE_FRAGMENT (vala_code_node_get_ccodenode (stmt))) {
						{
							GeeList* _tmp43_;
							GeeIterator* _tmp44_;
							GeeIterator* _cstmt_it;
							_tmp43_ = NULL;
							_tmp44_ = NULL;
							_cstmt_it = (_tmp44_ = gee_iterable_iterator ((GeeIterable*) (_tmp43_ = vala_ccode_fragment_get_children (VALA_CCODE_FRAGMENT (vala_code_node_get_ccodenode (stmt))))), (_tmp43_ == NULL) ? NULL : (_tmp43_ = (gee_collection_object_unref (_tmp43_), NULL)), _tmp44_);
							while (gee_iterator_next (_cstmt_it)) {
								ValaCCodeNode* cstmt;
								cstmt = (ValaCCodeNode*) gee_iterator_get (_cstmt_it);
								vala_ccode_block_add_statement (cblock, cstmt);
								(cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL));
							}
							(_cstmt_it == NULL) ? NULL : (_cstmt_it = (gee_collection_object_unref (_cstmt_it), NULL));
						}
					} else {
						vala_ccode_block_add_statement (cblock, vala_code_node_get_ccodenode (stmt));
					}
					(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
					(expr_stmt == NULL) ? NULL : (expr_stmt = (vala_code_node_unref (expr_stmt), NULL));
				}
				(_stmt_it == NULL) ? NULL : (_stmt_it = (gee_collection_object_unref (_stmt_it), NULL));
			}
			{
				GeeList* _tmp45_;
				GeeIterator* _tmp46_;
				GeeIterator* _local_it;
				_tmp45_ = NULL;
				_tmp46_ = NULL;
				_local_it = (_tmp46_ = gee_iterable_iterator ((GeeIterable*) (_tmp45_ = vala_block_get_local_variables (vala_method_get_body (m)))), (_tmp45_ == NULL) ? NULL : (_tmp45_ = (gee_collection_object_unref (_tmp45_), NULL)), _tmp46_);
				while (gee_iterator_next (_local_it)) {
					ValaLocalVariable* local;
					gboolean _tmp47_;
					local = (ValaLocalVariable*) gee_iterator_get (_local_it);
					_tmp47_ = FALSE;
					if (!vala_local_variable_get_floating (local)) {
						_tmp47_ = vala_ccode_base_module_requires_destroy ((ValaCCodeBaseModule*) self, vala_local_variable_get_variable_type (local));
					} else {
						_tmp47_ = FALSE;
					}
					if (_tmp47_) {
						ValaMemberAccess* ma;
						ValaCCodeExpressionStatement* _tmp50_;
						ValaCCodeExpression* _tmp49_;
						ValaCCodeExpression* _tmp48_;
						ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), NULL);
						vala_expression_set_symbol_reference ((ValaExpression*) ma, (ValaSymbol*) local);
						_tmp50_ = NULL;
						_tmp49_ = NULL;
						_tmp48_ = NULL;
						vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp50_ = vala_ccode_expression_statement_new (_tmp49_ = vala_ccode_base_module_get_unref_expression ((ValaCCodeBaseModule*) self, _tmp48_ = vala_ccode_base_module_get_variable_cexpression ((ValaCCodeBaseModule*) self, vala_symbol_get_name ((ValaSymbol*) local)), vala_local_variable_get_variable_type (local), (ValaExpression*) ma))));
						(_tmp50_ == NULL) ? NULL : (_tmp50_ = (vala_ccode_node_unref (_tmp50_), NULL));
						(_tmp49_ == NULL) ? NULL : (_tmp49_ = (vala_ccode_node_unref (_tmp49_), NULL));
						(_tmp48_ == NULL) ? NULL : (_tmp48_ = (vala_ccode_node_unref (_tmp48_), NULL));
						(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
					}
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				}
				(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
			}
			vala_code_node_set_ccodenode ((ValaCodeNode*) vala_method_get_body (m), (ValaCCodeNode*) cblock);
			(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
		}
		((ValaCCodeBaseModule*) self)->in_creation_method = old_in_creation_method;
	}
	inner_error = ((ValaCCodeBaseModule*) self)->current_method_inner_error;
	_tmp52_ = NULL;
	_tmp51_ = NULL;
	((ValaCCodeBaseModule*) self)->current_type_symbol = (_tmp52_ = (_tmp51_ = old_type_symbol, (_tmp51_ == NULL) ? NULL : vala_code_node_ref (_tmp51_)), (((ValaCCodeBaseModule*) self)->current_type_symbol == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_type_symbol = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_type_symbol), NULL)), _tmp52_);
	_tmp54_ = NULL;
	_tmp53_ = NULL;
	((ValaCCodeBaseModule*) self)->current_symbol = (_tmp54_ = (_tmp53_ = old_symbol, (_tmp53_ == NULL) ? NULL : vala_code_node_ref (_tmp53_)), (((ValaCCodeBaseModule*) self)->current_symbol == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_symbol = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_symbol), NULL)), _tmp54_);
	_tmp56_ = NULL;
	_tmp55_ = NULL;
	((ValaCCodeBaseModule*) self)->current_method = (_tmp56_ = (_tmp55_ = old_method, (_tmp55_ == NULL) ? NULL : vala_code_node_ref (_tmp55_)), (((ValaCCodeBaseModule*) self)->current_method == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_method = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_method), NULL)), _tmp56_);
	_tmp58_ = NULL;
	_tmp57_ = NULL;
	((ValaCCodeBaseModule*) self)->current_return_type = (_tmp58_ = (_tmp57_ = old_return_type, (_tmp57_ == NULL) ? NULL : vala_code_node_ref (_tmp57_)), (((ValaCCodeBaseModule*) self)->current_return_type == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_return_type = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_return_type), NULL)), _tmp58_);
	((ValaCCodeBaseModule*) self)->current_method_inner_error = old_method_inner_error;
	((ValaCCodeBaseModule*) self)->next_temp_var_id = old_next_temp_var_id;
	_tmp60_ = NULL;
	_tmp59_ = NULL;
	((ValaCCodeBaseModule*) self)->temp_vars = (_tmp60_ = (_tmp59_ = old_temp_vars, (_tmp59_ == NULL) ? NULL : gee_collection_object_ref (_tmp59_)), (((ValaCCodeBaseModule*) self)->temp_vars == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->temp_vars = (gee_collection_object_unref (((ValaCCodeBaseModule*) self)->temp_vars), NULL)), _tmp60_);
	_tmp62_ = NULL;
	_tmp61_ = NULL;
	((ValaCCodeBaseModule*) self)->temp_ref_vars = (_tmp62_ = (_tmp61_ = old_temp_ref_vars, (_tmp61_ == NULL) ? NULL : gee_collection_object_ref (_tmp61_)), (((ValaCCodeBaseModule*) self)->temp_ref_vars == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->temp_ref_vars = (gee_collection_object_unref (((ValaCCodeBaseModule*) self)->temp_ref_vars), NULL)), _tmp62_);
	_tmp64_ = NULL;
	_tmp63_ = NULL;
	((ValaCCodeBaseModule*) self)->variable_name_map = (_tmp64_ = (_tmp63_ = old_variable_name_map, (_tmp63_ == NULL) ? NULL : gee_collection_object_ref (_tmp63_)), (((ValaCCodeBaseModule*) self)->variable_name_map == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->variable_name_map = (gee_collection_object_unref (((ValaCCodeBaseModule*) self)->variable_name_map), NULL)), _tmp64_);
	_tmp66_ = NULL;
	_tmp65_ = NULL;
	((ValaCCodeBaseModule*) self)->current_try = (_tmp66_ = (_tmp65_ = old_try, (_tmp65_ == NULL) ? NULL : vala_code_node_ref (_tmp65_)), (((ValaCCodeBaseModule*) self)->current_try == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->current_try = (vala_code_node_unref (((ValaCCodeBaseModule*) self)->current_try), NULL)), _tmp66_);
	_tmp67_ = FALSE;
	_tmp68_ = FALSE;
	if (vala_method_get_is_abstract (m)) {
		_tmp68_ = TRUE;
	} else {
		_tmp68_ = vala_method_get_is_virtual (m);
	}
	if (_tmp68_) {
		_tmp67_ = TRUE;
	} else {
		gboolean _tmp69_;
		_tmp69_ = FALSE;
		if (vala_method_get_base_method (m) == NULL) {
			_tmp69_ = vala_method_get_base_interface_method (m) == NULL;
		} else {
			_tmp69_ = FALSE;
		}
		_tmp67_ = _tmp69_;
	}
	/* do not declare overriding methods and interface implementations*/
	if (_tmp67_) {
		vala_ccode_base_module_generate_method_declaration ((ValaCCodeBaseModule*) self, m, ((ValaCCodeBaseModule*) self)->source_declarations);
		if (!vala_symbol_is_internal_symbol ((ValaSymbol*) m)) {
			vala_ccode_base_module_generate_method_declaration ((ValaCCodeBaseModule*) self, m, ((ValaCCodeBaseModule*) self)->header_declarations);
		}
		vala_ccode_base_module_generate_method_declaration ((ValaCCodeBaseModule*) self, m, ((ValaCCodeBaseModule*) self)->internal_header_declarations);
	}
	_tmp71_ = NULL;
	_tmp70_ = NULL;
	((ValaCCodeBaseModule*) self)->function = (_tmp71_ = vala_ccode_function_new (_tmp70_ = vala_method_get_real_cname (m), "void"), (((ValaCCodeBaseModule*) self)->function == NULL) ? NULL : (((ValaCCodeBaseModule*) self)->function = (vala_ccode_node_unref (((ValaCCodeBaseModule*) self)->function), NULL)), _tmp71_);
	_tmp70_ = (g_free (_tmp70_), NULL);
	vala_code_node_set_ccodenode ((ValaCodeNode*) m, (ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function);
	if (vala_method_get_is_inline (m)) {
		vala_ccode_function_set_modifiers (((ValaCCodeBaseModule*) self)->function, vala_ccode_function_get_modifiers (((ValaCCodeBaseModule*) self)->function) | VALA_CCODE_MODIFIERS_INLINE);
	}
	cparam_map = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal);
	vala_ccode_base_module_generate_cparameters ((ValaCCodeBaseModule*) self, m, ((ValaCCodeBaseModule*) self)->source_declarations, (GeeMap*) cparam_map, ((ValaCCodeBaseModule*) self)->function, NULL, NULL, NULL, 3);
	_tmp72_ = FALSE;
	if (!vala_method_get_is_abstract (m)) {
		_tmp72_ = TRUE;
	} else {
		gboolean _tmp73_;
		_tmp73_ = FALSE;
		if (vala_method_get_is_abstract (m)) {
			_tmp73_ = VALA_IS_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol);
		} else {
			_tmp73_ = FALSE;
		}
		_tmp72_ = _tmp73_;
	}
	/* generate *_real_* functions for virtual methods
	 also generate them for abstract methods of classes to prevent faulty subclassing*/
	if (_tmp72_) {
		gboolean _tmp74_;
		_tmp74_ = FALSE;
		if (vala_method_get_base_method (m) != NULL) {
			_tmp74_ = TRUE;
		} else {
			_tmp74_ = vala_method_get_base_interface_method (m) != NULL;
		}
		if (_tmp74_) {
			ValaCCodeFunction* _tmp75_;
			/* declare *_real_* function*/
			vala_ccode_function_set_modifiers (((ValaCCodeBaseModule*) self)->function, vala_ccode_function_get_modifiers (((ValaCCodeBaseModule*) self)->function) | VALA_CCODE_MODIFIERS_STATIC);
			_tmp75_ = NULL;
			vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp75_ = vala_ccode_function_copy (((ValaCCodeBaseModule*) self)->function)));
			(_tmp75_ == NULL) ? NULL : (_tmp75_ = (vala_ccode_node_unref (_tmp75_), NULL));
		} else {
			if (vala_symbol_is_private_symbol ((ValaSymbol*) m)) {
				vala_ccode_function_set_modifiers (((ValaCCodeBaseModule*) self)->function, vala_ccode_function_get_modifiers (((ValaCCodeBaseModule*) self)->function) | VALA_CCODE_MODIFIERS_STATIC);
			}
		}
		/* Methods imported from a plain C file don't
		 * have a body, e.g. Vala.Parser.parse_file () */
		if (vala_method_get_body (m) != NULL) {
			ValaCCodeFragment* cinit;
			gboolean _tmp95_;
			gboolean _tmp227_;
			vala_ccode_function_set_block (((ValaCCodeBaseModule*) self)->function, VALA_CCODE_BLOCK (vala_code_node_get_ccodenode ((ValaCodeNode*) vala_method_get_body (m))));
			vala_ccode_node_set_line ((ValaCCodeNode*) vala_ccode_function_get_block (((ValaCCodeBaseModule*) self)->function), vala_ccode_node_get_line ((ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function));
			cinit = vala_ccode_fragment_new ();
			vala_ccode_block_prepend_statement (vala_ccode_function_get_block (((ValaCCodeBaseModule*) self)->function), (ValaCCodeNode*) cinit);
			if (vala_method_get_coroutine (m)) {
				char* _tmp77_;
				char* _tmp76_;
				ValaCCodeFunction* _tmp78_;
				ValaCCodeFunction* co_function;
				ValaCCodeFormalParameter* _tmp82_;
				char* _tmp81_;
				char* _tmp80_;
				char* _tmp79_;
				ValaCCodeFunction* _tmp83_;
				ValaCCodeMemberAccess* _tmp85_;
				ValaCCodeIdentifier* _tmp84_;
				ValaCCodeSwitchStatement* _tmp86_;
				ValaCCodeSwitchStatement* cswitch;
				ValaCCodeLabel* _tmp87_;
				ValaCCodeExpressionStatement* _tmp90_;
				ValaCCodeFunctionCall* _tmp89_;
				ValaCCodeIdentifier* _tmp88_;
				ValaCCodeCaseStatement* _tmp92_;
				ValaCCodeConstant* _tmp91_;
				ValaCCodeStatement* _tmp93_;
				ValaCCodeBlock* _tmp94_;
				_tmp77_ = NULL;
				_tmp76_ = NULL;
				_tmp78_ = NULL;
				co_function = (_tmp78_ = vala_ccode_function_new (_tmp77_ = g_strconcat (_tmp76_ = vala_method_get_real_cname (m), "_co", NULL), "gboolean"), _tmp77_ = (g_free (_tmp77_), NULL), _tmp76_ = (g_free (_tmp76_), NULL), _tmp78_);
				/* data struct to hold parameters, local variables, and the return value*/
				_tmp82_ = NULL;
				_tmp81_ = NULL;
				_tmp80_ = NULL;
				_tmp79_ = NULL;
				vala_ccode_function_add_parameter (co_function, _tmp82_ = vala_ccode_formal_parameter_new ("data", _tmp81_ = g_strconcat (_tmp80_ = vala_symbol_lower_case_to_camel_case (_tmp79_ = vala_method_get_cname (m)), "Data*", NULL)));
				(_tmp82_ == NULL) ? NULL : (_tmp82_ = (vala_ccode_node_unref (_tmp82_), NULL));
				_tmp81_ = (g_free (_tmp81_), NULL);
				_tmp80_ = (g_free (_tmp80_), NULL);
				_tmp79_ = (g_free (_tmp79_), NULL);
				vala_ccode_function_set_modifiers (co_function, vala_ccode_function_get_modifiers (co_function) | VALA_CCODE_MODIFIERS_STATIC);
				_tmp83_ = NULL;
				vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp83_ = vala_ccode_function_copy (co_function)));
				(_tmp83_ == NULL) ? NULL : (_tmp83_ = (vala_ccode_node_unref (_tmp83_), NULL));
				_tmp85_ = NULL;
				_tmp84_ = NULL;
				_tmp86_ = NULL;
				cswitch = (_tmp86_ = vala_ccode_switch_statement_new ((ValaCCodeExpression*) (_tmp85_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp84_ = vala_ccode_identifier_new ("data")), "state"))), (_tmp85_ == NULL) ? NULL : (_tmp85_ = (vala_ccode_node_unref (_tmp85_), NULL)), (_tmp84_ == NULL) ? NULL : (_tmp84_ = (vala_ccode_node_unref (_tmp84_), NULL)), _tmp86_);
				/* let gcc know that this can't happen*/
				_tmp87_ = NULL;
				vala_ccode_block_add_statement ((ValaCCodeBlock*) cswitch, (ValaCCodeNode*) (_tmp87_ = vala_ccode_label_new ("default")));
				(_tmp87_ == NULL) ? NULL : (_tmp87_ = (vala_ccode_node_unref (_tmp87_), NULL));
				_tmp90_ = NULL;
				_tmp89_ = NULL;
				_tmp88_ = NULL;
				vala_ccode_block_add_statement ((ValaCCodeBlock*) cswitch, (ValaCCodeNode*) (_tmp90_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp89_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp88_ = vala_ccode_identifier_new ("g_assert_not_reached")))))));
				(_tmp90_ == NULL) ? NULL : (_tmp90_ = (vala_ccode_node_unref (_tmp90_), NULL));
				(_tmp89_ == NULL) ? NULL : (_tmp89_ = (vala_ccode_node_unref (_tmp89_), NULL));
				(_tmp88_ == NULL) ? NULL : (_tmp88_ = (vala_ccode_node_unref (_tmp88_), NULL));
				/* initial coroutine state*/
				_tmp92_ = NULL;
				_tmp91_ = NULL;
				vala_ccode_block_add_statement ((ValaCCodeBlock*) cswitch, (ValaCCodeNode*) (_tmp92_ = vala_ccode_case_statement_new ((ValaCCodeExpression*) (_tmp91_ = vala_ccode_constant_new ("0")))));
				(_tmp92_ == NULL) ? NULL : (_tmp92_ = (vala_ccode_node_unref (_tmp92_), NULL));
				(_tmp91_ == NULL) ? NULL : (_tmp91_ = (vala_ccode_node_unref (_tmp91_), NULL));
				/* coroutine body*/
				vala_ccode_block_add_statement ((ValaCCodeBlock*) cswitch, (ValaCCodeNode*) vala_ccode_function_get_block (((ValaCCodeBaseModule*) self)->function));
				/* epilogue*/
				_tmp93_ = NULL;
				vala_ccode_block_add_statement ((ValaCCodeBlock*) cswitch, (ValaCCodeNode*) (_tmp93_ = vala_ccode_method_module_complete_async (self)));
				(_tmp93_ == NULL) ? NULL : (_tmp93_ = (vala_ccode_node_unref (_tmp93_), NULL));
				_tmp94_ = NULL;
				vala_ccode_function_set_block (co_function, _tmp94_ = vala_ccode_block_new ());
				(_tmp94_ == NULL) ? NULL : (_tmp94_ = (vala_ccode_node_unref (_tmp94_), NULL));
				vala_ccode_block_add_statement (vala_ccode_function_get_block (co_function), (ValaCCodeNode*) cswitch);
				vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) co_function);
				(co_function == NULL) ? NULL : (co_function = (vala_ccode_node_unref (co_function), NULL));
				(cswitch == NULL) ? NULL : (cswitch = (vala_ccode_node_unref (cswitch), NULL));
			}
			_tmp95_ = FALSE;
			if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m))) {
				_tmp95_ = !vala_method_get_coroutine (m);
			} else {
				_tmp95_ = FALSE;
			}
			if (_tmp95_) {
				ValaClass* _tmp96_;
				ValaClass* cl;
				gboolean _tmp97_;
				_tmp96_ = NULL;
				cl = (_tmp96_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp96_ == NULL) ? NULL : vala_code_node_ref (_tmp96_));
				_tmp97_ = FALSE;
				if (vala_method_get_overrides (m)) {
					_tmp97_ = TRUE;
				} else {
					gboolean _tmp98_;
					gboolean _tmp99_;
					_tmp98_ = FALSE;
					_tmp99_ = FALSE;
					if (vala_method_get_base_interface_method (m) != NULL) {
						_tmp99_ = !vala_method_get_is_abstract (m);
					} else {
						_tmp99_ = FALSE;
					}
					if (_tmp99_) {
						_tmp98_ = !vala_method_get_is_virtual (m);
					} else {
						_tmp98_ = FALSE;
					}
					_tmp97_ = _tmp98_;
				}
				if (_tmp97_) {
					ValaMethod* base_method;
					ValaReferenceType* base_expression_type;
					ValaObjectType* self_target_type;
					ValaCCodeIdentifier* _tmp106_;
					ValaCCodeExpression* _tmp107_;
					ValaCCodeExpression* cself;
					char* _tmp109_;
					char* _tmp108_;
					ValaCCodeDeclaration* _tmp110_;
					ValaCCodeDeclaration* _cdecl_;
					ValaCCodeVariableDeclarator* _tmp111_;
					base_method = NULL;
					base_expression_type = NULL;
					if (vala_method_get_overrides (m)) {
						ValaMethod* _tmp101_;
						ValaMethod* _tmp100_;
						ValaReferenceType* _tmp102_;
						_tmp101_ = NULL;
						_tmp100_ = NULL;
						base_method = (_tmp101_ = (_tmp100_ = vala_method_get_base_method (m), (_tmp100_ == NULL) ? NULL : vala_code_node_ref (_tmp100_)), (base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)), _tmp101_);
						_tmp102_ = NULL;
						base_expression_type = (_tmp102_ = (ValaReferenceType*) vala_object_type_new ((ValaObjectTypeSymbol*) VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) base_method))), (base_expression_type == NULL) ? NULL : (base_expression_type = (vala_code_node_unref (base_expression_type), NULL)), _tmp102_);
					} else {
						ValaMethod* _tmp104_;
						ValaMethod* _tmp103_;
						ValaReferenceType* _tmp105_;
						_tmp104_ = NULL;
						_tmp103_ = NULL;
						base_method = (_tmp104_ = (_tmp103_ = vala_method_get_base_interface_method (m), (_tmp103_ == NULL) ? NULL : vala_code_node_ref (_tmp103_)), (base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)), _tmp104_);
						_tmp105_ = NULL;
						base_expression_type = (_tmp105_ = (ValaReferenceType*) vala_object_type_new ((ValaObjectTypeSymbol*) VALA_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) base_method))), (base_expression_type == NULL) ? NULL : (base_expression_type = (vala_code_node_unref (base_expression_type), NULL)), _tmp105_);
					}
					self_target_type = vala_object_type_new ((ValaObjectTypeSymbol*) cl);
					_tmp106_ = NULL;
					_tmp107_ = NULL;
					cself = (_tmp107_ = vala_ccode_base_module_transform_expression ((ValaCCodeBaseModule*) self, (ValaCCodeExpression*) (_tmp106_ = vala_ccode_identifier_new ("base")), (ValaDataType*) base_expression_type, (ValaDataType*) self_target_type, NULL), (_tmp106_ == NULL) ? NULL : (_tmp106_ = (vala_ccode_node_unref (_tmp106_), NULL)), _tmp107_);
					_tmp109_ = NULL;
					_tmp108_ = NULL;
					_tmp110_ = NULL;
					_cdecl_ = (_tmp110_ = vala_ccode_declaration_new (_tmp109_ = g_strdup_printf ("%s *", _tmp108_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE))), _tmp109_ = (g_free (_tmp109_), NULL), _tmp108_ = (g_free (_tmp108_), NULL), _tmp110_);
					_tmp111_ = NULL;
					vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp111_ = vala_ccode_variable_declarator_new ("self", cself, NULL)));
					(_tmp111_ == NULL) ? NULL : (_tmp111_ = (vala_ccode_node_unref (_tmp111_), NULL));
					vala_ccode_fragment_append (cinit, (ValaCCodeNode*) _cdecl_);
					(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
					(base_expression_type == NULL) ? NULL : (base_expression_type = (vala_code_node_unref (base_expression_type), NULL));
					(self_target_type == NULL) ? NULL : (self_target_type = (vala_code_node_unref (self_target_type), NULL));
					(cself == NULL) ? NULL : (cself = (vala_ccode_node_unref (cself), NULL));
					(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
				} else {
					gboolean _tmp112_;
					_tmp112_ = FALSE;
					if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
						_tmp112_ = !VALA_IS_CREATION_METHOD (m);
					} else {
						_tmp112_ = FALSE;
					}
					if (_tmp112_) {
						ValaCCodeStatement* ccheckstmt;
						ccheckstmt = vala_ccode_method_module_create_method_type_check_statement (self, m, creturn_type, (ValaTypeSymbol*) cl, TRUE, "self");
						if (ccheckstmt != NULL) {
							vala_ccode_node_set_line ((ValaCCodeNode*) ccheckstmt, vala_ccode_node_get_line ((ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function));
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) ccheckstmt);
						}
						(ccheckstmt == NULL) ? NULL : (ccheckstmt = (vala_ccode_node_unref (ccheckstmt), NULL));
					}
				}
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			}
			{
				GeeList* _tmp113_;
				GeeIterator* _tmp114_;
				GeeIterator* _param_it;
				_tmp113_ = NULL;
				_tmp114_ = NULL;
				_param_it = (_tmp114_ = gee_iterable_iterator ((GeeIterable*) (_tmp113_ = vala_method_get_parameters (m))), (_tmp113_ == NULL) ? NULL : (_tmp113_ = (gee_collection_object_unref (_tmp113_), NULL)), _tmp114_);
				while (gee_iterator_next (_param_it)) {
					ValaFormalParameter* param;
					ValaTypeSymbol* _tmp115_;
					ValaTypeSymbol* t;
					gboolean _tmp116_;
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					if (vala_formal_parameter_get_ellipsis (param)) {
						(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
						break;
					}
					_tmp115_ = NULL;
					t = (_tmp115_ = vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param)), (_tmp115_ == NULL) ? NULL : vala_code_node_ref (_tmp115_));
					_tmp116_ = FALSE;
					if (t != NULL) {
						_tmp116_ = vala_typesymbol_is_reference_type (t);
					} else {
						_tmp116_ = FALSE;
					}
					if (_tmp116_) {
						if (vala_formal_parameter_get_direction (param) != VALA_PARAMETER_DIRECTION_OUT) {
							char* _tmp117_;
							ValaCCodeStatement* _tmp118_;
							ValaCCodeStatement* type_check;
							_tmp117_ = NULL;
							_tmp118_ = NULL;
							type_check = (_tmp118_ = vala_ccode_method_module_create_method_type_check_statement (self, m, creturn_type, t, !vala_data_type_get_nullable (vala_formal_parameter_get_parameter_type (param)), _tmp117_ = vala_ccode_base_module_get_variable_cname ((ValaCCodeBaseModule*) self, vala_symbol_get_name ((ValaSymbol*) param))), _tmp117_ = (g_free (_tmp117_), NULL), _tmp118_);
							if (type_check != NULL) {
								vala_ccode_node_set_line ((ValaCCodeNode*) type_check, vala_ccode_node_get_line ((ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function));
								vala_ccode_fragment_append (cinit, (ValaCCodeNode*) type_check);
							}
							(type_check == NULL) ? NULL : (type_check = (vala_ccode_node_unref (type_check), NULL));
						} else {
							ValaCCodeConstant* _tmp121_;
							ValaCCodeUnaryExpression* _tmp120_;
							ValaCCodeExpression* _tmp119_;
							ValaCCodeAssignment* _tmp122_;
							ValaCCodeAssignment* a;
							ValaCCodeExpressionStatement* _tmp123_;
							/* ensure that the passed reference for output parameter is cleared*/
							_tmp121_ = NULL;
							_tmp120_ = NULL;
							_tmp119_ = NULL;
							_tmp122_ = NULL;
							a = (_tmp122_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp120_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_POINTER_INDIRECTION, _tmp119_ = vala_ccode_base_module_get_variable_cexpression ((ValaCCodeBaseModule*) self, vala_symbol_get_name ((ValaSymbol*) param)))), (ValaCCodeExpression*) (_tmp121_ = vala_ccode_constant_new ("NULL")), VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE), (_tmp121_ == NULL) ? NULL : (_tmp121_ = (vala_ccode_node_unref (_tmp121_), NULL)), (_tmp120_ == NULL) ? NULL : (_tmp120_ = (vala_ccode_node_unref (_tmp120_), NULL)), (_tmp119_ == NULL) ? NULL : (_tmp119_ = (vala_ccode_node_unref (_tmp119_), NULL)), _tmp122_);
							_tmp123_ = NULL;
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp123_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) a)));
							(_tmp123_ == NULL) ? NULL : (_tmp123_ = (vala_ccode_node_unref (_tmp123_), NULL));
							(a == NULL) ? NULL : (a = (vala_ccode_node_unref (a), NULL));
						}
					}
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
					(t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL));
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			if (inner_error) {
				/* always separate error parameter and inner_error local variable
				 * as error may be set to NULL but we're always interested in inner errors
				 */
				if (vala_method_get_coroutine (m)) {
					vala_ccode_struct_add_field (((ValaCCodeBaseModule*) self)->closure_struct, "GError *", "_inner_error_");
				} else {
					ValaCCodeDeclaration* _cdecl_;
					ValaCCodeVariableDeclarator* _tmp125_;
					ValaCCodeConstant* _tmp124_;
					/* no initialization necessary, closure struct is zeroed*/
					_cdecl_ = vala_ccode_declaration_new ("GError *");
					_tmp125_ = NULL;
					_tmp124_ = NULL;
					vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp125_ = vala_ccode_variable_declarator_new ("_inner_error_", (ValaCCodeExpression*) (_tmp124_ = vala_ccode_constant_new ("NULL")), NULL)));
					(_tmp125_ == NULL) ? NULL : (_tmp125_ = (vala_ccode_node_unref (_tmp125_), NULL));
					(_tmp124_ == NULL) ? NULL : (_tmp124_ = (vala_ccode_node_unref (_tmp124_), NULL));
					vala_ccode_fragment_append (cinit, (ValaCCodeNode*) _cdecl_);
					(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
				}
			}
			if (!vala_method_get_coroutine (m)) {
				gboolean _tmp126_;
				_tmp126_ = FALSE;
				if (vala_code_node_get_source_reference ((ValaCodeNode*) m) != NULL) {
					_tmp126_ = vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) m)) != NULL;
				} else {
					_tmp126_ = FALSE;
				}
				if (_tmp126_) {
					ValaCCodeComment* _tmp127_;
					_tmp127_ = NULL;
					vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) (_tmp127_ = vala_ccode_comment_new (vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) m)))));
					(_tmp127_ == NULL) ? NULL : (_tmp127_ = (vala_ccode_node_unref (_tmp127_), NULL));
				}
				vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function);
			}
			if (VALA_IS_CREATION_METHOD (m)) {
				if (in_gobject_creation_method) {
					gint n_params;
					gboolean _tmp128_;
					n_params = vala_creation_method_get_n_construction_params (VALA_CREATION_METHOD (m));
					_tmp128_ = FALSE;
					if (n_params > 0) {
						_tmp128_ = TRUE;
					} else {
						GeeList* _tmp129_;
						_tmp129_ = NULL;
						_tmp128_ = gee_collection_get_size ((GeeCollection*) (_tmp129_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class))) > 0;
						(_tmp129_ == NULL) ? NULL : (_tmp129_ = (gee_collection_object_unref (_tmp129_), NULL));
					}
					if (_tmp128_) {
						ValaCCodeIdentifier* _tmp130_;
						ValaCCodeFunctionCall* _tmp131_;
						ValaCCodeFunctionCall* cparamsinit;
						ValaCCodeIdentifier* _tmp132_;
						ValaCCodeConstant* _tmp135_;
						char* _tmp134_;
						GeeList* _tmp133_;
						ValaCCodeDeclaration* _cdecl_;
						ValaCCodeVariableDeclarator* _tmp136_;
						ValaCCodeDeclaration* _tmp137_;
						ValaCCodeVariableDeclarator* _tmp139_;
						ValaCCodeIdentifier* _tmp138_;
						/* declare construction parameter array*/
						_tmp130_ = NULL;
						_tmp131_ = NULL;
						cparamsinit = (_tmp131_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp130_ = vala_ccode_identifier_new ("g_new0"))), (_tmp130_ == NULL) ? NULL : (_tmp130_ = (vala_ccode_node_unref (_tmp130_), NULL)), _tmp131_);
						_tmp132_ = NULL;
						vala_ccode_function_call_add_argument (cparamsinit, (ValaCCodeExpression*) (_tmp132_ = vala_ccode_identifier_new ("GParameter")));
						(_tmp132_ == NULL) ? NULL : (_tmp132_ = (vala_ccode_node_unref (_tmp132_), NULL));
						_tmp135_ = NULL;
						_tmp134_ = NULL;
						_tmp133_ = NULL;
						vala_ccode_function_call_add_argument (cparamsinit, (ValaCCodeExpression*) (_tmp135_ = vala_ccode_constant_new (_tmp134_ = g_strdup_printf ("%i", n_params + (3 * gee_collection_get_size ((GeeCollection*) (_tmp133_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class))))))));
						(_tmp135_ == NULL) ? NULL : (_tmp135_ = (vala_ccode_node_unref (_tmp135_), NULL));
						_tmp134_ = (g_free (_tmp134_), NULL);
						(_tmp133_ == NULL) ? NULL : (_tmp133_ = (gee_collection_object_unref (_tmp133_), NULL));
						_cdecl_ = vala_ccode_declaration_new ("GParameter *");
						_tmp136_ = NULL;
						vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp136_ = vala_ccode_variable_declarator_new ("__params", (ValaCCodeExpression*) cparamsinit, NULL)));
						(_tmp136_ == NULL) ? NULL : (_tmp136_ = (vala_ccode_node_unref (_tmp136_), NULL));
						vala_ccode_fragment_append (cinit, (ValaCCodeNode*) _cdecl_);
						_tmp137_ = NULL;
						_cdecl_ = (_tmp137_ = vala_ccode_declaration_new ("GParameter *"), (_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL)), _tmp137_);
						_tmp139_ = NULL;
						_tmp138_ = NULL;
						vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp139_ = vala_ccode_variable_declarator_new ("__params_it", (ValaCCodeExpression*) (_tmp138_ = vala_ccode_identifier_new ("__params")), NULL)));
						(_tmp139_ == NULL) ? NULL : (_tmp139_ = (vala_ccode_node_unref (_tmp139_), NULL));
						(_tmp138_ == NULL) ? NULL : (_tmp138_ = (vala_ccode_node_unref (_tmp138_), NULL));
						vala_ccode_fragment_append (cinit, (ValaCCodeNode*) _cdecl_);
						(cparamsinit == NULL) ? NULL : (cparamsinit = (vala_ccode_node_unref (cparamsinit), NULL));
						(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
					}
					/* type, dup func, and destroy func properties for generic types */
					{
						GeeList* _tmp140_;
						GeeIterator* _tmp141_;
						GeeIterator* _type_param_it;
						/* type, dup func, and destroy func properties for generic types */
						_tmp140_ = NULL;
						_tmp141_ = NULL;
						_type_param_it = (_tmp141_ = gee_iterable_iterator ((GeeIterable*) (_tmp140_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class))), (_tmp140_ == NULL) ? NULL : (_tmp140_ = (gee_collection_object_unref (_tmp140_), NULL)), _tmp141_);
						/* type, dup func, and destroy func properties for generic types */
						while (gee_iterator_next (_type_param_it)) {
							ValaTypeParameter* type_param;
							ValaCCodeConstant* prop_name;
							ValaCCodeIdentifier* param_name;
							ValaCCodeConstant* _tmp144_;
							char* _tmp143_;
							char* _tmp142_;
							ValaCCodeIdentifier* _tmp147_;
							char* _tmp146_;
							char* _tmp145_;
							ValaCCodeExpressionStatement* _tmp150_;
							ValaCCodeExpression* _tmp149_;
							ValaIntegerType* _tmp148_;
							ValaCCodeConstant* _tmp153_;
							char* _tmp152_;
							char* _tmp151_;
							ValaCCodeIdentifier* _tmp156_;
							char* _tmp155_;
							char* _tmp154_;
							ValaCCodeExpressionStatement* _tmp160_;
							ValaCCodeExpression* _tmp159_;
							ValaPointerType* _tmp158_;
							ValaVoidType* _tmp157_;
							ValaCCodeConstant* _tmp163_;
							char* _tmp162_;
							char* _tmp161_;
							ValaCCodeIdentifier* _tmp166_;
							char* _tmp165_;
							char* _tmp164_;
							ValaCCodeExpressionStatement* _tmp170_;
							ValaCCodeExpression* _tmp169_;
							ValaPointerType* _tmp168_;
							ValaVoidType* _tmp167_;
							/* type, dup func, and destroy func properties for generic types */
							type_param = (ValaTypeParameter*) gee_iterator_get (_type_param_it);
							prop_name = NULL;
							param_name = NULL;
							_tmp144_ = NULL;
							_tmp143_ = NULL;
							_tmp142_ = NULL;
							prop_name = (_tmp144_ = vala_ccode_constant_new (_tmp143_ = g_strdup_printf ("\"%s-type\"", _tmp142_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (prop_name == NULL) ? NULL : (prop_name = (vala_ccode_node_unref (prop_name), NULL)), _tmp144_);
							_tmp143_ = (g_free (_tmp143_), NULL);
							_tmp142_ = (g_free (_tmp142_), NULL);
							_tmp147_ = NULL;
							_tmp146_ = NULL;
							_tmp145_ = NULL;
							param_name = (_tmp147_ = vala_ccode_identifier_new (_tmp146_ = g_strdup_printf ("%s_type", _tmp145_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL)), _tmp147_);
							_tmp146_ = (g_free (_tmp146_), NULL);
							_tmp145_ = (g_free (_tmp145_), NULL);
							_tmp150_ = NULL;
							_tmp149_ = NULL;
							_tmp148_ = NULL;
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp150_ = vala_ccode_expression_statement_new (_tmp149_ = vala_ccode_module_get_construct_property_assignment ((ValaCCodeModule*) self, prop_name, (ValaDataType*) (_tmp148_ = vala_integer_type_new (VALA_STRUCT (((ValaCCodeBaseModule*) self)->gtype_type), NULL, NULL)), (ValaCCodeExpression*) param_name))));
							(_tmp150_ == NULL) ? NULL : (_tmp150_ = (vala_ccode_node_unref (_tmp150_), NULL));
							(_tmp149_ == NULL) ? NULL : (_tmp149_ = (vala_ccode_node_unref (_tmp149_), NULL));
							(_tmp148_ == NULL) ? NULL : (_tmp148_ = (vala_code_node_unref (_tmp148_), NULL));
							_tmp153_ = NULL;
							_tmp152_ = NULL;
							_tmp151_ = NULL;
							prop_name = (_tmp153_ = vala_ccode_constant_new (_tmp152_ = g_strdup_printf ("\"%s-dup-func\"", _tmp151_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (prop_name == NULL) ? NULL : (prop_name = (vala_ccode_node_unref (prop_name), NULL)), _tmp153_);
							_tmp152_ = (g_free (_tmp152_), NULL);
							_tmp151_ = (g_free (_tmp151_), NULL);
							_tmp156_ = NULL;
							_tmp155_ = NULL;
							_tmp154_ = NULL;
							param_name = (_tmp156_ = vala_ccode_identifier_new (_tmp155_ = g_strdup_printf ("%s_dup_func", _tmp154_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL)), _tmp156_);
							_tmp155_ = (g_free (_tmp155_), NULL);
							_tmp154_ = (g_free (_tmp154_), NULL);
							_tmp160_ = NULL;
							_tmp159_ = NULL;
							_tmp158_ = NULL;
							_tmp157_ = NULL;
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp160_ = vala_ccode_expression_statement_new (_tmp159_ = vala_ccode_module_get_construct_property_assignment ((ValaCCodeModule*) self, prop_name, (ValaDataType*) (_tmp158_ = vala_pointer_type_new ((ValaDataType*) (_tmp157_ = vala_void_type_new (NULL)), NULL)), (ValaCCodeExpression*) param_name))));
							(_tmp160_ == NULL) ? NULL : (_tmp160_ = (vala_ccode_node_unref (_tmp160_), NULL));
							(_tmp159_ == NULL) ? NULL : (_tmp159_ = (vala_ccode_node_unref (_tmp159_), NULL));
							(_tmp158_ == NULL) ? NULL : (_tmp158_ = (vala_code_node_unref (_tmp158_), NULL));
							(_tmp157_ == NULL) ? NULL : (_tmp157_ = (vala_code_node_unref (_tmp157_), NULL));
							_tmp163_ = NULL;
							_tmp162_ = NULL;
							_tmp161_ = NULL;
							prop_name = (_tmp163_ = vala_ccode_constant_new (_tmp162_ = g_strdup_printf ("\"%s-destroy-func\"", _tmp161_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (prop_name == NULL) ? NULL : (prop_name = (vala_ccode_node_unref (prop_name), NULL)), _tmp163_);
							_tmp162_ = (g_free (_tmp162_), NULL);
							_tmp161_ = (g_free (_tmp161_), NULL);
							_tmp166_ = NULL;
							_tmp165_ = NULL;
							_tmp164_ = NULL;
							param_name = (_tmp166_ = vala_ccode_identifier_new (_tmp165_ = g_strdup_printf ("%s_destroy_func", _tmp164_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL)), _tmp166_);
							_tmp165_ = (g_free (_tmp165_), NULL);
							_tmp164_ = (g_free (_tmp164_), NULL);
							_tmp170_ = NULL;
							_tmp169_ = NULL;
							_tmp168_ = NULL;
							_tmp167_ = NULL;
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp170_ = vala_ccode_expression_statement_new (_tmp169_ = vala_ccode_module_get_construct_property_assignment ((ValaCCodeModule*) self, prop_name, (ValaDataType*) (_tmp168_ = vala_pointer_type_new ((ValaDataType*) (_tmp167_ = vala_void_type_new (NULL)), NULL)), (ValaCCodeExpression*) param_name))));
							(_tmp170_ == NULL) ? NULL : (_tmp170_ = (vala_ccode_node_unref (_tmp170_), NULL));
							(_tmp169_ == NULL) ? NULL : (_tmp169_ = (vala_ccode_node_unref (_tmp169_), NULL));
							(_tmp168_ == NULL) ? NULL : (_tmp168_ = (vala_code_node_unref (_tmp168_), NULL));
							(_tmp167_ == NULL) ? NULL : (_tmp167_ = (vala_code_node_unref (_tmp167_), NULL));
							(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
							(prop_name == NULL) ? NULL : (prop_name = (vala_ccode_node_unref (prop_name), NULL));
							(param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL));
						}
						(_type_param_it == NULL) ? NULL : (_type_param_it = (gee_collection_object_unref (_type_param_it), NULL));
					}
				} else {
					if (vala_ccode_method_module_is_gtypeinstance_creation_method (self, m)) {
						ValaClass* _tmp171_;
						ValaClass* cl;
						char* _tmp173_;
						char* _tmp172_;
						ValaCCodeDeclaration* _tmp174_;
						ValaCCodeDeclaration* cdeclaration;
						ValaCCodeVariableDeclarator* _cdecl_;
						_tmp171_ = NULL;
						cl = (_tmp171_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp171_ == NULL) ? NULL : vala_code_node_ref (_tmp171_));
						_tmp173_ = NULL;
						_tmp172_ = NULL;
						_tmp174_ = NULL;
						cdeclaration = (_tmp174_ = vala_ccode_declaration_new (_tmp173_ = g_strconcat (_tmp172_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE), "*", NULL)), _tmp173_ = (g_free (_tmp173_), NULL), _tmp172_ = (g_free (_tmp172_), NULL), _tmp174_);
						_cdecl_ = vala_ccode_variable_declarator_new ("self", NULL, NULL);
						vala_ccode_declaration_add_declarator (cdeclaration, (ValaCCodeDeclarator*) _cdecl_);
						vala_ccode_fragment_append (cinit, (ValaCCodeNode*) cdeclaration);
						if (!vala_creation_method_get_chain_up (VALA_CREATION_METHOD (m))) {
							ValaCCodeIdentifier* _tmp175_;
							ValaCCodeFunctionCall* _tmp176_;
							ValaCCodeFunctionCall* ccall;
							ValaCCodeIdentifier* _tmp177_;
							ValaCCodeCastExpression* _tmp180_;
							char* _tmp179_;
							char* _tmp178_;
							/* TODO implicitly chain up to base class as in add_object_creation*/
							_tmp175_ = NULL;
							_tmp176_ = NULL;
							ccall = (_tmp176_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp175_ = vala_ccode_identifier_new ("g_type_create_instance"))), (_tmp175_ == NULL) ? NULL : (_tmp175_ = (vala_ccode_node_unref (_tmp175_), NULL)), _tmp176_);
							_tmp177_ = NULL;
							vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp177_ = vala_ccode_identifier_new ("object_type")));
							(_tmp177_ == NULL) ? NULL : (_tmp177_ = (vala_ccode_node_unref (_tmp177_), NULL));
							_tmp180_ = NULL;
							_tmp179_ = NULL;
							_tmp178_ = NULL;
							vala_ccode_variable_declarator_set_initializer (_cdecl_, (ValaCCodeExpression*) (_tmp180_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) ccall, _tmp179_ = g_strconcat (_tmp178_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE), "*", NULL))));
							(_tmp180_ == NULL) ? NULL : (_tmp180_ = (vala_ccode_node_unref (_tmp180_), NULL));
							_tmp179_ = (g_free (_tmp179_), NULL);
							_tmp178_ = (g_free (_tmp178_), NULL);
							(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
						}
						/* type, dup func, and destroy func fields for generic types */
						{
							GeeList* _tmp181_;
							GeeIterator* _tmp182_;
							GeeIterator* _type_param_it;
							/* type, dup func, and destroy func fields for generic types */
							_tmp181_ = NULL;
							_tmp182_ = NULL;
							_type_param_it = (_tmp182_ = gee_iterable_iterator ((GeeIterable*) (_tmp181_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class))), (_tmp181_ == NULL) ? NULL : (_tmp181_ = (gee_collection_object_unref (_tmp181_), NULL)), _tmp182_);
							/* type, dup func, and destroy func fields for generic types */
							while (gee_iterator_next (_type_param_it)) {
								ValaTypeParameter* type_param;
								ValaCCodeIdentifier* param_name;
								ValaCCodeAssignment* assign;
								ValaCCodeIdentifier* _tmp183_;
								ValaCCodeMemberAccess* _tmp184_;
								ValaCCodeMemberAccess* priv_access;
								ValaCCodeIdentifier* _tmp187_;
								char* _tmp186_;
								char* _tmp185_;
								ValaCCodeAssignment* _tmp189_;
								ValaCCodeMemberAccess* _tmp188_;
								ValaCCodeExpressionStatement* _tmp190_;
								ValaCCodeIdentifier* _tmp193_;
								char* _tmp192_;
								char* _tmp191_;
								ValaCCodeAssignment* _tmp195_;
								ValaCCodeMemberAccess* _tmp194_;
								ValaCCodeExpressionStatement* _tmp196_;
								ValaCCodeIdentifier* _tmp199_;
								char* _tmp198_;
								char* _tmp197_;
								ValaCCodeAssignment* _tmp201_;
								ValaCCodeMemberAccess* _tmp200_;
								ValaCCodeExpressionStatement* _tmp202_;
								/* type, dup func, and destroy func fields for generic types */
								type_param = (ValaTypeParameter*) gee_iterator_get (_type_param_it);
								param_name = NULL;
								assign = NULL;
								_tmp183_ = NULL;
								_tmp184_ = NULL;
								priv_access = (_tmp184_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp183_ = vala_ccode_identifier_new ("self")), "priv"), (_tmp183_ == NULL) ? NULL : (_tmp183_ = (vala_ccode_node_unref (_tmp183_), NULL)), _tmp184_);
								_tmp187_ = NULL;
								_tmp186_ = NULL;
								_tmp185_ = NULL;
								param_name = (_tmp187_ = vala_ccode_identifier_new (_tmp186_ = g_strdup_printf ("%s_type", _tmp185_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL)), _tmp187_);
								_tmp186_ = (g_free (_tmp186_), NULL);
								_tmp185_ = (g_free (_tmp185_), NULL);
								_tmp189_ = NULL;
								_tmp188_ = NULL;
								assign = (_tmp189_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp188_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) priv_access, vala_ccode_identifier_get_name (param_name))), (ValaCCodeExpression*) param_name, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE), (assign == NULL) ? NULL : (assign = (vala_ccode_node_unref (assign), NULL)), _tmp189_);
								(_tmp188_ == NULL) ? NULL : (_tmp188_ = (vala_ccode_node_unref (_tmp188_), NULL));
								_tmp190_ = NULL;
								vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp190_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) assign)));
								(_tmp190_ == NULL) ? NULL : (_tmp190_ = (vala_ccode_node_unref (_tmp190_), NULL));
								_tmp193_ = NULL;
								_tmp192_ = NULL;
								_tmp191_ = NULL;
								param_name = (_tmp193_ = vala_ccode_identifier_new (_tmp192_ = g_strdup_printf ("%s_dup_func", _tmp191_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL)), _tmp193_);
								_tmp192_ = (g_free (_tmp192_), NULL);
								_tmp191_ = (g_free (_tmp191_), NULL);
								_tmp195_ = NULL;
								_tmp194_ = NULL;
								assign = (_tmp195_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp194_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) priv_access, vala_ccode_identifier_get_name (param_name))), (ValaCCodeExpression*) param_name, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE), (assign == NULL) ? NULL : (assign = (vala_ccode_node_unref (assign), NULL)), _tmp195_);
								(_tmp194_ == NULL) ? NULL : (_tmp194_ = (vala_ccode_node_unref (_tmp194_), NULL));
								_tmp196_ = NULL;
								vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp196_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) assign)));
								(_tmp196_ == NULL) ? NULL : (_tmp196_ = (vala_ccode_node_unref (_tmp196_), NULL));
								_tmp199_ = NULL;
								_tmp198_ = NULL;
								_tmp197_ = NULL;
								param_name = (_tmp199_ = vala_ccode_identifier_new (_tmp198_ = g_strdup_printf ("%s_destroy_func", _tmp197_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1))), (param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL)), _tmp199_);
								_tmp198_ = (g_free (_tmp198_), NULL);
								_tmp197_ = (g_free (_tmp197_), NULL);
								_tmp201_ = NULL;
								_tmp200_ = NULL;
								assign = (_tmp201_ = vala_ccode_assignment_new ((ValaCCodeExpression*) (_tmp200_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) priv_access, vala_ccode_identifier_get_name (param_name))), (ValaCCodeExpression*) param_name, VALA_CCODE_ASSIGNMENT_OPERATOR_SIMPLE), (assign == NULL) ? NULL : (assign = (vala_ccode_node_unref (assign), NULL)), _tmp201_);
								(_tmp200_ == NULL) ? NULL : (_tmp200_ = (vala_ccode_node_unref (_tmp200_), NULL));
								_tmp202_ = NULL;
								vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp202_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) assign)));
								(_tmp202_ == NULL) ? NULL : (_tmp202_ = (vala_ccode_node_unref (_tmp202_), NULL));
								(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
								(param_name == NULL) ? NULL : (param_name = (vala_ccode_node_unref (param_name), NULL));
								(assign == NULL) ? NULL : (assign = (vala_ccode_node_unref (assign), NULL));
								(priv_access == NULL) ? NULL : (priv_access = (vala_ccode_node_unref (priv_access), NULL));
							}
							(_type_param_it == NULL) ? NULL : (_type_param_it = (gee_collection_object_unref (_type_param_it), NULL));
						}
						(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
						(cdeclaration == NULL) ? NULL : (cdeclaration = (vala_ccode_node_unref (cdeclaration), NULL));
						(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
					} else {
						if (VALA_IS_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol)) {
							ValaClass* _tmp203_;
							ValaClass* cl;
							char* _tmp205_;
							char* _tmp204_;
							ValaCCodeDeclaration* _tmp206_;
							ValaCCodeDeclaration* _cdecl_;
							ValaCCodeIdentifier* _tmp207_;
							ValaCCodeFunctionCall* _tmp208_;
							ValaCCodeFunctionCall* ccall;
							ValaCCodeIdentifier* _tmp210_;
							char* _tmp209_;
							ValaCCodeVariableDeclarator* _tmp211_;
							ValaCCodeIdentifier* _tmp214_;
							char* _tmp213_;
							char* _tmp212_;
							ValaCCodeFunctionCall* _tmp215_;
							ValaCCodeFunctionCall* cinitcall;
							ValaCCodeIdentifier* _tmp216_;
							ValaCCodeExpressionStatement* _tmp217_;
							_tmp203_ = NULL;
							cl = (_tmp203_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp203_ == NULL) ? NULL : vala_code_node_ref (_tmp203_));
							_tmp205_ = NULL;
							_tmp204_ = NULL;
							_tmp206_ = NULL;
							_cdecl_ = (_tmp206_ = vala_ccode_declaration_new (_tmp205_ = g_strconcat (_tmp204_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE), "*", NULL)), _tmp205_ = (g_free (_tmp205_), NULL), _tmp204_ = (g_free (_tmp204_), NULL), _tmp206_);
							_tmp207_ = NULL;
							_tmp208_ = NULL;
							ccall = (_tmp208_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp207_ = vala_ccode_identifier_new ("g_slice_new0"))), (_tmp207_ == NULL) ? NULL : (_tmp207_ = (vala_ccode_node_unref (_tmp207_), NULL)), _tmp208_);
							_tmp210_ = NULL;
							_tmp209_ = NULL;
							vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp210_ = vala_ccode_identifier_new (_tmp209_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE))));
							(_tmp210_ == NULL) ? NULL : (_tmp210_ = (vala_ccode_node_unref (_tmp210_), NULL));
							_tmp209_ = (g_free (_tmp209_), NULL);
							_tmp211_ = NULL;
							vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp211_ = vala_ccode_variable_declarator_new ("self", (ValaCCodeExpression*) ccall, NULL)));
							(_tmp211_ == NULL) ? NULL : (_tmp211_ = (vala_ccode_node_unref (_tmp211_), NULL));
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) _cdecl_);
							_tmp214_ = NULL;
							_tmp213_ = NULL;
							_tmp212_ = NULL;
							_tmp215_ = NULL;
							cinitcall = (_tmp215_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp214_ = vala_ccode_identifier_new (_tmp213_ = g_strdup_printf ("%s_instance_init", _tmp212_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) cl, NULL))))), (_tmp214_ == NULL) ? NULL : (_tmp214_ = (vala_ccode_node_unref (_tmp214_), NULL)), _tmp213_ = (g_free (_tmp213_), NULL), _tmp212_ = (g_free (_tmp212_), NULL), _tmp215_);
							_tmp216_ = NULL;
							vala_ccode_function_call_add_argument (cinitcall, (ValaCCodeExpression*) (_tmp216_ = vala_ccode_identifier_new ("self")));
							(_tmp216_ == NULL) ? NULL : (_tmp216_ = (vala_ccode_node_unref (_tmp216_), NULL));
							_tmp217_ = NULL;
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp217_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) cinitcall)));
							(_tmp217_ == NULL) ? NULL : (_tmp217_ = (vala_ccode_node_unref (_tmp217_), NULL));
							(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
							(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
							(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
							(cinitcall == NULL) ? NULL : (cinitcall = (vala_ccode_node_unref (cinitcall), NULL));
						} else {
							ValaStruct* _tmp218_;
							ValaStruct* st;
							ValaCCodeIdentifier* _tmp219_;
							ValaCCodeFunctionCall* _tmp220_;
							ValaCCodeFunctionCall* czero;
							ValaCCodeIdentifier* _tmp221_;
							ValaCCodeConstant* _tmp222_;
							ValaCCodeIdentifier* _tmp225_;
							char* _tmp224_;
							char* _tmp223_;
							ValaCCodeExpressionStatement* _tmp226_;
							_tmp218_ = NULL;
							st = (_tmp218_ = VALA_STRUCT (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp218_ == NULL) ? NULL : vala_code_node_ref (_tmp218_));
							/* memset needs string.h*/
							vala_ccode_declaration_space_add_include (((ValaCCodeBaseModule*) self)->source_declarations, "string.h", FALSE);
							_tmp219_ = NULL;
							_tmp220_ = NULL;
							czero = (_tmp220_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp219_ = vala_ccode_identifier_new ("memset"))), (_tmp219_ == NULL) ? NULL : (_tmp219_ = (vala_ccode_node_unref (_tmp219_), NULL)), _tmp220_);
							_tmp221_ = NULL;
							vala_ccode_function_call_add_argument (czero, (ValaCCodeExpression*) (_tmp221_ = vala_ccode_identifier_new ("self")));
							(_tmp221_ == NULL) ? NULL : (_tmp221_ = (vala_ccode_node_unref (_tmp221_), NULL));
							_tmp222_ = NULL;
							vala_ccode_function_call_add_argument (czero, (ValaCCodeExpression*) (_tmp222_ = vala_ccode_constant_new ("0")));
							(_tmp222_ == NULL) ? NULL : (_tmp222_ = (vala_ccode_node_unref (_tmp222_), NULL));
							_tmp225_ = NULL;
							_tmp224_ = NULL;
							_tmp223_ = NULL;
							vala_ccode_function_call_add_argument (czero, (ValaCCodeExpression*) (_tmp225_ = vala_ccode_identifier_new (_tmp224_ = g_strdup_printf ("sizeof (%s)", _tmp223_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE)))));
							(_tmp225_ == NULL) ? NULL : (_tmp225_ = (vala_ccode_node_unref (_tmp225_), NULL));
							_tmp224_ = (g_free (_tmp224_), NULL);
							_tmp223_ = (g_free (_tmp223_), NULL);
							_tmp226_ = NULL;
							vala_ccode_fragment_append (cinit, (ValaCCodeNode*) (_tmp226_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) czero)));
							(_tmp226_ == NULL) ? NULL : (_tmp226_ = (vala_ccode_node_unref (_tmp226_), NULL));
							(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
							(czero == NULL) ? NULL : (czero = (vala_ccode_node_unref (czero), NULL));
						}
					}
				}
			}
			_tmp227_ = FALSE;
			if (vala_code_context_get_module_init_method (vala_ccode_base_module_get_context ((ValaCCodeBaseModule*) self)) == m) {
				_tmp227_ = ((ValaCCodeBaseModule*) self)->in_plugin;
			} else {
				_tmp227_ = FALSE;
			}
			if (_tmp227_) {
				/* GTypeModule-based plug-in, register types*/
				vala_ccode_fragment_append (cinit, (ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->module_init_fragment);
			}
			{
				GeeList* _tmp228_;
				GeeIterator* _tmp229_;
				GeeIterator* _precondition_it;
				_tmp228_ = NULL;
				_tmp229_ = NULL;
				_precondition_it = (_tmp229_ = gee_iterable_iterator ((GeeIterable*) (_tmp228_ = vala_method_get_preconditions (m))), (_tmp228_ == NULL) ? NULL : (_tmp228_ = (gee_collection_object_unref (_tmp228_), NULL)), _tmp229_);
				while (gee_iterator_next (_precondition_it)) {
					ValaExpression* precondition;
					ValaCCodeStatement* check_stmt;
					precondition = (ValaExpression*) gee_iterator_get (_precondition_it);
					check_stmt = vala_ccode_method_module_create_precondition_statement (self, (ValaCodeNode*) m, creturn_type, precondition);
					if (check_stmt != NULL) {
						vala_ccode_fragment_append (cinit, (ValaCCodeNode*) check_stmt);
					}
					(precondition == NULL) ? NULL : (precondition = (vala_code_node_unref (precondition), NULL));
					(check_stmt == NULL) ? NULL : (check_stmt = (vala_ccode_node_unref (check_stmt), NULL));
				}
				(_precondition_it == NULL) ? NULL : (_precondition_it = (gee_collection_object_unref (_precondition_it), NULL));
			}
			(cinit == NULL) ? NULL : (cinit = (vala_ccode_node_unref (cinit), NULL));
		} else {
			if (vala_method_get_is_abstract (m)) {
				ValaCCodeBlock* cblock;
				ValaCCodeStatement* check_stmt;
				ValaCCodeIdentifier* _tmp230_;
				ValaCCodeFunctionCall* _tmp231_;
				ValaCCodeFunctionCall* type_from_instance_call;
				ValaCCodeIdentifier* _tmp232_;
				ValaCCodeIdentifier* _tmp233_;
				ValaCCodeFunctionCall* _tmp234_;
				ValaCCodeFunctionCall* type_name_call;
				char* _tmp235_;
				char* _tmp236_;
				char* error_string;
				ValaCCodeIdentifier* _tmp237_;
				ValaCCodeFunctionCall* _tmp238_;
				ValaCCodeFunctionCall* cerrorcall;
				ValaCCodeConstant* _tmp239_;
				ValaCCodeExpressionStatement* _tmp240_;
				ValaCCodeReturnStatement* _tmp242_;
				ValaCCodeExpression* _tmp241_;
				/* generate helpful error message if a sublcass does not implement an abstract method.
				 This is only meaningful for subclasses implemented in C since the vala compiler would
				 complain during compile time of such en error.*/
				cblock = vala_ccode_block_new ();
				/* add a typecheck statement for "self"*/
				check_stmt = vala_ccode_method_module_create_method_type_check_statement (self, m, creturn_type, ((ValaCCodeBaseModule*) self)->current_type_symbol, TRUE, "self");
				if (check_stmt != NULL) {
					vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) check_stmt);
				}
				/* add critical warning that this method should not have been called*/
				_tmp230_ = NULL;
				_tmp231_ = NULL;
				type_from_instance_call = (_tmp231_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp230_ = vala_ccode_identifier_new ("G_TYPE_FROM_INSTANCE"))), (_tmp230_ == NULL) ? NULL : (_tmp230_ = (vala_ccode_node_unref (_tmp230_), NULL)), _tmp231_);
				_tmp232_ = NULL;
				vala_ccode_function_call_add_argument (type_from_instance_call, (ValaCCodeExpression*) (_tmp232_ = vala_ccode_identifier_new ("self")));
				(_tmp232_ == NULL) ? NULL : (_tmp232_ = (vala_ccode_node_unref (_tmp232_), NULL));
				_tmp233_ = NULL;
				_tmp234_ = NULL;
				type_name_call = (_tmp234_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp233_ = vala_ccode_identifier_new ("g_type_name"))), (_tmp233_ == NULL) ? NULL : (_tmp233_ = (vala_ccode_node_unref (_tmp233_), NULL)), _tmp234_);
				vala_ccode_function_call_add_argument (type_name_call, (ValaCCodeExpression*) type_from_instance_call);
				_tmp235_ = NULL;
				_tmp236_ = NULL;
				error_string = (_tmp236_ = g_strdup_printf ("\"Type `%%s' does not implement abstract method `%s'\"", _tmp235_ = vala_method_get_cname (m)), _tmp235_ = (g_free (_tmp235_), NULL), _tmp236_);
				_tmp237_ = NULL;
				_tmp238_ = NULL;
				cerrorcall = (_tmp238_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp237_ = vala_ccode_identifier_new ("g_critical"))), (_tmp237_ == NULL) ? NULL : (_tmp237_ = (vala_ccode_node_unref (_tmp237_), NULL)), _tmp238_);
				_tmp239_ = NULL;
				vala_ccode_function_call_add_argument (cerrorcall, (ValaCCodeExpression*) (_tmp239_ = vala_ccode_constant_new (error_string)));
				(_tmp239_ == NULL) ? NULL : (_tmp239_ = (vala_ccode_node_unref (_tmp239_), NULL));
				vala_ccode_function_call_add_argument (cerrorcall, (ValaCCodeExpression*) type_name_call);
				_tmp240_ = NULL;
				vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp240_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) cerrorcall)));
				(_tmp240_ == NULL) ? NULL : (_tmp240_ = (vala_ccode_node_unref (_tmp240_), NULL));
				/* add return statement*/
				_tmp242_ = NULL;
				_tmp241_ = NULL;
				vala_ccode_block_add_statement (cblock, (ValaCCodeNode*) (_tmp242_ = vala_ccode_return_statement_new (_tmp241_ = vala_ccode_base_module_default_value_for_type ((ValaCCodeBaseModule*) self, creturn_type, FALSE))));
				(_tmp242_ == NULL) ? NULL : (_tmp242_ = (vala_ccode_node_unref (_tmp242_), NULL));
				(_tmp241_ == NULL) ? NULL : (_tmp241_ = (vala_ccode_node_unref (_tmp241_), NULL));
				vala_ccode_function_set_block (((ValaCCodeBaseModule*) self)->function, cblock);
				vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function);
				(cblock == NULL) ? NULL : (cblock = (vala_ccode_node_unref (cblock), NULL));
				(check_stmt == NULL) ? NULL : (check_stmt = (vala_ccode_node_unref (check_stmt), NULL));
				(type_from_instance_call == NULL) ? NULL : (type_from_instance_call = (vala_ccode_node_unref (type_from_instance_call), NULL));
				(type_name_call == NULL) ? NULL : (type_name_call = (vala_ccode_node_unref (type_name_call), NULL));
				error_string = (g_free (error_string), NULL);
				(cerrorcall == NULL) ? NULL : (cerrorcall = (vala_ccode_node_unref (cerrorcall), NULL));
			}
		}
	}
	_tmp243_ = FALSE;
	_tmp244_ = FALSE;
	if (vala_method_get_is_abstract (m)) {
		_tmp244_ = TRUE;
	} else {
		_tmp244_ = vala_method_get_is_virtual (m);
	}
	if (_tmp244_) {
		_tmp243_ = !vala_method_get_coroutine (m);
	} else {
		_tmp243_ = FALSE;
	}
	if (_tmp243_) {
		GeeHashMap* _tmp245_;
		GeeHashMap* carg_map;
		_tmp245_ = NULL;
		cparam_map = (_tmp245_ = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal), (cparam_map == NULL) ? NULL : (cparam_map = (gee_collection_object_unref (cparam_map), NULL)), _tmp245_);
		carg_map = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal);
		vala_ccode_method_module_generate_vfunc (self, m, creturn_type, (GeeMap*) cparam_map, (GeeMap*) carg_map, "", 3);
		(carg_map == NULL) ? NULL : (carg_map = (gee_collection_object_unref (carg_map), NULL));
	}
	if (vala_method_get_entry_point (m)) {
		ValaCCodeFunction* cmain;
		ValaCCodeFormalParameter* _tmp246_;
		ValaCCodeFormalParameter* _tmp247_;
		ValaCCodeBlock* main_block;
		ValaCCodeFunctionCall* _tmp253_;
		ValaCCodeIdentifier* _tmp252_;
		ValaCCodeExpressionStatement* _tmp254_;
		ValaCCodeExpressionStatement* type_init_call;
		ValaCCodeIdentifier* _tmp255_;
		ValaCCodeFunctionCall* _tmp256_;
		ValaCCodeFunctionCall* main_call;
		GeeList* _tmp257_;
		gboolean _tmp258_;
		/* m is possible entry point, add appropriate startup code*/
		cmain = vala_ccode_function_new ("main", "int");
		vala_ccode_node_set_line ((ValaCCodeNode*) cmain, vala_ccode_node_get_line ((ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function));
		_tmp246_ = NULL;
		vala_ccode_function_add_parameter (cmain, _tmp246_ = vala_ccode_formal_parameter_new ("argc", "int"));
		(_tmp246_ == NULL) ? NULL : (_tmp246_ = (vala_ccode_node_unref (_tmp246_), NULL));
		_tmp247_ = NULL;
		vala_ccode_function_add_parameter (cmain, _tmp247_ = vala_ccode_formal_parameter_new ("argv", "char **"));
		(_tmp247_ == NULL) ? NULL : (_tmp247_ = (vala_ccode_node_unref (_tmp247_), NULL));
		main_block = vala_ccode_block_new ();
		if (vala_code_context_get_thread (vala_ccode_base_module_get_context ((ValaCCodeBaseModule*) self))) {
			ValaCCodeIdentifier* _tmp248_;
			ValaCCodeFunctionCall* _tmp249_;
			ValaCCodeFunctionCall* thread_init_call;
			ValaCCodeConstant* _tmp250_;
			ValaCCodeExpressionStatement* _tmp251_;
			_tmp248_ = NULL;
			_tmp249_ = NULL;
			thread_init_call = (_tmp249_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp248_ = vala_ccode_identifier_new ("g_thread_init"))), (_tmp248_ == NULL) ? NULL : (_tmp248_ = (vala_ccode_node_unref (_tmp248_), NULL)), _tmp249_);
			vala_ccode_node_set_line ((ValaCCodeNode*) thread_init_call, vala_ccode_node_get_line ((ValaCCodeNode*) cmain));
			_tmp250_ = NULL;
			vala_ccode_function_call_add_argument (thread_init_call, (ValaCCodeExpression*) (_tmp250_ = vala_ccode_constant_new ("NULL")));
			(_tmp250_ == NULL) ? NULL : (_tmp250_ = (vala_ccode_node_unref (_tmp250_), NULL));
			_tmp251_ = NULL;
			vala_ccode_block_add_statement (main_block, (ValaCCodeNode*) (_tmp251_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) thread_init_call)));
			(_tmp251_ == NULL) ? NULL : (_tmp251_ = (vala_ccode_node_unref (_tmp251_), NULL));
			(thread_init_call == NULL) ? NULL : (thread_init_call = (vala_ccode_node_unref (thread_init_call), NULL));
		}
		_tmp253_ = NULL;
		_tmp252_ = NULL;
		_tmp254_ = NULL;
		type_init_call = (_tmp254_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp253_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp252_ = vala_ccode_identifier_new ("g_type_init"))))), (_tmp253_ == NULL) ? NULL : (_tmp253_ = (vala_ccode_node_unref (_tmp253_), NULL)), (_tmp252_ == NULL) ? NULL : (_tmp252_ = (vala_ccode_node_unref (_tmp252_), NULL)), _tmp254_);
		vala_ccode_node_set_line ((ValaCCodeNode*) type_init_call, vala_ccode_node_get_line ((ValaCCodeNode*) cmain));
		vala_ccode_block_add_statement (main_block, (ValaCCodeNode*) type_init_call);
		_tmp255_ = NULL;
		_tmp256_ = NULL;
		main_call = (_tmp256_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp255_ = vala_ccode_identifier_new (vala_ccode_function_get_name (((ValaCCodeBaseModule*) self)->function)))), (_tmp255_ == NULL) ? NULL : (_tmp255_ = (vala_ccode_node_unref (_tmp255_), NULL)), _tmp256_);
		_tmp257_ = NULL;
		if ((_tmp258_ = gee_collection_get_size ((GeeCollection*) (_tmp257_ = vala_method_get_parameters (m))) == 1, (_tmp257_ == NULL) ? NULL : (_tmp257_ = (gee_collection_object_unref (_tmp257_), NULL)), _tmp258_)) {
			ValaCCodeIdentifier* _tmp259_;
			ValaCCodeIdentifier* _tmp260_;
			_tmp259_ = NULL;
			vala_ccode_function_call_add_argument (main_call, (ValaCCodeExpression*) (_tmp259_ = vala_ccode_identifier_new ("argv")));
			(_tmp259_ == NULL) ? NULL : (_tmp259_ = (vala_ccode_node_unref (_tmp259_), NULL));
			_tmp260_ = NULL;
			vala_ccode_function_call_add_argument (main_call, (ValaCCodeExpression*) (_tmp260_ = vala_ccode_identifier_new ("argc")));
			(_tmp260_ == NULL) ? NULL : (_tmp260_ = (vala_ccode_node_unref (_tmp260_), NULL));
		}
		if (VALA_IS_VOID_TYPE (vala_method_get_return_type (m))) {
			ValaCCodeExpressionStatement* main_stmt;
			ValaCCodeConstant* _tmp261_;
			ValaCCodeReturnStatement* _tmp262_;
			ValaCCodeReturnStatement* ret_stmt;
			/* method returns void, always use 0 as exit code*/
			main_stmt = vala_ccode_expression_statement_new ((ValaCCodeExpression*) main_call);
			vala_ccode_node_set_line ((ValaCCodeNode*) main_stmt, vala_ccode_node_get_line ((ValaCCodeNode*) cmain));
			vala_ccode_block_add_statement (main_block, (ValaCCodeNode*) main_stmt);
			_tmp261_ = NULL;
			_tmp262_ = NULL;
			ret_stmt = (_tmp262_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp261_ = vala_ccode_constant_new ("0"))), (_tmp261_ == NULL) ? NULL : (_tmp261_ = (vala_ccode_node_unref (_tmp261_), NULL)), _tmp262_);
			vala_ccode_node_set_line ((ValaCCodeNode*) ret_stmt, vala_ccode_node_get_line ((ValaCCodeNode*) cmain));
			vala_ccode_block_add_statement (main_block, (ValaCCodeNode*) ret_stmt);
			(main_stmt == NULL) ? NULL : (main_stmt = (vala_ccode_node_unref (main_stmt), NULL));
			(ret_stmt == NULL) ? NULL : (ret_stmt = (vala_ccode_node_unref (ret_stmt), NULL));
		} else {
			ValaCCodeReturnStatement* main_stmt;
			main_stmt = vala_ccode_return_statement_new ((ValaCCodeExpression*) main_call);
			vala_ccode_node_set_line ((ValaCCodeNode*) main_stmt, vala_ccode_node_get_line ((ValaCCodeNode*) cmain));
			vala_ccode_block_add_statement (main_block, (ValaCCodeNode*) main_stmt);
			(main_stmt == NULL) ? NULL : (main_stmt = (vala_ccode_node_unref (main_stmt), NULL));
		}
		vala_ccode_function_set_block (cmain, main_block);
		vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) cmain);
		(cmain == NULL) ? NULL : (cmain = (vala_ccode_node_unref (cmain), NULL));
		(main_block == NULL) ? NULL : (main_block = (vala_ccode_node_unref (main_block), NULL));
		(type_init_call == NULL) ? NULL : (type_init_call = (vala_ccode_node_unref (type_init_call), NULL));
		(main_call == NULL) ? NULL : (main_call = (vala_ccode_node_unref (main_call), NULL));
	}
	(old_type_symbol == NULL) ? NULL : (old_type_symbol = (vala_code_node_unref (old_type_symbol), NULL));
	(old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL));
	(old_method == NULL) ? NULL : (old_method = (vala_code_node_unref (old_method), NULL));
	(old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL));
	(old_temp_vars == NULL) ? NULL : (old_temp_vars = (gee_collection_object_unref (old_temp_vars), NULL));
	(old_temp_ref_vars == NULL) ? NULL : (old_temp_ref_vars = (gee_collection_object_unref (old_temp_ref_vars), NULL));
	(old_variable_name_map == NULL) ? NULL : (old_variable_name_map = (gee_collection_object_unref (old_variable_name_map), NULL));
	(old_try == NULL) ? NULL : (old_try = (vala_code_node_unref (old_try), NULL));
	(creturn_type == NULL) ? NULL : (creturn_type = (vala_code_node_unref (creturn_type), NULL));
	(cparam_map == NULL) ? NULL : (cparam_map = (gee_collection_object_unref (cparam_map), NULL));
}


static void vala_ccode_method_module_real_generate_parameter (ValaCCodeMethodModule* self, ValaFormalParameter* param, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, GeeMap* carg_map) {
	gboolean _tmp8_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	g_return_if_fail (decl_space != NULL);
	g_return_if_fail (cparam_map != NULL);
	if (!vala_formal_parameter_get_ellipsis (param)) {
		char* ctypename;
		ValaCCodeFormalParameter* _tmp6_;
		char* _tmp5_;
		ctypename = vala_data_type_get_cname (vala_formal_parameter_get_parameter_type (param));
		vala_ccode_base_module_generate_type_declaration ((ValaCCodeBaseModule*) self, vala_formal_parameter_get_parameter_type (param), decl_space);
		/* pass non-simple structs always by reference*/
		if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param)))) {
			ValaStruct* _tmp0_;
			ValaStruct* st;
			gboolean _tmp1_;
			_tmp0_ = NULL;
			st = (_tmp0_ = VALA_STRUCT (vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (param))), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
			_tmp1_ = FALSE;
			if (!vala_struct_is_simple_type (st)) {
				_tmp1_ = vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN;
			} else {
				_tmp1_ = FALSE;
			}
			if (_tmp1_) {
				if (vala_struct_get_use_const (st)) {
					char* _tmp2_;
					_tmp2_ = NULL;
					ctypename = (_tmp2_ = g_strconcat ("const ", ctypename, NULL), ctypename = (g_free (ctypename), NULL), _tmp2_);
				}
				if (!vala_data_type_get_nullable (vala_formal_parameter_get_parameter_type (param))) {
					char* _tmp3_;
					_tmp3_ = NULL;
					ctypename = (_tmp3_ = g_strconcat (ctypename, "*", NULL), ctypename = (g_free (ctypename), NULL), _tmp3_);
				}
			}
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		if (vala_formal_parameter_get_direction (param) != VALA_PARAMETER_DIRECTION_IN) {
			char* _tmp4_;
			_tmp4_ = NULL;
			ctypename = (_tmp4_ = g_strconcat (ctypename, "*", NULL), ctypename = (g_free (ctypename), NULL), _tmp4_);
		}
		_tmp6_ = NULL;
		_tmp5_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) param, (ValaCCodeNode*) (_tmp6_ = vala_ccode_formal_parameter_new (_tmp5_ = vala_ccode_base_module_get_variable_cname ((ValaCCodeBaseModule*) self, vala_symbol_get_name ((ValaSymbol*) param)), ctypename)));
		(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_ccode_node_unref (_tmp6_), NULL));
		_tmp5_ = (g_free (_tmp5_), NULL);
		ctypename = (g_free (ctypename), NULL);
	} else {
		ValaCCodeFormalParameter* _tmp7_;
		_tmp7_ = NULL;
		vala_code_node_set_ccodenode ((ValaCodeNode*) param, (ValaCCodeNode*) (_tmp7_ = vala_ccode_formal_parameter_new_with_ellipsis ()));
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_ccode_node_unref (_tmp7_), NULL));
	}
	gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_formal_parameter_get_cparameter_position (param), FALSE)), VALA_CCODE_FORMAL_PARAMETER (vala_code_node_get_ccodenode ((ValaCodeNode*) param)));
	_tmp8_ = FALSE;
	if (carg_map != NULL) {
		_tmp8_ = !vala_formal_parameter_get_ellipsis (param);
	} else {
		_tmp8_ = FALSE;
	}
	if (_tmp8_) {
		ValaCCodeExpression* _tmp9_;
		_tmp9_ = NULL;
		gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_formal_parameter_get_cparameter_position (param), FALSE)), _tmp9_ = vala_ccode_base_module_get_variable_cexpression ((ValaCCodeBaseModule*) self, vala_symbol_get_name ((ValaSymbol*) param)));
		(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_ccode_node_unref (_tmp9_), NULL));
	}
}


void vala_ccode_method_module_generate_parameter (ValaCCodeMethodModule* self, ValaFormalParameter* param, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, GeeMap* carg_map) {
	VALA_CCODE_METHOD_MODULE_GET_CLASS (self)->generate_parameter (self, param, decl_space, cparam_map, carg_map);
}


static void vala_ccode_method_module_real_generate_cparameters (ValaCCodeBaseModule* base, ValaMethod* m, ValaCCodeDeclarationSpace* decl_space, GeeMap* cparam_map, ValaCCodeFunction* func, ValaCCodeFunctionDeclarator* vdeclarator, GeeMap* carg_map, ValaCCodeFunctionCall* vcall, gint direction) {
	ValaCCodeMethodModule * self;
	gboolean _tmp0_;
	gint last_pos;
	gint min_pos;
	self = (ValaCCodeMethodModule*) base;
	g_return_if_fail (m != NULL);
	g_return_if_fail (decl_space != NULL);
	g_return_if_fail (cparam_map != NULL);
	g_return_if_fail (func != NULL);
	_tmp0_ = FALSE;
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m))) {
		_tmp0_ = VALA_IS_CREATION_METHOD (m);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		ValaClass* _tmp1_;
		ValaClass* cl;
		gboolean _tmp2_;
		_tmp1_ = NULL;
		cl = (_tmp1_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
		_tmp2_ = FALSE;
		if (!vala_class_get_is_compact (cl)) {
			_tmp2_ = vcall == NULL;
		} else {
			_tmp2_ = FALSE;
		}
		if (_tmp2_) {
			ValaCCodeFormalParameter* _tmp3_;
			_tmp3_ = NULL;
			gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_cinstance_parameter_position (m), FALSE)), _tmp3_ = vala_ccode_formal_parameter_new ("object_type", "GType"));
			(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_ccode_node_unref (_tmp3_), NULL));
		}
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	} else {
		gboolean _tmp4_;
		_tmp4_ = FALSE;
		if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
			_tmp4_ = TRUE;
		} else {
			gboolean _tmp5_;
			_tmp5_ = FALSE;
			if (VALA_IS_STRUCT (vala_symbol_get_parent_symbol ((ValaSymbol*) m))) {
				_tmp5_ = VALA_IS_CREATION_METHOD (m);
			} else {
				_tmp5_ = FALSE;
			}
			_tmp4_ = _tmp5_;
		}
		if (_tmp4_) {
			ValaTypeSymbol* parent_type;
			ValaDataType* this_type;
			ValaCCodeFormalParameter* instance_param;
			gboolean _tmp10_;
			gboolean _tmp11_;
			parent_type = vala_ccode_method_module_find_parent_type (self, (ValaSymbol*) m);
			this_type = NULL;
			if (VALA_IS_CLASS (parent_type)) {
				ValaDataType* _tmp6_;
				_tmp6_ = NULL;
				this_type = (_tmp6_ = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) VALA_CLASS (parent_type)), (this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL)), _tmp6_);
			} else {
				if (VALA_IS_INTERFACE (parent_type)) {
					ValaDataType* _tmp7_;
					_tmp7_ = NULL;
					this_type = (_tmp7_ = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) VALA_INTERFACE (parent_type)), (this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL)), _tmp7_);
				} else {
					if (VALA_IS_STRUCT (parent_type)) {
						ValaDataType* _tmp8_;
						_tmp8_ = NULL;
						this_type = (_tmp8_ = (ValaDataType*) vala_struct_value_type_new (VALA_STRUCT (parent_type)), (this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL)), _tmp8_);
					} else {
						if (VALA_IS_ENUM (parent_type)) {
							ValaDataType* _tmp9_;
							_tmp9_ = NULL;
							this_type = (_tmp9_ = (ValaDataType*) vala_enum_value_type_new (VALA_ENUM (parent_type)), (this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL)), _tmp9_);
						} else {
							g_assert_not_reached ();
						}
					}
				}
			}
			vala_ccode_base_module_generate_type_declaration ((ValaCCodeBaseModule*) self, this_type, decl_space);
			instance_param = NULL;
			_tmp10_ = FALSE;
			_tmp11_ = FALSE;
			if (vala_method_get_base_interface_method (m) != NULL) {
				_tmp11_ = !vala_method_get_is_abstract (m);
			} else {
				_tmp11_ = FALSE;
			}
			if (_tmp11_) {
				_tmp10_ = !vala_method_get_is_virtual (m);
			} else {
				_tmp10_ = FALSE;
			}
			if (_tmp10_) {
				ValaObjectType* base_type;
				ValaCCodeFormalParameter* _tmp13_;
				char* _tmp12_;
				base_type = vala_object_type_new ((ValaObjectTypeSymbol*) VALA_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_method_get_base_interface_method (m))));
				_tmp13_ = NULL;
				_tmp12_ = NULL;
				instance_param = (_tmp13_ = vala_ccode_formal_parameter_new ("base", _tmp12_ = vala_data_type_get_cname ((ValaDataType*) base_type)), (instance_param == NULL) ? NULL : (instance_param = (vala_ccode_node_unref (instance_param), NULL)), _tmp13_);
				_tmp12_ = (g_free (_tmp12_), NULL);
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
			} else {
				if (vala_method_get_overrides (m)) {
					ValaObjectType* base_type;
					ValaCCodeFormalParameter* _tmp15_;
					char* _tmp14_;
					base_type = vala_object_type_new ((ValaObjectTypeSymbol*) VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_method_get_base_method (m))));
					_tmp15_ = NULL;
					_tmp14_ = NULL;
					instance_param = (_tmp15_ = vala_ccode_formal_parameter_new ("base", _tmp14_ = vala_data_type_get_cname ((ValaDataType*) base_type)), (instance_param == NULL) ? NULL : (instance_param = (vala_ccode_node_unref (instance_param), NULL)), _tmp15_);
					_tmp14_ = (g_free (_tmp14_), NULL);
					(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
				} else {
					gboolean _tmp16_;
					_tmp16_ = FALSE;
					if (VALA_IS_STRUCT (vala_symbol_get_parent_symbol ((ValaSymbol*) m))) {
						_tmp16_ = !vala_struct_is_simple_type (VALA_STRUCT (vala_symbol_get_parent_symbol ((ValaSymbol*) m)));
					} else {
						_tmp16_ = FALSE;
					}
					if (_tmp16_) {
						ValaCCodeFormalParameter* _tmp18_;
						char* _tmp17_;
						_tmp18_ = NULL;
						_tmp17_ = NULL;
						instance_param = (_tmp18_ = vala_ccode_formal_parameter_new ("*self", _tmp17_ = vala_data_type_get_cname (this_type)), (instance_param == NULL) ? NULL : (instance_param = (vala_ccode_node_unref (instance_param), NULL)), _tmp18_);
						_tmp17_ = (g_free (_tmp17_), NULL);
					} else {
						ValaCCodeFormalParameter* _tmp20_;
						char* _tmp19_;
						_tmp20_ = NULL;
						_tmp19_ = NULL;
						instance_param = (_tmp20_ = vala_ccode_formal_parameter_new ("self", _tmp19_ = vala_data_type_get_cname (this_type)), (instance_param == NULL) ? NULL : (instance_param = (vala_ccode_node_unref (instance_param), NULL)), _tmp20_);
						_tmp19_ = (g_free (_tmp19_), NULL);
					}
				}
			}
			gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_cinstance_parameter_position (m), FALSE)), instance_param);
			(parent_type == NULL) ? NULL : (parent_type = (vala_code_node_unref (parent_type), NULL));
			(this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL));
			(instance_param == NULL) ? NULL : (instance_param = (vala_ccode_node_unref (instance_param), NULL));
		} else {
			if (vala_method_get_binding (m) == MEMBER_BINDING_CLASS) {
				ValaTypeSymbol* parent_type;
				ValaDataType* this_type;
				ValaDataType* _tmp21_;
				char* _tmp22_;
				ValaCCodeFormalParameter* _tmp23_;
				ValaCCodeFormalParameter* class_param;
				parent_type = vala_ccode_method_module_find_parent_type (self, (ValaSymbol*) m);
				this_type = NULL;
				_tmp21_ = NULL;
				this_type = (_tmp21_ = (ValaDataType*) vala_class_type_new (VALA_CLASS (parent_type)), (this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL)), _tmp21_);
				_tmp22_ = NULL;
				_tmp23_ = NULL;
				class_param = (_tmp23_ = vala_ccode_formal_parameter_new ("klass", _tmp22_ = vala_data_type_get_cname (this_type)), _tmp22_ = (g_free (_tmp22_), NULL), _tmp23_);
				gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_cinstance_parameter_position (m), FALSE)), class_param);
				(parent_type == NULL) ? NULL : (parent_type = (vala_code_node_unref (parent_type), NULL));
				(this_type == NULL) ? NULL : (this_type = (vala_code_node_unref (this_type), NULL));
				(class_param == NULL) ? NULL : (class_param = (vala_ccode_node_unref (class_param), NULL));
			}
		}
	}
	if (vala_ccode_method_module_is_gtypeinstance_creation_method (self, m)) {
		gint type_param_index;
		ValaClass* _tmp24_;
		ValaClass* cl;
		/* memory management for generic types*/
		type_param_index = 0;
		_tmp24_ = NULL;
		cl = (_tmp24_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp24_ == NULL) ? NULL : vala_code_node_ref (_tmp24_));
		{
			GeeList* _tmp25_;
			GeeIterator* _tmp26_;
			GeeIterator* _type_param_it;
			_tmp25_ = NULL;
			_tmp26_ = NULL;
			_type_param_it = (_tmp26_ = gee_iterable_iterator ((GeeIterable*) (_tmp25_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) cl))), (_tmp25_ == NULL) ? NULL : (_tmp25_ = (gee_collection_object_unref (_tmp25_), NULL)), _tmp26_);
			while (gee_iterator_next (_type_param_it)) {
				ValaTypeParameter* type_param;
				ValaCCodeFormalParameter* _tmp29_;
				char* _tmp28_;
				char* _tmp27_;
				ValaCCodeFormalParameter* _tmp32_;
				char* _tmp31_;
				char* _tmp30_;
				ValaCCodeFormalParameter* _tmp35_;
				char* _tmp34_;
				char* _tmp33_;
				type_param = (ValaTypeParameter*) gee_iterator_get (_type_param_it);
				_tmp29_ = NULL;
				_tmp28_ = NULL;
				_tmp27_ = NULL;
				gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (0.1 * type_param_index) + 0.01, FALSE)), _tmp29_ = vala_ccode_formal_parameter_new (_tmp28_ = g_strdup_printf ("%s_type", _tmp27_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1)), "GType"));
				(_tmp29_ == NULL) ? NULL : (_tmp29_ = (vala_ccode_node_unref (_tmp29_), NULL));
				_tmp28_ = (g_free (_tmp28_), NULL);
				_tmp27_ = (g_free (_tmp27_), NULL);
				_tmp32_ = NULL;
				_tmp31_ = NULL;
				_tmp30_ = NULL;
				gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (0.1 * type_param_index) + 0.02, FALSE)), _tmp32_ = vala_ccode_formal_parameter_new (_tmp31_ = g_strdup_printf ("%s_dup_func", _tmp30_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1)), "GBoxedCopyFunc"));
				(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
				_tmp31_ = (g_free (_tmp31_), NULL);
				_tmp30_ = (g_free (_tmp30_), NULL);
				_tmp35_ = NULL;
				_tmp34_ = NULL;
				_tmp33_ = NULL;
				gee_map_set (cparam_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (0.1 * type_param_index) + 0.03, FALSE)), _tmp35_ = vala_ccode_formal_parameter_new (_tmp34_ = g_strdup_printf ("%s_destroy_func", _tmp33_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1)), "GDestroyNotify"));
				(_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL));
				_tmp34_ = (g_free (_tmp34_), NULL);
				_tmp33_ = (g_free (_tmp33_), NULL);
				if (carg_map != NULL) {
					ValaCCodeIdentifier* _tmp38_;
					char* _tmp37_;
					char* _tmp36_;
					ValaCCodeIdentifier* _tmp41_;
					char* _tmp40_;
					char* _tmp39_;
					ValaCCodeIdentifier* _tmp44_;
					char* _tmp43_;
					char* _tmp42_;
					_tmp38_ = NULL;
					_tmp37_ = NULL;
					_tmp36_ = NULL;
					gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (0.1 * type_param_index) + 0.01, FALSE)), (ValaCCodeExpression*) (_tmp38_ = vala_ccode_identifier_new (_tmp37_ = g_strdup_printf ("%s_type", _tmp36_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1)))));
					(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
					_tmp37_ = (g_free (_tmp37_), NULL);
					_tmp36_ = (g_free (_tmp36_), NULL);
					_tmp41_ = NULL;
					_tmp40_ = NULL;
					_tmp39_ = NULL;
					gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (0.1 * type_param_index) + 0.02, FALSE)), (ValaCCodeExpression*) (_tmp41_ = vala_ccode_identifier_new (_tmp40_ = g_strdup_printf ("%s_dup_func", _tmp39_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1)))));
					(_tmp41_ == NULL) ? NULL : (_tmp41_ = (vala_ccode_node_unref (_tmp41_), NULL));
					_tmp40_ = (g_free (_tmp40_), NULL);
					_tmp39_ = (g_free (_tmp39_), NULL);
					_tmp44_ = NULL;
					_tmp43_ = NULL;
					_tmp42_ = NULL;
					gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, (0.1 * type_param_index) + 0.03, FALSE)), (ValaCCodeExpression*) (_tmp44_ = vala_ccode_identifier_new (_tmp43_ = g_strdup_printf ("%s_destroy_func", _tmp42_ = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) type_param), -1)))));
					(_tmp44_ == NULL) ? NULL : (_tmp44_ = (vala_ccode_node_unref (_tmp44_), NULL));
					_tmp43_ = (g_free (_tmp43_), NULL);
					_tmp42_ = (g_free (_tmp42_), NULL);
				}
				type_param_index++;
				(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
			}
			(_type_param_it == NULL) ? NULL : (_type_param_it = (gee_collection_object_unref (_type_param_it), NULL));
		}
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	}
	{
		GeeList* _tmp45_;
		GeeIterator* _tmp46_;
		GeeIterator* _param_it;
		_tmp45_ = NULL;
		_tmp46_ = NULL;
		_param_it = (_tmp46_ = gee_iterable_iterator ((GeeIterable*) (_tmp45_ = vala_method_get_parameters (m))), (_tmp45_ == NULL) ? NULL : (_tmp45_ = (gee_collection_object_unref (_tmp45_), NULL)), _tmp46_);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			if (vala_formal_parameter_get_direction (param) != VALA_PARAMETER_DIRECTION_OUT) {
				if ((direction & 1) == 0) {
					/* no in paramters*/
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
					continue;
				}
			} else {
				if ((direction & 2) == 0) {
					/* no out paramters*/
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
					continue;
				}
			}
			vala_ccode_method_module_generate_parameter (self, param, decl_space, cparam_map, carg_map);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	if ((direction & 2) != 0) {
		vala_ccode_method_module_generate_method_result_declaration (self, m, decl_space, func, cparam_map, carg_map);
	}
	/* append C parameters in the right order*/
	last_pos = -1;
	min_pos = 0;
	while (TRUE) {
		ValaCCodeFormalParameter* _tmp51_;
		min_pos = -1;
		{
			GeeSet* _tmp47_;
			GeeIterator* _tmp48_;
			GeeIterator* _pos_it;
			_tmp47_ = NULL;
			_tmp48_ = NULL;
			_pos_it = (_tmp48_ = gee_iterable_iterator ((GeeIterable*) (_tmp47_ = gee_map_get_keys (cparam_map))), (_tmp47_ == NULL) ? NULL : (_tmp47_ = (gee_collection_object_unref (_tmp47_), NULL)), _tmp48_);
			while (gee_iterator_next (_pos_it)) {
				gint pos;
				gboolean _tmp49_;
				pos = GPOINTER_TO_INT (gee_iterator_get (_pos_it));
				_tmp49_ = FALSE;
				if (pos > last_pos) {
					gboolean _tmp50_;
					_tmp50_ = FALSE;
					if (min_pos == (-1)) {
						_tmp50_ = TRUE;
					} else {
						_tmp50_ = pos < min_pos;
					}
					_tmp49_ = _tmp50_;
				} else {
					_tmp49_ = FALSE;
				}
				if (_tmp49_) {
					min_pos = pos;
				}
			}
			(_pos_it == NULL) ? NULL : (_pos_it = (gee_collection_object_unref (_pos_it), NULL));
		}
		if (min_pos == (-1)) {
			break;
		}
		_tmp51_ = NULL;
		vala_ccode_function_add_parameter (func, _tmp51_ = (ValaCCodeFormalParameter*) gee_map_get (cparam_map, GINT_TO_POINTER (min_pos)));
		(_tmp51_ == NULL) ? NULL : (_tmp51_ = (vala_ccode_node_unref (_tmp51_), NULL));
		if (vdeclarator != NULL) {
			ValaCCodeFormalParameter* _tmp52_;
			_tmp52_ = NULL;
			vala_ccode_function_declarator_add_parameter (vdeclarator, _tmp52_ = (ValaCCodeFormalParameter*) gee_map_get (cparam_map, GINT_TO_POINTER (min_pos)));
			(_tmp52_ == NULL) ? NULL : (_tmp52_ = (vala_ccode_node_unref (_tmp52_), NULL));
		}
		if (vcall != NULL) {
			ValaCCodeExpression* arg;
			arg = (ValaCCodeExpression*) gee_map_get (carg_map, GINT_TO_POINTER (min_pos));
			if (arg != NULL) {
				vala_ccode_function_call_add_argument (vcall, arg);
			}
			(arg == NULL) ? NULL : (arg = (vala_ccode_node_unref (arg), NULL));
		}
		last_pos = min_pos;
	}
}


void vala_ccode_method_module_generate_vfunc (ValaCCodeMethodModule* self, ValaMethod* m, ValaDataType* return_type, GeeMap* cparam_map, GeeMap* carg_map, const char* suffix, gint direction) {
	char* _tmp1_;
	char* _tmp0_;
	ValaCCodeFunction* _tmp2_;
	ValaCCodeFunction* vfunc;
	ValaCCodeBlock* vblock;
	ValaCCodeFunctionCall* vcast;
	ValaCCodeIdentifier* _tmp15_;
	ValaCCodeMemberAccess* _tmp17_;
	char* _tmp16_;
	ValaCCodeFunctionCall* _tmp18_;
	ValaCCodeFunctionCall* vcall;
	ValaCCodeIdentifier* _tmp19_;
	ValaCCodeStatement* cstmt;
	GeeList* _tmp30_;
	gboolean _tmp31_;
	gboolean _tmp37_;
	gboolean _tmp38_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (return_type != NULL);
	g_return_if_fail (cparam_map != NULL);
	g_return_if_fail (carg_map != NULL);
	g_return_if_fail (suffix != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	vfunc = (_tmp2_ = vala_ccode_function_new (_tmp1_ = g_strconcat (_tmp0_ = vala_method_get_cname (m), suffix, NULL), "void"), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	vala_ccode_node_set_line ((ValaCCodeNode*) vfunc, vala_ccode_node_get_line ((ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function));
	vblock = vala_ccode_block_new ();
	{
		GeeList* _tmp3_;
		GeeIterator* _tmp4_;
		GeeIterator* _precondition_it;
		_tmp3_ = NULL;
		_tmp4_ = NULL;
		_precondition_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = vala_method_get_preconditions (m))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), _tmp4_);
		while (gee_iterator_next (_precondition_it)) {
			ValaExpression* precondition;
			ValaCCodeStatement* check_stmt;
			precondition = (ValaExpression*) gee_iterator_get (_precondition_it);
			check_stmt = vala_ccode_method_module_create_precondition_statement (self, (ValaCodeNode*) m, return_type, precondition);
			if (check_stmt != NULL) {
				vala_ccode_block_add_statement (vblock, (ValaCCodeNode*) check_stmt);
			}
			(precondition == NULL) ? NULL : (precondition = (vala_code_node_unref (precondition), NULL));
			(check_stmt == NULL) ? NULL : (check_stmt = (vala_ccode_node_unref (check_stmt), NULL));
		}
		(_precondition_it == NULL) ? NULL : (_precondition_it = (gee_collection_object_unref (_precondition_it), NULL));
	}
	vcast = NULL;
	if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) m))) {
		ValaInterface* _tmp5_;
		ValaInterface* iface;
		ValaCCodeFunctionCall* _tmp9_;
		ValaCCodeIdentifier* _tmp8_;
		char* _tmp7_;
		char* _tmp6_;
		_tmp5_ = NULL;
		iface = (_tmp5_ = VALA_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
		_tmp9_ = NULL;
		_tmp8_ = NULL;
		_tmp7_ = NULL;
		_tmp6_ = NULL;
		vcast = (_tmp9_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp8_ = vala_ccode_identifier_new (_tmp7_ = g_strdup_printf ("%s_GET_INTERFACE", _tmp6_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) iface, NULL))))), (vcast == NULL) ? NULL : (vcast = (vala_ccode_node_unref (vcast), NULL)), _tmp9_);
		(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_ccode_node_unref (_tmp8_), NULL));
		_tmp7_ = (g_free (_tmp7_), NULL);
		_tmp6_ = (g_free (_tmp6_), NULL);
		(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
	} else {
		ValaClass* _tmp10_;
		ValaClass* cl;
		ValaCCodeFunctionCall* _tmp14_;
		ValaCCodeIdentifier* _tmp13_;
		char* _tmp12_;
		char* _tmp11_;
		_tmp10_ = NULL;
		cl = (_tmp10_ = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp10_ == NULL) ? NULL : vala_code_node_ref (_tmp10_));
		_tmp14_ = NULL;
		_tmp13_ = NULL;
		_tmp12_ = NULL;
		_tmp11_ = NULL;
		vcast = (_tmp14_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_identifier_new (_tmp12_ = g_strdup_printf ("%s_GET_CLASS", _tmp11_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) cl, NULL))))), (vcast == NULL) ? NULL : (vcast = (vala_ccode_node_unref (vcast), NULL)), _tmp14_);
		(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL));
		_tmp12_ = (g_free (_tmp12_), NULL);
		_tmp11_ = (g_free (_tmp11_), NULL);
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	}
	_tmp15_ = NULL;
	vala_ccode_function_call_add_argument (vcast, (ValaCCodeExpression*) (_tmp15_ = vala_ccode_identifier_new ("self")));
	(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
	_tmp17_ = NULL;
	_tmp16_ = NULL;
	_tmp18_ = NULL;
	vcall = (_tmp18_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp17_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) vcast, _tmp16_ = g_strconcat (vala_method_get_vfunc_name (m), suffix, NULL)))), (_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL)), _tmp16_ = (g_free (_tmp16_), NULL), _tmp18_);
	_tmp19_ = NULL;
	gee_map_set (carg_map, GINT_TO_POINTER (vala_ccode_base_module_get_param_pos ((ValaCCodeBaseModule*) self, vala_method_get_cinstance_parameter_position (m), FALSE)), (ValaCCodeExpression*) (_tmp19_ = vala_ccode_identifier_new ("self")));
	(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
	vala_ccode_base_module_generate_cparameters ((ValaCCodeBaseModule*) self, m, ((ValaCCodeBaseModule*) self)->source_declarations, cparam_map, vfunc, NULL, carg_map, vcall, direction);
	cstmt = NULL;
	if (VALA_IS_VOID_TYPE (return_type)) {
		ValaCCodeStatement* _tmp20_;
		_tmp20_ = NULL;
		cstmt = (_tmp20_ = (ValaCCodeStatement*) vala_ccode_expression_statement_new ((ValaCCodeExpression*) vcall), (cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL)), _tmp20_);
	} else {
		GeeList* _tmp21_;
		gboolean _tmp22_;
		_tmp21_ = NULL;
		if ((_tmp22_ = gee_collection_get_size ((GeeCollection*) (_tmp21_ = vala_method_get_postconditions (m))) == 0, (_tmp21_ == NULL) ? NULL : (_tmp21_ = (gee_collection_object_unref (_tmp21_), NULL)), _tmp22_)) {
			ValaCCodeStatement* _tmp23_;
			/* pass method return value */
			_tmp23_ = NULL;
			cstmt = (_tmp23_ = (ValaCCodeStatement*) vala_ccode_return_statement_new ((ValaCCodeExpression*) vcall), (cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL)), _tmp23_);
		} else {
			char* _tmp25_;
			char* _tmp24_;
			ValaCCodeDeclaration* _tmp26_;
			ValaCCodeDeclaration* _cdecl_;
			ValaCCodeVariableDeclarator* _tmp27_;
			ValaCCodeStatement* _tmp29_;
			ValaCCodeStatement* _tmp28_;
			/* store method return value for postconditions */
			_tmp25_ = NULL;
			_tmp24_ = NULL;
			_tmp26_ = NULL;
			_cdecl_ = (_tmp26_ = vala_ccode_declaration_new (_tmp25_ = vala_ccode_method_module_get_creturn_type (self, m, _tmp24_ = vala_data_type_get_cname (return_type))), _tmp25_ = (g_free (_tmp25_), NULL), _tmp24_ = (g_free (_tmp24_), NULL), _tmp26_);
			_tmp27_ = NULL;
			vala_ccode_declaration_add_declarator (_cdecl_, (ValaCCodeDeclarator*) (_tmp27_ = vala_ccode_variable_declarator_new ("result", (ValaCCodeExpression*) vcall, NULL)));
			(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
			_tmp29_ = NULL;
			_tmp28_ = NULL;
			cstmt = (_tmp29_ = (_tmp28_ = (ValaCCodeStatement*) _cdecl_, (_tmp28_ == NULL) ? NULL : vala_ccode_node_ref (_tmp28_)), (cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL)), _tmp29_);
			(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
		}
	}
	vala_ccode_node_set_line ((ValaCCodeNode*) cstmt, vala_ccode_node_get_line ((ValaCCodeNode*) vfunc));
	vala_ccode_block_add_statement (vblock, (ValaCCodeNode*) cstmt);
	_tmp30_ = NULL;
	if ((_tmp31_ = gee_collection_get_size ((GeeCollection*) (_tmp30_ = vala_method_get_postconditions (m))) > 0, (_tmp30_ == NULL) ? NULL : (_tmp30_ = (gee_collection_object_unref (_tmp30_), NULL)), _tmp31_)) {
		{
			GeeList* _tmp32_;
			GeeIterator* _tmp33_;
			GeeIterator* _postcondition_it;
			_tmp32_ = NULL;
			_tmp33_ = NULL;
			_postcondition_it = (_tmp33_ = gee_iterable_iterator ((GeeIterable*) (_tmp32_ = vala_method_get_postconditions (m))), (_tmp32_ == NULL) ? NULL : (_tmp32_ = (gee_collection_object_unref (_tmp32_), NULL)), _tmp33_);
			while (gee_iterator_next (_postcondition_it)) {
				ValaExpression* postcondition;
				ValaCCodeStatement* _tmp34_;
				postcondition = (ValaExpression*) gee_iterator_get (_postcondition_it);
				_tmp34_ = NULL;
				vala_ccode_block_add_statement (vblock, (ValaCCodeNode*) (_tmp34_ = vala_ccode_method_module_create_postcondition_statement (self, postcondition)));
				(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
				(postcondition == NULL) ? NULL : (postcondition = (vala_code_node_unref (postcondition), NULL));
			}
			(_postcondition_it == NULL) ? NULL : (_postcondition_it = (gee_collection_object_unref (_postcondition_it), NULL));
		}
		if (!VALA_IS_VOID_TYPE (return_type)) {
			ValaCCodeIdentifier* _tmp35_;
			ValaCCodeReturnStatement* _tmp36_;
			ValaCCodeReturnStatement* cret_stmt;
			_tmp35_ = NULL;
			_tmp36_ = NULL;
			cret_stmt = (_tmp36_ = vala_ccode_return_statement_new ((ValaCCodeExpression*) (_tmp35_ = vala_ccode_identifier_new ("result"))), (_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL)), _tmp36_);
			vala_ccode_node_set_line ((ValaCCodeNode*) cret_stmt, vala_ccode_node_get_line ((ValaCCodeNode*) vfunc));
			vala_ccode_block_add_statement (vblock, (ValaCCodeNode*) cret_stmt);
			(cret_stmt == NULL) ? NULL : (cret_stmt = (vala_ccode_node_unref (cret_stmt), NULL));
		}
	}
	vala_ccode_function_set_block (vfunc, vblock);
	_tmp37_ = FALSE;
	_tmp38_ = FALSE;
	if (vala_method_get_is_abstract (m)) {
		_tmp38_ = vala_code_node_get_source_reference ((ValaCodeNode*) m) != NULL;
	} else {
		_tmp38_ = FALSE;
	}
	if (_tmp38_) {
		_tmp37_ = vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) m)) != NULL;
	} else {
		_tmp37_ = FALSE;
	}
	if (_tmp37_) {
		ValaCCodeComment* _tmp39_;
		_tmp39_ = NULL;
		vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) (_tmp39_ = vala_ccode_comment_new (vala_source_reference_get_comment (vala_code_node_get_source_reference ((ValaCodeNode*) m)))));
		(_tmp39_ == NULL) ? NULL : (_tmp39_ = (vala_ccode_node_unref (_tmp39_), NULL));
	}
	vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) vfunc);
	(vfunc == NULL) ? NULL : (vfunc = (vala_ccode_node_unref (vfunc), NULL));
	(vblock == NULL) ? NULL : (vblock = (vala_ccode_node_unref (vblock), NULL));
	(vcast == NULL) ? NULL : (vcast = (vala_ccode_node_unref (vcast), NULL));
	(vcall == NULL) ? NULL : (vcall = (vala_ccode_node_unref (vcall), NULL));
	(cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL));
}


static ValaCCodeStatement* vala_ccode_method_module_create_method_type_check_statement (ValaCCodeMethodModule* self, ValaMethod* m, ValaDataType* return_type, ValaTypeSymbol* t, gboolean non_null, const char* var_name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (m != NULL, NULL);
	g_return_val_if_fail (return_type != NULL, NULL);
	g_return_val_if_fail (t != NULL, NULL);
	g_return_val_if_fail (var_name != NULL, NULL);
	if (vala_method_get_coroutine (m)) {
		return NULL;
	} else {
		return vala_ccode_base_module_create_type_check_statement ((ValaCCodeBaseModule*) self, (ValaCodeNode*) m, return_type, t, non_null, var_name);
	}
}


static ValaCCodeStatement* vala_ccode_method_module_create_precondition_statement (ValaCCodeMethodModule* self, ValaCodeNode* method_node, ValaDataType* ret_type, ValaExpression* precondition) {
	ValaCCodeFunctionCall* ccheck;
	ValaCCodeStatement* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (method_node != NULL, NULL);
	g_return_val_if_fail (ret_type != NULL, NULL);
	g_return_val_if_fail (precondition != NULL, NULL);
	ccheck = vala_ccode_function_call_new (NULL);
	vala_ccode_function_call_add_argument (ccheck, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) precondition)));
	if (VALA_IS_VOID_TYPE (ret_type)) {
		ValaCCodeIdentifier* _tmp0_;
		/* void function */
		_tmp0_ = NULL;
		vala_ccode_function_call_set_call (ccheck, (ValaCCodeExpression*) (_tmp0_ = vala_ccode_identifier_new ("g_return_if_fail")));
		(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL));
	} else {
		ValaCCodeIdentifier* _tmp1_;
		ValaCCodeExpression* cdefault;
		_tmp1_ = NULL;
		vala_ccode_function_call_set_call (ccheck, (ValaCCodeExpression*) (_tmp1_ = vala_ccode_identifier_new ("g_return_val_if_fail")));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_ccode_node_unref (_tmp1_), NULL));
		cdefault = vala_ccode_base_module_default_value_for_type ((ValaCCodeBaseModule*) self, ret_type, FALSE);
		if (cdefault != NULL) {
			vala_ccode_function_call_add_argument (ccheck, cdefault);
		} else {
			ValaCCodeStatement* _tmp2_;
			_tmp2_ = NULL;
			return (_tmp2_ = NULL, (cdefault == NULL) ? NULL : (cdefault = (vala_ccode_node_unref (cdefault), NULL)), (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), _tmp2_);
		}
		(cdefault == NULL) ? NULL : (cdefault = (vala_ccode_node_unref (cdefault), NULL));
	}
	_tmp3_ = NULL;
	return (_tmp3_ = (ValaCCodeStatement*) vala_ccode_expression_statement_new ((ValaCCodeExpression*) ccheck), (ccheck == NULL) ? NULL : (ccheck = (vala_ccode_node_unref (ccheck), NULL)), _tmp3_);
}


static ValaCCodeStatement* vala_ccode_method_module_create_postcondition_statement (ValaCCodeMethodModule* self, ValaExpression* postcondition) {
	ValaCCodeIdentifier* _tmp0_;
	ValaCCodeFunctionCall* _tmp1_;
	ValaCCodeFunctionCall* cassert;
	ValaCCodeStatement* _tmp2_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (postcondition != NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	cassert = (_tmp1_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp0_ = vala_ccode_identifier_new ("g_assert"))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_ccode_node_unref (_tmp0_), NULL)), _tmp1_);
	vala_ccode_function_call_add_argument (cassert, VALA_CCODE_EXPRESSION (vala_code_node_get_ccodenode ((ValaCodeNode*) postcondition)));
	_tmp2_ = NULL;
	return (_tmp2_ = (ValaCCodeStatement*) vala_ccode_expression_statement_new ((ValaCCodeExpression*) cassert), (cassert == NULL) ? NULL : (cassert = (vala_ccode_node_unref (cassert), NULL)), _tmp2_);
}


static ValaTypeSymbol* vala_ccode_method_module_find_parent_type (ValaCCodeMethodModule* self, ValaSymbol* sym) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sym != NULL, NULL);
	while (sym != NULL) {
		if (VALA_IS_TYPESYMBOL (sym)) {
			ValaTypeSymbol* _tmp0_;
			_tmp0_ = NULL;
			return (_tmp0_ = VALA_TYPESYMBOL (sym), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		}
		sym = vala_symbol_get_parent_symbol (sym);
	}
	return NULL;
}


static void vala_ccode_method_module_add_object_creation (ValaCCodeMethodModule* self, ValaCCodeBlock* b, gboolean has_params) {
	ValaClass* _tmp0_;
	ValaClass* cl;
	gboolean chain_up;
	ValaCreationMethod* cm;
	gboolean _tmp7_;
	gboolean _tmp8_;
	ValaCCodeVariableDeclarator* _cdecl_;
	char* _tmp24_;
	char* _tmp23_;
	ValaCCodeDeclaration* _tmp25_;
	ValaCCodeDeclaration* cdeclaration;
	g_return_if_fail (self != NULL);
	g_return_if_fail (b != NULL);
	_tmp0_ = NULL;
	cl = (_tmp0_ = VALA_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	chain_up = FALSE;
	cm = NULL;
	if (vala_class_get_base_class (cl) != NULL) {
		ValaCreationMethod* _tmp3_;
		ValaCreationMethod* _tmp2_;
		ValaMethod* _tmp1_;
		gboolean _tmp4_;
		gboolean _tmp5_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		cm = (_tmp3_ = (_tmp2_ = (_tmp1_ = vala_class_get_default_construction_method (vala_class_get_base_class (cl)), VALA_IS_CREATION_METHOD (_tmp1_) ? ((ValaCreationMethod*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_)), (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), _tmp3_);
		_tmp4_ = FALSE;
		_tmp5_ = FALSE;
		if (cm != NULL) {
			GeeList* _tmp6_;
			_tmp6_ = NULL;
			_tmp5_ = gee_collection_get_size ((GeeCollection*) (_tmp6_ = vala_method_get_parameters ((ValaMethod*) cm))) == 0;
			(_tmp6_ == NULL) ? NULL : (_tmp6_ = (gee_collection_object_unref (_tmp6_), NULL));
		} else {
			_tmp5_ = FALSE;
		}
		if (_tmp5_) {
			_tmp4_ = vala_method_get_has_construct_function ((ValaMethod*) cm);
		} else {
			_tmp4_ = FALSE;
		}
		if (_tmp4_) {
			if (!has_params) {
				chain_up = TRUE;
			}
		}
	}
	_tmp7_ = FALSE;
	_tmp8_ = FALSE;
	if (!has_params) {
		_tmp8_ = !chain_up;
	} else {
		_tmp8_ = FALSE;
	}
	if (_tmp8_) {
		_tmp7_ = VALA_TYPESYMBOL (vala_class_get_base_class (cl)) != ((ValaCCodeBaseModule*) self)->gobject_type;
	} else {
		_tmp7_ = FALSE;
	}
	if (_tmp7_) {
	}
	/* possibly report warning or error about missing base call*/
	_cdecl_ = vala_ccode_variable_declarator_new ("self", NULL, NULL);
	if (chain_up) {
		ValaCCodeIdentifier* _tmp10_;
		char* _tmp9_;
		ValaCCodeFunctionCall* _tmp11_;
		ValaCCodeFunctionCall* ccall;
		ValaCCodeIdentifier* _tmp12_;
		ValaCCodeCastExpression* _tmp15_;
		char* _tmp14_;
		char* _tmp13_;
		vala_ccode_base_module_generate_method_declaration ((ValaCCodeBaseModule*) self, (ValaMethod*) cm, ((ValaCCodeBaseModule*) self)->source_declarations);
		_tmp10_ = NULL;
		_tmp9_ = NULL;
		_tmp11_ = NULL;
		ccall = (_tmp11_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp10_ = vala_ccode_identifier_new (_tmp9_ = vala_method_get_real_cname ((ValaMethod*) cm)))), (_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_ccode_node_unref (_tmp10_), NULL)), _tmp9_ = (g_free (_tmp9_), NULL), _tmp11_);
		_tmp12_ = NULL;
		vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp12_ = vala_ccode_identifier_new ("object_type")));
		(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_ccode_node_unref (_tmp12_), NULL));
		_tmp15_ = NULL;
		_tmp14_ = NULL;
		_tmp13_ = NULL;
		vala_ccode_variable_declarator_set_initializer (_cdecl_, (ValaCCodeExpression*) (_tmp15_ = vala_ccode_cast_expression_new ((ValaCCodeExpression*) ccall, _tmp14_ = g_strdup_printf ("%s*", _tmp13_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE)))));
		(_tmp15_ == NULL) ? NULL : (_tmp15_ = (vala_ccode_node_unref (_tmp15_), NULL));
		_tmp14_ = (g_free (_tmp14_), NULL);
		_tmp13_ = (g_free (_tmp13_), NULL);
		(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
	} else {
		ValaCCodeIdentifier* _tmp16_;
		ValaCCodeFunctionCall* _tmp17_;
		ValaCCodeFunctionCall* ccall;
		ValaCCodeIdentifier* _tmp18_;
		_tmp16_ = NULL;
		_tmp17_ = NULL;
		ccall = (_tmp17_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new ("g_object_newv"))), (_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL)), _tmp17_);
		_tmp18_ = NULL;
		vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp18_ = vala_ccode_identifier_new ("object_type")));
		(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_ccode_node_unref (_tmp18_), NULL));
		if (has_params) {
			ValaCCodeConstant* _tmp19_;
			ValaCCodeConstant* _tmp20_;
			_tmp19_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp19_ = vala_ccode_constant_new ("__params_it - __params")));
			(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_ccode_node_unref (_tmp19_), NULL));
			_tmp20_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp20_ = vala_ccode_constant_new ("__params")));
			(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_ccode_node_unref (_tmp20_), NULL));
		} else {
			ValaCCodeConstant* _tmp21_;
			ValaCCodeConstant* _tmp22_;
			_tmp21_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp21_ = vala_ccode_constant_new ("0")));
			(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_ccode_node_unref (_tmp21_), NULL));
			_tmp22_ = NULL;
			vala_ccode_function_call_add_argument (ccall, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_constant_new ("NULL")));
			(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL));
		}
		vala_ccode_variable_declarator_set_initializer (_cdecl_, (ValaCCodeExpression*) ccall);
		(ccall == NULL) ? NULL : (ccall = (vala_ccode_node_unref (ccall), NULL));
	}
	_tmp24_ = NULL;
	_tmp23_ = NULL;
	_tmp25_ = NULL;
	cdeclaration = (_tmp25_ = vala_ccode_declaration_new (_tmp24_ = g_strdup_printf ("%s *", _tmp23_ = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE))), _tmp24_ = (g_free (_tmp24_), NULL), _tmp23_ = (g_free (_tmp23_), NULL), _tmp25_);
	vala_ccode_declaration_add_declarator (cdeclaration, (ValaCCodeDeclarator*) _cdecl_);
	vala_ccode_block_add_statement (b, (ValaCCodeNode*) cdeclaration);
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
	(_cdecl_ == NULL) ? NULL : (_cdecl_ = (vala_ccode_node_unref (_cdecl_), NULL));
	(cdeclaration == NULL) ? NULL : (cdeclaration = (vala_ccode_node_unref (cdeclaration), NULL));
}


static void vala_ccode_method_module_real_visit_creation_method (ValaCCodeModule* base, ValaCreationMethod* m) {
	ValaCCodeMethodModule * self;
	gboolean visible;
	gboolean _tmp0_;
	gboolean _tmp1_;
	ValaDataType* creturn_type;
	gboolean _tmp9_;
	gboolean _tmp18_;
	gboolean _tmp19_;
	self = (ValaCCodeMethodModule*) base;
	g_return_if_fail (m != NULL);
	visible = !vala_symbol_is_private_symbol ((ValaSymbol*) m);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (vala_method_get_body ((ValaMethod*) m) != NULL) {
		_tmp1_ = VALA_IS_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol);
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class, ((ValaCCodeBaseModule*) self)->gobject_type);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		gint n_params;
		n_params = 0;
		{
			GeeList* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _stmt_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_stmt_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = vala_block_get_statements (vala_method_get_body ((ValaMethod*) m)))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL)), _tmp3_);
			while (gee_iterator_next (_stmt_it)) {
				ValaStatement* stmt;
				ValaExpressionStatement* _tmp5_;
				ValaStatement* _tmp4_;
				ValaExpressionStatement* expr_stmt;
				stmt = (ValaStatement*) gee_iterator_get (_stmt_it);
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				expr_stmt = (_tmp5_ = (_tmp4_ = stmt, VALA_IS_EXPRESSION_STATEMENT (_tmp4_) ? ((ValaExpressionStatement*) _tmp4_) : NULL), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
				if (expr_stmt != NULL) {
					ValaProperty* prop;
					gboolean _tmp6_;
					prop = vala_expression_statement_assigned_property (expr_stmt);
					_tmp6_ = FALSE;
					if (prop != NULL) {
						_tmp6_ = vala_property_accessor_get_construction (vala_property_get_set_accessor (prop));
					} else {
						_tmp6_ = FALSE;
					}
					if (_tmp6_) {
						n_params++;
					}
					(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				}
				(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
				(expr_stmt == NULL) ? NULL : (expr_stmt = (vala_code_node_unref (expr_stmt), NULL));
			}
			(_stmt_it == NULL) ? NULL : (_stmt_it = (gee_collection_object_unref (_stmt_it), NULL));
		}
		vala_creation_method_set_n_construction_params (m, n_params);
	}
	vala_ccode_module_visit_method (vala_ccode_module_get_head ((ValaCCodeModule*) self), (ValaMethod*) m);
	creturn_type = NULL;
	if (VALA_IS_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol)) {
		ValaDataType* _tmp7_;
		_tmp7_ = NULL;
		creturn_type = (_tmp7_ = (ValaDataType*) vala_object_type_new ((ValaObjectTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class), (creturn_type == NULL) ? NULL : (creturn_type = (vala_code_node_unref (creturn_type), NULL)), _tmp7_);
	} else {
		ValaDataType* _tmp8_;
		_tmp8_ = NULL;
		creturn_type = (_tmp8_ = (ValaDataType*) vala_void_type_new (NULL), (creturn_type == NULL) ? NULL : (creturn_type = (vala_code_node_unref (creturn_type), NULL)), _tmp8_);
	}
	_tmp9_ = FALSE;
	if (VALA_IS_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol)) {
		_tmp9_ = !vala_class_get_is_compact (((ValaCCodeBaseModule*) self)->current_class);
	} else {
		_tmp9_ = FALSE;
	}
	if (_tmp9_) {
		char* _tmp10_;
		ValaCCodeFunction* _tmp11_;
		ValaCCodeFunction* vfunc;
		GeeHashMap* cparam_map;
		GeeHashMap* carg_map;
		ValaCCodeBlock* vblock;
		ValaCCodeIdentifier* _tmp13_;
		char* _tmp12_;
		ValaCCodeFunctionCall* _tmp14_;
		ValaCCodeFunctionCall* vcall;
		ValaCCodeIdentifier* _tmp16_;
		char* _tmp15_;
		ValaCCodeStatement* cstmt;
		ValaCCodeFunction* _tmp17_;
		_tmp10_ = NULL;
		_tmp11_ = NULL;
		vfunc = (_tmp11_ = vala_ccode_function_new (_tmp10_ = vala_method_get_cname ((ValaMethod*) m), "void"), _tmp10_ = (g_free (_tmp10_), NULL), _tmp11_);
		vala_ccode_node_set_line ((ValaCCodeNode*) vfunc, vala_ccode_node_get_line ((ValaCCodeNode*) ((ValaCCodeBaseModule*) self)->function));
		cparam_map = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal);
		carg_map = gee_hash_map_new (G_TYPE_INT, NULL, NULL, VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_hash, g_direct_equal, g_direct_equal);
		vblock = vala_ccode_block_new ();
		_tmp13_ = NULL;
		_tmp12_ = NULL;
		_tmp14_ = NULL;
		vcall = (_tmp14_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp13_ = vala_ccode_identifier_new (_tmp12_ = vala_method_get_real_cname ((ValaMethod*) m)))), (_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_ccode_node_unref (_tmp13_), NULL)), _tmp12_ = (g_free (_tmp12_), NULL), _tmp14_);
		_tmp16_ = NULL;
		_tmp15_ = NULL;
		vala_ccode_function_call_add_argument (vcall, (ValaCCodeExpression*) (_tmp16_ = vala_ccode_identifier_new (_tmp15_ = vala_typesymbol_get_type_id ((ValaTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class))));
		(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_ccode_node_unref (_tmp16_), NULL));
		_tmp15_ = (g_free (_tmp15_), NULL);
		vala_ccode_base_module_generate_cparameters ((ValaCCodeBaseModule*) self, (ValaMethod*) m, ((ValaCCodeBaseModule*) self)->source_declarations, (GeeMap*) cparam_map, vfunc, NULL, (GeeMap*) carg_map, vcall, 3);
		cstmt = (ValaCCodeStatement*) vala_ccode_return_statement_new ((ValaCCodeExpression*) vcall);
		vala_ccode_node_set_line ((ValaCCodeNode*) cstmt, vala_ccode_node_get_line ((ValaCCodeNode*) vfunc));
		vala_ccode_block_add_statement (vblock, (ValaCCodeNode*) cstmt);
		if (!visible) {
			vala_ccode_function_set_modifiers (vfunc, vala_ccode_function_get_modifiers (vfunc) | VALA_CCODE_MODIFIERS_STATIC);
		}
		_tmp17_ = NULL;
		vala_ccode_declaration_space_add_type_member_declaration (((ValaCCodeBaseModule*) self)->source_declarations, (ValaCCodeNode*) (_tmp17_ = vala_ccode_function_copy (vfunc)));
		(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_ccode_node_unref (_tmp17_), NULL));
		vala_ccode_function_set_block (vfunc, vblock);
		vala_ccode_fragment_append (((ValaCCodeBaseModule*) self)->source_type_member_definition, (ValaCCodeNode*) vfunc);
		(vfunc == NULL) ? NULL : (vfunc = (vala_ccode_node_unref (vfunc), NULL));
		(cparam_map == NULL) ? NULL : (cparam_map = (gee_collection_object_unref (cparam_map), NULL));
		(carg_map == NULL) ? NULL : (carg_map = (gee_collection_object_unref (carg_map), NULL));
		(vblock == NULL) ? NULL : (vblock = (vala_ccode_node_unref (vblock), NULL));
		(vcall == NULL) ? NULL : (vcall = (vala_ccode_node_unref (vcall), NULL));
		(cstmt == NULL) ? NULL : (cstmt = (vala_ccode_node_unref (cstmt), NULL));
	}
	_tmp18_ = FALSE;
	_tmp19_ = FALSE;
	if (VALA_IS_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol)) {
		_tmp19_ = vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class, ((ValaCCodeBaseModule*) self)->gobject_type);
	} else {
		_tmp19_ = FALSE;
	}
	if (_tmp19_) {
		gboolean _tmp20_;
		_tmp20_ = FALSE;
		if (vala_creation_method_get_n_construction_params (VALA_CREATION_METHOD (m)) > 0) {
			_tmp20_ = TRUE;
		} else {
			GeeList* _tmp21_;
			_tmp21_ = NULL;
			_tmp20_ = gee_collection_get_size ((GeeCollection*) (_tmp21_ = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) ((ValaCCodeBaseModule*) self)->current_class))) > 0;
			(_tmp21_ == NULL) ? NULL : (_tmp21_ = (gee_collection_object_unref (_tmp21_), NULL));
		}
		_tmp18_ = _tmp20_;
	} else {
		_tmp18_ = FALSE;
	}
	if (_tmp18_) {
		ValaCCodeIdentifier* _tmp23_;
		ValaCCodeIdentifier* _tmp22_;
		ValaCCodeBinaryExpression* _tmp24_;
		ValaCCodeBinaryExpression* ccond;
		ValaCCodeBlock* cdofreeparam;
		ValaCCodeExpressionStatement* _tmp27_;
		ValaCCodeUnaryExpression* _tmp26_;
		ValaCCodeIdentifier* _tmp25_;
		ValaCCodeIdentifier* _tmp28_;
		ValaCCodeFunctionCall* _tmp29_;
		ValaCCodeFunctionCall* cunsetcall;
		ValaCCodeUnaryExpression* _tmp32_;
		ValaCCodeMemberAccess* _tmp31_;
		ValaCCodeIdentifier* _tmp30_;
		ValaCCodeExpressionStatement* _tmp33_;
		ValaCCodeWhileStatement* _tmp34_;
		ValaCCodeIdentifier* _tmp35_;
		ValaCCodeFunctionCall* _tmp36_;
		ValaCCodeFunctionCall* cfreeparams;
		ValaCCodeIdentifier* _tmp37_;
		ValaCCodeExpressionStatement* _tmp38_;
		_tmp23_ = NULL;
		_tmp22_ = NULL;
		_tmp24_ = NULL;
		ccond = (_tmp24_ = vala_ccode_binary_expression_new (VALA_CCODE_BINARY_OPERATOR_GREATER_THAN, (ValaCCodeExpression*) (_tmp22_ = vala_ccode_identifier_new ("__params_it")), (ValaCCodeExpression*) (_tmp23_ = vala_ccode_identifier_new ("__params"))), (_tmp23_ == NULL) ? NULL : (_tmp23_ = (vala_ccode_node_unref (_tmp23_), NULL)), (_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_ccode_node_unref (_tmp22_), NULL)), _tmp24_);
		cdofreeparam = vala_ccode_block_new ();
		_tmp27_ = NULL;
		_tmp26_ = NULL;
		_tmp25_ = NULL;
		vala_ccode_block_add_statement (cdofreeparam, (ValaCCodeNode*) (_tmp27_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) (_tmp26_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_PREFIX_DECREMENT, (ValaCCodeExpression*) (_tmp25_ = vala_ccode_identifier_new ("__params_it")))))));
		(_tmp27_ == NULL) ? NULL : (_tmp27_ = (vala_ccode_node_unref (_tmp27_), NULL));
		(_tmp26_ == NULL) ? NULL : (_tmp26_ = (vala_ccode_node_unref (_tmp26_), NULL));
		(_tmp25_ == NULL) ? NULL : (_tmp25_ = (vala_ccode_node_unref (_tmp25_), NULL));
		_tmp28_ = NULL;
		_tmp29_ = NULL;
		cunsetcall = (_tmp29_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp28_ = vala_ccode_identifier_new ("g_value_unset"))), (_tmp28_ == NULL) ? NULL : (_tmp28_ = (vala_ccode_node_unref (_tmp28_), NULL)), _tmp29_);
		_tmp32_ = NULL;
		_tmp31_ = NULL;
		_tmp30_ = NULL;
		vala_ccode_function_call_add_argument (cunsetcall, (ValaCCodeExpression*) (_tmp32_ = vala_ccode_unary_expression_new (VALA_CCODE_UNARY_OPERATOR_ADDRESS_OF, (ValaCCodeExpression*) (_tmp31_ = vala_ccode_member_access_new_pointer ((ValaCCodeExpression*) (_tmp30_ = vala_ccode_identifier_new ("__params_it")), "value")))));
		(_tmp32_ == NULL) ? NULL : (_tmp32_ = (vala_ccode_node_unref (_tmp32_), NULL));
		(_tmp31_ == NULL) ? NULL : (_tmp31_ = (vala_ccode_node_unref (_tmp31_), NULL));
		(_tmp30_ == NULL) ? NULL : (_tmp30_ = (vala_ccode_node_unref (_tmp30_), NULL));
		_tmp33_ = NULL;
		vala_ccode_block_add_statement (cdofreeparam, (ValaCCodeNode*) (_tmp33_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) cunsetcall)));
		(_tmp33_ == NULL) ? NULL : (_tmp33_ = (vala_ccode_node_unref (_tmp33_), NULL));
		_tmp34_ = NULL;
		vala_ccode_block_add_statement (vala_ccode_function_get_block (((ValaCCodeBaseModule*) self)->function), (ValaCCodeNode*) (_tmp34_ = vala_ccode_while_statement_new ((ValaCCodeExpression*) ccond, (ValaCCodeStatement*) cdofreeparam)));
		(_tmp34_ == NULL) ? NULL : (_tmp34_ = (vala_ccode_node_unref (_tmp34_), NULL));
		_tmp35_ = NULL;
		_tmp36_ = NULL;
		cfreeparams = (_tmp36_ = vala_ccode_function_call_new ((ValaCCodeExpression*) (_tmp35_ = vala_ccode_identifier_new ("g_free"))), (_tmp35_ == NULL) ? NULL : (_tmp35_ = (vala_ccode_node_unref (_tmp35_), NULL)), _tmp36_);
		_tmp37_ = NULL;
		vala_ccode_function_call_add_argument (cfreeparams, (ValaCCodeExpression*) (_tmp37_ = vala_ccode_identifier_new ("__params")));
		(_tmp37_ == NULL) ? NULL : (_tmp37_ = (vala_ccode_node_unref (_tmp37_), NULL));
		_tmp38_ = NULL;
		vala_ccode_block_add_statement (vala_ccode_function_get_block (((ValaCCodeBaseModule*) self)->function), (ValaCCodeNode*) (_tmp38_ = vala_ccode_expression_statement_new ((ValaCCodeExpression*) cfreeparams)));
		(_tmp38_ == NULL) ? NULL : (_tmp38_ = (vala_ccode_node_unref (_tmp38_), NULL));
		(ccond == NULL) ? NULL : (ccond = (vala_ccode_node_unref (ccond), NULL));
		(cdofreeparam == NULL) ? NULL : (cdofreeparam = (vala_ccode_node_unref (cdofreeparam), NULL));
		(cunsetcall == NULL) ? NULL : (cunsetcall = (vala_ccode_node_unref (cunsetcall), NULL));
		(cfreeparams == NULL) ? NULL : (cfreeparams = (vala_ccode_node_unref (cfreeparams), NULL));
	}
	if (VALA_IS_CLASS (((ValaCCodeBaseModule*) self)->current_type_symbol)) {
		ValaCCodeExpression* cresult;
		char* _tmp39_;
		gboolean _tmp40_;
		ValaCCodeReturnStatement* creturn;
		cresult = (ValaCCodeExpression*) vala_ccode_identifier_new ("self");
		_tmp39_ = NULL;
		if ((_tmp40_ = (_tmp39_ = vala_ccode_module_get_custom_creturn_type ((ValaCCodeModule*) self, (ValaMethod*) m)) != NULL, _tmp39_ = (g_free (_tmp39_), NULL), _tmp40_)) {
			ValaCCodeExpression* _tmp42_;
			char* _tmp41_;
			_tmp42_ = NULL;
			_tmp41_ = NULL;
			cresult = (_tmp42_ = (ValaCCodeExpression*) vala_ccode_cast_expression_new (cresult, _tmp41_ = vala_ccode_module_get_custom_creturn_type ((ValaCCodeModule*) self, (ValaMethod*) m)), (cresult == NULL) ? NULL : (cresult = (vala_ccode_node_unref (cresult), NULL)), _tmp42_);
			_tmp41_ = (g_free (_tmp41_), NULL);
		}
		creturn = vala_ccode_return_statement_new (NULL);
		vala_ccode_return_statement_set_return_expression (creturn, cresult);
		vala_ccode_block_add_statement (vala_ccode_function_get_block (((ValaCCodeBaseModule*) self)->function), (ValaCCodeNode*) creturn);
		(cresult == NULL) ? NULL : (cresult = (vala_ccode_node_unref (cresult), NULL));
		(creturn == NULL) ? NULL : (creturn = (vala_ccode_node_unref (creturn), NULL));
	}
	(creturn_type == NULL) ? NULL : (creturn_type = (vala_code_node_unref (creturn_type), NULL));
}


static void vala_ccode_method_module_class_init (ValaCCodeMethodModuleClass * klass) {
	vala_ccode_method_module_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_MODULE_CLASS (klass)->method_has_wrapper = vala_ccode_method_module_real_method_has_wrapper;
	VALA_CCODE_MODULE_CLASS (klass)->get_custom_creturn_type = vala_ccode_method_module_real_get_custom_creturn_type;
	VALA_CCODE_METHOD_MODULE_CLASS (klass)->generate_method_result_declaration = vala_ccode_method_module_real_generate_method_result_declaration;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_method_declaration = vala_ccode_method_module_real_generate_method_declaration;
	VALA_CCODE_MODULE_CLASS (klass)->visit_method = vala_ccode_method_module_real_visit_method;
	VALA_CCODE_METHOD_MODULE_CLASS (klass)->generate_parameter = vala_ccode_method_module_real_generate_parameter;
	VALA_CCODE_BASE_MODULE_CLASS (klass)->generate_cparameters = vala_ccode_method_module_real_generate_cparameters;
	VALA_CCODE_MODULE_CLASS (klass)->visit_creation_method = vala_ccode_method_module_real_visit_creation_method;
}


static void vala_ccode_method_module_instance_init (ValaCCodeMethodModule * self) {
}


GType vala_ccode_method_module_get_type (void) {
	static GType vala_ccode_method_module_type_id = 0;
	if (vala_ccode_method_module_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeMethodModuleClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_method_module_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeMethodModule), 0, (GInstanceInitFunc) vala_ccode_method_module_instance_init, NULL };
		vala_ccode_method_module_type_id = g_type_register_static (VALA_TYPE_CCODE_STRUCT_MODULE, "ValaCCodeMethodModule", &g_define_type_info, 0);
	}
	return vala_ccode_method_module_type_id;
}




