/* valafield.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valafield.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasymbol.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valasourcefile.h>
#include <vala/valaattribute.h>
#include <vala/valainterface.h>
#include <vala/valareport.h>
#include <vala/valaclass.h>
#include <vala/valavaluetype.h>
#include <vala/valanamespace.h>
#include <vala/valastringliteral.h>




struct _ValaFieldPrivate {
	MemberBinding _binding;
	gboolean _is_volatile;
	gboolean _no_array_length;
	char* cname;
	gboolean lock_used;
	ValaExpression* _initializer;
	ValaDataType* _data_type;
};

#define VALA_FIELD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FIELD, ValaFieldPrivate))
enum  {
	VALA_FIELD_DUMMY_PROPERTY
};
static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a);
static gboolean vala_field_real_get_lock_used (ValaLockable* base);
static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_field_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_field_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_field_parent_class = NULL;
static ValaLockableIface* vala_field_vala_lockable_parent_iface = NULL;
static void vala_field_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new field.
 *
 * @param name   field name
 * @param type   field type
 * @param init   initializer expression
 * @param source reference to source code
 * @return       newly created field
 */
ValaField* vala_field_construct (GType object_type, const char* name, ValaDataType* field_type, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaField* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (field_type != NULL, NULL);
	self = (ValaField*) vala_member_construct (object_type, name, source_reference);
	vala_field_set_field_type (self, field_type);
	vala_field_set_initializer (self, initializer);
	return self;
}


ValaField* vala_field_new (const char* name, ValaDataType* field_type, ValaExpression* initializer, ValaSourceReference* source_reference) {
	return vala_field_construct (VALA_TYPE_FIELD, name, field_type, initializer, source_reference);
}


static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_member (visitor, (ValaMember*) self);
	vala_code_visitor_visit_field (visitor, self);
}


static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_field_get_field_type (self), visitor);
	if (vala_field_get_initializer (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_field_get_initializer (self), visitor);
	}
}


/**
 * Returns the name of this field as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_field_get_cname (ValaField* self) {
	const char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_field_get_default_cname (self), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


/**
 * Sets the name of this field as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_field_set_cname (ValaField* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1);
}


/**
 * Returns the default name of this field as it is used in C code.
 *
 * @return the name to be used in C code by default
 */
char* vala_field_get_default_cname (ValaField* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_binding == MEMBER_BINDING_STATIC) {
		char* _tmp0;
		char* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = g_strconcat (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	} else {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = vala_symbol_get_name ((ValaSymbol*) self), (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
	}
}


static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_field_set_cname (self, _tmp0 = vala_attribute_get_string (a, "cname"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
				{
					vala_member_add_cheader_filename ((ValaMember*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_field_process_attributes (ValaField* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_field_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "NoArrayLength") == 0) {
						vala_field_set_no_array_length (self, TRUE);
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_field_real_get_lock_used (ValaLockable* base) {
	ValaField * self;
	self = (ValaField*) base;
	return self->priv->lock_used;
}


static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaField * self;
	self = (ValaField*) base;
	self->priv->lock_used = used;
}


static void vala_field_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_field_get_initializer (self) == old_node) {
		vala_field_set_initializer (self, new_node);
	}
}


static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_field_get_field_type (self) == old_type) {
		vala_field_set_field_type (self, new_type);
	}
}


char* vala_field_get_ctype (ValaField* self) {
	ValaAttribute* attr;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
	if (attr == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = NULL, (attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = vala_attribute_get_string (attr, "type"), (attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL)), _tmp1);
}


void vala_field_set_ctype (ValaField* self, const char* ctype) {
	ValaAttribute* attr;
	ValaStringLiteral* _tmp3;
	char* _tmp2;
	g_return_if_fail (self != NULL);
	g_return_if_fail (ctype != NULL);
	attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
	if (attr == NULL) {
		ValaAttribute* _tmp0;
		ValaAttribute* _tmp1;
		_tmp0 = NULL;
		attr = (_tmp0 = vala_attribute_new ("CCode", NULL), (attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL)), _tmp0);
		_tmp1 = NULL;
		((ValaCodeNode*) self)->attributes = g_list_append (((ValaCodeNode*) self)->attributes, (_tmp1 = attr, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)));
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	vala_attribute_add_argument (attr, "type", (ValaExpression*) (_tmp3 = vala_string_literal_new (_tmp2 = g_strdup_printf ("\"%s\"", ctype), NULL)));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
	_tmp2 = (g_free (_tmp2), NULL);
	(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
}


static gboolean vala_field_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaField * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	gboolean _tmp2;
	gboolean field_in_header;
	gboolean _tmp7;
	self = (ValaField*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_code_node_check ((ValaCodeNode*) vala_field_get_field_type (self), analyzer);
	vala_field_process_attributes (self);
	if (vala_field_get_initializer (self) != NULL) {
		vala_expression_set_target_type (vala_field_get_initializer (self), vala_field_get_field_type (self));
	}
	vala_code_node_check ((ValaCodeNode*) vala_field_get_field_type (self), analyzer);
	if (vala_field_get_initializer (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_field_get_initializer (self), analyzer);
	}
	_tmp2 = FALSE;
	if (self->priv->_binding == MEMBER_BINDING_INSTANCE) {
		_tmp2 = VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self));
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		gboolean _tmp3;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Interfaces may not have instance fields");
		return (_tmp3 = FALSE, (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), _tmp3);
	}
	field_in_header = !vala_symbol_is_internal_symbol ((ValaSymbol*) self);
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
		ValaClass* _tmp4;
		ValaClass* cl;
		gboolean _tmp5;
		_tmp4 = NULL;
		cl = (_tmp4 = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4));
		_tmp5 = FALSE;
		if (vala_class_get_is_compact (cl)) {
			_tmp5 = !vala_symbol_is_internal_symbol ((ValaSymbol*) cl);
		} else {
			_tmp5 = FALSE;
		}
		if (_tmp5) {
			/* compact classes don't have priv structs*/
			field_in_header = TRUE;
		}
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	}
	if (field_in_header) {
		if (VALA_IS_VALUE_TYPE (vala_field_get_field_type (self))) {
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_field_get_field_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
		} else {
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_field_get_field_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	} else {
		if (VALA_IS_NAMESPACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
			gboolean _tmp6;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Namespaces may not have private members");
			return (_tmp6 = FALSE, (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), _tmp6);
		}
		vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_field_get_field_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	return (_tmp7 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), _tmp7);
}


ValaDataType* vala_field_get_field_type (ValaField* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_field_set_field_type (ValaField* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
}


ValaExpression* vala_field_get_initializer (ValaField* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_initializer;
}


void vala_field_set_initializer (ValaField* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_initializer == NULL) ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL)), _tmp2);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_initializer, (ValaCodeNode*) self);
	}
}


MemberBinding vala_field_get_binding (ValaField* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_binding;
}


void vala_field_set_binding (ValaField* self, MemberBinding value) {
	g_return_if_fail (self != NULL);
	self->priv->_binding = value;
}


gboolean vala_field_get_is_volatile (ValaField* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_volatile;
}


void vala_field_set_is_volatile (ValaField* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_volatile = value;
}


gboolean vala_field_get_no_array_length (ValaField* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_no_array_length;
}


void vala_field_set_no_array_length (ValaField* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_array_length = value;
}


static void vala_field_class_init (ValaFieldClass * klass) {
	vala_field_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_field_finalize;
	g_type_class_add_private (klass, sizeof (ValaFieldPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_field_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_field_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_field_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_field_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_field_real_check;
}


static void vala_field_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_field_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_field_real_get_lock_used;
	iface->set_lock_used = vala_field_real_set_lock_used;
}


static void vala_field_instance_init (ValaField * self) {
	self->priv = VALA_FIELD_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->lock_used = FALSE;
}


static void vala_field_finalize (ValaCodeNode* obj) {
	ValaField * self;
	self = VALA_FIELD (obj);
	self->priv->cname = (g_free (self->priv->cname), NULL);
	(self->priv->_initializer == NULL) ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL));
	(self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL));
	VALA_CODE_NODE_CLASS (vala_field_parent_class)->finalize (obj);
}


GType vala_field_get_type (void) {
	static GType vala_field_type_id = 0;
	if (vala_field_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFieldClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_field_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaField), 0, (GInstanceInitFunc) vala_field_instance_init, NULL };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_field_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_field_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaField", &g_define_type_info, 0);
		g_type_add_interface_static (vala_field_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_field_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




