/* valacodewriter.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valacodewriter.h>
#include <stdio.h>
#include <gee/list.h>
#include <gee/arraylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/collection.h>
#include <float.h>
#include <math.h>
#include <vala/valascope.h>
#include <vala/valasymbol.h>
#include <vala/valacodenode.h>
#include <vala/valatypesymbol.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valatypeparameter.h>
#include <vala/valadatatype.h>
#include <vala/valaenumvalue.h>
#include <vala/valamember.h>
#include <vala/valaarraytype.h>
#include <vala/valavoidtype.h>
#include <vala/valapointertype.h>
#include <vala/valavaluetype.h>
#include <vala/valaformalparameter.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valastatement.h>
#include <vala/valaexpression.h>




static char* bool_to_string (gboolean self);
struct _ValaCodeWriterPrivate {
	ValaCodeContext* context;
	FILE* stream;
	gint indent;
	gboolean bol;
	ValaScope* current_scope;
	gboolean dump_tree;
};

#define VALA_CODE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CODE_WRITER, ValaCodeWriterPrivate))
enum  {
	VALA_CODE_WRITER_DUMMY_PROPERTY
};
static void vala_code_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_code_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_code_writer_visit_sorted (ValaCodeWriter* self, GeeList* symbols);
static void vala_code_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_code_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_code_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_code_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain);
static void vala_code_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode);
static void vala_code_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_code_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_code_writer_write_error_domains (ValaCodeWriter* self, GeeList* error_domains);
static gboolean vala_code_writer_float_equal (ValaCodeWriter* self, double d1, double d2);
static void vala_code_writer_write_params (ValaCodeWriter* self, GeeList* params);
static void vala_code_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb);
static void vala_code_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_code_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_code_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_code_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_code_writer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_code_writer_real_visit_empty_statement (ValaCodeVisitor* base, ValaEmptyStatement* stmt);
static void vala_code_writer_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt);
static void vala_code_writer_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local);
static void vala_code_writer_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_code_writer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_code_writer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_code_writer_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt);
static void vala_code_writer_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section);
static void vala_code_writer_real_visit_switch_label (ValaCodeVisitor* base, ValaSwitchLabel* label);
static void vala_code_writer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_code_writer_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_code_writer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_code_writer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_code_writer_real_visit_break_statement (ValaCodeVisitor* base, ValaBreakStatement* stmt);
static void vala_code_writer_real_visit_continue_statement (ValaCodeVisitor* base, ValaContinueStatement* stmt);
static void vala_code_writer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_code_writer_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* y);
static void vala_code_writer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_code_writer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_code_writer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_code_writer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static void vala_code_writer_real_visit_delete_statement (ValaCodeVisitor* base, ValaDeleteStatement* stmt);
static void vala_code_writer_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_code_writer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* lit);
static void vala_code_writer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* lit);
static void vala_code_writer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* lit);
static void vala_code_writer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* lit);
static void vala_code_writer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* lit);
static void vala_code_writer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* lit);
static void vala_code_writer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static void vala_code_writer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static void vala_code_writer_real_visit_method_call (ValaCodeVisitor* base, ValaMethodCall* expr);
static void vala_code_writer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_code_writer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr);
static void vala_code_writer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_code_writer_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_code_writer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr);
static void vala_code_writer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr);
static void vala_code_writer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static void vala_code_writer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr);
static void vala_code_writer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr);
static void vala_code_writer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr);
static void vala_code_writer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static void vala_code_writer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_code_writer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr);
static void vala_code_writer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr);
static void vala_code_writer_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* expr);
static void vala_code_writer_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static void vala_code_writer_write_indent (ValaCodeWriter* self);
static void vala_code_writer_write_identifier (ValaCodeWriter* self, const char* s);
static void vala_code_writer_write_return_type (ValaCodeWriter* self, ValaDataType* type);
static gboolean vala_code_writer_is_weak (ValaCodeWriter* self, ValaDataType* type);
static void vala_code_writer_write_type (ValaCodeWriter* self, ValaDataType* type);
static void vala_code_writer_write_string (ValaCodeWriter* self, const char* s);
static void vala_code_writer_write_newline (ValaCodeWriter* self);
static void vala_code_writer_write_code_block (ValaCodeWriter* self, ValaBlock* block);
static void vala_code_writer_write_begin_block (ValaCodeWriter* self);
static void vala_code_writer_write_end_block (ValaCodeWriter* self);
static gboolean vala_code_writer_check_accessibility (ValaCodeWriter* self, ValaSymbol* sym);
static void vala_code_writer_write_accessibility (ValaCodeWriter* self, ValaSymbol* sym);
static gpointer vala_code_writer_parent_class = NULL;
static void vala_code_writer_finalize (ValaCodeVisitor* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



static char* bool_to_string (gboolean self) {
	if (self) {
		return g_strdup ("true");
	} else {
		return g_strdup ("false");
	}
}


ValaCodeWriter* vala_code_writer_construct (GType object_type, gboolean dump_tree) {
	ValaCodeWriter* self;
	self = (ValaCodeWriter*) g_type_create_instance (object_type);
	self->priv->dump_tree = dump_tree;
	return self;
}


ValaCodeWriter* vala_code_writer_new (gboolean dump_tree) {
	return vala_code_writer_construct (VALA_TYPE_CODE_WRITER, dump_tree);
}


/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_code_writer_write_file (ValaCodeWriter* self, ValaCodeContext* context, const char* filename) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	FILE* _tmp2;
	char* _tmp4;
	char* _tmp3;
	ValaScope* _tmp6;
	ValaScope* _tmp5;
	ValaScope* _tmp7;
	FILE* _tmp8;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	g_return_if_fail (filename != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL) ? NULL : vala_code_context_ref (_tmp0)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1);
	_tmp2 = NULL;
	self->priv->stream = (_tmp2 = fopen (filename, "w"), (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp2);
	_tmp4 = NULL;
	_tmp3 = NULL;
	vala_code_writer_write_string (self, _tmp4 = g_strdup_printf ("/* %s generated by %s, do not modify. */", _tmp3 = g_path_get_basename (filename), g_get_prgname ()));
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp3 = (g_free (_tmp3), NULL);
	vala_code_writer_write_newline (self);
	vala_code_writer_write_newline (self);
	_tmp6 = NULL;
	_tmp5 = NULL;
	self->priv->current_scope = (_tmp6 = (_tmp5 = vala_symbol_get_scope ((ValaSymbol*) vala_code_context_get_root (context)), (_tmp5 == NULL) ? NULL : vala_scope_ref (_tmp5)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp6);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
	_tmp7 = NULL;
	self->priv->current_scope = (_tmp7 = NULL, (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp7);
	_tmp8 = NULL;
	self->priv->stream = (_tmp8 = NULL, (self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)), _tmp8);
}


static void vala_code_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaCodeWriter * self;
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	ValaScope* _tmp4;
	ValaScope* _tmp3;
	GeeList* _tmp5;
	GeeList* _tmp6;
	GeeList* _tmp7;
	GeeList* _tmp8;
	GeeList* _tmp9;
	GeeList* _tmp10;
	GeeList* _tmp11;
	GeeList* _tmp12;
	GeeList* _tmp13;
	GeeList* _tmp14;
	ValaScope* _tmp16;
	ValaScope* _tmp15;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (ns != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) ns)) {
		return;
	}
	if (vala_symbol_get_name ((ValaSymbol*) ns) == NULL) {
		vala_code_node_accept_children ((ValaCodeNode*) ns, (ValaCodeVisitor*) self);
		return;
	}
	vala_code_writer_write_indent (self);
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_code_writer_write_string (self, _tmp2 = g_strdup_printf ("[CCode (cprefix = \"%s\", lower_case_cprefix = \"%s\")]", _tmp0 = vala_symbol_get_cprefix ((ValaSymbol*) ns), _tmp1 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) ns)));
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp1 = (g_free (_tmp1), NULL);
	_tmp0 = (g_free (_tmp0), NULL);
	vala_code_writer_write_newline (self);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "namespace ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) ns));
	vala_code_writer_write_begin_block (self);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_scope = (_tmp4 = (_tmp3 = vala_symbol_get_scope ((ValaSymbol*) ns), (_tmp3 == NULL) ? NULL : vala_scope_ref (_tmp3)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp4);
	_tmp5 = NULL;
	vala_code_writer_visit_sorted (self, _tmp5 = vala_namespace_get_namespaces (ns));
	(_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL));
	_tmp6 = NULL;
	vala_code_writer_visit_sorted (self, _tmp6 = vala_namespace_get_classes (ns));
	(_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL));
	_tmp7 = NULL;
	vala_code_writer_visit_sorted (self, _tmp7 = vala_namespace_get_interfaces (ns));
	(_tmp7 == NULL) ? NULL : (_tmp7 = (gee_collection_object_unref (_tmp7), NULL));
	_tmp8 = NULL;
	vala_code_writer_visit_sorted (self, _tmp8 = vala_namespace_get_structs (ns));
	(_tmp8 == NULL) ? NULL : (_tmp8 = (gee_collection_object_unref (_tmp8), NULL));
	_tmp9 = NULL;
	vala_code_writer_visit_sorted (self, _tmp9 = vala_namespace_get_enums (ns));
	(_tmp9 == NULL) ? NULL : (_tmp9 = (gee_collection_object_unref (_tmp9), NULL));
	_tmp10 = NULL;
	vala_code_writer_visit_sorted (self, _tmp10 = vala_namespace_get_error_types (ns));
	(_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL));
	_tmp11 = NULL;
	vala_code_writer_visit_sorted (self, _tmp11 = vala_namespace_get_delegates (ns));
	(_tmp11 == NULL) ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL));
	_tmp12 = NULL;
	vala_code_writer_visit_sorted (self, _tmp12 = vala_namespace_get_fields (ns));
	(_tmp12 == NULL) ? NULL : (_tmp12 = (gee_collection_object_unref (_tmp12), NULL));
	_tmp13 = NULL;
	vala_code_writer_visit_sorted (self, _tmp13 = vala_namespace_get_constants (ns));
	(_tmp13 == NULL) ? NULL : (_tmp13 = (gee_collection_object_unref (_tmp13), NULL));
	_tmp14 = NULL;
	vala_code_writer_visit_sorted (self, _tmp14 = vala_namespace_get_methods (ns));
	(_tmp14 == NULL) ? NULL : (_tmp14 = (gee_collection_object_unref (_tmp14), NULL));
	_tmp16 = NULL;
	_tmp15 = NULL;
	self->priv->current_scope = (_tmp16 = (_tmp15 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp15 == NULL) ? NULL : vala_scope_ref (_tmp15)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp16);
	vala_code_writer_write_end_block (self);
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaCodeWriter * self;
	char* _tmp24;
	char* _tmp23;
	gboolean _tmp25;
	gboolean _tmp29;
	char* _tmp30;
	gboolean _tmp31;
	gboolean first;
	char* cheaders;
	char* _tmp42;
	GeeList* type_params;
	GeeList* base_types;
	ValaScope* _tmp44;
	ValaScope* _tmp43;
	GeeList* _tmp45;
	GeeList* _tmp46;
	GeeList* _tmp47;
	GeeList* _tmp48;
	GeeList* _tmp49;
	GeeList* _tmp50;
	GeeList* _tmp51;
	GeeList* _tmp52;
	GeeList* _tmp53;
	ValaScope* _tmp55;
	ValaScope* _tmp54;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (cl != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) cl)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) cl)) {
		return;
	}
	if (vala_class_get_is_compact (cl)) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[Compact]");
		vala_code_writer_write_newline (self);
	}
	if (vala_class_get_is_immutable (cl)) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[Immutable]");
		vala_code_writer_write_newline (self);
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "[CCode (");
	if (vala_typesymbol_is_reference_counting ((ValaTypeSymbol*) cl)) {
		gboolean _tmp0;
		gboolean _tmp1;
		gboolean _tmp7;
		gboolean _tmp8;
		_tmp0 = FALSE;
		_tmp1 = FALSE;
		if (vala_class_get_base_class (cl) == NULL) {
			_tmp1 = TRUE;
		} else {
			char* _tmp2;
			_tmp2 = NULL;
			_tmp1 = (_tmp2 = vala_typesymbol_get_ref_function ((ValaTypeSymbol*) vala_class_get_base_class (cl))) == NULL;
			_tmp2 = (g_free (_tmp2), NULL);
		}
		if (_tmp1) {
			_tmp0 = TRUE;
		} else {
			char* _tmp4;
			char* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			_tmp0 = _vala_strcmp0 (_tmp3 = vala_typesymbol_get_ref_function ((ValaTypeSymbol*) vala_class_get_base_class (cl)), _tmp4 = vala_typesymbol_get_ref_function ((ValaTypeSymbol*) cl)) != 0;
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
		}
		if (_tmp0) {
			char* _tmp6;
			char* _tmp5;
			_tmp6 = NULL;
			_tmp5 = NULL;
			vala_code_writer_write_string (self, _tmp6 = g_strdup_printf ("ref_function = \"%s\", ", _tmp5 = vala_typesymbol_get_ref_function ((ValaTypeSymbol*) cl)));
			_tmp6 = (g_free (_tmp6), NULL);
			_tmp5 = (g_free (_tmp5), NULL);
		}
		_tmp7 = FALSE;
		_tmp8 = FALSE;
		if (vala_class_get_base_class (cl) == NULL) {
			_tmp8 = TRUE;
		} else {
			char* _tmp9;
			_tmp9 = NULL;
			_tmp8 = (_tmp9 = vala_typesymbol_get_unref_function ((ValaTypeSymbol*) vala_class_get_base_class (cl))) == NULL;
			_tmp9 = (g_free (_tmp9), NULL);
		}
		if (_tmp8) {
			_tmp7 = TRUE;
		} else {
			char* _tmp11;
			char* _tmp10;
			_tmp11 = NULL;
			_tmp10 = NULL;
			_tmp7 = _vala_strcmp0 (_tmp10 = vala_typesymbol_get_unref_function ((ValaTypeSymbol*) vala_class_get_base_class (cl)), _tmp11 = vala_typesymbol_get_unref_function ((ValaTypeSymbol*) cl)) != 0;
			_tmp11 = (g_free (_tmp11), NULL);
			_tmp10 = (g_free (_tmp10), NULL);
		}
		if (_tmp7) {
			char* _tmp13;
			char* _tmp12;
			_tmp13 = NULL;
			_tmp12 = NULL;
			vala_code_writer_write_string (self, _tmp13 = g_strdup_printf ("unref_function = \"%s\", ", _tmp12 = vala_typesymbol_get_unref_function ((ValaTypeSymbol*) cl)));
			_tmp13 = (g_free (_tmp13), NULL);
			_tmp12 = (g_free (_tmp12), NULL);
		}
	} else {
		char* _tmp14;
		gboolean _tmp15;
		char* _tmp19;
		char* _tmp18;
		gboolean _tmp20;
		_tmp14 = NULL;
		if ((_tmp15 = (_tmp14 = vala_typesymbol_get_dup_function ((ValaTypeSymbol*) cl)) != NULL, _tmp14 = (g_free (_tmp14), NULL), _tmp15)) {
			char* _tmp17;
			char* _tmp16;
			_tmp17 = NULL;
			_tmp16 = NULL;
			vala_code_writer_write_string (self, _tmp17 = g_strdup_printf ("copy_function = \"%s\", ", _tmp16 = vala_typesymbol_get_dup_function ((ValaTypeSymbol*) cl)));
			_tmp17 = (g_free (_tmp17), NULL);
			_tmp16 = (g_free (_tmp16), NULL);
		}
		_tmp19 = NULL;
		_tmp18 = NULL;
		if ((_tmp20 = _vala_strcmp0 (_tmp18 = vala_typesymbol_get_free_function ((ValaTypeSymbol*) cl), _tmp19 = vala_class_get_default_free_function (cl)) != 0, _tmp19 = (g_free (_tmp19), NULL), _tmp18 = (g_free (_tmp18), NULL), _tmp20)) {
			char* _tmp22;
			char* _tmp21;
			_tmp22 = NULL;
			_tmp21 = NULL;
			vala_code_writer_write_string (self, _tmp22 = g_strdup_printf ("free_function = \"%s\", ", _tmp21 = vala_typesymbol_get_free_function ((ValaTypeSymbol*) cl)));
			_tmp22 = (g_free (_tmp22), NULL);
			_tmp21 = (g_free (_tmp21), NULL);
		}
	}
	_tmp24 = NULL;
	_tmp23 = NULL;
	if ((_tmp25 = _vala_strcmp0 (_tmp23 = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE), _tmp24 = vala_class_get_default_cname (cl)) != 0, _tmp24 = (g_free (_tmp24), NULL), _tmp23 = (g_free (_tmp23), NULL), _tmp25)) {
		char* _tmp27;
		char* _tmp26;
		_tmp27 = NULL;
		_tmp26 = NULL;
		vala_code_writer_write_string (self, _tmp27 = g_strdup_printf ("cname = \"%s\", ", _tmp26 = vala_typesymbol_get_cname ((ValaTypeSymbol*) cl, FALSE)));
		_tmp27 = (g_free (_tmp27), NULL);
		_tmp26 = (g_free (_tmp26), NULL);
	}
	if (vala_class_get_type_check_function (cl) != NULL) {
		char* _tmp28;
		_tmp28 = NULL;
		vala_code_writer_write_string (self, _tmp28 = g_strdup_printf ("type_check_function = \"%s\", ", vala_class_get_type_check_function (cl)));
		_tmp28 = (g_free (_tmp28), NULL);
	}
	_tmp29 = FALSE;
	_tmp30 = NULL;
	if ((_tmp31 = (_tmp30 = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) cl)) != NULL, _tmp30 = (g_free (_tmp30), NULL), _tmp31)) {
		gboolean _tmp32;
		_tmp32 = FALSE;
		if (vala_class_get_base_class (cl) == NULL) {
			_tmp32 = TRUE;
		} else {
			char* _tmp34;
			char* _tmp33;
			_tmp34 = NULL;
			_tmp33 = NULL;
			_tmp32 = _vala_strcmp0 (_tmp33 = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) cl), _tmp34 = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) vala_class_get_base_class (cl))) != 0;
			_tmp34 = (g_free (_tmp34), NULL);
			_tmp33 = (g_free (_tmp33), NULL);
		}
		_tmp29 = (_tmp32);
	} else {
		_tmp29 = FALSE;
	}
	if (_tmp29) {
		char* _tmp36;
		char* _tmp35;
		_tmp36 = NULL;
		_tmp35 = NULL;
		vala_code_writer_write_string (self, _tmp36 = g_strdup_printf ("param_spec_function = \"%s\", ", _tmp35 = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) cl)));
		_tmp36 = (g_free (_tmp36), NULL);
		_tmp35 = (g_free (_tmp35), NULL);
	}
	first = TRUE;
	cheaders = g_strdup ("");
	{
		GeeList* _tmp37;
		GeeIterator* _tmp38;
		GeeIterator* cheader_it;
		_tmp37 = NULL;
		_tmp38 = NULL;
		cheader_it = (_tmp38 = gee_iterable_iterator ((GeeIterable*) (_tmp37 = vala_symbol_get_cheader_filenames ((ValaSymbol*) cl))), (_tmp37 == NULL) ? NULL : (_tmp37 = (gee_collection_object_unref (_tmp37), NULL)), _tmp38);
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = (char*) gee_iterator_get (cheader_it);
			if (first) {
				char* _tmp40;
				const char* _tmp39;
				_tmp40 = NULL;
				_tmp39 = NULL;
				cheaders = (_tmp40 = (_tmp39 = cheader, (_tmp39 == NULL) ? NULL : g_strdup (_tmp39)), cheaders = (g_free (cheaders), NULL), _tmp40);
				first = FALSE;
			} else {
				char* _tmp41;
				_tmp41 = NULL;
				cheaders = (_tmp41 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp41);
			}
			cheader = (g_free (cheader), NULL);
		}
		(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
	}
	_tmp42 = NULL;
	vala_code_writer_write_string (self, _tmp42 = g_strdup_printf ("cheader_filename = \"%s\")]", cheaders));
	_tmp42 = (g_free (_tmp42), NULL);
	vala_code_writer_write_newline (self);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) cl);
	if (vala_class_get_is_abstract (cl)) {
		vala_code_writer_write_string (self, "abstract ");
	}
	vala_code_writer_write_string (self, "class ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) cl));
	type_params = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) cl);
	if (gee_collection_get_size ((GeeCollection*) type_params) > 0) {
		gboolean first;
		vala_code_writer_write_string (self, "<");
		first = TRUE;
		{
			GeeIterator* type_param_it;
			type_param_it = gee_iterable_iterator ((GeeIterable*) type_params);
			while (gee_iterator_next (type_param_it)) {
				ValaTypeParameter* type_param;
				type_param = (ValaTypeParameter*) gee_iterator_get (type_param_it);
				if (first) {
					first = FALSE;
				} else {
					vala_code_writer_write_string (self, ",");
				}
				vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) type_param));
				(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
			}
			(type_param_it == NULL) ? NULL : (type_param_it = (gee_collection_object_unref (type_param_it), NULL));
		}
		vala_code_writer_write_string (self, ">");
	}
	base_types = vala_class_get_base_types (cl);
	if (gee_collection_get_size ((GeeCollection*) base_types) > 0) {
		gboolean first;
		vala_code_writer_write_string (self, " : ");
		first = TRUE;
		{
			GeeIterator* base_type_it;
			base_type_it = gee_iterable_iterator ((GeeIterable*) base_types);
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = (ValaDataType*) gee_iterator_get (base_type_it);
				if (!first) {
					vala_code_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				vala_code_writer_write_type (self, base_type);
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
			}
			(base_type_it == NULL) ? NULL : (base_type_it = (gee_collection_object_unref (base_type_it), NULL));
		}
	}
	vala_code_writer_write_begin_block (self);
	_tmp44 = NULL;
	_tmp43 = NULL;
	self->priv->current_scope = (_tmp44 = (_tmp43 = vala_symbol_get_scope ((ValaSymbol*) cl), (_tmp43 == NULL) ? NULL : vala_scope_ref (_tmp43)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp44);
	_tmp45 = NULL;
	vala_code_writer_visit_sorted (self, _tmp45 = vala_class_get_classes (cl));
	(_tmp45 == NULL) ? NULL : (_tmp45 = (gee_collection_object_unref (_tmp45), NULL));
	_tmp46 = NULL;
	vala_code_writer_visit_sorted (self, _tmp46 = vala_class_get_structs (cl));
	(_tmp46 == NULL) ? NULL : (_tmp46 = (gee_collection_object_unref (_tmp46), NULL));
	_tmp47 = NULL;
	vala_code_writer_visit_sorted (self, _tmp47 = vala_class_get_enums (cl));
	(_tmp47 == NULL) ? NULL : (_tmp47 = (gee_collection_object_unref (_tmp47), NULL));
	_tmp48 = NULL;
	vala_code_writer_visit_sorted (self, _tmp48 = vala_class_get_delegates (cl));
	(_tmp48 == NULL) ? NULL : (_tmp48 = (gee_collection_object_unref (_tmp48), NULL));
	_tmp49 = NULL;
	vala_code_writer_visit_sorted (self, _tmp49 = vala_class_get_fields (cl));
	(_tmp49 == NULL) ? NULL : (_tmp49 = (gee_collection_object_unref (_tmp49), NULL));
	_tmp50 = NULL;
	vala_code_writer_visit_sorted (self, _tmp50 = vala_class_get_constants (cl));
	(_tmp50 == NULL) ? NULL : (_tmp50 = (gee_collection_object_unref (_tmp50), NULL));
	_tmp51 = NULL;
	vala_code_writer_visit_sorted (self, _tmp51 = vala_object_type_symbol_get_methods ((ValaObjectTypeSymbol*) cl));
	(_tmp51 == NULL) ? NULL : (_tmp51 = (gee_collection_object_unref (_tmp51), NULL));
	_tmp52 = NULL;
	vala_code_writer_visit_sorted (self, _tmp52 = vala_object_type_symbol_get_properties ((ValaObjectTypeSymbol*) cl));
	(_tmp52 == NULL) ? NULL : (_tmp52 = (gee_collection_object_unref (_tmp52), NULL));
	_tmp53 = NULL;
	vala_code_writer_visit_sorted (self, _tmp53 = vala_object_type_symbol_get_signals ((ValaObjectTypeSymbol*) cl));
	(_tmp53 == NULL) ? NULL : (_tmp53 = (gee_collection_object_unref (_tmp53), NULL));
	_tmp55 = NULL;
	_tmp54 = NULL;
	self->priv->current_scope = (_tmp55 = (_tmp54 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp54 == NULL) ? NULL : vala_scope_ref (_tmp54)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp55);
	vala_code_writer_write_end_block (self);
	vala_code_writer_write_newline (self);
	cheaders = (g_free (cheaders), NULL);
	(type_params == NULL) ? NULL : (type_params = (gee_collection_object_unref (type_params), NULL));
	(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
}


static void vala_code_writer_visit_sorted (ValaCodeWriter* self, GeeList* symbols) {
	GeeArrayList* sorted_symbols;
	g_return_if_fail (self != NULL);
	g_return_if_fail (symbols != NULL);
	sorted_symbols = gee_array_list_new (VALA_TYPE_SYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	{
		GeeIterator* sym_it;
		sym_it = gee_iterable_iterator ((GeeIterable*) symbols);
		while (gee_iterator_next (sym_it)) {
			ValaSymbol* sym;
			gint left;
			gint right;
			gboolean _tmp0;
			sym = (ValaSymbol*) gee_iterator_get (sym_it);
			left = 0;
			right = gee_collection_get_size ((GeeCollection*) sorted_symbols) - 1;
			_tmp0 = FALSE;
			if (left > right) {
				_tmp0 = TRUE;
			} else {
				ValaSymbol* _tmp1;
				_tmp1 = NULL;
				_tmp0 = _vala_strcmp0 (vala_symbol_get_name (sym), vala_symbol_get_name (_tmp1 = (ValaSymbol*) gee_list_get ((GeeList*) sorted_symbols, left))) < 0;
				(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
			}
			if (_tmp0) {
				gee_list_insert ((GeeList*) sorted_symbols, 0, sym);
			} else {
				ValaSymbol* _tmp2;
				gboolean _tmp3;
				_tmp2 = NULL;
				if ((_tmp3 = _vala_strcmp0 (vala_symbol_get_name (sym), vala_symbol_get_name (_tmp2 = (ValaSymbol*) gee_list_get ((GeeList*) sorted_symbols, right))) > 0, (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)), _tmp3)) {
					gee_collection_add ((GeeCollection*) sorted_symbols, sym);
				} else {
					while ((right - left) > 1) {
						gint i;
						ValaSymbol* _tmp4;
						gboolean _tmp5;
						i = (right + left) / 2;
						_tmp4 = NULL;
						if ((_tmp5 = _vala_strcmp0 (vala_symbol_get_name (sym), vala_symbol_get_name (_tmp4 = (ValaSymbol*) gee_list_get ((GeeList*) sorted_symbols, i))) > 0, (_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)), _tmp5)) {
							left = i;
						} else {
							right = i;
						}
					}
					gee_list_insert ((GeeList*) sorted_symbols, left + 1, sym);
				}
			}
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
		}
		(sym_it == NULL) ? NULL : (sym_it = (gee_collection_object_unref (sym_it), NULL));
	}
	{
		GeeIterator* sym_it;
		sym_it = gee_iterable_iterator ((GeeIterable*) sorted_symbols);
		while (gee_iterator_next (sym_it)) {
			ValaSymbol* sym;
			sym = (ValaSymbol*) gee_iterator_get (sym_it);
			vala_code_node_accept ((ValaCodeNode*) sym, (ValaCodeVisitor*) self);
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
		}
		(sym_it == NULL) ? NULL : (sym_it = (gee_collection_object_unref (sym_it), NULL));
	}
	(sorted_symbols == NULL) ? NULL : (sorted_symbols = (gee_collection_object_unref (sorted_symbols), NULL));
}


static void vala_code_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaCodeWriter * self;
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	gboolean _tmp5;
	gboolean first;
	char* cheaders;
	char* _tmp14;
	GeeList* base_types;
	ValaScope* _tmp18;
	ValaScope* _tmp17;
	GeeList* _tmp21;
	GeeList* _tmp22;
	ValaScope* _tmp24;
	ValaScope* _tmp23;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (st != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) st)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) st)) {
		return;
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "[CCode (");
	_tmp1 = NULL;
	_tmp0 = NULL;
	if ((_tmp2 = _vala_strcmp0 (_tmp0 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE), _tmp1 = vala_struct_get_default_cname (st)) != 0, _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2)) {
		char* _tmp4;
		char* _tmp3;
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_code_writer_write_string (self, _tmp4 = g_strdup_printf ("cname = \"%s\", ", _tmp3 = vala_typesymbol_get_cname ((ValaTypeSymbol*) st, FALSE)));
		_tmp4 = (g_free (_tmp4), NULL);
		_tmp3 = (g_free (_tmp3), NULL);
	}
	_tmp5 = FALSE;
	if (!vala_struct_is_simple_type (st)) {
		char* _tmp6;
		_tmp6 = NULL;
		_tmp5 = _vala_strcmp0 (_tmp6 = vala_typesymbol_get_type_id ((ValaTypeSymbol*) st), "G_TYPE_POINTER") != 0;
		_tmp6 = (g_free (_tmp6), NULL);
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		char* _tmp8;
		char* _tmp7;
		_tmp8 = NULL;
		_tmp7 = NULL;
		vala_code_writer_write_string (self, _tmp8 = g_strdup_printf ("type_id = \"%s\", ", _tmp7 = vala_typesymbol_get_type_id ((ValaTypeSymbol*) st)));
		_tmp8 = (g_free (_tmp8), NULL);
		_tmp7 = (g_free (_tmp7), NULL);
	}
	first = TRUE;
	cheaders = g_strdup ("");
	{
		GeeList* _tmp9;
		GeeIterator* _tmp10;
		GeeIterator* cheader_it;
		_tmp9 = NULL;
		_tmp10 = NULL;
		cheader_it = (_tmp10 = gee_iterable_iterator ((GeeIterable*) (_tmp9 = vala_symbol_get_cheader_filenames ((ValaSymbol*) st))), (_tmp9 == NULL) ? NULL : (_tmp9 = (gee_collection_object_unref (_tmp9), NULL)), _tmp10);
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = (char*) gee_iterator_get (cheader_it);
			if (first) {
				char* _tmp12;
				const char* _tmp11;
				_tmp12 = NULL;
				_tmp11 = NULL;
				cheaders = (_tmp12 = (_tmp11 = cheader, (_tmp11 == NULL) ? NULL : g_strdup (_tmp11)), cheaders = (g_free (cheaders), NULL), _tmp12);
				first = FALSE;
			} else {
				char* _tmp13;
				_tmp13 = NULL;
				cheaders = (_tmp13 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp13);
			}
			cheader = (g_free (cheader), NULL);
		}
		(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
	}
	_tmp14 = NULL;
	vala_code_writer_write_string (self, _tmp14 = g_strdup_printf ("cheader_filename = \"%s\")]", cheaders));
	_tmp14 = (g_free (_tmp14), NULL);
	vala_code_writer_write_newline (self);
	if (vala_struct_is_simple_type (st)) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[SimpleType]");
		vala_code_writer_write_newline (self);
	}
	if (vala_struct_is_integer_type (st)) {
		char* _tmp15;
		vala_code_writer_write_indent (self);
		_tmp15 = NULL;
		vala_code_writer_write_string (self, _tmp15 = g_strdup_printf ("[IntegerType (rank = %d)]", vala_struct_get_rank (st)));
		_tmp15 = (g_free (_tmp15), NULL);
		vala_code_writer_write_newline (self);
	}
	if (vala_struct_is_floating_type (st)) {
		char* _tmp16;
		vala_code_writer_write_indent (self);
		_tmp16 = NULL;
		vala_code_writer_write_string (self, _tmp16 = g_strdup_printf ("[FloatingType (rank = %d)]", vala_struct_get_rank (st)));
		_tmp16 = (g_free (_tmp16), NULL);
		vala_code_writer_write_newline (self);
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) st);
	vala_code_writer_write_string (self, "struct ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) st));
	base_types = vala_struct_get_base_types (st);
	if (gee_collection_get_size ((GeeCollection*) base_types) > 0) {
		gboolean first;
		vala_code_writer_write_string (self, " : ");
		first = TRUE;
		{
			GeeIterator* base_type_it;
			base_type_it = gee_iterable_iterator ((GeeIterable*) base_types);
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = (ValaDataType*) gee_iterator_get (base_type_it);
				if (!first) {
					vala_code_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				vala_code_writer_write_type (self, base_type);
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
			}
			(base_type_it == NULL) ? NULL : (base_type_it = (gee_collection_object_unref (base_type_it), NULL));
		}
	}
	vala_code_writer_write_begin_block (self);
	_tmp18 = NULL;
	_tmp17 = NULL;
	self->priv->current_scope = (_tmp18 = (_tmp17 = vala_symbol_get_scope ((ValaSymbol*) st), (_tmp17 == NULL) ? NULL : vala_scope_ref (_tmp17)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp18);
	{
		GeeList* _tmp19;
		GeeIterator* _tmp20;
		GeeIterator* field_it;
		_tmp19 = NULL;
		_tmp20 = NULL;
		field_it = (_tmp20 = gee_iterable_iterator ((GeeIterable*) (_tmp19 = vala_struct_get_fields (st))), (_tmp19 == NULL) ? NULL : (_tmp19 = (gee_collection_object_unref (_tmp19), NULL)), _tmp20);
		while (gee_iterator_next (field_it)) {
			ValaField* field;
			field = (ValaField*) gee_iterator_get (field_it);
			vala_code_node_accept ((ValaCodeNode*) field, (ValaCodeVisitor*) self);
			(field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL));
		}
		(field_it == NULL) ? NULL : (field_it = (gee_collection_object_unref (field_it), NULL));
	}
	_tmp21 = NULL;
	vala_code_writer_visit_sorted (self, _tmp21 = vala_struct_get_constants (st));
	(_tmp21 == NULL) ? NULL : (_tmp21 = (gee_collection_object_unref (_tmp21), NULL));
	_tmp22 = NULL;
	vala_code_writer_visit_sorted (self, _tmp22 = vala_struct_get_methods (st));
	(_tmp22 == NULL) ? NULL : (_tmp22 = (gee_collection_object_unref (_tmp22), NULL));
	_tmp24 = NULL;
	_tmp23 = NULL;
	self->priv->current_scope = (_tmp24 = (_tmp23 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp23 == NULL) ? NULL : vala_scope_ref (_tmp23)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp24);
	vala_code_writer_write_end_block (self);
	vala_code_writer_write_newline (self);
	cheaders = (g_free (cheaders), NULL);
	(base_types == NULL) ? NULL : (base_types = (gee_collection_object_unref (base_types), NULL));
}


static void vala_code_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaCodeWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp5;
	char* _tmp7;
	char* _tmp6;
	gboolean _tmp8;
	GeeList* type_params;
	GeeList* prerequisites;
	ValaScope* _tmp12;
	ValaScope* _tmp11;
	GeeList* _tmp13;
	GeeList* _tmp14;
	GeeList* _tmp15;
	GeeList* _tmp16;
	GeeList* _tmp17;
	GeeList* _tmp18;
	GeeList* _tmp19;
	GeeList* _tmp20;
	ValaScope* _tmp22;
	ValaScope* _tmp21;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (iface != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) iface)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) iface)) {
		return;
	}
	vala_code_writer_write_indent (self);
	first = TRUE;
	cheaders = g_strdup ("");
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* cheader_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		cheader_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_symbol_get_cheader_filenames ((ValaSymbol*) iface))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = (char*) gee_iterator_get (cheader_it);
			if (first) {
				char* _tmp3;
				const char* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				cheaders = (_tmp3 = (_tmp2 = cheader, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), cheaders = (g_free (cheaders), NULL), _tmp3);
				first = FALSE;
			} else {
				char* _tmp4;
				_tmp4 = NULL;
				cheaders = (_tmp4 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp4);
			}
			cheader = (g_free (cheader), NULL);
		}
		(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
	}
	_tmp5 = NULL;
	vala_code_writer_write_string (self, _tmp5 = g_strdup_printf ("[CCode (cheader_filename = \"%s\"", cheaders));
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp7 = NULL;
	_tmp6 = NULL;
	if ((_tmp8 = _vala_strcmp0 (_tmp6 = vala_interface_get_lower_case_csuffix (iface), _tmp7 = vala_interface_get_default_lower_case_csuffix (iface)) != 0, _tmp7 = (g_free (_tmp7), NULL), _tmp6 = (g_free (_tmp6), NULL), _tmp8)) {
		char* _tmp10;
		char* _tmp9;
		_tmp10 = NULL;
		_tmp9 = NULL;
		vala_code_writer_write_string (self, _tmp10 = g_strdup_printf (", lower_case_csuffix = \"%s\"", _tmp9 = vala_interface_get_lower_case_csuffix (iface)));
		_tmp10 = (g_free (_tmp10), NULL);
		_tmp9 = (g_free (_tmp9), NULL);
	}
	vala_code_writer_write_string (self, ")]");
	vala_code_writer_write_newline (self);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) iface);
	vala_code_writer_write_string (self, "interface ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) iface));
	type_params = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) iface);
	if (gee_collection_get_size ((GeeCollection*) type_params) > 0) {
		gboolean first;
		vala_code_writer_write_string (self, "<");
		first = TRUE;
		{
			GeeIterator* type_param_it;
			type_param_it = gee_iterable_iterator ((GeeIterable*) type_params);
			while (gee_iterator_next (type_param_it)) {
				ValaTypeParameter* type_param;
				type_param = (ValaTypeParameter*) gee_iterator_get (type_param_it);
				if (first) {
					first = FALSE;
				} else {
					vala_code_writer_write_string (self, ",");
				}
				vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) type_param));
				(type_param == NULL) ? NULL : (type_param = (vala_code_node_unref (type_param), NULL));
			}
			(type_param_it == NULL) ? NULL : (type_param_it = (gee_collection_object_unref (type_param_it), NULL));
		}
		vala_code_writer_write_string (self, ">");
	}
	prerequisites = vala_interface_get_prerequisites (iface);
	if (gee_collection_get_size ((GeeCollection*) prerequisites) > 0) {
		gboolean first;
		vala_code_writer_write_string (self, " : ");
		first = TRUE;
		{
			GeeIterator* prerequisite_it;
			prerequisite_it = gee_iterable_iterator ((GeeIterable*) prerequisites);
			while (gee_iterator_next (prerequisite_it)) {
				ValaDataType* prerequisite;
				prerequisite = (ValaDataType*) gee_iterator_get (prerequisite_it);
				if (!first) {
					vala_code_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				vala_code_writer_write_type (self, prerequisite);
				(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			}
			(prerequisite_it == NULL) ? NULL : (prerequisite_it = (gee_collection_object_unref (prerequisite_it), NULL));
		}
	}
	vala_code_writer_write_begin_block (self);
	_tmp12 = NULL;
	_tmp11 = NULL;
	self->priv->current_scope = (_tmp12 = (_tmp11 = vala_symbol_get_scope ((ValaSymbol*) iface), (_tmp11 == NULL) ? NULL : vala_scope_ref (_tmp11)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp12);
	_tmp13 = NULL;
	vala_code_writer_visit_sorted (self, _tmp13 = vala_interface_get_classes (iface));
	(_tmp13 == NULL) ? NULL : (_tmp13 = (gee_collection_object_unref (_tmp13), NULL));
	_tmp14 = NULL;
	vala_code_writer_visit_sorted (self, _tmp14 = vala_interface_get_structs (iface));
	(_tmp14 == NULL) ? NULL : (_tmp14 = (gee_collection_object_unref (_tmp14), NULL));
	_tmp15 = NULL;
	vala_code_writer_visit_sorted (self, _tmp15 = vala_interface_get_enums (iface));
	(_tmp15 == NULL) ? NULL : (_tmp15 = (gee_collection_object_unref (_tmp15), NULL));
	_tmp16 = NULL;
	vala_code_writer_visit_sorted (self, _tmp16 = vala_interface_get_delegates (iface));
	(_tmp16 == NULL) ? NULL : (_tmp16 = (gee_collection_object_unref (_tmp16), NULL));
	_tmp17 = NULL;
	vala_code_writer_visit_sorted (self, _tmp17 = vala_interface_get_fields (iface));
	(_tmp17 == NULL) ? NULL : (_tmp17 = (gee_collection_object_unref (_tmp17), NULL));
	_tmp18 = NULL;
	vala_code_writer_visit_sorted (self, _tmp18 = vala_object_type_symbol_get_methods ((ValaObjectTypeSymbol*) iface));
	(_tmp18 == NULL) ? NULL : (_tmp18 = (gee_collection_object_unref (_tmp18), NULL));
	_tmp19 = NULL;
	vala_code_writer_visit_sorted (self, _tmp19 = vala_object_type_symbol_get_properties ((ValaObjectTypeSymbol*) iface));
	(_tmp19 == NULL) ? NULL : (_tmp19 = (gee_collection_object_unref (_tmp19), NULL));
	_tmp20 = NULL;
	vala_code_writer_visit_sorted (self, _tmp20 = vala_object_type_symbol_get_signals ((ValaObjectTypeSymbol*) iface));
	(_tmp20 == NULL) ? NULL : (_tmp20 = (gee_collection_object_unref (_tmp20), NULL));
	_tmp22 = NULL;
	_tmp21 = NULL;
	self->priv->current_scope = (_tmp22 = (_tmp21 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp21 == NULL) ? NULL : vala_scope_ref (_tmp21)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp22);
	vala_code_writer_write_end_block (self);
	vala_code_writer_write_newline (self);
	cheaders = (g_free (cheaders), NULL);
	(type_params == NULL) ? NULL : (type_params = (gee_collection_object_unref (type_params), NULL));
	(prerequisites == NULL) ? NULL : (prerequisites = (gee_collection_object_unref (prerequisites), NULL));
}


static void vala_code_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaCodeWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp6;
	char* _tmp5;
	char* _tmp9;
	ValaScope* _tmp20;
	ValaScope* _tmp19;
	ValaScope* _tmp24;
	ValaScope* _tmp23;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (en != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) en)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) en)) {
		return;
	}
	vala_code_writer_write_indent (self);
	first = TRUE;
	cheaders = g_strdup ("");
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* cheader_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		cheader_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_symbol_get_cheader_filenames ((ValaSymbol*) en))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = (char*) gee_iterator_get (cheader_it);
			if (first) {
				char* _tmp3;
				const char* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				cheaders = (_tmp3 = (_tmp2 = cheader, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), cheaders = (g_free (cheaders), NULL), _tmp3);
				first = FALSE;
			} else {
				char* _tmp4;
				_tmp4 = NULL;
				cheaders = (_tmp4 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp4);
			}
			cheader = (g_free (cheader), NULL);
		}
		(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	vala_code_writer_write_string (self, _tmp6 = g_strdup_printf ("[CCode (cprefix = \"%s\", ", _tmp5 = vala_enum_get_cprefix (en)));
	_tmp6 = (g_free (_tmp6), NULL);
	_tmp5 = (g_free (_tmp5), NULL);
	if (!vala_enum_get_has_type_id (en)) {
		gint _tmp7;
		char* _tmp8;
		_tmp7 = 0;
		if (vala_enum_get_has_type_id (en)) {
			_tmp7 = 1;
		} else {
			_tmp7 = 0;
		}
		_tmp8 = NULL;
		vala_code_writer_write_string (self, _tmp8 = g_strdup_printf ("has_type_id = \"%d\", ", _tmp7));
		_tmp8 = (g_free (_tmp8), NULL);
	}
	_tmp9 = NULL;
	vala_code_writer_write_string (self, _tmp9 = g_strdup_printf ("cheader_filename = \"%s\")]", cheaders));
	_tmp9 = (g_free (_tmp9), NULL);
	if (vala_enum_get_is_flags (en)) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[Flags]");
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) en);
	vala_code_writer_write_string (self, "enum ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) en));
	vala_code_writer_write_begin_block (self);
	first = TRUE;
	{
		GeeList* _tmp10;
		GeeIterator* _tmp11;
		GeeIterator* ev_it;
		_tmp10 = NULL;
		_tmp11 = NULL;
		ev_it = (_tmp11 = gee_iterable_iterator ((GeeIterable*) (_tmp10 = vala_enum_get_values (en))), (_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL)), _tmp11);
		while (gee_iterator_next (ev_it)) {
			ValaEnumValue* ev;
			char* _tmp13;
			char* _tmp12;
			gboolean _tmp14;
			ev = (ValaEnumValue*) gee_iterator_get (ev_it);
			if (first) {
				first = FALSE;
			} else {
				vala_code_writer_write_string (self, ",");
				vala_code_writer_write_newline (self);
			}
			_tmp13 = NULL;
			_tmp12 = NULL;
			if ((_tmp14 = _vala_strcmp0 (_tmp12 = vala_enum_value_get_cname (ev), _tmp13 = vala_enum_value_get_default_cname (ev)) != 0, _tmp13 = (g_free (_tmp13), NULL), _tmp12 = (g_free (_tmp12), NULL), _tmp14)) {
				char* _tmp16;
				char* _tmp15;
				vala_code_writer_write_indent (self);
				_tmp16 = NULL;
				_tmp15 = NULL;
				vala_code_writer_write_string (self, _tmp16 = g_strdup_printf ("[CCode (cname = \"%s\")]", _tmp15 = vala_enum_value_get_cname (ev)));
				_tmp16 = (g_free (_tmp16), NULL);
				_tmp15 = (g_free (_tmp15), NULL);
			}
			vala_code_writer_write_indent (self);
			vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) ev));
			(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
		}
		(ev_it == NULL) ? NULL : (ev_it = (gee_collection_object_unref (ev_it), NULL));
	}
	if (!first) {
		GeeList* _tmp17;
		gboolean _tmp18;
		_tmp17 = NULL;
		if ((_tmp18 = gee_collection_get_size ((GeeCollection*) (_tmp17 = vala_enum_get_methods (en))) > 0, (_tmp17 == NULL) ? NULL : (_tmp17 = (gee_collection_object_unref (_tmp17), NULL)), _tmp18)) {
			vala_code_writer_write_string (self, ";");
		}
		vala_code_writer_write_newline (self);
	}
	_tmp20 = NULL;
	_tmp19 = NULL;
	self->priv->current_scope = (_tmp20 = (_tmp19 = vala_symbol_get_scope ((ValaSymbol*) en), (_tmp19 == NULL) ? NULL : vala_scope_ref (_tmp19)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp20);
	{
		GeeList* _tmp21;
		GeeIterator* _tmp22;
		GeeIterator* m_it;
		_tmp21 = NULL;
		_tmp22 = NULL;
		m_it = (_tmp22 = gee_iterable_iterator ((GeeIterable*) (_tmp21 = vala_enum_get_methods (en))), (_tmp21 == NULL) ? NULL : (_tmp21 = (gee_collection_object_unref (_tmp21), NULL)), _tmp22);
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (m_it);
			vala_code_node_accept ((ValaCodeNode*) m, (ValaCodeVisitor*) self);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(m_it == NULL) ? NULL : (m_it = (gee_collection_object_unref (m_it), NULL));
	}
	_tmp24 = NULL;
	_tmp23 = NULL;
	self->priv->current_scope = (_tmp24 = (_tmp23 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp23 == NULL) ? NULL : vala_scope_ref (_tmp23)), (self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL)), _tmp24);
	vala_code_writer_write_end_block (self);
	vala_code_writer_write_newline (self);
	cheaders = (g_free (cheaders), NULL);
}


static void vala_code_writer_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* edomain) {
	ValaCodeWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp6;
	char* _tmp5;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (edomain != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) edomain)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) edomain)) {
		return;
	}
	vala_code_writer_write_indent (self);
	first = TRUE;
	cheaders = g_strdup ("");
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* cheader_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		cheader_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_symbol_get_cheader_filenames ((ValaSymbol*) edomain))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = (char*) gee_iterator_get (cheader_it);
			if (first) {
				char* _tmp3;
				const char* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				cheaders = (_tmp3 = (_tmp2 = cheader, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), cheaders = (g_free (cheaders), NULL), _tmp3);
				first = FALSE;
			} else {
				char* _tmp4;
				_tmp4 = NULL;
				cheaders = (_tmp4 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp4);
			}
			cheader = (g_free (cheader), NULL);
		}
		(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	vala_code_writer_write_string (self, _tmp6 = g_strdup_printf ("[CCode (cprefix = \"%s\", cheader_filename = \"%s\")]", _tmp5 = vala_error_domain_get_cprefix (edomain), cheaders));
	_tmp6 = (g_free (_tmp6), NULL);
	_tmp5 = (g_free (_tmp5), NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) edomain);
	vala_code_writer_write_string (self, "errordomain ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) edomain));
	vala_code_writer_write_begin_block (self);
	vala_code_node_accept_children ((ValaCodeNode*) edomain, (ValaCodeVisitor*) self);
	vala_code_writer_write_end_block (self);
	vala_code_writer_write_newline (self);
	cheaders = (g_free (cheaders), NULL);
}


static void vala_code_writer_real_visit_error_code (ValaCodeVisitor* base, ValaErrorCode* ecode) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (ecode != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) ecode));
	vala_code_writer_write_string (self, ",");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaCodeWriter * self;
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	gboolean custom_cname;
	gboolean custom_cheaders;
	gboolean _tmp3;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (c != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) c)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) c)) {
		return;
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	custom_cname = (_tmp2 = (_vala_strcmp0 (_tmp0 = vala_constant_get_cname (c), _tmp1 = vala_constant_get_default_cname (c)) != 0), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	custom_cheaders = (VALA_IS_NAMESPACE (vala_symbol_get_parent_symbol ((ValaSymbol*) c)));
	_tmp3 = FALSE;
	if (custom_cname) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = custom_cheaders;
	}
	if (_tmp3) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[CCode (");
		if (custom_cname) {
			char* _tmp5;
			char* _tmp4;
			_tmp5 = NULL;
			_tmp4 = NULL;
			vala_code_writer_write_string (self, _tmp5 = g_strdup_printf ("cname = \"%s\"", _tmp4 = vala_constant_get_cname (c)));
			_tmp5 = (g_free (_tmp5), NULL);
			_tmp4 = (g_free (_tmp4), NULL);
		}
		if (custom_cheaders) {
			gboolean first;
			char* cheaders;
			char* _tmp11;
			if (custom_cname) {
				vala_code_writer_write_string (self, ", ");
			}
			first = TRUE;
			cheaders = g_strdup ("");
			{
				GeeList* _tmp6;
				GeeIterator* _tmp7;
				GeeIterator* cheader_it;
				_tmp6 = NULL;
				_tmp7 = NULL;
				cheader_it = (_tmp7 = gee_iterable_iterator ((GeeIterable*) (_tmp6 = vala_symbol_get_cheader_filenames ((ValaSymbol*) c))), (_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL)), _tmp7);
				while (gee_iterator_next (cheader_it)) {
					char* cheader;
					cheader = (char*) gee_iterator_get (cheader_it);
					if (first) {
						char* _tmp9;
						const char* _tmp8;
						_tmp9 = NULL;
						_tmp8 = NULL;
						cheaders = (_tmp9 = (_tmp8 = cheader, (_tmp8 == NULL) ? NULL : g_strdup (_tmp8)), cheaders = (g_free (cheaders), NULL), _tmp9);
						first = FALSE;
					} else {
						char* _tmp10;
						_tmp10 = NULL;
						cheaders = (_tmp10 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp10);
					}
					cheader = (g_free (cheader), NULL);
				}
				(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
			}
			_tmp11 = NULL;
			vala_code_writer_write_string (self, _tmp11 = g_strdup_printf ("cheader_filename = \"%s\"", cheaders));
			_tmp11 = (g_free (_tmp11), NULL);
			cheaders = (g_free (cheaders), NULL);
		}
		vala_code_writer_write_string (self, ")]");
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) c);
	vala_code_writer_write_string (self, "const ");
	vala_code_writer_write_type (self, vala_constant_get_type_reference (c));
	vala_code_writer_write_string (self, " ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) c));
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaCodeWriter * self;
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	gboolean custom_cname;
	char* _tmp3;
	gboolean _tmp4;
	gboolean custom_ctype;
	gboolean custom_cheaders;
	gboolean _tmp5;
	gboolean _tmp6;
	gboolean _tmp18;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (f != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) f)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) f)) {
		return;
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	custom_cname = (_tmp2 = (_vala_strcmp0 (_tmp0 = vala_field_get_cname (f), _tmp1 = vala_field_get_default_cname (f)) != 0), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	_tmp3 = NULL;
	custom_ctype = (_tmp4 = ((_tmp3 = vala_field_get_ctype (f)) != NULL), _tmp3 = (g_free (_tmp3), NULL), _tmp4);
	custom_cheaders = (VALA_IS_NAMESPACE (vala_symbol_get_parent_symbol ((ValaSymbol*) f)));
	_tmp5 = FALSE;
	_tmp6 = FALSE;
	if (custom_cname) {
		_tmp6 = TRUE;
	} else {
		_tmp6 = custom_ctype;
	}
	if (_tmp6) {
		_tmp5 = TRUE;
	} else {
		_tmp5 = custom_cheaders;
	}
	if (_tmp5) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[CCode (");
		if (custom_cname) {
			char* _tmp8;
			char* _tmp7;
			_tmp8 = NULL;
			_tmp7 = NULL;
			vala_code_writer_write_string (self, _tmp8 = g_strdup_printf ("cname = \"%s\"", _tmp7 = vala_field_get_cname (f)));
			_tmp8 = (g_free (_tmp8), NULL);
			_tmp7 = (g_free (_tmp7), NULL);
		}
		if (custom_ctype) {
			char* _tmp10;
			char* _tmp9;
			if (custom_cname) {
				vala_code_writer_write_string (self, ", ");
			}
			_tmp10 = NULL;
			_tmp9 = NULL;
			vala_code_writer_write_string (self, _tmp10 = g_strdup_printf ("type = \"%s\"", _tmp9 = vala_field_get_ctype (f)));
			_tmp10 = (g_free (_tmp10), NULL);
			_tmp9 = (g_free (_tmp9), NULL);
		}
		if (custom_cheaders) {
			gboolean _tmp11;
			gboolean first;
			char* cheaders;
			char* _tmp17;
			_tmp11 = FALSE;
			if (custom_cname) {
				_tmp11 = TRUE;
			} else {
				_tmp11 = custom_ctype;
			}
			if (_tmp11) {
				vala_code_writer_write_string (self, ", ");
			}
			first = TRUE;
			cheaders = g_strdup ("");
			{
				GeeList* _tmp12;
				GeeIterator* _tmp13;
				GeeIterator* cheader_it;
				_tmp12 = NULL;
				_tmp13 = NULL;
				cheader_it = (_tmp13 = gee_iterable_iterator ((GeeIterable*) (_tmp12 = vala_symbol_get_cheader_filenames ((ValaSymbol*) f))), (_tmp12 == NULL) ? NULL : (_tmp12 = (gee_collection_object_unref (_tmp12), NULL)), _tmp13);
				while (gee_iterator_next (cheader_it)) {
					char* cheader;
					cheader = (char*) gee_iterator_get (cheader_it);
					if (first) {
						char* _tmp15;
						const char* _tmp14;
						_tmp15 = NULL;
						_tmp14 = NULL;
						cheaders = (_tmp15 = (_tmp14 = cheader, (_tmp14 == NULL) ? NULL : g_strdup (_tmp14)), cheaders = (g_free (cheaders), NULL), _tmp15);
						first = FALSE;
					} else {
						char* _tmp16;
						_tmp16 = NULL;
						cheaders = (_tmp16 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp16);
					}
					cheader = (g_free (cheader), NULL);
				}
				(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
			}
			_tmp17 = NULL;
			vala_code_writer_write_string (self, _tmp17 = g_strdup_printf ("cheader_filename = \"%s\"", cheaders));
			_tmp17 = (g_free (_tmp17), NULL);
			cheaders = (g_free (cheaders), NULL);
		}
		vala_code_writer_write_string (self, ")]");
	}
	_tmp18 = FALSE;
	if (vala_field_get_no_array_length (f)) {
		_tmp18 = VALA_IS_ARRAY_TYPE (vala_field_get_field_type (f));
	} else {
		_tmp18 = FALSE;
	}
	if (_tmp18) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[NoArrayLength]");
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) f);
	if (vala_field_get_binding (f) == MEMBER_BINDING_STATIC) {
		vala_code_writer_write_string (self, "static ");
	} else {
		if (vala_field_get_binding (f) == MEMBER_BINDING_CLASS) {
			vala_code_writer_write_string (self, "class ");
		}
	}
	if (vala_code_writer_is_weak (self, vala_field_get_field_type (f))) {
		vala_code_writer_write_string (self, "weak ");
	}
	vala_code_writer_write_type (self, vala_field_get_field_type (f));
	vala_code_writer_write_string (self, " ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) f));
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_write_error_domains (ValaCodeWriter* self, GeeList* error_domains) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (error_domains != NULL);
	if (gee_collection_get_size ((GeeCollection*) error_domains) > 0) {
		gboolean first;
		vala_code_writer_write_string (self, " throws ");
		first = TRUE;
		{
			GeeIterator* type_it;
			type_it = gee_iterable_iterator ((GeeIterable*) error_domains);
			while (gee_iterator_next (type_it)) {
				ValaDataType* type;
				type = (ValaDataType*) gee_iterator_get (type_it);
				if (!first) {
					vala_code_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				vala_code_writer_write_type (self, type);
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			}
			(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
		}
	}
}


/* equality comparison with 3 digit precision*/
static gboolean vala_code_writer_float_equal (ValaCodeWriter* self, double d1, double d2) {
	g_return_val_if_fail (self != NULL, FALSE);
	return ((gint) (d1 * 1000)) == ((gint) (d2 * 1000));
}


static void vala_code_writer_write_params (ValaCodeWriter* self, GeeList* params) {
	gint i;
	g_return_if_fail (self != NULL);
	g_return_if_fail (params != NULL);
	vala_code_writer_write_string (self, "(");
	i = 1;
	{
		GeeIterator* param_it;
		param_it = gee_iterable_iterator ((GeeIterable*) params);
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			GString* ccode_params;
			char* separator;
			gboolean _tmp4;
			param = (ValaFormalParameter*) gee_iterator_get (param_it);
			if (i > 1) {
				vala_code_writer_write_string (self, ", ");
			}
			if (vala_formal_parameter_get_ellipsis (param)) {
				vala_code_writer_write_string (self, "...");
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				continue;
			}
			ccode_params = g_string_new ("");
			separator = g_strdup ("");
			if (vala_formal_parameter_get_ctype (param) != NULL) {
				char* _tmp0;
				g_string_append_printf (ccode_params, "%stype = \"%s\"", separator, vala_formal_parameter_get_ctype (param));
				_tmp0 = NULL;
				separator = (_tmp0 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp0);
			}
			if (!vala_code_writer_float_equal (self, vala_formal_parameter_get_carray_length_parameter_position (param), i + 0.1)) {
				char* _tmp1;
				g_string_append_printf (ccode_params, "%sarray_length_pos = %g", separator, vala_formal_parameter_get_carray_length_parameter_position (param));
				_tmp1 = NULL;
				separator = (_tmp1 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp1);
			}
			if (!vala_code_writer_float_equal (self, vala_formal_parameter_get_cdelegate_target_parameter_position (param), i + 0.1)) {
				char* _tmp2;
				g_string_append_printf (ccode_params, "%sdelegate_target_pos = %g", separator, vala_formal_parameter_get_cdelegate_target_parameter_position (param));
				_tmp2 = NULL;
				separator = (_tmp2 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp2);
			}
			if (ccode_params->len > 0) {
				char* _tmp3;
				_tmp3 = NULL;
				vala_code_writer_write_string (self, _tmp3 = g_strdup_printf ("[CCode (%s)] ", ccode_params->str));
				_tmp3 = (g_free (_tmp3), NULL);
			}
			if (vala_formal_parameter_get_params_array (param)) {
				vala_code_writer_write_string (self, "params ");
			}
			if (vala_formal_parameter_get_direction (param) != VALA_PARAMETER_DIRECTION_IN) {
				if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_REF) {
					vala_code_writer_write_string (self, "ref ");
				} else {
					if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_OUT) {
						vala_code_writer_write_string (self, "out ");
					}
				}
				if (vala_code_writer_is_weak (self, vala_formal_parameter_get_parameter_type (param))) {
					vala_code_writer_write_string (self, "weak ");
				}
			}
			vala_code_writer_write_type (self, vala_formal_parameter_get_parameter_type (param));
			_tmp4 = FALSE;
			if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_IN) {
				_tmp4 = vala_data_type_get_value_owned (vala_formal_parameter_get_parameter_type (param));
			} else {
				_tmp4 = FALSE;
			}
			if (_tmp4) {
				vala_code_writer_write_string (self, "#");
			}
			vala_code_writer_write_string (self, " ");
			vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) param));
			if (vala_formal_parameter_get_default_expression (param) != NULL) {
				char* _tmp5;
				vala_code_writer_write_string (self, " = ");
				_tmp5 = NULL;
				vala_code_writer_write_string (self, _tmp5 = vala_code_node_to_string ((ValaCodeNode*) vala_formal_parameter_get_default_expression (param)));
				_tmp5 = (g_free (_tmp5), NULL);
			}
			i++;
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			(ccode_params == NULL) ? NULL : (ccode_params = (g_string_free (ccode_params, TRUE), NULL));
			separator = (g_free (separator), NULL);
		}
		(param_it == NULL) ? NULL : (param_it = (gee_collection_object_unref (param_it), NULL));
	}
	vala_code_writer_write_string (self, ")");
}


static void vala_code_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb) {
	ValaCodeWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp5;
	GeeList* _tmp6;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (cb != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) cb)) {
		return;
	}
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) cb)) {
		return;
	}
	vala_code_writer_write_indent (self);
	first = TRUE;
	cheaders = g_strdup ("");
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* cheader_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		cheader_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_symbol_get_cheader_filenames ((ValaSymbol*) cb))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = (char*) gee_iterator_get (cheader_it);
			if (first) {
				char* _tmp3;
				const char* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				cheaders = (_tmp3 = (_tmp2 = cheader, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), cheaders = (g_free (cheaders), NULL), _tmp3);
				first = FALSE;
			} else {
				char* _tmp4;
				_tmp4 = NULL;
				cheaders = (_tmp4 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp4);
			}
			cheader = (g_free (cheader), NULL);
		}
		(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
	}
	_tmp5 = NULL;
	vala_code_writer_write_string (self, _tmp5 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders));
	_tmp5 = (g_free (_tmp5), NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) cb);
	if (!vala_delegate_get_has_target (cb)) {
		vala_code_writer_write_string (self, "static ");
	}
	vala_code_writer_write_string (self, "delegate ");
	vala_code_writer_write_return_type (self, vala_delegate_get_return_type (cb));
	vala_code_writer_write_string (self, " ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) cb));
	vala_code_writer_write_string (self, " ");
	_tmp6 = NULL;
	vala_code_writer_write_params (self, _tmp6 = vala_delegate_get_parameters (cb));
	(_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL));
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
	cheaders = (g_free (cheaders), NULL);
}


static void vala_code_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaCodeWriter * self;
	gboolean _tmp0;
	gboolean _tmp1;
	ValaAttribute* _tmp4;
	gboolean _tmp5;
	GString* ccode_params;
	char* separator;
	char* _tmp9;
	char* _tmp8;
	gboolean _tmp10;
	gboolean _tmp23;
	gboolean _tmp25;
	GeeList* _tmp29;
	GeeList* _tmp30;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (m != NULL);
	if (vala_symbol_get_external_package ((ValaSymbol*) m)) {
		return;
	}
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) m)) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = vala_method_get_overrides (m);
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp2;
		gboolean _tmp3;
		_tmp2 = FALSE;
		_tmp3 = FALSE;
		if (vala_method_get_base_interface_method (m) != NULL) {
			_tmp3 = !vala_method_get_is_abstract (m);
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			_tmp2 = !vala_method_get_is_virtual (m);
		} else {
			_tmp2 = FALSE;
		}
		_tmp0 = (_tmp2);
	}
	/* don't write interface implementation unless it's an abstract or virtual method*/
	if (_tmp0) {
		if (!self->priv->dump_tree) {
			return;
		}
	}
	_tmp4 = NULL;
	if ((_tmp5 = (_tmp4 = vala_code_node_get_attribute ((ValaCodeNode*) m, "NoWrapper")) != NULL, (_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)), _tmp5)) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[NoWrapper]");
	}
	if (vala_method_get_no_array_length (m)) {
		gboolean array_found;
		array_found = (VALA_IS_ARRAY_TYPE (vala_method_get_return_type (m)));
		{
			GeeList* _tmp6;
			GeeIterator* _tmp7;
			GeeIterator* param_it;
			_tmp6 = NULL;
			_tmp7 = NULL;
			param_it = (_tmp7 = gee_iterable_iterator ((GeeIterable*) (_tmp6 = vala_method_get_parameters (m))), (_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL)), _tmp7);
			while (gee_iterator_next (param_it)) {
				ValaFormalParameter* param;
				param = (ValaFormalParameter*) gee_iterator_get (param_it);
				if (VALA_IS_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param))) {
					array_found = TRUE;
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
					break;
				}
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
			}
			(param_it == NULL) ? NULL : (param_it = (gee_collection_object_unref (param_it), NULL));
		}
		if (array_found) {
			vala_code_writer_write_indent (self);
			vala_code_writer_write_string (self, "[NoArrayLength]");
		}
	}
	ccode_params = g_string_new ("");
	separator = g_strdup ("");
	_tmp9 = NULL;
	_tmp8 = NULL;
	if ((_tmp10 = _vala_strcmp0 (_tmp8 = vala_method_get_cname (m), _tmp9 = vala_method_get_default_cname (m)) != 0, _tmp9 = (g_free (_tmp9), NULL), _tmp8 = (g_free (_tmp8), NULL), _tmp10)) {
		char* _tmp11;
		char* _tmp12;
		_tmp11 = NULL;
		g_string_append_printf (ccode_params, "%scname = \"%s\"", separator, _tmp11 = vala_method_get_cname (m));
		_tmp11 = (g_free (_tmp11), NULL);
		_tmp12 = NULL;
		separator = (_tmp12 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp12);
	}
	if (VALA_IS_NAMESPACE (vala_symbol_get_parent_symbol ((ValaSymbol*) m))) {
		gboolean first;
		char* cheaders;
		char* _tmp18;
		first = TRUE;
		cheaders = g_strdup ("");
		{
			GeeList* _tmp13;
			GeeIterator* _tmp14;
			GeeIterator* cheader_it;
			_tmp13 = NULL;
			_tmp14 = NULL;
			cheader_it = (_tmp14 = gee_iterable_iterator ((GeeIterable*) (_tmp13 = vala_symbol_get_cheader_filenames ((ValaSymbol*) m))), (_tmp13 == NULL) ? NULL : (_tmp13 = (gee_collection_object_unref (_tmp13), NULL)), _tmp14);
			while (gee_iterator_next (cheader_it)) {
				char* cheader;
				cheader = (char*) gee_iterator_get (cheader_it);
				if (first) {
					char* _tmp16;
					const char* _tmp15;
					_tmp16 = NULL;
					_tmp15 = NULL;
					cheaders = (_tmp16 = (_tmp15 = cheader, (_tmp15 == NULL) ? NULL : g_strdup (_tmp15)), cheaders = (g_free (cheaders), NULL), _tmp16);
					first = FALSE;
				} else {
					char* _tmp17;
					_tmp17 = NULL;
					cheaders = (_tmp17 = g_strdup_printf ("%s,%s", cheaders, cheader), cheaders = (g_free (cheaders), NULL), _tmp17);
				}
				cheader = (g_free (cheader), NULL);
			}
			(cheader_it == NULL) ? NULL : (cheader_it = (gee_collection_object_unref (cheader_it), NULL));
		}
		g_string_append_printf (ccode_params, "%scheader_filename = \"%s\"", separator, cheaders);
		_tmp18 = NULL;
		separator = (_tmp18 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp18);
		cheaders = (g_free (cheaders), NULL);
	}
	if (!vala_code_writer_float_equal (self, vala_method_get_cinstance_parameter_position (m), (double) 0)) {
		char* _tmp19;
		g_string_append_printf (ccode_params, "%sinstance_pos = %g", separator, vala_method_get_cinstance_parameter_position (m));
		_tmp19 = NULL;
		separator = (_tmp19 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp19);
	}
	if (!vala_code_writer_float_equal (self, vala_method_get_carray_length_parameter_position (m), (double) (-3))) {
		char* _tmp20;
		g_string_append_printf (ccode_params, "%sarray_length_pos = %g", separator, vala_method_get_carray_length_parameter_position (m));
		_tmp20 = NULL;
		separator = (_tmp20 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp20);
	}
	if (!vala_code_writer_float_equal (self, vala_method_get_cdelegate_target_parameter_position (m), (double) (-3))) {
		char* _tmp21;
		g_string_append_printf (ccode_params, "%sdelegate_target_pos = %g", separator, vala_method_get_cdelegate_target_parameter_position (m));
		_tmp21 = NULL;
		separator = (_tmp21 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp21);
	}
	if (_vala_strcmp0 (vala_method_get_sentinel (m), VALA_METHOD_DEFAULT_SENTINEL) != 0) {
		char* _tmp22;
		g_string_append_printf (ccode_params, "%ssentinel = \"%s\"", separator, vala_method_get_sentinel (m));
		_tmp22 = NULL;
		separator = (_tmp22 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp22);
	}
	_tmp23 = FALSE;
	if (VALA_IS_CREATION_METHOD (m)) {
		_tmp23 = vala_creation_method_get_custom_return_type_cname ((VALA_CREATION_METHOD (m))) != NULL;
	} else {
		_tmp23 = FALSE;
	}
	if (_tmp23) {
		char* _tmp24;
		g_string_append_printf (ccode_params, "%stype = \"%s\"", separator, vala_creation_method_get_custom_return_type_cname ((VALA_CREATION_METHOD (m))));
		_tmp24 = NULL;
		separator = (_tmp24 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp24);
	}
	_tmp25 = FALSE;
	if (VALA_IS_CREATION_METHOD (m)) {
		_tmp25 = !vala_method_get_has_construct_function (m);
	} else {
		_tmp25 = FALSE;
	}
	if (_tmp25) {
		char* _tmp26;
		g_string_append_printf (ccode_params, "%shas_construct_function = false", separator);
		_tmp26 = NULL;
		separator = (_tmp26 = g_strdup (", "), separator = (g_free (separator), NULL), _tmp26);
	}
	if (ccode_params->len > 0) {
		char* _tmp27;
		vala_code_writer_write_indent (self);
		_tmp27 = NULL;
		vala_code_writer_write_string (self, _tmp27 = g_strdup_printf ("[CCode (%s)]", ccode_params->str));
		_tmp27 = (g_free (_tmp27), NULL);
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) m);
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaTypeSymbol* _tmp28;
		ValaTypeSymbol* datatype;
		_tmp28 = NULL;
		datatype = (_tmp28 = VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) m)), (_tmp28 == NULL) ? NULL : vala_code_node_ref (_tmp28));
		vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) datatype));
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) m), "new") != 0) {
			vala_code_writer_write_string (self, ".");
			vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) m));
		}
		vala_code_writer_write_string (self, " ");
		(datatype == NULL) ? NULL : (datatype = (vala_code_node_unref (datatype), NULL));
	} else {
		if (vala_method_get_binding (m) == MEMBER_BINDING_STATIC) {
			vala_code_writer_write_string (self, "static ");
		} else {
			if (vala_method_get_binding (m) == MEMBER_BINDING_CLASS) {
				vala_code_writer_write_string (self, "class ");
			} else {
				if (vala_method_get_is_abstract (m)) {
					vala_code_writer_write_string (self, "abstract ");
				} else {
					if (vala_method_get_is_virtual (m)) {
						vala_code_writer_write_string (self, "virtual ");
					}
				}
			}
		}
	}
	if (!(VALA_IS_CREATION_METHOD (m))) {
		vala_code_writer_write_return_type (self, vala_method_get_return_type (m));
		vala_code_writer_write_string (self, " ");
		vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) m));
		vala_code_writer_write_string (self, " ");
	}
	_tmp29 = NULL;
	vala_code_writer_write_params (self, _tmp29 = vala_method_get_parameters (m));
	(_tmp29 == NULL) ? NULL : (_tmp29 = (gee_collection_object_unref (_tmp29), NULL));
	_tmp30 = NULL;
	vala_code_writer_write_error_domains (self, _tmp30 = vala_code_node_get_error_types ((ValaCodeNode*) m));
	(_tmp30 == NULL) ? NULL : (_tmp30 = (gee_collection_object_unref (_tmp30), NULL));
	vala_code_writer_write_code_block (self, vala_method_get_body (m));
	vala_code_writer_write_newline (self);
	(ccode_params == NULL) ? NULL : (ccode_params = (g_string_free (ccode_params, TRUE), NULL));
	separator = (g_free (separator), NULL);
}


static void vala_code_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (m != NULL);
	vala_code_visitor_visit_method ((ValaCodeVisitor*) self, (ValaMethod*) m);
}


static void vala_code_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaCodeWriter * self;
	gboolean _tmp0;
	gboolean _tmp1;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (prop != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) prop)) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = vala_property_get_overrides (prop);
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp2;
		gboolean _tmp3;
		_tmp2 = FALSE;
		_tmp3 = FALSE;
		if (vala_property_get_base_interface_property (prop) != NULL) {
			_tmp3 = !vala_property_get_is_abstract (prop);
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			_tmp2 = !vala_property_get_is_virtual (prop);
		} else {
			_tmp2 = FALSE;
		}
		_tmp0 = (_tmp2);
	}
	if (_tmp0) {
		return;
	}
	if (vala_property_get_no_accessor_method (prop)) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[NoAccessorMethod]");
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) prop);
	if (vala_property_get_binding (prop) == MEMBER_BINDING_STATIC) {
		vala_code_writer_write_string (self, "static ");
	} else {
		if (vala_property_get_is_abstract (prop)) {
			vala_code_writer_write_string (self, "abstract ");
		} else {
			if (vala_property_get_is_virtual (prop)) {
				vala_code_writer_write_string (self, "virtual ");
			}
		}
	}
	vala_code_writer_write_type (self, vala_property_get_property_type (prop));
	if (vala_data_type_get_value_owned (vala_property_get_property_type (prop))) {
		vala_code_writer_write_string (self, "#");
	}
	vala_code_writer_write_string (self, " ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) prop));
	vala_code_writer_write_string (self, " {");
	if (vala_property_get_get_accessor (prop) != NULL) {
		vala_code_writer_write_string (self, " get");
		vala_code_writer_write_code_block (self, vala_property_accessor_get_body (vala_property_get_get_accessor (prop)));
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
			vala_code_writer_write_string (self, " set");
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (prop))) {
			vala_code_writer_write_string (self, " construct");
		}
		vala_code_writer_write_code_block (self, vala_property_accessor_get_body (vala_property_get_set_accessor (prop)));
	}
	vala_code_writer_write_string (self, " }");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaCodeWriter * self;
	GeeList* _tmp0;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (sig != NULL);
	if (!vala_code_writer_check_accessibility (self, (ValaSymbol*) sig)) {
		return;
	}
	if (vala_signal_get_has_emitter (sig)) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "[HasEmitter]");
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_accessibility (self, (ValaSymbol*) sig);
	if (vala_signal_get_is_virtual (sig)) {
		vala_code_writer_write_string (self, "virtual ");
	}
	vala_code_writer_write_string (self, "signal ");
	vala_code_writer_write_return_type (self, vala_signal_get_return_type (sig));
	vala_code_writer_write_string (self, " ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) sig));
	vala_code_writer_write_string (self, " ");
	_tmp0 = NULL;
	vala_code_writer_write_params (self, _tmp0 = vala_signal_get_parameters (sig));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL));
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (b != NULL);
	vala_code_writer_write_begin_block (self);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* stmt_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		stmt_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_block_get_statements (b))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (stmt_it)) {
			ValaStatement* stmt;
			stmt = (ValaStatement*) gee_iterator_get (stmt_it);
			vala_code_node_accept ((ValaCodeNode*) stmt, (ValaCodeVisitor*) self);
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
		}
		(stmt_it == NULL) ? NULL : (stmt_it = (gee_collection_object_unref (stmt_it), NULL));
	}
	vala_code_writer_write_end_block (self);
}


static void vala_code_writer_real_visit_empty_statement (ValaCodeVisitor* base, ValaEmptyStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
}


static void vala_code_writer_real_visit_declaration_statement (ValaCodeVisitor* base, ValaDeclarationStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_node_accept ((ValaCodeNode*) vala_declaration_statement_get_declaration (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (local != NULL);
	vala_code_writer_write_type (self, vala_local_variable_get_variable_type (local));
	vala_code_writer_write_string (self, " ");
	vala_code_writer_write_identifier (self, vala_symbol_get_name ((ValaSymbol*) local));
	if (vala_local_variable_get_initializer (local) != NULL) {
		vala_code_writer_write_string (self, " = ");
		vala_code_node_accept ((ValaCodeNode*) vala_local_variable_get_initializer (local), (ValaCodeVisitor*) self);
	}
}


static void vala_code_writer_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (list != NULL);
}


static void vala_code_writer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_node_accept ((ValaCodeNode*) vala_expression_statement_get_expression (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "if (");
	vala_code_node_accept ((ValaCodeNode*) vala_if_statement_get_condition (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ")");
	vala_code_node_accept ((ValaCodeNode*) vala_if_statement_get_true_statement (stmt), (ValaCodeVisitor*) self);
	if (vala_if_statement_get_false_statement (stmt) != NULL) {
		vala_code_writer_write_string (self, " else");
		vala_code_node_accept ((ValaCodeNode*) vala_if_statement_get_false_statement (stmt), (ValaCodeVisitor*) self);
	}
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_switch_statement (ValaCodeVisitor* base, ValaSwitchStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "switch (");
	vala_code_node_accept ((ValaCodeNode*) vala_switch_statement_get_expression (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ") {");
	vala_code_writer_write_newline (self);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* section_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		section_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_switch_statement_get_sections (stmt))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (section_it)) {
			ValaSwitchSection* section;
			section = (ValaSwitchSection*) gee_iterator_get (section_it);
			vala_code_node_accept ((ValaCodeNode*) section, (ValaCodeVisitor*) self);
			(section == NULL) ? NULL : (section = (vala_code_node_unref (section), NULL));
		}
		(section_it == NULL) ? NULL : (section_it = (gee_collection_object_unref (section_it), NULL));
	}
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "}");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (section != NULL);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* label_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		label_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_switch_section_get_labels (section))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (label_it)) {
			ValaSwitchLabel* label;
			label = (ValaSwitchLabel*) gee_iterator_get (label_it);
			vala_code_node_accept ((ValaCodeNode*) label, (ValaCodeVisitor*) self);
			(label == NULL) ? NULL : (label = (vala_code_node_unref (label), NULL));
		}
		(label_it == NULL) ? NULL : (label_it = (gee_collection_object_unref (label_it), NULL));
	}
	vala_code_visitor_visit_block ((ValaCodeVisitor*) self, (ValaBlock*) section);
}


static void vala_code_writer_real_visit_switch_label (ValaCodeVisitor* base, ValaSwitchLabel* label) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (label != NULL);
	if (vala_switch_label_get_expression (label) != NULL) {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "case ");
		vala_code_node_accept ((ValaCodeNode*) vala_switch_label_get_expression (label), (ValaCodeVisitor*) self);
		vala_code_writer_write_string (self, ":");
		vala_code_writer_write_newline (self);
	} else {
		vala_code_writer_write_indent (self);
		vala_code_writer_write_string (self, "default:");
		vala_code_writer_write_newline (self);
	}
}


static void vala_code_writer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "while (");
	vala_code_node_accept ((ValaCodeNode*) vala_while_statement_get_condition (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ")");
	vala_code_node_accept ((ValaCodeNode*) vala_while_statement_get_body (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "do");
	vala_code_node_accept ((ValaCodeNode*) vala_do_statement_get_body (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, "while (");
	vala_code_node_accept ((ValaCodeNode*) vala_do_statement_get_condition (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ");");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaCodeWriter * self;
	gboolean first;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "for (");
	first = TRUE;
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* initializer_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		initializer_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_for_statement_get_initializer (stmt))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (initializer_it)) {
			ValaExpression* initializer;
			initializer = (ValaExpression*) gee_iterator_get (initializer_it);
			if (!first) {
				vala_code_writer_write_string (self, ", ");
			}
			first = FALSE;
			vala_code_node_accept ((ValaCodeNode*) initializer, (ValaCodeVisitor*) self);
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
		}
		(initializer_it == NULL) ? NULL : (initializer_it = (gee_collection_object_unref (initializer_it), NULL));
	}
	vala_code_writer_write_string (self, "; ");
	vala_code_node_accept ((ValaCodeNode*) vala_for_statement_get_condition (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, "; ");
	first = TRUE;
	{
		GeeList* _tmp2;
		GeeIterator* _tmp3;
		GeeIterator* iterator_it;
		_tmp2 = NULL;
		_tmp3 = NULL;
		iterator_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = vala_for_statement_get_iterator (stmt))), (_tmp2 == NULL) ? NULL : (_tmp2 = (gee_collection_object_unref (_tmp2), NULL)), _tmp3);
		while (gee_iterator_next (iterator_it)) {
			ValaExpression* iterator;
			iterator = (ValaExpression*) gee_iterator_get (iterator_it);
			if (!first) {
				vala_code_writer_write_string (self, ", ");
			}
			first = FALSE;
			vala_code_node_accept ((ValaCodeNode*) iterator, (ValaCodeVisitor*) self);
			(iterator == NULL) ? NULL : (iterator = (vala_code_node_unref (iterator), NULL));
		}
		(iterator_it == NULL) ? NULL : (iterator_it = (gee_collection_object_unref (iterator_it), NULL));
	}
	vala_code_writer_write_string (self, ")");
	vala_code_node_accept ((ValaCodeNode*) vala_for_statement_get_body (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
}


static void vala_code_writer_real_visit_break_statement (ValaCodeVisitor* base, ValaBreakStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "break;");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_continue_statement (ValaCodeVisitor* base, ValaContinueStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "continue;");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "return");
	if (vala_return_statement_get_return_expression (stmt) != NULL) {
		vala_code_writer_write_string (self, " ");
		vala_code_node_accept ((ValaCodeNode*) vala_return_statement_get_return_expression (stmt), (ValaCodeVisitor*) self);
	}
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_yield_statement (ValaCodeVisitor* base, ValaYieldStatement* y) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (y != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "yield");
	if (vala_yield_statement_get_yield_expression (y) != NULL) {
		vala_code_writer_write_string (self, " ");
		vala_code_node_accept ((ValaCodeNode*) vala_yield_statement_get_yield_expression (y), (ValaCodeVisitor*) self);
	}
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "throw");
	if (vala_throw_statement_get_error_expression (stmt) != NULL) {
		vala_code_writer_write_string (self, " ");
		vala_code_node_accept ((ValaCodeNode*) vala_throw_statement_get_error_expression (stmt), (ValaCodeVisitor*) self);
	}
	vala_code_writer_write_string (self, ";");
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
	vala_code_writer_write_indent (self);
	vala_code_writer_write_string (self, "try");
	vala_code_node_accept ((ValaCodeNode*) vala_try_statement_get_body (stmt), (ValaCodeVisitor*) self);
	vala_code_writer_write_newline (self);
}


static void vala_code_writer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (clause != NULL);
}


static void vala_code_writer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
}


static void vala_code_writer_real_visit_delete_statement (ValaCodeVisitor* base, ValaDeleteStatement* stmt) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (stmt != NULL);
}


static void vala_code_writer_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
}


static void vala_code_writer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* lit) {
	ValaCodeWriter * self;
	char* _tmp0;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (lit != NULL);
	_tmp0 = NULL;
	vala_code_writer_write_string (self, _tmp0 = bool_to_string (vala_boolean_literal_get_value (lit)));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_code_writer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* lit) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (lit != NULL);
	vala_code_writer_write_string (self, vala_character_literal_get_value (lit));
}


static void vala_code_writer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* lit) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (lit != NULL);
	vala_code_writer_write_string (self, vala_integer_literal_get_value (lit));
}


static void vala_code_writer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* lit) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (lit != NULL);
	vala_code_writer_write_string (self, vala_real_literal_get_value (lit));
}


static void vala_code_writer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* lit) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (lit != NULL);
	vala_code_writer_write_string (self, vala_string_literal_get_value (lit));
}


static void vala_code_writer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* lit) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (lit != NULL);
	vala_code_writer_write_string (self, "null");
}


static void vala_code_writer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "(");
	vala_code_node_accept ((ValaCodeNode*) vala_parenthesized_expression_get_inner (expr), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ")");
}


static void vala_code_writer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	if (vala_member_access_get_inner (expr) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_member_access_get_inner (expr), (ValaCodeVisitor*) self);
		vala_code_writer_write_string (self, ".");
	}
	vala_code_writer_write_identifier (self, vala_member_access_get_member_name (expr));
}


static void vala_code_writer_real_visit_method_call (ValaCodeVisitor* base, ValaMethodCall* expr) {
	ValaCodeWriter * self;
	gboolean first;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_method_call_get_call (expr), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, " (");
	first = TRUE;
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* arg_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		arg_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_method_call_get_argument_list (expr))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (arg_it);
			if (!first) {
				vala_code_writer_write_string (self, ", ");
			}
			first = FALSE;
			vala_code_node_accept ((ValaCodeNode*) arg, (ValaCodeVisitor*) self);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL));
	}
	vala_code_writer_write_string (self, ")");
}


static void vala_code_writer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr) {
	ValaCodeWriter * self;
	gboolean first;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_element_access_get_container (expr), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, "[");
	first = TRUE;
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* index_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		index_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_element_access_get_indices (expr))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (index_it)) {
			ValaExpression* index;
			index = (ValaExpression*) gee_iterator_get (index_it);
			if (!first) {
				vala_code_writer_write_string (self, ", ");
			}
			first = FALSE;
			vala_code_node_accept ((ValaCodeNode*) index, (ValaCodeVisitor*) self);
			(index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL));
		}
		(index_it == NULL) ? NULL : (index_it = (gee_collection_object_unref (index_it), NULL));
	}
	vala_code_writer_write_string (self, "]");
}


static void vala_code_writer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "base");
}


static void vala_code_writer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_postfix_expression_get_inner (expr), (ValaCodeVisitor*) self);
	if (vala_postfix_expression_get_increment (expr)) {
		vala_code_writer_write_string (self, "++");
	} else {
		vala_code_writer_write_string (self, "--");
	}
}


static void vala_code_writer_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr) {
	ValaCodeWriter * self;
	gboolean first;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "new ");
	vala_code_writer_write_type (self, vala_object_creation_expression_get_type_reference (expr));
	vala_code_writer_write_string (self, " (");
	first = TRUE;
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* arg_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		arg_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_object_creation_expression_get_argument_list (expr))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (arg_it);
			if (!first) {
				vala_code_writer_write_string (self, ", ");
			}
			first = FALSE;
			vala_code_node_accept ((ValaCodeNode*) arg, (ValaCodeVisitor*) self);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL));
	}
	vala_code_writer_write_string (self, ")");
}


static void vala_code_writer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "sizeof (");
	vala_code_writer_write_type (self, vala_sizeof_expression_get_type_reference (expr));
	vala_code_writer_write_string (self, ")");
}


static void vala_code_writer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "typeof (");
	vala_code_writer_write_type (self, vala_typeof_expression_get_type_reference (expr));
	vala_code_writer_write_string (self, ")");
}


static void vala_code_writer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	switch (vala_unary_expression_get_operator (expr)) {
		case VALA_UNARY_OPERATOR_PLUS:
		{
			vala_code_writer_write_string (self, "+");
			break;
		}
		case VALA_UNARY_OPERATOR_MINUS:
		{
			vala_code_writer_write_string (self, "-");
			break;
		}
		case VALA_UNARY_OPERATOR_LOGICAL_NEGATION:
		{
			vala_code_writer_write_string (self, "!");
			break;
		}
		case VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT:
		{
			vala_code_writer_write_string (self, "~");
			break;
		}
		case VALA_UNARY_OPERATOR_INCREMENT:
		{
			vala_code_writer_write_string (self, "++");
			break;
		}
		case VALA_UNARY_OPERATOR_DECREMENT:
		{
			vala_code_writer_write_string (self, "--");
			break;
		}
		case VALA_UNARY_OPERATOR_REF:
		{
			vala_code_writer_write_string (self, "ref ");
			break;
		}
		case VALA_UNARY_OPERATOR_OUT:
		{
			vala_code_writer_write_string (self, "out ");
			break;
		}
		default:
		{
			g_assert_not_reached ();
		}
	}
	vala_code_node_accept ((ValaCodeNode*) vala_unary_expression_get_inner (expr), (ValaCodeVisitor*) self);
}


static void vala_code_writer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	if (!vala_cast_expression_get_is_silent_cast (expr)) {
		vala_code_writer_write_string (self, "(");
		vala_code_writer_write_type (self, vala_cast_expression_get_type_reference (expr));
		vala_code_writer_write_string (self, ") ");
	}
	vala_code_node_accept ((ValaCodeNode*) vala_cast_expression_get_inner (expr), (ValaCodeVisitor*) self);
	if (vala_cast_expression_get_is_silent_cast (expr)) {
		vala_code_writer_write_string (self, " as ");
		vala_code_writer_write_type (self, vala_cast_expression_get_type_reference (expr));
	}
}


static void vala_code_writer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "*");
	vala_code_node_accept ((ValaCodeNode*) vala_pointer_indirection_get_inner (expr), (ValaCodeVisitor*) self);
}


static void vala_code_writer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "&");
	vala_code_node_accept ((ValaCodeNode*) vala_addressof_expression_get_inner (expr), (ValaCodeVisitor*) self);
}


static void vala_code_writer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_writer_write_string (self, "#");
	vala_code_node_accept ((ValaCodeNode*) vala_reference_transfer_expression_get_inner (expr), (ValaCodeVisitor*) self);
}


static void vala_code_writer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_binary_expression_get_left (expr), (ValaCodeVisitor*) self);
	switch (vala_binary_expression_get_operator (expr)) {
		case VALA_BINARY_OPERATOR_PLUS:
		{
			vala_code_writer_write_string (self, " + ");
			break;
		}
		case VALA_BINARY_OPERATOR_MINUS:
		{
			vala_code_writer_write_string (self, " - ");
			break;
		}
		case VALA_BINARY_OPERATOR_MUL:
		{
			vala_code_writer_write_string (self, " * ");
			break;
		}
		case VALA_BINARY_OPERATOR_DIV:
		{
			vala_code_writer_write_string (self, " / ");
			break;
		}
		case VALA_BINARY_OPERATOR_MOD:
		{
			vala_code_writer_write_string (self, " % ");
			break;
		}
		case VALA_BINARY_OPERATOR_SHIFT_LEFT:
		{
			vala_code_writer_write_string (self, " << ");
			break;
		}
		case VALA_BINARY_OPERATOR_SHIFT_RIGHT:
		{
			vala_code_writer_write_string (self, " >> ");
			break;
		}
		case VALA_BINARY_OPERATOR_LESS_THAN:
		{
			vala_code_writer_write_string (self, " < ");
			break;
		}
		case VALA_BINARY_OPERATOR_GREATER_THAN:
		{
			vala_code_writer_write_string (self, " > ");
			break;
		}
		case VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL:
		{
			vala_code_writer_write_string (self, " <= ");
			break;
		}
		case VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL:
		{
			vala_code_writer_write_string (self, " >= ");
			break;
		}
		case VALA_BINARY_OPERATOR_EQUALITY:
		{
			vala_code_writer_write_string (self, " == ");
			break;
		}
		case VALA_BINARY_OPERATOR_INEQUALITY:
		{
			vala_code_writer_write_string (self, " != ");
			break;
		}
		case VALA_BINARY_OPERATOR_BITWISE_AND:
		{
			vala_code_writer_write_string (self, " & ");
			break;
		}
		case VALA_BINARY_OPERATOR_BITWISE_OR:
		{
			vala_code_writer_write_string (self, " | ");
			break;
		}
		case VALA_BINARY_OPERATOR_BITWISE_XOR:
		{
			vala_code_writer_write_string (self, " ^ ");
			break;
		}
		case VALA_BINARY_OPERATOR_AND:
		{
			vala_code_writer_write_string (self, " && ");
			break;
		}
		case VALA_BINARY_OPERATOR_OR:
		{
			vala_code_writer_write_string (self, " || ");
			break;
		}
		case VALA_BINARY_OPERATOR_IN:
		{
			vala_code_writer_write_string (self, " in ");
			break;
		}
		default:
		{
			g_assert_not_reached ();
		}
	}
	vala_code_node_accept ((ValaCodeNode*) vala_binary_expression_get_right (expr), (ValaCodeVisitor*) self);
}


static void vala_code_writer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_typecheck_get_expression (expr), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, " is ");
	vala_code_writer_write_type (self, vala_typecheck_get_type_reference (expr));
}


static void vala_code_writer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_conditional_expression_get_condition (expr), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, "?");
	vala_code_node_accept ((ValaCodeNode*) vala_conditional_expression_get_true_expression (expr), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, ":");
	vala_code_node_accept ((ValaCodeNode*) vala_conditional_expression_get_false_expression (expr), (ValaCodeVisitor*) self);
}


static void vala_code_writer_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* expr) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (expr != NULL);
}


static void vala_code_writer_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a) {
	ValaCodeWriter * self;
	self = (ValaCodeWriter*) base;
	g_return_if_fail (a != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_left (a), (ValaCodeVisitor*) self);
	vala_code_writer_write_string (self, " = ");
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_right (a), (ValaCodeVisitor*) self);
}


static void vala_code_writer_write_indent (ValaCodeWriter* self) {
	gint i;
	g_return_if_fail (self != NULL);
	i = 0;
	if (!self->priv->bol) {
		fputc ('\n', self->priv->stream);
	}
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
	self->priv->bol = FALSE;
}


static void vala_code_writer_write_identifier (ValaCodeWriter* self, const char* s) {
	gboolean _tmp0;
	gboolean _tmp1;
	gboolean _tmp2;
	gboolean _tmp3;
	gboolean _tmp4;
	gboolean _tmp5;
	gboolean _tmp6;
	gboolean _tmp7;
	gboolean _tmp8;
	gboolean _tmp9;
	gboolean _tmp10;
	gboolean _tmp11;
	gboolean _tmp12;
	gboolean _tmp13;
	gboolean _tmp14;
	g_return_if_fail (self != NULL);
	g_return_if_fail (s != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	_tmp3 = FALSE;
	_tmp4 = FALSE;
	_tmp5 = FALSE;
	_tmp6 = FALSE;
	_tmp7 = FALSE;
	_tmp8 = FALSE;
	_tmp9 = FALSE;
	_tmp10 = FALSE;
	_tmp11 = FALSE;
	_tmp12 = FALSE;
	_tmp13 = FALSE;
	_tmp14 = FALSE;
	if (_vala_strcmp0 (s, "base") == 0) {
		_tmp14 = TRUE;
	} else {
		_tmp14 = _vala_strcmp0 (s, "break") == 0;
	}
	if (_tmp14) {
		_tmp13 = TRUE;
	} else {
		_tmp13 = _vala_strcmp0 (s, "class") == 0;
	}
	if (_tmp13) {
		_tmp12 = TRUE;
	} else {
		_tmp12 = _vala_strcmp0 (s, "construct") == 0;
	}
	if (_tmp12) {
		_tmp11 = TRUE;
	} else {
		_tmp11 = _vala_strcmp0 (s, "delegate") == 0;
	}
	if (_tmp11) {
		_tmp10 = TRUE;
	} else {
		_tmp10 = _vala_strcmp0 (s, "delete") == 0;
	}
	if (_tmp10) {
		_tmp9 = TRUE;
	} else {
		_tmp9 = _vala_strcmp0 (s, "do") == 0;
	}
	if (_tmp9) {
		_tmp8 = TRUE;
	} else {
		_tmp8 = _vala_strcmp0 (s, "foreach") == 0;
	}
	if (_tmp8) {
		_tmp7 = TRUE;
	} else {
		_tmp7 = _vala_strcmp0 (s, "in") == 0;
	}
	if (_tmp7) {
		_tmp6 = TRUE;
	} else {
		_tmp6 = _vala_strcmp0 (s, "interface") == 0;
	}
	if (_tmp6) {
		_tmp5 = TRUE;
	} else {
		_tmp5 = _vala_strcmp0 (s, "lock") == 0;
	}
	if (_tmp5) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = _vala_strcmp0 (s, "namespace") == 0;
	}
	if (_tmp4) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = _vala_strcmp0 (s, "new") == 0;
	}
	if (_tmp3) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = _vala_strcmp0 (s, "out") == 0;
	}
	if (_tmp2) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = _vala_strcmp0 (s, "ref") == 0;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = _vala_strcmp0 (s, "signal") == 0;
	}
	if (_tmp0) {
		fputc ('@', self->priv->stream);
	}
	vala_code_writer_write_string (self, s);
}


static void vala_code_writer_write_return_type (ValaCodeWriter* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	if (vala_code_writer_is_weak (self, type)) {
		vala_code_writer_write_string (self, "weak ");
	}
	vala_code_writer_write_type (self, type);
}


static gboolean vala_code_writer_is_weak (ValaCodeWriter* self, ValaDataType* type) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type != NULL, FALSE);
	if (vala_data_type_get_value_owned (type)) {
		return FALSE;
	} else {
		gboolean _tmp1;
		_tmp1 = FALSE;
		if (VALA_IS_VOID_TYPE (type)) {
			_tmp1 = TRUE;
		} else {
			_tmp1 = VALA_IS_POINTER_TYPE (type);
		}
		if (_tmp1) {
			return FALSE;
		} else {
			if (VALA_IS_VALUE_TYPE (type)) {
				if (vala_data_type_get_nullable (type)) {
					/* nullable structs are heap allocated*/
					return TRUE;
				}
				/* TODO return true for structs with destroy*/
				return FALSE;
			}
		}
	}
	return TRUE;
}


static void vala_code_writer_write_type (ValaCodeWriter* self, ValaDataType* type) {
	char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	_tmp0 = NULL;
	vala_code_writer_write_string (self, _tmp0 = vala_data_type_to_qualified_string (type, self->priv->current_scope));
	_tmp0 = (g_free (_tmp0), NULL);
}


static void vala_code_writer_write_string (ValaCodeWriter* self, const char* s) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (s != NULL);
	fprintf (self->priv->stream, "%s", s);
	self->priv->bol = FALSE;
}


static void vala_code_writer_write_newline (ValaCodeWriter* self) {
	g_return_if_fail (self != NULL);
	fputc ('\n', self->priv->stream);
	self->priv->bol = TRUE;
}


static void vala_code_writer_write_code_block (ValaCodeWriter* self, ValaBlock* block) {
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	_tmp0 = FALSE;
	if (block == NULL) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = !self->priv->dump_tree;
	}
	if (_tmp0) {
		vala_code_writer_write_string (self, ";");
		return;
	}
	vala_code_node_accept ((ValaCodeNode*) block, (ValaCodeVisitor*) self);
}


static void vala_code_writer_write_begin_block (ValaCodeWriter* self) {
	g_return_if_fail (self != NULL);
	if (!self->priv->bol) {
		fputc (' ', self->priv->stream);
	} else {
		vala_code_writer_write_indent (self);
	}
	fputc ('{', self->priv->stream);
	vala_code_writer_write_newline (self);
	self->priv->indent++;
}


static void vala_code_writer_write_end_block (ValaCodeWriter* self) {
	g_return_if_fail (self != NULL);
	self->priv->indent--;
	vala_code_writer_write_indent (self);
	fprintf (self->priv->stream, "}");
}


static gboolean vala_code_writer_check_accessibility (ValaCodeWriter* self, ValaSymbol* sym) {
	gboolean _tmp0;
	gboolean _tmp1;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (sym != NULL, FALSE);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->dump_tree) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED;
	}
	if (_tmp0) {
		return TRUE;
	}
	return FALSE;
}


static void vala_code_writer_write_accessibility (ValaCodeWriter* self, ValaSymbol* sym) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (sym != NULL);
	if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
		vala_code_writer_write_string (self, "public ");
	} else {
		if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED) {
			vala_code_writer_write_string (self, "protected ");
		} else {
			if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_INTERNAL) {
				vala_code_writer_write_string (self, "internal ");
			} else {
				if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
					vala_code_writer_write_string (self, "private ");
				}
			}
		}
	}
}


static void vala_code_writer_class_init (ValaCodeWriterClass * klass) {
	vala_code_writer_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_code_writer_finalize;
	g_type_class_add_private (klass, sizeof (ValaCodeWriterPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_code_writer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_code_writer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_code_writer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_code_writer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_code_writer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_code_writer_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_code = vala_code_writer_real_visit_error_code;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_code_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_code_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_code_writer_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_code_writer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_code_writer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_code_writer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_code_writer_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_code_writer_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_empty_statement = vala_code_writer_real_visit_empty_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_declaration_statement = vala_code_writer_real_visit_declaration_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_local_variable = vala_code_writer_real_visit_local_variable;
	VALA_CODE_VISITOR_CLASS (klass)->visit_initializer_list = vala_code_writer_real_visit_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_code_writer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_code_writer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_statement = vala_code_writer_real_visit_switch_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_code_writer_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_label = vala_code_writer_real_visit_switch_label;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_code_writer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_code_writer_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_code_writer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_code_writer_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_break_statement = vala_code_writer_real_visit_break_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_continue_statement = vala_code_writer_real_visit_continue_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_code_writer_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_yield_statement = vala_code_writer_real_visit_yield_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_code_writer_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_code_writer_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_code_writer_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_code_writer_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delete_statement = vala_code_writer_real_visit_delete_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_array_creation_expression = vala_code_writer_real_visit_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_code_writer_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_code_writer_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_code_writer_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_code_writer_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_code_writer_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_code_writer_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_code_writer_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_code_writer_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method_call = vala_code_writer_real_visit_method_call;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_code_writer_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_code_writer_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_code_writer_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_object_creation_expression = vala_code_writer_real_visit_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_code_writer_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_code_writer_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_code_writer_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_code_writer_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_code_writer_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_code_writer_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_code_writer_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_code_writer_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_code_writer_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_code_writer_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_code_writer_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_code_writer_real_visit_assignment;
}


static void vala_code_writer_instance_init (ValaCodeWriter * self) {
	self->priv = VALA_CODE_WRITER_GET_PRIVATE (self);
	self->priv->bol = TRUE;
}


static void vala_code_writer_finalize (ValaCodeVisitor* obj) {
	ValaCodeWriter * self;
	self = VALA_CODE_WRITER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->stream == NULL) ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL));
	(self->priv->current_scope == NULL) ? NULL : (self->priv->current_scope = (vala_scope_unref (self->priv->current_scope), NULL));
	VALA_CODE_VISITOR_CLASS (vala_code_writer_parent_class)->finalize (obj);
}


GType vala_code_writer_get_type (void) {
	static GType vala_code_writer_type_id = 0;
	if (vala_code_writer_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeWriter), 0, (GInstanceInitFunc) vala_code_writer_instance_init, NULL };
		vala_code_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaCodeWriter", &g_define_type_info, 0);
	}
	return vala_code_writer_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




