/* valacharacterliteral.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valacharacterliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaexpression.h>




struct _ValaCharacterLiteralPrivate {
	char* _value;
};

#define VALA_CHARACTER_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CHARACTER_LITERAL, ValaCharacterLiteralPrivate))
enum  {
	VALA_CHARACTER_LITERAL_DUMMY_PROPERTY
};
static void vala_character_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_character_literal_real_is_pure (ValaExpression* base);
static gpointer vala_character_literal_parent_class = NULL;
static void vala_character_literal_finalize (ValaCodeNode* obj);



/**
 * Creates a new character literal.
 *
 * @param c      character
 * @param source reference to source code
 * @return       newly created character literal
 */
ValaCharacterLiteral* vala_character_literal_construct (GType object_type, const char* c, ValaSourceReference* source) {
	ValaCharacterLiteral* self;
	g_return_val_if_fail (c != NULL, NULL);
	g_return_val_if_fail (source != NULL, NULL);
	self = ((ValaCharacterLiteral*) (g_type_create_instance (object_type)));
	vala_character_literal_set_value (self, c);
	vala_code_node_set_source_reference (((ValaCodeNode*) (self)), source);
	return self;
}


ValaCharacterLiteral* vala_character_literal_new (const char* c, ValaSourceReference* source) {
	return vala_character_literal_construct (VALA_TYPE_CHARACTER_LITERAL, c, source);
}


static void vala_character_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaCharacterLiteral * self;
	self = ((ValaCharacterLiteral*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_character_literal (visitor, self);
	vala_code_visitor_visit_expression (visitor, ((ValaExpression*) (self)));
}


/**
 * Returns the unicode character value this character literal
 * represents.
 *
 * @return unicode character value
 */
gunichar vala_character_literal_get_char (ValaCharacterLiteral* self) {
	g_return_val_if_fail (self != NULL, 0U);
	return g_utf8_get_char (g_utf8_next_char (vala_character_literal_get_value (self)));
}


static gboolean vala_character_literal_real_is_pure (ValaExpression* base) {
	ValaCharacterLiteral * self;
	self = ((ValaCharacterLiteral*) (base));
	return TRUE;
}


const char* vala_character_literal_get_value (ValaCharacterLiteral* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value;
}


void vala_character_literal_set_value (ValaCharacterLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_value = (g_free (self->priv->_value), NULL)), _tmp2);
	if (!g_utf8_validate (value, -1, NULL) || (g_utf8_strlen (value, -1) != 3 && g_utf8_get_char (g_utf8_next_char (value)) != '\\')) {
		vala_code_node_set_error (((ValaCodeNode*) (self)), TRUE);
	}
}


static void vala_character_literal_class_init (ValaCharacterLiteralClass * klass) {
	vala_character_literal_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_character_literal_finalize;
	g_type_class_add_private (klass, sizeof (ValaCharacterLiteralPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_character_literal_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_character_literal_real_is_pure;
}


static void vala_character_literal_instance_init (ValaCharacterLiteral * self) {
	self->priv = VALA_CHARACTER_LITERAL_GET_PRIVATE (self);
}


static void vala_character_literal_finalize (ValaCodeNode* obj) {
	ValaCharacterLiteral * self;
	self = VALA_CHARACTER_LITERAL (obj);
	self->priv->_value = (g_free (self->priv->_value), NULL);
	VALA_CODE_NODE_CLASS (vala_character_literal_parent_class)->finalize (obj);
}


GType vala_character_literal_get_type (void) {
	static GType vala_character_literal_type_id = 0;
	if (vala_character_literal_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCharacterLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_character_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCharacterLiteral), 0, (GInstanceInitFunc) vala_character_literal_instance_init, NULL };
		vala_character_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaCharacterLiteral", &g_define_type_info, 0);
	}
	return vala_character_literal_type_id;
}




