/* valaobjectcreationexpression.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaobjectcreationexpression.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valamethod.h>
#include <vala/valamemberaccess.h>
#include <vala/valamemberinitializer.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaObjectCreationExpressionPrivate {
	ValaMethod* _constructor;
	ValaMemberAccess* _member_name;
	gboolean _struct_creation;
	GeeList* argument_list;
	GeeList* object_initializer;
	ValaDataType* _data_type;
};

#define VALA_OBJECT_CREATION_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_OBJECT_CREATION_EXPRESSION, ValaObjectCreationExpressionPrivate))
enum  {
	VALA_OBJECT_CREATION_EXPRESSION_DUMMY_PROPERTY
};
static void vala_object_creation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_object_creation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_object_creation_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_object_creation_expression_real_is_pure (ValaExpression* base);
static void vala_object_creation_expression_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_object_creation_expression_parent_class = NULL;
static void vala_object_creation_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new object creation expression.
 *
 * @param member_name      object type to create
 * @param source_reference reference to source code
 * @return                 newly created object creation expression
 */
ValaObjectCreationExpression* vala_object_creation_expression_construct (GType object_type, ValaMemberAccess* member_name, ValaSourceReference* source_reference) {
	ValaObjectCreationExpression* self;
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (member_name), NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaObjectCreationExpression*) (g_type_create_instance (object_type)));
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	vala_object_creation_expression_set_member_name (self, member_name);
	return self;
}


ValaObjectCreationExpression* vala_object_creation_expression_new (ValaMemberAccess* member_name, ValaSourceReference* source_reference) {
	return vala_object_creation_expression_construct (VALA_TYPE_OBJECT_CREATION_EXPRESSION, member_name, source_reference);
}


/**
 * Appends the specified expression to the list of arguments.
 *
 * @param arg an argument
 */
void vala_object_creation_expression_add_argument (ValaObjectCreationExpression* self, ValaExpression* arg) {
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	g_return_if_fail (VALA_IS_EXPRESSION (arg));
	gee_collection_add (GEE_COLLECTION (self->priv->argument_list), arg);
	vala_code_node_set_parent_node (VALA_CODE_NODE (arg), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the argument list.
 *
 * @return argument list
 */
GeeList* vala_object_creation_expression_get_argument_list (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->argument_list));
}


/**
 * Appends the specified member initializer to the object initializer.
 *
 * @param init a member initializer
 */
void vala_object_creation_expression_add_member_initializer (ValaObjectCreationExpression* self, ValaMemberInitializer* init) {
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	g_return_if_fail (VALA_IS_MEMBER_INITIALIZER (init));
	gee_collection_add (GEE_COLLECTION (self->priv->object_initializer), init);
	vala_code_node_set_parent_node (VALA_CODE_NODE (init), VALA_CODE_NODE (self));
}


/**
 * Returns the object initializer.
 *
 * @return member initializer list
 */
GeeList* vala_object_creation_expression_get_object_initializer (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_MEMBER_INITIALIZER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->object_initializer));
}


static void vala_object_creation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_object_creation_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, VALA_EXPRESSION (self));
}


static void vala_object_creation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_object_creation_expression_get_type_reference (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_object_creation_expression_get_type_reference (self)), visitor);
	}
	if (self->priv->_member_name != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (self->priv->_member_name), visitor);
	}
	{
		GeeList* arg_collection;
		int arg_it;
		arg_collection = self->priv->argument_list;
		for (arg_it = 0; arg_it < gee_collection_get_size (GEE_COLLECTION (arg_collection)); arg_it = arg_it + 1) {
			ValaExpression* arg;
			arg = ((ValaExpression*) (gee_list_get (GEE_LIST (arg_collection), arg_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (arg), visitor);
				(arg == NULL ? NULL : (arg = (vala_code_node_unref (arg), NULL)));
			}
		}
	}
	{
		GeeList* init_collection;
		int init_it;
		init_collection = self->priv->object_initializer;
		for (init_it = 0; init_it < gee_collection_get_size (GEE_COLLECTION (init_collection)); init_it = init_it + 1) {
			ValaMemberInitializer* init;
			init = ((ValaMemberInitializer*) (gee_list_get (GEE_LIST (init_collection), init_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (init), visitor);
				(init == NULL ? NULL : (init = (vala_code_node_unref (init), NULL)));
			}
		}
	}
}


static void vala_object_creation_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaObjectCreationExpression * self;
	gint index;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (VALA_IS_EXPRESSION (new_node));
	index = gee_list_index_of (self->priv->argument_list, old_node);
	if (index >= 0 && vala_code_node_get_parent_node (VALA_CODE_NODE (new_node)) == NULL) {
		gee_list_set (((GeeList*) (self->priv->argument_list)), index, new_node);
		vala_code_node_set_parent_node (VALA_CODE_NODE (new_node), VALA_CODE_NODE (self));
	}
}


static gboolean vala_object_creation_expression_real_is_pure (ValaExpression* base) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	return FALSE;
}


static void vala_object_creation_expression_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_object_creation_expression_get_type_reference (self) == old_type) {
		vala_object_creation_expression_set_type_reference (self, new_type);
	}
}


ValaDataType* vala_object_creation_expression_get_type_reference (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return self->priv->_data_type;
}


void vala_object_creation_expression_set_type_reference (ValaObjectCreationExpression* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


ValaMethod* vala_object_creation_expression_get_constructor (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return self->priv->_constructor;
}


void vala_object_creation_expression_set_constructor (ValaObjectCreationExpression* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (vala_code_node_unref (self->priv->_constructor), NULL))), _tmp2);
}


ValaMemberAccess* vala_object_creation_expression_get_member_name (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return self->priv->_member_name;
}


void vala_object_creation_expression_set_member_name (ValaObjectCreationExpression* self, ValaMemberAccess* value) {
	ValaMemberAccess* _tmp2;
	ValaMemberAccess* _tmp1;
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_member_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_member_name == NULL ? NULL : (self->priv->_member_name = (vala_code_node_unref (self->priv->_member_name), NULL))), _tmp2);
}


gboolean vala_object_creation_expression_get_struct_creation (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), FALSE);
	return self->priv->_struct_creation;
}


void vala_object_creation_expression_set_struct_creation (ValaObjectCreationExpression* self, gboolean value) {
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	self->priv->_struct_creation = value;
}


static void vala_object_creation_expression_class_init (ValaObjectCreationExpressionClass * klass) {
	vala_object_creation_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_object_creation_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaObjectCreationExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_object_creation_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_object_creation_expression_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_object_creation_expression_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_object_creation_expression_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_object_creation_expression_real_replace_type;
}


static void vala_object_creation_expression_instance_init (ValaObjectCreationExpression * self) {
	self->priv = VALA_OBJECT_CREATION_EXPRESSION_GET_PRIVATE (self);
	self->priv->argument_list = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
	self->priv->object_initializer = GEE_LIST (gee_array_list_new (VALA_TYPE_MEMBER_INITIALIZER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
}


static void vala_object_creation_expression_finalize (ValaCodeNode* obj) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (obj);
	(self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (vala_code_node_unref (self->priv->_constructor), NULL)));
	(self->priv->_member_name == NULL ? NULL : (self->priv->_member_name = (vala_code_node_unref (self->priv->_member_name), NULL)));
	(self->priv->argument_list == NULL ? NULL : (self->priv->argument_list = (gee_collection_object_unref (self->priv->argument_list), NULL)));
	(self->priv->object_initializer == NULL ? NULL : (self->priv->object_initializer = (gee_collection_object_unref (self->priv->object_initializer), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)));
	VALA_CODE_NODE_CLASS (vala_object_creation_expression_parent_class)->finalize (obj);
}


GType vala_object_creation_expression_get_type (void) {
	static GType vala_object_creation_expression_type_id = 0;
	if (vala_object_creation_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaObjectCreationExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_object_creation_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaObjectCreationExpression), 0, (GInstanceInitFunc) vala_object_creation_expression_instance_init, NULL };
		vala_object_creation_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaObjectCreationExpression", &g_define_type_info, 0);
	}
	return vala_object_creation_expression_type_id;
}




