/* valaifstatement.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaifstatement.h>
#include <vala/valablock.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaIfStatementPrivate {
	ValaBlock* _true_statement;
	ValaBlock* _false_statement;
	ValaExpression* _condition;
};

#define VALA_IF_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_IF_STATEMENT, ValaIfStatementPrivate))
enum  {
	VALA_IF_STATEMENT_DUMMY_PROPERTY
};
static void vala_if_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_if_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_if_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gpointer vala_if_statement_parent_class = NULL;
static ValaStatementIface* vala_if_statement_vala_statement_parent_iface = NULL;
static void vala_if_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new if statement.
 *
 * @param cond       a boolean condition
 * @param true_stmt  statement to be evaluated if condition is true
 * @param false_stmt statement to be evaluated if condition is false
 * @return           newly created if statement
 */
ValaIfStatement* vala_if_statement_construct (GType object_type, ValaExpression* cond, ValaBlock* true_stmt, ValaBlock* false_stmt, ValaSourceReference* source) {
	ValaIfStatement* self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (cond), NULL);
	g_return_val_if_fail (VALA_IS_BLOCK (true_stmt), NULL);
	g_return_val_if_fail (false_stmt == NULL || VALA_IS_BLOCK (false_stmt), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = ((ValaIfStatement*) (g_type_create_instance (object_type)));
	vala_if_statement_set_condition (self, cond);
	vala_if_statement_set_true_statement (self, true_stmt);
	vala_if_statement_set_false_statement (self, false_stmt);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


ValaIfStatement* vala_if_statement_new (ValaExpression* cond, ValaBlock* true_stmt, ValaBlock* false_stmt, ValaSourceReference* source) {
	return vala_if_statement_construct (VALA_TYPE_IF_STATEMENT, cond, true_stmt, false_stmt, source);
}


static void vala_if_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaIfStatement * self;
	self = VALA_IF_STATEMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_if_statement (visitor, self);
}


static void vala_if_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaIfStatement * self;
	self = VALA_IF_STATEMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_if_statement_get_condition (self)), visitor);
	vala_code_visitor_visit_end_full_expression (visitor, vala_if_statement_get_condition (self));
	vala_code_node_accept (VALA_CODE_NODE (self->priv->_true_statement), visitor);
	if (self->priv->_false_statement != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (self->priv->_false_statement), visitor);
	}
}


static void vala_if_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaIfStatement * self;
	self = VALA_IF_STATEMENT (base);
	g_return_if_fail (VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (VALA_IS_EXPRESSION (new_node));
	if (vala_if_statement_get_condition (self) == old_node) {
		vala_if_statement_set_condition (self, new_node);
	}
}


ValaExpression* vala_if_statement_get_condition (ValaIfStatement* self) {
	g_return_val_if_fail (VALA_IS_IF_STATEMENT (self), NULL);
	return self->priv->_condition;
}


void vala_if_statement_set_condition (ValaIfStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_IF_STATEMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_condition = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_condition == NULL ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_condition), VALA_CODE_NODE (self));
}


ValaBlock* vala_if_statement_get_true_statement (ValaIfStatement* self) {
	g_return_val_if_fail (VALA_IS_IF_STATEMENT (self), NULL);
	return self->priv->_true_statement;
}


void vala_if_statement_set_true_statement (ValaIfStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_IF_STATEMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_true_statement = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_true_statement == NULL ? NULL : (self->priv->_true_statement = (vala_code_node_unref (self->priv->_true_statement), NULL))), _tmp2);
}


ValaBlock* vala_if_statement_get_false_statement (ValaIfStatement* self) {
	g_return_val_if_fail (VALA_IS_IF_STATEMENT (self), NULL);
	return self->priv->_false_statement;
}


void vala_if_statement_set_false_statement (ValaIfStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_IF_STATEMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_false_statement = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_false_statement == NULL ? NULL : (self->priv->_false_statement = (vala_code_node_unref (self->priv->_false_statement), NULL))), _tmp2);
}


static void vala_if_statement_class_init (ValaIfStatementClass * klass) {
	vala_if_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_if_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaIfStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_if_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_if_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_if_statement_real_replace_expression;
}


static void vala_if_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_if_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_if_statement_instance_init (ValaIfStatement * self) {
	self->priv = VALA_IF_STATEMENT_GET_PRIVATE (self);
}


static void vala_if_statement_finalize (ValaCodeNode* obj) {
	ValaIfStatement * self;
	self = VALA_IF_STATEMENT (obj);
	(self->priv->_true_statement == NULL ? NULL : (self->priv->_true_statement = (vala_code_node_unref (self->priv->_true_statement), NULL)));
	(self->priv->_false_statement == NULL ? NULL : (self->priv->_false_statement = (vala_code_node_unref (self->priv->_false_statement), NULL)));
	(self->priv->_condition == NULL ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL)));
	VALA_CODE_NODE_CLASS (vala_if_statement_parent_class)->finalize (obj);
}


GType vala_if_statement_get_type (void) {
	static GType vala_if_statement_type_id = 0;
	if (vala_if_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaIfStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_if_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaIfStatement), 0, (GInstanceInitFunc) vala_if_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_if_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_if_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaIfStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_if_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_if_statement_type_id;
}




