/* valageniescanner.vala
 *
 * Copyright (C) 2008  Jamie McCracken, Jürg Billeter
 * Based on code by Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jamie McCracken jamiemcc gnome org
 */

#include <vala/valageniescanner.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <gobject/gvaluecollector.h>




struct _ValaGenieScannerPrivate {
	ValaSourceFile* _source_file;
	gint _indent_spaces;
	gchar* begin;
	gchar* current;
	gchar* end;
	gint line;
	gint column;
	gint current_indent_level;
	gint indent_level;
	gint pending_dedents;
	ValaGenieTokenType last_token;
	gboolean parse_started;
	char* _comment;
};

#define VALA_GENIE_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_GENIE_TYPE_SCANNER, ValaGenieScannerPrivate))
enum  {
	VALA_GENIE_SCANNER_DUMMY_PROPERTY
};
static gboolean vala_genie_scanner_is_ident_char (ValaGenieScanner* self, gchar c);
static ValaGenieTokenType vala_genie_scanner_get_identifier_or_keyword (ValaGenieScanner* self, gchar* begin, gint len);
static gint vala_genie_scanner_count_tabs (ValaGenieScanner* self);
static gboolean vala_genie_scanner_matches (ValaGenieScanner* self, gchar* begin, const char* keyword);
static gboolean vala_genie_scanner_whitespace (ValaGenieScanner* self);
static inline gboolean vala_genie_scanner_newline (ValaGenieScanner* self);
static gboolean vala_genie_scanner_skip_newlines (ValaGenieScanner* self);
static gboolean vala_genie_scanner_comment (ValaGenieScanner* self);
static gboolean vala_genie_scanner_skip_tabs (ValaGenieScanner* self);
static void vala_genie_scanner_skip_space_tabs (ValaGenieScanner* self);
static void vala_genie_scanner_space (ValaGenieScanner* self);
static void vala_genie_scanner_push_comment (ValaGenieScanner* self, const char* comment_item, gboolean file_comment);
static void vala_genie_scanner_set_source_file (ValaGenieScanner* self, ValaSourceFile* value);
static gpointer vala_genie_scanner_parent_class = NULL;
static void vala_genie_scanner_finalize (ValaGenieScanner* obj);



ValaGenieScanner* vala_genie_scanner_construct (GType object_type, ValaSourceFile* source_file) {
	ValaGenieScanner* self;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (source_file), NULL);
	self = ((ValaGenieScanner*) (g_type_create_instance (object_type)));
	vala_genie_scanner_set_source_file (self, source_file);
	self->priv->begin = vala_source_file_get_mapped_contents (source_file);
	self->priv->end = self->priv->begin + vala_source_file_get_mapped_length (source_file);
	self->priv->current = self->priv->begin;
	self->priv->_indent_spaces = 0;
	self->priv->line = 1;
	self->priv->column = 1;
	self->priv->current_indent_level = 0;
	self->priv->indent_level = 0;
	self->priv->pending_dedents = 0;
	self->priv->parse_started = FALSE;
	self->priv->last_token = VALA_GENIE_TOKEN_TYPE_NONE;
	return self;
}


ValaGenieScanner* vala_genie_scanner_new (ValaSourceFile* source_file) {
	return vala_genie_scanner_construct (VALA_GENIE_TYPE_SCANNER, source_file);
}


static gboolean vala_genie_scanner_is_ident_char (ValaGenieScanner* self, gchar c) {
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), FALSE);
	return (g_ascii_isalnum (c) || c == '_');
}


static ValaGenieTokenType vala_genie_scanner_get_identifier_or_keyword (ValaGenieScanner* self, gchar* begin, gint len) {
	gint _tmp107;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), 0);
	_tmp107 = len;
	if (_tmp107 == 2)
	do {
		gchar _tmp9;
		_tmp9 = begin[0];
		if (_tmp9 == 'a')
		do {
			if (vala_genie_scanner_matches (self, begin, "as")) {
				return VALA_GENIE_TOKEN_TYPE_AS;
			}
			break;
		} while (0); else if (_tmp9 == 'd')
		do {
			if (vala_genie_scanner_matches (self, begin, "do")) {
				return VALA_GENIE_TOKEN_TYPE_DO;
			}
			break;
		} while (0); else if (_tmp9 == 'i')
		do {
			gchar _tmp5;
			_tmp5 = begin[1];
			if (_tmp5 == 'f')
			do {
				return VALA_GENIE_TOKEN_TYPE_IF;
			} while (0); else if (_tmp5 == 'n')
			do {
				return VALA_GENIE_TOKEN_TYPE_IN;
			} while (0); else if (_tmp5 == 's')
			do {
				return VALA_GENIE_TOKEN_TYPE_IS;
			} while (0);
			break;
		} while (0); else if (_tmp9 == 'o')
		do {
			if (vala_genie_scanner_matches (self, begin, "of")) {
				return VALA_GENIE_TOKEN_TYPE_OF;
			}
			if (vala_genie_scanner_matches (self, begin, "or")) {
				return VALA_GENIE_TOKEN_TYPE_OP_OR;
			}
			break;
		} while (0); else if (_tmp9 == 't')
		do {
			if (vala_genie_scanner_matches (self, begin, "to")) {
				return VALA_GENIE_TOKEN_TYPE_TO;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 3)
	do {
		gchar _tmp23;
		_tmp23 = begin[0];
		if (_tmp23 == 'a')
		do {
			if (vala_genie_scanner_matches (self, begin, "and")) {
				return VALA_GENIE_TOKEN_TYPE_OP_AND;
			}
			break;
		} while (0); else if (_tmp23 == 'd')
		do {
			if (vala_genie_scanner_matches (self, begin, "def")) {
				return VALA_GENIE_TOKEN_TYPE_DEF;
			}
			break;
		} while (0); else if (_tmp23 == 'f')
		do {
			if (vala_genie_scanner_matches (self, begin, "for")) {
				return VALA_GENIE_TOKEN_TYPE_FOR;
			}
			break;
		} while (0); else if (_tmp23 == 'g')
		do {
			if (vala_genie_scanner_matches (self, begin, "get")) {
				return VALA_GENIE_TOKEN_TYPE_GET;
			}
			break;
		} while (0); else if (_tmp23 == 'i')
		do {
			if (vala_genie_scanner_matches (self, begin, "isa")) {
				return VALA_GENIE_TOKEN_TYPE_ISA;
			}
			break;
		} while (0); else if (_tmp23 == 'n')
		do {
			gchar _tmp17;
			_tmp17 = begin[1];
			if (_tmp17 == 'e')
			do {
				if (vala_genie_scanner_matches (self, begin, "new")) {
					return VALA_GENIE_TOKEN_TYPE_NEW;
				}
				break;
			} while (0); else if (_tmp17 == 'o')
			do {
				if (vala_genie_scanner_matches (self, begin, "not")) {
					return VALA_GENIE_TOKEN_TYPE_OP_NEG;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp23 == 'o')
		do {
			if (vala_genie_scanner_matches (self, begin, "out")) {
				return VALA_GENIE_TOKEN_TYPE_OUT;
			}
			break;
		} while (0); else if (_tmp23 == 'r')
		do {
			if (vala_genie_scanner_matches (self, begin, "ref")) {
				return VALA_GENIE_TOKEN_TYPE_REF;
			}
			break;
		} while (0); else if (_tmp23 == 's')
		do {
			if (vala_genie_scanner_matches (self, begin, "set")) {
				return VALA_GENIE_TOKEN_TYPE_SET;
			}
			break;
		} while (0); else if (_tmp23 == 't')
		do {
			if (vala_genie_scanner_matches (self, begin, "try")) {
				return VALA_GENIE_TOKEN_TYPE_TRY;
			}
			break;
		} while (0); else if (_tmp23 == 'v')
		do {
			if (vala_genie_scanner_matches (self, begin, "var")) {
				return VALA_GENIE_TOKEN_TYPE_VAR;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 4)
	do {
		gchar _tmp44;
		_tmp44 = begin[0];
		if (_tmp44 == 'c')
		do {
			if (vala_genie_scanner_matches (self, begin, "case")) {
				return VALA_GENIE_TOKEN_TYPE_CASE;
			}
			break;
		} while (0); else if (_tmp44 == 'd')
		do {
			if (vala_genie_scanner_matches (self, begin, "dict")) {
				return VALA_GENIE_TOKEN_TYPE_DICT;
			}
			break;
		} while (0); else if (_tmp44 == 'e')
		do {
			gchar _tmp28;
			_tmp28 = begin[1];
			if (_tmp28 == 'l')
			do {
				if (vala_genie_scanner_matches (self, begin, "else")) {
					return VALA_GENIE_TOKEN_TYPE_ELSE;
				}
				break;
			} while (0); else if (_tmp28 == 'n')
			do {
				if (vala_genie_scanner_matches (self, begin, "enum")) {
					return VALA_GENIE_TOKEN_TYPE_ENUM;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp44 == 'i')
		do {
			if (vala_genie_scanner_matches (self, begin, "init")) {
				return VALA_GENIE_TOKEN_TYPE_INIT;
			}
			break;
		} while (0); else if (_tmp44 == 'l')
		do {
			gchar _tmp32;
			_tmp32 = begin[1];
			if (_tmp32 == 'i')
			do {
				if (vala_genie_scanner_matches (self, begin, "list")) {
					return VALA_GENIE_TOKEN_TYPE_LIST;
				}
				break;
			} while (0); else if (_tmp32 == 'o')
			do {
				if (vala_genie_scanner_matches (self, begin, "lock")) {
					return VALA_GENIE_TOKEN_TYPE_LOCK;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp44 == 'n')
		do {
			if (vala_genie_scanner_matches (self, begin, "null")) {
				return VALA_GENIE_TOKEN_TYPE_NULL;
			}
			break;
		} while (0); else if (_tmp44 == 'p')
		do {
			gchar _tmp36;
			_tmp36 = begin[1];
			if (_tmp36 == 'a')
			do {
				if (vala_genie_scanner_matches (self, begin, "pass")) {
					return VALA_GENIE_TOKEN_TYPE_PASS;
				}
				break;
			} while (0); else if (_tmp36 == 'r')
			do {
				if (vala_genie_scanner_matches (self, begin, "prop")) {
					return VALA_GENIE_TOKEN_TYPE_PROP;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp44 == 's')
		do {
			if (vala_genie_scanner_matches (self, begin, "self")) {
				return VALA_GENIE_TOKEN_TYPE_THIS;
			}
			break;
		} while (0); else if (_tmp44 == 't')
		do {
			if (vala_genie_scanner_matches (self, begin, "true")) {
				return VALA_GENIE_TOKEN_TYPE_TRUE;
			}
			break;
		} while (0); else if (_tmp44 == 'u')
		do {
			if (vala_genie_scanner_matches (self, begin, "uses")) {
				return VALA_GENIE_TOKEN_TYPE_USES;
			}
			break;
		} while (0); else if (_tmp44 == 'v')
		do {
			if (vala_genie_scanner_matches (self, begin, "void")) {
				return VALA_GENIE_TOKEN_TYPE_VOID;
			}
			break;
		} while (0); else if (_tmp44 == 'w')
		do {
			gchar _tmp43;
			_tmp43 = begin[1];
			if (_tmp43 == 'e')
			do {
				if (vala_genie_scanner_matches (self, begin, "weak")) {
					return VALA_GENIE_TOKEN_TYPE_WEAK;
				}
				break;
			} while (0); else if (_tmp43 == 'h')
			do {
				if (vala_genie_scanner_matches (self, begin, "when")) {
					return VALA_GENIE_TOKEN_TYPE_WHEN;
				}
				break;
			} while (0);
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 5)
	do {
		gchar _tmp58;
		_tmp58 = begin[0];
		if (_tmp58 == 'a')
		do {
			if (vala_genie_scanner_matches (self, begin, "array")) {
				return VALA_GENIE_TOKEN_TYPE_ARRAY;
			}
			break;
		} while (0); else if (_tmp58 == 'b')
		do {
			if (vala_genie_scanner_matches (self, begin, "break")) {
				return VALA_GENIE_TOKEN_TYPE_BREAK;
			}
			break;
		} while (0); else if (_tmp58 == 'c')
		do {
			gchar _tmp49;
			_tmp49 = begin[1];
			if (_tmp49 == 'l')
			do {
				if (vala_genie_scanner_matches (self, begin, "class")) {
					return VALA_GENIE_TOKEN_TYPE_CLASS;
				}
				break;
			} while (0); else if (_tmp49 == 'o')
			do {
				if (vala_genie_scanner_matches (self, begin, "const")) {
					return VALA_GENIE_TOKEN_TYPE_CONST;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp58 == 'e')
		do {
			if (vala_genie_scanner_matches (self, begin, "event")) {
				return VALA_GENIE_TOKEN_TYPE_EVENT;
			}
			break;
		} while (0); else if (_tmp58 == 'f')
		do {
			gchar _tmp53;
			_tmp53 = begin[1];
			if (_tmp53 == 'a')
			do {
				if (vala_genie_scanner_matches (self, begin, "false")) {
					return VALA_GENIE_TOKEN_TYPE_FALSE;
				}
				break;
			} while (0); else if (_tmp53 == 'i')
			do {
				if (vala_genie_scanner_matches (self, begin, "final")) {
					return VALA_GENIE_TOKEN_TYPE_FINAL;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp58 == 'p')
		do {
			if (vala_genie_scanner_matches (self, begin, "print")) {
				return VALA_GENIE_TOKEN_TYPE_PRINT;
			}
			break;
		} while (0); else if (_tmp58 == 's')
		do {
			if (vala_genie_scanner_matches (self, begin, "super")) {
				return VALA_GENIE_TOKEN_TYPE_SUPER;
			}
			break;
		} while (0); else if (_tmp58 == 'r')
		do {
			if (vala_genie_scanner_matches (self, begin, "raise")) {
				return VALA_GENIE_TOKEN_TYPE_RAISE;
			}
			break;
		} while (0); else if (_tmp58 == 'w')
		do {
			if (vala_genie_scanner_matches (self, begin, "while")) {
				return VALA_GENIE_TOKEN_TYPE_WHILE;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 6)
	do {
		gchar _tmp78;
		_tmp78 = begin[0];
		if (_tmp78 == 'a')
		do {
			if (vala_genie_scanner_matches (self, begin, "assert")) {
				return VALA_GENIE_TOKEN_TYPE_ASSERT;
			}
			break;
		} while (0); else if (_tmp78 == 'd')
		do {
			gchar _tmp62;
			_tmp62 = begin[1];
			if (_tmp62 == 'e')
			do {
				if (vala_genie_scanner_matches (self, begin, "delete")) {
					return VALA_GENIE_TOKEN_TYPE_DELETE;
				}
				break;
			} while (0); else if (_tmp62 == 'o')
			do {
				if (vala_genie_scanner_matches (self, begin, "downto")) {
					return VALA_GENIE_TOKEN_TYPE_DOWNTO;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp78 == 'e')
		do {
			gchar _tmp66;
			_tmp66 = begin[1];
			if (_tmp66 == 'x')
			do {
				gchar _tmp65;
				_tmp65 = begin[2];
				if (_tmp65 == 'c')
				do {
					if (vala_genie_scanner_matches (self, begin, "except")) {
						return VALA_GENIE_TOKEN_TYPE_EXCEPT;
					}
					break;
				} while (0); else if (_tmp65 == 't')
				do {
					if (vala_genie_scanner_matches (self, begin, "extern")) {
						return VALA_GENIE_TOKEN_TYPE_EXTERN;
					}
					break;
				} while (0);
				break;
			} while (0);
			break;
		} while (0); else if (_tmp78 == 'i')
		do {
			if (vala_genie_scanner_matches (self, begin, "inline")) {
				return VALA_GENIE_TOKEN_TYPE_INLINE;
			}
			break;
		} while (0); else if (_tmp78 == 'p')
		do {
			if (vala_genie_scanner_matches (self, begin, "public")) {
				return VALA_GENIE_TOKEN_TYPE_PUBLIC;
			}
			break;
		} while (0); else if (_tmp78 == 'r')
		do {
			gchar _tmp71;
			_tmp71 = begin[1];
			if (_tmp71 == 'a')
			do {
				if (vala_genie_scanner_matches (self, begin, "raises")) {
					return VALA_GENIE_TOKEN_TYPE_RAISES;
				}
				break;
			} while (0); else if (_tmp71 == 'e')
			do {
				if (vala_genie_scanner_matches (self, begin, "return")) {
					return VALA_GENIE_TOKEN_TYPE_RETURN;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp78 == 's')
		do {
			gchar _tmp76;
			_tmp76 = begin[1];
			if (_tmp76 == 'i')
			do {
				if (vala_genie_scanner_matches (self, begin, "sizeof")) {
					return VALA_GENIE_TOKEN_TYPE_SIZEOF;
				}
				break;
			} while (0); else if (_tmp76 == 't')
			do {
				gchar _tmp75;
				_tmp75 = begin[2];
				if (_tmp75 == 'a')
				do {
					if (vala_genie_scanner_matches (self, begin, "static")) {
						return VALA_GENIE_TOKEN_TYPE_STATIC;
					}
					break;
				} while (0); else if (_tmp75 == 'r')
				do {
					if (vala_genie_scanner_matches (self, begin, "struct")) {
						return VALA_GENIE_TOKEN_TYPE_STRUCT;
					}
					break;
				} while (0);
				break;
			} while (0);
			break;
		} while (0); else if (_tmp78 == 't')
		do {
			if (vala_genie_scanner_matches (self, begin, "typeof")) {
				return VALA_GENIE_TOKEN_TYPE_TYPEOF;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 7)
	do {
		gchar _tmp88;
		_tmp88 = begin[0];
		if (_tmp88 == 'd')
		do {
			gchar _tmp81;
			_tmp81 = begin[1];
			if (_tmp81 == 'e')
			do {
				if (vala_genie_scanner_matches (self, begin, "default")) {
					return VALA_GENIE_TOKEN_TYPE_DEFAULT;
				}
				break;
			} while (0); else if (_tmp81 == 'y')
			do {
				if (vala_genie_scanner_matches (self, begin, "dynamic")) {
					return VALA_GENIE_TOKEN_TYPE_DYNAMIC;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp88 == 'e')
		do {
			if (vala_genie_scanner_matches (self, begin, "ensures")) {
				return VALA_GENIE_TOKEN_TYPE_ENSURES;
			}
			break;
		} while (0); else if (_tmp88 == 'f')
		do {
			gchar _tmp85;
			_tmp85 = begin[1];
			if (_tmp85 == 'i')
			do {
				if (vala_genie_scanner_matches (self, begin, "finally")) {
					return VALA_GENIE_TOKEN_TYPE_FINALLY;
				}
				break;
			} while (0); else if (_tmp85 == 'o')
			do {
				if (vala_genie_scanner_matches (self, begin, "foreach")) {
					return VALA_GENIE_TOKEN_TYPE_FOREACH;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp88 == 'p')
		do {
			if (vala_genie_scanner_matches (self, begin, "private")) {
				return VALA_GENIE_TOKEN_TYPE_PRIVATE;
			}
			break;
		} while (0); else if (_tmp88 == 'v')
		do {
			if (vala_genie_scanner_matches (self, begin, "virtual")) {
				return VALA_GENIE_TOKEN_TYPE_VIRTUAL;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 8)
	do {
		gchar _tmp97;
		_tmp97 = begin[0];
		if (_tmp97 == 'a')
		do {
			if (vala_genie_scanner_matches (self, begin, "abstract")) {
				return VALA_GENIE_TOKEN_TYPE_ABSTRACT;
			}
			break;
		} while (0); else if (_tmp97 == 'c')
		do {
			if (vala_genie_scanner_matches (self, begin, "continue")) {
				return VALA_GENIE_TOKEN_TYPE_CONTINUE;
			}
			break;
		} while (0); else if (_tmp97 == 'd')
		do {
			if (vala_genie_scanner_matches (self, begin, "delegate")) {
				return VALA_GENIE_TOKEN_TYPE_DELEGATE;
			}
			break;
		} while (0); else if (_tmp97 == 'o')
		do {
			if (vala_genie_scanner_matches (self, begin, "override")) {
				return VALA_GENIE_TOKEN_TYPE_OVERRIDE;
			}
			break;
		} while (0); else if (_tmp97 == 'r')
		do {
			gchar _tmp95;
			_tmp95 = begin[2];
			if (_tmp95 == 'a')
			do {
				if (vala_genie_scanner_matches (self, begin, "readonly")) {
					return VALA_GENIE_TOKEN_TYPE_READONLY;
				}
				break;
			} while (0); else if (_tmp95 == 'q')
			do {
				if (vala_genie_scanner_matches (self, begin, "requires")) {
					return VALA_GENIE_TOKEN_TYPE_REQUIRES;
				}
				break;
			} while (0);
			break;
		} while (0); else if (_tmp97 == 'v')
		do {
			if (vala_genie_scanner_matches (self, begin, "volatile")) {
				return VALA_GENIE_TOKEN_TYPE_VOLATILE;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 9)
	do {
		gchar _tmp104;
		_tmp104 = begin[0];
		if (_tmp104 == 'c')
		do {
			if (vala_genie_scanner_matches (self, begin, "construct")) {
				return VALA_GENIE_TOKEN_TYPE_CONSTRUCT;
			}
			break;
		} while (0); else if (_tmp104 == 'e')
		do {
			if (vala_genie_scanner_matches (self, begin, "exception")) {
				return VALA_GENIE_TOKEN_TYPE_ERRORDOMAIN;
			}
			break;
		} while (0); else if (_tmp104 == 'i')
		do {
			if (vala_genie_scanner_matches (self, begin, "interface")) {
				return VALA_GENIE_TOKEN_TYPE_INTERFACE;
			}
			break;
		} while (0); else if (_tmp104 == 'n')
		do {
			if (vala_genie_scanner_matches (self, begin, "namespace")) {
				return VALA_GENIE_TOKEN_TYPE_NAMESPACE;
			}
			break;
		} while (0); else if (_tmp104 == 'p')
		do {
			if (vala_genie_scanner_matches (self, begin, "protected")) {
				return VALA_GENIE_TOKEN_TYPE_PROTECTED;
			}
			break;
		} while (0); else if (_tmp104 == 'w')
		do {
			if (vala_genie_scanner_matches (self, begin, "writeonly")) {
				return VALA_GENIE_TOKEN_TYPE_WRITEONLY;
			}
			break;
		} while (0);
		break;
	} while (0); else if (_tmp107 == 10)
	do {
		gchar _tmp106;
		_tmp106 = begin[0];
		if (_tmp106 == 'i')
		do {
			if (vala_genie_scanner_matches (self, begin, "implements")) {
				return VALA_GENIE_TOKEN_TYPE_IMPLEMENTS;
			}
			break;
		} while (0);
		break;
	} while (0);
	return VALA_GENIE_TOKEN_TYPE_IDENTIFIER;
}


ValaGenieTokenType vala_genie_scanner_read_token (ValaGenieScanner* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end) {
	ValaGenieTokenType type;
	gchar* begin;
	gint token_length_in_chars;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), 0);
	/* emit dedents if outstanding before checking any other chars */
	if (self->priv->pending_dedents > 0) {
		self->priv->pending_dedents--;
		self->priv->indent_level--;
		(*token_begin).pos = self->priv->current;
		(*token_begin).line = self->priv->line;
		(*token_begin).column = self->priv->column;
		(*token_end).pos = self->priv->current;
		(*token_end).line = self->priv->line;
		(*token_end).column = self->priv->column;
		self->priv->last_token = VALA_GENIE_TOKEN_TYPE_DEDENT;
		return VALA_GENIE_TOKEN_TYPE_DEDENT;
	}
	if ((self->priv->_indent_spaces == 0) || (self->priv->last_token != VALA_GENIE_TOKEN_TYPE_EOL)) {
		/* scrub whitespace (excluding newlines) and comments */
		vala_genie_scanner_space (self);
	}
	while (self->priv->current < self->priv->end && self->priv->current[0] == '\\' && self->priv->current[1] == '\n') {
		self->priv->current = self->priv->current + (2);
		self->priv->line++;
		vala_genie_scanner_skip_space_tabs (self);
	}
	/* handle non-consecutive new line once parsing is underway - EOL */
	if (vala_genie_scanner_newline (self) && self->priv->parse_started && self->priv->last_token != VALA_GENIE_TOKEN_TYPE_EOL && self->priv->last_token != VALA_GENIE_TOKEN_TYPE_SEMICOLON) {
		(*token_begin).pos = self->priv->current;
		(*token_begin).line = self->priv->line;
		(*token_begin).column = self->priv->column;
		(*token_end).pos = self->priv->current;
		(*token_end).line = self->priv->line;
		(*token_end).column = self->priv->column;
		self->priv->last_token = VALA_GENIE_TOKEN_TYPE_EOL;
		return VALA_GENIE_TOKEN_TYPE_EOL;
	}
	while (vala_genie_scanner_skip_newlines (self)) {
		(*token_begin).pos = self->priv->current;
		(*token_begin).line = self->priv->line;
		(*token_begin).column = self->priv->column;
		self->priv->current_indent_level = vala_genie_scanner_count_tabs (self);
		/* if its an empty new line then ignore */
		if (self->priv->current_indent_level == -1) {
			continue;
		}
		if (self->priv->current_indent_level > self->priv->indent_level) {
			self->priv->indent_level = self->priv->current_indent_level;
			(*token_end).pos = self->priv->current;
			(*token_end).line = self->priv->line;
			(*token_end).column = self->priv->column;
			self->priv->last_token = VALA_GENIE_TOKEN_TYPE_INDENT;
			return VALA_GENIE_TOKEN_TYPE_INDENT;
		} else {
			if (self->priv->current_indent_level < self->priv->indent_level) {
				self->priv->indent_level--;
				self->priv->pending_dedents = (self->priv->indent_level - self->priv->current_indent_level);
				(*token_end).pos = self->priv->current;
				(*token_end).line = self->priv->line;
				(*token_end).column = self->priv->column;
				self->priv->last_token = VALA_GENIE_TOKEN_TYPE_DEDENT;
				return VALA_GENIE_TOKEN_TYPE_DEDENT;
			}
		}
	}
	type = 0;
	begin = self->priv->current;
	(*token_begin).pos = begin;
	(*token_begin).line = self->priv->line;
	(*token_begin).column = self->priv->column;
	token_length_in_chars = -1;
	self->priv->parse_started = TRUE;
	if (self->priv->current >= self->priv->end) {
		if (self->priv->indent_level > 0) {
			self->priv->indent_level--;
			self->priv->pending_dedents = self->priv->indent_level;
			type = VALA_GENIE_TOKEN_TYPE_DEDENT;
		} else {
			type = VALA_GENIE_TOKEN_TYPE_EOF;
		}
	} else {
		if (g_ascii_isalpha (self->priv->current[0]) || self->priv->current[0] == '_') {
			gint len;
			len = 0;
			while (self->priv->current < self->priv->end && vala_genie_scanner_is_ident_char (self, self->priv->current[0])) {
				self->priv->current++;
				len++;
			}
			type = vala_genie_scanner_get_identifier_or_keyword (self, begin, len);
		} else {
			if (self->priv->current[0] == '@') {
				gint len;
				len = 0;
				if (self->priv->current[1] == '@') {
					(*token_begin).pos = (*token_begin).pos + (2);
					/* @@ is not part of the identifier*/
					self->priv->current = self->priv->current + (2);
				} else {
					self->priv->current++;
					len = 1;
				}
				while (self->priv->current < self->priv->end && vala_genie_scanner_is_ident_char (self, self->priv->current[0])) {
					self->priv->current++;
					len++;
				}
				type = VALA_GENIE_TOKEN_TYPE_IDENTIFIER;
			} else {
				if (g_ascii_isdigit (self->priv->current[0])) {
					while (self->priv->current < self->priv->end && g_ascii_isdigit (self->priv->current[0])) {
						self->priv->current++;
					}
					type = VALA_GENIE_TOKEN_TYPE_INTEGER_LITERAL;
					if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
						self->priv->current++;
						if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
							self->priv->current++;
						}
					} else {
						if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'u') {
							self->priv->current++;
							if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
								self->priv->current++;
								if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'l') {
									self->priv->current++;
								}
							}
						} else {
							if (self->priv->current < self->priv->end - 1 && self->priv->current[0] == '.' && g_ascii_isdigit (self->priv->current[1])) {
								self->priv->current++;
								while (self->priv->current < self->priv->end && g_ascii_isdigit (self->priv->current[0])) {
									self->priv->current++;
								}
								if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'e') {
									self->priv->current++;
									if (self->priv->current < self->priv->end && (self->priv->current[0] == '+' || self->priv->current[0] == '-')) {
										self->priv->current++;
									}
									while (self->priv->current < self->priv->end && g_ascii_isdigit (self->priv->current[0])) {
										self->priv->current++;
									}
								}
								if (self->priv->current < self->priv->end && g_ascii_tolower (self->priv->current[0]) == 'f') {
									self->priv->current++;
								}
								type = VALA_GENIE_TOKEN_TYPE_REAL_LITERAL;
							} else {
								if (self->priv->current < self->priv->end && self->priv->current == begin + 1 && begin[0] == '0' && begin[1] == 'x' && g_ascii_isxdigit (begin[2])) {
									/* hexadecimal integer literal*/
									self->priv->current++;
									while (self->priv->current < self->priv->end && g_ascii_isxdigit (self->priv->current[0])) {
										self->priv->current++;
									}
								} else {
									if (self->priv->current < self->priv->end && vala_genie_scanner_is_ident_char (self, self->priv->current[0])) {
										while (self->priv->current < self->priv->end && vala_genie_scanner_is_ident_char (self, self->priv->current[0])) {
											self->priv->current++;
										}
										type = VALA_GENIE_TOKEN_TYPE_IDENTIFIER;
									}
								}
							}
						}
					}
				} else {
					gchar _tmp16;
					_tmp16 = self->priv->current[0];
					if (_tmp16 == '{')
					do {
						type = VALA_GENIE_TOKEN_TYPE_OPEN_BRACE;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '}')
					do {
						type = VALA_GENIE_TOKEN_TYPE_CLOSE_BRACE;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '(')
					do {
						type = VALA_GENIE_TOKEN_TYPE_OPEN_PARENS;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == ')')
					do {
						type = VALA_GENIE_TOKEN_TYPE_CLOSE_PARENS;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '[')
					do {
						type = VALA_GENIE_TOKEN_TYPE_OPEN_BRACKET;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == ']')
					do {
						type = VALA_GENIE_TOKEN_TYPE_CLOSE_BRACKET;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '.')
					do {
						type = VALA_GENIE_TOKEN_TYPE_DOT;
						self->priv->current++;
						if (self->priv->current < self->priv->end - 1) {
							if (self->priv->current[0] == '.' && self->priv->current[1] == '.') {
								type = VALA_GENIE_TOKEN_TYPE_ELLIPSIS;
								self->priv->current = self->priv->current + (2);
							}
						}
						break;
					} while (0); else if (_tmp16 == ':')
					do {
						type = VALA_GENIE_TOKEN_TYPE_COLON;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == ',')
					do {
						type = VALA_GENIE_TOKEN_TYPE_COMMA;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == ';')
					do {
						type = VALA_GENIE_TOKEN_TYPE_SEMICOLON;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '#')
					do {
						type = VALA_GENIE_TOKEN_TYPE_HASH;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '?')
					do {
						type = VALA_GENIE_TOKEN_TYPE_INTERR;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '|')
					do {
						type = VALA_GENIE_TOKEN_TYPE_BITWISE_OR;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp4;
							_tmp4 = self->priv->current[0];
							if (_tmp4 == '=')
							do {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_BITWISE_OR;
								self->priv->current++;
								break;
							} while (0); else if (_tmp4 == '|')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_OR;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp16 == '&')
					do {
						type = VALA_GENIE_TOKEN_TYPE_BITWISE_AND;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp5;
							_tmp5 = self->priv->current[0];
							if (_tmp5 == '=')
							do {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_BITWISE_AND;
								self->priv->current++;
								break;
							} while (0); else if (_tmp5 == '&')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_AND;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp16 == '^')
					do {
						type = VALA_GENIE_TOKEN_TYPE_CARRET;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_GENIE_TOKEN_TYPE_ASSIGN_BITWISE_XOR;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp16 == '~')
					do {
						type = VALA_GENIE_TOKEN_TYPE_TILDE;
						self->priv->current++;
						break;
					} while (0); else if (_tmp16 == '=')
					do {
						type = VALA_GENIE_TOKEN_TYPE_ASSIGN;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp6;
							_tmp6 = self->priv->current[0];
							if (_tmp6 == '=')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_EQ;
								self->priv->current++;
								break;
							} while (0); else if (_tmp6 == '>')
							do {
								type = VALA_GENIE_TOKEN_TYPE_LAMBDA;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp16 == '<')
					do {
						type = VALA_GENIE_TOKEN_TYPE_OP_LT;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp7;
							_tmp7 = self->priv->current[0];
							if (_tmp7 == '=')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_LE;
								self->priv->current++;
								break;
							} while (0); else if (_tmp7 == '<')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_SHIFT_LEFT;
								self->priv->current++;
								if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
									type = VALA_GENIE_TOKEN_TYPE_ASSIGN_SHIFT_LEFT;
									self->priv->current++;
								}
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp16 == '>')
					do {
						type = VALA_GENIE_TOKEN_TYPE_OP_GT;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_GENIE_TOKEN_TYPE_OP_GE;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp16 == '!')
					do {
						type = VALA_GENIE_TOKEN_TYPE_OP_NEG;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_GENIE_TOKEN_TYPE_OP_NE;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp16 == '+')
					do {
						type = VALA_GENIE_TOKEN_TYPE_PLUS;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp8;
							_tmp8 = self->priv->current[0];
							if (_tmp8 == '=')
							do {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_ADD;
								self->priv->current++;
								break;
							} while (0); else if (_tmp8 == '+')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_INC;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp16 == '-')
					do {
						type = VALA_GENIE_TOKEN_TYPE_MINUS;
						self->priv->current++;
						if (self->priv->current < self->priv->end) {
							gchar _tmp9;
							_tmp9 = self->priv->current[0];
							if (_tmp9 == '=')
							do {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_SUB;
								self->priv->current++;
								break;
							} while (0); else if (_tmp9 == '-')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_DEC;
								self->priv->current++;
								break;
							} while (0); else if (_tmp9 == '>')
							do {
								type = VALA_GENIE_TOKEN_TYPE_OP_PTR;
								self->priv->current++;
								break;
							} while (0);
						}
						break;
					} while (0); else if (_tmp16 == '*')
					do {
						type = VALA_GENIE_TOKEN_TYPE_STAR;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_GENIE_TOKEN_TYPE_ASSIGN_MUL;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp16 == '/')
					do {
						type = VALA_GENIE_TOKEN_TYPE_DIV;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_GENIE_TOKEN_TYPE_ASSIGN_DIV;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp16 == '%')
					do {
						type = VALA_GENIE_TOKEN_TYPE_PERCENT;
						self->priv->current++;
						if (self->priv->current < self->priv->end && self->priv->current[0] == '=') {
							type = VALA_GENIE_TOKEN_TYPE_ASSIGN_PERCENT;
							self->priv->current++;
						}
						break;
					} while (0); else if (_tmp16 == '\'' || _tmp16 == '"')
					do {
						if (begin[0] == '\'') {
							type = VALA_GENIE_TOKEN_TYPE_CHARACTER_LITERAL;
						} else {
							type = VALA_GENIE_TOKEN_TYPE_STRING_LITERAL;
						}
						token_length_in_chars = 2;
						self->priv->current++;
						while (self->priv->current < self->priv->end && self->priv->current[0] != begin[0]) {
							if (self->priv->current[0] == '\\') {
								self->priv->current++;
								token_length_in_chars++;
								if (self->priv->current < self->priv->end && self->priv->current[0] == 'x') {
									/* hexadecimal escape character*/
									self->priv->current++;
									token_length_in_chars++;
									while (self->priv->current < self->priv->end && g_ascii_isxdigit (self->priv->current[0])) {
										self->priv->current++;
										token_length_in_chars++;
									}
								} else {
									self->priv->current++;
									token_length_in_chars++;
								}
							} else {
								if (self->priv->current[0] == '\n') {
									break;
								} else {
									gunichar u;
									u = g_utf8_get_char_validated ((((const char*) (self->priv->current))), ((glong) ((self->priv->end - self->priv->current))));
									if (u != ((gunichar) ((-1)))) {
										self->priv->current = self->priv->current + (g_unichar_to_utf8 (u, NULL));
										token_length_in_chars++;
									} else {
										ValaSourceReference* _tmp10;
										_tmp10 = NULL;
										vala_report_error ((_tmp10 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars)), "invalid UTF-8 character");
										(_tmp10 == NULL ? NULL : (_tmp10 = (vala_source_reference_unref (_tmp10), NULL)));
									}
								}
							}
						}
						if (self->priv->current < self->priv->end && self->priv->current[0] != '\n') {
							self->priv->current++;
						} else {
							char* _tmp12;
							ValaSourceReference* _tmp11;
							_tmp12 = NULL;
							_tmp11 = NULL;
							vala_report_error ((_tmp11 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars)), (_tmp12 = g_strdup_printf ("syntax error, expected %c", ((gint) (begin[0])))));
							_tmp12 = (g_free (_tmp12), NULL);
							(_tmp11 == NULL ? NULL : (_tmp11 = (vala_source_reference_unref (_tmp11), NULL)));
						}
						break;
					} while (0); else
					do {
						gunichar u;
						u = g_utf8_get_char_validated ((((const char*) (self->priv->current))), ((glong) ((self->priv->end - self->priv->current))));
						if (u != ((gunichar) ((-1)))) {
							ValaSourceReference* _tmp13;
							self->priv->current = self->priv->current + (g_unichar_to_utf8 (u, NULL));
							_tmp13 = NULL;
							vala_report_error ((_tmp13 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column)), "syntax error, unexpected character");
							(_tmp13 == NULL ? NULL : (_tmp13 = (vala_source_reference_unref (_tmp13), NULL)));
						} else {
							ValaSourceReference* _tmp14;
							self->priv->current++;
							_tmp14 = NULL;
							vala_report_error ((_tmp14 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column)), "invalid UTF-8 character");
							(_tmp14 == NULL ? NULL : (_tmp14 = (vala_source_reference_unref (_tmp14), NULL)));
						}
						self->priv->column++;
						self->priv->last_token = VALA_GENIE_TOKEN_TYPE_STRING_LITERAL;
						return vala_genie_scanner_read_token (self, &(*token_begin), &(*token_end));
					} while (0);
				}
			}
		}
	}
	if (token_length_in_chars < 0) {
		self->priv->column = self->priv->column + (((gint) ((self->priv->current - begin))));
	} else {
		self->priv->column = self->priv->column + (token_length_in_chars);
	}
	(*token_end).pos = self->priv->current;
	(*token_end).line = self->priv->line;
	(*token_end).column = self->priv->column - 1;
	self->priv->last_token = type;
	return type;
}


static gint vala_genie_scanner_count_tabs (ValaGenieScanner* self) {
	gint tab_count;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), 0);
	tab_count = 0;
	if (self->priv->_indent_spaces == 0) {
		while (self->priv->current < self->priv->end && self->priv->current[0] == '\t') {
			self->priv->current++;
			self->priv->column++;
			tab_count++;
		}
	} else {
		gint space_count;
		space_count = 0;
		while (self->priv->current < self->priv->end && self->priv->current[0] == ' ') {
			self->priv->current++;
			self->priv->column++;
			space_count++;
		}
		tab_count = space_count / self->priv->_indent_spaces;
	}
	/* ignore comments and whitspace and other lines that contain no code */
	vala_genie_scanner_space (self);
	if ((self->priv->current < self->priv->end) && (self->priv->current[0] == '\n')) {
		return -1;
	}
	return tab_count;
}


static gboolean vala_genie_scanner_matches (ValaGenieScanner* self, gchar* begin, const char* keyword) {
	gchar* keyword_array;
	glong len;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), FALSE);
	g_return_val_if_fail (keyword != NULL, FALSE);
	keyword_array = keyword;
	len = g_utf8_strlen (keyword, -1);
	{
		gint i;
		i = 0;
		for (; i < len; i++) {
			if (begin[i] != keyword_array[i]) {
				return FALSE;
			}
		}
	}
	return TRUE;
}


static gboolean vala_genie_scanner_whitespace (ValaGenieScanner* self) {
	gboolean found;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), FALSE);
	found = FALSE;
	while (self->priv->current < self->priv->end && g_ascii_isspace (self->priv->current[0]) && self->priv->current[0] != '\n') {
		found = TRUE;
		self->priv->current++;
		self->priv->column++;
	}
	return found;
}


static inline gboolean vala_genie_scanner_newline (ValaGenieScanner* self) {
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), FALSE);
	if (self->priv->current[0] == '\n') {
		return TRUE;
	}
	return FALSE;
}


static gboolean vala_genie_scanner_skip_newlines (ValaGenieScanner* self) {
	gboolean new_lines;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), FALSE);
	new_lines = FALSE;
	while (vala_genie_scanner_newline (self)) {
		self->priv->current++;
		self->priv->line++;
		self->priv->column = 1;
		self->priv->current_indent_level = 0;
		new_lines = TRUE;
	}
	return new_lines;
}


static gboolean vala_genie_scanner_comment (ValaGenieScanner* self) {
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), FALSE);
	if (self->priv->current > self->priv->end - 2 || self->priv->current[0] != '/' || (self->priv->current[1] != '/' && self->priv->current[1] != '*')) {
		return FALSE;
	}
	if (self->priv->current[1] == '/') {
		gchar* begin;
		char* _tmp1;
		/* single-line comment*/
		self->priv->current = self->priv->current + (2);
		begin = self->priv->current;
		while (self->priv->current < self->priv->end && self->priv->current[0] != '\n') {
			self->priv->current++;
		}
		_tmp1 = NULL;
		vala_genie_scanner_push_comment (self, (_tmp1 = g_strndup ((((const char*) (begin))), ((gulong) (((glong) ((self->priv->current - begin))))))), self->priv->line == 1);
		_tmp1 = (g_free (_tmp1), NULL);
		if (self->priv->current[0] == '\n') {
			self->priv->current++;
			self->priv->line++;
			self->priv->column = 1;
			self->priv->current_indent_level = 0;
		}
	} else {
		gchar* begin;
		gint begin_line;
		char* _tmp4;
		/* delimited comment*/
		self->priv->current = self->priv->current + (2);
		begin = self->priv->current;
		begin_line = self->priv->line;
		while (self->priv->current < self->priv->end - 1 && (self->priv->current[0] != '*' || self->priv->current[1] != '/')) {
			if (self->priv->current[0] == '\n') {
				self->priv->line++;
				self->priv->column = 0;
			}
			self->priv->current++;
			self->priv->column++;
		}
		if (self->priv->current == self->priv->end - 1) {
			ValaSourceReference* _tmp2;
			_tmp2 = NULL;
			vala_report_error ((_tmp2 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column)), "syntax error, expected */");
			(_tmp2 == NULL ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL)));
			return TRUE;
		}
		_tmp4 = NULL;
		vala_genie_scanner_push_comment (self, (_tmp4 = g_strndup ((((const char*) (begin))), ((gulong) (((glong) ((self->priv->current - begin))))))), begin_line == 1);
		_tmp4 = (g_free (_tmp4), NULL);
		self->priv->current = self->priv->current + (2);
		self->priv->column = self->priv->column + (2);
	}
	return TRUE;
}


static gboolean vala_genie_scanner_skip_tabs (ValaGenieScanner* self) {
	gboolean found;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), FALSE);
	found = FALSE;
	while (self->priv->current < self->priv->end && self->priv->current[0] == '\t') {
		self->priv->current++;
		self->priv->column++;
		found = TRUE;
	}
	return found;
}


static void vala_genie_scanner_skip_space_tabs (ValaGenieScanner* self) {
	g_return_if_fail (VALA_GENIE_IS_SCANNER (self));
	while (vala_genie_scanner_whitespace (self) || vala_genie_scanner_skip_tabs (self) || vala_genie_scanner_comment (self)) {
	}
}


static void vala_genie_scanner_space (ValaGenieScanner* self) {
	g_return_if_fail (VALA_GENIE_IS_SCANNER (self));
	while (vala_genie_scanner_whitespace (self) || vala_genie_scanner_comment (self)) {
	}
}


static void vala_genie_scanner_push_comment (ValaGenieScanner* self, const char* comment_item, gboolean file_comment) {
	g_return_if_fail (VALA_GENIE_IS_SCANNER (self));
	g_return_if_fail (comment_item != NULL);
	if (self->priv->_comment == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->_comment = (_tmp1 = (_tmp0 = comment_item, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp1);
	} else {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->_comment = (_tmp2 = g_strdup_printf ("%s\n%s", self->priv->_comment, comment_item), (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp2);
	}
	if (file_comment) {
		char* _tmp3;
		vala_source_file_set_comment (self->priv->_source_file, self->priv->_comment);
		_tmp3 = NULL;
		self->priv->_comment = (_tmp3 = NULL, (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp3);
	}
}


/**
 * Clears and returns the content of the comment stack.
 *
 * @return saved comment
 */
char* vala_genie_scanner_pop_comment (ValaGenieScanner* self) {
	GString* result;
	char* _tmp1;
	const char* index;
	const char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), NULL);
	if (self->priv->_comment == NULL) {
		return NULL;
	}
	result = g_string_new (self->priv->_comment);
	_tmp1 = NULL;
	self->priv->_comment = (_tmp1 = NULL, (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp1);
	index = NULL;
	while ((index = g_utf8_strchr (result->str, ((glong) (-1)), ((gunichar) ('\t')))) != NULL) {
		g_string_erase (result, g_utf8_pointer_to_offset (result->str, index), ((glong) (1)));
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = result->str, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), _tmp3);
}


ValaSourceFile* vala_genie_scanner_get_source_file (ValaGenieScanner* self) {
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), NULL);
	return self->priv->_source_file;
}


static void vala_genie_scanner_set_source_file (ValaGenieScanner* self, ValaSourceFile* value) {
	ValaSourceFile* _tmp2;
	ValaSourceFile* _tmp1;
	g_return_if_fail (VALA_GENIE_IS_SCANNER (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_source_file = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_source_file_ref (_tmp1))), (self->priv->_source_file == NULL ? NULL : (self->priv->_source_file = (vala_source_file_unref (self->priv->_source_file), NULL))), _tmp2);
}


gint vala_genie_scanner_get_indent_spaces (ValaGenieScanner* self) {
	g_return_val_if_fail (VALA_GENIE_IS_SCANNER (self), 0);
	return self->priv->_indent_spaces;
}


void vala_genie_scanner_set_indent_spaces (ValaGenieScanner* self, gint value) {
	g_return_if_fail (VALA_GENIE_IS_SCANNER (self));
	self->priv->_indent_spaces = value;
}


static void vala_genie_value_scanner_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_genie_value_scanner_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_genie_scanner_unref (value->data[0].v_pointer);
	}
}


static void vala_genie_value_scanner_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_genie_scanner_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_genie_value_scanner_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_genie_value_scanner_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaGenieScanner* object;
		object = value->data[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_OBJECT_TYPE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", G_OBJECT_TYPE (object), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_genie_value_scanner_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaGenieScanner** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_genie_scanner_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_genie_param_spec_scanner (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaGenieParamSpecScanner* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_GENIE_TYPE_SCANNER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_genie_value_get_scanner (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_GENIE_TYPE_SCANNER), NULL);
	return value->data[0].v_pointer;
}


void vala_genie_value_set_scanner (GValue* value, gpointer v_object) {
	ValaGenieScanner* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_GENIE_TYPE_SCANNER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_GENIE_TYPE_SCANNER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_genie_scanner_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_genie_scanner_unref (old);
	}
}


static void vala_genie_scanner_class_init (ValaGenieScannerClass * klass) {
	vala_genie_scanner_parent_class = g_type_class_peek_parent (klass);
	VALA_GENIE_SCANNER_CLASS (klass)->finalize = vala_genie_scanner_finalize;
	g_type_class_add_private (klass, sizeof (ValaGenieScannerPrivate));
}


static void vala_genie_scanner_instance_init (ValaGenieScanner * self) {
	self->priv = VALA_GENIE_SCANNER_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_genie_scanner_finalize (ValaGenieScanner* obj) {
	ValaGenieScanner * self;
	self = VALA_GENIE_SCANNER (obj);
	(self->priv->_source_file == NULL ? NULL : (self->priv->_source_file = (vala_source_file_unref (self->priv->_source_file), NULL)));
	self->priv->_comment = (g_free (self->priv->_comment), NULL);
}


GType vala_genie_scanner_get_type (void) {
	static GType vala_genie_scanner_type_id = 0;
	if (vala_genie_scanner_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_genie_value_scanner_init, vala_genie_value_scanner_free_value, vala_genie_value_scanner_copy_value, vala_genie_value_scanner_peek_pointer, "p", vala_genie_value_scanner_collect_value, "p", vala_genie_value_scanner_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaGenieScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_genie_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGenieScanner), 0, (GInstanceInitFunc) vala_genie_scanner_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_genie_scanner_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaGenieScanner", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_genie_scanner_type_id;
}


gpointer vala_genie_scanner_ref (gpointer instance) {
	ValaGenieScanner* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_genie_scanner_unref (gpointer instance) {
	ValaGenieScanner* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_GENIE_SCANNER_GET_CLASS (self)->finalize (self);
		g_type_free_instance (((GTypeInstance *) (self)));
	}
}




