/* valaforstatement.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaforstatement.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valaexpression.h>
#include <vala/valablock.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaForStatementPrivate {
	GeeList* initializer;
	GeeList* iterator;
	ValaExpression* _condition;
	ValaBlock* _body;
};

#define VALA_FOR_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FOR_STATEMENT, ValaForStatementPrivate))
enum  {
	VALA_FOR_STATEMENT_DUMMY_PROPERTY
};
static void vala_for_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_for_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_for_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gpointer vala_for_statement_parent_class = NULL;
static ValaStatementIface* vala_for_statement_vala_statement_parent_iface = NULL;
static void vala_for_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new for statement.
 *
 * @param cond             loop condition
 * @param body             loop body
 * @param source_reference reference to source code
 * @return                 newly created for statement
 */
ValaForStatement* vala_for_statement_construct (GType object_type, ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaForStatement* self;
	g_return_val_if_fail (condition == NULL || VALA_IS_EXPRESSION (condition), NULL);
	g_return_val_if_fail (VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaForStatement*) (g_type_create_instance (object_type)));
	vala_for_statement_set_condition (self, condition);
	vala_for_statement_set_body (self, body);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaForStatement* vala_for_statement_new (ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	return vala_for_statement_construct (VALA_TYPE_FOR_STATEMENT, condition, body, source_reference);
}


/**
 * Appends the specified expression to the list of initializers.
 *
 * @param init an initializer expression
 */
void vala_for_statement_add_initializer (ValaForStatement* self, ValaExpression* init) {
	g_return_if_fail (VALA_IS_FOR_STATEMENT (self));
	g_return_if_fail (VALA_IS_EXPRESSION (init));
	vala_code_node_set_parent_node (VALA_CODE_NODE (init), VALA_CODE_NODE (self));
	gee_collection_add (GEE_COLLECTION (self->priv->initializer), init);
}


/**
 * Returns a copy of the list of initializers.
 *
 * @return initializer list
 */
GeeList* vala_for_statement_get_initializer (ValaForStatement* self) {
	g_return_val_if_fail (VALA_IS_FOR_STATEMENT (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->initializer));
}


/**
 * Appends the specified expression to the iterator.
 *
 * @param iter an iterator expression
 */
void vala_for_statement_add_iterator (ValaForStatement* self, ValaExpression* iter) {
	g_return_if_fail (VALA_IS_FOR_STATEMENT (self));
	g_return_if_fail (VALA_IS_EXPRESSION (iter));
	vala_code_node_set_parent_node (VALA_CODE_NODE (iter), VALA_CODE_NODE (self));
	gee_collection_add (GEE_COLLECTION (self->priv->iterator), iter);
}


/**
 * Returns a copy of the iterator.
 *
 * @return iterator
 */
GeeList* vala_for_statement_get_iterator (ValaForStatement* self) {
	g_return_val_if_fail (VALA_IS_FOR_STATEMENT (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->iterator));
}


static void vala_for_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaForStatement * self;
	self = VALA_FOR_STATEMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_for_statement (visitor, self);
}


static void vala_for_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaForStatement * self;
	self = VALA_FOR_STATEMENT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* init_expr_collection;
		int init_expr_it;
		init_expr_collection = self->priv->initializer;
		for (init_expr_it = 0; init_expr_it < gee_collection_get_size (GEE_COLLECTION (init_expr_collection)); init_expr_it = init_expr_it + 1) {
			ValaExpression* init_expr;
			init_expr = ((ValaExpression*) (gee_list_get (GEE_LIST (init_expr_collection), init_expr_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (init_expr), visitor);
				vala_code_visitor_visit_end_full_expression (visitor, init_expr);
				(init_expr == NULL ? NULL : (init_expr = (vala_code_node_unref (init_expr), NULL)));
			}
		}
	}
	if (vala_for_statement_get_condition (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_for_statement_get_condition (self)), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_for_statement_get_condition (self));
	}
	{
		GeeList* it_expr_collection;
		int it_expr_it;
		it_expr_collection = self->priv->iterator;
		for (it_expr_it = 0; it_expr_it < gee_collection_get_size (GEE_COLLECTION (it_expr_collection)); it_expr_it = it_expr_it + 1) {
			ValaExpression* it_expr;
			it_expr = ((ValaExpression*) (gee_list_get (GEE_LIST (it_expr_collection), it_expr_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (it_expr), visitor);
				vala_code_visitor_visit_end_full_expression (visitor, it_expr);
				(it_expr == NULL ? NULL : (it_expr = (vala_code_node_unref (it_expr), NULL)));
			}
		}
	}
	vala_code_node_accept (VALA_CODE_NODE (vala_for_statement_get_body (self)), visitor);
}


static void vala_for_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaForStatement * self;
	self = VALA_FOR_STATEMENT (base);
	g_return_if_fail (VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (VALA_IS_EXPRESSION (new_node));
	if (vala_for_statement_get_condition (self) == old_node) {
		vala_for_statement_set_condition (self, new_node);
		return;
	}
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size (GEE_COLLECTION (self->priv->initializer)); i++) {
			ValaExpression* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = ((ValaExpression*) (gee_list_get (((GeeList*) (self->priv->initializer)), i)))) == old_node, (_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL))), _tmp1)) {
				gee_list_set (((GeeList*) (self->priv->initializer)), i, new_node);
				return;
			}
		}
	}
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size (GEE_COLLECTION (self->priv->iterator)); i++) {
			ValaExpression* _tmp2;
			gboolean _tmp3;
			_tmp2 = NULL;
			if ((_tmp3 = (_tmp2 = ((ValaExpression*) (gee_list_get (((GeeList*) (self->priv->iterator)), i)))) == old_node, (_tmp2 == NULL ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL))), _tmp3)) {
				gee_list_set (((GeeList*) (self->priv->iterator)), i, new_node);
				return;
			}
		}
	}
}


ValaExpression* vala_for_statement_get_condition (ValaForStatement* self) {
	g_return_val_if_fail (VALA_IS_FOR_STATEMENT (self), NULL);
	return self->priv->_condition;
}


void vala_for_statement_set_condition (ValaForStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_FOR_STATEMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_condition = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_condition == NULL ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL))), _tmp2);
	if (self->priv->_condition != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_condition), VALA_CODE_NODE (self));
	}
}


ValaBlock* vala_for_statement_get_body (ValaForStatement* self) {
	g_return_val_if_fail (VALA_IS_FOR_STATEMENT (self), NULL);
	return self->priv->_body;
}


void vala_for_statement_set_body (ValaForStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_FOR_STATEMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_body), VALA_CODE_NODE (self));
}


static void vala_for_statement_class_init (ValaForStatementClass * klass) {
	vala_for_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_for_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaForStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_for_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_for_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_for_statement_real_replace_expression;
}


static void vala_for_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_for_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_for_statement_instance_init (ValaForStatement * self) {
	self->priv = VALA_FOR_STATEMENT_GET_PRIVATE (self);
	self->priv->initializer = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
	self->priv->iterator = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
}


static void vala_for_statement_finalize (ValaCodeNode* obj) {
	ValaForStatement * self;
	self = VALA_FOR_STATEMENT (obj);
	(self->priv->initializer == NULL ? NULL : (self->priv->initializer = (gee_collection_object_unref (self->priv->initializer), NULL)));
	(self->priv->iterator == NULL ? NULL : (self->priv->iterator = (gee_collection_object_unref (self->priv->iterator), NULL)));
	(self->priv->_condition == NULL ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL)));
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)));
	VALA_CODE_NODE_CLASS (vala_for_statement_parent_class)->finalize (obj);
}


GType vala_for_statement_get_type (void) {
	static GType vala_for_statement_type_id = 0;
	if (vala_for_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaForStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_for_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaForStatement), 0, (GInstanceInitFunc) vala_for_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_for_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_for_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaForStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_for_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_for_statement_type_id;
}




