/* valasymbolresolver.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valasymbolresolver.h>
#include <gee/list.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodenode.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valatypesymbol.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valaunresolvedtype.h>
#include <vala/valavoidtype.h>
#include <vala/valainvalidtype.h>
#include <vala/valatypeparameter.h>
#include <vala/valatypeparametertype.h>
#include <vala/valadelegatetype.h>
#include <vala/valaerrortype.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valavaluetype.h>
#include <vala/valareferencetype.h>




struct _ValaSymbolResolverPrivate {
	ValaSymbol* root_symbol;
	ValaScope* current_scope;
	GeeList* current_using_directives;
};

#define VALA_SYMBOL_RESOLVER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SYMBOL_RESOLVER, ValaSymbolResolverPrivate))
enum  {
	VALA_SYMBOL_RESOLVER_DUMMY_PROPERTY
};
static void vala_symbol_resolver_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_symbol_resolver_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_symbol_resolver_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_symbol_resolver_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_symbol_resolver_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_symbol_resolver_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed);
static void vala_symbol_resolver_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb);
static void vala_symbol_resolver_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_symbol_resolver_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_symbol_resolver_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_symbol_resolver_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_symbol_resolver_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_symbol_resolver_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_symbol_resolver_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_symbol_resolver_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_symbol_resolver_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_symbol_resolver_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_symbol_resolver_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_symbol_resolver_real_visit_namespace_reference (ValaCodeVisitor* base, ValaNamespaceReference* ns);
static ValaSymbol* vala_symbol_resolver_resolve_symbol (ValaSymbolResolver* self, ValaUnresolvedSymbol* unresolved_symbol);
static ValaDataType* vala_symbol_resolver_resolve_type (ValaSymbolResolver* self, ValaUnresolvedType* unresolved_type);
static void vala_symbol_resolver_real_visit_data_type (ValaCodeVisitor* base, ValaDataType* data_type);
static void vala_symbol_resolver_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local);
static void vala_symbol_resolver_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_symbol_resolver_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_symbol_resolver_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section);
static void vala_symbol_resolver_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_symbol_resolver_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt);
static void vala_symbol_resolver_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_symbol_resolver_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_symbol_resolver_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_symbol_resolver_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_symbol_resolver_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_symbol_resolver_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_symbol_resolver_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* e);
static void vala_symbol_resolver_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static void vala_symbol_resolver_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_symbol_resolver_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_symbol_resolver_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static void vala_symbol_resolver_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static void vala_symbol_resolver_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_symbol_resolver_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_symbol_resolver_parent_class = NULL;
static void vala_symbol_resolver_dispose (GObject * obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Resolve symbol names in the specified code context.
 *
 * @param context a code context
 */
void vala_symbol_resolver_resolve (ValaSymbolResolver* self, ValaCodeContext* context) {
	ValaSymbol* _tmp1;
	ValaSymbol* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	g_return_if_fail (VALA_IS_SYMBOL_RESOLVER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->root_symbol = (_tmp1 = (_tmp0 = VALA_SYMBOL (vala_code_context_get_root (context)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL))), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_symbol_get_scope (self->priv->root_symbol), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp3);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file) {
	ValaSymbolResolver * self;
	GeeList* _tmp0;
	ValaScope* _tmp2;
	ValaScope* _tmp1;
	GeeList* _tmp3;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	_tmp0 = NULL;
	self->priv->current_using_directives = (_tmp0 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), _tmp0);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_scope = (_tmp2 = (_tmp1 = vala_symbol_get_scope (self->priv->root_symbol), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp2);
	vala_source_file_accept_children (file, VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	self->priv->current_using_directives = (_tmp3 = NULL, (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), _tmp3);
}


static void vala_symbol_resolver_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp10;
	ValaScope* _tmp9;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (cl)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	{
		GeeList* type_collection;
		int type_it;
		type_collection = vala_class_get_base_types (cl);
		for (type_it = 0; type_it < gee_collection_get_size (GEE_COLLECTION (type_collection)); type_it = type_it + 1) {
			ValaDataType* type;
			type = ((ValaDataType*) (gee_list_get (GEE_LIST (type_collection), type_it)));
			{
				if (VALA_IS_CLASS (vala_data_type_get_data_type (type))) {
					if (vala_class_get_base_class (cl) != NULL) {
						char* _tmp5;
						char* _tmp4;
						char* _tmp3;
						char* _tmp2;
						vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
						_tmp5 = NULL;
						_tmp4 = NULL;
						_tmp3 = NULL;
						_tmp2 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (type)), (_tmp5 = g_strdup_printf ("%s: Classes cannot have multiple base classes (`%s' and `%s')", (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (_tmp3 = vala_symbol_get_full_name (VALA_SYMBOL (vala_class_get_base_class (cl)))), (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (vala_data_type_get_data_type (type)))))));
						_tmp5 = (g_free (_tmp5), NULL);
						_tmp4 = (g_free (_tmp4), NULL);
						_tmp3 = (g_free (_tmp3), NULL);
						_tmp2 = (g_free (_tmp2), NULL);
						(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
						(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
						return;
					}
					vala_class_set_base_class (cl, VALA_CLASS (vala_data_type_get_data_type (type)));
					if (vala_typesymbol_is_subtype_of (VALA_TYPESYMBOL (vala_class_get_base_class (cl)), VALA_TYPESYMBOL (cl))) {
						char* _tmp8;
						char* _tmp7;
						char* _tmp6;
						vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
						_tmp8 = NULL;
						_tmp7 = NULL;
						_tmp6 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (type)), (_tmp8 = g_strdup_printf ("Base class cycle (`%s' and `%s')", (_tmp6 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (_tmp7 = vala_symbol_get_full_name (VALA_SYMBOL (vala_class_get_base_class (cl)))))));
						_tmp8 = (g_free (_tmp8), NULL);
						_tmp7 = (g_free (_tmp7), NULL);
						_tmp6 = (g_free (_tmp6), NULL);
						(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
						(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
						return;
					}
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
	}
	_tmp10 = NULL;
	_tmp9 = NULL;
	self->priv->current_scope = (_tmp10 = (_tmp9 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp10);
}


static void vala_symbol_resolver_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (st)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp3);
}


static void vala_symbol_resolver_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp6;
	ValaScope* _tmp5;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (iface)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	{
		GeeList* type_collection;
		int type_it;
		type_collection = vala_interface_get_prerequisites (iface);
		for (type_it = 0; type_it < gee_collection_get_size (GEE_COLLECTION (type_collection)); type_it = type_it + 1) {
			ValaDataType* type;
			type = ((ValaDataType*) (gee_list_get (GEE_LIST (type_collection), type_it)));
			{
				if (vala_data_type_get_data_type (type) != NULL && vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (type), VALA_TYPESYMBOL (iface))) {
					char* _tmp4;
					char* _tmp3;
					char* _tmp2;
					vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE);
					_tmp4 = NULL;
					_tmp3 = NULL;
					_tmp2 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (type)), (_tmp4 = g_strdup_printf ("Prerequisite cycle (`%s' and `%s')", (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (iface))), (_tmp3 = vala_symbol_get_full_name (VALA_SYMBOL (vala_data_type_get_data_type (type)))))));
					_tmp4 = (g_free (_tmp4), NULL);
					_tmp3 = (g_free (_tmp3), NULL);
					_tmp2 = (g_free (_tmp2), NULL);
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
					return;
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	self->priv->current_scope = (_tmp6 = (_tmp5 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp5 == NULL ? NULL : g_object_ref (_tmp5))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp6);
}


static void vala_symbol_resolver_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_ENUM (en));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (en)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (en), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp3);
}


static void vala_symbol_resolver_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (ed));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (ed)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (ed), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp3);
}


static void vala_symbol_resolver_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_DELEGATE (cb));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (cb)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (cb), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp3);
}


static void vala_symbol_resolver_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (c)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (f)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (f), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp3);
}


static void vala_symbol_resolver_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaSymbolResolver * self;
	ValaScope* _tmp1;
	ValaScope* _tmp0;
	ValaScope* _tmp3;
	ValaScope* _tmp2;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_scope = (_tmp1 = (_tmp0 = vala_symbol_get_scope (VALA_SYMBOL (m)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_scope = (_tmp3 = (_tmp2 = vala_scope_get_parent_scope (self->priv->current_scope), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL))), _tmp3);
}


static void vala_symbol_resolver_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (p));
	vala_code_node_accept_children (VALA_CODE_NODE (p), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	vala_code_node_accept_children (VALA_CODE_NODE (sig), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	vala_code_node_accept_children (VALA_CODE_NODE (b), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_namespace_reference (ValaCodeVisitor* base, ValaNamespaceReference* ns) {
	ValaSymbolResolver * self;
	ValaSymbol* _tmp0;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_NAMESPACE_REFERENCE (ns));
	_tmp0 = NULL;
	vala_namespace_reference_set_namespace_symbol (ns, (_tmp0 = vala_scope_lookup (self->priv->current_scope, vala_namespace_reference_get_name (ns))));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	if (vala_namespace_reference_get_namespace_symbol (ns) == NULL) {
		char* _tmp1;
		vala_code_node_set_error (VALA_CODE_NODE (ns), TRUE);
		_tmp1 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ns)), (_tmp1 = g_strdup_printf ("The namespace name `%s' could not be found", vala_namespace_reference_get_name (ns))));
		_tmp1 = (g_free (_tmp1), NULL);
		return;
	}
}


static ValaSymbol* vala_symbol_resolver_resolve_symbol (ValaSymbolResolver* self, ValaUnresolvedSymbol* unresolved_symbol) {
	g_return_val_if_fail (VALA_IS_SYMBOL_RESOLVER (self), NULL);
	g_return_val_if_fail (VALA_IS_UNRESOLVED_SYMBOL (unresolved_symbol), NULL);
	if (vala_unresolved_symbol_get_qualified (unresolved_symbol)) {
		/* qualified access to global symbol*/
		return vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), vala_unresolved_symbol_get_name (unresolved_symbol));
	} else {
		if (vala_unresolved_symbol_get_inner (unresolved_symbol) == NULL) {
			ValaSymbol* sym;
			ValaScope* _tmp1;
			ValaScope* scope;
			ValaSymbol* _tmp11;
			sym = NULL;
			_tmp1 = NULL;
			scope = (_tmp1 = self->priv->current_scope, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
			while (sym == NULL && scope != NULL) {
				ValaSymbol* _tmp2;
				ValaScope* _tmp4;
				ValaScope* _tmp3;
				_tmp2 = NULL;
				sym = (_tmp2 = vala_scope_lookup (scope, vala_unresolved_symbol_get_name (unresolved_symbol)), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp2);
				_tmp4 = NULL;
				_tmp3 = NULL;
				scope = (_tmp4 = (_tmp3 = vala_scope_get_parent_scope (scope), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (scope == NULL ? NULL : (scope = (g_object_unref (scope), NULL))), _tmp4);
			}
			if (sym == NULL) {
				{
					GeeList* ns_collection;
					int ns_it;
					ns_collection = self->priv->current_using_directives;
					for (ns_it = 0; ns_it < gee_collection_get_size (GEE_COLLECTION (ns_collection)); ns_it = ns_it + 1) {
						ValaNamespaceReference* ns;
						ns = ((ValaNamespaceReference*) (gee_list_get (GEE_LIST (ns_collection), ns_it)));
						{
							ValaSymbol* local_sym;
							if (vala_code_node_get_error (VALA_CODE_NODE (ns))) {
								(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
								continue;
							}
							local_sym = vala_scope_lookup (vala_symbol_get_scope (vala_namespace_reference_get_namespace_symbol (ns)), vala_unresolved_symbol_get_name (unresolved_symbol));
							if (local_sym != NULL) {
								ValaSymbol* _tmp10;
								ValaSymbol* _tmp9;
								if (sym != NULL) {
									char* _tmp7;
									char* _tmp6;
									char* _tmp5;
									ValaSymbol* _tmp8;
									vala_code_node_set_error (VALA_CODE_NODE (unresolved_symbol), TRUE);
									_tmp7 = NULL;
									_tmp6 = NULL;
									_tmp5 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (unresolved_symbol)), (_tmp7 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_unresolved_symbol_get_name (unresolved_symbol), (_tmp5 = vala_symbol_get_full_name (sym)), (_tmp6 = vala_symbol_get_full_name (local_sym)))));
									_tmp7 = (g_free (_tmp7), NULL);
									_tmp6 = (g_free (_tmp6), NULL);
									_tmp5 = (g_free (_tmp5), NULL);
									_tmp8 = NULL;
									return (_tmp8 = NULL, (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), (local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), (scope == NULL ? NULL : (scope = (g_object_unref (scope), NULL))), _tmp8);
								}
								_tmp10 = NULL;
								_tmp9 = NULL;
								sym = (_tmp10 = (_tmp9 = local_sym, (_tmp9 == NULL ? NULL : g_object_ref (_tmp9))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp10);
							}
							(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
							(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
						}
					}
				}
			}
			_tmp11 = NULL;
			return (_tmp11 = sym, (scope == NULL ? NULL : (scope = (g_object_unref (scope), NULL))), _tmp11);
		} else {
			ValaSymbol* parent_symbol;
			ValaSymbol* _tmp14;
			parent_symbol = vala_symbol_resolver_resolve_symbol (self, vala_unresolved_symbol_get_inner (unresolved_symbol));
			if (parent_symbol == NULL) {
				char* _tmp12;
				ValaSymbol* _tmp13;
				vala_code_node_set_error (VALA_CODE_NODE (unresolved_symbol), TRUE);
				_tmp12 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_unresolved_symbol_get_inner (unresolved_symbol))), (_tmp12 = g_strdup_printf ("The symbol `%s' could not be found", vala_unresolved_symbol_get_name (vala_unresolved_symbol_get_inner (unresolved_symbol)))));
				_tmp12 = (g_free (_tmp12), NULL);
				_tmp13 = NULL;
				return (_tmp13 = NULL, (parent_symbol == NULL ? NULL : (parent_symbol = (g_object_unref (parent_symbol), NULL))), _tmp13);
			}
			_tmp14 = NULL;
			return (_tmp14 = vala_scope_lookup (vala_symbol_get_scope (parent_symbol), vala_unresolved_symbol_get_name (unresolved_symbol)), (parent_symbol == NULL ? NULL : (parent_symbol = (g_object_unref (parent_symbol), NULL))), _tmp14);
		}
	}
}


static ValaDataType* vala_symbol_resolver_resolve_type (ValaSymbolResolver* self, ValaUnresolvedType* unresolved_type) {
	ValaDataType* type;
	ValaSymbol* sym;
	ValaDataType* _tmp19;
	g_return_val_if_fail (VALA_IS_SYMBOL_RESOLVER (self), NULL);
	g_return_val_if_fail (VALA_IS_UNRESOLVED_TYPE (unresolved_type), NULL);
	type = NULL;
	/* still required for vapigen*/
	if (_vala_strcmp0 (vala_unresolved_symbol_get_name (vala_unresolved_type_get_unresolved_symbol (unresolved_type)), "void") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = VALA_DATA_TYPE (vala_void_type_new ()), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp0);
	}
	sym = vala_symbol_resolver_resolve_symbol (self, vala_unresolved_type_get_unresolved_symbol (unresolved_type));
	if (sym == NULL) {
		ValaDataType* _tmp3;
		/* don't report same error twice*/
		if (!vala_code_node_get_error (VALA_CODE_NODE (vala_unresolved_type_get_unresolved_symbol (unresolved_type)))) {
			char* _tmp2;
			char* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (unresolved_type)), (_tmp2 = g_strdup_printf ("The type name `%s' could not be found", (_tmp1 = vala_code_node_to_string (VALA_CODE_NODE (vala_unresolved_type_get_unresolved_symbol (unresolved_type)))))));
			_tmp2 = (g_free (_tmp2), NULL);
			_tmp1 = (g_free (_tmp1), NULL);
		}
		_tmp3 = NULL;
		return (_tmp3 = VALA_DATA_TYPE (vala_invalid_type_new ()), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp3);
	}
	if (VALA_IS_TYPEPARAMETER (sym)) {
		ValaDataType* _tmp4;
		_tmp4 = NULL;
		type = (_tmp4 = VALA_DATA_TYPE (vala_typeparameter_type_new (VALA_TYPEPARAMETER (sym))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp4);
	} else {
		if (VALA_IS_TYPESYMBOL (sym)) {
			if (VALA_IS_DELEGATE (sym)) {
				ValaDataType* _tmp5;
				_tmp5 = NULL;
				type = (_tmp5 = VALA_DATA_TYPE (vala_delegate_type_new (VALA_DELEGATE (sym))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp5);
			} else {
				if (VALA_IS_CLASS (sym)) {
					ValaClass* _tmp6;
					ValaClass* cl;
					_tmp6 = NULL;
					cl = (_tmp6 = VALA_CLASS (sym), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
					if (vala_class_get_is_error_base (cl)) {
						ValaDataType* _tmp7;
						_tmp7 = NULL;
						type = (_tmp7 = VALA_DATA_TYPE (vala_error_type_new (NULL, vala_code_node_get_source_reference (VALA_CODE_NODE (unresolved_type)))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp7);
					} else {
						ValaDataType* _tmp8;
						_tmp8 = NULL;
						type = (_tmp8 = VALA_DATA_TYPE (vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (cl))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp8);
					}
					(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
				} else {
					if (VALA_IS_INTERFACE (sym)) {
						ValaDataType* _tmp9;
						_tmp9 = NULL;
						type = (_tmp9 = VALA_DATA_TYPE (vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (VALA_INTERFACE (sym)))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp9);
					} else {
						if (VALA_IS_STRUCT (sym)) {
							ValaDataType* _tmp10;
							_tmp10 = NULL;
							type = (_tmp10 = VALA_DATA_TYPE (vala_value_type_new (VALA_TYPESYMBOL (VALA_STRUCT (sym)))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp10);
						} else {
							if (VALA_IS_ENUM (sym)) {
								ValaDataType* _tmp11;
								_tmp11 = NULL;
								type = (_tmp11 = VALA_DATA_TYPE (vala_value_type_new (VALA_TYPESYMBOL (VALA_ENUM (sym)))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp11);
							} else {
								if (VALA_IS_ERROR_DOMAIN (sym)) {
									ValaDataType* _tmp12;
									_tmp12 = NULL;
									type = (_tmp12 = VALA_DATA_TYPE (vala_error_type_new (VALA_ERROR_DOMAIN (sym), vala_code_node_get_source_reference (VALA_CODE_NODE (unresolved_type)))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp12);
								} else {
									char* _tmp14;
									char* _tmp13;
									ValaDataType* _tmp15;
									_tmp14 = NULL;
									_tmp13 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (unresolved_type)), (_tmp14 = g_strdup_printf ("internal error: `%s' is not a supported type", (_tmp13 = vala_symbol_get_full_name (sym)))));
									_tmp14 = (g_free (_tmp14), NULL);
									_tmp13 = (g_free (_tmp13), NULL);
									_tmp15 = NULL;
									return (_tmp15 = VALA_DATA_TYPE (vala_invalid_type_new ()), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp15);
								}
							}
						}
					}
				}
			}
		} else {
			char* _tmp17;
			char* _tmp16;
			ValaDataType* _tmp18;
			_tmp17 = NULL;
			_tmp16 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (unresolved_type)), (_tmp17 = g_strdup_printf ("`%s' is not a type", (_tmp16 = vala_symbol_get_full_name (sym)))));
			_tmp17 = (g_free (_tmp17), NULL);
			_tmp16 = (g_free (_tmp16), NULL);
			_tmp18 = NULL;
			return (_tmp18 = VALA_DATA_TYPE (vala_invalid_type_new ()), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp18);
		}
	}
	vala_code_node_set_source_reference (VALA_CODE_NODE (type), vala_code_node_get_source_reference (VALA_CODE_NODE (unresolved_type)));
	vala_data_type_set_value_owned (type, vala_data_type_get_value_owned (VALA_DATA_TYPE (unresolved_type)));
	vala_data_type_set_nullable (type, vala_data_type_get_nullable (VALA_DATA_TYPE (unresolved_type)));
	vala_data_type_set_is_dynamic (type, vala_data_type_get_is_dynamic (VALA_DATA_TYPE (unresolved_type)));
	{
		GeeList* type_arg_collection;
		int type_arg_it;
		type_arg_collection = vala_data_type_get_type_arguments (VALA_DATA_TYPE (unresolved_type));
		for (type_arg_it = 0; type_arg_it < gee_collection_get_size (GEE_COLLECTION (type_arg_collection)); type_arg_it = type_arg_it + 1) {
			ValaDataType* type_arg;
			type_arg = ((ValaDataType*) (gee_list_get (GEE_LIST (type_arg_collection), type_arg_it)));
			{
				vala_data_type_add_type_argument (type, type_arg);
				(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
			}
		}
		(type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL)));
	}
	_tmp19 = NULL;
	return (_tmp19 = type, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp19);
}


static void vala_symbol_resolver_real_visit_data_type (ValaCodeVisitor* base, ValaDataType* data_type) {
	ValaSymbolResolver * self;
	ValaUnresolvedType* _tmp0;
	ValaUnresolvedType* unresolved_type;
	ValaDataType* _tmp1;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (data_type));
	vala_code_node_accept_children (VALA_CODE_NODE (data_type), VALA_CODE_VISITOR (self));
	if (!(VALA_IS_UNRESOLVED_TYPE (data_type))) {
		return;
	}
	_tmp0 = NULL;
	unresolved_type = (_tmp0 = VALA_UNRESOLVED_TYPE (data_type), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	_tmp1 = NULL;
	vala_code_node_replace_type (vala_code_node_get_parent_node (VALA_CODE_NODE (unresolved_type)), VALA_DATA_TYPE (unresolved_type), (_tmp1 = vala_symbol_resolver_resolve_type (self, unresolved_type)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	(unresolved_type == NULL ? NULL : (unresolved_type = (g_object_unref (unresolved_type), NULL)));
}


static void vala_symbol_resolver_real_visit_local_variable (ValaCodeVisitor* base, ValaLocalVariable* local) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_LOCAL_VARIABLE (local));
	vala_code_node_accept_children (VALA_CODE_NODE (local), VALA_CODE_VISITOR (self));
	if (VALA_IS_REFERENCE_TYPE (vala_local_variable_get_variable_type (local))) {
		vala_data_type_set_nullable (vala_local_variable_get_variable_type (local), TRUE);
	}
}


static void vala_symbol_resolver_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	vala_code_node_accept_children (VALA_CODE_NODE (list), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_IF_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_SWITCH_SECTION (section));
	vala_code_node_accept_children (VALA_CODE_NODE (section), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_WHILE_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_do_statement (ValaCodeVisitor* base, ValaDoStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_DO_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_FOR_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_THROW_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_TRY_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (clause));
	vala_code_node_accept_children (VALA_CODE_NODE (clause), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* e) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (e));
	vala_code_node_accept_children (VALA_CODE_NODE (e), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (l));
	vala_code_node_accept_children (VALA_CODE_NODE (l), VALA_CODE_VISITOR (self));
}


static void vala_symbol_resolver_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	vala_code_node_accept_children (VALA_CODE_NODE (a), VALA_CODE_VISITOR (self));
}


/**
 * Code visitor resolving symbol names.
 */
ValaSymbolResolver* vala_symbol_resolver_new (void) {
	ValaSymbolResolver * self;
	self = g_object_newv (VALA_TYPE_SYMBOL_RESOLVER, 0, NULL);
	return self;
}


static void vala_symbol_resolver_class_init (ValaSymbolResolverClass * klass) {
	vala_symbol_resolver_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSymbolResolverPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_symbol_resolver_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_symbol_resolver_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_symbol_resolver_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_symbol_resolver_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_symbol_resolver_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_symbol_resolver_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_symbol_resolver_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_symbol_resolver_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_symbol_resolver_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_symbol_resolver_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_symbol_resolver_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_symbol_resolver_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_symbol_resolver_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_symbol_resolver_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_symbol_resolver_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_symbol_resolver_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_symbol_resolver_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_symbol_resolver_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_symbol_resolver_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace_reference = vala_symbol_resolver_real_visit_namespace_reference;
	VALA_CODE_VISITOR_CLASS (klass)->visit_data_type = vala_symbol_resolver_real_visit_data_type;
	VALA_CODE_VISITOR_CLASS (klass)->visit_local_variable = vala_symbol_resolver_real_visit_local_variable;
	VALA_CODE_VISITOR_CLASS (klass)->visit_initializer_list = vala_symbol_resolver_real_visit_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_symbol_resolver_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_symbol_resolver_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_symbol_resolver_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_symbol_resolver_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_symbol_resolver_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_symbol_resolver_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_symbol_resolver_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_symbol_resolver_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_symbol_resolver_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_symbol_resolver_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_array_creation_expression = vala_symbol_resolver_real_visit_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_symbol_resolver_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_invocation_expression = vala_symbol_resolver_real_visit_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_object_creation_expression = vala_symbol_resolver_real_visit_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_symbol_resolver_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_symbol_resolver_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_symbol_resolver_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_symbol_resolver_real_visit_assignment;
}


static void vala_symbol_resolver_instance_init (ValaSymbolResolver * self) {
	self->priv = VALA_SYMBOL_RESOLVER_GET_PRIVATE (self);
}


static void vala_symbol_resolver_dispose (GObject * obj) {
	ValaSymbolResolver * self;
	self = VALA_SYMBOL_RESOLVER (obj);
	(self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL)));
	(self->priv->current_scope == NULL ? NULL : (self->priv->current_scope = (g_object_unref (self->priv->current_scope), NULL)));
	(self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL)));
	G_OBJECT_CLASS (vala_symbol_resolver_parent_class)->dispose (obj);
}


GType vala_symbol_resolver_get_type (void) {
	static GType vala_symbol_resolver_type_id = 0;
	if (G_UNLIKELY (vala_symbol_resolver_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSymbolResolverClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_symbol_resolver_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSymbolResolver), 0, (GInstanceInitFunc) vala_symbol_resolver_instance_init };
		vala_symbol_resolver_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSymbolResolver", &g_define_type_info, 0);
	}
	return vala_symbol_resolver_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




