/* valaswitchstatement.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaswitchstatement.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valaexpression.h>
#include <vala/valaswitchsection.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaSwitchStatementPrivate {
	ValaExpression* _expression;
	GeeList* sections;
};
#define VALA_SWITCH_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SWITCH_STATEMENT, ValaSwitchStatementPrivate))
enum  {
	VALA_SWITCH_STATEMENT_DUMMY_PROPERTY,
	VALA_SWITCH_STATEMENT_EXPRESSION
};
static void vala_switch_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_switch_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gpointer vala_switch_statement_parent_class = NULL;
static ValaStatementIface* vala_switch_statement_vala_statement_parent_iface = NULL;
static void vala_switch_statement_dispose (GObject * obj);



/**
 * Creates a new switch statement.
 *
 * @param expression       switch expression
 * @param source_reference reference to source code
 * @return                 newly created switch statement
 */
ValaSwitchStatement* vala_switch_statement_new (ValaExpression* expression, ValaSourceReference* source_reference) {
	GParameter * __params;
	GParameter * __params_it;
	ValaSwitchStatement * self;
	g_return_val_if_fail (expression == NULL || VALA_IS_EXPRESSION (expression), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "expression";
	g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION);
	g_value_set_object (&__params_it->value, expression);
	__params_it++;
	self = g_object_newv (VALA_TYPE_SWITCH_STATEMENT, __params_it - __params, __params);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


/**
 * Appends the specified section to the list of switch sections.
 *
 * @param section a switch section
 */
void vala_switch_statement_add_section (ValaSwitchStatement* self, ValaSwitchSection* section) {
	g_return_if_fail (VALA_IS_SWITCH_STATEMENT (self));
	g_return_if_fail (section == NULL || VALA_IS_SWITCH_SECTION (section));
	vala_code_node_set_parent_node (VALA_CODE_NODE (section), VALA_CODE_NODE (self));
	gee_collection_add (GEE_COLLECTION (self->priv->sections), section);
}


/**
 * Returns a copy of the list of switch sections.
 *
 * @return section list
 */
GeeCollection* vala_switch_statement_get_sections (ValaSwitchStatement* self) {
	g_return_val_if_fail (VALA_IS_SWITCH_STATEMENT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_SWITCH_SECTION, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->sections)));
}


static void vala_switch_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSwitchStatement * self;
	self = VALA_SWITCH_STATEMENT (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_switch_statement_get_expression (self)), visitor);
	vala_code_visitor_visit_end_full_expression (visitor, vala_switch_statement_get_expression (self));
	{
		GeeList* section_collection;
		GeeIterator* section_it;
		section_collection = self->priv->sections;
		section_it = gee_iterable_iterator (GEE_ITERABLE (section_collection));
		while (gee_iterator_next (section_it)) {
			ValaSwitchSection* section;
			section = ((ValaSwitchSection*) gee_iterator_get (section_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (section), visitor);
				(section == NULL ? NULL : (section = (g_object_unref (section), NULL)));
			}
		}
		(section_it == NULL ? NULL : (section_it = (g_object_unref (section_it), NULL)));
	}
	vala_code_visitor_visit_switch_statement (visitor, self);
}


static void vala_switch_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaSwitchStatement * self;
	self = VALA_SWITCH_STATEMENT (base);
	g_return_if_fail (old_node == NULL || VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (new_node == NULL || VALA_IS_EXPRESSION (new_node));
	if (vala_switch_statement_get_expression (self) == old_node) {
		vala_switch_statement_set_expression (self, new_node);
	}
}


ValaExpression* vala_switch_statement_get_expression (ValaSwitchStatement* self) {
	g_return_val_if_fail (VALA_IS_SWITCH_STATEMENT (self), NULL);
	return self->priv->_expression;
}


void vala_switch_statement_set_expression (ValaSwitchStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_SWITCH_STATEMENT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_expression == NULL ? NULL : (self->priv->_expression = (g_object_unref (self->priv->_expression), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_expression), VALA_CODE_NODE (self));
}


static void vala_switch_statement_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaSwitchStatement * self;
	self = VALA_SWITCH_STATEMENT (object);
	switch (property_id) {
		case VALA_SWITCH_STATEMENT_EXPRESSION:
		g_value_set_object (value, vala_switch_statement_get_expression (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_switch_statement_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaSwitchStatement * self;
	self = VALA_SWITCH_STATEMENT (object);
	switch (property_id) {
		case VALA_SWITCH_STATEMENT_EXPRESSION:
		vala_switch_statement_set_expression (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_switch_statement_class_init (ValaSwitchStatementClass * klass) {
	vala_switch_statement_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSwitchStatementPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_switch_statement_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_switch_statement_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_switch_statement_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_switch_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_switch_statement_real_replace_expression;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SWITCH_STATEMENT_EXPRESSION, g_param_spec_object ("expression", "expression", "expression", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}


static void vala_switch_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_switch_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_switch_statement_init (ValaSwitchStatement * self) {
	self->priv = VALA_SWITCH_STATEMENT_GET_PRIVATE (self);
	self->priv->sections = GEE_LIST (gee_array_list_new (VALA_TYPE_SWITCH_SECTION, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_switch_statement_dispose (GObject * obj) {
	ValaSwitchStatement * self;
	self = VALA_SWITCH_STATEMENT (obj);
	(self->priv->_expression == NULL ? NULL : (self->priv->_expression = (g_object_unref (self->priv->_expression), NULL)));
	(self->priv->sections == NULL ? NULL : (self->priv->sections = (g_object_unref (self->priv->sections), NULL)));
	G_OBJECT_CLASS (vala_switch_statement_parent_class)->dispose (obj);
}


GType vala_switch_statement_get_type (void) {
	static GType vala_switch_statement_type_id = 0;
	if (G_UNLIKELY (vala_switch_statement_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSwitchStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_switch_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSwitchStatement), 0, (GInstanceInitFunc) vala_switch_statement_init };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_switch_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_switch_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaSwitchStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_switch_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_switch_statement_type_id;
}




