/* valacodecontext.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valacodecontext.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valamethod.h>
#include <vala/valasourcefile.h>
#include <vala/valanamespace.h>
#include <vala/valasourcereference.h>
#include <vala/valasourcefilecycle.h>
#include <vala/valacodegenerator.h>
#include <vala/valacodevisitor.h>
#include <vala/valaclass.h>
#include <vala/valastruct.h>
#include <vala/valainterface.h>
#include <vala/valaenum.h>
#include <vala/valaenumvalue.h>
#include <vala/valaexpression.h>
#include <vala/valacallback.h>
#include <vala/valatypereference.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valacreationmethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valasignal.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valatypeparameter.h>
#include <vala/valaemptystatement.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valalocalvariabledeclaration.h>
#include <vala/valavariabledeclarator.h>
#include <vala/valainitializerlist.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valaswitchstatement.h>
#include <vala/valaswitchsection.h>
#include <vala/valaswitchlabel.h>
#include <vala/valawhilestatement.h>
#include <vala/valadostatement.h>
#include <vala/valaforstatement.h>
#include <vala/valaforeachstatement.h>
#include <vala/valabreakstatement.h>
#include <vala/valacontinuestatement.h>
#include <vala/valareturnstatement.h>
#include <vala/valathrowstatement.h>
#include <vala/valatrystatement.h>
#include <vala/valacatchclause.h>
#include <vala/valalockstatement.h>
#include <vala/valaarraycreationexpression.h>
#include <vala/valabooleanliteral.h>
#include <vala/valacharacterliteral.h>
#include <vala/valaintegerliteral.h>
#include <vala/valarealliteral.h>
#include <vala/valastringliteral.h>
#include <vala/valanullliteral.h>
#include <vala/valaliteralexpression.h>
#include <vala/valaliteral.h>
#include <vala/valaparenthesizedexpression.h>
#include <vala/valamemberaccess.h>
#include <vala/valainvocationexpression.h>
#include <vala/valaelementaccess.h>
#include <vala/valabaseaccess.h>
#include <vala/valapostfixexpression.h>
#include <vala/valaobjectcreationexpression.h>
#include <vala/valasizeofexpression.h>
#include <vala/valatypeofexpression.h>
#include <vala/valacastexpression.h>
#include <vala/valapointerindirection.h>
#include <vala/valaaddressofexpression.h>
#include <vala/valareferencetransferexpression.h>
#include <vala/valatypecheck.h>
#include <vala/valaconditionalexpression.h>
#include <vala/valalambdaexpression.h>

struct _ValaCodeContextPrivate {
	char* _library;
	gboolean _memory_management;
	gboolean _assert;
	gboolean _checking;
	gboolean _ccode_only;
	gboolean _compile_only;
	char* _output;
	char* _basedir;
	char* _directory;
	gboolean _debug;
	gint _optlevel;
	gboolean _thread;
	ValaMethod* _module_init_method;
	gboolean _save_temps;
	GeeList* source_files;
	GeeList* c_source_files;
	ValaNamespace* _root;
	GeeList* cycles;
	GeeList* packages;
	ValaCodeGenerator* _codegen;
};
#define VALA_CODE_CONTEXT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CODE_CONTEXT, ValaCodeContextPrivate))
enum  {
	VALA_CODE_CONTEXT_DUMMY_PROPERTY,
	VALA_CODE_CONTEXT_LIBRARY,
	VALA_CODE_CONTEXT_MEMORY_MANAGEMENT,
	VALA_CODE_CONTEXT_ASSERT,
	VALA_CODE_CONTEXT_CHECKING,
	VALA_CODE_CONTEXT_CCODE_ONLY,
	VALA_CODE_CONTEXT_COMPILE_ONLY,
	VALA_CODE_CONTEXT_OUTPUT,
	VALA_CODE_CONTEXT_BASEDIR,
	VALA_CODE_CONTEXT_DIRECTORY,
	VALA_CODE_CONTEXT_DEBUG,
	VALA_CODE_CONTEXT_OPTLEVEL,
	VALA_CODE_CONTEXT_THREAD,
	VALA_CODE_CONTEXT_MODULE_INIT_METHOD,
	VALA_CODE_CONTEXT_SAVE_TEMPS,
	VALA_CODE_CONTEXT_SAVE_CSOURCES,
	VALA_CODE_CONTEXT_SAVE_CHEADERS,
	VALA_CODE_CONTEXT_ROOT,
	VALA_CODE_CONTEXT_CODEGEN
};
static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file);
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GeeCollection* chain);
static GObject * vala_code_context_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_code_context_parent_class = NULL;
static void vala_code_context_dispose (GObject * obj);


ValaCodeContext* vala_code_context_new (void) {
	ValaCodeContext * self;
	self = g_object_newv (VALA_TYPE_CODE_CONTEXT, 0, NULL);
	return self;
}


/**
 * Returns a copy of the list of source files.
 *
 * @return list of source files
 */
GeeCollection* vala_code_context_get_source_files (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_SOURCE_FILE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->source_files)));
}


/**
 * Returns a copy of the list of C source files.
 *
 * @return list of C source files
 */
GeeCollection* vala_code_context_get_c_source_files (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, GEE_COLLECTION (self->priv->c_source_files)));
}


/**
 * Adds the specified file to the list of source files.
 *
 * @param file a source file
 */
void vala_code_context_add_source_file (ValaCodeContext* self, ValaSourceFile* file) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	gee_collection_add (GEE_COLLECTION (self->priv->source_files), file);
}


/**
 * Adds the specified file to the list of C source files.
 *
 * @param file a C source file
 */
void vala_code_context_add_c_source_file (ValaCodeContext* self, const char* file) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (file != NULL);
	gee_collection_add (GEE_COLLECTION (self->priv->c_source_files), file);
}


/**
 * Returns a copy of the list of used packages.
 *
 * @return list of used packages
 */
GeeCollection* vala_code_context_get_packages (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, GEE_COLLECTION (self->priv->packages)));
}


/**
 * Returns whether the specified package is being used.
 *
 * @param pkg a package name
 * @return    true if the specified package is being used
 */
gboolean vala_code_context_has_package (ValaCodeContext* self, const char* pkg) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	return gee_collection_contains (GEE_COLLECTION (self->priv->packages), pkg);
}


/**
 * Adds the specified package to the list of used packages.
 *
 * @param pkg a package name
 */
void vala_code_context_add_package (ValaCodeContext* self, const char* pkg) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (pkg != NULL);
	gee_collection_add (GEE_COLLECTION (self->priv->packages), pkg);
}


/**
 * Visits the complete code tree file by file.
 *
 * @param visitor the visitor to be called when traversing
 */
void vala_code_context_accept (ValaCodeContext* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_code_context_get_root (self)), visitor);
	{
		GeeList* file_collection;
		GeeIterator* file_it;
		file_collection = self->priv->source_files;
		file_it = gee_iterable_iterator (GEE_ITERABLE (file_collection));
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = gee_iterator_get (file_it);
			{
				vala_source_file_accept (file, visitor);
				(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
			}
		}
		(file_it == NULL ? NULL : (file_it = (g_object_unref (file_it), NULL)));
	}
}


/**
 * Find and resolve cycles in source file dependencies.
 */
void vala_code_context_find_header_cycles (ValaCodeContext* self) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	{
		GeeList* file_collection;
		GeeIterator* file_it;
		file_collection = self->priv->source_files;
		file_it = gee_iterable_iterator (GEE_ITERABLE (file_collection));
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = gee_iterator_get (file_it);
			{
				/* find cycles in dependencies between source files 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file)) {
					(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
					continue;
				}
				if (vala_source_file_get_mark (file) == 0) {
					GeeArrayList* _tmp0;
					_tmp0 = NULL;
					vala_code_context_visit (self, file, GEE_COLLECTION ((_tmp0 = gee_array_list_new (VALA_TYPE_SOURCE_FILE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal))));
					(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
				}
				(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
			}
		}
		(file_it == NULL ? NULL : (file_it = (g_object_unref (file_it), NULL)));
	}
	{
		GeeList* cycle_collection;
		GeeIterator* cycle_it;
		cycle_collection = self->priv->cycles;
		cycle_it = gee_iterable_iterator (GEE_ITERABLE (cycle_collection));
		while (gee_iterator_next (cycle_it)) {
			ValaSourceFileCycle* cycle;
			cycle = gee_iterator_get (cycle_it);
			{
				/* find one head for each cycle, it must not have any
				 * hard dependencies on other files in the cycle
				 */
				cycle->head = vala_code_context_find_cycle_head (self, gee_list_get (cycle->files, 0));
				vala_source_file_set_is_cycle_head (cycle->head, TRUE);
				(cycle == NULL ? NULL : (cycle = (g_object_unref (cycle), NULL)));
			}
		}
		(cycle_it == NULL ? NULL : (cycle_it = (g_object_unref (cycle_it), NULL)));
	}
	{
		GeeList* file2_collection;
		GeeIterator* file2_it;
		file2_collection = self->priv->source_files;
		file2_it = gee_iterable_iterator (GEE_ITERABLE (file2_collection));
		while (gee_iterator_next (file2_it)) {
			ValaSourceFile* file2;
			file2 = gee_iterator_get (file2_it);
			{
				/* connect the source files in a non-cyclic way
				 * cycle members connect to the head of their cycle
				 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file2)) {
					(file2 == NULL ? NULL : (file2 = (g_object_unref (file2), NULL)));
					continue;
				}
				{
					GeeCollection* dep_collection;
					GeeIterator* dep_it;
					dep_collection = vala_source_file_get_header_internal_dependencies (file2);
					dep_it = gee_iterable_iterator (GEE_ITERABLE (dep_collection));
					while (gee_iterator_next (dep_it)) {
						ValaSourceFile* _tmp3;
						ValaSourceFile* dep;
						_tmp3 = NULL;
						dep = (_tmp3 = gee_iterator_get (dep_it), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
						{
							if (vala_source_file_get_cycle (file2) != NULL && vala_source_file_get_cycle (dep) == vala_source_file_get_cycle (file2)) {
								/* in the same cycle */
								if (!vala_source_file_get_is_cycle_head (file2)) {
									char* _tmp1;
									/* include header of cycle head */
									_tmp1 = NULL;
									vala_source_file_add_header_internal_include (file2, (_tmp1 = vala_source_file_get_cinclude_filename (vala_source_file_get_cycle (file2)->head)));
									(_tmp1 = (g_free (_tmp1), NULL));
								}
							} else {
								char* _tmp2;
								/* we can just include the headers if they are not in a cycle or not in the same cycle as the current file */
								_tmp2 = NULL;
								vala_source_file_add_header_internal_include (file2, (_tmp2 = vala_source_file_get_cinclude_filename (dep)));
								(_tmp2 = (g_free (_tmp2), NULL));
							}
							(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
						}
					}
					(dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL)));
					(dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL)));
				}
				(file2 == NULL ? NULL : (file2 = (g_object_unref (file2), NULL)));
			}
		}
		(file2_it == NULL ? NULL : (file2_it = (g_object_unref (file2_it), NULL)));
	}
}


static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (file), NULL);
	{
		GeeCollection* dep_collection;
		GeeIterator* dep_it;
		dep_collection = vala_source_file_get_header_internal_full_dependencies (file);
		dep_it = gee_iterable_iterator (GEE_ITERABLE (dep_collection));
		while (gee_iterator_next (dep_it)) {
			ValaSourceFile* _tmp2;
			ValaSourceFile* dep;
			_tmp2 = NULL;
			dep = (_tmp2 = gee_iterator_get (dep_it), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
			{
				if (dep == file) {
					(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
					continue;
				}
				{
					GeeList* cycle_file_collection;
					GeeIterator* cycle_file_it;
					cycle_file_collection = vala_source_file_get_cycle (file)->files;
					cycle_file_it = gee_iterable_iterator (GEE_ITERABLE (cycle_file_collection));
					while (gee_iterator_next (cycle_file_it)) {
						ValaSourceFile* _tmp1;
						ValaSourceFile* cycle_file;
						_tmp1 = NULL;
						cycle_file = (_tmp1 = gee_iterator_get (cycle_file_it), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
						{
							/* ignore file-internal dependencies */
							if (dep == cycle_file) {
								ValaSourceFile* _tmp0;
								_tmp0 = NULL;
								return (_tmp0 = vala_code_context_find_cycle_head (self, dep), (cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL))), (cycle_file_it == NULL ? NULL : (cycle_file_it = (g_object_unref (cycle_file_it), NULL))), (dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL))), (dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL))), (dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL))), _tmp0);
							}
							(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
						}
					}
					(cycle_file_it == NULL ? NULL : (cycle_file_it = (g_object_unref (cycle_file_it), NULL)));
				}
				(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
			}
		}
		(dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL)));
		(dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL)));
	}
	return file;
}


/* no hard dependencies on members of the same cycle found
 * source file suitable as cycle head
 */
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GeeCollection* chain) {
	GeeList* l;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	g_return_if_fail (chain == NULL || GEE_IS_COLLECTION (chain));
	l = GEE_LIST (gee_array_list_new (VALA_TYPE_SOURCE_FILE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	{
		GeeCollection* chain_file_collection;
		GeeIterator* chain_file_it;
		chain_file_collection = chain;
		chain_file_it = gee_iterable_iterator (GEE_ITERABLE (chain_file_collection));
		while (gee_iterator_next (chain_file_it)) {
			ValaSourceFile* chain_file;
			chain_file = gee_iterator_get (chain_file_it);
			{
				gee_collection_add (GEE_COLLECTION (l), chain_file);
				(chain_file == NULL ? NULL : (chain_file = (g_object_unref (chain_file), NULL)));
			}
		}
		(chain_file_it == NULL ? NULL : (chain_file_it = (g_object_unref (chain_file_it), NULL)));
	}
	gee_collection_add (GEE_COLLECTION (l), file);
	/* mark file as currently being visited */
	vala_source_file_set_mark (file, 1);
	{
		GeeCollection* dep_collection;
		GeeIterator* dep_it;
		dep_collection = vala_source_file_get_header_internal_dependencies (file);
		dep_it = gee_iterable_iterator (GEE_ITERABLE (dep_collection));
		while (gee_iterator_next (dep_it)) {
			ValaSourceFile* _tmp2;
			ValaSourceFile* dep;
			_tmp2 = NULL;
			dep = (_tmp2 = gee_iterator_get (dep_it), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
			{
				if (file == dep) {
					(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
					continue;
				}
				if (vala_source_file_get_mark (dep) == 1) {
					ValaSourceFileCycle* cycle;
					gboolean cycle_start_found;
					/* found cycle */
					cycle = vala_source_file_cycle_new ();
					gee_collection_add (GEE_COLLECTION (self->priv->cycles), cycle);
					cycle_start_found = FALSE;
					{
						GeeList* cycle_file_collection;
						GeeIterator* cycle_file_it;
						cycle_file_collection = l;
						cycle_file_it = gee_iterable_iterator (GEE_ITERABLE (cycle_file_collection));
						while (gee_iterator_next (cycle_file_it)) {
							ValaSourceFile* cycle_file;
							cycle_file = gee_iterator_get (cycle_file_it);
							{
								ValaSourceFileCycle* _tmp0;
								ValaSourceFileCycle* ref_cycle_file_cycle;
								_tmp0 = NULL;
								ref_cycle_file_cycle = (_tmp0 = vala_source_file_get_cycle (cycle_file), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
								if (!cycle_start_found) {
									if (cycle_file == dep) {
										cycle_start_found = TRUE;
									}
								}
								if (!cycle_start_found) {
									(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
									(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
									continue;
								}
								if (vala_source_file_get_cycle (cycle_file) != NULL) {
									/* file already in a cycle */
									if (vala_source_file_get_cycle (cycle_file) == cycle) {
										(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
										(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
										continue;
									}
									/* file is in the same cycle, nothing to do 
									 file is in an other cycle, merge the two cycles */
									gee_collection_remove (GEE_COLLECTION (self->priv->cycles), vala_source_file_get_cycle (cycle_file));
									{
										GeeList* inner_cycle_file_collection;
										GeeIterator* inner_cycle_file_it;
										inner_cycle_file_collection = vala_source_file_get_cycle (cycle_file)->files;
										inner_cycle_file_it = gee_iterable_iterator (GEE_ITERABLE (inner_cycle_file_collection));
										while (gee_iterator_next (inner_cycle_file_it)) {
											ValaSourceFile* _tmp1;
											ValaSourceFile* inner_cycle_file;
											_tmp1 = NULL;
											inner_cycle_file = (_tmp1 = gee_iterator_get (inner_cycle_file_it), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
											{
												if (vala_source_file_get_cycle (inner_cycle_file) != cycle) {
													/* file in inner cycle not yet added to outer cycle */
													gee_collection_add (GEE_COLLECTION (cycle->files), inner_cycle_file);
													vala_source_file_set_cycle (inner_cycle_file, cycle);
												}
												(inner_cycle_file == NULL ? NULL : (inner_cycle_file = (g_object_unref (inner_cycle_file), NULL)));
											}
										}
										(inner_cycle_file_it == NULL ? NULL : (inner_cycle_file_it = (g_object_unref (inner_cycle_file_it), NULL)));
									}
								} else {
									gee_collection_add (GEE_COLLECTION (cycle->files), cycle_file);
									vala_source_file_set_cycle (cycle_file, cycle);
								}
								(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
								(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
							}
						}
						(cycle_file_it == NULL ? NULL : (cycle_file_it = (g_object_unref (cycle_file_it), NULL)));
					}
					(cycle == NULL ? NULL : (cycle = (g_object_unref (cycle), NULL)));
				} else {
					if (vala_source_file_get_mark (dep) == 0) {
						/* found not yet visited file */
						vala_code_context_visit (self, dep, GEE_COLLECTION (l));
					}
				}
				(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
			}
		}
		(dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL)));
		(dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL)));
	}
	/* mark file as successfully visited */
	vala_source_file_set_mark (file, 2);
	(l == NULL ? NULL : (l = (g_object_unref (l), NULL)));
}


ValaNamespace* vala_code_context_create_namespace (ValaCodeContext* self, const char* name, ValaSourceReference* source_reference) {
	ValaNamespace* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_namespace_new (name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_namespace_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaClass* vala_code_context_create_class (ValaCodeContext* self, const char* name, ValaSourceReference* source_reference) {
	ValaClass* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_class_new (name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_class_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaStruct* vala_code_context_create_struct (ValaCodeContext* self, const char* name, ValaSourceReference* source_reference) {
	ValaStruct* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_struct_new (name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_struct_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaInterface* vala_code_context_create_interface (ValaCodeContext* self, const char* name, ValaSourceReference* source_reference) {
	ValaInterface* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_interface_new (name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_interface_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaEnum* vala_code_context_create_enum (ValaCodeContext* self, const char* name, ValaSourceReference* source_reference) {
	ValaEnum* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_enum_new (name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_enum_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaEnumValue* vala_code_context_create_enum_value (ValaCodeContext* self, const char* name) {
	ValaEnumValue* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	node = vala_enum_value_new (name);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_enum_value_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaEnumValue* vala_code_context_create_enum_value_with_value (ValaCodeContext* self, const char* name, ValaExpression* value) {
	ValaEnumValue* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (value == NULL || VALA_IS_EXPRESSION (value), NULL);
	node = vala_enum_value_new_with_value (name, value);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_enum_value_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaCallback* vala_code_context_create_callback (ValaCodeContext* self, const char* name, ValaTypeReference* return_type, ValaSourceReference* source_reference) {
	ValaCallback* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (return_type == NULL || VALA_IS_TYPE_REFERENCE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_callback_new (name, return_type, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_callback_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaConstant* vala_code_context_create_constant (ValaCodeContext* self, const char* name, ValaTypeReference* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaConstant* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_constant_new (name, type_reference, initializer, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_constant_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaField* vala_code_context_create_field (ValaCodeContext* self, const char* name, ValaTypeReference* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaField* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_field_new (name, type_reference, initializer, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_field_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaMethod* vala_code_context_create_method (ValaCodeContext* self, const char* name, ValaTypeReference* return_type, ValaSourceReference* source_reference) {
	ValaMethod* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (return_type == NULL || VALA_IS_TYPE_REFERENCE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_method_new (name, return_type, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_method_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaCreationMethod* vala_code_context_create_creation_method (ValaCodeContext* self, const char* type_name, const char* name, ValaSourceReference* source_reference) {
	ValaCreationMethod* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_creation_method_new (type_name, name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_creation_method_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaFormalParameter* vala_code_context_create_formal_parameter (ValaCodeContext* self, const char* name, ValaTypeReference* type_reference, ValaSourceReference* source_reference) {
	ValaFormalParameter* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (type_reference == NULL || VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_formal_parameter_new (name, type_reference, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_formal_parameter_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaFormalParameter* vala_code_context_create_formal_parameter_with_ellipsis (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaFormalParameter* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_formal_parameter_new_with_ellipsis (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_formal_parameter_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaProperty* vala_code_context_create_property (ValaCodeContext* self, const char* name, ValaTypeReference* type_reference, ValaPropertyAccessor* get_accessor, ValaPropertyAccessor* set_accessor, ValaSourceReference* source_reference) {
	ValaProperty* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (get_accessor == NULL || VALA_IS_PROPERTY_ACCESSOR (get_accessor), NULL);
	g_return_val_if_fail (set_accessor == NULL || VALA_IS_PROPERTY_ACCESSOR (set_accessor), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_property_new (name, type_reference, get_accessor, set_accessor, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_property_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaPropertyAccessor* vala_code_context_create_property_accessor (ValaCodeContext* self, gboolean readable, gboolean writable, gboolean construction, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaPropertyAccessor* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_property_accessor_new (readable, writable, construction, body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_property_accessor_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaSignal* vala_code_context_create_signal (ValaCodeContext* self, const char* name, ValaTypeReference* return_type, ValaSourceReference* source_reference) {
	ValaSignal* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_signal_new (name, return_type, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_signal_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaConstructor* vala_code_context_create_constructor (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaConstructor* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_constructor_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_constructor_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaDestructor* vala_code_context_create_destructor (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaDestructor* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_destructor_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_destructor_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaTypeParameter* vala_code_context_create_type_parameter (ValaCodeContext* self, const char* name, ValaSourceReference* source_reference) {
	ValaTypeParameter* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_type_parameter_new (name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_type_parameter_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaBlock* vala_code_context_create_block (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaBlock* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_block_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_block_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaEmptyStatement* vala_code_context_create_empty_statement (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaEmptyStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_empty_statement_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_empty_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaDeclarationStatement* vala_code_context_create_declaration_statement (ValaCodeContext* self, ValaLocalVariableDeclaration* declaration, ValaSourceReference* source_reference) {
	ValaDeclarationStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_LOCAL_VARIABLE_DECLARATION (declaration), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_declaration_statement_new (declaration, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_declaration_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaLocalVariableDeclaration* vala_code_context_create_local_variable_declaration (ValaCodeContext* self, ValaTypeReference* type_reference, ValaSourceReference* source_reference) {
	ValaLocalVariableDeclaration* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (type_reference == NULL || VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_local_variable_declaration_new (type_reference, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_local_variable_declaration_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaLocalVariableDeclaration* vala_code_context_create_local_variable_declaration_var_type (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaLocalVariableDeclaration* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_local_variable_declaration_new_var_type (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_local_variable_declaration_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaVariableDeclarator* vala_code_context_create_variable_declarator (ValaCodeContext* self, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaVariableDeclarator* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_variable_declarator_new (name, initializer, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_variable_declarator_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaInitializerList* vala_code_context_create_initializer_list (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaInitializerList* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_initializer_list_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_initializer_list_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaExpressionStatement* vala_code_context_create_expression_statement (ValaCodeContext* self, ValaExpression* expression, ValaSourceReference* source_reference) {
	ValaExpressionStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expression), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_expression_statement_new (expression, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_expression_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaIfStatement* vala_code_context_create_if_statement (ValaCodeContext* self, ValaExpression* condition, ValaBlock* true_statement, ValaBlock* false_statement, ValaSourceReference* source_reference) {
	ValaIfStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (condition), NULL);
	g_return_val_if_fail (VALA_IS_BLOCK (true_statement), NULL);
	g_return_val_if_fail (false_statement == NULL || VALA_IS_BLOCK (false_statement), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_if_statement_new (condition, true_statement, false_statement, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_if_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaSwitchStatement* vala_code_context_create_switch_statement (ValaCodeContext* self, ValaExpression* expression, ValaSourceReference* source_reference) {
	ValaSwitchStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expression), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_switch_statement_new (expression, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_switch_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaSwitchSection* vala_code_context_create_switch_section (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaSwitchSection* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_switch_section_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_switch_section_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaSwitchLabel* vala_code_context_create_switch_label (ValaCodeContext* self, ValaExpression* expression, ValaSourceReference* source_reference) {
	ValaSwitchLabel* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (expression == NULL || VALA_IS_EXPRESSION (expression), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_switch_label_new (expression, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_switch_label_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaSwitchLabel* vala_code_context_create_switch_label_with_default (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaSwitchLabel* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_switch_label_new_with_default (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_switch_label_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaWhileStatement* vala_code_context_create_while_statement (ValaCodeContext* self, ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaWhileStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (condition), NULL);
	g_return_val_if_fail (VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_while_statement_new (condition, body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_while_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaDoStatement* vala_code_context_create_do_statement (ValaCodeContext* self, ValaBlock* body, ValaExpression* condition, ValaSourceReference* source_reference) {
	ValaDoStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (condition), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_do_statement_new (body, condition, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_do_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaForStatement* vala_code_context_create_for_statement (ValaCodeContext* self, ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaForStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (condition == NULL || VALA_IS_EXPRESSION (condition), NULL);
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_for_statement_new (condition, body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_for_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaForeachStatement* vala_code_context_create_foreach_statement (ValaCodeContext* self, ValaTypeReference* type_reference, const char* variable_name, ValaExpression* collection, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaForeachStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (variable_name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (collection), NULL);
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_foreach_statement_new (type_reference, variable_name, collection, body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_foreach_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaBreakStatement* vala_code_context_create_break_statement (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaBreakStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_break_statement_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_break_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaContinueStatement* vala_code_context_create_continue_statement (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaContinueStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_continue_statement_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_continue_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaReturnStatement* vala_code_context_create_return_statement (ValaCodeContext* self, ValaExpression* return_expression, ValaSourceReference* source_reference) {
	ValaReturnStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (return_expression == NULL || VALA_IS_EXPRESSION (return_expression), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_return_statement_new (return_expression, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_return_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaThrowStatement* vala_code_context_create_throw_statement (ValaCodeContext* self, ValaExpression* error_expression, ValaSourceReference* source_reference) {
	ValaThrowStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (error_expression), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_throw_statement_new (error_expression, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_throw_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaTryStatement* vala_code_context_create_try_statement (ValaCodeContext* self, ValaBlock* body, ValaBlock* finally_body, ValaSourceReference* source_reference) {
	ValaTryStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (finally_body == NULL || VALA_IS_BLOCK (finally_body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_try_statement_new (body, finally_body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_try_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaCatchClause* vala_code_context_create_catch_clause (ValaCodeContext* self, ValaTypeReference* type_reference, const char* variable_name, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaCatchClause* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (type_reference == NULL || VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_catch_clause_new (type_reference, variable_name, body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_catch_clause_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaLockStatement* vala_code_context_create_lock_statement (ValaCodeContext* self, ValaExpression* resource, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaLockStatement* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (resource == NULL || VALA_IS_EXPRESSION (resource), NULL);
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_lock_statement_new (resource, body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_lock_statement_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaArrayCreationExpression* vala_code_context_create_array_creation_expression (ValaCodeContext* self, ValaTypeReference* element_type, gint rank, ValaInitializerList* initializer_list, ValaSourceReference* source_reference) {
	ValaArrayCreationExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (element_type == NULL || VALA_IS_TYPE_REFERENCE (element_type), NULL);
	g_return_val_if_fail (initializer_list == NULL || VALA_IS_INITIALIZER_LIST (initializer_list), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_array_creation_expression_new (element_type, rank, initializer_list, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_array_creation_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaBooleanLiteral* vala_code_context_create_boolean_literal (ValaCodeContext* self, gboolean value, ValaSourceReference* source_reference) {
	ValaBooleanLiteral* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_boolean_literal_new (value, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_boolean_literal_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaCharacterLiteral* vala_code_context_create_character_literal (ValaCodeContext* self, const char* value, ValaSourceReference* source_reference) {
	ValaCharacterLiteral* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (value != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_character_literal_new (value, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_character_literal_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaIntegerLiteral* vala_code_context_create_integer_literal (ValaCodeContext* self, const char* value, ValaSourceReference* source_reference) {
	ValaIntegerLiteral* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (value != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_integer_literal_new (value, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_integer_literal_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaRealLiteral* vala_code_context_create_real_literal (ValaCodeContext* self, const char* value, ValaSourceReference* source_reference) {
	ValaRealLiteral* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_real_literal_new (value, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_real_literal_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaStringLiteral* vala_code_context_create_string_literal (ValaCodeContext* self, const char* value, ValaSourceReference* source_reference) {
	ValaStringLiteral* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_string_literal_new (value, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_string_literal_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaNullLiteral* vala_code_context_create_null_literal (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaNullLiteral* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_null_literal_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_null_literal_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaLiteralExpression* vala_code_context_create_literal_expression (ValaCodeContext* self, ValaLiteral* literal, ValaSourceReference* source_reference) {
	ValaLiteralExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_LITERAL (literal), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_literal_expression_new (literal, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_literal_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaParenthesizedExpression* vala_code_context_create_parenthesized_expression (ValaCodeContext* self, ValaExpression* inner, ValaSourceReference* source_reference) {
	ValaParenthesizedExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_parenthesized_expression_new (inner, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_parenthesized_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaMemberAccess* vala_code_context_create_member_access (ValaCodeContext* self, ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (inner == NULL || VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_member_access_new (inner, member_name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_member_access_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaMemberAccess* vala_code_context_create_member_access_simple (ValaCodeContext* self, const char* member_name, ValaSourceReference* source_reference) {
	ValaMemberAccess* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_member_access_new_simple (member_name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_member_access_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaInvocationExpression* vala_code_context_create_invocation_expression (ValaCodeContext* self, ValaExpression* call, ValaSourceReference* source_reference) {
	ValaInvocationExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (call), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_invocation_expression_new (call, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_invocation_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaElementAccess* vala_code_context_create_element_access (ValaCodeContext* self, ValaExpression* container, ValaSourceReference* source_reference) {
	ValaElementAccess* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (container == NULL || VALA_IS_EXPRESSION (container), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_element_access_new (container, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_element_access_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaBaseAccess* vala_code_context_create_base_access (ValaCodeContext* self, ValaSourceReference* source_reference) {
	ValaBaseAccess* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_base_access_new (source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_base_access_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaPostfixExpression* vala_code_context_create_postfix_expression (ValaCodeContext* self, ValaExpression* inner, gboolean increment, ValaSourceReference* source_reference) {
	ValaPostfixExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_postfix_expression_new (inner, increment, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_postfix_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaObjectCreationExpression* vala_code_context_create_object_creation_expression (ValaCodeContext* self, ValaMemberAccess* member_name, ValaSourceReference* source_reference) {
	ValaObjectCreationExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (member_name), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_object_creation_expression_new (member_name, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_object_creation_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaSizeofExpression* vala_code_context_create_sizeof_expression (ValaCodeContext* self, ValaTypeReference* type_reference, ValaSourceReference* source_reference) {
	ValaSizeofExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_sizeof_expression_new (type_reference, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_sizeof_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaTypeofExpression* vala_code_context_create_typeof_expression (ValaCodeContext* self, ValaTypeReference* type_reference, ValaSourceReference* source_reference) {
	ValaTypeofExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_typeof_expression_new (type_reference, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_typeof_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaUnaryExpression* vala_code_context_create_unary_expression (ValaCodeContext* self, ValaUnaryOperator operator, ValaExpression* inner, ValaSourceReference* source_reference) {
	ValaUnaryExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_unary_expression_new (operator, inner, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_unary_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaCastExpression* vala_code_context_create_cast_expression (ValaCodeContext* self, ValaExpression* inner, ValaTypeReference* type_reference, ValaSourceReference* source_reference, gboolean is_silent_cast) {
	ValaCastExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_cast_expression_new (inner, type_reference, source_reference, is_silent_cast);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_cast_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaPointerIndirection* vala_code_context_create_pointer_indirection (ValaCodeContext* self, ValaExpression* inner, ValaSourceReference* source_reference) {
	ValaPointerIndirection* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_pointer_indirection_new (inner, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_pointer_indirection_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaAddressofExpression* vala_code_context_create_addressof_expression (ValaCodeContext* self, ValaExpression* inner, ValaSourceReference* source_reference) {
	ValaAddressofExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_addressof_expression_new (inner, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_addressof_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaReferenceTransferExpression* vala_code_context_create_reference_transfer_expression (ValaCodeContext* self, ValaExpression* inner, ValaSourceReference* source_reference) {
	ValaReferenceTransferExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_reference_transfer_expression_new (inner, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_reference_transfer_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaBinaryExpression* vala_code_context_create_binary_expression (ValaCodeContext* self, ValaBinaryOperator operator, ValaExpression* left, ValaExpression* right, ValaSourceReference* source_reference) {
	ValaBinaryExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (left), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (right), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_binary_expression_new (operator, left, right, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_binary_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaTypeCheck* vala_code_context_create_type_check (ValaCodeContext* self, ValaExpression* expression, ValaTypeReference* type_reference, ValaSourceReference* source_reference) {
	ValaTypeCheck* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expression), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_type_check_new (expression, type_reference, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_type_check_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaConditionalExpression* vala_code_context_create_conditional_expression (ValaCodeContext* self, ValaExpression* condition, ValaExpression* true_expression, ValaExpression* false_expression, ValaSourceReference* source_reference) {
	ValaConditionalExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (condition), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (true_expression), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (false_expression), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_conditional_expression_new (condition, true_expression, false_expression, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_conditional_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaLambdaExpression* vala_code_context_create_lambda_expression (ValaCodeContext* self, ValaExpression* expression_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expression_body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_lambda_expression_new (expression_body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_lambda_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaLambdaExpression* vala_code_context_create_lambda_expression_with_statement_body (ValaCodeContext* self, ValaBlock* statement_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_BLOCK (statement_body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_lambda_expression_new_with_statement_body (statement_body, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_lambda_expression_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


ValaAssignment* vala_code_context_create_assignment (ValaCodeContext* self, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	ValaAssignment* node;
	ValaCodeBinding* _tmp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (left), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (right), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	node = vala_assignment_new (left, right, operator, source_reference);
	_tmp0 = NULL;
	vala_code_node_set_code_binding (VALA_CODE_NODE (node), (_tmp0 = vala_code_generator_create_assignment_binding (vala_code_context_get_codegen (self), node)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	return node;
	(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
}


char* vala_code_context_get_library (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_library;
}


void vala_code_context_set_library (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_library = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_library = (g_free (self->priv->_library), NULL)), _tmp2);
}


gboolean vala_code_context_get_memory_management (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_memory_management;
}


void vala_code_context_set_memory_management (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_memory_management = value;
}


gboolean vala_code_context_get_assert (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_assert;
}


void vala_code_context_set_assert (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_assert = value;
}


gboolean vala_code_context_get_checking (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_checking;
}


void vala_code_context_set_checking (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_checking = value;
}


gboolean vala_code_context_get_ccode_only (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_ccode_only;
}


void vala_code_context_set_ccode_only (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_ccode_only = value;
}


gboolean vala_code_context_get_compile_only (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_compile_only;
}


void vala_code_context_set_compile_only (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_compile_only = value;
}


char* vala_code_context_get_output (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_output;
}


void vala_code_context_set_output (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_output = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_output = (g_free (self->priv->_output), NULL)), _tmp2);
}


char* vala_code_context_get_basedir (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_basedir;
}


void vala_code_context_set_basedir (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_basedir = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_basedir = (g_free (self->priv->_basedir), NULL)), _tmp2);
}


char* vala_code_context_get_directory (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_directory;
}


void vala_code_context_set_directory (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_directory = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_directory = (g_free (self->priv->_directory), NULL)), _tmp2);
}


gboolean vala_code_context_get_debug (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_debug;
}


void vala_code_context_set_debug (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_debug = value;
}


gint vala_code_context_get_optlevel (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), 0);
	return self->priv->_optlevel;
}


void vala_code_context_set_optlevel (ValaCodeContext* self, gint value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_optlevel = value;
}


gboolean vala_code_context_get_thread (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_thread;
}


void vala_code_context_set_thread (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_thread = value;
}


ValaMethod* vala_code_context_get_module_init_method (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_module_init_method;
}


void vala_code_context_set_module_init_method (ValaCodeContext* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_module_init_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL))), _tmp2);
}


gboolean vala_code_context_get_save_temps (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_save_temps;
}


void vala_code_context_set_save_temps (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_save_temps = value;
}


gboolean vala_code_context_get_save_csources (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return vala_code_context_get_save_temps (self);
}


gboolean vala_code_context_get_save_cheaders (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return vala_code_context_get_save_csources (self) || NULL != vala_code_context_get_library (self);
}


ValaNamespace* vala_code_context_get_root (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_root;
}


ValaCodeGenerator* vala_code_context_get_codegen (ValaCodeContext* self) {
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_codegen;
}


void vala_code_context_set_codegen (ValaCodeContext* self, ValaCodeGenerator* value) {
	ValaCodeGenerator* _tmp2;
	ValaCodeGenerator* _tmp1;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_codegen = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_codegen == NULL ? NULL : (self->priv->_codegen = (g_object_unref (self->priv->_codegen), NULL))), _tmp2);
}


static GObject * vala_code_context_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	ValaCodeContextClass * klass;
	GObjectClass * parent_class;
	ValaCodeContext * self;
	klass = VALA_CODE_CONTEXT_CLASS (g_type_class_peek (VALA_TYPE_CODE_CONTEXT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_CODE_CONTEXT (obj);
	{
		ValaCodeGenerator* _tmp0;
		_tmp0 = NULL;
		vala_code_context_set_codegen (self, (_tmp0 = vala_code_generator_new ()));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	}
	return obj;
}


static void vala_code_context_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		g_value_set_string (value, vala_code_context_get_library (self));
		break;
		case VALA_CODE_CONTEXT_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_code_context_get_memory_management (self));
		break;
		case VALA_CODE_CONTEXT_ASSERT:
		g_value_set_boolean (value, vala_code_context_get_assert (self));
		break;
		case VALA_CODE_CONTEXT_CHECKING:
		g_value_set_boolean (value, vala_code_context_get_checking (self));
		break;
		case VALA_CODE_CONTEXT_CCODE_ONLY:
		g_value_set_boolean (value, vala_code_context_get_ccode_only (self));
		break;
		case VALA_CODE_CONTEXT_COMPILE_ONLY:
		g_value_set_boolean (value, vala_code_context_get_compile_only (self));
		break;
		case VALA_CODE_CONTEXT_OUTPUT:
		g_value_set_string (value, vala_code_context_get_output (self));
		break;
		case VALA_CODE_CONTEXT_BASEDIR:
		g_value_set_string (value, vala_code_context_get_basedir (self));
		break;
		case VALA_CODE_CONTEXT_DIRECTORY:
		g_value_set_string (value, vala_code_context_get_directory (self));
		break;
		case VALA_CODE_CONTEXT_DEBUG:
		g_value_set_boolean (value, vala_code_context_get_debug (self));
		break;
		case VALA_CODE_CONTEXT_OPTLEVEL:
		g_value_set_int (value, vala_code_context_get_optlevel (self));
		break;
		case VALA_CODE_CONTEXT_THREAD:
		g_value_set_boolean (value, vala_code_context_get_thread (self));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		g_value_set_object (value, vala_code_context_get_module_init_method (self));
		break;
		case VALA_CODE_CONTEXT_SAVE_TEMPS:
		g_value_set_boolean (value, vala_code_context_get_save_temps (self));
		break;
		case VALA_CODE_CONTEXT_SAVE_CSOURCES:
		g_value_set_boolean (value, vala_code_context_get_save_csources (self));
		break;
		case VALA_CODE_CONTEXT_SAVE_CHEADERS:
		g_value_set_boolean (value, vala_code_context_get_save_cheaders (self));
		break;
		case VALA_CODE_CONTEXT_ROOT:
		g_value_set_object (value, vala_code_context_get_root (self));
		break;
		case VALA_CODE_CONTEXT_CODEGEN:
		g_value_set_object (value, vala_code_context_get_codegen (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_code_context_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		vala_code_context_set_library (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_MEMORY_MANAGEMENT:
		vala_code_context_set_memory_management (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_ASSERT:
		vala_code_context_set_assert (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_CHECKING:
		vala_code_context_set_checking (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_CCODE_ONLY:
		vala_code_context_set_ccode_only (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_COMPILE_ONLY:
		vala_code_context_set_compile_only (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_OUTPUT:
		vala_code_context_set_output (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_BASEDIR:
		vala_code_context_set_basedir (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_DIRECTORY:
		vala_code_context_set_directory (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_DEBUG:
		vala_code_context_set_debug (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_OPTLEVEL:
		vala_code_context_set_optlevel (self, g_value_get_int (value));
		break;
		case VALA_CODE_CONTEXT_THREAD:
		vala_code_context_set_thread (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		vala_code_context_set_module_init_method (self, g_value_get_object (value));
		break;
		case VALA_CODE_CONTEXT_SAVE_TEMPS:
		vala_code_context_set_save_temps (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_CODEGEN:
		vala_code_context_set_codegen (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_code_context_class_init (ValaCodeContextClass * klass) {
	vala_code_context_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCodeContextPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_code_context_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_code_context_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_code_context_constructor;
	G_OBJECT_CLASS (klass)->dispose = vala_code_context_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_LIBRARY, g_param_spec_string ("library", "library", "library", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "memory-management", "memory-management", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_ASSERT, g_param_spec_boolean ("assert", "assert", "assert", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_CHECKING, g_param_spec_boolean ("checking", "checking", "checking", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_CCODE_ONLY, g_param_spec_boolean ("ccode-only", "ccode-only", "ccode-only", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_COMPILE_ONLY, g_param_spec_boolean ("compile-only", "compile-only", "compile-only", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_OUTPUT, g_param_spec_string ("output", "output", "output", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_BASEDIR, g_param_spec_string ("basedir", "basedir", "basedir", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_DIRECTORY, g_param_spec_string ("directory", "directory", "directory", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_DEBUG, g_param_spec_boolean ("debug", "debug", "debug", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_OPTLEVEL, g_param_spec_int ("optlevel", "optlevel", "optlevel", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_THREAD, g_param_spec_boolean ("thread", "thread", "thread", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_MODULE_INIT_METHOD, g_param_spec_object ("module-init-method", "module-init-method", "module-init-method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_SAVE_TEMPS, g_param_spec_boolean ("save-temps", "save-temps", "save-temps", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_SAVE_CSOURCES, g_param_spec_boolean ("save-csources", "save-csources", "save-csources", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_SAVE_CHEADERS, g_param_spec_boolean ("save-cheaders", "save-cheaders", "save-cheaders", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_ROOT, g_param_spec_object ("root", "root", "root", VALA_TYPE_NAMESPACE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_CODEGEN, g_param_spec_object ("codegen", "codegen", "codegen", VALA_TYPE_CODE_GENERATOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_code_context_init (ValaCodeContext * self) {
	self->priv = VALA_CODE_CONTEXT_GET_PRIVATE (self);
	self->priv->source_files = GEE_LIST (gee_array_list_new (VALA_TYPE_SOURCE_FILE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->c_source_files = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal));
	self->priv->_root = vala_namespace_new (NULL, NULL);
	self->priv->cycles = GEE_LIST (gee_array_list_new (VALA_TYPE_SOURCE_FILE_CYCLE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->packages = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_str_equal));
}


static void vala_code_context_dispose (GObject * obj) {
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (obj);
	(self->priv->_library = (g_free (self->priv->_library), NULL));
	(self->priv->_output = (g_free (self->priv->_output), NULL));
	(self->priv->_basedir = (g_free (self->priv->_basedir), NULL));
	(self->priv->_directory = (g_free (self->priv->_directory), NULL));
	(self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL)));
	(self->priv->source_files == NULL ? NULL : (self->priv->source_files = (g_object_unref (self->priv->source_files), NULL)));
	(self->priv->c_source_files == NULL ? NULL : (self->priv->c_source_files = (g_object_unref (self->priv->c_source_files), NULL)));
	(self->priv->_root == NULL ? NULL : (self->priv->_root = (g_object_unref (self->priv->_root), NULL)));
	(self->priv->cycles == NULL ? NULL : (self->priv->cycles = (g_object_unref (self->priv->cycles), NULL)));
	(self->priv->packages == NULL ? NULL : (self->priv->packages = (g_object_unref (self->priv->packages), NULL)));
	(self->priv->_codegen == NULL ? NULL : (self->priv->_codegen = (g_object_unref (self->priv->_codegen), NULL)));
	G_OBJECT_CLASS (vala_code_context_parent_class)->dispose (obj);
}


GType vala_code_context_get_type (void) {
	static GType vala_code_context_type_id = 0;
	if (G_UNLIKELY (vala_code_context_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeContextClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_context_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeContext), 0, (GInstanceInitFunc) vala_code_context_init };
		vala_code_context_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCodeContext", &g_define_type_info, 0);
	}
	return vala_code_context_type_id;
}




