/* valacodecontext.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valacodecontext.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valamethod.h>
#include <vala/valanamespace.h>
#include <vala/valasourcereference.h>
#include <vala/valacodecontext.h>
#include <vala/valasourcefile.h>
#include <vala/valacodevisitor.h>
#include <vala/valasourcefilecycle.h>

struct _ValaCodeContextPrivate {
	char* _library;
	gboolean _memory_management;
	gboolean _assert;
	gboolean _checking;
	gboolean _ccode_only;
	gboolean _compile_only;
	char* _output;
	gboolean _debug;
	gint _optlevel;
	gboolean _thread;
	ValaMethod* _module_init_method;
	GeeList* source_files;
	ValaNamespace* _root;
	GeeList* cycles;
	GeeList* packages;
};
#define VALA_CODE_CONTEXT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CODE_CONTEXT, ValaCodeContextPrivate))
enum  {
	VALA_CODE_CONTEXT_DUMMY_PROPERTY,
	VALA_CODE_CONTEXT_LIBRARY,
	VALA_CODE_CONTEXT_MEMORY_MANAGEMENT,
	VALA_CODE_CONTEXT_ASSERT,
	VALA_CODE_CONTEXT_CHECKING,
	VALA_CODE_CONTEXT_CCODE_ONLY,
	VALA_CODE_CONTEXT_COMPILE_ONLY,
	VALA_CODE_CONTEXT_OUTPUT,
	VALA_CODE_CONTEXT_DEBUG,
	VALA_CODE_CONTEXT_OPTLEVEL,
	VALA_CODE_CONTEXT_THREAD,
	VALA_CODE_CONTEXT_MODULE_INIT_METHOD,
	VALA_CODE_CONTEXT_ROOT
};
static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file);
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GeeCollection* chain);
static gpointer vala_code_context_parent_class = NULL;
static void vala_code_context_dispose (GObject * obj);


/**
 * Returns a copy of the list of source files.
 *
 * @return list of source files
 */
GeeCollection* vala_code_context_get_source_files (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->source_files)));
}


/**
 * Adds the specified file to the list of source files.
 *
 * @param file a source file
 */
void vala_code_context_add_source_file (ValaCodeContext* self, ValaSourceFile* file)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	gee_collection_add (GEE_COLLECTION (self->priv->source_files), file);
}


/**
 * Returns a copy of the list of used packages.
 *
 * @return list of used packages
 */
GeeCollection* vala_code_context_get_packages (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_strdup, g_free, GEE_COLLECTION (self->priv->packages)));
}


/**
 * Returns whether the specified package is being used.
 *
 * @param pkg a package name
 * @return    true if the specified package is being used
 */
gboolean vala_code_context_has_package (ValaCodeContext* self, const char* pkg)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	return gee_collection_contains (GEE_COLLECTION (self->priv->packages), pkg);
}


/**
 * Adds the specified package to the list of used packages.
 *
 * @param pkg a package name
 */
void vala_code_context_add_package (ValaCodeContext* self, const char* pkg)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (pkg != NULL);
	gee_collection_add (GEE_COLLECTION (self->priv->packages), pkg);
}


/**
 * Visits the complete code tree file by file.
 *
 * @param visitor the visitor to be called when traversing
 */
void vala_code_context_accept (ValaCodeContext* self, ValaCodeVisitor* visitor)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_code_context_get_root (self)), visitor);
	{
		GeeList* file_collection;
		GeeIterator* file_it;
		file_collection = self->priv->source_files;
		file_it = gee_iterable_iterator (GEE_ITERABLE (file_collection));
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = gee_iterator_get (file_it);
			{
				vala_source_file_accept (file, visitor);
				(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
			}
		}
		(file_it == NULL ? NULL : (file_it = (g_object_unref (file_it), NULL)));
	}
}


/**
 * Find and resolve cycles in source file dependencies.
 */
void vala_code_context_find_header_cycles (ValaCodeContext* self)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	{
		GeeList* file_collection;
		GeeIterator* file_it;
		file_collection = self->priv->source_files;
		file_it = gee_iterable_iterator (GEE_ITERABLE (file_collection));
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = gee_iterator_get (file_it);
			{
				/* find cycles in dependencies between source files 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file)) {
					(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
					continue;
				}
				if (vala_source_file_get_mark (file) == 0) {
					GeeArrayList* __temp3;
					__temp3 = NULL;
					vala_code_context_visit (self, file, GEE_COLLECTION ((__temp3 = gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal))));
					(__temp3 == NULL ? NULL : (__temp3 = (g_object_unref (__temp3), NULL)));
				}
				(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
			}
		}
		(file_it == NULL ? NULL : (file_it = (g_object_unref (file_it), NULL)));
	}
	{
		GeeList* cycle_collection;
		GeeIterator* cycle_it;
		cycle_collection = self->priv->cycles;
		cycle_it = gee_iterable_iterator (GEE_ITERABLE (cycle_collection));
		while (gee_iterator_next (cycle_it)) {
			ValaSourceFileCycle* cycle;
			cycle = gee_iterator_get (cycle_it);
			{
				/* find one head for each cycle, it must not have any
				 * hard dependencies on other files in the cycle
				 */
				cycle->head = vala_code_context_find_cycle_head (self, gee_list_get (cycle->files, 0));
				vala_source_file_set_is_cycle_head (cycle->head, TRUE);
				(cycle == NULL ? NULL : (cycle = (g_object_unref (cycle), NULL)));
			}
		}
		(cycle_it == NULL ? NULL : (cycle_it = (g_object_unref (cycle_it), NULL)));
	}
	{
		GeeList* file2_collection;
		GeeIterator* file2_it;
		file2_collection = self->priv->source_files;
		file2_it = gee_iterable_iterator (GEE_ITERABLE (file2_collection));
		while (gee_iterator_next (file2_it)) {
			ValaSourceFile* file2;
			file2 = gee_iterator_get (file2_it);
			{
				/* connect the source files in a non-cyclic way
				 * cycle members connect to the head of their cycle
				 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file2)) {
					(file2 == NULL ? NULL : (file2 = (g_object_unref (file2), NULL)));
					continue;
				}
				{
					GeeCollection* dep_collection;
					GeeIterator* dep_it;
					dep_collection = vala_source_file_get_header_internal_dependencies (file2);
					dep_it = gee_iterable_iterator (GEE_ITERABLE (dep_collection));
					while (gee_iterator_next (dep_it)) {
						ValaSourceFile* __temp6;
						ValaSourceFile* dep;
						__temp6 = NULL;
						dep = (__temp6 = gee_iterator_get (dep_it), (__temp6 == NULL ? NULL : g_object_ref (__temp6)));
						{
							if (vala_source_file_get_cycle (file2) != NULL && vala_source_file_get_cycle (dep) == vala_source_file_get_cycle (file2)) {
								/* in the same cycle */
								if (!vala_source_file_get_is_cycle_head (file2)) {
									char* __temp4;
									/* include header of cycle head */
									__temp4 = NULL;
									vala_source_file_add_header_internal_include (file2, (__temp4 = vala_source_file_get_cinclude_filename (vala_source_file_get_cycle (file2)->head)));
									(__temp4 = (g_free (__temp4), NULL));
								}
							} else {
								char* __temp5;
								/* we can just include the headers if they are not in a cycle or not in the same cycle as the current file */
								__temp5 = NULL;
								vala_source_file_add_header_internal_include (file2, (__temp5 = vala_source_file_get_cinclude_filename (dep)));
								(__temp5 = (g_free (__temp5), NULL));
							}
							(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
						}
					}
					(dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL)));
					(dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL)));
				}
				(file2 == NULL ? NULL : (file2 = (g_object_unref (file2), NULL)));
			}
		}
		(file2_it == NULL ? NULL : (file2_it = (g_object_unref (file2_it), NULL)));
	}
}


static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (file), NULL);
	{
		GeeCollection* dep_collection;
		GeeIterator* dep_it;
		dep_collection = vala_source_file_get_header_internal_full_dependencies (file);
		dep_it = gee_iterable_iterator (GEE_ITERABLE (dep_collection));
		while (gee_iterator_next (dep_it)) {
			ValaSourceFile* __temp9;
			ValaSourceFile* dep;
			__temp9 = NULL;
			dep = (__temp9 = gee_iterator_get (dep_it), (__temp9 == NULL ? NULL : g_object_ref (__temp9)));
			{
				if (dep == file) {
					(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
					continue;
				}
				{
					GeeList* cycle_file_collection;
					GeeIterator* cycle_file_it;
					cycle_file_collection = vala_source_file_get_cycle (file)->files;
					cycle_file_it = gee_iterable_iterator (GEE_ITERABLE (cycle_file_collection));
					while (gee_iterator_next (cycle_file_it)) {
						ValaSourceFile* __temp8;
						ValaSourceFile* cycle_file;
						__temp8 = NULL;
						cycle_file = (__temp8 = gee_iterator_get (cycle_file_it), (__temp8 == NULL ? NULL : g_object_ref (__temp8)));
						{
							/* ignore file-internal dependencies */
							if (dep == cycle_file) {
								ValaSourceFile* __temp7;
								__temp7 = NULL;
								return (__temp7 = vala_code_context_find_cycle_head (self, dep), (cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL))), (cycle_file_it == NULL ? NULL : (cycle_file_it = (g_object_unref (cycle_file_it), NULL))), (dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL))), (dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL))), (dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL))), __temp7);
							}
							(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
						}
					}
					(cycle_file_it == NULL ? NULL : (cycle_file_it = (g_object_unref (cycle_file_it), NULL)));
				}
				(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
			}
		}
		(dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL)));
		(dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL)));
	}
	return file;
}


/* no hard dependencies on members of the same cycle found
 * source file suitable as cycle head
 */
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GeeCollection* chain)
{
	GeeList* l;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	g_return_if_fail (chain == NULL || GEE_IS_COLLECTION (chain));
	l = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	{
		GeeCollection* chain_file_collection;
		GeeIterator* chain_file_it;
		chain_file_collection = chain;
		chain_file_it = gee_iterable_iterator (GEE_ITERABLE (chain_file_collection));
		while (gee_iterator_next (chain_file_it)) {
			ValaSourceFile* chain_file;
			chain_file = gee_iterator_get (chain_file_it);
			{
				gee_collection_add (GEE_COLLECTION (l), chain_file);
				(chain_file == NULL ? NULL : (chain_file = (g_object_unref (chain_file), NULL)));
			}
		}
		(chain_file_it == NULL ? NULL : (chain_file_it = (g_object_unref (chain_file_it), NULL)));
	}
	gee_collection_add (GEE_COLLECTION (l), file);
	/* mark file as currently being visited */
	vala_source_file_set_mark (file, 1);
	{
		GeeCollection* dep_collection;
		GeeIterator* dep_it;
		dep_collection = vala_source_file_get_header_internal_dependencies (file);
		dep_it = gee_iterable_iterator (GEE_ITERABLE (dep_collection));
		while (gee_iterator_next (dep_it)) {
			ValaSourceFile* __temp13;
			ValaSourceFile* dep;
			__temp13 = NULL;
			dep = (__temp13 = gee_iterator_get (dep_it), (__temp13 == NULL ? NULL : g_object_ref (__temp13)));
			{
				if (file == dep) {
					(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
					continue;
				}
				if (vala_source_file_get_mark (dep) == 1) {
					ValaSourceFileCycle* cycle;
					gboolean cycle_start_found;
					/* found cycle */
					cycle = g_object_new (VALA_TYPE_SOURCE_FILE_CYCLE, NULL);
					gee_collection_add (GEE_COLLECTION (self->priv->cycles), cycle);
					cycle_start_found = FALSE;
					{
						GeeList* cycle_file_collection;
						GeeIterator* cycle_file_it;
						cycle_file_collection = l;
						cycle_file_it = gee_iterable_iterator (GEE_ITERABLE (cycle_file_collection));
						while (gee_iterator_next (cycle_file_it)) {
							ValaSourceFile* cycle_file;
							cycle_file = gee_iterator_get (cycle_file_it);
							{
								ValaSourceFileCycle* __temp11;
								ValaSourceFileCycle* ref_cycle_file_cycle;
								__temp11 = NULL;
								ref_cycle_file_cycle = (__temp11 = vala_source_file_get_cycle (cycle_file), (__temp11 == NULL ? NULL : g_object_ref (__temp11)));
								if (!cycle_start_found) {
									if (cycle_file == dep) {
										cycle_start_found = TRUE;
									}
								}
								if (!cycle_start_found) {
									(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
									(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
									continue;
								}
								if (vala_source_file_get_cycle (cycle_file) != NULL) {
									/* file already in a cycle */
									if (vala_source_file_get_cycle (cycle_file) == cycle) {
										(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
										(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
										continue;
									}
									/* file is in the same cycle, nothing to do 
									 file is in an other cycle, merge the two cycles */
									gee_collection_remove (GEE_COLLECTION (self->priv->cycles), vala_source_file_get_cycle (cycle_file));
									{
										GeeList* inner_cycle_file_collection;
										GeeIterator* inner_cycle_file_it;
										inner_cycle_file_collection = vala_source_file_get_cycle (cycle_file)->files;
										inner_cycle_file_it = gee_iterable_iterator (GEE_ITERABLE (inner_cycle_file_collection));
										while (gee_iterator_next (inner_cycle_file_it)) {
											ValaSourceFile* __temp12;
											ValaSourceFile* inner_cycle_file;
											__temp12 = NULL;
											inner_cycle_file = (__temp12 = gee_iterator_get (inner_cycle_file_it), (__temp12 == NULL ? NULL : g_object_ref (__temp12)));
											{
												if (vala_source_file_get_cycle (inner_cycle_file) != cycle) {
													/* file in inner cycle not yet added to outer cycle */
													gee_collection_add (GEE_COLLECTION (cycle->files), inner_cycle_file);
													vala_source_file_set_cycle (inner_cycle_file, cycle);
												}
												(inner_cycle_file == NULL ? NULL : (inner_cycle_file = (g_object_unref (inner_cycle_file), NULL)));
											}
										}
										(inner_cycle_file_it == NULL ? NULL : (inner_cycle_file_it = (g_object_unref (inner_cycle_file_it), NULL)));
									}
								} else {
									gee_collection_add (GEE_COLLECTION (cycle->files), cycle_file);
									vala_source_file_set_cycle (cycle_file, cycle);
								}
								(cycle_file == NULL ? NULL : (cycle_file = (g_object_unref (cycle_file), NULL)));
								(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
							}
						}
						(cycle_file_it == NULL ? NULL : (cycle_file_it = (g_object_unref (cycle_file_it), NULL)));
					}
					(cycle == NULL ? NULL : (cycle = (g_object_unref (cycle), NULL)));
				} else {
					if (vala_source_file_get_mark (dep) == 0) {
						/* found not yet visited file */
						vala_code_context_visit (self, dep, GEE_COLLECTION (l));
					}
				}
				(dep == NULL ? NULL : (dep = (g_object_unref (dep), NULL)));
			}
		}
		(dep_collection == NULL ? NULL : (dep_collection = (g_object_unref (dep_collection), NULL)));
		(dep_it == NULL ? NULL : (dep_it = (g_object_unref (dep_it), NULL)));
	}
	/* mark file as successfully visited */
	vala_source_file_set_mark (file, 2);
	(l == NULL ? NULL : (l = (g_object_unref (l), NULL)));
}


char* vala_code_context_get_library (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_library;
}


void vala_code_context_set_library (ValaCodeContext* self, const char* value)
{
	char* __temp16;
	const char* __temp15;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	__temp16 = NULL;
	__temp15 = NULL;
	self->priv->_library = (__temp16 = (__temp15 = value, (__temp15 == NULL ? NULL : g_strdup (__temp15))), (self->priv->_library = (g_free (self->priv->_library), NULL)), __temp16);
}


gboolean vala_code_context_get_memory_management (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_memory_management;
}


void vala_code_context_set_memory_management (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_memory_management = value;
}


gboolean vala_code_context_get_assert (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_assert;
}


void vala_code_context_set_assert (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_assert = value;
}


gboolean vala_code_context_get_checking (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_checking;
}


void vala_code_context_set_checking (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_checking = value;
}


gboolean vala_code_context_get_ccode_only (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_ccode_only;
}


void vala_code_context_set_ccode_only (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_ccode_only = value;
}


gboolean vala_code_context_get_compile_only (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_compile_only;
}


void vala_code_context_set_compile_only (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_compile_only = value;
}


char* vala_code_context_get_output (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_output;
}


void vala_code_context_set_output (ValaCodeContext* self, const char* value)
{
	char* __temp24;
	const char* __temp23;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	__temp24 = NULL;
	__temp23 = NULL;
	self->priv->_output = (__temp24 = (__temp23 = value, (__temp23 == NULL ? NULL : g_strdup (__temp23))), (self->priv->_output = (g_free (self->priv->_output), NULL)), __temp24);
}


gboolean vala_code_context_get_debug (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_debug;
}


void vala_code_context_set_debug (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_debug = value;
}


gint vala_code_context_get_optlevel (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), 0);
	return self->priv->_optlevel;
}


void vala_code_context_set_optlevel (ValaCodeContext* self, gint value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_optlevel = value;
}


gboolean vala_code_context_get_thread (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_thread;
}


void vala_code_context_set_thread (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_thread = value;
}


ValaMethod* vala_code_context_get_module_init_method (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_module_init_method;
}


void vala_code_context_set_module_init_method (ValaCodeContext* self, ValaMethod* value)
{
	ValaMethod* __temp30;
	ValaMethod* __temp29;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	__temp30 = NULL;
	__temp29 = NULL;
	self->priv->_module_init_method = (__temp30 = (__temp29 = value, (__temp29 == NULL ? NULL : g_object_ref (__temp29))), (self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL))), __temp30);
}


ValaNamespace* vala_code_context_get_root (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_root;
}


static void vala_code_context_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		g_value_set_string (value, vala_code_context_get_library (self));
		break;
		case VALA_CODE_CONTEXT_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_code_context_get_memory_management (self));
		break;
		case VALA_CODE_CONTEXT_ASSERT:
		g_value_set_boolean (value, vala_code_context_get_assert (self));
		break;
		case VALA_CODE_CONTEXT_CHECKING:
		g_value_set_boolean (value, vala_code_context_get_checking (self));
		break;
		case VALA_CODE_CONTEXT_CCODE_ONLY:
		g_value_set_boolean (value, vala_code_context_get_ccode_only (self));
		break;
		case VALA_CODE_CONTEXT_COMPILE_ONLY:
		g_value_set_boolean (value, vala_code_context_get_compile_only (self));
		break;
		case VALA_CODE_CONTEXT_OUTPUT:
		g_value_set_string (value, vala_code_context_get_output (self));
		break;
		case VALA_CODE_CONTEXT_DEBUG:
		g_value_set_boolean (value, vala_code_context_get_debug (self));
		break;
		case VALA_CODE_CONTEXT_OPTLEVEL:
		g_value_set_int (value, vala_code_context_get_optlevel (self));
		break;
		case VALA_CODE_CONTEXT_THREAD:
		g_value_set_boolean (value, vala_code_context_get_thread (self));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		g_value_set_object (value, vala_code_context_get_module_init_method (self));
		break;
		case VALA_CODE_CONTEXT_ROOT:
		g_value_set_object (value, vala_code_context_get_root (self));
		break;
	}
}


static void vala_code_context_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		vala_code_context_set_library (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_MEMORY_MANAGEMENT:
		vala_code_context_set_memory_management (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_ASSERT:
		vala_code_context_set_assert (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_CHECKING:
		vala_code_context_set_checking (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_CCODE_ONLY:
		vala_code_context_set_ccode_only (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_COMPILE_ONLY:
		vala_code_context_set_compile_only (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_OUTPUT:
		vala_code_context_set_output (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_DEBUG:
		vala_code_context_set_debug (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_OPTLEVEL:
		vala_code_context_set_optlevel (self, g_value_get_int (value));
		break;
		case VALA_CODE_CONTEXT_THREAD:
		vala_code_context_set_thread (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		vala_code_context_set_module_init_method (self, g_value_get_object (value));
		break;
	}
}


static void vala_code_context_class_init (ValaCodeContextClass * klass)
{
	vala_code_context_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCodeContextPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_code_context_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_code_context_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_code_context_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_LIBRARY, g_param_spec_string ("library", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_ASSERT, g_param_spec_boolean ("assert", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_CHECKING, g_param_spec_boolean ("checking", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_CCODE_ONLY, g_param_spec_boolean ("ccode-only", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_COMPILE_ONLY, g_param_spec_boolean ("compile-only", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_OUTPUT, g_param_spec_string ("output", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_DEBUG, g_param_spec_boolean ("debug", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_OPTLEVEL, g_param_spec_int ("optlevel", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_THREAD, g_param_spec_boolean ("thread", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_MODULE_INIT_METHOD, g_param_spec_object ("module-init-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_ROOT, g_param_spec_object ("root", "foo", "bar", VALA_TYPE_NAMESPACE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
}


static void vala_code_context_init (ValaCodeContext * self)
{
	self->priv = VALA_CODE_CONTEXT_GET_PRIVATE (self);
	self->priv->source_files = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->_root = vala_namespace_new (NULL, NULL);
	self->priv->cycles = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->packages = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_str_equal));
}


static void vala_code_context_dispose (GObject * obj)
{
	ValaCodeContext * self;
	ValaCodeContextClass * klass;
	GObjectClass * parent_class;
	self = VALA_CODE_CONTEXT (obj);
	(self->priv->_library = (g_free (self->priv->_library), NULL));
	(self->priv->_output = (g_free (self->priv->_output), NULL));
	(self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL)));
	(self->priv->source_files == NULL ? NULL : (self->priv->source_files = (g_object_unref (self->priv->source_files), NULL)));
	(self->priv->_root == NULL ? NULL : (self->priv->_root = (g_object_unref (self->priv->_root), NULL)));
	(self->priv->cycles == NULL ? NULL : (self->priv->cycles = (g_object_unref (self->priv->cycles), NULL)));
	(self->priv->packages == NULL ? NULL : (self->priv->packages = (g_object_unref (self->priv->packages), NULL)));
	klass = VALA_CODE_CONTEXT_CLASS (g_type_class_peek (VALA_TYPE_CODE_CONTEXT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_code_context_get_type (void)
{
	static GType vala_code_context_type_id = 0;
	if (G_UNLIKELY (vala_code_context_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeContextClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_context_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeContext), 0, (GInstanceInitFunc) vala_code_context_init };
		vala_code_context_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCodeContext", &g_define_type_info, 0);
	}
	return vala_code_context_type_id;
}




