/* valainitializerlist.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valainitializerlist.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valainitializerlist.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaInitializerListPrivate {
	GeeList* initializers;
};
#define VALA_INITIALIZER_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListPrivate))
enum  {
	VALA_INITIALIZER_LIST_DUMMY_PROPERTY
};
static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_initializer_list_parent_class = NULL;
static void vala_initializer_list_dispose (GObject * obj);


/**
 * Appends the specified expression to this initializer list.
 *
 * @param expr an expression
 */
void vala_initializer_list_append (ValaInitializerList* self, ValaExpression* expr)
{
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (self));
	g_return_if_fail (VALA_IS_EXPRESSION (expr));
	gee_collection_add (GEE_COLLECTION (self->priv->initializers), expr);
}


/**
 * Returns a copy of the expression list.
 *
 * @return expression list
 */
GeeCollection* vala_initializer_list_get_initializers (ValaInitializerList* self)
{
	g_return_val_if_fail (VALA_IS_INITIALIZER_LIST (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->initializers)));
}


/**
 * Creates a new initializer list.
 *
 * @param source_reference reference to source code
 * @return                 newly created initializer list
 */
ValaInitializerList* vala_initializer_list_new (ValaSourceReference* source_reference)
{
	ValaInitializerList * self;
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_INITIALIZER_LIST, 0, NULL);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaInitializerList * self;
	self = ((ValaInitializerList*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_begin_initializer_list (visitor, self);
	{
		GeeList* expr_collection;
		GeeIterator* expr_it;
		expr_collection = self->priv->initializers;
		expr_it = gee_iterable_iterator (GEE_ITERABLE (expr_collection));
		while (gee_iterator_next (expr_it)) {
			ValaExpression* expr;
			expr = gee_iterator_get (expr_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (expr), visitor);
				(expr == NULL ? NULL : (expr = (g_object_unref (expr), NULL)));
			}
		}
		(expr_it == NULL ? NULL : (expr_it = (g_object_unref (expr_it), NULL)));
	}
	vala_code_visitor_visit_end_initializer_list (visitor, self);
}


static void vala_initializer_list_class_init (ValaInitializerListClass * klass)
{
	vala_initializer_list_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInitializerListPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_initializer_list_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_initializer_list_real_accept;
}


static void vala_initializer_list_init (ValaInitializerList * self)
{
	self->priv = VALA_INITIALIZER_LIST_GET_PRIVATE (self);
	self->priv->initializers = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_initializer_list_dispose (GObject * obj)
{
	ValaInitializerList * self;
	ValaInitializerListClass * klass;
	GObjectClass * parent_class;
	self = VALA_INITIALIZER_LIST (obj);
	(self->priv->initializers == NULL ? NULL : (self->priv->initializers = (g_object_unref (self->priv->initializers), NULL)));
	klass = VALA_INITIALIZER_LIST_CLASS (g_type_class_peek (VALA_TYPE_INITIALIZER_LIST));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_initializer_list_get_type (void)
{
	static GType vala_initializer_list_type_id = 0;
	if (G_UNLIKELY (vala_initializer_list_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInitializerListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_initializer_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInitializerList), 0, (GInstanceInitFunc) vala_initializer_list_init };
		vala_initializer_list_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaInitializerList", &g_define_type_info, 0);
	}
	return vala_initializer_list_type_id;
}




