/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * soup-httpd.c: Soup HTTPD server.
 *
 * Authors:
 *      Alex Graveley (alex@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <ctype.h>
#include <errno.h>
#include <glib.h>
#include <gmodule.h>
#include <popt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include <libsoup/soup-headers.h>
#include <libsoup/soup-private.h>
#include <libsoup/soup-server.h>
#include <libsoup/soup-ssl.h>
#include <libsoup/soup-transfer.h>

static int verbose = 0;
static GSList *modules = NULL;

typedef struct {
	const gchar       *filename;
	GModule           *module;
	SoupServerHandler *handler_list;
	GSList            *added_handlers;
	time_t             mtime;
} SoupHttpdModule;

typedef void (*ModuleInitFn) (void);

typedef struct {
	int         port;
	gboolean    secure;
	SoupSocket *sock;
} SoupHttpdServerSock;

SoupHttpdServerSock server = { SOUP_SERVER_ANY_PORT, FALSE, NULL };
SoupHttpdServerSock ssl_server = { SOUP_SERVER_ANY_PORT, TRUE, NULL };

static inline void
destroy_message (SoupMessage *req)
{
	soup_socket_unref ((SoupSocket *) req->priv->conn);
	req->priv->conn = NULL;
	soup_message_free (req);
}

static void 
soup_httpd_error_cb (gboolean     body_started,
		     SoupMessage *req)
{
	destroy_message (req);
}

static void
soup_httpd_write_done_cb (SoupMessage *req)
{
	req->priv->write_tag = 0;
	destroy_message (req);
}

static SoupTransferDone
soup_httpd_read_headers_cb (const GString *headers,
			    guint         *content_len,
			    SoupMessage   *msg)
{
	SoupContext *ctx;
	gchar *req_host = NULL, *req_path = NULL, *req_method, *url;
	gchar *connection, *length, *enc;

	msg->request_headers = g_hash_table_new (soup_str_case_hash, 
						 soup_str_case_equal);

	if (!soup_headers_parse_request (headers->str, 
					 headers->len, 
					 msg->request_headers, 
					 &req_method, 
					 &req_path,
					 &msg->priv->http_version))
		goto THROW_MALFORMED_HEADER;

	/* Handle connection persistence */
	connection = g_hash_table_lookup (msg->request_headers, "Connection");
	/*
	if (connection && g_strcasecmp (connection, "close") == 0)
		soup_connection_set_keep_alive (req->priv->conn, FALSE);
	*/

	/* Handle Content-Length or Chunked encoding */
	length = g_hash_table_lookup (msg->request_headers, "Content-Length");
	enc = g_hash_table_lookup (msg->request_headers, "Transfer-Encoding");

	if (length) {
		*content_len = atoi (length);
		if (*content_len < 0) 
			goto THROW_MALFORMED_HEADER;
	} else if (enc) {
		if (g_strcasecmp (enc, "chunked") == 0)
			*content_len = SOUP_TRANSFER_CHUNKED;
		else {
			g_warning ("Unknown encoding type in HTTP request.");
			goto THROW_MALFORMED_HEADER;
		}
	}

	/* Check for POST or M-POST request HTTP method */
	if ((g_strcasecmp (req_method, "POST") != 0 && 
	     g_strcasecmp (req_method, "M-POST") != 0))
		goto THROW_MALFORMED_HEADER;

	/* Generate correct context for request */
	req_host = g_hash_table_lookup (msg->request_headers, "Host");
	if (req_host) 
		url = g_strconcat ("http://", req_host, req_path, NULL);
	else 
		url = g_strdup (req_path);

	ctx = soup_context_get (url);
	g_free (url);

	/* No Host, no AbsoluteUri */
	if (!ctx) {
		url = g_strconcat ("http://localhost/", req_path, NULL);
		ctx = soup_context_get (url);
		g_free (url);
	}

	if (!ctx) goto THROW_MALFORMED_HEADER;

	soup_context_unref (msg->context);
	msg->context = ctx;

	g_free (req_method);
	g_free (req_path);

	return SOUP_TRANSFER_CONTINUE;

 THROW_MALFORMED_HEADER:
	g_free (req_method);
	g_free (req_path);

	destroy_message (msg);

	return SOUP_TRANSFER_END;
}

static void
soup_httpd_write_header (gchar *key, gchar *val, SoupMessage *msg)
{
	g_string_sprintfa (msg->priv->req_header, "%s: %s\r\n", key, val);
}

static GString *
soup_httpd_get_response_header (SoupMessage *req)
{
	GString *ret = g_string_new (NULL);

	g_string_sprintfa (ret, 
			   "HTTP/1.1 %d %s\r\n", 
			   req->response_code, 
			   req->response_phrase);

	g_string_sprintfa (ret, 
			   "Content-Length: %d\r\n",  
			   req->response.length);

	if (req->response_headers) 
		g_hash_table_foreach (req->response_headers, 
				      (GHFunc) soup_httpd_write_header,
				      req);

	g_string_append (ret, "\r\n");

	return ret;
}

static inline void
set_response_error (SoupMessage    *req,
		    gint            code,
		    gchar          *phrase,
		    gchar          *body)
{
	req->response_code = code;
	req->response_phrase = g_strdup (phrase);
	req->response.owner = SOUP_BUFFER_STATIC;
	req->response.body = body;
	req->response.length = strlen (req->response.body);
}

static void
soup_httpd_read_done_cb (const SoupDataBuffer *data,
			 SoupMessage          *req)
{
	SoupServerHandler *hand;
	const gchar *action;
	GIOChannel *channel;

	req->response.owner = data->owner;
	req->response.length = data->length;
	req->response.body = data->body;

	req->status = SOUP_STATUS_FINISHED;

	action = soup_message_get_request_header (req, "SOAPAction");
	if (!action) {
		g_print ("No SOAPAction found in request.\n");
		set_response_error (
			req, 
			403, 
			"Missing SOAPAction", 
			"The required SOAPAction header was not supplied.");
		goto START_WRITE;
	}

	g_print ("Handling request for \"%s\"... ", action);

	hand = soup_server_get_handler (action);
	if (!hand) {
		g_print ("ERROR.\n\tNo Handler found.\n");
		set_response_error (req, 
				    403, 
				    "Invalid SOAPAction",
				    "The SOAPAction specified is unknown.");
		goto START_WRITE;
	}

	req->response_code = 200;
	req->response_phrase = g_strdup ("OK");

	soup_message_set_response_header (req, 
					  "Content-Type",
					  "text/xml; charset=\"utf-8\"");
	soup_message_set_response_header (req, 
					  "Connection", 
					  "Keep-Alive");

	/* Call method handler */
	if (hand->cb) (*hand->cb) (req, hand->user_data);

	g_print ("OK.\n");

 START_WRITE:
	channel = soup_socket_get_iochannel ((SoupSocket *) req->priv->conn);

	req->priv->req_header = soup_httpd_get_response_header (req);

	req->priv->read_tag = 0;

	req->priv->write_tag = 
		soup_transfer_write (channel,
				     req->priv->req_header,
				     &req->response,
				     NULL,
				     (SoupWriteDoneFn) soup_httpd_write_done_cb,
				     (SoupWriteErrorFn) soup_httpd_error_cb,
				     req);
	return;
}

static void 
soup_httpd_conn_accept (GIOChannel          *chan,
			GIOCondition         condition, 
			SoupHttpdServerSock *serv)
{
	SoupSocket *sock;
	SoupContext *ctx;
	SoupMessage *msg;
	SoupUri uri = { 
		serv->secure ? SOUP_PROTOCOL_HTTPS : SOUP_PROTOCOL_HTTP,
		NULL, 
		NULL, 
		NULL, 
		"localhost", 
		serv->port, 
		"/",
		NULL,
		NULL 
	};

	sock = soup_socket_server_accept (serv->sock);
	if (!sock) return;
	
	/* 
	 * Create a fake context until the request is read 
	 * and we can generate a valid one.
	 */
	ctx = soup_context_from_uri (&uri);
	msg = soup_message_new (ctx, NULL);

	/* FIXME: Use a SoupConnection here */
	msg->priv->conn = (SoupConnection *) sock;

	chan = soup_socket_get_iochannel (sock);
	if (serv->secure) chan = soup_ssl_get_iochannel (chan);

	msg->priv->read_tag = 
		soup_transfer_read (
			chan,
			FALSE,
			(SoupReadHeadersDoneFn) soup_httpd_read_headers_cb,
			NULL,
			(SoupReadDoneFn) soup_httpd_read_done_cb,
			(SoupReadErrorFn) soup_httpd_error_cb,
			msg);
}

static void
soup_httpd_print_handler (SoupServerHandler *handler, gpointer notused)
{
	if (!handler || !handler->methodname) return;
	g_print ("\n\tRegistering \"%s\"\n", handler->methodname);
}

static void
soup_httpd_load_module (const gchar *file)
{
	GModule *gmod;
	SoupHttpdModule *mod;
	SoupServerHandler *handlers = NULL;
	ModuleInitFn *init = NULL;
	struct stat mod_stat;
	
	mod = g_new0 (SoupHttpdModule, 1);
	mod->filename = file;

	if (stat (file, &mod_stat) == 0) 
		mod->mtime = mod_stat.st_mtime;
	else { 
		g_print ("ERROR.\n\tFile Not Found.\n");
		goto ADD_MODULE;
	}

	gmod = g_module_open (file, 0);
	if (!gmod) {
		g_print ("ERROR.\n\tUnable to load module \"%s\":\n\t%s\n", 
			 file,
			 g_module_error ());
		goto ADD_MODULE;
	}

	if (g_module_symbol (gmod, 
			     "soup_server_handlers", 
			     (gpointer *) &handlers)) {
		mod->handler_list = handlers;

		while (handlers && handlers->methodname) {
			soup_server_handlers = 
				g_slist_prepend (soup_server_handlers, 
						 handlers);
			soup_httpd_print_handler (handlers, NULL);
			handlers++;
		}
	}

	if (g_module_symbol (gmod, "soup_server_init", (gpointer *) init)) {
		GSList *old_handlers = soup_server_handlers;
		soup_server_handlers = NULL;

		(*init) ();

		mod->added_handlers = soup_server_handlers;

		g_slist_foreach (soup_server_handlers, 
				 (GFunc) soup_httpd_print_handler, 
				 NULL);

		old_handlers = g_slist_concat (old_handlers, 
					       soup_server_handlers);
		soup_server_handlers = old_handlers;
	}

	/* 
	 * Even if nothing is found to register, leave the module around so we
	 * can monitor the file and reload it when it changes. 
	 */
	if (!handlers && !init) {
		g_print ("ERROR.\n");
		g_print ("Module \"%s\" appers to have neither an "
			 "initialization function nor a URI handler table\n", 
			 file);
	} else if (!mod->handler_list && !mod->added_handlers)
		g_print ("NO HANDLERS FOUND.\n");
	else 
		g_print ("OK.\n");		

 ADD_MODULE:
	modules = g_slist_prepend (modules, mod);
}

static gboolean 
soup_httpd_check_modules (gpointer unused)
{
	GSList *iter = modules;

	while (iter) {
		SoupHttpdModule *mod = iter->data;
		gchar *mod_name;
		struct stat mod_stat;

		mod_name = g_strdup (mod->filename);
		mod_stat.st_mtime = 0;

		stat (mod_name, &mod_stat);

		if (mod_stat.st_mtime && mod_stat.st_mtime > mod->mtime) {
			SoupServerHandler *handlers = mod->handler_list;
			GSList *hand_iter = mod->added_handlers;

			while (handlers && handlers->methodname) {
				soup_server_handlers = 
					g_slist_remove (soup_server_handlers,
							handlers);
				handlers++;
			}

			while (hand_iter) {
				soup_server_handlers = 
					g_slist_remove (soup_server_handlers,
							hand_iter->data);
				hand_iter = hand_iter->next;
			}

			g_slist_free (mod->added_handlers);
			g_module_close (mod->module);

			modules = g_slist_remove (modules, mod);
			g_free (mod);

			g_print ("Reloading module \"%s\"... ", mod_name);
			soup_httpd_load_module (mod_name);
		}

		g_free (mod_name);
		iter = iter->next;
	}

	return TRUE;
}

static struct poptOption options[] = {
        { "port", 
	  'p', 
	  POPT_ARG_INT, 
	  &server.port, 
	  0, 
	  "Server Port", 
	  "PORT" },
        { "ssl-port", 
	  '\0', 
	  POPT_ARG_INT, 
	  &ssl_server.port, 
	  0, 
	  "Server secure SSL Port", 
	  "PORT" },
        { "verbose", 
	  'v', 
	  POPT_ARG_NONE, 
	  &verbose, 
	  0, 
	  "Show verbose output", 
	  NULL },
	POPT_AUTOHELP
        { NULL, 0, 0, NULL, 0 }
};

int
main (int argc, const char **argv)
{
	poptContext ctx;
	int nextopt;
	const gchar **mod_names;
	GMainLoop *loop;

	if (!g_module_supported ()) {
		g_print ("Dynamically loadable modules not supported by the "
			 "version of GLIB installed on this system.\n");
		exit (1);
	}

	ctx = poptGetContext(argv [0], argc, argv, options, 0);
	poptSetOtherOptionHelp(ctx, "MODULE...");

        while ((nextopt = poptGetNextOpt (ctx)) > 0)
                /* do nothing */ ;

	if (nextopt != -1) {
		g_print("Error on option %s: %s.\nRun '%s --help' to see a "
			"full list of command line options.\n",
			poptBadOption (ctx, 0),
			poptStrerror (nextopt),
			argv [0]);
		exit (1);
	}

	mod_names = poptGetArgs (ctx);

	if (!mod_names) {
		g_print ("No soup server modules specified.\n");
		poptPrintUsage (ctx, stderr, 0);
		exit (1);
	}

	server.sock = soup_socket_server_new (server.port);
	if (!server.sock) {
		g_print ("Unable to bind to server port %d\n", server.port);
		exit (1);
	}

	ssl_server.sock = soup_socket_server_new (ssl_server.port);
	if (!ssl_server.sock) {
		g_print ("Unable to bind to SSL server port %d\n", 
			 ssl_server.port);
		exit (1);
	}

	server.port = soup_socket_get_port (server.sock);
	ssl_server.port = soup_socket_get_port (ssl_server.sock);

	g_io_add_watch (soup_socket_get_iochannel (server.sock),
			G_IO_IN,
			(GIOFunc) soup_httpd_conn_accept, 
			&server);

	g_io_add_watch (soup_socket_get_iochannel (ssl_server.sock),
			G_IO_IN,
			(GIOFunc) soup_httpd_conn_accept, 
			&ssl_server);

	g_timeout_add (1000, (GSourceFunc) soup_httpd_check_modules, NULL);

	while (*mod_names) {
		g_print ("Loading module \"%s\"... ", *mod_names);
		soup_httpd_load_module (*mod_names);
		mod_names++;
	}

	poptFreeContext (ctx);

	g_print ("\nStarting Server on port %d\n", server.port);
	g_print ("Starting SSL Server on port %d\n", ssl_server.port);
	g_print ("\nWaiting for requests...\n");

	loop = g_main_new (TRUE);
	g_main_run (loop);
}


/*
 * Dummy Apache method implementations, to allow for Soup apache modules to be
 * loaded by soup-httpd. These are never called.
 */

void ap_construct_url           (void);
void ap_table_get               (void);
void ap_table_do                (void);
void ap_setup_client_block      (void);
void ap_should_client_block     (void);
void ap_palloc                  (void);
void ap_hard_timeout            (void);
void ap_get_client_block        (void);
void ap_reset_timeout           (void);
void ap_kill_timeout            (void);
void ap_table_set               (void);
void ap_log_error               (void);
void ap_auth_type               (void);
void ap_get_basic_auth_pw       (void);
void ap_note_basic_auth_failure (void);
void ap_psprintf                (void);
void ap_send_http_header        (void);
void ap_rwrite                  (void);

void
ap_construct_url (void)
{
}

void
ap_table_get (void)
{
}

void
ap_table_do (void)
{
}

void
ap_setup_client_block (void)
{
}

void
ap_should_client_block (void)
{
}

void
ap_palloc (void)
{
}

void
ap_hard_timeout (void)
{
}

void
ap_get_client_block (void)
{
}

void
ap_reset_timeout (void)
{
}

void
ap_kill_timeout (void)
{
}

void
ap_table_set (void)
{
}

void
ap_log_error (void)
{
}

void
ap_auth_type (void)
{
}

void
ap_get_basic_auth_pw (void)
{
}

void
ap_note_basic_auth_failure (void)
{
}

void
ap_psprintf (void)
{
}

void
ap_send_http_header (void)
{
}

void
ap_rwrite (void)
{
}
