/*
 * wsdl-typecodes.c: Type code definitions for WSDL types
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <glib.h>
#include <config.h>
#include <string.h>

#include "wsdl-typecodes.h"
#include "wsdl-schema.h"

const wsdl_typecode WSDL_TC_glib_null_struct={WSDL_TK_GLIB_NULL,
					      "null", "glib", GLIBNS,
					      FALSE, 0, NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_void_struct={WSDL_TK_GLIB_VOID,
					      "void", "glib", GLIBNS,
					      FALSE, 0, NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_boolean_struct={WSDL_TK_GLIB_BOOLEAN,
						 "boolean", "glib",
						 GLIBNS, FALSE, 0,
						 NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_char_struct={WSDL_TK_GLIB_CHAR,
					      "char", "glib", GLIBNS,
					      FALSE, 0, NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_uchar_struct={WSDL_TK_GLIB_UCHAR,
					       "uchar", "glib",
					       GLIBNS, FALSE, 0, NULL,
					       NULL, NULL};

const wsdl_typecode WSDL_TC_glib_int_struct={WSDL_TK_GLIB_INT, "int",
					     "glib", GLIBNS, FALSE, 0,
					     NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_uint_struct={WSDL_TK_GLIB_UINT,
					      "uint", "glib", GLIBNS,
					      FALSE, 0, NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_short_struct={WSDL_TK_GLIB_SHORT,
					       "short", "glib",
					       GLIBNS, FALSE, 0, NULL,
					       NULL, NULL};

const wsdl_typecode WSDL_TC_glib_ushort_struct={WSDL_TK_GLIB_USHORT,
						"ushort", "glib",
						GLIBNS, FALSE, 0,
						NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_long_struct={WSDL_TK_GLIB_LONG,
					      "long", "glib", GLIBNS,
					      FALSE, 0, NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_ulong_struct={WSDL_TK_GLIB_ULONG,
					       "ulong", "glib",
					       GLIBNS, FALSE, 0, NULL,
					       NULL, NULL};

const wsdl_typecode WSDL_TC_glib_int8_struct={WSDL_TK_GLIB_INT8,
					      "int8", "glib", GLIBNS,
					      FALSE, 0, NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_uint8_struct={WSDL_TK_GLIB_UINT8,
					       "uint8", "glib",
					       GLIBNS, FALSE, 0, NULL,
					       NULL, NULL};

const wsdl_typecode WSDL_TC_glib_int16_struct={WSDL_TK_GLIB_INT16,
					       "int16", "glib",
					       GLIBNS, FALSE, 0, NULL,
					       NULL, NULL};

const wsdl_typecode WSDL_TC_glib_uint16_struct={WSDL_TK_GLIB_UINT16,
						"uint16", "glib",
						GLIBNS, FALSE, 0,
						NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_int32_struct={WSDL_TK_GLIB_INT32,
					       "int32", "glib",
					       GLIBNS, FALSE, 0, NULL,
					       NULL, NULL};

const wsdl_typecode WSDL_TC_glib_uint32_struct={WSDL_TK_GLIB_UINT32,
						"uint32", "glib",
						GLIBNS, FALSE, 0,
						NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_float_struct={WSDL_TK_GLIB_FLOAT,
					       "float", "glib",
					       GLIBNS, FALSE, 0, NULL,
					       NULL, NULL};

const wsdl_typecode WSDL_TC_glib_double_struct={WSDL_TK_GLIB_DOUBLE,
						"double", "glib",
						GLIBNS, FALSE, 0,
						NULL, NULL, NULL};

const wsdl_typecode WSDL_TC_glib_string_struct={WSDL_TK_GLIB_STRING,
						"string", "glib",
						GLIBNS, FALSE, 0,
						NULL, NULL, NULL};


/* Keep this synchronised with wsdl_typecode_kind_t in
 * wsdl-typecodes.h
 */
static const guchar *wsdl_typecode_kind_names[]={
	"WSDL_TK_GLIB_NULL",
	"WSDL_TK_GLIB_VOID",
	"WSDL_TK_GLIB_BOOLEAN",
	"WSDL_TK_GLIB_CHAR",
	"WSDL_TK_GLIB_UCHAR",
	"WSDL_TK_GLIB_INT",
	"WSDL_TK_GLIB_UINT",
	"WSDL_TK_GLIB_SHORT",
	"WSDL_TK_GLIB_USHORT",
	"WSDL_TK_GLIB_LONG",
	"WSDL_TK_GLIB_ULONG",
	"WSDL_TK_GLIB_INT8",
	"WSDL_TK_GLIB_UINT8",
	"WSDL_TK_GLIB_INT16",
	"WSDL_TK_GLIB_UINT16",
	"WSDL_TK_GLIB_INT32",
	"WSDL_TK_GLIB_UINT32",
	"WSDL_TK_GLIB_FLOAT",
	"WSDL_TK_GLIB_DOUBLE",
	"WSDL_TK_GLIB_STRING",
	"WSDL_TK_GLIB_ELEMENT",
	"WSDL_TK_GLIB_STRUCT",
	"WSDL_TK_GLIB_LIST",
	"Not a valid WSDL_TK_GLIB type"
};

static GSList *wsdl_typecodes=NULL;

const guchar *wsdl_typecode_kind_name(wsdl_typecode_kind_t kind)
{
	g_assert(kind<WSDL_TK_GLIB_MAX);
	
	return(wsdl_typecode_kind_names[kind]);
}

wsdl_typecode_kind_t wsdl_typecode_kind(const wsdl_typecode *const tc)
{
	g_assert(tc!=NULL);
	g_assert(tc->kind<WSDL_TK_GLIB_MAX);
	
	return(tc->kind);
}

gboolean wsdl_typecode_is_simple(const wsdl_typecode *const tc)
{
	g_assert(tc!=NULL);
	g_assert(tc->kind<WSDL_TK_GLIB_MAX);
	
	if(tc->kind<WSDL_TK_GLIB_ELEMENT) {
		return(TRUE);
	} else if (tc->kind==WSDL_TK_GLIB_ELEMENT) {
		return(wsdl_typecode_is_simple(tc->subtypes[0]));
	} else {
		return(FALSE);
	}
}

guint wsdl_typecode_member_count(const wsdl_typecode *const tc)
{
	g_assert(tc!=NULL);

	if(tc->kind!=WSDL_TK_GLIB_STRUCT) {
		return(0);
	}
	
	return(tc->sub_parts);
}

const guchar *wsdl_typecode_member_name(const wsdl_typecode *const tc,
					guint member)
{
	g_assert(tc!=NULL);

	if(tc->kind!=WSDL_TK_GLIB_STRUCT || member>=tc->sub_parts) {
		return(NULL);
	}
	
	return(tc->subnames[member]);
}

const wsdl_typecode *wsdl_typecode_member_type(const wsdl_typecode *const tc,
					       guint member)
{
	g_assert(tc!=NULL);

	if(tc->kind!=WSDL_TK_GLIB_STRUCT || member>=tc->sub_parts) {
		return(NULL);
	}
	
	return(tc->subtypes[member]);
}

const wsdl_typecode *wsdl_typecode_content_type(const wsdl_typecode *const tc)
{
	g_assert(tc!=NULL);

	if(tc->kind!=WSDL_TK_GLIB_LIST || tc->kind!=WSDL_TK_GLIB_ELEMENT) {
		return(NULL);
	}
	
	return(tc->subtypes[0]);
}

const guchar *wsdl_typecode_name(const wsdl_typecode *const tc)
{
	g_assert(tc!=NULL);
	
	return(tc->name);
}

const guchar *wsdl_typecode_ns(const wsdl_typecode *const tc)
{
	g_assert(tc!=NULL);
	
	return(tc->ns);
}

const guchar *wsdl_typecode_nsuri(const wsdl_typecode *const tc)
{
	g_assert(tc!=NULL);
	
	return(tc->nsuri);
}

guchar *wsdl_typecode_type(const wsdl_typecode *const tc)
{
	guchar *str=NULL;
	
	g_assert(tc!=NULL);
	
	switch(tc->kind) {
	case WSDL_TK_GLIB_NULL:
		break;
	case WSDL_TK_GLIB_VOID:
		break;
	case WSDL_TK_GLIB_BOOLEAN:
		str=g_strdup("gboolean");
		break;
	case WSDL_TK_GLIB_CHAR:
		str=g_strdup("gchar");
		break;
	case WSDL_TK_GLIB_UCHAR:
		str=g_strdup("guchar");
		break;
	case WSDL_TK_GLIB_INT:
		str=g_strdup("gint");
		break;
	case WSDL_TK_GLIB_UINT:
		str=g_strdup("guint");
		break;
	case WSDL_TK_GLIB_SHORT:
		str=g_strdup("gshort");
		break;
	case WSDL_TK_GLIB_USHORT:
		str=g_strdup("gushort");
		break;
	case WSDL_TK_GLIB_LONG:
		str=g_strdup("glong");
		break;
	case WSDL_TK_GLIB_ULONG:
		str=g_strdup("gulong");
		break;
	case WSDL_TK_GLIB_INT8:
		str=g_strdup("gint8");
		break;
	case WSDL_TK_GLIB_UINT8:
		str=g_strdup("guint8");
		break;
	case WSDL_TK_GLIB_INT16:
		str=g_strdup("gint16");
		break;
	case WSDL_TK_GLIB_UINT16:
		str=g_strdup("guint16");
		break;
	case WSDL_TK_GLIB_INT32:
		str=g_strdup("gint32");
		break;
	case WSDL_TK_GLIB_UINT32:
		str=g_strdup("guint32");
		break;
	case WSDL_TK_GLIB_FLOAT:
		str=g_strdup("gfloat");
		break;
	case WSDL_TK_GLIB_DOUBLE:
		str=g_strdup("gdouble");
		break;
	case WSDL_TK_GLIB_STRING:
		str=g_strdup("guchar *");
		break;
	case WSDL_TK_GLIB_ELEMENT:
		str=g_strdup_printf("%s_%s", tc->ns, tc->name);
		break;
	case WSDL_TK_GLIB_STRUCT:
		str=g_strdup_printf("%s_%s", tc->ns, tc->name);
		break;
	case WSDL_TK_GLIB_LIST:
		str=g_strdup("GSList *");
		break;
	case WSDL_TK_GLIB_MAX:
		break;
	}

	if(str==NULL) {
		str=g_strdup("NULL");
	}
	
	return(str);
}

guchar *wsdl_typecode_param_type(const wsdl_typecode *const tc)
{
	guchar *str=NULL;
	
	g_assert(tc!=NULL);
	
	switch(tc->kind) {
	case WSDL_TK_GLIB_ELEMENT: 
	{
		const wsdl_typecode *subtc=tc->subtypes[0];
		
		/* This one gets tricky. If the aliased type is a struct we
		 * need to add the '*'
		 */
		while(subtc!=NULL) {
			if(subtc->kind==WSDL_TK_GLIB_ELEMENT) {
				subtc=subtc->subtypes[0];
			} else if (subtc->kind==WSDL_TK_GLIB_STRUCT) {
				str=g_strdup_printf("%s_%s *", tc->ns,
						    tc->name);
				break;
			} else {
				str=g_strdup_printf("%s_%s", tc->ns, tc->name);
				break;
			}
		}
		
		break;
	}
	
	case WSDL_TK_GLIB_STRUCT:
		str=g_strdup_printf("%s_%s *", tc->ns, tc->name);
		break;
	case WSDL_TK_GLIB_NULL:
	case WSDL_TK_GLIB_VOID:
	case WSDL_TK_GLIB_BOOLEAN:
	case WSDL_TK_GLIB_CHAR:
	case WSDL_TK_GLIB_UCHAR:
	case WSDL_TK_GLIB_INT:
	case WSDL_TK_GLIB_UINT:
	case WSDL_TK_GLIB_SHORT:
	case WSDL_TK_GLIB_USHORT:
	case WSDL_TK_GLIB_LONG:
	case WSDL_TK_GLIB_ULONG:
	case WSDL_TK_GLIB_INT8:
	case WSDL_TK_GLIB_UINT8:
	case WSDL_TK_GLIB_INT16:
	case WSDL_TK_GLIB_UINT16:
	case WSDL_TK_GLIB_INT32:
	case WSDL_TK_GLIB_UINT32:
	case WSDL_TK_GLIB_FLOAT:
	case WSDL_TK_GLIB_DOUBLE:
	case WSDL_TK_GLIB_STRING:
	case WSDL_TK_GLIB_LIST:
	case WSDL_TK_GLIB_MAX:
		str=wsdl_typecode_type(tc);
		break;
	}

	if(str==NULL) {
		str=g_strdup("NULL");
	}
	
	return(str);
}

static void tc_indent(guint ind)
{
	unsigned int i;

	for(i=0; i<ind; i++) {
		g_print(" ");
	}
}

void wsdl_typecode_print(const wsdl_typecode *const tc, guint ind)
{
	guint i;
	
	g_assert(tc!=NULL);
	
	tc_indent(ind);g_print("%s ", wsdl_typecode_kind_name(tc->kind));
	g_print("%s (%s:%s):\n", tc->name, tc->ns, tc->nsuri);

	switch(tc->kind) {
	default:
		g_print("\n");
		break;
		
	case WSDL_TK_GLIB_ELEMENT:
		wsdl_typecode_print(tc->subtypes[0], ind+4);
		break;
		
	case WSDL_TK_GLIB_STRUCT:
		for(i=0; i<tc->sub_parts; i++) {
			tc_indent(ind+2);g_print("%s:\n", tc->subnames[i]);
			
			wsdl_typecode_print(tc->subtypes[i], ind+4);
		}
		break;
		
	case WSDL_TK_GLIB_LIST:
		wsdl_typecode_print(tc->subtypes[0], ind+4);
		break;
	}
}

static void wsdl_typecode_init(void)
{
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_void_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_boolean_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_char_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_uchar_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_int_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_uint_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_short_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_ushort_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_long_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_ulong_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_int8_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_uint8_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_int16_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_uint16_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_int32_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_uint32_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_float_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_double_struct);
	wsdl_typecodes=g_slist_append(wsdl_typecodes,
				      (gpointer)&WSDL_TC_glib_string_struct);
}

const wsdl_typecode *wsdl_typecode_lookup(const guchar *name,
					  const guchar *nsuri)
{
	GSList *iter;
	guchar *colon;
	const guchar *compare;
	
	if(wsdl_typecodes==NULL) {
		wsdl_typecode_init();
	}

	if((colon=strchr(name, ':'))!=NULL) {
		compare=colon+1;
	} else {
		compare=name;
	}
	
	iter=wsdl_typecodes;
	while(iter!=NULL) {
		wsdl_typecode *tc=(wsdl_typecode *)iter->data;
		
		/* Check for namespace-qualified names */
		if(nsuri!=NULL &&
		   !strcmp(compare, tc->name) &&
		   !strcmp(nsuri, tc->nsuri)) {
			return(tc);
		}
		
		/* Check for any match */
		if(nsuri==NULL &&
		   !strcmp(compare, tc->name)) {
			return(tc);
		}
		
		iter=iter->next;
	}

	return(NULL);
}

void wsdl_typecode_register(const wsdl_typecode *const tc)
{
	if(wsdl_typecodes==NULL) {
		wsdl_typecode_init();
	}
	
	/* g_slist_append really should declare the data arg const */
	wsdl_typecodes=g_slist_append(wsdl_typecodes, (gpointer)tc);
}

void wsdl_typecode_unregister(const guchar *name, const guchar *nsuri)
{
	const wsdl_typecode *tc;
	
	g_return_if_fail(name==NULL);
	
	tc=wsdl_typecode_lookup(name, nsuri);
	if(tc==NULL) {
		return;
	}
	
	wsdl_typecodes=g_slist_remove(wsdl_typecodes, (gpointer)tc);

	if(tc->dynamic) {
		wsdl_typecode_free((gpointer)tc);
	}
}

void wsdl_typecode_free(wsdl_typecode *tc)
{
	gulong i;
	
	g_return_if_fail(tc==NULL);
	g_return_if_fail(tc->dynamic==FALSE);
	
	if(tc->name!=NULL) {
		g_free((gpointer)tc->name);
	}
	if(tc->ns!=NULL) {
		g_free((gpointer)tc->ns);
	}
	if(tc->nsuri!=NULL) {
		g_free((gpointer)tc->nsuri);
	}
	
	if(tc->kind==WSDL_TK_GLIB_ELEMENT ||
	   tc->kind==WSDL_TK_GLIB_STRUCT ||
	   tc->kind==WSDL_TK_GLIB_LIST) {
		for(i=0; i<tc->sub_parts; i++) {
			if(tc->subnames[i]!=NULL) {
				g_free((gpointer)tc->subnames[i]);
			}
			if(tc->kind==WSDL_TK_GLIB_STRUCT &&
			   tc->subtypes[i]!=NULL) {
				wsdl_typecode_free((gpointer)tc->subtypes[i]);
			}
		}
	}

	g_free(tc);
}

void wsdl_typecode_free_all(void)
{
	GSList *iter=wsdl_typecodes;
	
	while(iter!=NULL) {
		wsdl_typecode *tc=(wsdl_typecode *)iter->data;
		
	
		wsdl_typecodes=g_slist_remove(wsdl_typecodes, tc);

		if(tc->dynamic) {
			wsdl_typecode_free(tc);
		}

		iter=iter->next;
	}
}

void wsdl_typecode_foreach(gboolean simple,
			   void (*callback)(const wsdl_typecode *const,
					    gpointer),
			   gpointer user_data)
{
	GSList *iter=wsdl_typecodes;
	
	while(iter!=NULL) {
		const wsdl_typecode *const tc=(wsdl_typecode *)iter->data;
		
		if(simple==TRUE || tc->kind>=WSDL_TK_GLIB_ELEMENT) {
			callback(tc, user_data);
		}
		
		iter=iter->next;
	}
}

guint wsdl_typecode_find_alignment(const wsdl_typecode *const tc)
{
	switch(tc->kind) {
	case WSDL_TK_GLIB_NULL:
		g_warning("Invalid typecode NULL in " G_GNUC_FUNCTION);
		return(0);
		break;
		
	case WSDL_TK_GLIB_VOID:
		return(0);
		break;
		
	case WSDL_TK_GLIB_BOOLEAN:
		return(ALIGNOF_GBOOLEAN);
		break;
		
	case WSDL_TK_GLIB_CHAR:
		return(ALIGNOF_GCHAR);
		break;
		
	case WSDL_TK_GLIB_UCHAR:
		return(ALIGNOF_GUCHAR);
		break;
		
	case WSDL_TK_GLIB_INT:
		return(ALIGNOF_GINT);
		break;
		
	case WSDL_TK_GLIB_UINT:
		return(ALIGNOF_GUINT);
		break;
		
	case WSDL_TK_GLIB_SHORT:
		return(ALIGNOF_GSHORT);
		break;
		
	case WSDL_TK_GLIB_USHORT:
		return(ALIGNOF_GUSHORT);
		break;
		
	case WSDL_TK_GLIB_LONG:
		return(ALIGNOF_GLONG);
		break;
		
	case WSDL_TK_GLIB_ULONG:
		return(ALIGNOF_GULONG);
		break;
		
	case WSDL_TK_GLIB_INT8:
		return(ALIGNOF_GINT8);
		break;
		
	case WSDL_TK_GLIB_UINT8:
		return(ALIGNOF_GUINT8);
		break;
		
	case WSDL_TK_GLIB_INT16:
		return(ALIGNOF_GINT16);
		break;
		
	case WSDL_TK_GLIB_UINT16:
		return(ALIGNOF_GUINT16);
		break;
		
	case WSDL_TK_GLIB_INT32:
		return(ALIGNOF_GINT32);
		break;
		
	case WSDL_TK_GLIB_UINT32:
		return(ALIGNOF_GUINT32);
		break;
		
	case WSDL_TK_GLIB_FLOAT:
		return(ALIGNOF_GFLOAT);
		break;
		
	case WSDL_TK_GLIB_DOUBLE:
		return(ALIGNOF_GDOUBLE);
		break;
		
	case WSDL_TK_GLIB_STRING:
		return(ALIGNOF_GPOINTER);
		break;
		
	case WSDL_TK_GLIB_ELEMENT:
		return(MAX(1, wsdl_typecode_find_alignment(tc->subtypes[0])));
		break;
		
	case WSDL_TK_GLIB_STRUCT: 
	{
		gulong i;
		guint align=1;
		
		for(i=0; i<tc->sub_parts; i++) {
			align=MAX(align, wsdl_typecode_find_alignment(tc->subtypes[i]));
		}
		
		return(align);
		break;
	}
		
	case WSDL_TK_GLIB_LIST:
		return(ALIGNOF_GPOINTER);
		break;
		
	case WSDL_TK_GLIB_MAX:
		g_warning("Invalid typecode MAX in " G_GNUC_FUNCTION);
		return(0);
		break;
	}

	/* gcc should know that it can't reach here */
	g_assert_not_reached();
	return(0);
}

guint wsdl_typecode_size(const wsdl_typecode *const tc)
{
	switch(tc->kind){
	case WSDL_TK_GLIB_NULL:
		g_warning("Invalid typecode NULL in " G_GNUC_FUNCTION);
		return(0);
		break;
		
	case WSDL_TK_GLIB_VOID:
		return(0);
		break;
		
	case WSDL_TK_GLIB_BOOLEAN:
		return(sizeof(gboolean));
		break;
		
	case WSDL_TK_GLIB_CHAR:
		return(sizeof(gchar));
		break;
		
	case WSDL_TK_GLIB_UCHAR:
		return(sizeof(guchar));
		break;
		
	case WSDL_TK_GLIB_INT:
		return(sizeof(gint));
		break;
		
	case WSDL_TK_GLIB_UINT:
		return(sizeof(guint));
		break;
		
	case WSDL_TK_GLIB_SHORT:
		return(sizeof(gshort));
		break;
		
	case WSDL_TK_GLIB_USHORT:
		return(sizeof(gushort));
		break;
		
	case WSDL_TK_GLIB_LONG:
		return(sizeof(glong));
		break;
		
	case WSDL_TK_GLIB_ULONG:
		return(sizeof(gulong));
		break;
		
	case WSDL_TK_GLIB_INT8:
		return(sizeof(gint8));
		break;
		
	case WSDL_TK_GLIB_UINT8:
		return(sizeof(guint8));
		break;
		
	case WSDL_TK_GLIB_INT16:
		return(sizeof(gint16));
		break;
		
	case WSDL_TK_GLIB_UINT16:
		return(sizeof(guint16));
		break;
		
	case WSDL_TK_GLIB_INT32:
		return(sizeof(gint32));
		break;
		
	case WSDL_TK_GLIB_UINT32:
		return(sizeof(guint32));
		break;
		
	case WSDL_TK_GLIB_FLOAT:
		return(sizeof(gfloat));
		break;
		
	case WSDL_TK_GLIB_DOUBLE:
		return(sizeof(gdouble));
		break;
		
	case WSDL_TK_GLIB_STRING:
		return(sizeof(guchar *));
		break;
		
	case WSDL_TK_GLIB_ELEMENT:
		return(wsdl_typecode_size(tc->subtypes[0]));
		break;
		
	case WSDL_TK_GLIB_STRUCT: 
	{
		gulong i;
		guint sum=0;
		
		for(i=0; i<tc->sub_parts; i++) {
			sum=GPOINTER_TO_INT(ALIGN_ADDRESS(sum, wsdl_typecode_find_alignment(tc->subtypes[i])));
			sum+=wsdl_typecode_size(tc->subtypes[i]);
		}
		sum=GPOINTER_TO_INT(ALIGN_ADDRESS(sum, wsdl_typecode_find_alignment(tc)));
		
		return(sum);
		break;
	}
		
	case WSDL_TK_GLIB_LIST:
		return(sizeof(GSList *));
		break;
		
	case WSDL_TK_GLIB_MAX:
		g_warning("Invalid typecode MAX in " G_GNUC_FUNCTION);
		return(0);
		break;
	}

	/* gcc should know that it can't reach here */
	g_assert_not_reached();
	return(0);
}

gpointer wsdl_typecode_alloc(const wsdl_typecode *const tc)
{
	guint size;
	gpointer ret;
	
	size=wsdl_typecode_size(tc);
	ret=g_malloc0(size);
	
	return(ret);
}

const wsdl_typecode *wsdl_typecode_offset(const wsdl_typecode *const tc,
					  const guchar *name,
					  guint *offset)
{
	const wsdl_typecode *ret=NULL;
	gulong i;
	
	*offset=0;

	for(i=0; i<tc->sub_parts; i++) {
		if(!strcmp(name, tc->subnames[i])) {
			ret=tc->subtypes[i];
			break;
		}
		*offset+=wsdl_typecode_size(tc->subtypes[i]);
	}
	
	return(ret);
}
