/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * soup-nss.c: Asyncronous Callback-based SOAP Request Queue.
 *
 * Authors:
 *      Alex Graveley (alex@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <config.h>

#ifdef HAVE_SECURITY_SSL_H

#include <glib.h>
#include <security/nss.h>
#include <security/ssl.h>
#include <security/pk11func.h>

#include "soup-nss.h"

/* NSPR Private function */
extern PRFileDesc *PR_ImportTCPSocket (int fd);

typedef struct {
	GIOChannel   channel;
	gint         fd;
	GIOChannel  *real_sock;
	PRFileDesc  *fdesc;
} SoupNSSChannel;

static GIOError
soup_nss_read (GIOChannel   *channel,
	       gchar        *buf,
	       guint         count,
	       guint        *bytes_read)
{
	SoupNSSChannel *chan = (SoupNSSChannel *) channel;
	gint result;

	result = PR_Read (chan->fdesc, buf, count);

	if (result < 0) {
		*bytes_read = 0;
		switch (PR_GetError ()) {
		case PR_INVALID_ARGUMENT_ERROR:
			return G_IO_ERROR_INVAL;
		case PR_WOULD_BLOCK_ERROR:
			return G_IO_ERROR_AGAIN;
		default:
			return G_IO_ERROR_UNKNOWN;
		}
	} else {
		*bytes_read = result;
		return G_IO_ERROR_NONE;
	}
}

static GIOError
soup_nss_write (GIOChannel   *channel,
		gchar        *buf,
		guint         count,
		guint        *bytes_written)
{
	SoupNSSChannel *chan = (SoupNSSChannel *) channel;
	gint result;

	result = PR_Write (chan->fdesc, buf, count);

	if (result < 0) {
		*bytes_written = 0;
		switch (PR_GetError ()) {
		case PR_INVALID_ARGUMENT_ERROR:
			return G_IO_ERROR_INVAL;
		case PR_WOULD_BLOCK_ERROR:
			return G_IO_ERROR_AGAIN;
		default:
			return G_IO_ERROR_UNKNOWN;
		}
	} else {
		*bytes_written = result;
		return G_IO_ERROR_NONE;
	}
}

static GIOError
soup_nss_seek (GIOChannel *channel, gint offset, GSeekType type)
{
	SoupNSSChannel *chan = (SoupNSSChannel *) channel;
	int whence;
	off_t result;
	
	switch (type) {
	case G_SEEK_SET:
		whence = PR_SEEK_SET;
		break;
	case G_SEEK_CUR:
		whence = PR_SEEK_CUR;
		break;
	case G_SEEK_END:
		whence = PR_SEEK_END;
		break;
	default:
		g_warning ("soup_nss_seek: unknown seek type");
		return G_IO_ERROR_UNKNOWN;
	}
	
	result = PR_Seek (chan->fdesc, offset, whence);
	
	if (result < 0) {
		switch (PR_GetError ()) {
		case PR_INVALID_ARGUMENT_ERROR:
			return G_IO_ERROR_INVAL;
		case PR_WOULD_BLOCK_ERROR:
			return G_IO_ERROR_AGAIN;
		default:
			return G_IO_ERROR_UNKNOWN;
		}
	} else
		return G_IO_ERROR_NONE;
}

static void
soup_nss_close (GIOChannel   *channel)
{
	SoupNSSChannel *chan = (SoupNSSChannel *) channel;
	PR_Close (chan->fdesc);
	g_io_channel_close (chan->real_sock);
}

static void
soup_nss_free (GIOChannel   *channel)
{
	SoupNSSChannel *chan = (SoupNSSChannel *) channel;
	g_io_channel_unref (chan->real_sock);
	g_free (chan);
}

typedef struct {
	GIOFunc         func;
	gpointer        user_data;
} SoupNSSReadData;

static gboolean 
soup_nss_read_cb (GIOChannel   *channel, 
		  GIOCondition  condition, 
		  gpointer      user_data)
{
	SoupNSSChannel *chan = (SoupNSSChannel *) channel;
	SoupNSSReadData *data = user_data;

	if (condition & G_IO_IN) {
		if (SSL_DataPending (chan->fdesc) && 
		    !(*data->func) (channel, condition, data->user_data)) {
			g_free (data);
			return FALSE;
		}
		return TRUE;
	} else return (*data->func) (channel, condition, data->user_data);
}

static guint
soup_nss_add_watch (GIOChannel     *channel,
		    gint            priority,
		    GIOCondition    condition,
		    GIOFunc         func,
		    gpointer        user_data,
		    GDestroyNotify  notify)
{
	SoupNSSChannel *chan = (SoupNSSChannel *) channel;
	if (condition & G_IO_IN) {
		SoupNSSReadData *data = g_new0 (SoupNSSReadData, 1);
		data->func = func;
		data->user_data = user_data;

		return chan->real_sock->funcs->io_add_watch (channel, 
							     priority, 
							     condition,
							     soup_nss_read_cb,
							     data,
							     notify);
	} else return chan->real_sock->funcs->io_add_watch (channel, 
							    priority, 
							    condition,
							    func,
							    user_data,
							    notify);
}

GIOFuncs soup_nss_channel_funcs = {
	soup_nss_read,
	soup_nss_write,
	soup_nss_seek,
	soup_nss_close,
	soup_nss_add_watch,
	soup_nss_free,
};

static gboolean nss_initialized = FALSE;

static SECStatus 
soup_nss_bad_cert (void *data, PRFileDesc *fd)
{
	return SECSuccess;
}

GIOChannel *
soup_nss_get_iochannel (GIOChannel *sock)
{
	SoupNSSChannel *chan;
	GIOChannel *gchan;
	PRFileDesc *fdesc, *ssl_fdesc;
	int sockfd;

        g_return_val_if_fail (sock != NULL, NULL);

	if (!nss_initialized && !soup_nss_init ()) goto THROW_CREATE_ERROR;
	
	sockfd = g_io_channel_unix_get_fd (sock);
	if (!sockfd) goto THROW_CREATE_ERROR;

	fdesc = PR_ImportTCPSocket (sockfd);
	if (!fdesc) {
		g_warning ("SSL socket creation failure.");
		goto THROW_CREATE_ERROR;
	}

	ssl_fdesc = SSL_ImportFD (NULL, fdesc);
	if (!ssl_fdesc) {
		g_warning ("SSL object creation failure.");
		goto THROW_CREATE_ERROR;
	}

	SSL_BadCertHook (ssl_fdesc, soup_nss_bad_cert, NULL);

	chan = g_new0 (SoupNSSChannel, 1);
	chan->fd = sockfd;
	chan->real_sock = sock;
	chan->fdesc = ssl_fdesc;
	g_io_channel_ref (sock);

	gchan = (GIOChannel *) chan;
	gchan->funcs = &soup_nss_channel_funcs;
	g_io_channel_init (gchan);

	return gchan;

 THROW_CREATE_ERROR:
	return NULL;
}

static char *
soup_nss_fail_password_ask (PK11SlotInfo *slot,
			    PRBool        retry,
			    void         *arg)
{
	return NULL;
}

gboolean
soup_nss_init (void)
{
	gchar *nss_dir = NULL;

	PK11_SetPasswordFunc (soup_nss_fail_password_ask);

	/* FIXME: NSS needs a directory with valid DB files in order to
	          work. Need to get a working copy of the keygen tools, and deal
	          with generation of new keys on install. */

	if (NSS_InitReadWrite (nss_dir) != SECSuccess) {
		g_warning ("Unable to initialize NSS SSL Library.");
		return FALSE;
	}

	g_free (nss_dir);

	nss_initialized = TRUE;
	return TRUE;
}

void 
soup_nss_set_security_policy (SoupSecurityPolicy policy)
{
	switch (policy) {
	case SOUP_SECURITY_DOMESTIC:
		NSS_SetDomesticPolicy ();
		break;
	case SOUP_SECURITY_EXPORT:
		NSS_SetExportPolicy ();
		break;
	case SOUP_SECURITY_FRANCE:
		NSS_SetFrancePolicy ();
		break;
	}

	SSL_ClearSessionCache ();
}

#endif /*HAVE_SECURITY_SSL_H*/

