#define SP_ITEM_C

#include <gnome.h>
#include "svg/svg.h"
#include "helper/sp-canvas-util.h"
#include "desktop.h"
#include "sp-item.h"

static void sp_item_class_init (SPItemClass * klass);
static void sp_item_init (SPItem * item);
static void sp_item_destroy (GtkObject * object);

static void sp_item_build (SPObject * object, SPDocument * document, SPRepr * repr);
static void sp_item_read_attr (SPObject * object, const gchar * key);

static gchar * sp_item_private_description (SPItem * item);

static GnomeCanvasItem * sp_item_private_show (SPItem * item, SPDesktop * desktop, GnomeCanvasGroup * canvas_group);
static void sp_item_private_hide (SPItem * item, SPDesktop * desktop);

static void sp_item_private_menu (SPItem * item, GtkMenu * menu);
void sp_item_reset_transformation (GtkMenuItem * menuitem, SPItem * item);
void sp_item_toggle_sensitivity (GtkMenuItem * menuitem, SPItem * item);

static SPObjectClass * parent_class;

GtkType
sp_item_get_type (void)
{
	static GtkType item_type = 0;
	if (!item_type) {
		GtkTypeInfo item_info = {
			"SPItem",
			sizeof (SPItem),
			sizeof (SPItemClass),
			(GtkClassInitFunc) sp_item_class_init,
			(GtkObjectInitFunc) sp_item_init,
			NULL, /* reserved_1 */
			NULL, /* reserved_2 */
			(GtkClassInitFunc) NULL
		};
		item_type = gtk_type_unique (sp_object_get_type (), &item_info);
	}
	return item_type;
}

static void
sp_item_class_init (SPItemClass * klass)
{
	GtkObjectClass * gtk_object_class;
	SPObjectClass * sp_object_class;

	gtk_object_class = (GtkObjectClass *) klass;
	sp_object_class = (SPObjectClass *) klass;

	parent_class = gtk_type_class (sp_object_get_type ());

	gtk_object_class->destroy = sp_item_destroy;

	sp_object_class->build = sp_item_build;
	sp_object_class->read_attr = sp_item_read_attr;

	klass->description = sp_item_private_description;
	klass->show = sp_item_private_show;
	klass->hide = sp_item_private_hide;
	klass->menu = sp_item_private_menu;
}

static void
sp_item_init (SPItem * item)
{
	art_affine_identity (item->affine);
	item->display = NULL;
}

static void
sp_item_destroy (GtkObject * object)
{
	SPItem * item;

	item = (SPItem *) object;

	while (item->display) {
		gtk_object_destroy (GTK_OBJECT (item->display->canvasitem));
		item->display = sp_item_view_list_remove (item->display, item->display);
	}

	if (((GtkObjectClass *) (parent_class))->destroy)
		(* ((GtkObjectClass *) (parent_class))->destroy) (object);
}

static void
sp_item_build (SPObject * object, SPDocument * document, SPRepr * repr)
{
	if (((SPObjectClass *) (parent_class))->build)
		(* ((SPObjectClass *) (parent_class))->build) (object, document, repr);

	sp_item_read_attr (object, "transform");
}

static void
sp_item_read_attr (SPObject * object, const gchar * key)
{
	SPItem * item, * i;
	SPItemView * v;
	gdouble a[6];
	const gchar * astr;

	item = SP_ITEM (object);

	astr = sp_repr_attr (object->repr, key);

	if (strcmp (key, "transform") == 0) {
		art_affine_identity (item->affine);
		if (astr != NULL) {
			sp_svg_read_affine (item->affine, astr);
		}
		for (v = item->display; v != NULL; v = v->next) {
			gnome_canvas_item_affine_absolute (v->canvasitem, item->affine);
		}
		art_affine_identity (a);
		for (i = item; i != NULL; i = (SPItem *) ((SPObject *)i)->parent)
			art_affine_multiply (a, a, i->affine);
		sp_item_update (item, a);
		return;
	}

	if (((SPObjectClass *) (parent_class))->read_attr)
		(* ((SPObjectClass *) (parent_class))->read_attr) (object, key);
}

/* Update indicates that affine is changed */

void
sp_item_update (SPItem * item, gdouble affine[])
{
	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (affine != NULL);

	if (SP_ITEM_CLASS (((GtkObject *)(item))->klass)->update)
		(* SP_ITEM_CLASS (((GtkObject *)(item))->klass)->update) (item, affine);
}

void sp_item_bbox (SPItem * item, ArtDRect * bbox)
{
	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (bbox != NULL);

	bbox->x0 = bbox->y0 = 1.0;
	bbox->x1 = bbox->y1 = 0.0;

	if (SP_ITEM_CLASS (((GtkObject *)(item))->klass)->bbox)
		(* SP_ITEM_CLASS (((GtkObject *)(item))->klass)->bbox) (item, bbox);
}

void sp_item_print (SPItem * item, GnomePrintContext * gpc)
{
	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (gpc != NULL);

	if (SP_ITEM_CLASS (((GtkObject *)(item))->klass)->print)
		(* SP_ITEM_CLASS (((GtkObject *)(item))->klass)->print) (item, gpc);
}

static gchar *
sp_item_private_description (SPItem * item)
{
	return g_strdup (_("Unknown item :-("));
}

gchar * sp_item_description (SPItem * item)
{
	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));

	if (SP_ITEM_CLASS (((GtkObject *)(item))->klass)->description)
		return (* SP_ITEM_CLASS (((GtkObject *)(item))->klass)->description) (item);

	g_assert_not_reached ();
	return NULL;
}

static GnomeCanvasItem *
sp_item_private_show (SPItem * item, SPDesktop * desktop, GnomeCanvasGroup * canvas_group)
{
	return NULL;
}

GnomeCanvasItem *
sp_item_show (SPItem * item, SPDesktop * desktop, GnomeCanvasGroup * canvas_group)
{
	GnomeCanvasItem * canvasitem;

	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (desktop != NULL);
	g_assert (SP_IS_DESKTOP (desktop));
	g_assert (canvas_group != NULL);
	g_assert (GNOME_IS_CANVAS_GROUP (canvas_group));

	canvasitem = NULL;

	if (SP_ITEM_CLASS (((GtkObject *)(item))->klass)->show) {
		canvasitem =  (* SP_ITEM_CLASS (((GtkObject *)(item))->klass)->show) (item, desktop, canvas_group);
	} else {
		canvasitem = NULL;
	}

	if (canvasitem != NULL) {
		item->display = sp_item_view_new_prepend (item->display, item, desktop, canvasitem);
		gnome_canvas_item_affine_absolute (canvasitem, item->affine);
		sp_desktop_connect_item (desktop, item, canvasitem);
	}

	return canvasitem;
}

static void
sp_item_private_hide (SPItem * item, SPDesktop * desktop)
{
	SPItemView * v;

	for (v = item->display; v != NULL; v = v->next) {
		if (v->desktop == desktop) {
			gtk_object_destroy (GTK_OBJECT (v->canvasitem));
			item->display = sp_item_view_list_remove (item->display, v);
			return;
		}
	}

	g_assert_not_reached ();
}

void sp_item_hide (SPItem * item, SPDesktop * desktop)
{
	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (desktop != NULL);
	g_assert (SP_IS_DESKTOP (desktop));

	if (SP_ITEM_CLASS (((GtkObject *)(item))->klass)->hide)
		(* SP_ITEM_CLASS (((GtkObject *)(item))->klass)->hide) (item, desktop);
}

gboolean sp_item_paint (SPItem * item, ArtPixBuf * buf, gdouble affine[])
{
	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (buf != NULL);
	g_assert (affine != NULL);

	if (SP_ITEM_STOP_PAINT (item)) return TRUE;

	if (SP_ITEM_CLASS (((GtkObject *)(item))->klass)->paint)
		return (* SP_ITEM_CLASS (((GtkObject *)(item))->klass)->paint) (item, buf, affine);

	return FALSE;
}

GnomeCanvasItem *
sp_item_canvas_item (SPItem * item, SPDesktop * desktop)
{
	SPItemView * v;

	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (desktop != NULL);
	g_assert (SP_IS_DESKTOP (desktop));

	for (v = item->display; v != NULL; v = v->next) {
		if (v->desktop == desktop) return v->canvasitem;
	}

	return NULL;
}

void
sp_item_request_canvas_update (SPItem * item)
{
	SPItemView * v;

	g_return_if_fail (item != NULL);
	g_return_if_fail (SP_IS_ITEM (item));

	for (v = item->display; v != NULL; v = v->next) {
		gnome_canvas_item_request_update (v->canvasitem);
	}
}

gdouble *
sp_item_i2d_affine (SPItem * item, gdouble affine[])
{
	g_return_val_if_fail (item != NULL, NULL);
	g_return_val_if_fail (SP_IS_ITEM (item), NULL);
	g_return_val_if_fail (affine != NULL, NULL);

	art_affine_identity (affine);

	while (item) {
		art_affine_multiply (affine, affine, item->affine);
		item = (SPItem *) SP_OBJECT (item)->parent;
	}

	return affine;
}

void
sp_item_set_i2d_affine (SPItem * item, gdouble affine[])
{
	SPItemView * v;
	gdouble p2d[6], d2p[6];
	gint i;

	g_return_if_fail (item != NULL);
	g_return_if_fail (SP_IS_ITEM (item));
	g_return_if_fail (affine != NULL);

	if (SP_OBJECT (item)->parent != NULL) {
		sp_item_i2d_affine (SP_ITEM (SP_OBJECT (item)->parent), p2d);
		art_affine_invert (d2p, p2d);
		art_affine_multiply (affine, affine, d2p);
	}

	for (i = 0; i < 6; i++) item->affine[i] = affine[i];

#if 0
	/* fixme: do the updating right way */
	sp_item_update (item, affine);
#else
	for (v = item->display; v != NULL; v = v->next) {
		gnome_canvas_item_affine_absolute (v->canvasitem, affine);
	}
#endif
}

gdouble *
sp_item_i2doc_affine (SPItem * item, gdouble affine[])
{
	g_return_val_if_fail (item != NULL, NULL);
	g_return_val_if_fail (SP_IS_ITEM (item), NULL);
	g_return_val_if_fail (affine != NULL, NULL);

	art_affine_identity (affine);

	while (SP_OBJECT (item)->parent) {
		art_affine_multiply (affine, affine, item->affine);
		item = (SPItem *) SP_OBJECT (item)->parent;
	}

	return affine;
}

void
sp_item_change_canvasitem_position (SPItem * item, gint delta)
{
	SPItemView * v;

	g_return_if_fail (item != NULL);
	g_return_if_fail (SP_IS_ITEM (item));

	if (delta == 0) return;

	for (v = item->display; v != NULL; v = v->next) {
		if (delta > 0) {
			gnome_canvas_item_raise (v->canvasitem, delta);
		} else {
			gnome_canvas_item_lower (v->canvasitem, -delta);
		}
	}
}

void
sp_item_raise_canvasitem_to_top (SPItem * item)
{
	SPItemView * v;

	g_return_if_fail (item != NULL);
	g_return_if_fail (SP_IS_ITEM (item));

	for (v = item->display; v != NULL; v = v->next) {
		gnome_canvas_item_raise_to_top (v->canvasitem);
	}
}

static void
sp_item_private_menu (SPItem * item, GtkMenu * menu)
{
	GtkWidget * i, * m, * w;

	i = gtk_menu_item_new_with_label ("Item");

	/* Reset transformations */
	m = gtk_menu_new ();
	w = gtk_menu_item_new_with_label (_("Reset transformation"));
	gtk_signal_connect (GTK_OBJECT (w), "activate",
			    GTK_SIGNAL_FUNC (sp_item_reset_transformation), item);
	gtk_widget_show (w);
	gtk_menu_append (GTK_MENU (m), w);
	/* Toggle sensitivity */
	/* "sensitive" attribute is handled in shape & image */
	w = gtk_menu_item_new_with_label (_("Toggle sensitivity"));
	gtk_signal_connect (GTK_OBJECT (w), "activate",
			    GTK_SIGNAL_FUNC (sp_item_toggle_sensitivity), item);
	gtk_widget_show (w);
	gtk_menu_append (GTK_MENU (m), w);
	gtk_widget_show (m);

	gtk_menu_item_set_submenu (GTK_MENU_ITEM (i), m);

	gtk_menu_append (menu, i);
	gtk_widget_show (i);
}

void
sp_item_menu (SPItem * item, GtkMenu * menu)
{
	g_assert (SP_IS_ITEM (item));
	g_assert (GTK_IS_MENU (menu));

	if (SP_ITEM_CLASS (((GtkObject *) (item))->klass)->menu)
		(* SP_ITEM_CLASS (((GtkObject *) (item))->klass)->menu) (item, menu);
}

void
sp_item_reset_transformation (GtkMenuItem * menuitem, SPItem * item)
{
	g_assert (SP_IS_ITEM (item));

	sp_repr_set_attr (((SPObject *) item)->repr, "transform", NULL);
}

void
sp_item_toggle_sensitivity (GtkMenuItem * menuitem, SPItem * item)
{
	const gchar * val;

	g_assert (SP_IS_ITEM (item));

	/* fixme: reprs suck */
	val = sp_repr_attr (((SPObject *) item)->repr, "insensitive");
	if (val != NULL) {
		val = NULL;
	} else {
		val = "1";
	}
	sp_repr_set_attr (((SPObject *) item)->repr, "insensitive", val);
}

/* Item views */

SPItemView *
sp_item_view_new_prepend (SPItemView * list, SPItem * item, SPDesktop * desktop, GnomeCanvasItem * canvasitem)
{
	SPItemView * new;

	g_assert (item != NULL);
	g_assert (SP_IS_ITEM (item));
	g_assert (desktop != NULL);
	g_assert (SP_IS_DESKTOP (desktop));
	g_assert (canvasitem != NULL);
	g_assert (GNOME_IS_CANVAS_ITEM (canvasitem));

	new = g_new (SPItemView, 1);

	new->next = list;
	new->prev = NULL;
	if (list) list->prev = new;
	new->item = item;
	new->desktop = desktop;
	new->canvasitem = canvasitem;

	return new;
}

SPItemView *
sp_item_view_list_remove (SPItemView * list, SPItemView * view)
{
	SPItemView * v;

	g_assert (list != NULL);
	g_assert (view != NULL);

	for (v = list; v != NULL; v = v->next) {
		if (v == view) {
			if (v->next) v->next->prev = v->prev;
			if (v->prev) {
				v->prev->next = v->next;
			} else {
				list = v->next;
			}
			g_free (v);
			return list;
		}
	}

	g_assert_not_reached ();
	return NULL;
}

