///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pdf.h"
#include <string>
#include <map>
#include <iomanip>
#include <iostream>
#include <glibmm/convert.h>
#include <util/stringutil.h>

namespace PDF {

  Part::~Part()
  {
    // Delete managed parts:
    for(Managed::iterator i=managed.begin(); i!=managed.end(); i++)
      delete *i;
  }

  class Header: public Part {
  private:
    struct {int major, minor;} version;
  public:
    Header(Document *document): Part(document) {}
    void set_version(int major_version=1, int minor_version=0)
    {version.major=major_version; version.minor=minor_version;}
    bytecount write(std::ostream *out, bool binary=true, 
		    bytecount offset=0)
    {
      Part::write(out, binary, offset);
      std::string tmp="%PDF-"+tostr(version.major)+"."+tostr(version.minor);
      tmp+="\r\n"; // newline ?
      // Binary (>127) junk comment to make sure that e.g. ftp programs 
      // treat the file as binary data:
      if(binary)
	tmp+="%\xff\xff\xff\xff\r\n";
      if(out)
	(*out) << tmp;
      return offset+tmp.length();
    }
  };

  
  class Ref;

  class Object: public Part{
  public:
    Object(Document *document):Part(document) {}
    void set_ref(Ref *ref) {ref_=ref;} /// \todo set ref=0 in constructor
    Ref *get_ref() const {return ref_;}
    
  private:
    Ref *ref_;
  };

  class XRefs: public Part {
    friend class Ref;
  public:
    XRefs(Document *document): Part(document) {}
    Object* add_object(Object* object) 
    {
      objects.push_back(object); 
      object->set_ref(&create_ref(object)); 
      return object;
    }
    bytecount get_xref_offset() {return xref_offset;}
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0);
    refnum get_num_of_refs() const {return refs.size();}

  private:
    bytecount xref_offset;

    refnum get_refnum(const Ref *ref) const
    {
      // yes I know, it's inefficient ...
      for(refnum i=1; ((unsigned long) i)<=refs.size(); i++) 
	if(refs[i-1]==ref) // number 0 is always defined
	  return i;
      return -1; // ref not found
    }
    Ref &create_ref(Object *object);

    typedef std::vector<Ref*> Refs;
    Refs refs;  
    typedef std::vector<Object*> Objects;
    Objects objects;  
  };

  class Name: public Object{
  public:
    Name(std::string name_, Document *document): 
      Object(document), name(name_) 
    {}
    bool operator<(Name x) const {return this->name<x.name;} 
    const std::string &get_name() const {return name;}
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0) 
    {
      Part::write(out, binary, offset);
      if(out)
	(*out) << "/" << name;
      return offset+name.length()+1;
    }

  private:
    std::string name;
  };

  class Integer: public Object{
  public:
    Integer(long integer_, Document *document): 
      Object(document), integer(integer_) 
    {}
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0) 
    {
      Part::write(out, binary, offset);
      std::string tmp=tostr(integer);
      if(out)
	(*out) << tmp;
      return offset+tmp.length();
    }
  private:
    long integer;
  };

  class Array: public Object{
  public:
    Array(Document *document):Object(document) {}
    Object *push_back(Object *object) {items.push_back(object); return object;}
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0) 
    {
      Part::write(out, binary, offset);
      std::string tmp="[";
      offset+=tmp.length();
      if(out) (*out) << tmp; 
      for(Items::iterator i=items.begin(); i!=items.end(); i++)
	{
	  offset=(*i)->write(out, binary, offset);
	  if(*i!=items.back())
	    {
	      if(out) (*out) << " "; 
	      offset++;
	    }
	}
      tmp="]";
      if(out) (*out) << tmp; 
      offset+=tmp.length();
      return offset;

    }
  private:
    typedef std::vector<Object*> Items;
    Items items;
  };

  class Dictionary: public Object{
  public:
    Dictionary(Document *document): Object(document) {}
    Object *set_entry(std::string name, Object *object) 
    {
      entries[name]=object; 
      return object;
    } 
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0) 
    {
      Part::write(out, binary, offset);
      std::string tmp="<<\r\n";
      offset+=tmp.length();
      if(out) (*out) << tmp; 
      for(Entries::iterator i=entries.begin(); i!=entries.end(); i++)
	{
	  tmp="/"+i->first+" ";
	  offset+=tmp.length();
	  if(*out) (*out) << tmp; 
	  offset=i->second->write(out, binary, offset);
	  tmp="\r\n";
	  offset+=tmp.length();
	  if(*out) (*out) << tmp; 
	}
      tmp=">>\r\n";
      if(out) (*out) << tmp; 
      offset+=tmp.length();
      return offset;
    }

  private:
    typedef std::map<std::string, Object*> Entries;
    Entries entries;
  };

  class Ref: public Object{
    friend class XRefs;
  public:
    refnum get_num() const {return parent.get_refnum(this);}
    generation get_generation() const {return 0;}
    std::string str() const {return tostr((long) get_num())+ " "
			       + tostr((long) get_generation())
			       + " R";} // no newline
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0) 
    {
      Part::write(out, binary, offset);
      std::string str(str());
      if(out)
	(*out) << str;
      return offset+str.length();
    }

  protected:
    Ref(XRefs &parent_, Object &object, Document *document): 
      Object(document), parent(parent_) {}
    XRefs &parent;

  private:
    /// \todo eliminate default constructors
  }; 

  class Stream: public Dictionary{
  public:
    Stream(Document *document): Dictionary(document), subtract_from_count(0) {}
    void add_string(const std::string &s, bool count=true)
    {
      for(std::string::const_iterator i=s.begin(); i!=s.end(); i++)
	stream.push_back(*i);
      if(!count)
	subtract_from_count+=s.length();
    }
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0) 
    {
      Part::write(out, binary, offset);
      set_entry("Length", 
		manage(new Integer(stream.size()-subtract_from_count,
				   document_)));
      offset=Dictionary::write(out, binary, offset);
      std::string tmp="stream\r\n";
      offset+=tmp.length();
      if(out) (*out) << tmp; 
      if(out)
	for(A_Stream::const_iterator i=stream.begin(); i!=stream.end(); i++)
	  (*out) << *i;
      offset+=stream.size();
      tmp="\r\nendstream\r\n";
      if(out) (*out) << tmp; 
      offset+=tmp.length();
      return offset;
    }
  private:
    typedef std::vector<unsigned char> A_Stream;
    A_Stream stream;
    int subtract_from_count;
  };

  class Content: public Stream{ // only text so far
  public:   
    Content(Document *document): 
      Stream(document), last_xpos(0), last_ypos(0)
    {add_string("BT\r\n");}
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0)
    {
      add_string("ET");
      return Stream::write(out, binary, offset);
    }
    void selectfont(int size) {add_string("/F1 "+tostr(size)+" Tf\r\n");}
    void moveto(float xpos, float ypos) 
    {
      add_string(tostr(xpos-last_xpos)+" "
		 +tostr(ypos-last_ypos)+" Td\r\n");
      last_xpos=xpos; last_ypos=ypos;
    }
    void rmoveto(float dx, float dy) {} // no easy way to do this
    void show(Glib::ustring s) {
      std::string iso; /// \todo PDF doesn't really use iso-latin-1
      try { 
	iso = Glib::convert_with_fallback(s.raw(), 
					  "ISO-8859-1", "UTF-8");
      } catch(Glib::ConvertError e) {
	iso = "????";
      }
      add_string("(" + iso + ") Tj\r\n");
    } 
    /// \todo chack for unallowed characters
    void whitespace(float width) 
    {add_string("["+tostr((int) -width)+"( )] TJ\r\n");}

  private:
    float last_xpos, last_ypos;
  };

  class Page: public Dictionary{
  public:
    Page(int width, int height, Document *document):
      Dictionary(document), content(0)
    {
      set_entry("Type",
		manage(new Name("Page", document_)));

      Dictionary *resources=manage(new Dictionary(document_));
      Dictionary *font_resource=manage(new Dictionary(document_));
      Dictionary *font=manage(new Dictionary(document_));
      document_->get_xrefs()->add_object(font);
      font->set_entry("Type", 
		      manage(new Name("Font", document_)));
      font->set_entry("Subtype",
		      manage(new Name("Type1", document_)));
      font->set_entry("Name",
		      manage(new Name("F1", document_)));
      font->set_entry("BaseFont",
		      manage(new Name("Helvetica", document_)));
      font->set_entry("Encoding",
		      manage(new Name("MacRomanEncoding", document_)));
      font_resource->set_entry("F1", font->get_ref());
      resources->set_entry("Font", font_resource);
      set_entry("Resources", resources);

      Array *mediabox=manage(new Array(document_));
      mediabox->push_back(manage(new Integer(0, document_)));
      mediabox->push_back(manage(new Integer(0, document_)));
      mediabox->push_back(manage(new Integer(width, document_)));
      mediabox->push_back(manage(new Integer(height, document_)));
      set_entry("MediaBox", mediabox);
    }
    Content *get_content()
    {
      if(!content)
	{
	  content=manage(new Content(document_));
	  document_->get_xrefs()->add_object(content);
	  set_entry("Contents", content->get_ref());
	}
      return content;
    }
    void set_parent(Ref *ref)
    {
      set_entry("Parent", ref);
    }
  private:
    Content *content;  
  };

  class Trailer: public Part {
  public:
    Trailer(Document *document): Part(document) {}
    void set_num_of_refs(long num_of_refs_) {num_of_refs=num_of_refs_;}
    void set_catalog_ref(Ref* catalog_ref_) {catalog_ref=catalog_ref_;}
    void set_xref_offset(long xref_offset_) {xref_offset=xref_offset_;}    
    bytecount write(std::ostream *out, bool binary=true, bytecount offset=0) 
    {
      Part::write(out, binary, offset);
      std::string tmp="trailer\r\n"; 
      if(out)
	(*out) << tmp; 
      offset+=tmp.length();
      Dictionary dict(document_);
      Integer num_of_refs_object(num_of_refs, document_);
      dict.set_entry("Size", &num_of_refs_object);
      dict.set_entry("Root", catalog_ref);
      offset=dict.write(out, binary, offset);
      tmp="startxref\r\n"+tostr(xref_offset)+"\r\n%%EOF\r\n";
      if(out)
	(*out) << tmp; 
      offset+=tmp.length();
      return offset;
    }

  private:
    long num_of_refs;
    Ref* catalog_ref;
    bytecount xref_offset;
  };

  Document::Document():
    Part(this)
  {
    xrefs=manage(new XRefs(this));
  }

  Ref &XRefs::create_ref(Object *object) 
  {
    refs.push_back(manage(new Ref(*this, *object, document_))); 
    return *refs.back();
  }

  bytecount XRefs::write(std::ostream *out, bool binary, 
			 bytecount offset) 
  {
    Part::write(out, binary, offset);
    
    for(Objects::iterator i=objects.begin(); i!=objects.end(); i++)
      {
	bytecount saved_offset=offset;
	Ref *ref=(*i)->get_ref();
	std::string tmp=tostr(ref->get_num()) + " " 
	  + tostr(ref->get_generation()) + " obj\r\n"; 
	if(out)
	  (*out) << tmp; 
	offset+=tmp.length();
	offset=(*i)->write(out, binary, offset);
	(*i)->set_offset(saved_offset); // override write
	tmp="endobj\r\n";
	if(out)
	  (*out) << tmp; 
	offset+=tmp.length();
      }
    
    xref_offset=offset;
    
    std::string tmp="xref\r\n0 "+tostr(refs.size()+1)+"\r\n";
    offset+=tmp.length();
    if(out)
      {
	(*out) << tmp;
	(*out) << std::setw(10) << std::setfill('0') << 0 << " " 
	       << std::setw(0) << std::setfill(' ') << "65535 f\r\n";
	for(Objects::iterator i=objects.begin(); i!=objects.end(); i++)
	  (*out) << std::setw(10) << std::setfill('0') << (*i)->get_offset() 
		 << " " 
		 << std::setw(0) << std::setfill(' ') << "00000 n\r\n";
      }
    offset+=refs.size()*20; // an entry is exactly 20 bytes long
    return offset;      
  }
  
  void Document::selectfont(int size)
  {
    pages.back()->get_content()->selectfont(size);
  }

  void Document::moveto(float xpos, float ypos)
  {
    pages.back()->get_content()->moveto(xpos, ypos);
  }

  void Document::rmoveto(float dx, float dy)
  {
    pages.back()->get_content()->rmoveto(dx, dy);
  }

  void Document::show(Glib::ustring s)
  {
    pages.back()->get_content()->show(s);
  }

  void Document::whitespace(float width)
  {
    pages.back()->get_content()->whitespace(width);
  }

  void Document::add_page(int width, int height)
  {
    pages.push_back(dynamic_cast<Page*>(xrefs->add_object(manage(new Page(width, height, this)))));
  }

  bytecount Document::write(std::ostream *out, bool binary, 
			    bytecount offset)
  {
    {
      Header header(this);
      header.set_version();
      offset=header.write(out, binary, offset);
    }
    
    Dictionary catalog(this);
    Dictionary page_tree_root(this);    
    xrefs->add_object(&catalog);
    xrefs->add_object(&page_tree_root);

    catalog.set_entry("Type", manage(new Name("Catalog", this)));
    catalog.set_entry("Pages", page_tree_root.get_ref());
    
    page_tree_root.set_entry("Type", 
			     manage(new Name("Pages", this)));
    page_tree_root.set_entry("Count", 
			     manage(new Integer(pages.size(), this)));
    Array kids(this);
    for(Pages::iterator i=pages.begin(); i!=pages.end(); i++)
      kids.push_back((*i)->get_ref());
    page_tree_root.set_entry("Kids", &kids);

    offset=xrefs->write(out, binary, offset);
    
    Trailer trailer(this);
    trailer.set_num_of_refs(xrefs->get_num_of_refs());
    trailer.set_catalog_ref(catalog.get_ref());
    trailer.set_xref_offset(xrefs->get_xref_offset());
    offset=trailer.write(out, binary, offset);

    return offset;
  }

};

#ifdef STAND_ALONE
int main(int argc, char *argv[])
{
  PDF::Document document;
  document.add_page(200, 300);
  document.selectfont(20);
  document.moveto(20, 20);
  document.show("Hello, World!");
  std::cerr<<document.write(&std::cout)<<" bytes written"<<std::endl;
}
#endif
