///
// Copyright (C) 2003, 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "postscriptviewent.h"
#include "config.h"
#include "document/basicframe.h" // For Gen_Pic_Error; Todo: remove!
#include "util/tempfile.h"
#include "util/processman.h"
#include "util/os.h"
#include "util/filesys.h"
#include "util/warning.h"
#include <sstream>

/// \todo Remote the following line!
typedef BasicFrame::GenPicError GenPicError;

PostscriptViewent::PostscriptViewent(View& view, const FrameRef rasterframe,
				     int _scale_factor)
  : Viewent(view), frame(rasterframe), 
    pid(-1), tmp_file(Tempfile::find_new_name()),
    scale_factor(_scale_factor),
    content_is_broken(false),
    procstopcon(ProcessManager::instance().process_stopped.connect
		(slot(*this, &PostscriptViewent::process_stopped))),
    /// \todo  Only clear when necessary, i.e. not when just moving ...
    propscon(frame->props_changed_signal.connect
	     (SigC::hide<Pagent*>(slot(*this, &PostscriptViewent::clear)))),
    objectcon(frame->object_changed_signal.connect
	      (SigC::hide<Pagent*>(slot(*this, &PostscriptViewent::clear)))),
    zoomcon(view.connect_zoom_change
	    (SigC::hide<float>(slot(*this, &PostscriptViewent::clear))))
    
{}

PostscriptViewent::~PostscriptViewent() {
  try { unlink(tmp_file); }
  catch(std::exception& err) {
    warning << "Failed to delete postscript view render file." << err.what()
	    << std::endl;
  }
  if(pid != -1) {
    ProcessManager::instance().stop(pid);
  }
}

void PostscriptViewent::clear() {
  pixbuf.clear();
  if(pid != -1) {
    verbose << "PostscriptViewent stopping gs " << pid << std::endl;
    ProcessManager::instance().stop(pid);
    pid = -1;
  }
}

void PostscriptViewent::set_image(const std::string& psfile) {
  clear();			// make sure we don't have or create image

  std::string &psinterpreter = config.PSInterpreter.values.front();
  if(psinterpreter.empty())
    throw GenPicError(BasicFrame::GENERATION,
			"No PostScript interpreter specified");
  
  const Matrix &m = frame->get_matrix();
  // Size of frame as pixels in this view.
  // get_width() / get_height() returns the unscaled size (yes, I am
  // confused myself)
  const Gdk::Point pv(int(view.pt2scr(frame->get_width() * m.sc_x())),
		      int(view.pt2scr(frame->get_height() * m.sc_y())));
  if(pv.get_x() == 0 || pv.get_y() == 0) 
    throw GenPicError(BasicFrame::ZEROSIZE, "Picture has zero size");

  std::ostringstream tmp;
  /// \todo -sFONTPATH parameter ?
  tmp << psinterpreter //assuming it is ghostscript
      << " -q -dSAFER -dNOPAUSE -dBATCH -sDEVICE=ppmraw -r" 
      << view.get_scrres() * scale_factor
      << " -g" << scale_factor * pv.get_x() << 'x' << scale_factor * pv.get_y()
      << " -sOutputFile=- " << psfile
      << " > " << tmp_file;
  pid = ProcessManager::instance().start(tmp.str());
  
  if(pid == -1)
    throw GenPicError(BasicFrame::GENERATION,
		      "Failed to run " + psinterpreter);

  verbose << pid << ": " << tmp.str() << std::endl;
}

bool PostscriptViewent::draw_content() {
  
  if(content_is_broken) {
    draw_content_broken();
    return true;
  }
  if(pixbuf) {
    // p is the top left corner, since that is what draw_pixmap wants.
    const Gdk::Point p = view.pt2scr(frame->get_matrix().transform
				     (Vector(0, frame->get_height())));
    view.get_win()->draw_pixbuf(view.get_gc(), pixbuf, 0, 0,
				p.get_x(), p.get_y(), -1, -1,
				Gdk::RGB_DITHER_NORMAL, 0, 0);
    //    view.get_win()->draw_line(view.get_gc(), p.get_x(), p.get_y(), 0, 0);
    return true;
  } else if(pid != -1) {
    // Allready running gs ...
    draw_content_wait();
    return false;
  } else {
    /// \todo is_rehaping has been removed, what to do?
    if(true /* !reshaping */)  {	// generate_picture takes too long time
      try {
	draw_content_wait();
	frame->generate_picture(SigC::slot
				(*this, &PostscriptViewent::set_image));
      } catch(GenPicError e) {
	debug << frame->get_name() << ": " << e.what() << std::endl;
	switch(e.type) {
	case BasicFrame::ASSOCIATION: draw_content_missing(); break;
	case BasicFrame::GENERATION:  draw_content_broken();  break;
	case BasicFrame::ZEROSIZE:                            break;
	}
      }
    } else // if reshaping
      draw_content_wait();
    return false;
  }
}

void PostscriptViewent::process_stopped(pid_t _pid, bool exited_normally, 
					int exit_code)
{
  if(_pid != pid)
    return;

  debug << _pid << ", exit code: " << exit_code << std::endl;
  if(!exited_normally) {
    warning << "Process " << _pid << " exited abnormally" << std::endl;
    return;
  }

  pixbuf.clear();
  if(exists(tmp_file)) {
    try {
      pixbuf = Gdk::Pixbuf::create_from_file(tmp_file);

      // Make all white pixels transparent.
      if(config.FakeTransparency.values.front()
	 && pixbuf && !pixbuf->get_has_alpha())
	pixbuf = pixbuf->add_alpha(true, 255, 255, 255); 
      // had some trouble with some images becoming totally transparent

      // scale
      if(scale_factor != 1)
	pixbuf = pixbuf->scale_simple(pixbuf->get_width() / scale_factor,
				      pixbuf->get_height() / scale_factor,
				      Gdk::INTERP_BILINEAR);

    } catch (const Glib::Error& e) { 
      warning << frame->get_name() << ": " << e.what() << std::endl; 
    }
  
  } else warning << frame->get_name() << ": " << tmp_file 
		 << " does not exist" << std::endl; 
  
  content_is_broken = !pixbuf;
  pid = -1;
  frame->ready_to_draw_signal(frame);
}
