///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "page.h"
#include "fileerrors.h"
#include "loader.h"
#include "document.h"		/// \todo  Remove if/when possible
#include "util/stringutil.h"
#include "util/warning.h"
#include <algorithm>

SigC::Signal1<void, Pagent*> Page::ready_to_draw_page_signal;

#ifdef PNUM_HACK
namespace {
  struct PNumHack { //page numbering hack
    static const int xpos=40; //points
    static const int ypos=40; //points
    static std::string font;
    static const int font_size=10;
  } pnum_hack;

  //  std::string PNumHack::font="Palatino-Italic";
  std::string PNumHack::font="URWPalladioL-Ital";
}
#endif

Guide::Guide(const xmlpp::Element& node)
  : is_horizontal(false)
{
  if(const xmlpp::Attribute* orientation = node.get_attribute("orientation")) {
    const std::string o = orientation->get_value();
    if(o == "h")
      is_horizontal = true;
    else if(o != "v") // "v" is default
      throw Error::Read("\"" + o + "\" is not a proper " 
			"guide orientation.\n"
			"Allowed values are \"v\" and \"h\"");
  }
  if(const xmlpp::Attribute* position_attr = node.get_attribute("pos"))
    try {
      position = to<float>(position_attr->get_value());
    } catch(std::runtime_error e) {
      throw Error::Read("Bad guide position: " 
			+ position_attr->get_value());
    }
  else
    throw Error::Read("Guide position is undefined");
}

int Page::get_page_num() const {
  return document.get_page_num_of_page(this);
}

Page::Page(Document& d)
  :Group(0, ""),
   document(d)
{}

Page::Page(const xmlpp::Element& node, Document& d, const FileContext &context)
  :Group(0, ""),
   document(d)
{
  if(const xmlpp::Attribute* name_attr = node.get_attribute("name"))
    set_name(name_attr->get_value());
  
  const xmlpp::Node::NodeList children = node.get_children();
  for(xmlpp::Node::NodeList::const_iterator i = children.begin();
      i != children.end(); i++) {
    if(const xmlpp::Element* elem = dynamic_cast<const xmlpp::Element*>(*i)) {
      const std::string name = elem->get_name();
      if(name == "frame") { 
	add(load(*elem, this, context));
	
      } else if(name == "guide") {
	guides.push_back(Guide(*elem));
	
      } else {
	warning << "Unknown node <" << name
		<< "> ignored in <page>." << std::endl;
      }
    }
  }
  /// \todo Maybe throw an error if *i was a text node (but not if it was a
  /// comment) ...
}

Page::~Page()
{}

xmlpp::Element *Page::save(xmlpp::Element& parent_node,
			   const FileContext &context) 
{
  xmlpp::Element *node = parent_node.add_child("page");
  for(Guides::const_iterator i = guides.begin(); i != guides.end(); i++) {
    xmlpp::Element *guide_node = node->add_child("guide");
    guide_node->set_attribute("orientation", i->is_horizontal ? "h" : "v");
    guide_node->set_attribute("pos", tostr(i->position));
  }
  save_childs(*node, context);
  if(!name.empty())
    node->set_attribute("name", name);
  return node;
}

void Page::print(std::ostream &out, bool grayscale) {
#ifdef PNUM_HACK
  // Page numbering hack:
  {
    int page_num=get_page_num();
    bool odd=page_num%2;
    int xpos=odd?(int) get_width()-pnum_hack.xpos:pnum_hack.xpos;
    int ypos=pnum_hack.ypos;
    out << '/' << pnum_hack.font << " findfont "
	<< pnum_hack.font_size << " scalefont setfont" << std::endl
	<< xpos << ' ' << ypos << " moveto (" << page_num << ") show"
	<< std::endl;
  }
#endif

  Group::print(out, grayscale);
  
  out << std::endl << "showpage"
      << std::endl << std::endl;
}

std::string Page::get_name() {
  /// \todo is this a bad idea?
  return name.empty() ? tostr(get_page_num()) : std::string(name);
}

float Page::get_width() const {
  return document.get_width();
}

float Page::get_height() const {
  return document.get_height();
}

float Page::get_xpos() const {
  return 0;
}

float Page::get_ypos() const {
  return 0;
}

void Page::addObject(Pagent* obj) {
  add(obj);
  Document::containing(*this).select(obj);
}

void Page::child_ready_to_draw(Pagent *pagent) {
  if(pagent)
    debug << pagent->get_name() << " ready to draw itself" << std::endl;
  ready_to_draw_page_signal(this);
}

void Page::select_all(bool select) {
  document.select_all_on_page(this, select);
}

Page& Page::containing(Pagent& obj) {
  try {
    if(Page* result = dynamic_cast<Page*>(&obj))
      return *result;
    else
      return Page::containing(obj.get_parent());
  } catch(const Error::NoParent& err) {
    throw std::logic_error
      ("Tried to get Page containing pagent that was not on a Page.");
  }
}

const Page& Page::containing(const Pagent& obj) {
  try {
    if(const Page* result = dynamic_cast<const Page*>(&obj))
      return *result;
    else
      return Page::containing(obj.get_parent());
  } catch(const Error::NoParent& err) {
    throw std::logic_error
      ("Tried to get Page containing pagent that was not on a Page.");
  }
}
