/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 *  liboaf: A library for accessing oafd in a nice way.
 *
 *  Copyright (C) 1999, 2000 Red Hat, Inc.
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#ifndef LIBOAF_H
#define LIBOAF_H 1

#ifdef __cplusplus
extern "C" {
#endif

#include <liboaf/liboaf-version.h>
#include <liboaf/oaf.h>

/* Need to conditionalize this */
#include <orb/orbit.h>

/* If you wish to manipulate the internals of this structure, please
   use g_malloc/g_free to allocate memory. */
typedef struct
{
	char *iid;		/* Implementation ID */
	char *user;		/* You owe this field to boc */
	char *host;		/* DNS name or IP address */
	char *domain;		/* This is not a DNS domain, but an activation domain */
}
OAFActivationInfo;

OAF_ActivationID oaf_actinfo_stringify (const OAFActivationInfo * actinfo);
OAFActivationInfo *oaf_servinfo_to_actinfo (const OAF_ServerInfo * servinfo);
OAFActivationInfo *oaf_actid_parse (const OAF_ActivationID actid);
OAFActivationInfo *oaf_actinfo_new (void);
void oaf_actinfo_free (OAFActivationInfo * actinfo);

gboolean oaf_is_initialized (void);
CORBA_ORB oaf_init (int argc, char **argv);
CORBA_ORB oaf_orb_get (void);
CORBA_Context oaf_context_get (void);	/* Just makes getting hold of the default context a bit easier */

#define oaf_username_get() g_get_user_name()
const char *oaf_hostname_get (void);
const char *oaf_session_name_get (void);
const char *oaf_domain_get (void);

typedef struct
{
	const char *iid;

	/* This routine should call oaf_plugin_use(servant, impl_ptr), 
         * as should all routines which activate CORBA objects
	 * implemented by this shared library. This needs to be done 
         * before making any CORBA calls on the object, or
	 * passing that object around. First thing after servant creation 
         * always works. :) 
         */

        CORBA_Object (*activate) (PortableServer_POA poa,
                                  const char *iid, 
                                  gpointer impl_ptr,	/* This pointer should be stored by the implementation
                                                         * to be passed to oaf_plugin_unuse() in the 
                                                         * implementation's destruction routine. */
				    CORBA_Environment * ev);
}
OAFPluginObject;

typedef struct
{
	const OAFPluginObject *plugin_object_list;
	const char *description;
}
OAFPlugin;

void oaf_plugin_use (PortableServer_Servant servant, gpointer impl_ptr);
void oaf_plugin_unuse (gpointer impl_ptr);

CORBA_Object oaf_server_activate_shlib (OAF_ActivationResult * sh, 
                                        CORBA_Environment * ev);


OAF_ServerInfoList *oaf_query (const char *requirements,
			       char *const *selection_order,
			       CORBA_Environment * ev);
CORBA_Object oaf_activate (const char *requirements,
			   char *const *selection_order,
			   OAF_ActivationFlags flags,
			   OAF_ActivationID * ret_aid,
			   CORBA_Environment * ev);
CORBA_Object oaf_activate_from_id (const OAF_ActivationID aid,
				   OAF_ActivationFlags flags,
				   OAF_ActivationID * ret_aid,
				   CORBA_Environment * ev);

/* oaf-registration.c - not intended for application use */
typedef struct
{
	const char *name;
	const char *session_name;
	const char *username, *hostname, *domain;
}
OAFRegistrationCategory;

typedef struct _OAFRegistrationLocation OAFRegistrationLocation;
struct _OAFRegistrationLocation
{
	void (*lock) (const OAFRegistrationLocation * regloc,
		      gpointer user_data);
	void (*unlock) (const OAFRegistrationLocation * regloc,
			gpointer user_data);
	char *(*check) (const OAFRegistrationLocation * regloc,
			const OAFRegistrationCategory * regcat,
			int *ret_distance, gpointer user_data);

	void (*register_new) (const OAFRegistrationLocation * regloc,
			      const char *ior,
			      const OAFRegistrationCategory * regcat,
			      gpointer user_data);
	void (*unregister) (const OAFRegistrationLocation * regloc,
			    const char *ior,
			    const OAFRegistrationCategory * regcat,
			    gpointer user_data);
};

void oaf_registration_location_add (const OAFRegistrationLocation * regloc,
				    int priority, gpointer user_data);

CORBA_Object oaf_registration_check (const OAFRegistrationCategory * regcat,
				     CORBA_Environment * ev);
void oaf_registration_set (const OAFRegistrationCategory * regcat,
			   CORBA_Object obj, CORBA_Environment * ev);
void oaf_registration_unset (const OAFRegistrationCategory * regcat,
			     CORBA_Object obj, CORBA_Environment * ev);

/* Do not release() the returned value */
CORBA_Object oaf_service_get (const OAFRegistrationCategory * regcat);

typedef CORBA_Object (*OAFServiceActivator) (const OAFRegistrationCategory *
					     regcat, const char **cmd,
					     int ior_fd,
					     CORBA_Environment * ev);
void oaf_registration_activator_add (OAFServiceActivator act_func,
				     int priority);

/* oaf-servreg.c */
OAF_RegistrationResult oaf_active_server_register (const char *iid,
						   CORBA_Object obj);
void oaf_active_server_unregister (const char *iid, CORBA_Object obj);

/* oaf-util.c */
OAF_Property *oaf_server_info_prop_find (OAF_ServerInfo * server,
                                         const char *prop_name);
const char *oaf_server_info_prop_lookup (OAF_ServerInfo * server,
					 const char *prop_name,
					 GSList * i18n_languages);
void OAF_PropertyValue_copy (OAF_PropertyValue *copy,
                             const OAF_PropertyValue *original);
void OAF_Property_copy (OAF_Property *copy,
                        const OAF_Property *original);
void
CORBA_sequence_OAF_Property_copy (CORBA_sequence_OAF_Property *copy,
                                  const CORBA_sequence_OAF_Property *original);
void
OAF_ServerInfo_copy (OAF_ServerInfo *copy, const OAF_ServerInfo *original);
OAF_ServerInfo *
OAF_ServerInfo_duplicate (const OAF_ServerInfo *original);

/* Optional stuff for libgnome to use */
#ifdef HAVE_POPT_H
#include <popt.h>
#endif

#ifdef POPT_AUTOHELP
extern struct poptOption oaf_popt_options[];
#endif

CORBA_ORB oaf_orb_init (int *argc, char **argv);
void oaf_preinit (gpointer app, gpointer mod_info);
void oaf_postinit (gpointer app, gpointer mod_info);
const char *oaf_activation_iid_get (void);
extern const guint liboaf_major_version,
	liboaf_minor_version, liboaf_micro_version;
extern const char liboaf_version[];

CORBA_Object oaf_name_service_get (CORBA_Environment * ev);

#ifdef __cplusplus
}
#endif

#endif /* LIBOAF_H */

