/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* fm-directory-view.h
 *
 * Copyright (C) 1999, 2000  Free Software Foundaton
 * Copyright (C) 2000  Eazel, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Ettore Perazzoli
 */

#ifndef FM_DIRECTORY_VIEW_H
#define FM_DIRECTORY_VIEW_H

#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkscrolledwindow.h>
#include <libnautilus-extensions/nautilus-background.h>
#include <libnautilus-extensions/nautilus-directory.h>
#include <libnautilus-extensions/nautilus-file.h>
#include <libnautilus-extensions/nautilus-icon-container.h>
#include <libnautilus-extensions/nautilus-link.h>
#include <libnautilus-extensions/nautilus-string-list.h>
#include <libnautilus/nautilus-view.h>

typedef struct FMDirectoryView FMDirectoryView;
typedef struct FMDirectoryViewClass FMDirectoryViewClass;

/* Paths to use when referring to bonobo menu items.
 * These are the new ones defined by FMDirectoryView. The
 * Nautilus-wide ones are in <libnautilus/nautilus-bonobo-ui.h>
 * Note that this may change as we complete the switchover to the
 * new Bonobo UI code.
 */
#define FM_DIRECTORY_VIEW_COMMAND_OPEN                      	"/commands/Open"
#define FM_DIRECTORY_VIEW_COMMAND_OPEN_IN_NEW_WINDOW        	"/commands/OpenNew"
#define FM_DIRECTORY_VIEW_COMMAND_OPEN_WITH			"/commands/Open With"
#define FM_DIRECTORY_VIEW_COMMAND_NEW_FOLDER			"/commands/New Folder"
#define FM_DIRECTORY_VIEW_COMMAND_DELETE                    	"/commands/Delete"
#define FM_DIRECTORY_VIEW_COMMAND_TRASH                    	"/commands/Trash"
#define FM_DIRECTORY_VIEW_COMMAND_EMPTY_TRASH                   "/commands/Empty Trash"
#define FM_DIRECTORY_VIEW_COMMAND_DUPLICATE                	"/commands/Duplicate"
#define FM_DIRECTORY_VIEW_COMMAND_CREATE_LINK                	"/commands/Create Link"
#define FM_DIRECTORY_VIEW_COMMAND_SHOW_PROPERTIES         	"/commands/Show Properties"
#define FM_DIRECTORY_VIEW_COMMAND_RESET_BACKGROUND		"/commands/Reset Background"
#define FM_DIRECTORY_VIEW_COMMAND_REMOVE_CUSTOM_ICONS		"/commands/Remove Custom Icons"
#define FM_DIRECTORY_VIEW_COMMAND_OTHER_APPLICATION    		"/commands/OtherApplication"
#define FM_DIRECTORY_VIEW_COMMAND_OTHER_VIEWER	   		"/commands/OtherViewer"


#define FM_TYPE_DIRECTORY_VIEW			(fm_directory_view_get_type ())
#define FM_DIRECTORY_VIEW(obj)			(GTK_CHECK_CAST ((obj), FM_TYPE_DIRECTORY_VIEW, FMDirectoryView))
#define FM_DIRECTORY_VIEW_CLASS(klass)		(GTK_CHECK_CLASS_CAST ((klass), FM_TYPE_DIRECTORY_VIEW, FMDirectoryViewClass))
#define FM_IS_DIRECTORY_VIEW(obj)		(GTK_CHECK_TYPE ((obj), FM_TYPE_DIRECTORY_VIEW))
#define FM_IS_DIRECTORY_VIEW_CLASS(klass)	(GTK_CHECK_CLASS_TYPE ((klass), FM_TYPE_DIRECTORY_VIEW))

typedef struct FMDirectoryViewDetails FMDirectoryViewDetails;

struct FMDirectoryView {
	GtkScrolledWindow parent;
	FMDirectoryViewDetails *details;
};

struct FMDirectoryViewClass {
	GtkScrolledWindowClass parent_class;

	/* The 'clear' signal is emitted to empty the view of its contents.
	 * It must be replaced by each subclass.
	 */
	void 	(* clear) 		 (FMDirectoryView *view);
	
	/* The 'begin_adding_files' signal is emitted before a set of files
	 * are added to the view. It can be replaced by a subclass to do any 
	 * necessary preparation for a set of new files. The default
	 * implementation does nothing.
	 */
	void 	(* begin_adding_files) (FMDirectoryView *view);
	
	/* The 'add_file' signal is emitted to add one file to the view.
	 * It must be replaced by each subclass.
	 */
	void    (* add_file) 		 (FMDirectoryView *view, 
					  NautilusFile *file);

	/* The 'file_changed' signal is emitted to signal a change in a file,
	 * including the file being removed.
	 * It must be replaced by each subclass.
	 */
	void 	(* file_changed)         (FMDirectoryView *view, 
					  NautilusFile *file);

	/* The 'done_adding_files' signal is emitted after a set of files
	 * are added to the view. It can be replaced by a subclass to do any 
	 * necessary cleanup (typically, cleanup for code in begin_adding_files).
	 * The default implementation does nothing.
	 */
	void 	(* done_adding_files)    (FMDirectoryView *view);
	
	/* The 'begin_loading' signal is emitted before any of the contents
	 * of a directory are added to the view. It can be replaced by a 
	 * subclass to do any necessary preparation to start dealing with a
	 * new directory. The default implementation does nothing.
	 */
	void 	(* begin_loading) 	 (FMDirectoryView *view);

	/* The 'create_selection_context_menu_items' signal is emitted 
	 * when creating a context menu for the selected items. @files is
	 * the list of selected files; it isn't destroyed until the menu
	 * is destroyed so it can be used in callbacks.
	 * Subclasses might want to override this function to
	 * modify the menu contents.
	 */
	void 	(* create_selection_context_menu_items) 	 
					 (FMDirectoryView *view,
					  GtkMenu *menu,
					  GList *files);

	/* The 'create_background_context_menu_items' signal is emitted 
	 * when creating a context menu, either an item-specific one or
	 * a background one. Subclasses might want to override this to
	 * modify the menu contents.
	 */
	void 	(* create_background_context_menu_items) 	 
					 (FMDirectoryView *view,
					  GtkMenu *menu);
	 
	/* Function pointers that don't have corresponding signals */

	/* get_selection is not a signal; it is just a function pointer for
	 * subclasses to replace (override). Subclasses must replace it
	 * with a function that returns a newly-allocated GList of
	 * NautilusFile pointers.
	 */
	GList *	(* get_selection) 	 	(FMDirectoryView *view);
	
        /* bump_zoom_level is a function pointer that subclasses must override
         * to change the zoom level of an object. */
        void    (* bump_zoom_level)      	(FMDirectoryView *view,
					  	 int zoom_increment);

        /* zoom_to_level is a function pointer that subclasses must override
         * to set the zoom level of an object to the specified level. */
        void    (* zoom_to_level) 		(FMDirectoryView *view, 
        				         gint 		 level);

        /* restore_default_zoom_level is a function pointer that subclasses must override
         * to restore the zoom level of an object to a default setting. */
        void    (* restore_default_zoom_level) (FMDirectoryView *view);

        /* can_zoom_in is a function pointer that subclasses must override to
         * return whether the view is at maximum size (furthest-in zoom level) */
        gboolean (* can_zoom_in)	 	(FMDirectoryView *view);

        /* can_zoom_out is a function pointer that subclasses must override to
         * return whether the view is at minimum size (furthest-out zoom level) */
        gboolean (* can_zoom_out)	 	(FMDirectoryView *view);
        
        /* select_all is a function pointer that subclasses must override to
         * select all of the items in the view */
        void     (* select_all)	         	(FMDirectoryView *view);

        /* set_selection is a function pointer that subclasses must
         * override to select the specified items (and unselect all
         * others). The argument is a list of NautilusFiles. */

        void     (* set_selection)	 	(FMDirectoryView *view, 
        					 GList *selection);

        /* reveal_selection is a function pointer that subclasses may
         * override to make sure the selected items are sufficiently
         * apparent to the user (e.g., scrolled into view). By default,
         * this does nothing.
         */
        void     (* reveal_selection)	 	(FMDirectoryView *view);

        /* get_background is a function pointer that subclasses must
         * override to return the NautilusBackground for this view.
         */
        GtkWidget * (* get_background_widget)	(FMDirectoryView *view);

        /* merge_menus is a function pointer that subclasses can override to
         * add their own menu items to the window's menu bar.
         * If overridden, subclasses must call parent class's function.
         */
        void    (* merge_menus)         	(FMDirectoryView *view);

        /* update_menus is a function pointer that subclasses can override to
         * update the sensitivity or wording of menu items in the menu bar.
         * It is called (at least) whenever the selection changes. If overridden, 
         * subclasses must call parent class's function.
         */
        void    (* update_menus)         	(FMDirectoryView *view);

	/* get_emblem_names_to_exclude is a function pointer that subclasses
	 * may override to specify a set of emblem names that should not
	 * be displayed with each file. By default, all emblems returned by
	 * NautilusFile are displayed.
	 */
	NautilusStringList * (* get_emblem_names_to_exclude)	(FMDirectoryView *view);

	/* supports_properties is a function pointer that subclasses may
	 * override to control whether the "Show Properties" menu item
	 * should be enabled for selected items. The default implementation 
	 * returns TRUE.
	 */
	gboolean (* supports_properties)	(FMDirectoryView *view);

	/* supports_zooming is a function pointer that subclasses may
	 * override to control whether or not the zooming control and
	 * menu items should be enabled. The default implementation
	 * returns TRUE.
	 */
	gboolean (* supports_zooming)		(FMDirectoryView *view);

	/* is_read_only is a function pointer that subclasses may
	 * override to control whether or not the user is allowed to
	 * change the contents of the currently viewed directory. The
	 * default implementation checks the permissions of the
	 * directory.
	 */
	gboolean (* is_read_only)	(FMDirectoryView *view);

	/* is_empty is a function pointer that subclasses must
	 * override to report whether the view contains any items.
	 */
	gboolean (* is_empty)	(FMDirectoryView *view);

	/* supports_creating_files is a function pointer that subclasses may
	 * override to control whether or not new items can be created.
	 * be accepted. The default implementation checks whether the
	 * user has write permissions for the viewed directory, and whether
	 * the viewed directory is in the trash.
	 */
	gboolean (* supports_creating_files)	(FMDirectoryView *view);

	/* accepts_dragged_files is a function pointer that subclasses may
	 * override to control whether or not files can be dropped in this
	 * location. The default implementation returns TRUE.
	 */
	gboolean (* accepts_dragged_files)	(FMDirectoryView *view);

	void	(* start_renaming_item)	 	(FMDirectoryView *view,
					  	 const char *uri);

	/* Preference change callbacks, overriden by icon and list views. 
	 * Icon and list views respond by synchronizing to the new preference
	 * values and forcing an update if appropriate.
	 */
	void	(* text_attribute_names_changed)(FMDirectoryView *view);
	void	(* embedded_text_policy_changed)(FMDirectoryView *view);
	void	(* image_display_policy_changed)(FMDirectoryView *view);
	void	(* font_family_changed)		(FMDirectoryView *view);
	void	(* click_policy_changed)	(FMDirectoryView *view);
	void	(* smooth_graphics_mode_changed)(FMDirectoryView *view);
};

/* GtkObject support */
GtkType            fm_directory_view_get_type                       (void);

/* Component embedding support */
NautilusView *     fm_directory_view_get_nautilus_view              (FMDirectoryView       *view);

/* Functions callable from the user interface and elsewhere. */
char *             fm_directory_view_get_uri		            (FMDirectoryView       *view);
gboolean           fm_directory_view_can_accept_item                (NautilusFile          *target_item,
								     const char            *item_uri,
								     FMDirectoryView       *view);
void		   fm_directory_view_display_selection_info	    (FMDirectoryView	   *view);
GList *            fm_directory_view_get_selection                  (FMDirectoryView       *view);
void               fm_directory_view_stop                           (FMDirectoryView       *view);
gboolean           fm_directory_view_can_zoom_in                    (FMDirectoryView       *view);
gboolean           fm_directory_view_can_zoom_out                   (FMDirectoryView       *view);
GtkWidget *	   fm_directory_view_get_background_widget 	    (FMDirectoryView	   *view);
void               fm_directory_view_bump_zoom_level                (FMDirectoryView       *view,
								     int                    zoom_increment);
void               fm_directory_view_zoom_to_level                  (FMDirectoryView        *view,
								     int                    zoom_level);
void               fm_directory_view_set_zoom_level		    (FMDirectoryView        *view,
								     int                    zoom_level);
void               fm_directory_view_restore_default_zoom_level     (FMDirectoryView       *view);
void               fm_directory_view_select_all                     (FMDirectoryView       *view);
void               fm_directory_view_set_selection                  (FMDirectoryView       *view,
								     GList                 *selection);
void               fm_directory_view_reveal_selection               (FMDirectoryView       *view);
gboolean	   fm_directory_view_is_empty			    (FMDirectoryView	   *view);
gboolean	   fm_directory_view_is_read_only		    (FMDirectoryView	   *view);
gboolean	   fm_directory_view_supports_creating_files	    (FMDirectoryView	   *view);
gboolean	   fm_directory_view_accepts_dragged_files	    (FMDirectoryView	   *view);
gboolean	   fm_directory_view_supports_properties	    (FMDirectoryView	   *view);
gboolean	   fm_directory_view_supports_zooming	    	    (FMDirectoryView	   *view);
void               fm_directory_view_move_copy_items                (const GList           *item_uris,
								     GdkPoint              *relative_item_points,
								     const char            *target_uri,
								     int                    copy_action,
								     int                    x,
								     int                    y,
								     FMDirectoryView       *view);
gint               fm_directory_view_get_context_menu_index         (GtkMenu 		   *menu,
								     const char            *verb_path);
gboolean	   fm_directory_link_type_in_selection 		    (FMDirectoryView 	   *view, 
								     NautilusLinkType 	   link_type);

/* Wrappers for signal emitters. These are normally called 
 * only by FMDirectoryView itself. They have corresponding signals
 * that observers might want to connect with.
 */
void               fm_directory_view_clear                          (FMDirectoryView       *view);
void               fm_directory_view_begin_loading                  (FMDirectoryView       *view);

/* Hooks for subclasses to call. These are normally called only by 
 * FMDirectoryView and its subclasses 
 */
void               fm_directory_view_activate_files                 (FMDirectoryView       *view,
								     GList          	   *files);								     
void		   fm_directory_view_start_batching_selection_changes (FMDirectoryView	   *view);
void		   fm_directory_view_stop_batching_selection_changes (FMDirectoryView	   *view);
gboolean	   fm_directory_view_confirm_multiple_windows	    (FMDirectoryView	   *view,
								     int		    window_count);
void		   fm_directory_view_queue_file_change	    	    (FMDirectoryView	   *view,
								     NautilusFile 	   *file);
void               fm_directory_view_notify_selection_changed       (FMDirectoryView       *view);
Bonobo_UIContainer fm_directory_view_get_bonobo_ui_container        (FMDirectoryView       *view);
NautilusStringList *fm_directory_view_get_emblem_names_to_exclude   (FMDirectoryView	   *view);
NautilusDirectory  *fm_directory_view_get_model                     (FMDirectoryView       *view);
NautilusFile       *fm_directory_view_get_directory_as_file         (FMDirectoryView       *view);
NautilusBackground *fm_directory_view_get_background		    (FMDirectoryView 	   *view);
void               fm_directory_view_pop_up_background_context_menu (FMDirectoryView       *view);
void               fm_directory_view_pop_up_selection_context_menu  (FMDirectoryView       *view); 
void               fm_directory_view_update_menus                   (FMDirectoryView       *view);
GtkMenuItem 	  *fm_directory_view_insert_context_menu_item	    (FMDirectoryView 	   *view, 
								     GtkMenu 		   *menu, 
								     const char 	   *label,
								     const char 	   *identifier,
								     int		    position,
								     void 		  (* callback) (GtkMenuItem *, FMDirectoryView *),
								     gboolean 		   sensitive);
void		   fm_directory_view_new_folder			    (FMDirectoryView       *view);

#endif /* FM_DIRECTORY_VIEW_H */
