/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 1999, 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 *
 *  medusa-utils.c -  Utility functions used across medusa
 */


#include <glib.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <string.h>

#include "medusa-utils.h"

static gboolean     directory_name_ends_with_slash        (char *directory_name);

char *
medusa_full_path_from_directory_and_file (char *directory_name,
                                          char *file_name)
{
  
        
        g_return_val_if_fail (directory_name != NULL, NULL);
        g_return_val_if_fail (file_name != NULL, NULL);
        /* This could probably be less horrid with canonical uri's */
        if (directory_name_ends_with_slash (directory_name)) {
                return g_strdup_printf ("%s%s", directory_name, file_name);
        }
        else {
                return g_strdup_printf ("%s/%s", directory_name, file_name);
        }
}

char *
medusa_full_uri_from_directory_and_file (char *directory_uri,
                                         char *file_name)
{
        g_return_val_if_fail (directory_uri != NULL, NULL);
        g_return_val_if_fail (file_name != NULL, NULL);

        if (strcmp (directory_uri, "file:///") == 0) {
                return g_strdup_printf ("%s%s", directory_uri, file_name);
        }
        else {
                return g_strdup_printf ("%s/%s",directory_uri, file_name);
        }
}

static gboolean     
directory_name_ends_with_slash (char *directory_name)
{
        g_return_val_if_fail (directory_name != NULL, FALSE);
        return directory_name[strlen (directory_name) - 1] == '/';

}


gboolean
medusa_group_contains (int gid, int uid)
{
        struct group *group;
        struct passwd *password_entry;
        int i;
        group = getgrgid (gid);
        for (i = 0; group->gr_mem[i] != NULL; i++) {
                password_entry = getpwnam (group->gr_mem[i]);
                if (password_entry->pw_uid == uid) {
                        return TRUE;
                }
        }
        return FALSE;
  
}


gboolean
medusa_string_has_prefix (char *string,
                          char *prefix)
{
        g_return_val_if_fail (string != NULL, FALSE);
        g_return_val_if_fail (prefix != NULL, FALSE);

        return (strncmp (string, prefix, strlen (prefix)) == 0);
}



/**
 * medusa_g_list_partition
 * 
 * Parition a list into two parts depending on whether the data
 * elements satisfy a provided predicate. Order is preserved in both
 * of the resulting lists, and the original list is consumed. A list
 * of the items that satisfy the predicate is returned, and the list
 * of items not satisfying the predicate is returned via the failed
 * out argument.
 * 
 * @list: List to partition.
 * @predicate: Function to call on each element.
 * @user_data: Data to pass to function.  
 * @removed: The GList * variable pinted to by this argument will be
 * set to the list of elements for which the predicate returned
 * false. 
 * Note: this is written by Maciej Stachowiak <mjs@eazel.com> and
 * clipped from nautilus/libnautilus-extensions/nautilus-glib-extensions.c */

GList *
medusa_g_list_partition (GList	          *list,
                         MedusaGPredicateFunc  predicate,
                         gpointer	           user_data,
                         GList                 **failed)
{
	GList *predicate_true;
	GList *predicate_false;
	GList *reverse;
	GList *p;
	GList *next;

	predicate_true = NULL;
	predicate_false = NULL;

	reverse = g_list_reverse (list);

	for (p = reverse; p != NULL; p = next) {
		next = p->next;
		
		if (next != NULL) {
			next->prev = NULL;
		}

		if (predicate (p->data, user_data)) {
			p->next = predicate_true;
 			if (predicate_true != NULL) {
				predicate_true->prev = p;
			}
			predicate_true = p;
		} else {
			p->next = predicate_false;
 			if (predicate_false != NULL) {
				predicate_false->prev = p;
			}
			predicate_false = p;
		}
	}

	*failed = predicate_false;
	return predicate_true;
}


gint32 *             
medusa_intersect_two_descending_integer_lists (gint32 *first_list,
                                               int first_list_entry_count,
                                               gint32 *second_list,
                                               int second_list_entry_count,
                                               int *number_of_results)
{
        int first_list_position, second_list_position;
        gint32 *results;
        
        /* If there are no entries in one of the lists, we are done */
        if (first_list_entry_count == 0 || 
            second_list_entry_count == 0) {
                *number_of_results = 0;
                return NULL;
        }
        /* Since their entry counts are non-zero, these can't be NULL */
        g_assert (first_list != NULL && second_list != NULL);

        results = g_new0 (gint32, MIN (first_list_entry_count, second_list_entry_count));
        first_list_position = 0;
        second_list_position = 0;
        *number_of_results = 0;
        while (first_list_position < first_list_entry_count &&
               second_list_position < second_list_entry_count) {
                if (first_list[first_list_position] > second_list[second_list_position]) {
                        first_list_position++;
                        continue;
                }
                if (second_list[second_list_position] > first_list[first_list_position]) {
                        second_list_position++;
                        continue;
                }
                results[*number_of_results] = first_list[first_list_position++];
                (*number_of_results)++;
        }
        return results;
}

gint32 *              
medusa_difference_of_two_descending_integer_lists (gint32 *first_list,
                                                   int first_list_entry_count,
                                                   gint32 *second_list,
                                                   int second_list_entry_count,
                                                   int *number_of_results)
{
        int first_list_position, second_list_position;
        gint32 *results;
        

        /* If there are no entries the first list, we are done */
        if (first_list_entry_count == 0) {
                *number_of_results = 0;
                return NULL;
        }
        /* Since their entry counts are non-zero, these can't be NULL */
        g_assert (first_list != NULL && second_list != NULL);
        
        results = g_new0 (gint32, first_list_entry_count);
        first_list_position = 0;
        second_list_position = 0;
        *number_of_results = 0;
        while (first_list_position < first_list_entry_count &&
               second_list_position < second_list_entry_count) {
                if (first_list[first_list_position] > second_list[second_list_position]) {
                        results[*number_of_results] = first_list[first_list_position];
                        (*number_of_results)++;
                        first_list_position++;
                        continue;
                }
                if (second_list[second_list_position] > first_list[first_list_position]) {
                        second_list_position++;
                        continue;
                }
                else {
                        first_list_position++;
                        second_list_position++;
                }
        }
        /* If there are still more entries left in the first list,
           they should all be included */
        while (first_list_position < first_list_entry_count) {
                results[*number_of_results] = first_list_position++;
                        (*number_of_results)++;
        }
        return results;
}


/**
 * medusa_g_list_free_deep_custom
 *
 * Frees the elements of a list and then the list, using a custom free function.
 *
 * @list: List of elements that can be freed with the provided free function.
 * @element_free_func: function to call with the data pointer and user_data to free it.
 * @user_data: User data to pass to element_free_func
 **/
void
medusa_g_list_free_deep_custom (GList *list, GFunc element_free_func, gpointer user_data)
{
	g_list_foreach (list, element_free_func, user_data);
	g_list_free (list);
}

/**
 * medusa_g_list_free_deep
 *
 * Frees the elements of a list and then the list.
 * @list: List of elements that can be freed with g_free.
 **/
void
medusa_g_list_free_deep (GList *list)
{
	medusa_g_list_free_deep_custom (list, (GFunc) g_free, NULL);
}
