/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 */

/* medusa-rdb-table.c:  The implementation of the relational database
   table object */

#include <medusa-rdb-table-private.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <medusa-conf.h>
#include <medusa-test.h>
#include <string.h>
#include <time.h>


static void        medusa_rdb_table_erase_record (MedusaRDBRecord record,
						  int record_size);

MedusaRDBTable *      
medusa_rdb_table_all_rows (MedusaRDBFile *file)
{
	MedusaRDBTable *table;
	int i;
	int *record;

	table = g_new0 (MedusaRDBTable, 1);
  
	table->file = file;

	table->number_of_records = file->number_of_records;
	if (table->number_of_records == 0) {
		table->first_record = NULL;
	}
	else {
		/* Don't include the first one..it contains meta db info */
		for (i = 1; i < table->number_of_records; i++) {
			record = g_new0 (int, 1);
			*record = i;
			table->first_record = g_list_prepend (table->first_record, 
							      record);
      
		}
	}
	return table;
}

void                         
medusa_rdb_table_insert (MedusaRDBTable *table,
			 gpointer data) 
{
	int *new_record;
	new_record = g_new0 (int, 1);
	if (table->deleted_records) {
		new_record = (MedusaRDBRecordNumber *) table->deleted_records->data;
		table->deleted_records = table->deleted_records->next;
		medusa_rdb_table_replace_record_by_number (table, *(MedusaRDBRecordNumber *) new_record,
							   data);
	}
	else {
		*new_record = table->file->number_of_records;
		medusa_io_handler_append_data (table->file->io_handler, data, table->file->field_info->record_size);
		/* Assure that contents pointer is correct in case of remap */
		/* FIXME:  This should be a method within medusa-rdb-file */
		table->file->header = table->file->io_handler->mapped_region + table->file->io_handler->header_length;
		table->file->metainfo = table->file->header + table->file->header_length;
		table->file->contents = table->file->metainfo + table->file->metainfo_length;
  
	}
	table->first_record = g_list_prepend (table->first_record, new_record);
	table->file->number_of_records++;
	table->number_of_records++;
  
}

/* FIXME:  This function assumes record number is in table.  It may not be.
   is this a problem?  Perhaps this function should go in medusa-rdb-file */
void
medusa_rdb_table_replace_record_by_number (MedusaRDBTable *table,
					   int record_number,
					   gpointer new_record)
{
	g_return_if_fail (record_number < table->file->number_of_records);
	memcpy (&table->file->contents[record_number * table->file->field_info->record_size],
		new_record,
		table->file->field_info->record_size);
}

void 
medusa_rdb_table_delete_record (MedusaRDBTable *table,
				MedusaRDBRecordNumber number)
{
	GList *record;
	int *record_number;
	g_return_if_fail (number <= table->file->number_of_records);
#ifdef QUERY_DEBUG
	g_message ("Trying to delete record %d\n", number);
#endif
	record_number = g_new0 (int, 1);
	*record_number = number;
	table->deleted_records = g_list_prepend (table->deleted_records, record_number);
	/* Get rid of the data in the record for security reasons */
	medusa_rdb_table_erase_record (medusa_rdb_record_number_to_record (table,
									   (MedusaRDBRecordNumber) number),
				       table->file->field_info->record_size);
	for (record = table->first_record; record != NULL; record = record->next) {
		if (*(int *)record->data == number) {
			table->first_record = g_list_remove (table->first_record, record->data);
			table->number_of_records--;
			table->file->number_of_records--;
			g_free (record->data);
#ifdef QUERY_DEBUG
			g_message ("Removed record %d\n", number);
#endif
			return;
		}
	}
}

/* Erases the record on disk */
static void
medusa_rdb_table_erase_record (MedusaRDBRecord record,
			       int record_size)
{
	memset (record, 0, record_size);
}


MedusaRDBTable *
medusa_rdb_table_select (MedusaRDBTable *table,
			 MedusaRDBQuery *query)
{
	MedusaRDBTable *result;
	MedusaRDBRecordNumbers *record_list;


	result = g_new0 (MedusaRDBTable, 1);
	result->file = table->file;
	result->number_of_records = 0;
	/* Skip the first record, because it is database information */
	/* if the table has no records, nothing matches the query, so return
	   immediately */
	if (table->number_of_records == 0) {
		return result;
	}
	for (record_list = table->first_record; record_list->next != NULL; record_list = record_list->next) {
		if (medusa_rdb_query_match  (query,
					     medusa_rdb_record_number_to_record 
					     (table, 
					      *(MedusaRDBRecordNumber *) record_list->data),
					     table->file->field_info)) {
			medusa_rdb_table_virtual_insert_record (result, *(MedusaRDBRecordNumber *) record_list->data);
		}
	}
	return result;
  
}

void
medusa_rdb_table_delete (MedusaRDBTable *table,
			 MedusaRDBQuery *query)
{
	MedusaRDBRecordNumbers *record_list;
	MedusaRDBRecord record;

	for (record_list = table->first_record; record_list != NULL; record_list = record_list->next) {
		record = medusa_rdb_record_number_to_record (table,
							     *(MedusaRDBRecordNumber *)record_list->data);
		if (record == NULL) {
			g_warning ("Record in table = NULL!");
			continue;
		}
		if (medusa_rdb_query_match  (query, record, table->file->field_info)) {
			medusa_rdb_table_delete_record (table, *(MedusaRDBRecordNumber *) record_list->data);
		}
	}
}




/* Adds a single record to the table.  Does not insert into the physical file */
void        
medusa_rdb_table_virtual_insert_record  (MedusaRDBTable *table,
					 MedusaRDBRecordNumber number)
{
	int *num_record;
	num_record = g_new0 (int, 1);
	*num_record = number;
	table->first_record = g_list_prepend (table->first_record, num_record);
	table->number_of_records++;
}

void
medusa_rdb_table_free (MedusaRDBTable *table)
{
	GList *record;
	/* FIXME:  This should be an unref...many tables can have
	   one file as an object */
	medusa_rdb_file_free (table->file);
	for (record = table->first_record;  record != NULL; record = record->next) {
		g_free (record->data);
	}
	/* FIXME:  Need to free deleted records? */
	g_list_free (table->first_record);
	g_list_free (table->deleted_records);
	g_free (table);
}

void
medusa_rdb_table_virtual_free (MedusaRDBTable *table)
{
	GList *record;

	for (record = table->first_record;  record != NULL; record = record->next) {
		g_free (record->data);
	}
	/* FIXME:  Need to free deleted records? */
	g_list_free (table->first_record);
	g_list_free (table->deleted_records);
	g_free (table);
}

MedusaRDBRecord    
medusa_rdb_record_number_to_record (MedusaRDBTable *table,
				    MedusaRDBRecordNumber number)
{
	g_return_val_if_fail (number <= table->file->number_of_records, NULL);
	return &table->file->contents[table->file->field_info->record_size * number];
}

int
medusa_rdb_table_get_number_of_records (MedusaRDBTable *table)
{
	return table->number_of_records;
}







