/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 1999, 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 *
 *  medusa-file-queries.c  Builds and performs common queries on a file index
 *
 */

#include <glib.h>
#include <grp.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>

#include "medusa-file-index.h"
#include "medusa-conf.h"
#include "medusa-query-clauses.h"
#include "medusa-rdb-query.h"
#include "medusa-rdb-record.h"
#include "medusa-file-index-queries.h"
#include "medusa-utils.h"

static int          unix_time_at_midnight              (void);
static int          unix_time_seven_days_ago           (void);
static int          unix_time_a_month_ago              (void);
static int          username_to_uid                    (char *username);
static int          group_to_gid                       (char *group);

/* These are the crappiest templates ever! */
#define MEDUSA_FILE_INDEX_QUERY_NO_ARG(query_name, field_name, operator, value, desired_result) \
gboolean \
medusa_file_index_##query_name (MedusaFileSystemDB *file_system_db, \
                                MedusaRDBRecord record, \
                                gpointer arg) \
{ \
        MedusaRDBField *field; \
        MedusaRDBQuery *query; \
        MedusaRDBFieldInfo *field_info; \
        gboolean query_result; \
   \
        g_return_val_if_fail (file_system_db != NULL, FALSE); \
\
        query = medusa_rdb_query_new (); \
        field_info = medusa_file_system_db_get_field_info (file_system_db); \
        field = medusa_rdb_field_get_field_structure (field_info, \
                                                      field_name); \
        query = medusa_rdb_query_add_selection_criterion (query, \
                                                          field, \
                                                          operator, \
                                                          value, \
                                                          desired_result, \
                                                          file_system_db); \
\
        query_result = medusa_rdb_query_match (query, \
                                               record, \
                                               field_info); \
        medusa_rdb_query_free (query); \
        return query_result;\
}
                                              

#define MEDUSA_FILE_INDEX_QUERY_CHAR_ARG(query_name, arg_name, field_name, operator, desired_result) \
gboolean \
medusa_file_index_##query_name (MedusaFileSystemDB *file_system_db, \
                                MedusaRDBRecord record, \
                                char * arg_name) \
{ \
        MedusaRDBField *field; \
        MedusaRDBQuery *query; \
        MedusaRDBFieldInfo *field_info; \
        gboolean query_result; \
\
        g_return_val_if_fail (record != NULL, FALSE); \
        g_return_val_if_fail (file_system_db != NULL, FALSE); \
\
        query = medusa_rdb_query_new (); \
        field_info = medusa_file_system_db_get_field_info (file_system_db); \
        field = medusa_rdb_field_get_field_structure (field_info, \
                                                      field_name); \
        query = medusa_rdb_query_add_selection_criterion (query, \
                                                          field, \
                                                          operator, \
                                                          arg_name, \
                                                          desired_result, \
                                                          file_system_db); \
\
        query_result = medusa_rdb_query_match (query, \
                                              record, \
                                              field_info); \
        medusa_rdb_query_free (query); \
        return query_result; \
}

                                               

#define MEDUSA_FILE_INDEX_QUERY_INT_ARG(query_name, arg_name, field_name, operator, desired_result) \
gboolean \
medusa_file_index_##query_name (MedusaFileSystemDB *file_system_db, \
                                MedusaRDBRecord record, \
                                int arg_name) \
{ \
        MedusaRDBField *field; \
        MedusaRDBQuery *query; \
        MedusaRDBFieldInfo *field_info; \
        gboolean query_result; \
   \
        g_return_val_if_fail (record != NULL, FALSE); \
        g_return_val_if_fail (file_system_db != NULL, FALSE); \
\
        query = medusa_rdb_query_new (); \
        field_info = medusa_file_system_db_get_field_info (file_system_db); \
        field = medusa_rdb_field_get_field_structure (field_info, \
                                                      field_name); \
        query = medusa_rdb_query_add_selection_criterion (query, \
                                                          field, \
                                                          operator, \
                                                          &arg_name, \
                                                          desired_result, \
                                                          file_system_db); \
        query_result = medusa_rdb_query_match (query, \
                                               record, \
                                               field_info); \
        medusa_rdb_query_free (query); \
        return query_result; \
} 




gboolean
medusa_file_index_is_of_type (MedusaFileSystemDB *file_system_db,
                              MedusaRDBRecord record,
                              char *type)
{
        if (strcasecmp (type, "file") == 0) {
                return medusa_file_index_is_file (file_system_db,
                                                  record,
                                                  NULL);
        }

        if (strcasecmp (type, "text_file") == 0) {
                return medusa_file_index_is_text_file (file_system_db,
                                                       record,
                                                       NULL);
        }
        if (strcasecmp (type, "application") == 0) {
                return medusa_file_index_is_application (file_system_db,
                                                  record,
                                                  NULL);
        }

        if (strcasecmp (type, "music") == 0) {
                return medusa_file_index_is_music (file_system_db,
                                            record,
                                            NULL);
        }
        if (strcasecmp (type, "directory") == 0) {
                return medusa_file_index_is_directory (file_system_db,
                                                record,
                                                NULL);
        }

        return FALSE;
}

MEDUSA_FILE_INDEX_QUERY_NO_ARG(is_file, MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_TITLE, MEDUSA_RDB_REGEXP_MATCH, "^x-special", FALSE)

     MEDUSA_FILE_INDEX_QUERY_NO_ARG(is_text_file, MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_TITLE, MEDUSA_RDB_REGEXP_MATCH, "^text/", TRUE)
     

MEDUSA_FILE_INDEX_QUERY_NO_ARG(is_application, MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_TITLE, MEDUSA_RDB_REGEXP_MATCH, "^application/", TRUE)

MEDUSA_FILE_INDEX_QUERY_NO_ARG(is_music, MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_TITLE, MEDUSA_RDB_REGEXP_MATCH, "^audio/", TRUE)

MEDUSA_FILE_INDEX_QUERY_NO_ARG(is_directory, MEDUSA_FILE_INDEX_MIME_TYPE_FIELD_TITLE, MEDUSA_RDB_STRING_EQUALS, "x-special/directory", TRUE)

MEDUSA_FILE_INDEX_QUERY_INT_ARG(has_uid, uid, MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE, MEDUSA_RDB_NUMBER_EQUALS, TRUE)

MEDUSA_FILE_INDEX_QUERY_INT_ARG(has_gid, gid, MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE, MEDUSA_RDB_NUMBER_EQUALS, TRUE)

     MEDUSA_FILE_INDEX_QUERY_INT_ARG(does_not_have_uid, uid, MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE, MEDUSA_RDB_NUMBER_EQUALS, FALSE)

MEDUSA_FILE_INDEX_QUERY_INT_ARG(does_not_have_gid, gid, MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE, MEDUSA_RDB_NUMBER_EQUALS, FALSE)
 


MEDUSA_FILE_INDEX_QUERY_INT_ARG(is_modified_after, time_desired, MEDUSA_FILE_INDEX_MTIME_FIELD_TITLE, MEDUSA_RDB_GREATER_THAN, TRUE)

MEDUSA_FILE_INDEX_QUERY_INT_ARG(is_modified_before, time_desired, MEDUSA_FILE_INDEX_MTIME_FIELD_TITLE, MEDUSA_RDB_LESS_THAN, TRUE)

MEDUSA_FILE_INDEX_QUERY_INT_ARG(is_larger_than, size_desired, MEDUSA_FILE_INDEX_SIZE_FIELD_TITLE, MEDUSA_RDB_GREATER_THAN, TRUE)

MEDUSA_FILE_INDEX_QUERY_INT_ARG(is_smaller_than, size_desired, MEDUSA_FILE_INDEX_SIZE_FIELD_TITLE, MEDUSA_RDB_LESS_THAN, TRUE)

     MEDUSA_FILE_INDEX_QUERY_INT_ARG(is_of_size, size_desired, MEDUSA_FILE_INDEX_SIZE_FIELD_TITLE, MEDUSA_RDB_NUMBER_EQUALS, TRUE)




/* Special functions that require more processing */


gboolean
medusa_file_index_is_owned_by (MedusaFileSystemDB *file_system_db,
                               MedusaRDBRecord record,
                               char *username)
{
        return medusa_file_index_has_uid (file_system_db, 
                                   record, 
                                   username_to_uid (username));
}


gboolean
medusa_file_index_is_in_group (MedusaFileSystemDB *file_system_db,
                               MedusaRDBRecord record,
                               char *group)
{
        return medusa_file_index_has_gid (file_system_db,
                                   record, 
                                   group_to_gid (group));

}

gboolean
medusa_file_index_is_not_owned_by (MedusaFileSystemDB *file_system_db,
                                   MedusaRDBRecord record,
                                   char *username)
{
        return medusa_file_index_does_not_have_uid (file_system_db, 
                                             record, 
                                             username_to_uid (username));

}

gboolean
medusa_file_index_is_not_in_group (MedusaFileSystemDB *file_system_db,
                                   MedusaRDBRecord record,
                                   char *group)
{
        return medusa_file_index_has_uid (file_system_db,
                                   record, 
                                   group_to_gid (group));
}

gboolean
medusa_file_index_is_modified_today (MedusaFileSystemDB *file_system_db,
                                     MedusaRDBRecord record,
                                     gpointer data)
{
        return medusa_file_index_is_modified_after (file_system_db,
                                                    record, 
                                                    unix_time_at_midnight ());

}


gboolean
medusa_file_index_is_modified_in_last_seven_days (MedusaFileSystemDB *file_system_db,
                                                  MedusaRDBRecord record,
                                                  gpointer data)
{
        return medusa_file_index_is_modified_after (file_system_db,
                                                    record, 
                                                    unix_time_seven_days_ago ());

}


gboolean
medusa_file_index_is_modified_this_month (MedusaFileSystemDB *file_system_db,
                                          MedusaRDBRecord record,
                                          gpointer data)
{
        return medusa_file_index_is_modified_after (file_system_db, 
                                             record, 
                                             unix_time_a_month_ago ());

}



gboolean
medusa_file_index_is_not_modified_today (MedusaFileSystemDB *file_system_db,
                                         MedusaRDBRecord record,
                                         gpointer data)
{
        return medusa_file_index_is_modified_before (file_system_db, 
                                              record, 
                                              unix_time_at_midnight ());

}


gboolean
medusa_file_index_is_not_modified_in_last_seven_days (MedusaFileSystemDB *file_system_db,
                                                      MedusaRDBRecord record,
                                                      gpointer data)
{
        return medusa_file_index_is_modified_before (file_system_db, 
                                              record, 
                                              unix_time_seven_days_ago ());

}


gboolean
medusa_file_index_is_not_modified_this_month (MedusaFileSystemDB *file_system_db,
                                              MedusaRDBRecord record,
                                              gpointer data)
{
        return medusa_file_index_is_modified_before (file_system_db, 
                                              record, 
                                              unix_time_a_month_ago ());
}


gboolean
medusa_file_index_uid_can_read_file (MedusaFileSystemDB *file_system_db,
                                     MedusaRDBRecord record,
                                     int uid)
{
        char *uid_field, *gid_field, *permissions_field;
        MedusaRDBDecodeFunc uid_decode, gid_decode, permissions_decode;
        MedusaRDBFieldInfo *field_info;
        int owner_uid, owner_gid;
        GnomeVFSFilePermissions permissions;
        
        g_return_val_if_fail (record != NULL, FALSE); 
        g_return_val_if_fail (file_system_db != NULL, FALSE); 
        /* We can skip all of this stuff if it's root */
        if (uid == 0) {
                printf ("user is root, it's ok to read.\n");
                return TRUE;
        }
        
        field_info = medusa_file_system_db_get_field_info (file_system_db);


        permissions_field = medusa_rdb_record_get_field_contents (record,
                                                                  field_info,
                                                                  MEDUSA_FILE_INDEX_PERMISSIONS_FIELD_TITLE);
        permissions_decode = medusa_rdb_field_get_decoder (field_info,
                                                           MEDUSA_FILE_INDEX_PERMISSIONS_FIELD_TITLE);
        permissions_decode (&permissions, permissions_field, file_system_db);
        
        if (permissions & S_IROTH) {
#ifdef PERMISSIONS_CHECK_DEBUG
                printf ("Other can read the file, it is readable to everyone\n");
#endif
                return TRUE;
        }
        
        uid_field = medusa_rdb_record_get_field_contents (record,
                                                          field_info,
                                                          MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE);
        uid_decode = medusa_rdb_field_get_decoder (field_info,
                                                   MEDUSA_FILE_INDEX_OWNER_FIELD_TITLE);
        uid_decode (&owner_uid, uid_field, file_system_db);
#ifdef PERMISSIONS_CHECK_DEBUG
        printf ("File has uid %d\n", owner_uid);
#endif
        if ((permissions & S_IRUSR) &&
            owner_uid == uid) {
#ifdef PERMISSIONS_CHECK_DEBUG
                printf ("File is readable by owner, and owner is the uid here. returning\n");
#endif
                return TRUE;
        }

        gid_field = medusa_rdb_record_get_field_contents (record,
                                                          field_info,
                                                          MEDUSA_FILE_INDEX_GROUP_FIELD_TITLE);
        gid_decode = medusa_rdb_field_get_decoder (field_info,
                                                   MEDUSA_FILE_INDEX_GROUP_FIELD_TITLE);
        gid_decode (&owner_gid, gid_field, file_system_db);
#ifdef PERMISSIONS_CHECK_DEBUG
        printf ("File has gid %d\n", owner_gid);
#endif
        if ((permissions & S_IRGRP) &&
            medusa_group_contains (owner_gid, uid)) {
#ifdef PERMISSIONS_CHECK_DEBUG
                printf ("File is readable by group, and owner is in the uid here. returning\n");
#endif
                return TRUE;
        }
#ifdef PERMISSIONS_CHECK_DEBUG
        printf ("Hmm, done of the above. returning false\n");
#endif
        return FALSE;
        
}

static int
unix_time_at_midnight ()
{
        struct tm *time_data;
        time_t current_unix_time, unix_time_at_midnight;



        current_unix_time = time(NULL);
        time_data = localtime (&current_unix_time);
        unix_time_at_midnight = current_unix_time - 3600 * time_data->tm_hour - 60 * time_data->tm_min - time_data->tm_sec;
        free (time_data);
        return unix_time_at_midnight;
}


static int
unix_time_seven_days_ago ()
{
        struct tm *time_data;
        time_t current_unix_time, unix_time_seven_days_ago;
        
        current_unix_time = time(NULL);
        time_data = localtime (&current_unix_time);
        unix_time_seven_days_ago = current_unix_time - 7 * 24 * 3600 -
                3600 * time_data->tm_hour - 60 * time_data->tm_min - time_data->tm_sec;
        free (time_data);
        return unix_time_seven_days_ago;

}


static int
unix_time_a_month_ago ()
{
        struct tm *time_data;
        time_t current_unix_time, unix_time_a_month_ago;
        
        current_unix_time = time(NULL);
        time_data = localtime (&current_unix_time);
        /* First go back to midnight */
        unix_time_a_month_ago = current_unix_time - 3600 * time_data->tm_hour - 60 * time_data->tm_min  - time_data->tm_sec;
        /* 30 days has september... */
        switch (time_data->tm_mon) {
        case 8: 
        case 3: 
        case 5: 
        case 10:
                unix_time_a_month_ago -= 24 * 3600 * 30;
                break;
        case 0: 
        case 2: 
        case 4: 
        case 6: 
        case 7: 
        case 9: 
        case 11:
                unix_time_a_month_ago -= 24 * 3600 * 31;
                break;
        case 1:
                unix_time_a_month_ago -= 24 * 3600 * 29;
                break;
        }
        free (time_data);
        return unix_time_a_month_ago;
}


static int
username_to_uid (char *username)
{
        struct passwd *user_info;
        int uid;

        user_info = getpwnam (username);
        uid = user_info->pw_uid;

        /* It doesn't seem like you need this */
        /* free (user_info); */
        return uid;

}


static int
group_to_gid (char *group)
{
        struct group *group_info;
        int gid;

        group_info = getgrnam (group);
        gid = group_info->gr_gid;

        /* It doesn't seem like you need this */
        /* free (group_info); */
        return gid;

}





