/* 
 * Copyright 2000 Karl Nelson
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkstock.h>
#include "libgnomeuimm/app-helper.h"
#include <libgnomeui/gnome-uidefs.h>
#include <glibmm/exceptionhandler.h>
#include <gtkmm/stock.h>

#include <sigc++/hide.h>


namespace Gnome
{

namespace UI
{

namespace Items
{

/*******************************************************************/
gchar* cpp_strdup(const Glib::ustring& s)
{
  gchar* c=new gchar[s.bytes()+1];
  memcpy(c,s.c_str(),s.bytes()+1);
  return c;
}


Info_::Info_()
: callback_(), subtree_()
{
  ref_count_ = 1;
  label_ = 0;
  hint_= 0 ;
}

Info_::Info_(const Glib::ustring& label, const Glib::ustring& hint)
: callback_(), subtree_()
{
  ref_count_ = 1;
  label_ = cpp_strdup(label);
  hint_ = 0;

  if (!hint.empty())
    hint_ = cpp_strdup(hint);
}

Info_::~Info_()
{
  delete label_;
  delete hint_;
}

void Info_::ref()
{
  ref_count_++;
}

void Info_::unref()
{
  if (!--ref_count_)
    delete this;
}

extern "C"
{

static void libgnomeuimm_info_call(void* object, void* data)
{
  try
  {
    Info_* d=(Info_*) data;
    d->callback_(Glib::wrap((GtkWidget*)object));
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }
}

} // "C"

void Info_::connect(Info& info)
{
  info.label=label_;
  info.hint=hint_;
  info.unused_data=this;
  if (callback_.connected())
  {
    info.user_data=this;
    info.moreinfo = (void*)&libgnomeuimm_info_call;
  }
  if (info.type() == Info::SUBTREE)
  {
    info.moreinfo = (void*)subtree_.gobj();
  }
  // Dirty hack
  else if (info.type() == Info::HELP)
  {
    info.moreinfo = (void*) info.label;
    info.label = 0;
  }
}

/* This is a bit of a kludge to get around some problems with the
   Items::Info structure.  Many of the actions on Items::Info imply the
   availablity of the Widget - something which as a rule does not
   happen in gtkmm.  That is, we wash the object because most often
   it isn't relevent.  Items::Info is different - since the user never
   constructed the object they don't know its address.  

   The solution is to allow both gtk+ type slots and gtkmm style
   slots with and without the object.  To do this it is wired such
   that Info_ only stores the with-object style.  The following converter
   changes a with to a without.  

   All info methods now take both slot styles.  
*/
    
void Info_::set_callback(const Info::Callback& callback)
{
  if (callback.connected())
    callback_=SigC::hide<Gtk::Widget*>(callback);
}

void Info_::set_callback(const Info::CallbackW& callback)
{
  callback_=callback;
}


void Info_::set_subtree(const Array<Info>& subtree)
{
  subtree_=subtree;
}

Array<Info>& Info_::get_subtree()
{
  return subtree_; //Removes Begin and End.
}


/*******************************************************************/
Info::~Info()
{
  if (unused_data) info_()->unref();
}

Info::Info()
{
  GnomeUIInfo::type = GnomeUIInfoType(0);
  GnomeUIInfo::label = 0;
  GnomeUIInfo::hint = 0;
  GnomeUIInfo::moreinfo = 0;
  GnomeUIInfo::user_data = 0;
  GnomeUIInfo::unused_data = 0;
  GnomeUIInfo::pixmap_type = GnomeUIPixmapType(0);
  GnomeUIInfo::pixmap_info = 0;
  GnomeUIInfo::accelerator_key = 0;
  GnomeUIInfo::ac_mods = GdkModifierType(0);
  GnomeUIInfo::widget = 0;
}

Info::Info(const Info& i)
{
  memcpy(this,&i, sizeof(Info));

  if (unused_data)
    info_()->ref();
}

Info& Info::operator =(const Info& i)
{
  if(this==&i)
    return *this;

  if(unused_data)
    info_()->unref();

  memcpy(this, &i, sizeof(Info));

  if(unused_data)
    info_()->ref();

  return *this;
}

Info::Type Info::type() const
{
  return Type(GnomeUIInfo::type);
}

Info_* Info::info_()
{
  return static_cast<Info_*>(GnomeUIInfo::unused_data);
}

Gtk::Widget* Info::get_widget() const
{
  return Glib::wrap(GnomeUIInfo::widget);
}

void Info::set_icon(const Icon& icon_)
{
  pixmap_type = icon_.pixmap_type;
  GnomeUIInfo::pixmap_info = icon_.pixmap_info;
}

void Info::set_accel(Gtk::Menu_Helpers::AccelKey key)
{
  GnomeUIInfo::accelerator_key = key.get_key();
  GnomeUIInfo::ac_mods = GdkModifierType(key.get_mod());
  if (accelerator_key == GDK_VoidSymbol)
  {
    accelerator_key = 0;
    GnomeUIInfo::ac_mods = GdkModifierType(0);
  }
}

void Info::init(Type type_)
{
  GnomeUIInfo::type=GnomeUIInfoType(type_);
}

void Info::init_cb(Type                 type_,
	   const Icon          &icon_,
	   const Glib::ustring &label_,
	   const Callback      &callback_,
	   const Glib::ustring &hint_)
{
  GnomeUIInfo::type=GnomeUIInfoType(type_);
  Info_ *info_=new Info_(label_,hint_);
  info_->set_callback(callback_);

  set_icon(icon_);
  set_accel();

  info_->connect(*this);
}

void Info::init_cbw(Type type_,const Icon& icon_,
          const Glib::ustring& label_,const CallbackW& callback_,
          const Glib::ustring& hint_)
{
  GnomeUIInfo::type=GnomeUIInfoType(type_);
  Info_ *info_=new Info_(label_,hint_);
  info_->set_callback(callback_);

  set_icon(icon_);
  set_accel();

  info_->connect(*this);
}


void Info::init_sub(Type type_,const Icon& icon_,
          const Glib::ustring& label_,const Array<Info>& array,
          const Glib::ustring& hint_)
{
  GnomeUIInfo::type=GnomeUIInfoType(type_);
  Info_ *info_=new Info_(label_, hint_);
  info_->set_subtree(array);

  set_icon(icon_);
  set_accel();

  info_->connect(*this);
}



extern "C"
{

static void
libgnomeuimm_info_connect(GnomeUIInfo* uiinfo, const char* signal_name, GnomeUIBuilderData* uibdata)
{
  if(uiinfo->moreinfo)
  {
    g_signal_connect_data(G_OBJECT(uiinfo->widget), signal_name, (GCallback)(uiinfo->moreinfo),
                          (gpointer)(uiinfo->user_data), 0, (GConnectFlags)0);
  }
}

} // "C"


GnomeUIBuilderData Begin::build_data_ = { &libgnomeuimm_info_connect, 0, 0, 0, 0 };

/*******************************************************************/

Separator::operator Gtk::Menu_Helpers::Element()
{
  Gtk::MenuItem *item=manage(new Gtk::MenuItem());
  item->show();
  return *item;
}

/*******************************************************************/

SubTree::SubTree()
{
}


SubTree::SubTree(const Glib::ustring& label,const Array<Info>& uitree,
                 const Glib::ustring& tip /* =Glib::ustring() */)
{
  init_sub(SUBTREE, Icon(), label, uitree, tip);
}

SubTree::SubTree(const Icon& icon, const Glib::ustring& label,
               const Array<Info>& uitree, const Glib::ustring& tip /* =Glib::ustring() */)
{
  init_sub(SUBTREE, icon, label, uitree, tip);
}

SubTree::~SubTree()
{
}


Array<Info>& SubTree::get_uitree()
{
  Info_* pInfo_ = info_();
  return pInfo_->get_subtree();
}

extern "C"
{

static void 
libgnomeuimm_radio_info_call(GtkWidget* w, gpointer data)
{
  Info_* d = (Info_*)data;
  if(d->callback_)
  {
    if (GTK_IS_TOGGLE_BUTTON(w) && GTK_TOGGLE_BUTTON(w)->active)
    {
      d->callback_(Glib::wrap(w));
      return;
    }

    if (GTK_IS_CHECK_MENU_ITEM(w) && GTK_CHECK_MENU_ITEM(w)->active)
    {
      d->callback_(Glib::wrap(w));
      return;
    }
  }
}

static void
libgnomeuimm_radio_info_connect (GnomeUIInfo* uiinfo, const gchar* signal_name, GnomeUIBuilderData* uibdata)
{
  g_signal_connect_data(G_OBJECT (uiinfo->widget), signal_name, GCallback(libgnomeuimm_radio_info_call),
                       (gpointer)(uiinfo->user_data), 0, (GConnectFlags)0);
}

} // extern "C"


GnomeUIBuilderData RadioTree::build_data_ = { &libgnomeuimm_radio_info_connect, 0, 0, 0, 0 };

RadioTree::RadioTree(const Items::Array<Info>& array)
{
  GnomeUIInfo::type = GnomeUIInfoType(RADIOITEMS);
  Items::Info_ *info_ = new Items::Info_();
  info_->set_subtree(array);
  info_->subtree_.gobj()->moreinfo = (void*)&build_data_;
  moreinfo = (void*)info_->subtree_.gobj();
}

/*******************************************************************/

void ConfigureItem::init(const Callback &cv,GnomeUIInfoConfigurableTypes ct)
{
  GnomeUIInfo::type = GnomeUIInfoType(GNOME_APP_UI_ITEM_CONFIGURABLE);
  Info_ *info_= new Info_();
  info_->set_callback(cv);

  set_icon(Icon());
  GnomeUIInfo::accelerator_key = ct;

  info_->connect(*this);
}

void ConfigureItem::init(const CallbackW &cv,GnomeUIInfoConfigurableTypes ct)
{
  GnomeUIInfo::type = GnomeUIInfoType(GNOME_APP_UI_ITEM_CONFIGURABLE);
  Info_ *info_ = new Info_();
  info_->set_callback(cv);

  set_icon(Icon());
  GnomeUIInfo::accelerator_key = ct;

  info_->connect(*this);
}


void ConfigureItem::init(const Callback &cv, GnomeUIInfoConfigurableTypes ct, const Glib::ustring& strLabel, const Glib::ustring& strHint)
{
  GnomeUIInfo::type = GnomeUIInfoType(GNOME_APP_UI_ITEM_CONFIGURABLE);
  Info_ *info_ = new Info_();
  info_->set_callback(cv);

  set_icon(Icon());
  GnomeUIInfo::accelerator_key = ct;

  info_->label_ = cpp_strdup(strLabel); //e.g. "Create a new foo".
  info_->hint_ = cpp_strdup(strHint); //e.g. "_New foo".

  info_->connect(*this);
}

void ConfigureItem::init(const CallbackW& cv, GnomeUIInfoConfigurableTypes ct, const Glib::ustring& strLabel, const Glib::ustring& strHint)
{
  GnomeUIInfo::type = GnomeUIInfoType(GNOME_APP_UI_ITEM_CONFIGURABLE);
  Info_ *info_ = new Info_();
  info_->set_callback(cv);

  set_icon(Icon());
  GnomeUIInfo::accelerator_key = ct;

  info_->label_ = cpp_strdup(strLabel); //e.g. "Create a new foo".
  info_->hint_ = cpp_strdup(strHint); //e.g. "_New foo".

  info_->connect(*this);
}


//Icon:

Icon::Icon(const Gtk::StockID &stock_id)
: pixmap_type(GNOME_APP_PIXMAP_STOCK), pixmap_info(stock_id.get_c_str ())
{}

Icon::Icon()
: pixmap_type(GNOME_APP_PIXMAP_NONE), pixmap_info(0)
{}

Icon::Icon(Type type, gconstpointer info)
: pixmap_type(GnomeUIPixmapType(type)), pixmap_info(info)
{}

Icon::~Icon()
{}

//IconXpm:

IconXpm::IconXpm(const gchar**const xpm)
: Icon(DATA, gconstpointer(xpm))
{}

IconXpm::~IconXpm()
{}

//IconFile:

IconFile::IconFile(const gchar* file)
: Icon(FILENAME, gconstpointer(file))
{}

IconFile::~IconFile()
{}

} /* namespace Items */

namespace Menus
{
    
New::New (const Items::Array<Items::Info> &tree)
  : Items::Menu("_New", tree)
{
  set_icon( Items::Icon(Gtk::Stock::NEW) );
  GnomeUIInfo::accelerator_key = GNOME_KEY_NAME_NEW;
  GnomeUIInfo::ac_mods = GdkModifierType(GNOME_KEY_MOD_NEW);
}
    
} /* namespace Menus */


//Gtk::MenuShell* MenuShell_find_menu_pos(const Gtk::MenuShell& parent,
//                                               const Glib::ustring &path,
//                                               gint& pos)
//{
  //gnome_app_find_menu_pos() actually wants a GtkMenuShell*, but the arg is a GtkWidget*.
//  GtkWidget* pGtkMenuShell = gnome_app_find_menu_pos(GTK_WIDGET(parent.gobj()), path.c_str(), &pos);
//  return Gtk::wrap(GTK_MENU_SHELL(pGtkMenuShell));
//}

namespace
{

struct UIArrayHolder
{
   Items::Array<Items::Info> info_;
   UIArrayHolder(const Items::Array<Items::Info>& info): info_(info) {}

   static void destroy(void* d)
   {
     delete ((UIArrayHolder*)d);
   }
};

} // namespace


void fill (Gtk::MenuShell& menu_shell,
           const Items::Array<Items::Info>& info,
           Glib::RefPtr<Gtk::AccelGroup>& accel_group,
           bool uline_accels,
           int pos)
{
  menu_shell.set_data ("gnomemm-uihold", new UIArrayHolder (info), UIArrayHolder::destroy);
  gnome_app_fill_menu (menu_shell.gobj (), info.gobj (),
		       accel_group->gobj (), (gboolean)uline_accels,
		       pos);
}

void fill (Gtk::Toolbar& toolbar,
           const Items::Array<Items::Info>& info,
           Glib::RefPtr<Gtk::AccelGroup>& accel_group)
{
  toolbar.set_data("gnomemm-uihold", new UIArrayHolder(info),UIArrayHolder::destroy);
  gnome_app_fill_toolbar(toolbar.gobj(), info.gobj(), accel_group->gobj());
}

} /*namespace UI */

} /* namespace Gnome */

