#define __TESTPRINT_C__

/*
 * libgnomeprint test program
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *   Unknown author
 *   Lauris Kaplinski <lauris@helixcode.com>
 *
 * Copyright 2000-2001 Ximian, Inc. and authors
 *
 */

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <popt.h>
#include <libbonobo.h>
#include <libgnomeprint/gnome-print-i18n.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-meta.h>
#include <libgnomeprint/gnome-print-bonobo.h>
#include <libgnomeprint/gnome-print-bonobo-client.h>

static gchar *dumpfile = NULL;
static gboolean bonobo = FALSE;

static struct poptOption options[] = {
	{ "dump", 'd', POPT_ARG_STRING, &dumpfile, 0,
	  N_("Dump printout to metafile"), N_("PATH")},
	{ "bonobo", 'b', POPT_ARG_NONE, &bonobo, 0,
	  N_("Split job between bonobo"), NULL},
	POPT_AUTOHELP
	{ NULL, '\0', 0, NULL, 0 }
};

#define G_P_PIXELS 256

static void
print_line (GnomePrintContext *ctx, gdouble x, gdouble y)
{
	gdouble px, py;

	gnome_print_newpath (ctx);
	gnome_print_moveto (ctx, x, y);
	gnome_print_lineto (ctx, x, y + 200.0);
	gnome_print_arcto (ctx, x + 50, y + 200, 50, 180.0, 45.0, 1);

	px = 50 * cos (0.25);
	py = 50 * sin (0.25);

	gnome_print_lineto (ctx, x + 50 + px + 100, y + 100 + py);

	gnome_print_stroke (ctx);
}

static void
print_lines (GnomePrintContext *ctx)
{
	gdouble d0[] = {10.0, 10.0};

	gnome_print_scale (ctx, 0.2, 0.2);

	gnome_print_setlinewidth (ctx, 32.0);
	gnome_print_setdash (ctx, 0, NULL, 0);
	print_line (ctx, 0, 0);
	gnome_print_setdash (ctx, 2, d0, 0);
	print_line (ctx, 200, 0);
	gnome_print_setdash (ctx, 2, d0, 3);
	print_line (ctx, 400, 0);
	gnome_print_setdash (ctx, 2, d0, 7);
	print_line (ctx, 600, 0);
}

static void
print_page (GnomePrintContext *pc)
{
	GnomeGlyphList *gl;
	GnomeFont *font;
	ArtDRect bbox;

	double matrix_slanted[6] = {0.9, 0.1, -0.8, 0.9, 0, 0};
	double matrix2[6] = {1, 0, 0, 1, 0, 100};
	double matrix3[6] = {100, 0, 0, 100, 50, 300};
	double matrix4[6] = {100, 0, 0, 100, 50, 410};

	char img      [G_P_PIXELS] [G_P_PIXELS];
	char colorimg [G_P_PIXELS] [G_P_PIXELS] [3];
	int x, y;
	gint pixels;

	/* Rotated text (using glyphlist) */
	gnome_print_gsave (pc);
	gnome_print_translate (pc, 50, 600);
	gnome_print_rotate (pc, 25);
	font = gnome_font_find_closest_from_weight_slant ("Times", GNOME_FONT_BOLD, 1, 36);
	if (font == NULL) g_error ("Cannot find suitable font, exiting...");
	gnome_print_setfont (pc, font);
	gl = gnome_glyphlist_from_text_dumb (font, 0x000000ff, 0.0, 0.0, _("Gnome-print test page, rotated"));
	gnome_glyphlist_bbox (gl, NULL, 0, &bbox);
	gnome_print_setrgbcolor (pc, 0.9, 0.8, 0.6);
	g_print ("bbox: %g %g %g %g\n", bbox.x0, bbox.y0, bbox.x1, bbox.y1);
	gnome_print_rect_filled (pc, bbox.x0 - 2.0, bbox.y0 - 2.0, bbox.x1 + 2.0 - bbox.x0, bbox.y1 + 2.0 - bbox.y0);
	gnome_print_moveto (pc, 0.0, 0.0);
	gnome_print_glyphlist (pc, gl);
	gnome_glyphlist_unref (gl);
	gnome_print_grestore (pc);

	/* Slanted text (using show) */
	gnome_print_gsave (pc);
	gnome_print_setfont (pc, font);
	gnome_print_moveto (pc, 150, 650);
	gnome_print_concat (pc, matrix_slanted);
	gnome_print_show (pc, _("Gnome-print test page, slanted"));
	gnome_print_grestore (pc);

	gnome_print_gsave (pc);
	font = gnome_font_find_closest_from_weight_slant ("Times", GNOME_FONT_BOLD, 1, 10);
	gnome_print_setfont (pc, font);
	gnome_print_moveto (pc, 150, 700);
	gnome_print_concat (pc, matrix_slanted);
	gnome_print_show (pc, _("Gnome-print test page, slanted. Small"));
	gnome_print_grestore (pc);

	/* Lines */
	gnome_print_gsave (pc);
	gnome_print_translate (pc, 50, 700);
	print_lines (pc);
	gnome_print_grestore (pc);

#if 0
  font = gnome_font_new ("NimbusRomNo9L-Regu", 36);
  gnome_print_setfont (pc, font);
  gnome_print_show (pc, "Gnome-print test page");
  gnome_print_grestore (pc);

  gnome_print_setrgbcolor (pc, 1, 0, 0.5);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);

  gnome_print_show (pc, "Gnome-print test page");
  gnome_print_grestore (pc);
#endif

  gnome_print_setrgbcolor (pc, 1, 0, 0.5);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);
  gnome_print_closepath (pc);
  gnome_print_fill (pc);

  pixels = G_P_PIXELS;
  
  for (y = 0; y < pixels; y++)
      for (x = 0; x < pixels; x++)
	img[y][x] = ((x+y)*256/pixels)/2;

  gnome_print_gsave (pc);
    gnome_print_concat (pc, matrix3);
    gnome_print_moveto (pc, 0, 0);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
  gnome_print_grestore (pc);
  
#if 0
  gnome_print_gsave (pc);
    g_print ("Image slanted\n");
    gnome_print_concat (pc, matrix_slanted);
    gnome_print_moveto (pc, 10, 10);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
    gnome_print_moveto (pc, 10, 100);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
    gnome_print_moveto (pc, 10, 100);
    gnome_print_grayimage (pc, (char *)img, pixels, pixels, pixels);
  gnome_print_grestore (pc);
#endif	

  for (y = 0; y < pixels; y++)
      for (x = 0; x < pixels; x++)
	{
	  colorimg[y][x][0] = (x + y) >> 1;
	  colorimg[y][x][1] = (x + (pixels - 1  - y)) >> 1;
	  colorimg[y][x][2] = ((pixels - 1 - x) + y) >> 1;
	}

  gnome_print_gsave (pc);
    gnome_print_concat (pc, matrix4);
    gnome_print_moveto (pc, 0, 0);
    gnome_print_rgbimage (pc, (char *)colorimg, pixels, pixels, pixels * 3);
  gnome_print_grestore (pc);

  gnome_print_concat (pc, matrix2);
  gnome_print_setrgbcolor (pc, 0, 0.5, 0);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);
  gnome_print_closepath (pc);
  gnome_print_stroke (pc);

  gnome_print_concat (pc, matrix2);
  gnome_print_setrgbcolor (pc, 0, 0, 0.5);
  gnome_print_moveto (pc, 50, 50);
  gnome_print_lineto (pc, 50, 90);
  gnome_print_curveto (pc, 70, 90, 90, 70, 90, 50);
  gnome_print_closepath (pc);
  gnome_print_gsave (pc);
  gnome_print_stroke (pc);
  gnome_print_grestore (pc);
  gnome_print_clip (pc);

  gnome_print_moveto (pc, 50, 50);
  gnome_font_unref (font);
  font = gnome_font_find ("Courier", 18);
  gnome_print_setfont (pc, font);
  gnome_print_show (pc, "clip!");

  gnome_font_unref (font);
}

static void
print_test_page (GnomePrintContext *ctx)
{
	gnome_print_beginpage (ctx, "testprint demo page");

	print_page (ctx);

	gnome_print_showpage (ctx);
}

static void
print_bonobo_page (GnomePrintBonobo *gpb, GnomePrintContext *ctx, gdouble width, gdouble height,
		   const Bonobo_PrintScissor *opt_scissor, gpointer data)
{
	g_print ("Printing bonobo page\n");

	print_page (ctx);
}

int
main (int argc, const char **argv)
{
	GnomePrintConfig *config;
	GnomePrintContext *pctx;
	poptContext ctx;
	char **args;

	if (!bonobo_init (&argc, (char **)argv)) 
		g_error ("Could not initialize bonobo");

	/* Parse arguments */
	ctx = poptGetContext (NULL, argc, argv, options, 0);
	g_return_val_if_fail (ctx != NULL, 1);
	g_return_val_if_fail (poptGetNextOpt (ctx) == -1, 1);
	args = (char **) poptGetArgs (ctx);

	if (dumpfile) {
		pctx = gnome_print_meta_new ();
		g_object_ref (G_OBJECT (pctx));
	} else {
		config = gnome_print_config_default ();
		if (args && args[0]) {
			gint i;
			for (i = 0; args[i] != NULL; i ++) {
				if (strchr (args[i], '=')) {
					gchar *k, *v;
					k = g_strdup (args[i]);
					v = strchr (k, '=');
					*v++ = '\0';
					if (!gnome_print_config_set (config, k, v)) {
						g_warning ("Setting %s = %s failed", k, v);
					}
				} else {
					g_warning ("Argument is not key=value pair: %s", args[i]);
				}
			}
		}
		pctx = gnome_print_context_new (config);
		g_return_val_if_fail (pctx != NULL, 1);
	}

	/* Free popt context */
	poptFreeContext (ctx);

	/* Now pctx is final output context */
	if (bonobo) {
		CORBA_ORB orb;
		GnomePrintBonobo *gpb;
		Bonobo_Print bp;
		GnomePrintBonoboData *gpbd;
		CORBA_Environment ev;
		CORBA_char *ior;
		GnomePrintBonoboDimensions *dim;
		orb = bonobo_orb ();
		gpb = gnome_print_bonobo_new (print_bonobo_page, NULL);
		CORBA_exception_init (&ev);
		ior = CORBA_ORB_object_to_string (orb, BONOBO_OBJREF (gpb), &ev);
		CORBA_exception_init (&ev);
		bp = CORBA_ORB_string_to_object (orb, ior, &ev);
		CORBA_free (ior);
		dim = gnome_print_bonobo_dimensions_new (21.0 * 72.0 / 2.54, 29.7 * 72.0 / 2.54);
		gpbd = gnome_print_bonobo_client_remote_render (bp, dim, NULL);

		gnome_print_beginpage (pctx, "testprint demo page");
		gnome_print_bonobo_data_re_render (pctx, 0, 0, gpbd, 0, 0);
		gnome_print_showpage (pctx);
	} else {
		print_test_page (pctx);
	}

	/* Close output (if we have metafile, it is still referenced) */
	gnome_print_context_close (pctx);

	if (dumpfile) {
		const guchar *buf;
		gint len;
		gint fh;
		buf = gnome_print_meta_get_buffer (GNOME_PRINT_META (pctx));
		len = gnome_print_meta_get_length (GNOME_PRINT_META (pctx));
		fh = open (dumpfile, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
		g_return_val_if_fail (fh > 0, 1);
		write (fh, buf, len);
		close (fh);

		g_object_unref (G_OBJECT (pctx));
	}

	return 0;
}

