#define __GNOME_PRINT_PAPER_C__

/*
 * GnomePrintPaper and GnomePrintUnit
 *
 * It is pure convenience stuff, that hopefully meakes the life
 * easier for someone
 *
 * Copyright (C) 1998 the Free Software Foundation and Ximian, Inc.
 *
 * Authors:
 *   Dirk Luetjens <dirk@luedi.oche.de>
 *   Yves Arrouye <Yves.Arrouye@marin.fdn.fr>
 *   Lauris Kaplinski <lauris@ximian.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include "gnome-print-i18n.h"
#include "gnome-print-paper.h"

/* fixme: fill in all papers and units */
/* fixme: load papers from file */
/* fixme: use some fancy unit program */

/*
 * WARNING! Do not mess up with that - we use hardcoded numbers for base units!
 */

static const GnomePrintUnit gp_units[] = {
	{0, GNOME_PRINT_UNIT_DIMENSIONLESS, 1.0, N_("Unit"), N_("Unit"), N_("Units"), N_("Units")},
	{0, GNOME_PRINT_UNIT_ABSOLUTE, 1.0, N_("Point"), N_("Pt"), N_("Points"), N_("Pts")},
	{0, GNOME_PRINT_UNIT_USERSPACE, 1.0, N_("Userspace unit"), N_("User"), N_("Userspace units"), N_("User")},
	{0, GNOME_PRINT_UNIT_DEVICE, 1.0, N_("Pixel"), N_("Px"), N_("Pixels"), N_("Px")},
	{0, GNOME_PRINT_UNIT_ABSOLUTE, 72.0 / 25.4, N_("Millimeter"), N_("mm"), N_("Millimeters"), N_("mm")},
};

#define gp_num_units (sizeof (gp_units) / sizeof (gp_units[0]))

static const GnomePrintPaper gp_papers[] = {
	{0, N_("A4"), 21.0 * 72.0 / 2.54, 29.7 * 72.0 / 2.54}
};

#define gp_num_papers (sizeof (gp_papers) / sizeof (gp_papers[0]))

/* Returned papers are const, but lists have to be freed */

const GnomePrintPaper *
gnome_print_paper_get_default (void)
{
	return &gp_papers[0];
}

const GnomePrintPaper *
gnome_print_paper_get_by_name (const guchar *name)
{
	gint i;

	g_return_val_if_fail (name != NULL, NULL);

	for (i = 0; i < gp_num_papers; i++) {
		if (!strcasecmp (name, gp_papers[i].name)) return &gp_papers[i];
	}

	return NULL;
}

#define GP_CLOSE_ENOUGH(a,b) (fabs ((a) - (b)) < 0.1)
#define GP_LESS_THAN(a,b) ((a) - (b) < 0.01)

const GnomePrintPaper *
gnome_print_paper_get_by_size (gdouble width, gdouble height)
{
	gint i;

	/* Should we allow papers <= 1/5184 sq inch? */
	g_return_val_if_fail (width > 1.0, NULL);
	g_return_val_if_fail (height > 1.0, NULL);

	for (i = 0; i < gp_num_papers; i++) {
		if (GP_CLOSE_ENOUGH (gp_papers[i].width, width) && GP_CLOSE_ENOUGH (gp_papers[i].height, height)) return &gp_papers[i];
	}

	return NULL;
}

const GnomePrintPaper *
gnome_print_paper_get_closest_by_size (gdouble width, gdouble height, gboolean mustfit)
{
	const GnomePrintPaper *bestpaper;
	gdouble dist, best;
	gint i;

	/* Should we allow papers <= 1/5184 sq inch? */
	g_return_val_if_fail (width > 1.0, NULL);
	g_return_val_if_fail (height > 1.0, NULL);

	bestpaper = NULL;
	best = 1e18;

	for (i = 0; i < gp_num_papers; i++) {
		if (!mustfit || (GP_LESS_THAN (width, gp_papers[i].width) && GP_LESS_THAN (height, gp_papers[i].height))) {
			gdouble dx, dy;
			/* We fit, or it is not important */
			dx = width - gp_papers[i].width;
			dy = height - gp_papers[i].height;
			dist = sqrt (dx * dx + dy * dy);
			if (dist < best) {
				best = dist;
				bestpaper = &gp_papers[i];
			}
		}
	}

	return bestpaper;
}

GList *
gnome_print_paper_get_list (void)
{
	GList *papers;
	gint i;

	papers = NULL;

	for (i = 0; i < gp_num_papers; i++) {
		papers = g_list_prepend (papers, (gpointer) &gp_papers[i]);
	}

	papers = g_list_reverse (papers);

	return papers;
}

void
gnome_print_paper_free_list (GList *papers)
{
	g_list_free (papers);
}

/* Base units are the ones used by gnome-print and paper descriptions */

const GnomePrintUnit *
gnome_print_unit_get_identity (guint base)
{
	switch (base) {
	case GNOME_PRINT_UNIT_DIMENSIONLESS:
		return &gp_units[0];
		break;
	case GNOME_PRINT_UNIT_ABSOLUTE:
		return &gp_units[1];
		break;
	case GNOME_PRINT_UNIT_DEVICE:
		return &gp_units[2];
		break;
	case GNOME_PRINT_UNIT_USERSPACE:
		return &gp_units[3];
		break;
	default:
		g_warning ("file %s: line %d: Illegal unit base %d", __FILE__, __LINE__, base);
		return FALSE;
		break;
	}
}

/* fixme: */
const GnomePrintUnit *
gnome_print_unit_get_default (void)
{
	return &gp_units[0];
}

const GnomePrintUnit *
gnome_print_unit_get_by_name (const guchar *name)
{
	gint i;

	g_return_val_if_fail (name != NULL, NULL);

	for (i = 0; i < gp_num_units; i++) {
		if (!strcasecmp (name, gp_units[i].name)) return &gp_units[i];
	}

	return NULL;
}

const GnomePrintUnit *
gnome_print_unit_get_by_abbreviation (const guchar *abbreviation)
{
	gint i;

	g_return_val_if_fail (abbreviation != NULL, NULL);

	for (i = 0; i < gp_num_units; i++) {
		if (!strcasecmp (abbreviation, gp_units[i].abbr)) return &gp_units[i];
	}

	return NULL;
}

GList *
gnome_print_unit_get_list (guint bases)
{
	GList *units;
	gint i;

	g_return_val_if_fail ((bases & ~GNOME_PRINT_UNITS_ALL) == 0, NULL);

	units = NULL;

	for (i = 0; i < gp_num_units; i++) {
		if (bases & gp_units[i].base) {
			units = g_list_prepend (units, (gpointer) &gp_units[i]);
		}
	}

	units = g_list_reverse (units);

	return units;
}

void
gnome_print_unit_free_list (GList *units)
{
	g_list_free (units);
}

/* These are pure utility */
/* Return TRUE if conversion is possible */
gboolean
gnome_print_convert_distance (gdouble *distance, GnomePrintUnit *from, GnomePrintUnit *to)
{
	g_return_val_if_fail (distance != NULL, FALSE);
	g_return_val_if_fail (from != NULL, FALSE);
	g_return_val_if_fail (to != NULL, FALSE);

	if (from->base != to->base) return FALSE;

	*distance = *distance * from->unittobase / to->unittobase;

	return TRUE;
}

/* ctm is for userspace, devicetransform is for device units */
gboolean
gnome_print_convert_distance_full (gdouble *distance, GnomePrintUnit *from, GnomePrintUnit *to,
				   gdouble ctmscale, gdouble devicescale)
{
	gdouble absolute;

	g_return_val_if_fail (distance != NULL, FALSE);
	g_return_val_if_fail (from != NULL, FALSE);
	g_return_val_if_fail (to != NULL, FALSE);

	if (from->base == to->base) return gnome_print_convert_distance (distance, from, to);

	switch (from->base) {
	case GNOME_PRINT_UNIT_ABSOLUTE:
		absolute = *distance * from->unittobase;
		break;
	case GNOME_PRINT_UNIT_DEVICE:
		if (devicescale) {
			absolute = *distance * from->unittobase * devicescale;
		} else {
			return FALSE;
		}
		break;
	case GNOME_PRINT_UNIT_USERSPACE:
		if (ctmscale) {
			absolute = *distance * from->unittobase * ctmscale;
		} else {
			return FALSE;
		}
		break;
	default:
		g_warning ("file %s: line %d: Illegal unit (base %d)", __FILE__, __LINE__, from->base);
		return FALSE;
		break;
	}

	switch (to->base) {
	case GNOME_PRINT_UNIT_ABSOLUTE:
		*distance = absolute / to->unittobase;
		break;
	case GNOME_PRINT_UNIT_DEVICE:
		if (devicescale) {
			*distance = absolute / (to->unittobase * devicescale);
		} else {
			return FALSE;
		}
		break;
	case GNOME_PRINT_UNIT_USERSPACE:
		if (ctmscale) {
			*distance = absolute / (to->unittobase * ctmscale);
		} else {
			return FALSE;
		}
		break;
	default:
		g_warning ("file %s: line %d: Illegal unit (base %d)", __FILE__, __LINE__, to->base);
		return FALSE;
		break;
	}

	return TRUE;
}

