/* -*- mode: C; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
#include <config.h>
#include <glib/gi18n-lib.h>
#include <string.h>
#include <gtk/gtkwidget.h>
#include "hippo-canvas-widget.h"
#include "hippo-canvas-style.h"

static void      hippo_canvas_widget_init                (HippoCanvasWidget       *widget);
static void      hippo_canvas_widget_class_init          (HippoCanvasWidgetClass  *klass);
static void      hippo_canvas_widget_iface_init          (HippoCanvasItemIface   *item_class);
static void      hippo_canvas_widget_dispose             (GObject                *object);
static void      hippo_canvas_widget_finalize            (GObject                *object);

static void hippo_canvas_widget_set_property (GObject      *object,
                                              guint         prop_id,
                                              const GValue *value,
                                              GParamSpec   *pspec);
static void hippo_canvas_widget_get_property (GObject      *object,
                                              guint         prop_id,
                                              GValue       *value,
                                              GParamSpec   *pspec);


/* Canvas item methods */
static void hippo_canvas_widget_set_context (HippoCanvasItem    *item,
                                             HippoCanvasContext *context);
static void hippo_canvas_widget_allocate    (HippoCanvasItem    *item,
                                             int                 width,
                                             int                 height,
                                             gboolean            origin_changed);

/* Canvas box methods */
static void hippo_canvas_widget_paint_below_children        (HippoCanvasBox  *box,
                                                             cairo_t         *cr,
                                                             HippoRectangle  *damage_box);
static void  hippo_canvas_widget_get_content_width_request  (HippoCanvasBox  *box,
                                                             int             *min_width_p,
                                                             int             *natural_width_p);
static void  hippo_canvas_widget_get_content_height_request (HippoCanvasBox  *box,
                                                             int              for_width,
                                                             int             *min_height_p,
                                                             int             *natural_height_p);

enum {
    NO_SIGNALS_YET,
    LAST_SIGNAL
};

/* static int signals[LAST_SIGNAL]; */

enum {
    PROP_0,
    PROP_WIDGET
};

G_DEFINE_TYPE_WITH_CODE(HippoCanvasWidget, hippo_canvas_widget, HIPPO_TYPE_CANVAS_BOX,
                        G_IMPLEMENT_INTERFACE(HIPPO_TYPE_CANVAS_ITEM, hippo_canvas_widget_iface_init));

static void
hippo_canvas_widget_init(HippoCanvasWidget *widget)
{

}

static HippoCanvasItemIface *item_parent_class;

static void
hippo_canvas_widget_iface_init(HippoCanvasItemIface *item_class)
{
    item_parent_class = g_type_interface_peek_parent(item_class);

    item_class->set_context = hippo_canvas_widget_set_context;
    item_class->allocate = hippo_canvas_widget_allocate;
}

static void
hippo_canvas_widget_class_init(HippoCanvasWidgetClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    HippoCanvasBoxClass *box_class = HIPPO_CANVAS_BOX_CLASS(klass);

    object_class->set_property = hippo_canvas_widget_set_property;
    object_class->get_property = hippo_canvas_widget_get_property;

    object_class->dispose = hippo_canvas_widget_dispose;
    object_class->finalize = hippo_canvas_widget_finalize;

    box_class->paint_below_children = hippo_canvas_widget_paint_below_children;
    box_class->get_content_width_request = hippo_canvas_widget_get_content_width_request;
    box_class->get_content_height_request = hippo_canvas_widget_get_content_height_request;

    g_object_class_install_property(object_class,
                                    PROP_WIDGET,
                                    g_param_spec_object("widget",
                                                        _("Widget"),
                                                        _("Widget to put in the canvas item"),
                                                        GTK_TYPE_WIDGET,
                                                        G_PARAM_READABLE | G_PARAM_WRITABLE));
}

static void
hippo_canvas_widget_dispose(GObject *object)
{
    HippoCanvasWidget *widget = HIPPO_CANVAS_WIDGET(object);

    if (widget->widget) {
        g_object_unref(widget->widget);
        widget->widget = NULL;
        g_object_notify(object, "widget");
    }

    G_OBJECT_CLASS(hippo_canvas_widget_parent_class)->dispose(object);
}

static void
hippo_canvas_widget_finalize(GObject *object)
{
    /* HippoCanvasWidget *widget = HIPPO_CANVAS_WIDGET(object); */


    G_OBJECT_CLASS(hippo_canvas_widget_parent_class)->finalize(object);
}

HippoCanvasItem*
hippo_canvas_widget_new(void)
{
    HippoCanvasWidget *widget = g_object_new(HIPPO_TYPE_CANVAS_WIDGET, NULL);


    return HIPPO_CANVAS_ITEM(widget);
}

static void
update_widget_visibility(HippoCanvasWidget *widget)
{
    int w, h;

    if (widget->widget == NULL)
        return;
    
    hippo_canvas_item_get_allocation(HIPPO_CANVAS_ITEM(widget), &w, &h);

    if (w == 0 || h == 0)
        gtk_widget_hide(widget->widget);
    else
        gtk_widget_show(widget->widget);
}

static void
hippo_canvas_widget_set_property(GObject         *object,
                                 guint            prop_id,
                                 const GValue    *value,
                                 GParamSpec      *pspec)
{
    HippoCanvasWidget *widget;

    widget = HIPPO_CANVAS_WIDGET(object);

    switch (prop_id) {
    case PROP_WIDGET:
        {
            GtkWidget *w = (GtkWidget*) g_value_get_object(value);
            if (widget->widget != w) {
                if (w) {
                    gtk_object_ref(GTK_OBJECT(w));
                    gtk_object_sink(GTK_OBJECT(w));
                }
                if (widget->widget)
                    g_object_unref(widget->widget);
                widget->widget = w;

                update_widget_visibility(widget);

                hippo_canvas_item_emit_request_changed(HIPPO_CANVAS_ITEM(widget));
            }
        }
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
hippo_canvas_widget_get_property(GObject         *object,
                                 guint            prop_id,
                                 GValue          *value,
                                 GParamSpec      *pspec)
{
    HippoCanvasWidget *widget;

    widget = HIPPO_CANVAS_WIDGET (object);

    switch (prop_id) {
    case PROP_WIDGET:
        g_value_set_object(value, widget->widget);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
on_context_style_changed(HippoCanvasContext   *context,
                         gboolean              resize_needed,
                         HippoCanvasWidget    *widget)
{
    HippoCanvasStyle *style = NULL;
    HippoCanvasTheme *theme = NULL;

    if (context != NULL)
        style = hippo_canvas_context_get_style(context);
    if (style != NULL)
        theme = hippo_canvas_style_get_theme(style);

    if (theme != widget->theme) {
        widget->theme = theme;

        if (HIPPO_CANVAS_WIDGET_GET_CLASS(widget)->theme_changed)
            HIPPO_CANVAS_WIDGET_GET_CLASS(widget)->theme_changed(widget);
    }
}

static void
hippo_canvas_widget_set_context(HippoCanvasItem    *item,
                                HippoCanvasContext *context)
{
    HippoCanvasBox *box = HIPPO_CANVAS_BOX(item);

    if (context == box->context)
        return;

    if (box->context) {
        hippo_canvas_context_unregister_widget_item(box->context, item);

        g_signal_handlers_disconnect_by_func(box->context,
                                             (gpointer)on_context_style_changed,
                                             context);
    }

    /* chain up, which invalidates our old context */
    item_parent_class->set_context(item, context);

    if (box->context) {
        hippo_canvas_context_register_widget_item(box->context, item);

        g_signal_connect(box->context, "style-changed",
                         G_CALLBACK(on_context_style_changed), item);

    }

    on_context_style_changed(context, TRUE, HIPPO_CANVAS_WIDGET(item));
}

static void
hippo_canvas_widget_allocate(HippoCanvasItem *item,
                             int              width,
                             int              height,
                             gboolean         origin_changed)
{
    int x, y, w, h;
    int widget_x, widget_y;
    GtkAllocation child_allocation;
    HippoCanvasWidget *widget;
    HippoCanvasBox *box;

    widget = HIPPO_CANVAS_WIDGET(item);
    box = HIPPO_CANVAS_BOX(item);
    
    /* get the box set up */
    item_parent_class->allocate(item, width, height, origin_changed);

    if (widget->widget == NULL)
        return;
    
    /* this probably queues a resize, which is not ideal */
    update_widget_visibility(widget);
    
    /* Now do the GTK allocation for the child widget */
    w = widget->widget->requisition.width;
    h = widget->widget->requisition.height;

    hippo_canvas_box_align(box, w, h, &x, &y, &w, &h);

    widget_x = 0;
    widget_y = 0;
    if (box->context)
        hippo_canvas_context_translate_to_widget(box->context, item,
                                                 &widget_x, &widget_y);

    child_allocation.x = widget_x + x;
    child_allocation.y = widget_y + y;
    child_allocation.width = MAX(w, 1);
    child_allocation.height = MAX(h, 1);

    gtk_widget_size_allocate(widget->widget, &child_allocation);
}

static void
hippo_canvas_widget_paint_below_children(HippoCanvasBox  *box,
                                         cairo_t         *cr,
                                         HippoRectangle  *damage_box)
{
    HippoCanvasWidget *widget = HIPPO_CANVAS_WIDGET(box);

    if (widget->widget == NULL)
        return;

    /* For now the HippoCanvas is responsible for drawing all widgets; it
     * plops them all on top, after rending all canvas items.
     * 
     * For no-window widgets, adding a canvas_context_paint_widget_item() to
     * call right here
     * is a simple way to get them in the right z-order, if we ever
     * need it.
     */
}

static void
hippo_canvas_widget_get_content_width_request(HippoCanvasBox *box,
                                              int            *min_width_p,
                                              int            *natural_width_p)
{
    HippoCanvasWidget *widget = HIPPO_CANVAS_WIDGET(box);
    int children_min_width, children_natural_width;
    int widget_width;
    GtkRequisition req;
    
    HIPPO_CANVAS_BOX_CLASS(hippo_canvas_widget_parent_class)->get_content_width_request(box,
                                                                                        &children_min_width,
                                                                                        &children_natural_width);

    /* Note that we get the widget's size request even if the GTK widget visibility flag is
     * false; that's because we are slaving GTK visibility to whether we have a nonzero
     * allocation (canvas item visibility), so it would make a circular mess if
     * our size request depended on visibility. This means you can't set visibility
     * on the underlying GtkWidget and expect anything sensible to happen.
     */
    
    if (widget->widget) {
        gtk_widget_size_request(widget->widget, &req);
        widget_width = req.width;
    } else {
        widget_width = 0;
    }

    if (min_width_p)
        *min_width_p = MAX(widget_width, children_min_width);
    if (natural_width_p)
        *natural_width_p = MAX(widget_width, children_natural_width);
}

static void
hippo_canvas_widget_get_content_height_request(HippoCanvasBox  *box,
                                               int              for_width,
                                               int             *min_height_p,
                                               int             *natural_height_p)
{
    HippoCanvasWidget *widget = HIPPO_CANVAS_WIDGET(box);
    int children_min_height, children_natural_height;
    int widget_height;
    GtkRequisition req;
    
    /* get height of children and the box padding */
    HIPPO_CANVAS_BOX_CLASS(hippo_canvas_widget_parent_class)->get_content_height_request(box,
                                                                                         for_width,
                                                                                         &children_min_height,
                                                                                         &children_natural_height);

    /* Note that we get the widget's size request even if the GTK widget visibility flag is
     * false; that's because we are slaving GTK visibility to whether we have a nonzero
     * allocation (canvas item visibility), so it would make a circular mess if
     * our size request depended on visibility. This means you can't set visibility
     * on the underlying GtkWidget and expect anything sensible to happen.
     */
    
    if (widget->widget) {
        /* We know a get_height_request was done first, so we can
         * just get widget->requisition instead of doing the size request
         * computation again.
         */
        gtk_widget_get_child_requisition(widget->widget, &req);
        widget_height = req.height;
    } else {
        widget_height = 0;
    }
    
    if (min_height_p)
        *min_height_p = MAX(widget_height, children_min_height);
    if (natural_height_p)
        *natural_height_p = MAX(widget_height, children_natural_height);
}
