/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* gturing.c - a Turing machine simulator.
 * Copyright (C) 1998 The Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <gnome.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnomeui/gnome-window.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include <gconf/gconf-client.h>

#include "turing.h"
#include "gturing.h"
#include "graph_editor.h"

#define SCROLL_WIDTH 10000

#define STATE 0
#define READ 1
#define WRITE 2
#define MOVE 3
#define NEW_STATE 4
#define ENTRIES_MAX 5

/* A few globals. */
const gchar *font_name = "courier Bold 12";

static char *progname;
static char *prog_message = NULL;
static char tape_string[1024] = "";
static char states_fname[1024] = "";

static long speed = 0;
static int stop_flag = 1;

static GtkWidget *rootw;
static GtkWidget *dialog = NULL;
static GtkWidget *state_clist = NULL;
static GtkWidget *entry;

static GtkWidget *vbox;
static GtkWidget *tapelabel;
static GtkWidget *headlabel;
/* static GtkWidget *hbox; */
static GtkTreeView *treeview_editor = NULL;
/* static GtkWidget *graph_editor = NULL; */

static GtkWidget *power;
/*static GtkWidget *stop;
static GtkWidget *play;
static GtkWidget *step; */
static GnomeAppBar *statusline;
/* static GtkWidget *save; */
static GtkWidget *edit_save = NULL;

GConfClient *gengine;
GError *gerror;

void
set_save_sens (gboolean saves)
{
	/* gtk_widget_set_sensitive (save, saves);

	   if (edit_save)
	   gtk_widget_set_sensitive (edit_save, saves); */
	return;
}

void
set_toolbar_sens (gboolean powers, gboolean stops,
		  gboolean plays, gboolean steps)
{
	/*
	   gtk_widget_set_sensitive (power, powers);
	   gtk_widget_set_sensitive (stop, stops);
	   gtk_widget_set_sensitive (play, plays);
	   gtk_widget_set_sensitive (step, steps);
	 */
	return;
}

void
set_tape (char *str)
{
	int len, i;
	gchar *tmp;

	len = strlen (str);
	tmp = g_malloc (len + 1);

	for (i = 0; i < len; i++)
		if (i == tm->pos)
			tmp[i] = '^';
		else
			tmp[i] = ' ';

	tmp[len] = 0;

	gtk_label_set (GTK_LABEL (tapelabel), str);
	gtk_label_set (GTK_LABEL (headlabel), tmp);

	g_free (tmp);
}

/* view_comment: Show the comments gotten from file and show them in a message box */
void
view_comment (void)
{
	GtkWidget *dialog;
	gchar *mess;

	if (prog_message && *prog_message && *prog_message != '\n')
		mess = prog_message;
	else
		mess = _("No comment for this program.");

	dialog = gtk_message_dialog_new (GTK_WINDOW (rootw),
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
					 "%s", mess);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

void
set_states_fname (char *str)
{
	gchar *c;

	if ((*states_fname != 0) && (*tape_string != 0))
		power_callback (power, NULL);
	else
		set_toolbar_sens (FALSE, FALSE, FALSE, FALSE);

	if ((c = strrchr (states_fname, '/')) != NULL)
		c++;
	else
		c = states_fname;

	state_clist_load_states ();
	state_clist_select_state (tm);
	gtk_window_set_title (GTK_WINDOW (rootw), c);

	turing_table_editor_set_model (treeview_editor, tm);
}

void
states_fname_examples_callback (GtkWidget * ok_button, gpointer data)
{
	GtkFileSelection *fsel;

	fsel = GTK_FILE_SELECTION (data);
#define GTURING_EXAMPLES_DIR "foo"
	gtk_file_selection_set_filename (fsel, GTURING_EXAMPLES_DIR);
}

void
states_fname_callback (GtkWidget * ok_button, gpointer data)
{
	gboolean action_save;
	gchar *fname, *comment;
	GtkWidget *text;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;

	fname =
	    g_strdup (gtk_file_selection_get_filename
		      (GTK_FILE_SELECTION (dialog)));

	action_save = (gboolean) data;

	if (action_save) {
		text = g_object_get_data (G_OBJECT (dialog), "text");
		/*
		   comment =
		   gtk_editable_get_chars (GTK_EDITABLE (text), 0, -1); */
		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
		gtk_text_buffer_get_bounds (buffer, &start, &end);
		comment =
		    gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
		g_print ("Comentario: %s\n", comment);

		if (turing_fwrite_states (tm->statehead, fname, comment))
			*states_fname = 0;	/*error */
		else
			strncpy (states_fname, fname, 1024);

		if (prog_message)
			free (prog_message);

		prog_message = strdup (comment);
	} else {
		if (turing_fread_states (tm, fname)) {
			*states_fname = 0;	/*error */
		} else {
			strncpy (states_fname, fname, 1024);

			if (prog_message)
				free (prog_message);

			prog_message = turing_fread_comments (fname);
			view_comment ();
			/*
			   set_save_sens (TRUE); 
			 */
		}
	}

	set_states_fname (fname);
	g_free (fname);

	gnome_config_set_string ("/gTuring/Options/program", states_fname);
	gconf_client_set_string (gengine, "/apps/gturing/program", states_fname, NULL);
	gtk_widget_destroy (dialog);
	dialog = NULL;
}

void
tape_string_callback (GtkWidget * ok_button, gpointer data)
{
	strncpy (tape_string, gtk_entry_get_text (GTK_ENTRY (entry)),
		 1024);
	turing_set_tape (tm, tape_string);
	set_tape (tape_string);

	gtk_widget_destroy (dialog);
	dialog = NULL;

	if (tm->statehead && (*tape_string != 0))
		power_callback (power, NULL);
	else
		set_toolbar_sens (FALSE, FALSE, FALSE, FALSE);

	gnome_config_set_string ("/gTuring/Options/tape", tape_string);
	gconf_client_set_string (gengine, "/apps/gturing/tape", tape_string, NULL);
}

void
speed_callback (GtkWidget * ok_button, gpointer data)
{
	speed = atoi (gtk_entry_get_text (GTK_ENTRY (entry)));

	gtk_widget_destroy (dialog);
	dialog = NULL;

	gconf_client_set_int (gengine, "/apps/gturing/speed", speed, NULL);
	gnome_config_set_int ("/gTuring/Options/speed", speed);
}

void
cancel_callback (GtkWidget * widget, gpointer data)
{
	void *p;

	if ((p = gtk_object_get_user_data (GTK_OBJECT (widget))) != NULL)
		g_free (p);

	gtk_widget_destroy (dialog);
	dialog = NULL;
}

void
states_view_edit_set_clicked_callback (GtkWidget * widget, gpointer data)
{
	GtkWidget **entries, *message;
	gchar *c[ENTRIES_MAX];
	turing_state state, *s;
	int i;

	entries = gtk_object_get_data (GTK_OBJECT (widget), "entries");

	for (i = 0; i < ENTRIES_MAX; i++)
		c[i] = gtk_entry_get_text (GTK_ENTRY (entries[i]));

	/* FIXME: states should be able to be greater than 9. */
	if (!(*c[STATE] >= '0' && *c[STATE] <= '9') ||
	    (*c[MOVE] != 'l' && *c[MOVE] != 'r') ||
	    !(*c[NEW_STATE] >= '0' && *c[NEW_STATE] <= '9')) {
		gtk_widget_grab_focus (entries[0]);
		message =
		    gnome_message_box_new (_
					   ("States must be numbers. Move can only be 'r' or 'l'."),
					   GNOME_MESSAGE_BOX_ERROR,
					   GNOME_STOCK_BUTTON_OK, NULL);
		gtk_widget_show (message);
		return;
	}

	state.no = atoi (c[STATE]);
	state.read = *c[READ];
	state.write = *c[WRITE];
	state.move = *c[MOVE];
	state.new = atoi (c[NEW_STATE]);
	state.next = NULL;

	turing_set_state (tm, state);

	gtk_clist_freeze (GTK_CLIST (state_clist));
	state_clist_load_states ();
	gtk_clist_thaw (GTK_CLIST (state_clist));

	for (s = tm->statehead; s; s = s->next)
		if ((s->no == state.no) && (s->read == state.read))
			break;

	if (s)
		for (i = 0;; i++)
			if (gtk_clist_get_row_data
			    (GTK_CLIST (state_clist), i) == s) {
				gtk_clist_select_row (GTK_CLIST
						      (state_clist), i, 0);
				break;
			}

	if (tm->tapehead)
		power_do ();

	set_save_sens (TRUE);

	gtk_widget_grab_focus (entries[0]);
}

void
states_view_entry_grab_focus_callback (GtkWidget * widget, gpointer data)
{
	GtkWidget **entries;
	int i;

	entries = gtk_object_get_data (GTK_OBJECT (widget), "entries");

	for (i = 0; i < ENTRIES_MAX; i++)
		gtk_editable_select_region (GTK_EDITABLE (entries[i]), 0,
					    0);

	gtk_editable_select_region (GTK_EDITABLE (widget), 0, -1);
}

void
states_view_edit_callback (GtkWidget * widget, gpointer data)
{
	GtkWidget *w;
	GtkWidget *table, *label, *frame, *button;
	GtkWidget **entries;
	gint i;
	gchar *labels[] =
	    { N_("State"), N_("Read"), N_("Write"), N_("Move"),
		N_("New State"), NULL
	};

	dialog = gtk_message_dialog_new (GTK_WINDOW (rootw),
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_OK, "%s",
					 _
					 ("Editing is still alpha. Come back later or hack."));
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	gtk_widget_set_sensitive (widget, FALSE);

	w = GTK_WIDGET (data);
	entries = g_new (GtkWidget *, ENTRIES_MAX);
	gtk_object_set_data (GTK_OBJECT (state_clist), "entries", entries);

	frame = gtk_frame_new (_("Edit Or Create Row"));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (w)->vbox), frame, FALSE,
			    FALSE, 0);
	table = gtk_table_new (2, 2 * ENTRIES_MAX + 2, FALSE);
	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_container_set_border_width (GTK_CONTAINER (table), 8);
	gtk_table_set_row_spacings (GTK_TABLE (table), 4);
	gtk_table_set_col_spacings (GTK_TABLE (table), 8);

	edit_save = button = gtk_button_new_from_stock (GTK_STOCK_SAVE_AS);
	/*gnome_pixmap_button (gnome_stock_pixmap_widget
	   (w, GNOME_STOCK_PIXMAP_SAVE_AS),
	   _("Save")); */

	g_signal_connect (G_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (save_call), NULL);
	/* segfault: FIXME */
	/* gtk_widget_set_sensitive (edit_save,
	   GTK_WIDGET_IS_SENSITIVE (save)); */
	gtk_table_attach (GTK_TABLE (table), button, ENTRIES_MAX * 2 + 1,
			  ENTRIES_MAX * 2 + 2, 1, 2, 0,
			  GTK_FILL | GTK_EXPAND, 0, 0);


	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	/*
	   gnome_pixmap_button (gnome_stock_pixmap_widget
	   (w, GNOME_STOCK_PIXMAP_ADD), _("Set"));
	 */
	gtk_object_set_data (GTK_OBJECT (button), "entries", entries);
	gtk_table_attach (GTK_TABLE (table), button, ENTRIES_MAX * 2,
			  ENTRIES_MAX * 2 + 1, 1, 2, 0,
			  GTK_FILL | GTK_EXPAND, 0, 0);

	for (i = 0; labels[i]; i++) {
#ifdef ENABLE_NLS
		label = gtk_label_new (_(labels[i]));
#else
		label = gtk_label_new (labels[i]);
#endif
		entries[i] = gtk_entry_new_with_max_length (1);
		gtk_object_set_data (GTK_OBJECT (entries[i]), "entries",
				     entries);
		gtk_widget_set_usize (entries[i], 20, 0);

		g_signal_connect (GTK_OBJECT (entries[i]), "grab_focus",
				  GTK_SIGNAL_FUNC
				  (states_view_entry_grab_focus_callback),
				  NULL);
		if (i != 0)
			g_signal_connect_swapped (GTK_OBJECT
						  (entries[i - 1]),
						  "activate",
						  GTK_SIGNAL_FUNC
						  (gtk_widget_grab_focus),
						  GTK_OBJECT (entries[i]));

		gtk_table_attach (GTK_TABLE (table), label, i * 2,
				  i * 2 + 1, 0, 1, 0, 0, 0, 0);
		gtk_table_attach (GTK_TABLE (table), entries[i], i * 2,
				  i * 2 + 1, 1, 2, 0, 0, 0, 0);
		gtk_table_attach (GTK_TABLE (table), gtk_vseparator_new (),
				  i * 2 + 1, i * 2 + 2, 0, 2, 0,
				  GTK_FILL | GTK_EXPAND, 0, 0);
	}
	g_signal_connect_swapped (GTK_OBJECT (entries[i - 1]), "activate",
				  GTK_SIGNAL_FUNC (gtk_widget_grab_focus),
				  GTK_OBJECT (button));
	g_signal_connect_swapped (GTK_OBJECT (button), "clicked",
				  GTK_SIGNAL_FUNC (gtk_widget_grab_focus),
				  GTK_OBJECT (entries[0]));
	g_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC
			  (states_view_edit_set_clicked_callback), NULL);

	gtk_widget_show_all (frame);
}

void
states_view_close_callback (GtkWidget * w, gpointer data)
{
	GtkWidget **entries;

	entries = gtk_object_get_data (GTK_OBJECT (w), "entries");
	if (entries)
		g_free (entries);

	state_clist = NULL;
	edit_save = NULL;

	gtk_widget_destroy (w);
}

void
state_clist_select_state (turing * t)
{
	gint i, tmp;
	char buff[20];

	if (t->actualstate) {
		snprintf (buff, 20, _("State: %d"), t->actualstate->no);
		gnome_appbar_set_status (GNOME_APPBAR (statusline), buff);
	} else {
		gnome_appbar_set_status (GNOME_APPBAR (statusline),
					 _("Stopped"));
	}

	if (state_clist) {
		tmp = i =
		    GPOINTER_TO_INT (gtk_object_get_user_data
				     (GTK_OBJECT (state_clist)));
		while (i--)
			if (t->actualstate == (turing_state *)
			    gtk_clist_get_row_data (GTK_CLIST
						    (state_clist), i))
				break;

		if (i >= 0) {
			gtk_clist_select_row (GTK_CLIST (state_clist), i,
					      0);
			/* bug: moveto corrupts the clist if all the rows appear in the list */
/*			if (tmp > 11)*/
			gtk_clist_moveto (GTK_CLIST (state_clist),
					  (i - 5 < 5) ? 0 : i - 5, 0, 0,
					  0);

		}
	}
}

int
next_state (turing * t)
{
	int ret;

	ret = turing_run_state (t);
	state_clist_select_state (t);

	return ret;
}

void
power_do (void)
{
	stop_flag = 1;
	tm->state = 0;
	tm->pos = 0;

	turing_set_tape (tm, tape_string);
	set_tape (tape_string);

	next_state (tm);
	set_toolbar_sens (FALSE, FALSE, TRUE, TRUE);
}

void
power_callback (GtkWidget * power_button, gpointer data)
{
	power_do ();
}

gint
do_play (gpointer data)
{
	char *tmp;
	int cont;

	cont = FALSE;

	if (!stop_flag) {
		tmp = turing_get_tape (tm);
		set_tape (tmp);
		free (tmp);

		cont = next_state (tm);

		if (!cont)
			set_toolbar_sens (TRUE, FALSE, stop_flag,
					  stop_flag);
	}

	return cont;
}

void
play_callback (GtkWidget * play_button, gpointer data)
{
	set_toolbar_sens (TRUE, TRUE, FALSE, FALSE);

	stop_flag = 0;
	gtk_timeout_add (speed, do_play, NULL);
}

void
step_callback (GtkWidget * step_buttton, gpointer data)
{
	char *tmp;

	tmp = turing_get_tape (tm);
	set_tape (tmp);
	free (tmp);

	if (!next_state (tm)) {
		set_toolbar_sens (TRUE, FALSE, FALSE, FALSE);
		gnome_appbar_set_status (GNOME_APPBAR (statusline),
					 _("Stopped"));
	}

	gtk_widget_set_sensitive (power, TRUE);
}

void
stop_callback (GtkWidget * stop_button, gpointer data)
{
	stop_flag = 1;

	set_toolbar_sens (TRUE, FALSE, TRUE, TRUE);
}

void
prompt (char *title, char *msg, GtkSignalFunc callback, char *def)
{
	GtkWidget *vbox, *label;

	if (dialog != NULL)
		return;

	dialog = gnome_dialog_new (title, GNOME_STOCK_BUTTON_CANCEL,
				   GNOME_STOCK_BUTTON_OK, NULL);
	vbox = GNOME_DIALOG (dialog)->vbox;
	label = gtk_label_new (msg);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
	entry = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (entry), def);
	gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);

	g_signal_connect (G_OBJECT (dialog), "destroy",
			  GTK_SIGNAL_FUNC (cancel_callback), NULL);
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
				     GTK_SIGNAL_FUNC (callback), NULL);
	gnome_dialog_button_connect_object (GNOME_DIALOG (dialog), 0,
					    GTK_SIGNAL_FUNC
					    (cancel_callback),
					    GTK_OBJECT (dialog));

	gtk_widget_show_all (vbox);
	gtk_widget_show (dialog);
}

void
save_call (GtkWidget * w, gpointer data)
{
	GtkWidget *fsel;
	GtkWidget *frame, *frame2, *scrolled, *text;
	GtkTextBuffer *buffer;

	dialog = fsel =
	    gtk_file_selection_new (_("Save gTuring Program File"));

	frame2 = gtk_frame_new (_("Comment"));
	gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (fsel)->main_vbox),
			    frame2, TRUE, TRUE, 0);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 8);
	gtk_container_add (GTK_CONTAINER (frame2), frame);
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (frame), scrolled);
	buffer = gtk_text_buffer_new (NULL);
	text = gtk_text_view_new_with_buffer (buffer);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (text), TRUE);

	if (prog_message)
		gtk_text_buffer_set_text (buffer, prog_message,
					  strlen (prog_message));

	gtk_container_add (GTK_CONTAINER (scrolled), text);

	gtk_widget_show_all (frame2);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION (fsel),
					 states_fname);
	gtk_object_set_data (GTK_OBJECT (fsel), "text", text);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->ok_button),
			  "clicked",
			  GTK_SIGNAL_FUNC (states_fname_callback),
			  (gpointer) TRUE);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->cancel_button),
			  "clicked", GTK_SIGNAL_FUNC (cancel_callback),
			  NULL);
	gtk_widget_show (fsel);
}

void
open_call (GtkWidget * w, gpointer data)
{
	GtkWidget *fsel;
	GtkWidget *button;

	dialog = fsel =
	    gtk_file_selection_new (_("Open gTuring Program File"));

	button = gtk_button_new_with_label ("Examples");
	gtk_container_add (GTK_CONTAINER
			   (GTK_FILE_SELECTION (fsel)->button_area),
			   button);
	gtk_widget_show (button);

	g_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC
			  (states_fname_examples_callback), fsel);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->ok_button),
			  "clicked",
			  GTK_SIGNAL_FUNC (states_fname_callback),
			  (gpointer) FALSE);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->cancel_button),
			  "clicked", GTK_SIGNAL_FUNC (cancel_callback),
			  NULL);
	gtk_widget_show (fsel);
}

void
tape_call (GtkWidget * w, gpointer data)
{
	prompt (_("Tape Setting"), _("Please enter the tape:"),
		GTK_SIGNAL_FUNC (tape_string_callback), tape_string);
}

void
state_clist_load_states (void)
{
	char *text[5];
	turing_state *s;
	gint i;

	if (state_clist) {
		gtk_clist_clear (GTK_CLIST (state_clist));

		for (i = 0; i < 5; i++)
			text[i] = malloc (6);

		i = 0;
		for (s = tm->statehead; s; s = s->next) {
			snprintf (text[0], 6, "%d", s->no);
			text[1][0] = s->read;
			text[1][1] = 0;
			text[2][0] = s->write;
			text[2][1] = 0;
			text[3][0] = s->move;
			text[3][1] = 0;
			snprintf (text[4], 6, "%d", s->new);
			i++;
			gtk_clist_insert (GTK_CLIST (state_clist), 0,
					  text);
		}

		gtk_object_set_user_data (GTK_OBJECT (state_clist),
					  GINT_TO_POINTER (i));

		for (s = tm->statehead; i--; s = s->next)
			gtk_clist_set_row_data (GTK_CLIST (state_clist), i,
						s);

		for (i = 0; i < 5; i++)
			free (text[i]);
	}
}

void
state_clist_select_row_callback (GtkCList * clist, gint row, gint column,
				 GdkEventButton * event,
				 gpointer user_data)
{
	GtkWidget **entries;

	entries = gtk_object_get_data (GTK_OBJECT (clist), "entries");
	if (entries) {
		int i;
		gchar *txt;

		for (i = 0; i < ENTRIES_MAX; i++) {
			gtk_clist_get_text (GTK_CLIST (state_clist), row,
					    i, &txt);
			gtk_entry_set_text (GTK_ENTRY (entries[i]), txt);
		}
	}
}

void
view_states_call (GtkWidget * widget, gpointer data)
{
	char *text[5] = { N_("State"), N_("Read"), N_("Write"), N_("Move"),
		N_("New State")
	};
	GtkWidget *w;
	GtkWidget *scrolled;
	int i;

	if (state_clist)
		return;

	w = gnome_dialog_new (_("Machine's states"), NULL);
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (w),
						"  Edit >>",
						GTK_STOCK_PROPERTIES);
	gnome_dialog_append_button (GNOME_DIALOG (w), GTK_STOCK_CLOSE);
	gtk_window_set_policy (GTK_WINDOW (w), TRUE, TRUE, FALSE);
	gnome_dialog_button_connect (GNOME_DIALOG (w), 0,
				     GTK_SIGNAL_FUNC
				     (states_view_edit_callback), w);
	gnome_dialog_button_connect_object (GNOME_DIALOG (w), 1,
					    GTK_SIGNAL_FUNC
					    (states_view_close_callback),
					    GTK_OBJECT (w));

#ifdef ENABLE_NLS
	{
		int i = 0;
		for (i = 0; i < 5; i++)
			text[i] = _(text[i]);
	}
#endif

	state_clist = gtk_clist_new_with_titles (5, text);
	gtk_clist_set_selection_mode (GTK_CLIST (state_clist),
				      GTK_SELECTION_SINGLE);
	gtk_clist_column_titles_passive (GTK_CLIST (state_clist));
	for (i = 0; i < 5; i++)
		gtk_clist_set_column_width (GTK_CLIST (state_clist), i,
					    60);
	gtk_widget_set_usize (state_clist, 360, 200);

	state_clist_load_states ();
	state_clist_select_state (tm);

	g_signal_connect (GTK_OBJECT (state_clist), "select_row",
			  GTK_SIGNAL_FUNC
			  (state_clist_select_row_callback), NULL);

	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled), state_clist);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (w)->vbox), scrolled,
			    TRUE, TRUE, 0);
	gtk_widget_show_all (GNOME_DIALOG (w)->vbox);
	gtk_widget_show (w);
}

void
view_comment_call (GtkWidget * w, gpointer data)
{
	view_comment ();
}

void
exit_call (GtkWidget * w, gpointer data)
{
	gtk_main_quit ();
	exit (0);
}

void
playspeed_call (GtkWidget * w, gpointer data)
{
	char buff[1024];

	sprintf (buff, "%ld", speed);
	prompt (_("Animation Speed"), _("Miliseconds between steps:"),
		GTK_SIGNAL_FUNC (speed_callback), buff);
}

void
viewstates_call (GtkWidget * w, gpointer data)
{
}


void
operation_call (GtkWidget * w, gpointer data)
{
}

void
about_call (GtkWidget * w, gpointer data)
{
	/* TODO: Add logo */
	GtkWidget *about_box = NULL;
	gchar *authors[] = { "Arturo Espinosa",
		g_locale_to_utf8 ("Germn Poo-Caamao", -1, NULL, NULL,
				  NULL),
		NULL
	};
	gchar *documentors[] = { NULL };
	gchar *translators[] = { "Vasif Ismailogu MD", 
		"Zbigniew Chyla",
		"Abel Cheung", 
		"Christophe Merlet",
		g_locale_to_utf8 ("Carlos Perell Marn", -1, NULL, NULL,
				  NULL),
		"Pablo Saratxaga",
		NULL
	};

	about_box = gnome_about_new ("gTuring",
				    VERSION,
				    g_locale_to_utf8
				    ("(c) 1997-2001 Free Software Foundation",
				      -1, NULL, NULL, NULL),
				    _("A Turing machine for GNOME"), 
					(const char **) authors,
				    (const char **) documentors,
				    (const char **) translators /*"Translation creds" */ ,
				    NULL /* logo pixbuf */ );

	gtk_dialog_run (GTK_DIALOG (about_box));

	gtk_widget_destroy (about_box);
}

void
create_machine (void)
{
	PangoFontDescription *font_desc = NULL;

	font_desc = pango_font_description_from_string (font_name);
	g_return_if_fail (font_desc != NULL);

	gtk_widget_modify_font (GTK_WIDGET (tapelabel), font_desc);
	gtk_widget_modify_font (GTK_WIDGET (headlabel), font_desc);

	pango_font_description_free (font_desc);
}

/* MENUS & TOOLBARS - gpoo */
/* Used as a callback for menu items in the GnomeAppHelper test; just prints the string contents of
 * the data pointer.
 */
static void
item_activated (GtkWidget * widget, gpointer data)
{
	printf ("%s activated\n", (char *) data);
}

/* Menu definitions for the GnomeAppHelper test */

static GnomeUIInfo helper_file_menu[] = {
	{GNOME_APP_UI_ITEM, "_New", "Create a new file", item_activated,
	 "file/new", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 'n',
	 GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, "_Open...", "Open an existing file", open_call,
	 "file/open", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 'o',
	 GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, "_Save", "Save the current file", save_call,
	 "file/save", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 's',
	 GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, "Save _as...",
	 "Save the current file with a new name", item_activated,
	 "file/save as", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE_AS, 0, 0, NULL},

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "_Print...", "Print the current file",
	 item_activated, "file/print", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PRINT, 'p',
	 GDK_CONTROL_MASK, NULL},

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "E_xit", "Exit the program", exit_call,
	 "file/exit", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 'q',
	 GDK_CONTROL_MASK, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo helper_view_menu[] = {
	{GNOME_APP_UI_ITEM, "_Comment...", "View the program's comment.",
	 view_comment_call, "view/comment", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UNDO, 'z',
	 GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, "_States...",
	 "Open a table with the machine's states.",
	 view_states_call, "view/states", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REDO, 0, 0, NULL},

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "Graph editor",
	 "Open a graphical interface to edit the machine's states.",
	 item_activated, "view/graph", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT, 'x',
	 GDK_CONTROL_MASK, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo helper_settings_menu[] = {
	{GNOME_APP_UI_ITEM, "_Play speed...", "Set play speed",
	 playspeed_call, "settings/play", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UNDO, 'z',
	 GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, "_Tape...", "Set the tape to play", tape_call,
	 "settings/tape", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REDO, 0, 0, NULL},

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "P_references... ",
	 "Personalize your gturing's settings", item_activated,
	 "settings/preferences", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REDO, 0, 0, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo helper_help_menu[] = {
	{GNOME_APP_UI_ITEM, "_About...",
	 "Displays information about the program", about_call,
	 "help/about", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT, 0, 0, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo helper_main_menu[] = {
	{GNOME_APP_UI_SUBTREE, "_File", "File operations",
	 helper_file_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	{GNOME_APP_UI_SUBTREE, "_View", "Views of states machine",
	 helper_view_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	{GNOME_APP_UI_SUBTREE, "_Settings", "gTturing settings",
	 helper_settings_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	{GNOME_APP_UI_SUBTREE, "_Help", "Help on the program",
	 helper_help_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	GNOMEUIINFO_END
};

/* Toolbar definition for the GnomeAppHelper test */

static GnomeUIInfo helper_toolbar[] = {
	/* New machine's states should be call an machine's editor, graph or
	   table */
	/* {GNOME_APP_UI_ITEM, "New", "Create a new machine's states", item_activated,
	   "toolbar/new", NULL,
	   GNOME_APP_PIXMAP_STOCK, GTK_STOCK_NEW, 0, 0, NULL}, */
	{GNOME_APP_UI_ITEM, "Open", "Open an existing file", open_call,
	 "toolbar/open", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_OPEN, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Save", "Save the current file", save_call,
	 "toolbar/save", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_SAVE, 0, 0, NULL},
	/*  Print will not ready early. The print stuff must be related
	   to print the graph and machine's states */
	/*{GNOME_APP_UI_ITEM, "Print", "Print the current file",
	   item_activated, "toolbar/print", NULL,
	   GNOME_APP_PIXMAP_STOCK, GTK_STOCK_PRINT, 0, 0, NULL}, */

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "Reset", "Reset the machine's states",
	 power_callback, "toolbar/undo", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_REFRESH, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Stop", "Stop the execution of tape",
	 stop_callback, "toolbar/redo", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_STOP, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Play", "Start the tape", play_callback,
	 "toolbar/cut", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GO_FORWARD, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Step", "Play the tape step by step",
	 step_callback, "toolbar/copy", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GOTO_LAST, 0, 0, NULL},

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "Exit", "Exit of gTuring", exit_call,
	 "toolbar/print", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_QUIT, 0, 0, NULL},

	GNOMEUIINFO_END
};


void
init_interface (int argc, char *argv[])
{
	static GtkWidget *hpaned;
	/* GtkWidget *graph; */
	GtkWidget *treeview; /* treeview with scrolled window */

	rootw = gnome_app_new ("gTuring", _("A Turing machine"));
	gnome_app_create_menus (GNOME_APP (rootw), helper_main_menu);
	gnome_app_create_toolbar (GNOME_APP (rootw), helper_toolbar);

	statusline =
	    GNOME_APPBAR (gnome_appbar_new
			  (FALSE, TRUE, GNOME_PREFERENCES_NEVER));
	gnome_app_set_statusbar (GNOME_APP (rootw),
				 GTK_WIDGET (statusline));

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	tapelabel = gtk_label_new (_("Welcome to gTuring"));
	headlabel = gtk_label_new (_("^                 "));
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (tapelabel), FALSE,
			    FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (headlabel), FALSE,
			    FALSE, 0);

	hpaned = gtk_hpaned_new ();

	gtk_container_set_border_width (GTK_CONTAINER (hpaned), 5);

	/*graph = graph_editor_new (&(graph_editor));*/
	treeview = turing_table_editor_new (&treeview_editor);

	/*gtk_paned_add1 (GTK_PANED (hpaned), graph);*/
	gtk_paned_add2 (GTK_PANED (hpaned), treeview);

	gtk_box_pack_start (GTK_BOX (vbox), hpaned, TRUE,TRUE, 0);
/*	set_save_sens (FALSE);
	set_toolbar_sens (FALSE, FALSE, FALSE, FALSE);*/

	gnome_app_set_contents (GNOME_APP (rootw), vbox);
	g_signal_connect (GTK_OBJECT (rootw), "destroy",
			  GTK_SIGNAL_FUNC (exit_call), NULL);

	create_machine ();
	gtk_widget_show_all (rootw);
}

void
init_globals (void)
{
	gchar * tape, * fname;
	
	tm = turing_new ();
	
	tape = gconf_client_get_string (gengine, "/apps/gturing/tape", NULL);
	strncpy (tape_string, tape, strlen (tape));
	set_tape (tape);
	/*strncpy (tape_string,
		gconf_client_get_string (gengine, "/gTuring/Options/tape", NULL), 1024);*/
	fname =
		gconf_client_get_string (gengine, "/apps/gturing/program", NULL);
	/*strncpy (states_fname, fname, strlen (fname));*/
	
	speed = gconf_client_get_int (gengine, "/apps/gturing/speed", NULL);

	/*strncpy (tape_string,
		 gnome_config_get_string ("/gTuring/Options/tape="), 1024);
	strncpy (states_fname,
		 gnome_config_get_string ("/gTuring/Options/program="),
		 1024);
	speed = gnome_config_get_int ("/gTuring/Options/speed=50");*/
}

/* Want to add command-line options? This is the right place. */
void
parse_args (int argc, char *argv[])
{
/*	int i;
	char help[] = N_("%s:\n\nUsage: %s [options] [states_file] [tape_string]\n\n\
-?  --help           Display this help and exit.\n");*/

	progname = argv[0];
/*	i = 1;
	
	while(i < argc)
		{
			if ((strcmp(argv[i], _("--help")) == 0) ||
							 (strcmp(argv[i], "-?") == 0))
				{
					fprintf(stderr, help, progname, progname);
					exit (1);
				}
			else if (*states_fname == 0)
				strcpy(states_fname, argv[i]);
			else if (*tape_string == 0)
				strcpy(tape_string, argv[i]);
			else
				{
					fprintf(stderr, help, progname, progname);
					exit (1);
				}
			
			i++;
		}*/
}

/* The main. */
int
main (int argc, char *argv[])
{
	gnome_score_init ("gturing");


#ifdef ENABLE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, GTURING_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	parse_args (argc, argv);
	gnome_init ("gturing", VERSION, argc, argv);

	gconf_init (argc, argv, NULL);
	gengine = gconf_client_get_default ();
	
	init_interface (argc, argv);
	init_globals ();
	
	if (*states_fname != 0)
		set_states_fname (states_fname);

	gtk_main ();

	return 0;
}
