/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <string.h>
#include "gtm.h"
#include "file-data.h"
#include "download-info.h"
#include "dialogs.h"
#include "main-window.h"

typedef struct
{
    GtkWidget *url;              /* Label */
    GtkWidget *file;             /* Label */
    GtkWidget *size;             /* Label */
    GtkWidget *state;            /* Label */
    GtkWidget *retr_progress;    /* ProgressBar */
    GtkWidget *retrieved;        /* Label */
    GtkWidget *estimated;        /* Label */
    GtkWidget *remain;           /* Label */
    GtkWidget *total_time;       /* Label */
    GtkWidget *total_speed;      /* Label */
    GtkWidget *sess_time;        /* Label */
    GtkWidget *sess_speed;       /* Label */
    GtkWidget *local_dir;        /* Label */
    GtkWidget *use_proxy;        /* Toggle Button */
    GtkWidget *use_auto_dl;      /* Toggle Button */
} Info;

static Info info;

/* Function to convert time to string */
static void
convert_time2str (gchar *buffer, guint32 time)
{
    sprintf (buffer, "%02d:%02d:%02d", (gint)(time / 3600), 
	     (gint)((time % 3600) / 60), (gint)(time % 60));
}


/* Functions to create the various frames with download information */
static GtkWidget *
create_frame_location (void)
{
    GtkWidget *frame, *table, *label, *sw;

    frame = gtk_frame_new (_ ("Location"));
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
    table = gtk_table_new (3, 2, FALSE);
    gtk_table_set_col_spacing (GTK_TABLE (table), 0, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (table),
                                    GNOME_PAD_SMALL);
    sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw),
                                           table);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				    GTK_POLICY_ALWAYS,
				    GTK_POLICY_NEVER);
    gtk_container_add (GTK_CONTAINER (frame), sw);
    
    label = gtk_label_new (_ ("Url:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.url = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.url), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.url, 1, 2, 0, 1);

    label = gtk_label_new (_ ("File:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.file = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.file), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.file, 1, 2, 1, 2);
    
    label = gtk_label_new (_ ("Size:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.size = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.size), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.size, 1, 2, 2, 3);

    return frame;
}

static GtkWidget *
create_frame_status (void)
{
    GtkWidget *frame, *table, *label;

    frame = gtk_frame_new (_ ("Status"));
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
    table = gtk_table_new (5, 2, FALSE);
    gtk_table_set_col_spacing (GTK_TABLE (table), 0, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (table),
                                    GNOME_PAD_SMALL);
    gtk_container_add (GTK_CONTAINER (frame), table);
    
    label = gtk_label_new (_ ("State:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.state = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.state), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.state, 1, 2, 0, 1);

    label = gtk_label_new (_ ("Retrieve progress:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.retr_progress = gtk_progress_bar_new ();
    gtk_table_attach_defaults (GTK_TABLE (table), info.retr_progress,
                               1, 2, 1, 2);

    label = gtk_label_new (_ ("Retrieved:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.retrieved = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.retrieved), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.retrieved, 1, 2, 2, 3);

    label = gtk_label_new (_ ("Estimated:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.estimated = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.estimated), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.estimated, 1, 2, 3, 4);

    label = gtk_label_new (_ ("Remain:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.remain = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.remain), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.remain, 1, 2, 4, 5);

    return frame;
}

static GtkWidget *
create_frame_total (void)
{
    GtkWidget *frame, *table, *label;

    frame = gtk_frame_new (_ ("Total"));
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
    table = gtk_table_new (2, 2, FALSE);
    gtk_table_set_col_spacing (GTK_TABLE (table), 0, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (table),
                                    GNOME_PAD_SMALL);
    gtk_container_add (GTK_CONTAINER (frame), table);
    
    label = gtk_label_new (_ ("Total time:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.total_time = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.total_time), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.total_time,
                               1, 2, 0, 1);

    label = gtk_label_new (_ ("Total speed:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.total_speed = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.total_speed), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.total_speed,
                               1, 2, 1, 2);

    return frame;
}

static GtkWidget *
create_frame_sess (void)
{
    GtkWidget *frame, *table, *label;

    frame = gtk_frame_new (_ ("Session"));
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
    table = gtk_table_new (2, 2, FALSE);
    gtk_table_set_col_spacing (GTK_TABLE (table), 0, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (table),
                                    GNOME_PAD_SMALL);
    gtk_container_add (GTK_CONTAINER (frame), table);
    
    label = gtk_label_new (_ ("Session time:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.sess_time = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.sess_time), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.sess_time,
                               1, 2, 0, 1);

    label = gtk_label_new (_ ("Session speed:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.sess_speed = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.sess_speed), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.sess_speed,
                               1, 2, 1, 2);

    return frame;
}

static void
toggle_proxy_cb (GtkWidget *widget, gpointer data)
{
    FileData *file_data;

    file_data = gtk_clist_get_row_data (GTK_CLIST (dl_file_list),
                                        GTK_CLIST (dl_file_list)->focus_row);
    file_data_set_use_proxy (file_data,
                             gtk_toggle_button_get_active (
                                 GTK_TOGGLE_BUTTON (info.use_proxy)));
    download_info_update (file_data);
    dialog_prop (FALSE, file_data);
}

static void
toggle_auto_dl_cb (GtkWidget *widget, gpointer data)
{
    FileData *file_data;

    file_data = gtk_clist_get_row_data (GTK_CLIST (dl_file_list),
                                        GTK_CLIST (dl_file_list)->focus_row);
    file_data_set_use_auto_dl (file_data,
                                   gtk_toggle_button_get_active (
                                       GTK_TOGGLE_BUTTON (
                                           info.use_auto_dl)));
    download_info_update (file_data);
    dialog_prop (FALSE, file_data);
}

static GtkWidget *
create_frame_options (void)
{
    GtkWidget *frame, *table, *label, *sw;

    frame = gtk_frame_new (_ ("Options"));
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
    table = gtk_table_new (3, 2, FALSE);
    gtk_table_set_col_spacing (GTK_TABLE (table), 0, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (table),
                                    GNOME_PAD_SMALL);
    sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw),
                                           table);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				    GTK_POLICY_ALWAYS,
				    GTK_POLICY_NEVER);
    gtk_container_add (GTK_CONTAINER (frame), sw);
    
    label = gtk_label_new (_ ("Local dir:"));
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                      GTK_SHRINK | GTK_FILL, 0, 0, 0);
    info.local_dir = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (info.local_dir), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), info.local_dir, 1, 2, 0, 1);

    info.use_proxy = gtk_check_button_new_with_label (_ ("Use proxy"));
    gtk_table_attach_defaults (GTK_TABLE (table), info.use_proxy,
                               0, 2, 1, 2);
    gtk_signal_connect (GTK_OBJECT (info.use_proxy), "toggled", 
                        GTK_SIGNAL_FUNC (toggle_proxy_cb), NULL);

    info.use_auto_dl =
        gtk_check_button_new_with_label (_ ("Use auto download"));
    gtk_table_attach_defaults (GTK_TABLE (table), info.use_auto_dl,
                               0, 2, 2, 3);
    gtk_signal_connect (GTK_OBJECT (info.use_auto_dl), "toggled", 
                        GTK_SIGNAL_FUNC (toggle_auto_dl_cb), NULL);

    return frame;
}


/* Function to create the download info page */
GtkWidget *
download_info_create (void)
{
    GtkWidget *table;

    /* Create the for the frames with information */
    table = gtk_table_new (4, 2, FALSE);
    gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);

    /* Create the frame with location info */
    gtk_table_attach_defaults (GTK_TABLE (table), create_frame_location (), 
                               0, 2, 0, 1);
    
    /* Create frame with status */
    gtk_table_attach_defaults (GTK_TABLE (table), create_frame_status (),
                               0, 2, 1, 2);

    /* Create frame with total information */
    gtk_table_attach_defaults (GTK_TABLE (table), create_frame_total (),
                               0, 1, 2, 3);

    /* Create frame with session information */
    gtk_table_attach_defaults (GTK_TABLE (table), create_frame_sess (),
                               1, 2, 2, 3);

    /* Create frame with options */
    gtk_table_attach_defaults (GTK_TABLE (table), create_frame_options (),
                               0, 2, 3, 4);

    /* We have built everything now show it */
    gtk_widget_show_all (table);

    return table;
}


/* Functions to update the download info page */
void
download_info_update_state (FileData *file_data)
{
    gtk_label_set_text (GTK_LABEL (info.state),
                        file_data_state_name (file_data));
}

void
download_info_update_total_size (FileData *file_data)
{
    gchar buffer[20];

    if (file_data->state == DL_NOT_STARTED)
	strcpy (buffer, "");
    else if (file_data->total_size == 0)
	strcpy (buffer, _ ("Unspecified"));
    else
	sprintf (buffer, "%d kB", (file_data->total_size + 512) / 1024);
    gtk_label_set_text (GTK_LABEL (info.size), buffer);

    if (file_data->state != DL_NOT_STARTED) {
        if (file_data->cur_size <= file_data->total_size)
            gtk_progress_configure (GTK_PROGRESS (info.retr_progress),
                                    file_data->cur_size, 0, 
                                    file_data->total_size);
    }
    else {
        gtk_progress_configure (GTK_PROGRESS (info.retr_progress),
                                0, 0, file_data->total_size);
    }
}

void
download_info_update_statistics (FileData *file_data)
{
    gchar buffer[20];
    guint32 retr_size, remain_size;
    time_t estimated;

    /* Update retrieved information */
    if (file_data->state == DL_NOT_STARTED || file_data->state == DL_COMPLETED)
        strcpy (buffer, "");
    else
        sprintf (buffer, "%d kB", (file_data->cur_size + 512) / 1024);
    gtk_label_set_text (GTK_LABEL (info.retrieved), buffer);
    if (file_data->state == DL_COMPLETED)
        gtk_progress_set_percentage (GTK_PROGRESS (info.retr_progress), 1.0);
    else
        gtk_progress_set_value (GTK_PROGRESS (info.retr_progress),
                                file_data->cur_size);

    /* Update total and session times */
    if (file_data->state == DL_NOT_STARTED)
        strcpy (buffer, "");
    else
        convert_time2str (buffer, file_data->total_time);
    gtk_label_set_text (GTK_LABEL (info.total_time), buffer);
    if (file_data->session_elapsed != 0)
        convert_time2str (buffer, file_data->session_elapsed);
    gtk_label_set_text (GTK_LABEL (info.sess_time), buffer);

    /* The rest of the statistics is calculated so we show them only
     * if requested */
    if (!gtm_pref.show_page_stat) {
        strcpy (buffer, "");
        gtk_label_set_text (GTK_LABEL (info.estimated), buffer);
        gtk_label_set_text (GTK_LABEL (info.remain), buffer);
        gtk_label_set_text (GTK_LABEL (info.total_speed), buffer);
        gtk_label_set_text (GTK_LABEL (info.sess_speed), buffer);
        return;
    }

    retr_size = file_data->cur_size - file_data->session_start_size;
    remain_size = file_data->total_size - file_data->cur_size;

    /* Update estimated and remain times */
    if (file_data->state != DL_NOT_STARTED 
        && file_data->state != DL_NOT_RUNNING 
        && file_data->state != DL_COMPLETED) {
        if (retr_size != 0)
            estimated =
                (((gfloat) (file_data->total_size
                            - file_data->session_start_size)
                  * file_data->session_elapsed) / retr_size)
                + ((gfloat) (file_data->total_time
                             - file_data->session_elapsed));
        else
            estimated = 0;
    } else {
        if (file_data->cur_size != 0 && file_data->state != DL_COMPLETED)
            estimated =
                ((gfloat) file_data->total_size * file_data->total_time)
                / file_data->cur_size;
        else
            estimated = 0;
    }
    if (estimated == 0)
        strcpy (buffer, "");
    else
        convert_time2str (buffer, estimated);
    gtk_label_set_text (GTK_LABEL (info.estimated), buffer);
    if (estimated == 0)
        strcpy (buffer, "");
    else
        convert_time2str (buffer, estimated - file_data->total_time);
    gtk_label_set_text (GTK_LABEL (info.remain), buffer);

    /* Update total and session speeds */
    if (file_data->total_time != 0) {
        if (file_data->state == DL_COMPLETED) {
            if (file_data->total_size != 0)
                sprintf (buffer, "%.2f kB/s", 
                         ((gfloat) file_data->total_size
                                   / file_data->total_time) / 1024);
            else
                strcpy (buffer, "");
        } else if (file_data->cur_size != 0)
            sprintf (buffer, "%.2f kB/s", 
                     ((gfloat) file_data->cur_size / file_data->total_time)
                              / 1024);
        else
            strcpy (buffer, "");
    } else
        strcpy (buffer, "");
    gtk_label_set_text (GTK_LABEL (info.total_speed), buffer);
    if (file_data->state != DL_NOT_STARTED 
        && file_data->state != DL_NOT_RUNNING 
        && file_data->state != DL_COMPLETED) {
        if (file_data->session_elapsed != 0) {
            if (retr_size == 0)
                strcpy (buffer, _ ("staled"));
            else
                sprintf (buffer, "%.2f kB/s",
                         ((gfloat) retr_size / file_data->session_elapsed) 
                          / 1024);
        }
    } else {
        strcpy (buffer, "");
        gtk_label_set_text (GTK_LABEL (info.sess_time), buffer);
    }
    gtk_label_set_text (GTK_LABEL (info.sess_speed), buffer);
}

void
download_info_update (FileData *file_data)
{
    /* Check if we have data to put on download info page */
    if (file_data == NULL) {
        /* Clear the download info page */
	gtk_label_set_text (GTK_LABEL (info.url), "");
	gtk_label_set_text (GTK_LABEL (info.file), "");
	gtk_label_set_text (GTK_LABEL (info.size), "");
	gtk_label_set_text (GTK_LABEL (info.state), "");
        gtk_progress_set_value (GTK_PROGRESS (info.retr_progress), 0);
	gtk_label_set_text (GTK_LABEL (info.retrieved), "");
	gtk_label_set_text (GTK_LABEL (info.estimated), "");
	gtk_label_set_text (GTK_LABEL (info.remain), "");
	gtk_label_set_text (GTK_LABEL (info.total_time), "");
	gtk_label_set_text (GTK_LABEL (info.total_speed), "");
	gtk_label_set_text (GTK_LABEL (info.sess_time), "");
	gtk_label_set_text (GTK_LABEL (info.sess_speed), "");
	gtk_label_set_text (GTK_LABEL (info.local_dir), "");
        gtk_widget_set_sensitive (GTK_WIDGET (info.use_auto_dl), FALSE);
        gtk_widget_set_sensitive (GTK_WIDGET (info.use_proxy), FALSE);
    } else {
        /* Update state, total size and statistics */
        download_info_update_state (file_data);
        download_info_update_total_size (file_data);
        download_info_update_statistics (file_data);
        
	/* Update location */
	gtk_label_set_text (GTK_LABEL (info.url), file_data->url);
	gtk_label_set_text (GTK_LABEL (info.file), file_data->filename);

	/* Update options */
	gtk_label_set_text (GTK_LABEL (info.local_dir), file_data->dir);
        gtk_widget_set_sensitive (GTK_WIDGET (info.use_auto_dl), TRUE);
        gtk_widget_set_sensitive (GTK_WIDGET (info.use_proxy), TRUE);
        gtk_toggle_button_set_active 
            (GTK_TOGGLE_BUTTON (info.use_auto_dl),
             file_data->use_auto_dl);
        gtk_toggle_button_set_active 
            (GTK_TOGGLE_BUTTON (info.use_proxy),
             file_data->use_proxy);

        /* Update the state of the toggle auto download button on the toolbar
           and menu */
        if (file_data->file_list)
            menu_toolbar_set_auto_dl (GTK_CLIST (file_data->file_list));
    }
}
