/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "gtm_applet.h"
#include "dialogs.h"

/* Structure that keeps the widgets with the information of the new download
 * file */
typedef struct 
{
    GtkWidget *dlg;              /* GNOME dialog */
    GtkWidget *url;              /* Entry */
    GtkWidget *file_entry;       /* File Entry */
    GtkWidget *download_dir;     /* Entry */
    GtkWidget *use_proxy;        /* Check button */
    GtkWidget *use_auto_dl;      /* Check button */
    GtkWidget *override_dd;      /* Check button */
} NewDlData;


/* Call back function used when the override download dir is changed */
static void
override_dd_cb (GtkWidget *widget, NewDlData *new_dl_data)
{
    /* Change the sensitivity of download dir text box */
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (new_dl_data->
                                                         override_dd)))
        gtk_widget_set_sensitive (new_dl_data->file_entry, TRUE);
    else
        gtk_widget_set_sensitive (new_dl_data->file_entry, FALSE);
}


/* Call back function used when the OK and CANCEL buttons get pressed */
static void
ok_new_file (GtkWidget *widget, NewDlData *new_dl_data)
{
    gchar *url, *dir, *filename, empty[] = "";
    GTMAppletError result;
    gboolean use_proxy, use_auto_dl;
    GtkWidget *error;

    /* Get the URL data and make sure it's right */
    url = g_strdup (gtk_entry_get_text (GTK_ENTRY (new_dl_data->url)));
    if (strncasecmp (url, "http://", 7) && strncasecmp (url, "ftp://", 6)) {
	error = gnome_message_box_new (
	    _ ("Invalid URL"), GNOME_MESSAGE_BOX_ERROR, 
	    GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_set_parent (GNOME_DIALOG (error), 
                                 GTK_WINDOW (new_dl_data->dlg));
	gnome_dialog_run_and_close (GNOME_DIALOG (error));
        g_free (url);
	return;
    }

    /* Get download filename data */
    filename = &url[strlen (url)];
    while (*filename != '/' && filename != url)
	filename--;
    filename++;

    /* TODO: If filename == NULL instead of reminding the user that it is not
     * possible yet to do multiple downloads, when the group stuff is 
     * implemented this should enter a new dialog like dialog-multiple and 
     * ask there for download depth and get the available files to download. 
     * Then the user can remove what he doesn't want. */
    if (strlen (filename) == 0) {
	error = gnome_message_box_new (
	    _ ("Multiple download not implemented yet"), 
            GNOME_MESSAGE_BOX_ERROR, 
	    GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_set_parent (GNOME_DIALOG (error), 
                                 GTK_WINDOW (new_dl_data->dlg));
	gnome_dialog_run_and_close (GNOME_DIALOG (error));
        g_free (url);
	return;
    }

    /* Get download directory data */
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (new_dl_data->
                                                         override_dd)))
        dir = gtk_entry_get_text (GTK_ENTRY (new_dl_data->download_dir));
    else
        dir = empty;

    /* Get the use proxy option data */
    use_proxy =
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
            new_dl_data->use_proxy));

    /* Get the use auto download option data */
    use_auto_dl = 
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (
	    new_dl_data->use_auto_dl));

    /* Add the new download file to the file list (if not a duplicate) */
    result = gtm_add_url (url, dir, use_proxy, use_auto_dl);
    if (result != GTM_ADDURL_NO_ERROR) {
        gchar *error_msg;

        if (result == GTM_ADDURL_GTM_ERROR) {
            /* GTM couldn't be started */
            error_msg = g_strconcat (_ ("Couldn't start GTM"), NULL);
        } else if (result == GTM_ADDURL_CORBA_ERROR) {
            /* CORBA error */
            error_msg = g_strconcat (_ ("CORBA error"), NULL);
        } else if (result == GTM_ADDURL_REPEAT_ERROR) {
            /* URL on the file list */
            error_msg = g_strconcat (_ ("URL:\n"), url, "\n",
                                     _ ("already in the list"), NULL);
        } else {
            /* Unknown code */
            error_msg = g_strconcat ("", NULL);
        }
        error = gnome_message_box_new (error_msg, GNOME_MESSAGE_BOX_ERROR,
                                       GNOME_STOCK_BUTTON_OK, NULL);
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
        g_free (error_msg);
    }
    
    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (url);
    g_free (new_dl_data);
}

static void
dialog_new_drag_data_received (GtkWidget *widget, GdkDragContext *context,
                               gint x, gint y, 
                               GtkSelectionData *selection_data,
                               guint info, guint time,
                               NewDlData *new_dl_data)
{
    g_return_if_fail (widget != NULL);
    g_return_if_fail (selection_data->data != NULL);
    gtk_entry_set_text (GTK_ENTRY (new_dl_data->url), selection_data->data);
}

static void
cancel_new_file (GtkWidget *widget, NewDlData *new_dl_data)
{
    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (new_dl_data);
}


/* Dialog used to add a new file to download */
void
dialog_new (void)
{
    GtkWidget *label, *file_entry, *hbox1;
    NewDlData *new_dl_data;
    gchar *default_protocol;

    static GtkTargetEntry target_table[] = {
        { "text/uri-list", 0, 0 },
        { "text/plain", 0, 0 },
        { "x-url/http", 0, 0 },
        { "x-url/ftp", 0, 0 },
        { "_NETSCAPE_URL", 0, 0 }
    };

    new_dl_data = g_new (NewDlData, 1);
    switch (gtm_applet_prop.presume_protocol) {
    case 0:
        default_protocol = g_strconcat("", NULL);
        break;
    case 1:
        default_protocol = g_strconcat("http://", NULL);
        break;
    case 2:
        default_protocol = g_strconcat("ftp://", NULL);
	break;
    default:
	default_protocol = g_strconcat("", NULL);
    }

    /* Create dialog box */
    new_dl_data->dlg = 
	gnome_dialog_new (_ ("New file to download"),
			  GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL,
			  NULL);

    /* Set it's contents */
    label = gtk_label_new (_ ("Please enter URL of the file to download:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
			label, FALSE, FALSE, 0);
    new_dl_data->url = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(new_dl_data->url), default_protocol);
    gtk_widget_set_usize (new_dl_data->url, 300, -2);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
			new_dl_data->url, FALSE, FALSE, 0);
    gtk_widget_grab_focus (new_dl_data->url);

    new_dl_data->override_dd = 
        gtk_check_button_new_with_label (_ ("Override Download dir"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
                        new_dl_data->override_dd, FALSE, FALSE, 0);
    file_entry = gnome_file_entry_new ("NewDownloadDir", 
                                       _("New Download Dir"));
    new_dl_data->download_dir = 
        gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (file_entry));
    gnome_file_entry_set_directory (GNOME_FILE_ENTRY (file_entry), 1);
    gtk_entry_set_text (GTK_ENTRY (new_dl_data->download_dir), 
                        gnome_util_user_home ());
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox),
                        file_entry, FALSE, FALSE, 0);
    new_dl_data->file_entry = file_entry;
    gtk_widget_set_sensitive (file_entry, FALSE);

    hbox1 = gtk_hbox_new(FALSE, FALSE);
    new_dl_data->use_proxy = 
	gtk_check_button_new_with_label (_ ("Use proxy"));
    gtk_box_pack_start (GTK_BOX (hbox1), new_dl_data->use_proxy,
                        FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        new_dl_data->use_proxy), TRUE);
    new_dl_data->use_auto_dl =
        gtk_check_button_new_with_label (_ ("Use auto download"));
    gtk_box_pack_start (GTK_BOX (hbox1), new_dl_data->use_auto_dl,
                        FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox),
                        hbox1, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        new_dl_data->use_auto_dl), TRUE);

    /* Configure the dialog box */
    gnome_dialog_set_close (GNOME_DIALOG (new_dl_data->dlg), FALSE);
    gnome_dialog_set_default (GNOME_DIALOG (new_dl_data->dlg), GNOME_OK);

    /* Set drag 'n drop stuff */
    gtk_drag_dest_set (new_dl_data->url, GTK_DEST_DEFAULT_MOTION 
		       | GTK_DEST_DEFAULT_DROP,
                       target_table, 
		       sizeof (target_table) / sizeof (target_table[0]),
                       GDK_ACTION_COPY | GDK_ACTION_MOVE);

    /* Set the signal handlers */
    gtk_signal_connect (GTK_OBJECT (new_dl_data->override_dd), "toggled", 
                        GTK_SIGNAL_FUNC (override_dd_cb), 
                        (gpointer) new_dl_data);
    gnome_dialog_button_connect (GNOME_DIALOG (new_dl_data->dlg), GNOME_OK,
				 GTK_SIGNAL_FUNC (ok_new_file),
				 (gpointer) new_dl_data);
    gnome_dialog_button_connect (GNOME_DIALOG (new_dl_data->dlg), GNOME_CANCEL,
				 GTK_SIGNAL_FUNC (cancel_new_file),
				 (gpointer) new_dl_data);
    gtk_signal_connect (GTK_OBJECT (new_dl_data->url), "activate",
			GTK_SIGNAL_FUNC (ok_new_file), 
			(gpointer) new_dl_data);
    gtk_signal_connect (GTK_OBJECT (new_dl_data->url), "drag_data_received",
                        GTK_SIGNAL_FUNC (dialog_new_drag_data_received),
                        (gpointer) new_dl_data);

    /* Show the dialog box and every thing it contains */
    gtk_widget_show_all (new_dl_data->dlg);
}
