/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "dialogs.h"
#include "download-info.h"
#include "main-window.h"
#include "robot.xpm"

/* Download info page */
static GtkWidget *download_info;


/* Callbacks functions */
static void
new_cmd (GtkWidget *widget, gpointer data)
{
    dialog_new ();
}

void
remove_cmd (GtkWidget *widget, gpointer data)
{
    if (GTK_CLIST (dl_file_list)->selection != NULL)
        dialog_remove ();
}

void
stats_cmd (GtkWidget *widget, gpointer data)
{
    dialog_stats_show ();
}

void
start_cmd (GtkWidget *widget, gpointer data)
{
    file_list_func_on_selection (GTK_CLIST (dl_file_list),
                                 file_data_start_download);
}

void
stop_cmd (GtkWidget *widget, gpointer data)
{
    file_list_func_on_selection (GTK_CLIST (dl_file_list),
                                 file_data_stop_download);
}

void
restart_cmd (GtkWidget *widget, gpointer data)
{
    dialog_restart ();
}

static void
num_down_cmd (GtkWidget *widget, gpointer data)
{
    dialog_numdown ();
}

static void
quit_cmd (GtkWidget *widget, gpointer data)
{
    GtkWidget *warning;
    FileData *file_data;
    gint row;

    /* Before quitting the program check that no files are being downloaded */
    warning = NULL;
    row = 0;
    while ((file_data = gtk_clist_get_row_data (GTK_CLIST (dl_file_list),
                                                row++))
	   != NULL) {
	if (file_data_run (file_data)) {
	    /* File is being downloaded so ask the user if he really wants to
	     * quit */
	    warning = gnome_message_box_new (_ (
		"Some files being downloaded now.\n"
		"Do you really want to quit?"), 
					     GNOME_MESSAGE_BOX_WARNING,
					     GNOME_STOCK_BUTTON_OK, 
					     GNOME_STOCK_BUTTON_CANCEL,
					     NULL);
	    gnome_dialog_set_parent (GNOME_DIALOG (warning), 
				     GTK_WINDOW (gtm_app));
	    break;
	}
    }
    if (warning != NULL) {
	if (gnome_dialog_run_and_close (GNOME_DIALOG (warning)) == GNOME_OK) {
	    /* The user wants to quit so stop the running processes and quit
	     * program */
	    row = 0;
	    while ((file_data = 
                    gtk_clist_get_row_data (GTK_CLIST (dl_file_list), row))
		   != NULL) {
                file_data_stop_download (file_data);
		row++;
	    }
	    gtk_main_quit ();
	}
    } else
	gtk_main_quit ();
}

static void 
about_cmd (void)
{
    dialog_about ();
}

static void
preferences_cmd (void)
{
    dialog_prefs ();
}

void
properties_cmd (GtkWidget *widget, gpointer data)
{
    FileData *file_data;

    if (g_list_find (GTK_CLIST (dl_file_list)->selection,
                     (gpointer) GTK_CLIST (dl_file_list)->focus_row) != NULL)
        file_data = (FileData *)
            gtk_clist_get_row_data (GTK_CLIST (dl_file_list),
                                    GTK_CLIST (dl_file_list)->focus_row);
    else
        file_data = NULL;
    dialog_prop (TRUE, file_data);
}

static void
select_all_cmd (void)
{
    file_list_select_all (GTK_CLIST (dl_file_list));
}

static void
unselect_all_cmd (void)
{
    file_list_unselect_all (GTK_CLIST (dl_file_list));
}

static void
toggle_auto_dl_cmd (GtkWidget *widget, gpointer data)
{
    FileData *file_data;
    GtkCList *file_list;
    GList *p;
    
    file_list = GTK_CLIST (dl_file_list);
    gtk_clist_freeze (file_list);
    p = file_list->selection;
    while (p != NULL) {
        file_data =
            (FileData *) gtk_clist_get_row_data (file_list, 
                                                 (gint) p->data);
        if (!GTK_IS_TOGGLE_BUTTON (widget))
            file_data_set_use_auto_dl (
                file_data,
                GTK_CHECK_MENU_ITEM (widget)->active);
        else
            file_data_set_use_auto_dl (
                file_data,
                GTK_TOGGLE_BUTTON (widget)->active);
        p = p->next;
    }
    gtk_clist_thaw (file_list);

    if (g_list_find (file_list->selection,
                     (gpointer) file_list->focus_row) != NULL)
        file_data = (FileData *)
            gtk_clist_get_row_data (file_list,
                                    file_list->focus_row);
    else
        file_data = NULL;
    download_info_update (file_data);
    dialog_prop (FALSE, file_data);
}


/* The menus */
static GnomeUIInfo main_menu_file[] = {
    { GNOME_APP_UI_ITEM, N_ ("_New..."), N_ ("New file to download"),
      new_cmd, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW,
      'N', GDK_CONTROL_MASK, NULL },
#define MAIN_MENU_FILE_REMOVE 2
    { GNOME_APP_UI_ITEM, N_ ("_Remove from list..."),
      N_ ("Remove selected files from the list"),
      remove_cmd, NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
      'D', GDK_CONTROL_MASK, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_ ("_List Statistics..."),
      N_ ("Show file list summary"),
      stats_cmd, NULL, NULL, GNOME_APP_PIXMAP_STOCK, 
      GNOME_STOCK_MENU_BOOK_YELLOW, 'L', GDK_CONTROL_MASK, NULL },
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_EXIT_ITEM (quit_cmd, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo main_menu_edit[] = {
    { GNOME_APP_UI_ITEM, N_ ("Select _All"),
      N_ ("Select all files from the list"), select_all_cmd, NULL, NULL,
      0, 0, 'A', GDK_CONTROL_MASK, NULL },
    { GNOME_APP_UI_ITEM, N_ ("Unselect All"),
      N_ ("Unselect all files from the list"), unselect_all_cmd, NULL, NULL,
      0, 0, 0, (GdkModifierType) 0, NULL },
#define MAIN_MENU_EDIT_TOGGLE_AUTO_DL 2
    GNOMEUIINFO_TOGGLEITEM (N_ ("Toggles auto download"),
                            N_ ("Toggles auto download of selected files"),
                            toggle_auto_dl_cmd, robot_xpm),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_PROPERTIES_ITEM (properties_cmd, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo main_menu_download[] = {
#define MAIN_MENU_DOWNLOAD_START 0
    GNOMEUIINFO_ITEM_STOCK (N_ ("Start"),
			    N_ ("Start/Resume the download of the selected "
                                "files"),
                            start_cmd, GNOME_STOCK_MENU_EXEC),
#define MAIN_MENU_DOWNLOAD_STOP 1
    GNOMEUIINFO_ITEM_STOCK (N_ ("Stop"),
                            N_ ("Stop the download of the selected files"),
			    stop_cmd, GNOME_STOCK_MENU_STOP),
#define MAIN_MENU_DOWNLOAD_RESTART 2
    GNOMEUIINFO_ITEM_STOCK (N_ ("Restart"),
			    N_ ("Restart the download of selected files from "
                                "the beginning"), restart_cmd, 
			    GNOME_STOCK_MENU_REFRESH),
    GNOMEUIINFO_END
};

static GnomeUIInfo main_menu_settings[] = {
    GNOMEUIINFO_MENU_PREFERENCES_ITEM (preferences_cmd, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo main_menu_tools[] = {
    GNOMEUIINFO_ITEM_STOCK (N_ ("_Numeric Download"),
                            N_ ("Download a group of URLs differing by "
                                "number"),
                            num_down_cmd, GNOME_STOCK_MENU_BOOK_OPEN ),
    GNOMEUIINFO_END
};
	
static GnomeUIInfo main_menu_help[] = {
    GNOMEUIINFO_HELP ("gtm"),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_ABOUT_ITEM (about_cmd, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo main_menu[] = {
    GNOMEUIINFO_MENU_FILE_TREE (main_menu_file),
    GNOMEUIINFO_MENU_EDIT_TREE (main_menu_edit),
    GNOMEUIINFO_SUBTREE (N_ ("_Download"), main_menu_download),
    GNOMEUIINFO_SUBTREE (N_ ("_Tools"), main_menu_tools),
    GNOMEUIINFO_MENU_SETTINGS_TREE (main_menu_settings),
    GNOMEUIINFO_MENU_HELP_TREE (main_menu_help),
    GNOMEUIINFO_END
};


/* The toolbar */
static GnomeUIInfo main_toolbar[] = {
    GNOMEUIINFO_ITEM_STOCK (N_ ("New"), N_ ("Add new file to download"),
			    new_cmd, GNOME_STOCK_PIXMAP_NEW),
    GNOMEUIINFO_SEPARATOR,
#define MAIN_TOOLBAR_REMOVE 2
    GNOMEUIINFO_ITEM_STOCK (N_ ("Remove"), 
			    N_ ("Remove selected files from the list"),
			    remove_cmd, GNOME_STOCK_PIXMAP_CLOSE),
    GNOMEUIINFO_SEPARATOR,
#define MAIN_TOOLBAR_START 4
    GNOMEUIINFO_ITEM_STOCK (N_ ("Start"),
			    N_ ("Start/Resume the download of selected "
                                "files"), 
                            start_cmd, GNOME_STOCK_PIXMAP_EXEC),
#define MAIN_TOOLBAR_STOP 5
    GNOMEUIINFO_ITEM_STOCK (N_ ("Stop"),
			    N_ ("Stop the download of selected files"),
			    stop_cmd, GNOME_STOCK_PIXMAP_STOP),
#define MAIN_TOOLBAR_RESTART 6
    GNOMEUIINFO_ITEM_STOCK (N_ ("Restart"),
			    N_ ("Restart the download of selected files from "
                                "the beginning"), restart_cmd,
			    GNOME_STOCK_PIXMAP_REFRESH),
    GNOMEUIINFO_SEPARATOR,
#define MAIN_TOOLBAR_TOGGLE_AUTO_DL 8
    GNOMEUIINFO_TOGGLEITEM (N_ ("Toggle Auto Dl"),
                            N_ ("Toggle auto download of selected files"),
                            toggle_auto_dl_cmd, robot_xpm),
    GNOMEUIINFO_SEPARATOR,
#define MAIN_TOOLBAR_PROPERTIES 10
    GNOMEUIINFO_ITEM_STOCK (N_ ("Properties"),
                            N_ ("Open properties for selected file"),
                            properties_cmd, GNOME_STOCK_PIXMAP_PROPERTIES),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_ITEM_STOCK (N_ ("Quit"), N_ ("Quit gtm"), quit_cmd,
			    GNOME_STOCK_PIXMAP_QUIT),
    GNOMEUIINFO_END
};


/* Function to set the state of the menu and toolbar according to the 
 * selection on the file list */
void
menu_toolbar_set_state (GtkCList *file_list)
{
    FileData *file_data;

    if (file_list->selection == NULL) {
        /* No file selected */
	gtk_widget_set_sensitive (
	    main_menu_file[MAIN_MENU_FILE_REMOVE].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_REMOVE].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_menu_edit[MAIN_MENU_EDIT_TOGGLE_AUTO_DL].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_TOGGLE_AUTO_DL].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_menu_download[MAIN_MENU_DOWNLOAD_START].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_START].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_menu_download[MAIN_MENU_DOWNLOAD_STOP].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_STOP].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_menu_download[MAIN_MENU_DOWNLOAD_RESTART].widget, FALSE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_RESTART].widget, FALSE);
        gtk_widget_set_sensitive (
            main_toolbar[MAIN_TOOLBAR_PROPERTIES].widget, FALSE);
    } else if (file_list->selection->next == NULL) {
        /* One file selected */
        file_data = gtk_clist_get_row_data (file_list, 
                                            (gint) file_list->selection->data);
	gtk_widget_set_sensitive (
	    main_menu_file[MAIN_MENU_FILE_REMOVE].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_REMOVE].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_menu_edit[MAIN_MENU_EDIT_TOGGLE_AUTO_DL].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_TOGGLE_AUTO_DL].widget, TRUE);
        gtk_widget_set_sensitive (
            main_toolbar[MAIN_TOOLBAR_PROPERTIES].widget, TRUE );
        if (file_data->state == DL_NOT_STARTED 
            || file_data->state == DL_NOT_RUNNING) {
            gtk_widget_set_sensitive (
                main_menu_download[MAIN_MENU_DOWNLOAD_START].widget, TRUE);
            gtk_widget_set_sensitive (
                main_toolbar[MAIN_TOOLBAR_START].widget, TRUE);
        } else {
            gtk_widget_set_sensitive (
                main_menu_download[MAIN_MENU_DOWNLOAD_START].widget, FALSE);
            gtk_widget_set_sensitive (
                main_toolbar[MAIN_TOOLBAR_START].widget, FALSE);
        }
        if (file_data->state != DL_NOT_STARTED
            && file_data->state != DL_NOT_RUNNING
            && file_data->state != DL_COMPLETED) {
            gtk_widget_set_sensitive (
                main_menu_download[MAIN_MENU_DOWNLOAD_STOP].widget, TRUE);
            gtk_widget_set_sensitive (
                main_toolbar[MAIN_TOOLBAR_STOP].widget, TRUE);
        } else {
            gtk_widget_set_sensitive (
                main_menu_download[MAIN_MENU_DOWNLOAD_STOP].widget, FALSE);
            gtk_widget_set_sensitive (
                main_toolbar[MAIN_TOOLBAR_STOP].widget, FALSE);
        }
        if (file_data->state == DL_NOT_RUNNING
            || file_data->state == DL_COMPLETED) {
            gtk_widget_set_sensitive (
                main_menu_download[MAIN_MENU_DOWNLOAD_RESTART].widget, TRUE);
            gtk_widget_set_sensitive (
                main_toolbar[MAIN_TOOLBAR_RESTART].widget, TRUE);
        } else {
            gtk_widget_set_sensitive (
                main_menu_download[MAIN_MENU_DOWNLOAD_RESTART].widget, FALSE);
            gtk_widget_set_sensitive (
                main_toolbar[MAIN_TOOLBAR_RESTART].widget, FALSE);
        }
    } else {
        /* More than one file selected */
	gtk_widget_set_sensitive (
	    main_menu_file[MAIN_MENU_FILE_REMOVE].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_REMOVE].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_menu_edit[MAIN_MENU_EDIT_TOGGLE_AUTO_DL].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_TOGGLE_AUTO_DL].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_menu_download[MAIN_MENU_DOWNLOAD_START].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_START].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_menu_download[MAIN_MENU_DOWNLOAD_STOP].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_STOP].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_menu_download[MAIN_MENU_DOWNLOAD_RESTART].widget, TRUE);
	gtk_widget_set_sensitive (
	    main_toolbar[MAIN_TOOLBAR_RESTART].widget, TRUE);
        gtk_widget_set_sensitive (
            main_toolbar[MAIN_TOOLBAR_PROPERTIES].widget, FALSE);
    }
}


/* Function to set the auto download toggle button according to the 
 * selection on the file list */
void
menu_toolbar_set_auto_dl (GtkCList *file_list)
{
    FileData *file_data;
    gboolean all_use_auto_dl;
    GList *p;

    /* Disable the callbacks to toggle_auto_dl_cmd so we don't
     * homogenise the selection */
    gtk_signal_handler_block_by_func (
        GTK_OBJECT (main_menu_edit[MAIN_MENU_EDIT_TOGGLE_AUTO_DL].widget),
        GTK_SIGNAL_FUNC (toggle_auto_dl_cmd),
        file_list);
    gtk_signal_handler_block_by_func (
        GTK_OBJECT (main_toolbar[MAIN_TOOLBAR_TOGGLE_AUTO_DL].widget),
        GTK_SIGNAL_FUNC (toggle_auto_dl_cmd),
        file_list);

    if (file_list->selection == NULL) {
        /* No file selected */
    } else if (file_list->selection->next == NULL) {
        /* One file selected */
        file_data = gtk_clist_get_row_data (file_list, 
                                            (gint) file_list->selection->data);
        gtk_check_menu_item_set_active (
            GTK_CHECK_MENU_ITEM (
                main_menu_edit[MAIN_MENU_EDIT_TOGGLE_AUTO_DL].widget),
            file_data->use_auto_dl);
        gtk_toggle_button_set_active (
            GTK_TOGGLE_BUTTON (
                main_toolbar[MAIN_TOOLBAR_TOGGLE_AUTO_DL].widget),
            file_data->use_auto_dl);
    } else {
        /* More than one file selected */
        p = file_list->selection;
        all_use_auto_dl = TRUE;
        while (p != NULL) {
            file_data = (FileData *) gtk_clist_get_row_data (file_list, 
                                                             (gint) p->data);
            if (file_data->use_auto_dl == FALSE) {
                all_use_auto_dl = FALSE;
                break;
            }
            p = p->next;
        }
        gtk_check_menu_item_set_active (
            GTK_CHECK_MENU_ITEM (
                main_menu_edit[MAIN_MENU_EDIT_TOGGLE_AUTO_DL].widget),
            all_use_auto_dl);
        gtk_toggle_button_set_active (
            GTK_TOGGLE_BUTTON (
                main_toolbar[MAIN_TOOLBAR_TOGGLE_AUTO_DL].widget),
            all_use_auto_dl);
    }

    /* Reenable the disabled callbacks */
    gtk_signal_handler_unblock_by_func (
        GTK_OBJECT (main_menu_edit[MAIN_MENU_EDIT_TOGGLE_AUTO_DL].widget),
        GTK_SIGNAL_FUNC (toggle_auto_dl_cmd),
        file_list);
    gtk_signal_handler_unblock_by_func (
        GTK_OBJECT (main_toolbar[MAIN_TOOLBAR_TOGGLE_AUTO_DL].widget),
        GTK_SIGNAL_FUNC (toggle_auto_dl_cmd),
        file_list);
}


/* Function to change the toolbar style */
void
toolbar_update_style (void)
{
    GnomeDockItem *item;
    GtkWidget *toolbar;
    
    item = gnome_app_get_dock_item_by_name (GNOME_APP (gtm_app),
					    GNOME_APP_TOOLBAR_NAME);
    toolbar = gnome_dock_item_get_child (item);
    gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), gtm_pref.toolbar_style);
    gtk_widget_queue_resize (gtm_app);
}


/* Function to update the download info page visibility */
void
main_window_update_dl_info_page_visibility (void)
{
    if (gtm_pref.show_dl_info_page)
        gtk_widget_show (download_info);
    else
        gtk_widget_hide (download_info);
}


/* Init the application */
GtkWidget *
main_window_create (gchar *geometry)
{
    GtkWidget *file_list, *file_list_sw, *hbox, *appbar;
    gint xpos, ypos, width, height, cur_w, cur_h;

    /* Create the application and handle some important signals */
    gtm_app = gnome_app_new ("gtm", "GNOME Transfer Manager");
    gtk_signal_connect (GTK_OBJECT (gtm_app), "delete_event",
			GTK_SIGNAL_FUNC (quit_cmd),
			NULL);

    /* Create the list with files to download */
    file_list_sw = file_list_create ();
    file_list = GTK_BIN (file_list_sw)->child;
    gtk_widget_show (file_list_sw);

    /* Create the download information */
    download_info = download_info_create ();
    main_window_update_dl_info_page_visibility ();

    /* Create the menu, toolbar and status bar */
    gnome_app_create_menus_with_data (GNOME_APP (gtm_app), main_menu, 
                                      file_list);
    gnome_app_create_toolbar_with_data (GNOME_APP (gtm_app), main_toolbar,
                                        file_list);
    appbar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_NEVER);
    gnome_app_set_statusbar (GNOME_APP (gtm_app), appbar);
    gnome_app_install_appbar_menu_hints (GNOME_APPBAR (appbar), 
					 main_menu);
    gtk_widget_show (appbar);
    menu_toolbar_set_state (GTK_CLIST (file_list));
    menu_toolbar_set_auto_dl (GTK_CLIST (file_list));
    toolbar_update_style ();

    /* Create a hbox widget and add the file list and download information */
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), file_list_sw, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), download_info, FALSE, FALSE, 0);
    gtk_widget_show (hbox);
    gnome_app_set_contents (GNOME_APP (gtm_app), hbox);
    download_info_update (NULL);

    /* Realize the main window */
    gtk_widget_realize (gtm_app);

    /* Parse the old geometry and check if the window is clipped with this
     * old geometry (if the font is changed that might happen) */
    gnome_parse_geometry (geometry, &xpos, &ypos, &width, &height);
    gdk_window_get_size (gtm_app->window, &cur_w, &cur_h);
    if (cur_w > width)
        width = cur_w;
    if (cur_h > height)
        height = cur_h;

    /* Set the main window geometry and other useful parameters */
    gtk_window_set_policy (GTK_WINDOW (gtm_app), FALSE, TRUE, FALSE);
    gtk_widget_set_uposition (gtm_app, xpos, ypos);
    gtk_window_set_default_size (GTK_WINDOW (gtm_app), width, height);

    /* We are ready to show the application */
    gtk_widget_show (gtm_app);

    return file_list;
}
