/*
 *  Copyright (C) 1999, 2000 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <fnmatch.h>
#include "gtm.h"
#include "download-dirs.h"

/* Directory list columns */
#define DIR_LIST_TOTAL_COLUMNS 2
#define DIR_LIST_COL_PATTERN 0
#define DIR_LIST_COL_DIR 1

/* Structure that keeps the widgets with the new download dir info */
typedef struct
{
    GtkWidget* dlg;         /* GNOME dialog */
    GtkWidget* pattern;     /* Entry */
    GtkWidget* dir;         /* Entry */
    GtkWidget* dir_list;    /* GtkCBox */
    gboolean replace;
} NewDirData;

#define DIR_LIST_BUTTON_INSERT 0
#define DIR_LIST_BUTTON_EDIT   1
#define DIR_LIST_BUTTON_DELETE 2
#define DIR_LIST_TOTAL_BUTTONS 3
static GtkWidget* buttons[DIR_LIST_TOTAL_BUTTONS];

static void
ok_add_dir (GtkWidget *widget, NewDirData* new_dir_data) {
    gchar *pattern, *dir;
    gchar *data[DIR_LIST_TOTAL_COLUMNS];
    GList *p;
    
    /* Get regular expression data and directory data*/
    pattern = gtk_entry_get_text (GTK_ENTRY (new_dir_data->pattern));
    dir = gtk_entry_get_text (GTK_ENTRY (new_dir_data->dir));
    
    p = GTK_CLIST (new_dir_data->dir_list)->selection;

    if (new_dir_data->replace) {
        if (p != NULL) {
            gtk_clist_set_text (GTK_CLIST (new_dir_data->dir_list),
                                (gint) p->data,
                                DIR_LIST_COL_PATTERN,
                                pattern);
            
            gtk_clist_set_text (GTK_CLIST (new_dir_data->dir_list),
                                (gint) p->data,
                                DIR_LIST_COL_DIR,
                                dir);
            gtk_signal_emit_by_name (GTK_OBJECT (new_dir_data->dir_list),
                                     "grab-focus");
        }
    } else {
        /* if exactly one list item is selected, insert at that position
           otherwise append to the list */
        data[DIR_LIST_COL_PATTERN] = pattern;
        data[DIR_LIST_COL_DIR] = dir;
        if (p == NULL)
            gtk_clist_append (GTK_CLIST (new_dir_data->dir_list), data);
        else
            gtk_clist_insert (GTK_CLIST (new_dir_data->dir_list), 
                              (gint) p->data, 
                              data); 
        gtk_signal_emit_by_name (GTK_OBJECT (new_dir_data->dir_list),
                                 "grab-focus");
    }
    
    gtk_widget_destroy (new_dir_data->dlg);
    g_free (new_dir_data);
}

static void
cancel_add_dir (GtkWidget *widget, NewDirData* new_dir_data) {
    gtk_widget_destroy (new_dir_data->dlg);
    g_free (new_dir_data);
}

static void 
download_dirs_add_dir (GtkWidget** clist, gboolean edit_current) {
    GtkWidget *label, *file_entry;
    NewDirData *new_dir_data;
    gchar *current_pattern, *current_dir;
    GList *p;

    new_dir_data = g_new (NewDirData, 1);

    new_dir_data->dir_list = *clist;
    new_dir_data->replace = edit_current;
    
    /* Create dialog box */
    new_dir_data->dlg = 
        gnome_dialog_new (_ ("Target directory"),
                          GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL,
                          NULL);
    gnome_dialog_set_parent (GNOME_DIALOG (new_dir_data->dlg),
                             GTK_WINDOW (gtm_app));
    label = gtk_label_new (_ ("Pattern:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dir_data->dlg)->vbox),
                        label, FALSE, FALSE, 0);
    new_dir_data->pattern = gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dir_data->dlg)->vbox),
                        new_dir_data->pattern, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Download matching URLs to directory:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dir_data->dlg)->vbox),
                        label, FALSE, FALSE, 0);
    /* file entry shares history with dialog-add */
    file_entry = gnome_file_entry_new ("NewDownloadDir",
                                       _ ("Add Download Dir"));
    new_dir_data->dir = 
        gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (file_entry));
    gnome_file_entry_set_directory (GNOME_FILE_ENTRY (file_entry), 1);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dir_data->dlg)->vbox),
                        file_entry, FALSE, FALSE, 0);
    
    /* if we are editing an existing entry, fill in the current data */
    if (edit_current) {
        p = GTK_CLIST(*clist)->selection;
        if (p != NULL) {
            gtk_clist_get_text (GTK_CLIST(new_dir_data->dir_list),
                                (gint) p->data,
                                DIR_LIST_COL_PATTERN,
                                &current_pattern);
            gtk_entry_set_text (GTK_ENTRY (new_dir_data->pattern),
                                current_pattern);
            gtk_clist_get_text (GTK_CLIST(new_dir_data->dir_list),
                                (gint) p->data,
                                DIR_LIST_COL_DIR,
                                &current_dir);
            gtk_entry_set_text (GTK_ENTRY(new_dir_data->dir),
                                current_dir);                                
        }
    }

    /* Configure the dialog box */
    gnome_dialog_set_close (GNOME_DIALOG (new_dir_data->dlg), FALSE);
    gnome_dialog_set_default (GNOME_DIALOG (new_dir_data->dlg), GNOME_OK);
    
    /* Set the signal handlers */
    gnome_dialog_button_connect (GNOME_DIALOG (new_dir_data->dlg), GNOME_OK,
                                 GTK_SIGNAL_FUNC (ok_add_dir),
                                 (gpointer) new_dir_data);
    gnome_dialog_button_connect (GNOME_DIALOG (new_dir_data->dlg), GNOME_CANCEL,
                                 GTK_SIGNAL_FUNC (cancel_add_dir),
                                 (gpointer) new_dir_data);

    /* Show the dialog box and everything it contains */
    gtk_widget_show_all (new_dir_data->dlg);

}

static void 
add_dir_cb (GtkWidget *widget, GtkWidget **data) {
    download_dirs_add_dir (data, FALSE);
}

static void 
edit_dir_cb (GtkWidget *widget, GtkWidget **data) {
    download_dirs_add_dir (data, TRUE);
}

static void 
delete_dir_cb (GtkWidget *widget, GtkWidget **data) {
    
    GList *p;
    gint tmp;

    gtk_clist_freeze (GTK_CLIST (*data));
    p = GTK_CLIST(*data)->selection;
    while (p != NULL) {
        tmp = (gint)p->data;
        p = p->next;
        gtk_clist_remove (GTK_CLIST (*data), tmp);
    }
    gtk_clist_thaw (GTK_CLIST (*data));
}

static void
dir_list_keypress_cb (GtkWidget *widget, GdkEventKey *event, GtkWidget **data)
{
    if (GTK_CLIST (*data)->selection != NULL) {
	if (event->keyval == GDK_Delete || event->keyval == GDK_KP_Delete)
	    delete_dir_cb(widget, data);
    }
}

static void
download_dirs_load_data (GtkCList* dir_list)
{
    gint i, total_dirs;
    gchar *data[DIR_LIST_TOTAL_COLUMNS];
    char *prefix;
    total_dirs = download_dirs_get_total_dirs();
    
    for (i = 0; i < total_dirs; i++) {
        prefix = g_strdup_printf ("/gtm/Dir%d/", i);
        gnome_config_push_prefix (prefix);
        data[DIR_LIST_COL_PATTERN] = gnome_config_get_string("pattern");
        data[DIR_LIST_COL_DIR] = gnome_config_get_string("dir");
        gtk_clist_append (dir_list, data);
        gnome_config_pop_prefix();
        g_free (prefix);
    }    
}

static void
dir_list_buttons_set_state(GtkCList* dir_list) {
    GList *p;
    /* set sensitivity of buttons depending on what is possible
       with the current selection */
    p = dir_list->selection;
    switch (g_list_length(p)) {
        case 0:
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_INSERT], TRUE);
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_EDIT], FALSE);
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_DELETE], FALSE);
            break;
        case 1:
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_INSERT], TRUE);
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_EDIT], TRUE);
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_DELETE], TRUE);
            break;
        default:
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_INSERT], FALSE);
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_EDIT], FALSE);
            gtk_widget_set_sensitive(buttons[DIR_LIST_BUTTON_DELETE], TRUE);
    }
}

static void
dir_list_change_selection_cb (GtkCList *dir_list, gint row, gint column,
                              GdkEventButton *event, gpointer data)
{
    dir_list_buttons_set_state (dir_list);
}


GtkWidget*
download_dirs_list_create (GtkWidget **dir_list) {    
    int i; 
    GtkWidget *hbox, *vbox, *dir_list_sw;

    gchar *titles[] = {
        N_ ("Pattern"),
        N_ ("Download to Directory")
    };
    gchar* t_titles[DIR_LIST_TOTAL_COLUMNS];

    /* Create hbox for columns */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), GNOME_PAD);

    /* vbox for list operation buttons */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);
    
    /* List operation buttons */
    buttons[DIR_LIST_BUTTON_INSERT] = gtk_button_new_with_label (_ ("Insert..."));
    buttons[DIR_LIST_BUTTON_EDIT] = gtk_button_new_with_label(_ ("Edit..."));
    buttons[DIR_LIST_BUTTON_DELETE] = gtk_button_new_with_label(_ ("Delete"));

    gtk_box_pack_start(GTK_BOX (vbox), buttons[DIR_LIST_BUTTON_INSERT], 
                       TRUE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX (vbox), buttons[DIR_LIST_BUTTON_EDIT], 
                       TRUE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX (vbox), buttons[DIR_LIST_BUTTON_DELETE],
                       TRUE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

    /* Regex/Directory list */
    for (i = 0; i < DIR_LIST_TOTAL_COLUMNS; i++)
        t_titles[i] = _ (titles[i]);
    *dir_list = gtk_clist_new_with_titles (DIR_LIST_TOTAL_COLUMNS,
                                          t_titles);
    
    gtk_clist_set_reorderable (GTK_CLIST (*dir_list), TRUE);
    gtk_clist_set_use_drag_icons (GTK_CLIST (*dir_list), FALSE);
    gtk_clist_column_titles_passive (GTK_CLIST (*dir_list));
    gtk_clist_set_column_auto_resize (GTK_CLIST (*dir_list), 
                                      DIR_LIST_COL_PATTERN, TRUE);
    gtk_clist_set_column_auto_resize (GTK_CLIST (*dir_list), 
                                      DIR_LIST_COL_DIR, TRUE);
    gtk_clist_set_selection_mode (GTK_CLIST (*dir_list), 
                                  GTK_SELECTION_EXTENDED);

    download_dirs_load_data (GTK_CLIST (*dir_list));

    /* Scrolled window for directory list */
    dir_list_sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_container_add (GTK_CONTAINER (dir_list_sw), 
                       *dir_list);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (dir_list_sw),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX (hbox), dir_list_sw, TRUE, TRUE, 0);

    /* Connect the button signals to callback functions */
    gtk_signal_connect (GTK_OBJECT (buttons[DIR_LIST_BUTTON_INSERT]), 
                                    "clicked", 
                                    GTK_SIGNAL_FUNC (add_dir_cb),
                                    (gpointer) dir_list);
    gtk_signal_connect (GTK_OBJECT (buttons[DIR_LIST_BUTTON_EDIT]), 
                        "clicked", 
                        GTK_SIGNAL_FUNC (edit_dir_cb),
                        (gpointer) dir_list);
    gtk_signal_connect (GTK_OBJECT (buttons[DIR_LIST_BUTTON_DELETE]), 
                        "clicked", 
                        GTK_SIGNAL_FUNC (delete_dir_cb),
                        (gpointer) dir_list);

    /* Connect the clist to callback functions */
    gtk_signal_connect (GTK_OBJECT (*dir_list), "select_row",
                        GTK_SIGNAL_FUNC (dir_list_change_selection_cb),
                        NULL);
    gtk_signal_connect (GTK_OBJECT (*dir_list), "unselect_row",
                        GTK_SIGNAL_FUNC (dir_list_change_selection_cb),
                        NULL);
    gtk_signal_connect (GTK_OBJECT (*dir_list), "key_press_event",
                        GTK_SIGNAL_FUNC (dir_list_keypress_cb),
                        (gpointer) dir_list);

    dir_list_buttons_set_state (GTK_CLIST(*dir_list));

    return hbox;
}


void
download_dirs_apply (GtkWidget* dir_list)
{
    gint i, old_total_dirs;
    char *pattern, *dir, *prefix;
    for (i = 0; ; i++) {
        if (gtk_clist_get_text ( 
                GTK_CLIST (dir_list), i, DIR_LIST_COL_PATTERN, &pattern) &&
            gtk_clist_get_text (
                GTK_CLIST (dir_list), i, DIR_LIST_COL_DIR, &dir)) {
            prefix = g_strdup_printf ("/gtm/Dir%d/", i);
            gnome_config_push_prefix (prefix);
            gnome_config_set_string ("pattern", pattern);
            gnome_config_set_string ("dir", dir);
            gnome_config_pop_prefix ();
            g_free (prefix);
        } else {
            break;
        }
    }
        
    /* Delete any remaining old data (if there are now fewer rows) */
    old_total_dirs = gnome_config_get_int ("/gtm/DirList/total_dirs");
    gnome_config_set_int ("/gtm/DirList/total_dirs", i);
    for (; i < old_total_dirs; i++) {
        prefix = g_strdup_printf ("/gtm/Dir%d/", i);
        gnome_config_clean_section (prefix);
        g_free (prefix);
    }        
}

gboolean
download_dirs_suggest_dir (char *url, char **dir) {
    gint total_dirs, i;
    char* prefix, *pattern;

    total_dirs = download_dirs_get_total_dirs();

    for (i = 0; i < total_dirs; i++) {    
        prefix = g_strdup_printf ("/gtm/Dir%d/", i);
        gnome_config_push_prefix (prefix);
        pattern = gnome_config_get_string ("pattern");
        if (fnmatch (pattern, url, 0) == 0) {
            *dir = gnome_config_get_string ("dir");
            gnome_config_pop_prefix ();
            g_free (prefix);
            return TRUE;
        }
        gnome_config_pop_prefix ();
        g_free (prefix);
    }

    return FALSE;
}

gint
download_dirs_get_total_dirs (void) 
{
    return gnome_config_get_int ("/gtm/DirList/total_dirs");
}
