/*
 *  Copyright (C) 1999, 2000 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <popt-gnome.h>
#include <gnome.h>
#include <libgnorba/gnorba.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <sys/utsname.h>
#include <sys/socket.h>
#include <net/route.h>
#include <net/if.h>
#include <sys/ioctl.h> 
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "main-window.h"
#include "Gtm.h"

/* The application */
GtkWidget *gtm_app = NULL;

/* Program preferences */
Preferences gtm_pref;

/* The program download list */
GtkWidget *dl_file_list = NULL;

/* Number of active automatic download */
gint num_of_download = 0;

/* ID used on the next file data created */
guint next_file_id = 0;

/* Variable used to disable events during startup */
static gboolean startup;

/* Session management */
static gint 
die (GnomeClient *client, gpointer client_data)
{
    /* When we receive this signal we leave the gtk_main */
    gtk_main_quit ();

    return FALSE;
}


/* Functions to load program preferences and save program geometry */
static void
save_geometry (void)
{
    gchar *geometry;

    /* Save the geometry to the configuration file */
    gnome_config_push_prefix ("/gtm/Display/");
    geometry = gnome_geometry_string (gtm_app->window);
    gnome_config_set_string ("geometry", geometry);
    g_free (geometry);
    gnome_config_pop_prefix ();
    gnome_config_sync ();
}

static gchar *
load_preferences (void)
{
    gboolean def;
    gchar *geometry;

    /* Read the geometry from the configuration file */
    gnome_config_push_prefix ("/gtm/Display/");
    geometry = gnome_config_get_string ("geometry=500x300");
    /* This geometry is g_free further ahead on init_main_window */
    gnome_config_pop_prefix ();

    /* Read global program preferences */
    gnome_config_push_prefix ("/gtm/Preferences/");

    gtm_pref.use_proxy =
        gnome_config_get_bool_with_default ("use_proxy=FALSE", &def);
    if (!def) {
        /* Remove old proxy user, pass, http and ftp from the visible config
         * since now I correctly use the private */
        gtm_pref.proxy_user = gnome_config_get_string ("proxy_user=");
        gtm_pref.proxy_pass = gnome_config_get_string ("proxy_pass=");
        gtm_pref.proxy_http = gnome_config_get_string ("proxy_http=");
        gtm_pref.proxy_ftp = gnome_config_get_string ("proxy_ftp=");
        gnome_config_clean_key ("use_proxy");
        gnome_config_clean_key ("proxy_user");
        gnome_config_clean_key ("proxy_pass");
        gnome_config_clean_key ("proxy_http");
        gnome_config_clean_key ("proxy_ftp");
        gnome_config_private_set_bool ("use_proxy", gtm_pref.use_proxy);
        gnome_config_private_set_string ("proxy_user", gtm_pref.proxy_user);
        gnome_config_private_set_string ("proxy_pass", gtm_pref.proxy_pass);
        gnome_config_private_set_string ("proxy_http", gtm_pref.proxy_http);
        gnome_config_private_set_string ("proxy_ftp", gtm_pref.proxy_ftp);
    } else {
        gtm_pref.use_proxy = gnome_config_private_get_bool ("use_proxy=FALSE");
        gtm_pref.proxy_user = gnome_config_private_get_string ("proxy_user=");
        gtm_pref.proxy_pass = gnome_config_private_get_string ("proxy_pass=");
        gtm_pref.proxy_http = gnome_config_private_get_string ("proxy_http=");
        gtm_pref.proxy_ftp = gnome_config_private_get_string ("proxy_ftp=");
    }
    gtm_pref.toolbar_style = 
	gnome_config_get_int_with_default ("toolbar_style", &def);
    if (def)
	gtm_pref.toolbar_style = GTK_TOOLBAR_BOTH;
    gtm_pref.def_download_dir = 
	gnome_config_get_string_with_default ("def_download_dir", &def);
    if (def)
        gtm_pref.def_download_dir = gnome_util_user_home ();
    gtm_pref.file_list_refresh = 
	gnome_config_get_int ("file_list_refresh=1000");
    gtm_pref.num_retries = gnome_config_get_int ("num_retries=0");
    gtm_pref.interface = gnome_config_get_string ("interface=ppp0");
    gtm_pref.num_of_auto_download = 
        gnome_config_get_int ("num_of_autodownload=1");
    gtm_pref.chk_interf_time = 
	gnome_config_get_int ("chk_interf_time=1000");
    gtm_pref.use_passive_ftp =
        gnome_config_get_bool ("use_passive_ftp=FALSE");
    gtm_pref.addnew_changes_def_dl_dir =
        gnome_config_get_bool ("addnew_changes_def_dl_dir=FALSE");
    gtm_pref.show_list_stat =
        gnome_config_get_bool ("show_list_stat=TRUE");
    gtm_pref.show_page_stat =
        gnome_config_get_bool ("show_page_stat=TRUE");
    gtm_pref.confirm_remove_incomplete =
        gtm_pref.confirm_remove_completed =
        gnome_config_get_bool_with_default ("confirm_remove=TRUE", &def);
    if (!def) {
        gnome_config_clean_key ("confirm_remove");
        gnome_config_set_bool ("confirm_remove_incomplete",
                               gtm_pref.confirm_remove_incomplete);
        gnome_config_set_bool ("confirm_remove_completed", 
                               gtm_pref.confirm_remove_completed);
        gnome_config_sync ();
    } else {
        gtm_pref.confirm_remove_incomplete =
            gnome_config_get_bool ("confirm_remove=TRUE");
        gtm_pref.confirm_remove_completed =
            gnome_config_get_bool ("confirm_remove_completed=TRUE");
    }
    gtm_pref.confirm_restart =
        gnome_config_get_bool ("confirm_restart=TRUE");
    gtm_pref.manual_stop_disables_auto_dl =
        gnome_config_get_bool ("manual_stop_disables_auto_dl=FALSE");
    gtm_pref.show_dl_info_page =
        gnome_config_get_bool ("show_dl_info_page=TRUE");
    gtm_pref.confirm_dnd_url =
        gnome_config_get_bool ("confirm_dnd_url=FALSE");
    gtm_pref.use_sound =
        gnome_config_get_bool ("use_sound=FALSE");
    gtm_pref.use_beep =
        gnome_config_get_bool ("use_beep=FALSE");
    gtm_pref.stop_dl_on_link_loss = 
        gnome_config_get_bool ("stop_dl_on_link_loss=FALSE");
    gnome_config_pop_prefix ();

    return geometry;
}


/* CORBA functions */
static void
Exception (CORBA_Environment *ev)
{
    switch (ev->_major) {
	case CORBA_SYSTEM_EXCEPTION:
	    g_log ("GNOME Transfer Manager", G_LOG_LEVEL_DEBUG, 
		   "CORBA system exception %s.\n", CORBA_exception_id (ev));
	    exit (1);
	case CORBA_USER_EXCEPTION:
	    g_log ("GNOME Transfer Manager", G_LOG_LEVEL_DEBUG, 
		   "CORBA user exception: %s.\n",
		   CORBA_exception_id (ev));
	    exit (1);
	default:
	    break;
    }
}

GTM_Download
impl_GTM_Download__create (PortableServer_POA poa, CORBA_Environment *ev);


/* Functions to handle the signals */
static void
signal_handler (int signal_num)
{
    /* Save program geometry */
    save_geometry ();

    /* Print error message and exit */
    g_printerr ("GTM: Signal %s caught - saving and exiting\n", 
		g_strsignal (signal_num));
    exit (1);
}

/* Function to check network and automaticaly start downloads if possible */
#if defined (__NetBSD__) || defined (linux)
static gint
check_network (gpointer data)
{
    struct ifreq ifr;
    int fd;
    
    fd = socket (PF_INET, SOCK_DGRAM, 0);
    if (fd == -1) {
        perror ("Couldn't open a socket");
        exit (1);
    }
    
    memset (&ifr, 0, sizeof (ifr)); 
    strcpy (ifr.ifr_name, gtm_pref.interface);
    if (ioctl (fd, SIOCGIFFLAGS, &ifr) < 0) {
       close (fd);
       return TRUE;
    }
    close (fd);

    if (ifr.ifr_flags & IFF_UP)
        /* We are connected to the network so we need to add more
        * downloads if possible */
        file_list_start_auto_downloads (GTK_CLIST (dl_file_list));
    else
        /* The interface is down, so stop downloads */
       file_list_stop_auto_downloads (GTK_CLIST (dl_file_list));

    return TRUE;
}

#else

static gboolean
check_network (gpointer data)
{
    /* We don't know if we are connected on the interface so we don't start
     * anything automatically */

    return TRUE;
}

#endif

/* Report an event by playing a sound or/and beeping */
void
report_event (gchar *which)
{
    if (!startup) {
        if (gtm_pref.use_sound)
            gnome_triggers_do (NULL, NULL, "gtm", which, NULL);

        if (gtm_pref.use_beep)
            gdk_beep ();
    }
}


/* The main program where all begins */
int 
main (int argc, char *argv[])
{
    CORBA_ORB orb;
    CORBA_Environment ev;
    CORBA_Object gtm_servant;
    PortableServer_POA root_poa;
    PortableServer_POAManager pm;
    GnomeClient *client;
    gchar *geometry;
    gint status;
    gint check_time;
    gchar *url_to_get = NULL;
    struct poptOption gtm_popt_options[] = {
        { "get-url", 'g', POPT_ARG_STRING, &url_to_get, 0,
          N_ ("url to fetch"), N_ ("URL")},
        { NULL, '\0', 0, NULL, 0 } /* end the list */
    };
    
    startup = TRUE;

    CORBA_exception_init (&ev);

    /* Initialize the i18n stuff */
    bindtextdomain (PACKAGE, GNOMELOCALEDIR);
    textdomain (PACKAGE);
   
    /* Initialize the application and CORBA stuff */
    orb = gnome_CORBA_init_with_popt_table ("gtm", VERSION, &argc, argv,
                                            gtm_popt_options, 0, NULL, 
                                            GNORBA_INIT_SERVER_FUNC, &ev);
    Exception (&ev);

    /* If we have a URL from the command line connect to the running gtm
     * object and send him the URL to fetch */
    if (url_to_get != NULL) {
        GTM_Download gtm_download_client;
        gboolean result;

        gtm_download_client = 
            goad_server_activate_with_repo_id (NULL,
                                               "IDL:gtm/download:1.0", 0,
                                               NULL);

        if (gtm_download_client != NULL) {
            result = GTM_Download_add_url (gtm_download_client,
                                           url_to_get, "", FALSE, FALSE, &ev);
            if (ev._major != CORBA_NO_EXCEPTION)
                g_printerr ("Exception thrown: %s\n", 
                            CORBA_exception_id (&ev));
            else if (result == FALSE) {
                g_print (_ ("URL:\n"));
                g_print ("%s\n", url_to_get);
                g_print (_ ("already in the list"));
                g_print ("\n");
            }
        }
        
        return 0;
    }

    /* Load program preferences */
    geometry = load_preferences ();

    /* Get the master client, that was hopefully connected to the
     * session manager int the 'gnome_init' call.  All communication to
     * the session manager will be done with this master client. */
    client = gnome_master_client ();
  
    /* Arrange to be told when something interesting happens. */
    gtk_signal_connect (GTK_OBJECT (client), "die",
			GTK_SIGNAL_FUNC (die), NULL);

    /* Initialize the main window */
    dl_file_list = main_window_create (geometry);
    g_free (geometry);

    /* Load file list */
    file_list_load_data (GTK_CLIST (dl_file_list));

    /* Install the timer */
    check_time = gtk_timeout_add (gtm_pref.chk_interf_time, check_network,
                                  NULL);

    /* Install a signal handler that saves the state of the application */
    signal (SIGINT, signal_handler);
    signal (SIGTERM, signal_handler);
    signal (SIGHUP, signal_handler);
    signal (SIGPIPE, signal_handler);
   
    /* Register the gtm CORBA object */
    root_poa = (PortableServer_POA) 
	CORBA_ORB_resolve_initial_references (orb, "RootPOA", &ev);
    Exception (&ev);
    gtm_servant = impl_GTM_Download__create (root_poa, &ev);
    Exception (&ev);
    pm = PortableServer_POA__get_the_POAManager (root_poa, &ev);
    Exception (&ev);
    PortableServer_POAManager_activate (pm, &ev);
    Exception (&ev);
    status = goad_server_register (CORBA_OBJECT_NIL, gtm_servant, 
				   "gtm_download", "object", &ev);
    Exception (&ev);
    
    /* Check if we are not registering another gtm object */
    if (status != 0) {
	/* Another GTM is already running and has already registered it self
	 * on the server */
	GtkWidget *error;

	error = gnome_message_box_new (_ (
	    "There is Another GNOME Transfer Manager running.\n"
	    "Only one GNOME Transfer Manager can be running."), 
				       GNOME_MESSAGE_BOX_ERROR,
				       GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_run_and_close (GNOME_DIALOG (error));
	exit (1);
    }

    /* Finished the startup */
    startup = FALSE;

    /* Enter the application main loop */
    gtk_main ();

    /* Destroy the timer */
    gtk_timeout_remove(check_time);
    
    /* Save program geometry */
    save_geometry ();

    return 0;
}
