/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gnome-scan
 * Copyright (C) Étienne Bersac 2007 <bersace03@laposte.net>
 * 
 * gnome-scan is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with gnome-scan.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

/**
 * SECTION: gnome-scan-job
 * @short_description: Handling acquisition, processing and outputting
 * @include: gnome-scan.h
 *
 * In Gnome Scan, the job of scanning is divided in three parts :
 * acquiring, processing and outputting. The first stage is handled by
 * #GnomeScanner plugin. The second is handled using
 * #GeglOperation. The latter is handled by #GnomeScanSink plugin.
 *
 * The purpose of GnomeScanJob is to marshal the configuration and
 * execution of these three part, allowing to easily delegate job
 * execution in a worker thread.
 **/
#include <glib/gi18n.h>
#include "gnome-scan-private.h"
#include <gegl.h>
#include "gnome-scan-processor-common.h"
#include "gnome-scan-job.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE((o), GNOME_TYPE_SCAN_JOB, GnomeScanJobPrivate))

typedef struct _GnomeScanJobPrivate GnomeScanJobPrivate;

struct _GnomeScanJobPrivate
{
	gboolean			disposed;
	GnomeScanSettings	*settings;
	GnomeScanner		*scanner;
	GSList				*processors;
	GnomeScanSink		*sink;
	
	GeglNode			*root;
	GeglNode			*gegl_convert;
	GeglNode			*gegl_sink;
	GList				*nodes;
	GeglProcessor		*gegl_processor;
	gboolean			ready;
};

enum
{
	PROP_0,
	PROP_SETTINGS,
	PROP_SCANNER,
	PROP_SINK
};


static GObjectClass* parent_class = NULL;

G_DEFINE_TYPE (GnomeScanJob, gnome_scan_job, G_TYPE_OBJECT);

static void
gnome_scan_job_init (GnomeScanJob *object)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE (object);
	
	priv->processors = NULL;
	priv->settings  = NULL;
	priv->sink		= NULL;
}

static void
gnome_scan_job_dispose (GObject *object)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE(object);
	
	if (!priv->disposed) { 
		g_object_unref(priv->settings);
		g_object_unref(priv->sink);
		priv->disposed = TRUE;
	}
	
	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
gnome_scan_job_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_JOB (object));

	switch (prop_id)
		{
		case PROP_SETTINGS:
			gnome_scan_job_set_settings (GNOME_SCAN_JOB (object),
										 GNOME_SCAN_SETTINGS (g_value_get_object (value)));
			break;
		case PROP_SCANNER:
			gnome_scan_job_set_scanner (GNOME_SCAN_JOB (object),
										GNOME_SCANNER (g_value_get_object (value)));
			break;
		case PROP_SINK:
			gnome_scan_job_set_sink (GNOME_SCAN_JOB (object),
									 GNOME_SCAN_SINK (g_value_get_object (value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
		}
}

static void
gnome_scan_job_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_JOB (object));

	switch (prop_id)
		{
		case PROP_SETTINGS:
			g_value_set_object (value,
								gnome_scan_job_get_settings (GNOME_SCAN_JOB (object)));
			break;
		case PROP_SCANNER:
			g_value_set_object (value,
								gnome_scan_job_get_scanner (GNOME_SCAN_JOB (object)));
			break;
		case PROP_SINK:
			g_value_set_object (value,
								gnome_scan_job_get_sink (GNOME_SCAN_JOB (object)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
		}
}

static void
gnome_scan_job_class_init (GnomeScanJobClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));

	g_type_class_add_private (klass, sizeof (GnomeScanJobPrivate));
	
	object_class->dispose = gnome_scan_job_dispose;
	object_class->set_property = gnome_scan_job_set_property;
	object_class->get_property = gnome_scan_job_get_property;

	/**
	 * GnomeScanJob:settings:
	 *
	 * The #GnomeScanSettings object containing all user settings for
	 * plugins.
	 **/	
	g_object_class_install_property (object_class,
	                                 PROP_SETTINGS,
	                                 g_param_spec_object ("settings",
	                                                      "Settings",
	                                                      "The set of parameters value for scanner and sink plugin.",
	                                                      GNOME_TYPE_SCAN_SETTINGS,
														  G_PARAM_READWRITE));

	/**
	 * GnomeScanJob:scanner:
	 *
	 * The #GnomeScanner responsible of image acquisition.
	 **/
	g_object_class_install_property (object_class,
	                                 PROP_SCANNER,
	                                 g_param_spec_object ("scanner",
	                                                      "Scanner",
	                                                      "The input scanner used for this job.",
	                                                      GNOME_TYPE_SCANNER,
	                                                      G_PARAM_READABLE | G_PARAM_WRITABLE));


	/**
	 * GnomeScanJob:sink:
	 *
	 * The sink responsible to output acquired images.
	 **/
	g_object_class_install_property (object_class,
	                                 PROP_SINK,
	                                 g_param_spec_object ("sink",
	                                                      "Sink",
	                                                      "The sink responsible to connect the job to the application.",
	                                                      GNOME_TYPE_SCAN_SINK,
	                                                      G_PARAM_READABLE | G_PARAM_WRITABLE));
}


/**
 * gnome_scan_job_new:
 * @settings:	a #GnomeScanSettings or NULL
 * @sink:		the #GnomeScanSink
 *
 * 
 * Instanciate a new job. If @settings is null, a new
 * #GnomeScanSettings is created.
 * 
 * Returns: a new #GnomeScanJob
 **/
GnomeScanJob*
gnome_scan_job_new (GnomeScanSettings *settings,
					GnomeScanSink *sink)
{
	if (settings == NULL) {
		settings = gnome_scan_settings_new();
	}
	else 
		g_object_ref(settings);

	GObject *object = g_object_new (GNOME_TYPE_SCAN_JOB,
									"settings", settings,
									"sink", sink,
									NULL);
	g_object_unref(settings);
	
	gnome_scan_job_add_processor (GNOME_SCAN_JOB(object),
								  gnome_scan_processor_common_new());
	
	return GNOME_SCAN_JOB (object);
}

/**
 * gnome_scan_job_set_settings:
 * @job: a #GnomeScanJob
 * @settings: a #GnomeScanSettings
 * 
 * Set the settings for this job.
 **/
void
gnome_scan_job_set_settings (GnomeScanJob *job,
							 GnomeScanSettings *settings)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE(job);
	
	if (priv->settings)
		g_object_unref(priv->settings);
	
	priv->settings = g_object_ref(settings);
}

/**
 * gnome_scan_job_get_settings:
 * @job: a #GnomeScanJob
 * 
 * Retrieve the settings used by the job.
 * 
 * Returns: the @job #GnomeScanSettings
 **/
GnomeScanSettings*
gnome_scan_job_get_settings (GnomeScanJob *job)
{
	return GET_PRIVATE (job)->settings;
}


/**
 * gnome_scan_job_set_scanner:
 * @job: a #GnomeScanJob
 * @scanner: a #GnomeScanner
 * 
 * Set the source of the acquisition.
 **/
void
gnome_scan_job_set_scanner (GnomeScanJob *job, GnomeScanner *scanner)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE (job);
	
	if (priv->scanner)
		g_object_unref (priv->scanner);
	
	priv->scanner = g_object_ref (scanner);
}

/**
 * gnome_scan_job_get_scanner:
 * @job: a #GnomeScanJob
 * 
 * Retrieve the scanner used to acquire image.
 * 
 * Returns: a #GnomeScanner
 **/
GnomeScanner*
gnome_scan_job_get_scanner (GnomeScanJob *job)
{
	return GET_PRIVATE (job)->scanner;
}

/**
 * gnome_scan_job_add_processor:
 * @job:		a #GnomeScanJob
 * @processor:	a #GnomeScanPlugin
 *
 * Add @processor as plugin for building the pipeline.
 **/
void
gnome_scan_job_add_processor (GnomeScanJob *job,
							  GnomeScanPlugin *processor)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE(job);
	priv->processors = g_slist_append(priv->processors, processor);
}

/**
 * gnome_scan_job_get_processors:
 * @job:	a #GnomeScanJob
 *
 * Returns: the list of processors
 */
GSList*
gnome_scan_job_get_processors (GnomeScanJob *job)
{
	return GET_PRIVATE(job)->processors;
}

/**
 * gnome_scan_job_set_sink:
 * @job: a #GnomeScanJob
 * @sink: a #GnomeScanSink
 * 
 * Set the scan sink.
 **/
void
gnome_scan_job_set_sink (GnomeScanJob *job,
						 GnomeScanSink *sink)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE(job);
	
	if (priv->sink)
		g_object_unref(priv->sink);
		
	priv->sink = g_object_ref(sink);
}

/**
 * gnome_scan_job_get_sink:
 * @job: a #GnomeScanJob
 * 
 * Retrieve the sink used to output acquired image.
 * 
 * Returns: a #GnomeScanSink
 **/
GnomeScanSink*
gnome_scan_job_get_sink (GnomeScanJob *job)
{
	return GET_PRIVATE (job)->sink;
}

/**
 * gnome_scan_job_configure:
 * @job: a #GnomeScanJob
 *
 * Must be called before running @job. This will ask @job plugins to
 * read their settings from :settings in order to bear ready for
 * processing images.
 **/
void
gnome_scan_job_configure (GnomeScanJob *job)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE (job);
	
	job->done		= FALSE;
	job->stage		= _("Configuring");
	job->progress	= 0.;
	
	gnome_scan_plugin_configure (GNOME_SCAN_PLUGIN (priv->scanner),
								 priv->settings);
	gnome_scan_plugin_configure (GNOME_SCAN_PLUGIN (priv->sink),
								 priv->settings);

	priv->ready = TRUE;
}

void
gsj_build_pipeline(GnomeScanJob *job)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE(job);
	GList *node, *node0;
	GSList *snode;
	GeglNode *root;
	gchar *op0, *op1;
	
	priv->root = root = gegl_node_new();
	/* scanner */
	node0 = node = gnome_scan_plugin_get_child_nodes (GNOME_SCAN_PLUGIN (priv->scanner),
													  root);
	/* processing */
	for (snode = priv->processors; snode; snode = snode->next) {
		gnome_scan_plugin_configure(GNOME_SCAN_PLUGIN(snode->data), priv->settings);
		node = g_list_concat (node,
							  gnome_scan_plugin_get_child_nodes (GNOME_SCAN_PLUGIN (snode->data),
																 root));
	}
	
	/* convert */
	priv->gegl_convert = gegl_node_new_child(root,
											 "operation", "convert-format",
											 NULL);
	node = g_list_append (node, priv->gegl_convert);
	
	/* sink  */
	node = g_list_concat (node,
						  gnome_scan_plugin_get_child_nodes (GNOME_SCAN_PLUGIN (priv->sink),
															 root));

	for (node = node0; node->next ; node = node->next) {
		gegl_node_get (node->data, "operation", &op0, NULL);
		gegl_node_get (node->next->data, "operation", &op1, NULL);
		g_debug ("Connecting %s to %s", op0, op1);
		g_free (op0);
		g_free (op1);
		gegl_node_connect_to (node->data, "output",
							  node->next->data, "input");
	}

	priv->gegl_sink = g_object_ref(node->data);
	priv->nodes = node0;
}

void
gsj_destroy_pipeline(GnomeScanJob *job)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE(job);

	if (!priv->root)
		return;

	g_object_unref(priv->root);
	priv->root = NULL;
	g_list_free(priv->nodes);
	priv->nodes = NULL;
}

/**
 * gnome_scan_job_run:
 * @job: a #GnomeScanJob
 * 
 * Configure, iterate each part of the job and monitor task
 * progress. You should call this function in a new thread. Ideally,
 * you should never use this function directly and use
 * #GnomeScanAcquisitionDialog instead.
 *
 * See: gnome_scan_job_run_once(), #GnomeScanAcquisitionDialog or g_thread_create()
 **/
void
gnome_scan_job_run (GnomeScanJob *job)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE (job);
	job->stop = FALSE;
	
	/* acquire */
	gint count = 0;
	
	while (gnome_scan_job_run_once (job))
		count++;

	/* unconfigure */
	gnome_scan_plugin_end_scan (GNOME_SCAN_PLUGIN (priv->sink));

	/* finish */
	job->progress	= 1.;
	job->stage		= _("Job completed");
	job->done		= TRUE;
}

/**
 * gnome_scan_job_run_once:
 * @job:	a #GnomeScanJob
 *
 * Acquire one image and process it, updated job progress, stage,
 * etc. Can be stop at any time using gnome_scan_job_cancel().
 **/
gboolean
gnome_scan_job_run_once	(GnomeScanJob *job)
{
	GnomeScanJobPrivate *priv = GET_PRIVATE (job);
	gdouble progress = 0.;

#define	stop(job)			if(job->stop){gsj_destroy_pipeline(job);return FALSE;}

	job->stage = _("Waiting for device");

	gsj_build_pipeline(job);

	if (!gnome_scan_plugin_start_frame (GNOME_SCAN_PLUGIN (priv->scanner))) {
		gsj_destroy_pipeline(job);
		return FALSE;
	}

	gegl_node_set (priv->gegl_convert,
				   "format", gnome_scanner_get_output_format (priv->scanner),
				   NULL);

	job->stage = _("Acquiring from scanner");

	while (!job->stop && gnome_scan_plugin_work (GNOME_SCAN_PLUGIN (priv->scanner), &progress)) {
		job->progress = progress / 3;
	}

	stop(job);

	/* configure the pipeline just before launching process. */
	gnome_scan_plugin_end_frame (GNOME_SCAN_PLUGIN (priv->scanner));
	gnome_scan_plugin_configure_frame (GNOME_SCAN_PLUGIN (priv->sink));

	/* process */
	job->progress = .33;
	job->stage = _("Processing page");
	progress = 0.;

	GeglProcessor *processor = gegl_node_new_processor (priv->gegl_sink, NULL);
	while (!job->stop && gegl_processor_work (processor, &progress)) {
		job->progress = .33 + progress / 3;
	}
	gegl_processor_destroy (processor);

	stop(job);
	
	/* sink */
	job->progress = .66;
	job->stage = _("Outputting page");
	progress = 0.;

	if (!gnome_scan_plugin_start_frame (GNOME_SCAN_PLUGIN (priv->sink))) {
		gsj_destroy_pipeline(job);
		return FALSE;
	}

	while (!job->stop && gnome_scan_plugin_work (GNOME_SCAN_PLUGIN (priv->sink), &progress)) {
		job->progress = .66 + progress / 3;
	}
	
	stop(job);
	
	gnome_scan_plugin_end_frame (GNOME_SCAN_PLUGIN (priv->sink));

	job->progress = 1.;
	
	gsj_destroy_pipeline(job);

#undef	stop

	return TRUE;
}

/**
 * gnome_scan_job_cancel:
 * @job: a running #GnomeScanJob
 * 
 * Ask the job to stop. The function returns immediatly, however the
 * job will actually stop as soon as possible.
 **/
void
gnome_scan_job_cancel (GnomeScanJob *job)
{
	job->stop = TRUE;
}

