/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescandialog.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gnomescan.h>
#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#define GET_PRIVATE(obj)			(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_DIALOG, GnomeScanDialogPrivate))
#define	GNOME_SCAN_DIALOG_PARENT_CLASS(klass)	(GTK_DIALOG_CLASS (g_type_class_peek_parent ((klass))))
#define GNOME_SCAN_DIALOG_ERROR			(g_type_qname (GNOME_TYPE_SCAN_DIALOG))

typedef struct _GnomeScanDialogPrivate		GnomeScanDialogPrivate;

struct _GnomeScanDialogPrivate {
  gboolean		dispose_has_run;
  GtkWidget		*acquisitiondialog;
  GtkWidget		*messagedialog;
  GtkWidget		*ui;
  GtkWidget		*about;

  /* messagedialog */
  /* errordialog */

};

enum {
  PROP_0,
  PROP_CONTEXT
};

void				about						(GtkButton *button,
										 GtkWindow *dialog);

void				dialog_close					(GtkWidget *widget,
										 GnomeScanDialog *dialog);

void				gnome_scan_dialog_probe_done			(GnomeScanContext *context,
										 GnomeScanDialog *dialog);

void				ok						(GtkButton *button,
										 GnomeScanDialog *dialog);

void				gnome_scan_dialog_scanner_selected		(GnomeScanContext *context,
										 GnomeScanner *scanner,
										 GnomeScanDialog *dialog);

void				gnome_scan_dialog_acquisition_started		(GnomeScanContext *context,
										 gint size,
										 GtkWidget *dialog);

void				gnome_scan_dialog_acquisition_terminated	(GnomeScanContext *context,
										 GtkWidget *dialog);

/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_dialog_dispose			(GObject *obj);

void				gnome_scan_dialog_set_property 			(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_dialog_get_property 			(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

G_DEFINE_TYPE (GnomeScanDialog, gnome_scan_dialog, GTK_TYPE_WINDOW);

void
gnome_scan_dialog_init (GnomeScanDialog *dialog)
{
  GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);

  dialog->context 		= NULL;

  priv->dispose_has_run 	= FALSE;
  priv->acquisitiondialog	= NULL;
  priv->messagedialog		= NULL;
  priv->ui 			= NULL;
  priv->about 			= NULL;
}

void
gnome_scan_dialog_dispose (GObject *obj)
{
  GnomeScanDialog *dialog = GNOME_SCAN_DIALOG (obj);
  GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
  GnomeScanDialogClass *b_klass = GNOME_SCAN_DIALOG_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref context */
  g_object_unref (dialog->context);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_DIALOG_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_dialog_class_init (GnomeScanDialogClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property	= gnome_scan_dialog_set_property;
  gobject_class->get_property	= gnome_scan_dialog_get_property;
  gobject_class->dispose 	= gnome_scan_dialog_dispose;

  g_type_class_add_private (gobject_class, sizeof (GnomeScanDialogPrivate));

  /* PROPERTIES */

  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"A GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

}

void
gnome_scan_dialog_set_property (GObject *obj,
				guint property_id,
				const GValue *value,
				GParamSpec *pspec)
{
  GnomeScanDialog *dialog = GNOME_SCAN_DIALOG (obj);

  switch (property_id) {
  case PROP_CONTEXT:
    dialog->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id,pspec);
    break;
  }
}

void
gnome_scan_dialog_get_property (GObject *obj,
				guint property_id,
				GValue *value,
				GParamSpec *pspec)
{
  GnomeScanDialog *dialog = GNOME_SCAN_DIALOG (obj);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, dialog->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,property_id,pspec);
    break;
  }
}



/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_dialog_new:
 * @context: a #GnomeScanContext
 * @parent: a #GtkWidget or NULL
 * @ui: a #GtkWidget
 * 
 * Create a new #GnomeScanDialog connnected to @context. If @parent is
 * not NULL, then the new dialog will be transient for @parent. @ui is
 * the widget tree containing configuration widget showed in the body
 * of the #GnomeScanDialog.
 * 
 * Return value: a new #GnomeScanDialog
 */
GtkWidget*
gnome_scan_dialog_new (GnomeScanContext *context,
		       GtkWindow *parent,
		       GtkWidget *ui)
{
  GtkWidget *dialog, *button, *box, *bbox, *hbox;
  GnomeScanDialogPrivate *priv;
  gchar *string;

  dialog = GTK_WIDGET (g_object_new (GNOME_TYPE_SCAN_DIALOG,
				     "context", context,
				     NULL));

  priv = GET_PRIVATE (dialog);

  /* customize widget */
  gtk_window_set_icon_name (GTK_WINDOW (dialog), PACKAGE);

/*   gtk_window_set_default_size (GTK_WINDOW (dialog), */
/* 			       200, /\* width *\/ */
/* 			       200); /\* height *\/ */

  gtk_window_set_title (GTK_WINDOW (dialog), _("Scan"));

  gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);

  if (GTK_IS_WINDOW (parent)) {
    gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);
  }
  else {
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ALWAYS);
  }

  /* UI */
  box = gtk_vbox_new (FALSE, 6);
  gtk_container_add (GTK_CONTAINER (dialog),
		     box);

  priv->ui = ui;
  gtk_box_pack_start (GTK_BOX (box),
		      ui,
		      TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX (box),
		      gtk_hseparator_new (),
		      FALSE, TRUE, 0);

  /* BUTTONS */
  bbox = gtk_hbutton_box_new ();
  gtk_box_set_spacing (GTK_BOX (bbox), 6);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox),
			     GTK_BUTTONBOX_END);

  gtk_box_pack_start (GTK_BOX (box),
		      bbox,
		      FALSE, TRUE, 0);

  /* about */
  button = gtk_button_new_from_stock (GTK_STOCK_ABOUT);
  gtk_box_pack_start (GTK_BOX (bbox),
		      button,
		      FALSE, FALSE, 0);
  gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (bbox),
				      button,
				      TRUE);

  g_signal_connect (GTK_BUTTON (button),
		    "clicked",
		    G_CALLBACK (about),
		    GTK_WINDOW (dialog));

  /* add: help */

  /* close */
  button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
  gtk_box_pack_end (GTK_BOX (bbox),
		    button,
		    FALSE, FALSE, 0);

  g_signal_connect (GTK_BUTTON (button),
		    "clicked",
		    G_CALLBACK (dialog_close),
		    dialog);

  /* scan */
  button = gtk_button_new_from_stock (GS_STOCK_SCAN);
  gtk_box_pack_end (GTK_BOX (bbox),
		    button,
		    FALSE, FALSE, 0);

  g_signal_connect (GTK_BUTTON (button),
		    "clicked",
		    G_CALLBACK (ok),
		    dialog);


  /* Acquisition dialog */
  priv->acquisitiondialog = gnome_scan_acquisition_dialog_new (context,
							       GTK_WINDOW (dialog));

  /* Message Dialog */
  string = g_strconcat ("<big><b>", _("Probing devices"), "</b></big>", NULL);
  priv->messagedialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog),
							    GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
							    GTK_MESSAGE_INFO,
							    GTK_BUTTONS_NONE,
							    string);
  g_free (string);

  gtk_window_set_icon_name (GTK_WINDOW (priv->messagedialog),
			    PACKAGE);

  gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (priv->messagedialog),
					    _("The program is probing devices. Please be patient ..."));

  /* Signals */
  g_signal_connect (context,
		    "probe-done",
		    (GCallback) gnome_scan_dialog_probe_done,
		    dialog);

  g_signal_connect (context,
		    "scanner-selected",
		    (GCallback) gnome_scan_dialog_scanner_selected,
		    GNOME_SCAN_DIALOG (dialog));

  g_signal_connect (context,
		    "preview-started",
		    (GCallback) gnome_scan_dialog_acquisition_started,
		    dialog);

  g_signal_connect (context,
		    "preview-terminated",
		    (GCallback) gnome_scan_dialog_acquisition_terminated,
		    dialog);

  g_signal_connect (context,
		    "acquisition-started",
		    (GCallback) gnome_scan_dialog_acquisition_started,
		    dialog);

  g_signal_connect (context,
		    "acquisition-terminated",
		    (GCallback) gnome_scan_dialog_acquisition_terminated,
		    dialog);


  g_signal_connect (GTK_WIDGET (dialog),
		    "delete-event",
		    G_CALLBACK (dialog_close),
		    dialog);

  return (dialog);
}

/**
 * gnome_scan_dialog_run:
 * @dialog: a #GnomeScanDialog
 * 
 * Run the dialog waiting for the use to configure the scan and
 * trigger acquisition.
 */
void
gnome_scan_dialog_run (GnomeScanDialog *dialog)
{
  GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);

  gtk_widget_set_sensitive (GTK_WIDGET (dialog), FALSE);

  /*   gtk_window_set_transient_for (GTK_WINDOW (messagedialog), */
  /* 				GTK_WINDOW (dialog)); */

  gtk_widget_show_all (GTK_WIDGET (dialog));
  gtk_widget_show (priv->messagedialog);

  while (gtk_events_pending ()) 
    gtk_main_iteration ();

  gnome_scan_context_probe_scanners (dialog->context);
}

/* CALLBACKS */

/* i really hate that about handling */
void
about (GtkButton *button,
       GtkWindow *dialog)
{
  const gchar *authors[] = {
    "Étienne Bersac <bersace03@laposte.net>",
    NULL
  };

  const gchar *license[] = {
    "This program is free software; you can redistribute it and/or"
    "modify it under the terms of the GNU Lesser General Public License"
    "as published by the Free Software Foundation; either version 2 of"
    "the License, or (at your option) any later version.",
    "This program is distributed in the hope that it will be useful, but"
    "WITHOUT ANY WARRANTY; without even the implied warranty of"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU"
    "General Public License for more details.",
    "You should have received a copy of the GNU General Public License"
    "along with this program; if not, write to the Free Software"
    "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307"
    "USA"
  };
		     
  gtk_show_about_dialog (dialog,
			 "name", PACKAGE,
			 "version", VERSION,
			 "comments", g_strconcat (_("A Gnome scan utility"), "\n\n",
						  "In Memoriam Mark Finlay\n",
						  NULL),
			 "authors", authors,
			 "logo-icon-name", PACKAGE,
			 "license", g_strjoin("\n\n", license[0], license[1], license[2]),
			 "wrap-license", TRUE,
			 "copyright", "Copyright © 2006 Étienne Bersac",
			 NULL);
}

void
dialog_close (GtkWidget *widget,
	      GnomeScanDialog *dialog)
{
  /*   g_debug ("%s:%i", __FILE__, __LINE__); */

  gtk_widget_hide (GTK_WIDGET (dialog));
  gtk_main_quit ();
}

/* Enable UI after probe */
void
gnome_scan_dialog_probe_done (GnomeScanContext *context,
			      GnomeScanDialog *dialog)
{

  gtk_widget_set_sensitive (GTK_WIDGET (dialog), TRUE);
  gtk_widget_hide (GET_PRIVATE (dialog)->messagedialog);

  gtk_main ();
}

/* Apply the user choice */
void
ok (GtkButton *button,
    GnomeScanDialog *dialog)
{
  gnome_scan_context_start_acquisition (dialog->context);
  dialog_close (GTK_WIDGET (button), dialog);
}

/* Update the title of the window */
void
gnome_scan_dialog_scanner_selected (GnomeScanContext *context,
				    GnomeScanner *scanner,
				    GnomeScanDialog *dialog)
{
  gchar *text = g_strdup_printf (_("Scan from %s"),
				 gnome_scanner_get_product (scanner));

  gtk_window_set_title (GTK_WINDOW (dialog),
			text);
}

void
gnome_scan_dialog_acquisition_started (GnomeScanContext *context,
				       gint size,
				       GtkWidget *dialog)
{
  gtk_widget_set_sensitive (dialog, FALSE);
}

void
gnome_scan_dialog_acquisition_terminated (GnomeScanContext *context,
					  GtkWidget *dialog)
{
  gtk_widget_set_sensitive (dialog, TRUE);
}
