/* libgnome-ppp - The GNOME PPP Dialer Library
 * Copyright (C) 1997 Jay Painter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <errno.h>
#include "gnome-ppp.h"
#include "misc.h"
#include "modem.h"
#include "pppd.h"


/* XXX: this is lame. -JMP */
/* various timeouts in seconds for operations */
#define GET_TIMEOUT        5
#define DETECT_TIMEOUT     5
#define DIAL_TIMEOUT      60
#define SCRIPT_TIMEOUT    30
#define PPPD_TIMEOUT       5
#define FIRST_STATE        0


/* a couple of macros to make the code more readable */
#define SET_TIMEOUT(t, s) ((t) = time(NULL) + (s))
#define TIMEOUT(t)        ((t) < time(NULL))
#define NEXT_STATE(s)     ((s)++)
#define CALLBACK(c, t, m) ((*(c)->func)((c)->account, (t), (m), ((c)->data)))


/* master state */
enum
{
  _GET_MODEM,
  _DETECT_MODEM,
  _DIAL,
  _RUN_SCRIPT,
  _EXEC_PPPD,
  _CONNECTED
};


/* message indexes for callback messages */
enum
{
  _MSG_NO_PPPD_EXEC_PERMISSION,
  _MSG_MODEM_LOCKED,
  _MSG_PPPD_FORK,
  _MSG_CLOSING,
  _MSG_MODEM_FD_PASS_TIMEOUT,
  _MSG_MODEM_DETECT,
  _MSG_MODEM_DETECT_FAILED,
  _MSG_NO_PHONE_NUMBER,
  _MSG_DIALING,
  _MSG_DIAL_TIMEOUT,
  _MSG_NO_DIALTONE,
  _MSG_BUSY,
  _MSG_CONNECTED,
  _MSG_RUNNING_SCRIPT,
  _MSG_SCRIPT_COMPLETE,
  _MSG_PPP_CONNECTED,
  _MSG_PPP_FAIL,
  _MSG_LAST_MESSAGE
};


typedef struct _Connect
{
  Account *account;

  ConnectCBFunc func;
  gpointer data;

  Modem *modem;
  GString *session_buffer;
  GString *modem_buffer;
  time_t timeout;
  time_t connect_time;

  /* state keepers */
  gint state;
  gint get_state;
  gint detect_state;
  gint dial_state;
  gint pppd_state;

  /* dial phone number list node */
  GList *phone_number_node;

  /* current script entry */
  GList *script_entry_node;

  gboolean close;
  gboolean dead;
} Connect;


static Connect*
malloc_connect()
{
  Connect *connect = g_malloc(sizeof(Connect));
  connect->account = NULL;
  connect->func = NULL;
  connect->data = NULL;
  connect->modem = NULL;
  connect->session_buffer = g_string_new("");
  connect->modem_buffer = g_string_new("");
  connect->timeout = 0;
  connect->connect_time = 0;

  connect->state = FIRST_STATE;
  connect->get_state = FIRST_STATE;
  connect->detect_state = FIRST_STATE;
  connect->dial_state = FIRST_STATE;
  connect->pppd_state = FIRST_STATE;

  connect->phone_number_node = NULL;
  connect->script_entry_node = NULL;
  connect->close = FALSE;
  connect->dead = FALSE;
  return connect;
}


static void
free_connect(Connect *connect)
{
  g_assert(connect != NULL);

  g_string_free(connect->session_buffer, TRUE);
  g_string_free(connect->modem_buffer, TRUE);
  g_free(connect);
}


/* list of all active connections  */
static gchar           *__messages[_MSG_LAST_MESSAGE];
static GHashTable      *__connect_hash = NULL;
static gchar           *__modem_responses[] =
{
  "NO DIALTONE\r",
  "NO CARRIER\r",
  "BUSY\r",
  "CONNECT",
  NULL
};


/* state machine for the entire connection sequence */
static gint state_engine_iteration(Connect *connect);
static void state_get_modem(Connect *connect);
static void state_detect_modem(Connect *connect);
static void state_dial(Connect *connect);
static void state_run_script(Connect *connect);
static void state_exec_pppd(Connect *connect);
static void state_connected(Connect *connect);


/* callbacks */
static void pppd_die_cb(gpointer data);


/* misc */
static void destroy_connect(Connect *connect);
static gboolean read_charactor(Connect *connect);
static gboolean search_modem_buffer(Connect *connect, gchar *find);
static gchar *search_modem_buffer_list(Connect *connect, gchar *find_list[]);


/* initalizer registered with init.c */
void
__init_connect()
{
  static gboolean done_init = FALSE;

  if (!done_init)
    {
      done_init = TRUE;
      __connect_hash = g_hash_table_new(g_direct_hash, NULL);

      /* initalize messages */
      __messages[_MSG_NO_PPPD_EXEC_PERMISSION] =
	_("You don't have permission to execute pppd.");

      __messages[_MSG_MODEM_LOCKED] =
	_("The modem is in use by another program.");

      __messages[_MSG_PPPD_FORK] =
	_("Running pppd.");

      __messages[_MSG_CLOSING] =
	_("Closing connection.");

      __messages[_MSG_MODEM_FD_PASS_TIMEOUT] =
	_("The pppd program failed to run gnome-ppp-chat.");

      __messages[_MSG_MODEM_DETECT] =
	_("Initalizing modem.");

      __messages[_MSG_MODEM_DETECT_FAILED] =
	_("Your modem is not responding to initalization.");

      __messages[_MSG_NO_PHONE_NUMBER] =
	_("No phone numbers specified for this account.");

      __messages[_MSG_DIALING] =
	_("Dialing.");

      __messages[_MSG_NO_DIALTONE] =
	_("No dialtone.");

      __messages[_MSG_BUSY] =
	_("Number Busy.");

      __messages[_MSG_CONNECTED] =
	_("Connected.");

      __messages[_MSG_RUNNING_SCRIPT] =
	_("Running Script.");

      __messages[_MSG_SCRIPT_COMPLETE] =
	_("Script Complete.");

      __messages[_MSG_PPP_CONNECTED] =
	_("PPP connection established.");

      __messages[_MSG_PPP_FAIL] =
	_("Failed to establish a PPP connection.");
    }
}


gint
connect_start(Account *account, ConnectCBFunc func, gpointer data)
{
  Connect *connect;

  g_assert(account != NULL);

  /* don't start a connection on a account that's already in use */
  if (g_hash_table_lookup(__connect_hash, account))
    {
      return;
    }

  /* create connect info struct */
  connect = malloc_connect();
  connect->account = account;
  connect->func = func;
  connect->data = data;

  /* add to the hash */
  g_hash_table_insert(__connect_hash, account, connect);

  /* for pppd -- this forks pppd with the gnome-ppp-chat */
  if (pppd_exec(connect->account, pppd_die_cb, connect))
    {
      CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_PPPD_FORK]);
    }
  else
    {
      switch (pppd_errno)
	{
	case PPPD_NO_EXEC_PERMISSION:
	  CALLBACK(connect, PPP_ERROR, __messages[_MSG_NO_PPPD_EXEC_PERMISSION]);
	  goto error;

	case PPPD_MODEM_LOCKED:
	  CALLBACK(connect, PPP_ERROR, __messages[_MSG_MODEM_LOCKED]);
	  goto error;

	default:
	  g_error("connect_start(): unknown pppd error");
	  break;
	}
    }

  return 1;

 error:
  /* remove connection from hash */
  CALLBACK(connect, PPP_DISCONNECTED, NULL);
  g_hash_table_remove(__connect_hash, connect->account);
  free_connect(connect);
  return 0;
}


gint
connect_engine_iteration(Account *account)
{
  Connect *connect;
  
  g_assert(account != NULL);

  connect = g_hash_table_lookup(__connect_hash, account);
  if (!connect)
    {
      return;
    }

  return state_engine_iteration(connect);
}


void
connect_stop(Account *account)
{
  Connect *connect;

  g_assert(account != NULL);
  
  connect = g_hash_table_lookup(__connect_hash, account);
  if (!connect)
    {
      return;
    }

  /* inform the user the connection is being closed, make sure
   * to use the proper connect state in the callback
   */
  if (connect->state == _CONNECTED)
    {
      CALLBACK(connect, PPP_CONNECTED, __messages[_MSG_CLOSING]);
    }
  else
    {
      CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_CLOSING]);
    }
  connect->close = TRUE;
}


/* this is the main state machine for the ENTIRE connection cycle */
static gint
state_engine_iteration(Connect *connect)
{
  /* get the modem's file descriptor and create a modem 
   * communication class
   */
  if (!connect->close && connect->state == _GET_MODEM)
    {
      state_get_modem(connect);
    }

  /* detect the modem */
  if (!connect->close && connect->state == _DETECT_MODEM)
    {
      state_detect_modem(connect);
    }

  /* dial the number */
  if (!connect->close && connect->state == _DIAL)
    {
      state_dial(connect);
    }

  /* run the user script */
  if (!connect->close && connect->state == _RUN_SCRIPT)
    {
      state_run_script(connect);
    }

  /* run the pppd daemon */
  if (!connect->close && connect->state == _EXEC_PPPD)
    {
      state_exec_pppd(connect);
    }

  /* connected */
  if (!connect->close && connect->state == _CONNECTED)
    {
      state_connected(connect);
    }

  /* end the connection */
  if (connect->close)
    {
      if (!connect->dead)
	{
	  pppd_kill(connect->account);
	}

      if (connect->dead)
	{
	  goto destroy;
	}
    }

  return 1;

 destroy:
  destroy_connect(connect);
  return 0;
}


static void
state_get_modem(Connect *connect)
{
  gint modem_fd;

  enum
  {
    __SET_TIMEOUT,
    __RECIEVE_FD
  };

  switch (connect->get_state)
    {
    case __SET_TIMEOUT:
      SET_TIMEOUT(connect->timeout, GET_TIMEOUT);
      NEXT_STATE(connect->get_state);
      break;


    case __RECIEVE_FD:
      if (pppd_modem_fd(connect->account, &modem_fd))
	{
	  connect->modem = modem_new(modem_fd, connect->account->speed);
	  if (!connect->modem)
	    {
	      g_error("state_get_modem(): ioctl failure on modem");
	    }
	  NEXT_STATE(connect->state);
	}
      else
	{
	  switch (pppd_errno)
	    {
	    case PPPD_PENDING:
	      if (TIMEOUT(connect->timeout))
		{
		  CALLBACK(connect, PPP_ERROR, __messages[_MSG_MODEM_FD_PASS_TIMEOUT]);
		  connect->close = TRUE;
		  goto error;
		}
	      break;

	    case PPPD_LOOKUP_FAIL:
	      goto error;

	    default:
	      g_error("state_get_modem(): unknown pppd errno");
	      break;
	    }
	}
    }

 error:
  return;
}


static void
state_detect_modem(Connect *connect)
{
  gint index;
  GString *gstr;

  enum
  {
    __AT_INIT,
    __OK
  };

  gstr = g_string_new("");

  switch (connect->detect_state)
    {
    case __AT_INIT:
      CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_MODEM_DETECT]);

      g_string_sprintf(gstr, "%s\r", connect->account->modem_init->str);
      modem_write(connect->modem, gstr->str, gstr->len);

      SET_TIMEOUT(connect->timeout, DETECT_TIMEOUT);
      NEXT_STATE(connect->detect_state);
      break;
      

    case __OK:
      index = g_string_find(connect->modem_buffer, "OK\r", TRUE);

      /* modem deteted -- flush the modem buffers and go to the
       * next stage of dialup
       */
      if (search_modem_buffer(connect, "OK\r"))
	{
	  modem_drain(connect->modem);
	  NEXT_STATE(connect->state);
	}
      else
	{
	  /* we've timed out while trying to detect the modem, inform
	   * the GUI and set the close flag for this connection
	   */
	  if (TIMEOUT(connect->timeout))
	    {
	      CALLBACK(connect, PPP_ERROR, __messages[_MSG_MODEM_DETECT_FAILED]);
	      connect->close = TRUE;
	    }
	}
      break;
    }

 cleanup:
  g_string_free(gstr, TRUE);
}


static void
state_dial(Connect *connect)
{
  gchar *result;
  GString *gstr;
  GString *phone_number;

  enum
  {
    __CALL,
    __ANSWER
  };

  gstr = g_string_new("");

  switch (connect->dial_state)
    {
    case __CALL:

      /* the account HAS to have at least one phone number to dial */
      if (g_list_length(connect->account->phone_list) == 0)
	{
	  CALLBACK(connect, PPP_ERROR, __messages[_MSG_NO_PHONE_NUMBER]);
	  connect->close = TRUE;
	  goto cleanup;
	}
      
      /* get a phone number node */
      if (connect->phone_number_node == NULL ||
	  connect->phone_number_node->next == NULL)
	{
	  connect->phone_number_node = connect->account->phone_list;
	}
      else
	{
	  connect->phone_number_node = connect->phone_number_node->next;
	}

      phone_number = (GString *) connect->phone_number_node->data;
      if (phone_number)
	{
	  CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_DIALING]);

	  modem_drain(connect->modem);

	  g_string_sprintf(gstr, "ATDT%s\r", phone_number->str);
	  modem_write(connect->modem, gstr->str, gstr->len);
	  SET_TIMEOUT(connect->timeout, DIAL_TIMEOUT);
	  connect->dial_state = __ANSWER;
	}
      break;


    case __ANSWER:
      result = search_modem_buffer_list(connect, __modem_responses);

      /* TIMEOUT */
      if (!result && TIMEOUT(connect->timeout))
	{
	  g_string_assign(connect->modem_buffer, "");
	  connect->dial_state = __CALL;
	  goto cleanup;
	}

      /* no match -- continue with life */
      if (!result)
	{
	  goto cleanup;
	}

      /* no dialtone */
      if (strcmp(result, "NO DIALTONE\r") == 0)
	{
	  CALLBACK(connect, PPP_ERROR, __messages[_MSG_NO_DIALTONE]);
	  connect->close = TRUE;
	  goto cleanup;
	}

      /* no carrior */
      if (strcmp(result, "NO CARRIER\r") == 0)
	{
	  CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_BUSY]);
	  connect->dial_state = __CALL;
	  goto cleanup;
	}

      /* busy */
      if (strcmp(result, "BUSY\r") == 0)
	{
	  CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_BUSY]);
	  connect->dial_state = __CALL;
	  goto cleanup;
	}

      /* we've connected */
      if (strcmp(result, "CONNECT") == 0)
	{
	  CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_CONNECTED]);
	  NEXT_STATE(connect->state);
	  goto cleanup;
	}
      break;
    }

 cleanup:
  g_string_free(gstr, TRUE);
}


static void
state_run_script(Connect *connect)
{
  gboolean next;
  ScriptEntry *script_entry;
  GString *gstr;

  gstr = g_string_new("");

  /* initalize the script entry glist to the first
   * script entry in the account
   */
  if (!connect->script_entry_node)
    {
      CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_RUNNING_SCRIPT]);

      if (connect->account->script_list)
	{
	  connect->script_entry_node = connect->account->script_list;
	}
      else
	{
	  /* if there's no script for this account, then jump
	   * to the next connection state
	   */
	  NEXT_STATE(connect->state);
	  goto cleanup;
	}
    }

  next = FALSE;
  script_entry = (ScriptEntry *) connect->script_entry_node->data;

  switch (script_entry->type)
    {
    case SCRIPT_ENTRY_EXPECT:
      if (search_modem_buffer(connect, script_entry->text->str))
	{
	  next = TRUE;
	}
      break;


    case SCRIPT_ENTRY_SEND:
      g_string_sprintf(gstr, "%s\r\n", script_entry->text->str);
      modem_write(connect->modem, gstr->str, gstr->len);
      next = TRUE;
      break;


    case SCRIPT_ENTRY_SEND_USER:
      g_string_sprintf(gstr, "%s\r\n", connect->account->user->str);
      modem_write(connect->modem, gstr->str, gstr->len);
      next = TRUE;
      break;

      
    case SCRIPT_ENTRY_SEND_PASSWORD:
      g_string_sprintf(gstr, "%s\r\n", connect->account->passwd->str);
      modem_write(connect->modem, gstr->str, gstr->len);
      next = TRUE;
      break;
    }

  /* get the next element of the script to execute,
   * and if there's no more go to the next state of the connect
   * state engine
   */
  if (next)
    {
      connect->script_entry_node = connect->script_entry_node->next;
 
      if (connect->script_entry_node == NULL)
	{
	  CALLBACK(connect, PPP_IN_PROGRESS, __messages[_MSG_SCRIPT_COMPLETE]);
	  NEXT_STATE(connect->state);
	}
    }

 cleanup:
  g_string_free(gstr, TRUE);
}


static void
state_exec_pppd(Connect *connect)
{
  gint device_number;

  enum
  {
    __START_PPP,
    __DETECT_PPP_CONNECTION
  };
  

  switch (connect->pppd_state)
    {
    case __START_PPP:
      /* close the connection to the modem */
      modem_destroy(connect->modem);
      connect->modem = NULL;
      
      /* ends gnome-ppp-chat allowing pppd to attempt to 
       * establish a PPP connection
       */
      if (pppd_end_chat(connect->account))
	{
	  SET_TIMEOUT(connect->timeout, PPPD_TIMEOUT);
	  NEXT_STATE(connect->pppd_state);
	}
      else
	{
	  switch (pppd_errno)
	    {
	    case PPPD_LOOKUP_FAIL:
	      goto error;
 
	    default:
	      g_error("state_exec_pppd(): unknown error from pppd_end_chat");
	      break;
	    }
	}
      break;


    case __DETECT_PPP_CONNECTION:
      if (pppd_device_number(connect->account, &device_number))
	{
	  CALLBACK(connect, PPP_CONNECTED, __messages[_MSG_PPP_CONNECTED]);
	  NEXT_STATE(connect->state);
	}
      else
	{
	  switch (pppd_errno)
	    {
	    case PPPD_PENDING:
	      if (TIMEOUT(connect->timeout))
		{
		  CALLBACK(connect, PPP_ERROR, __messages[_MSG_PPP_FAIL]);
		  connect->close = TRUE;
		  goto error;
		}
	      break;

	    case PPPD_LOOKUP_FAIL:
	      goto error;
	      
	    default:
	      g_error("state_exec_pppd(): unknown error from pppd_device_number");
	      break;
	    }
	}
      break;
    }

 error:
}


static void
state_connected(Connect *connect)
{
#if 0
  GString *gstr;
  gint total_seconds, hours, minutes;

  gstr = g_string_new("");

  /* set the inital time of connect */
  if (connect->connect_time == 0)
    {
      connect->connect_time = time(NULL);
    }

  total_seconds = time(NULL) - connect->connect_time;

  hours = total_seconds / 3600;
  minutes = (total_seconds % 3600) / 60;

  g_string_sprintf(gstr, "Connected %d:%d", hours, minutes);
  CALLBACK(connect, PPP_CONNECTED, gstr->str);

 cleanup:
  g_string_free(gstr, TRUE);
#endif
}


/* callbacks */
static void
pppd_die_cb(gpointer data)
{
  Connect *connect;

  g_assert(data != NULL);

  connect = (Connect *) data;
  connect->close = TRUE;
  connect->dead = TRUE;
}


/* MISC */
static void
destroy_connect(Connect *connect)
{
  /* destroy the modem communictation object */
  if (connect->modem)
    {
      modem_hangup(connect->modem);
      modem_destroy(connect->modem);
      connect->modem = NULL;
    }

  CALLBACK(connect, PPP_DISCONNECTED, NULL);

  /* remove connection from hash */
  g_hash_table_remove(__connect_hash, connect->account);
  free_connect(connect);
}


static gboolean
read_charactor(Connect *connect)
{
  gint len;
  gchar c;

  if (modem_read_ready(connect->modem))
    {
      len = modem_read(connect->modem, &c, 1);
      if (len == 1)
	{
	  g_string_append_c(connect->session_buffer, c);
	  g_string_append_c(connect->modem_buffer, c);
	}
      else
	{
	  connect->close = TRUE;
	  return FALSE;
	}

      return TRUE;
    }
  else
    {
      return FALSE;
    }
}


static gboolean
search_modem_buffer(Connect *connect, gchar *find)
{
  while (read_charactor(connect))
    {
      if (g_string_find(connect->modem_buffer, find, FALSE) >= 0)
	{
	  g_string_assign(connect->modem_buffer, "");
	  return TRUE;
	}
    }

  /* just in case there was nothing to read, but we need to search
   * the recieve buffer anyways
   */
  if (g_string_find(connect->modem_buffer, find, FALSE) >= 0)
    {
      g_string_assign(connect->modem_buffer, "");
      return TRUE;
    }

  return FALSE;
}


static gchar*
search_modem_buffer_list(Connect *connect, gchar *find_list[])
{
  gint i;
  gchar *find;

  while (read_charactor(connect))
    {
      for (i = 0; find_list[i]; i++)
	{
	  find = find_list[i];
	  
	  if (g_string_find(connect->modem_buffer, find, FALSE) >= 0)
	    {
	      g_string_assign(connect->modem_buffer, "");
	      return find;
	    }
	}
    }

  /* just in case there was nothing to read, but we need to search
   * the recieve buffer anyways
   */
  for (i = 0; find_list[i]; i++)
    {
      find = find_list[i];
      
      if (g_string_find(connect->modem_buffer, find, FALSE) >= 0)
	{
	  g_string_assign(connect->modem_buffer, "");
	  return find;
	}
    }
  
  return NULL;
}
