/*  -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4; coding: utf-8 -*-
 * 
 * Copyright (C) 2003 Gustavo Giráldez
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 * 
 * Author: Gustavo Giráldez <gustavo.giraldez@gmx.net>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdarg.h>
#include <glade/glade-xml.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <bonobo/bonobo-file-selector-util.h>
#include <gdl/gdl-icons.h>

#include "gbf-i18n.h"
#include "gbf-tree-data.h"
#include "gbf-project-view.h"
#include "gbf-project.h"
#include "gbf-project-model.h"
#include "gbf-project-util.h"
#include "eggtreemodelfilter.h"


#define GLADE_FILE GBF_GLADEDIR "/create_dialogs.glade"

static GladeXML *
load_interface (const gchar *top_widget)
{
    GladeXML *xml;

    xml = glade_xml_new (GLADE_FILE, top_widget, GETTEXT_PACKAGE);

    if (!xml) {
	g_warning (_("Couldn't load glade file"));
	return NULL;
    }
    
    return xml;
}

static gboolean
groups_filter_fn (GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data)
{
    GbfTreeData *data = NULL;
    gboolean retval = FALSE;

    gtk_tree_model_get (model, iter,
                        GBF_PROJECT_MODEL_COLUMN_DATA, &data, -1);
    retval = (data && data->type == GBF_TREE_NODE_GROUP);
    gbf_tree_data_free (data);
    
    return retval;
}

static void 
setup_groups_treeview (GbfProjectModel *model,
		       GtkWidget       *view,
                       const gchar     *select_group)
{
    GtkTreeModel *filter;
    GtkTreePath *path;
    GtkTreeIter iter;
    
    g_return_if_fail (model != NULL);
    g_return_if_fail (view != NULL && GBF_IS_PROJECT_VIEW (view));

    filter = egg_tree_model_filter_new (GTK_TREE_MODEL (model), NULL);
    egg_tree_model_filter_set_visible_func (EGG_TREE_MODEL_FILTER (filter),
					    groups_filter_fn, NULL, NULL);
    gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (filter));
    g_object_unref (filter);
    gtk_tree_view_expand_all (GTK_TREE_VIEW (view));

    /* select default group */
    if (select_group && gbf_project_model_find_id (model, &iter,
                                                   GBF_TREE_NODE_GROUP, select_group)) {
        GtkTreeIter iter_filter;
        
        egg_tree_model_filter_convert_child_iter_to_iter (
            EGG_TREE_MODEL_FILTER (filter), &iter_filter, &iter);
        path = gtk_tree_model_get_path (filter, &iter_filter);

    } else {
        GtkTreePath *root_path;

        root_path = gbf_project_model_get_project_root (model);
        if (root_path) {
            path = egg_tree_model_filter_convert_child_path_to_path (
                EGG_TREE_MODEL_FILTER (filter), root_path);
            gtk_tree_path_free (root_path);
        } else {
            path = gtk_tree_path_new_first ();
        }
    }
    
    gtk_tree_view_set_cursor (GTK_TREE_VIEW (view), path, NULL, FALSE);
    gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (view), path, NULL, TRUE, 0.5, 0.0);
    gtk_tree_path_free (path);
}

static void
entry_changed_cb (GtkEditable *editable, gpointer user_data)
{
    GtkWidget *button = user_data;
    gchar *text;

    if (!button)
        return;
    
    text = gtk_editable_get_chars (editable, 0, -1);
    if (strlen (text) > 0) {
        gtk_widget_set_sensitive (button, TRUE);
        gtk_widget_grab_default (button);
    } else {
        gtk_widget_set_sensitive (button, FALSE);
    }
    g_free (text);
}

static void
error_dialog (GtkWindow *parent, const gchar *msg, ...)
{
    va_list ap;
    gchar *tmp;
    GtkWidget *dialog;
    
    va_start (ap, msg);
    tmp = g_strdup_vprintf (msg, ap);
    va_end (ap);
    
    dialog = gtk_message_dialog_new (parent,
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_OK,
                                     tmp);
    g_free (tmp);
    
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
}

void 
gbf_project_util_new_group (GbfProjectModel *model,
			    GtkWindow       *parent,
			    const gchar     *default_group)
{
    GladeXML *gui;
    GtkWidget *dialog, *groups_ph, *group_name_entry, *ok_button;
    GtkWidget *groups_view;
    gint response;
    GbfProject *project;
    gboolean finished = FALSE;
    
    g_return_if_fail (model != NULL);
    
    project = gbf_project_model_get_project (model);
    if (!project)
        return;

    gui = load_interface ("new_group_dialog");
    g_return_if_fail (gui != NULL);
    
    /* get all needed widgets */
    dialog = glade_xml_get_widget (gui, "new_group_dialog");
    groups_ph = glade_xml_get_widget (gui, "groups_ph");
    group_name_entry = glade_xml_get_widget (gui, "group_name_entry");
    ok_button = glade_xml_get_widget (gui, "ok_button");
    
    /* set up dialog */
    g_signal_connect (group_name_entry, "changed",
                      (GCallback) entry_changed_cb, ok_button);
    gtk_widget_set_sensitive (ok_button, FALSE);
    
    groups_view = gbf_project_view_new ();
    gtk_widget_show (groups_view);
    gtk_container_add (GTK_CONTAINER (groups_ph), groups_view);
    setup_groups_treeview (model, groups_view, default_group);
    
    if (parent) {
	gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    }
    
    /* execute dialog */
    while (!finished) {
        response = gtk_dialog_run (GTK_DIALOG (dialog));

        switch (response) {
            case GTK_RESPONSE_OK: 
            {
                GError *err = NULL;
                GbfTreeData *data;
                gchar *parent_id = NULL, *name;
            
                name = gtk_editable_get_chars (
                    GTK_EDITABLE (group_name_entry), 0, -1);
                data = gbf_project_view_find_selected (GBF_PROJECT_VIEW (groups_view),
                                                       GBF_TREE_NODE_GROUP);
                if (data) {
                    gchar *new_group = NULL;
                    
                    gtk_widget_hide (dialog);
                    
                    parent_id = g_strdup (data->id);
                    gbf_tree_data_free (data);
            
                    new_group = gbf_project_add_group (project, parent_id, name, &err);
                    if (err) {
                        error_dialog (parent, _("Can't add group: %s"),
                                      err->message);
                        g_error_free (err);
                    }
            
                    g_free (new_group);
                    g_free (parent_id);
                    finished = TRUE;
                    
                } else {
                    error_dialog (parent, _("No parent group selected"));
                
                }
            
                g_free (name);

                break;
            }
            default:
                finished = TRUE;
                break;
        }
    }
        
    /* destroy stuff */
    gtk_widget_destroy (dialog);
    g_object_unref (gui);
}

static GtkWidget *
build_types_menu (GbfProject *project)
{
    GtkWidget *menu;
    gchar **types;
    gint i;
    GdlIcons *icons;
    
    types = gbf_project_get_types (project);
    menu = gtk_menu_new ();
    g_object_set_data_full (G_OBJECT (menu), "types-stringv",
                            types, (GDestroyNotify) g_strfreev);
    icons = gdl_icons_new (24, 16.0);
    
    for (i = 0; types [i]; i++) {
        GtkWidget *item, *image = NULL;
        GdkPixbuf *pixbuf;
        const gchar *name;

        name = gbf_project_name_for_type (project, types [i]);
        pixbuf = gdl_icons_get_mime_icon (icons,
                                          gbf_project_mimetype_for_type (project, types [i]));
        item = gtk_image_menu_item_new_with_label (name);
        if (pixbuf) {
            image = gtk_image_new_from_pixbuf (pixbuf);
            gtk_widget_show (image);
            gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), image);
            g_object_unref (pixbuf);
        }
        gtk_widget_show (item);
        gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    }

    g_object_unref (icons);
    
    return menu;
}

static void 
type_changed_cb (GtkWidget *widget, gpointer user_data)
{
    GtkWidget *menu;
    gchar **types, *selected_type;
    
    g_return_if_fail (GTK_IS_OPTION_MENU (widget));
    menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widget));

    types = g_object_get_data (G_OBJECT (menu), "types-stringv");
    selected_type = types [gtk_option_menu_get_history (
                               GTK_OPTION_MENU (widget))];
    g_object_set_data_full (G_OBJECT (widget), "type",
                            g_strdup (selected_type), (GDestroyNotify) g_free);
}

void 
gbf_project_util_new_target (GbfProjectModel *model,
			     GtkWindow       *parent,
			     const gchar     *default_group)
{
    GladeXML *gui;
    GtkWidget *dialog, *groups_ph, *target_name_entry, *ok_button;
    GtkWidget *target_type_optionmenu, *groups_view, *types_menu;
    gint response;
    GbfProject *project;
    gboolean finished = FALSE;
    
    g_return_if_fail (model != NULL);
    
    project = gbf_project_model_get_project (model);
    if (!project)
        return;

    gui = load_interface ("new_target_dialog");
    g_return_if_fail (gui != NULL);
    
    /* get all needed widgets */
    dialog = glade_xml_get_widget (gui, "new_target_dialog");
    groups_ph = glade_xml_get_widget (gui, "groups_ph");
    target_name_entry = glade_xml_get_widget (gui, "target_name_entry");
    target_type_optionmenu =
        glade_xml_get_widget (gui, "target_type_optionmenu");
    ok_button = glade_xml_get_widget (gui, "ok_button");
    
    /* set up dialog */
    g_signal_connect (target_name_entry, "changed",
                      (GCallback) entry_changed_cb, ok_button);
    gtk_widget_set_sensitive (ok_button, FALSE);
    
    groups_view = gbf_project_view_new ();
    gtk_widget_show (groups_view);
    gtk_container_add (GTK_CONTAINER (groups_ph), groups_view);
    setup_groups_treeview (model, groups_view, default_group);

    types_menu = build_types_menu (project);
    gtk_option_menu_set_menu (GTK_OPTION_MENU (target_type_optionmenu),
                              types_menu);
    g_signal_connect (target_type_optionmenu, "changed",
                      (GCallback) type_changed_cb, NULL);
    gtk_option_menu_set_history (GTK_OPTION_MENU (target_type_optionmenu), 0);
    /* force data setting */
    type_changed_cb (target_type_optionmenu, NULL);
    
    if (parent) {
	gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    }
    
    /* execute dialog */
    while (!finished) {
        response = gtk_dialog_run (GTK_DIALOG (dialog));

        switch (response) {
            case GTK_RESPONSE_OK: 
            {
                GError *err = NULL;
                GbfTreeData *data;
                gchar *group_id = NULL, *name, *type;
                
                name = gtk_editable_get_chars (
                    GTK_EDITABLE (target_name_entry), 0, -1);
                data = gbf_project_view_find_selected (GBF_PROJECT_VIEW (groups_view),
                                                       GBF_TREE_NODE_GROUP);
                type = g_object_get_data (G_OBJECT (target_type_optionmenu), "type");
                
                if (data) {
                    gchar *new_target = NULL;
                    
                    gtk_widget_hide (dialog);
                    
                    group_id = g_strdup (data->id);
                    gbf_tree_data_free (data);
            
                    new_target = gbf_project_add_target (project, group_id, name, type, &err);
                    if (err) {
                        error_dialog (parent, _("Can't add target: %s"),
                                      err->message);
                        g_error_free (err);
                    }
            
                    g_free (new_target);
                    g_free (group_id);
                    finished = TRUE;
                    
                } else {
                    error_dialog (parent, _("No group selected"));
                
                }
            
                g_free (name);

                break;
            }
            default:
                finished = TRUE;
                break;
        }
    }
    
    /* destroy stuff */
    gtk_option_menu_remove_menu (GTK_OPTION_MENU (target_type_optionmenu));
    gtk_widget_destroy (dialog);
    g_object_unref (gui);
}

static gboolean
targets_filter_fn (GtkTreeModel *model, GtkTreeIter *iter, gpointer user_data)
{
    GbfTreeData *data = NULL;
    gboolean retval = FALSE;

    gtk_tree_model_get (model, iter,
                        GBF_PROJECT_MODEL_COLUMN_DATA, &data, -1);
    retval = (data && !data->is_shortcut &&
              (data->type == GBF_TREE_NODE_GROUP ||
               data->type == GBF_TREE_NODE_TARGET));
    gbf_tree_data_free (data);
    
    return retval;
}

static void 
setup_targets_treeview (GbfProjectModel *model,
                        GtkWidget       *view,
                        const gchar     *select_target)
{
    GtkTreeModel *filter;
    GtkTreeIter iter, iter_filter;
    
    g_return_if_fail (model != NULL);
    g_return_if_fail (view != NULL && GBF_IS_PROJECT_VIEW (view));

    filter = egg_tree_model_filter_new (GTK_TREE_MODEL (model), NULL);
    egg_tree_model_filter_set_visible_func (EGG_TREE_MODEL_FILTER (filter),
					    targets_filter_fn, NULL, NULL);
    gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (filter));
    g_object_unref (filter);
    gtk_tree_view_expand_all (GTK_TREE_VIEW (view));

    /* select default target */
    if (select_target) {
        if (gbf_project_model_find_id (model, &iter,
                                       GBF_TREE_NODE_TARGET, select_target)) {
            GtkTreePath *path;
            
            egg_tree_model_filter_convert_child_iter_to_iter (
                EGG_TREE_MODEL_FILTER (filter), &iter_filter, &iter);
            path = gtk_tree_model_get_path (filter, &iter_filter);
            gtk_tree_view_set_cursor (GTK_TREE_VIEW (view), path, NULL, FALSE);
            gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (view), path, NULL, TRUE, 0.5, 0.0);
            gtk_tree_path_free (path);
        }
    }
}

static void
browse_button_clicked_cb (GtkWidget *widget, gpointer user_data)
{
    GtkEntry *entry = user_data;
    gchar *uri, *selected_uri;
    GnomeVFSURI *tmp_uri;
    
    g_return_if_fail (user_data != NULL && GTK_IS_ENTRY (user_data));

    uri = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    tmp_uri = gnome_vfs_uri_new (uri);
    g_free (uri);
    uri = NULL;
    if (tmp_uri) {
        uri = gnome_vfs_uri_extract_dirname (tmp_uri);
        gnome_vfs_uri_unref (tmp_uri);
    }
    selected_uri = bonobo_file_selector_open (
        GTK_WINDOW (gtk_widget_get_toplevel (widget)),
        FALSE, _("Add source..."), NULL,
        uri ? uri : g_object_get_data (G_OBJECT (widget), "root"));
    g_free (uri);

    if (selected_uri) {
        gtk_entry_set_text (entry, selected_uri);
        g_free (selected_uri);
    }
}

void 
gbf_project_util_add_source (GbfProjectModel *model,
			     GtkWindow       *parent,
			     const gchar     *default_target,
			     const gchar     *default_uri)
{
    GladeXML *gui;
    GtkWidget *dialog, *targets_ph, *source_file_entry;
    GtkWidget *ok_button, *browse_button;
    GtkWidget *targets_view;
    gint response;
    GbfProject *project;
    gboolean finished = FALSE;
    gchar *project_root;
    
    g_return_if_fail (model != NULL);
    
    project = gbf_project_model_get_project (model);
    if (!project)
        return;

    gui = load_interface ("add_source_dialog");
    g_return_if_fail (gui != NULL);
    
    /* get all needed widgets */
    dialog = glade_xml_get_widget (gui, "add_source_dialog");
    targets_ph = glade_xml_get_widget (gui, "targets_ph");
    source_file_entry = glade_xml_get_widget (gui, "source_file_entry");
    browse_button = glade_xml_get_widget (gui, "browse_button");
    ok_button = glade_xml_get_widget (gui, "ok_button");
    
    /* set up dialog */
    g_signal_connect (source_file_entry, "changed",
                      (GCallback) entry_changed_cb, ok_button);
    if (default_uri) {
        gtk_entry_set_text (GTK_ENTRY (source_file_entry), default_uri);
        gtk_widget_set_sensitive (ok_button, TRUE);
    } else {
        gtk_widget_set_sensitive (ok_button, FALSE);
    }
    g_signal_connect (browse_button, "clicked",
                      (GCallback) browse_button_clicked_cb, source_file_entry);
    g_object_get (project, "project-dir", &project_root, NULL);
    g_object_set_data_full (G_OBJECT (browse_button), "root",
                            project_root, g_free);
    
    targets_view = gbf_project_view_new ();
    gtk_widget_show (targets_view);
    gtk_container_add (GTK_CONTAINER (targets_ph), targets_view);
    setup_targets_treeview (model, targets_view, default_target);
    
    if (parent) {
	gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    }

    if (default_target)
        gtk_widget_grab_focus (source_file_entry);
    else
        gtk_widget_grab_focus (targets_view);
    
    /* execute dialog */
    while (!finished) {
        response = gtk_dialog_run (GTK_DIALOG (dialog));

        switch (response) {
            case GTK_RESPONSE_OK: 
            {
                GError *err = NULL;
                GbfTreeData *data;
                gchar *target_id = NULL, *source_file;
            
                data = gbf_project_view_find_selected (GBF_PROJECT_VIEW (targets_view),
                                                       GBF_TREE_NODE_TARGET);
                if (data) {
                    gchar *new_source = NULL;
                    
                    gtk_widget_hide (dialog);
                    
                    target_id = g_strdup (data->id);
                    gbf_tree_data_free (data);
                    source_file = gtk_editable_get_chars (
                        GTK_EDITABLE (source_file_entry), 0, -1);

                    new_source = gbf_project_add_source (project, target_id,
                                                         source_file, &err);
                    if (err) {
                        error_dialog (parent, _("Can't add source: %s"),
                                      err->message);
                        g_error_free (err);
                    }

                    g_free (new_source);
                    g_free (source_file);
                    g_free (target_id);
                    finished = TRUE;
                    
                } else {
                    error_dialog (parent, _("No target has been selected"));
                
                }
	
                break;
            }
            default:
                finished = TRUE;
                break;
        }
    }
    
    /* destroy stuff */
    gtk_widget_destroy (dialog);
    g_object_unref (gui);
}

