/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gbf-build-info.c
 *
 * Copyright (C) 2001  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <libgnome/gnome-macros.h>
#include "gbf-i18n.h"
#include "gbf-project.h"
#include "gbf-build-info.h"
#include "libgbfmarshal.h"

enum {
	WARNING_SELECTED,
	ERROR_SELECTED,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_PROJECT
};

struct _GbfBuildInfoPrivate {
	GbfProject *prj;

	GtkWidget *text_view;
	GtkWidget *text_sw;
	GtkTextBuffer *buffer;

	GHashTable *warnings;
	GHashTable *errors;
};

static guint info_signals [LAST_SIGNAL] = { 0 };

static void get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);
static void set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void dispose      (GObject *object);

static void build_callback (GbfProject     *project,
			    GbfBuildMessage msg,
			    gpointer        data,
			    gpointer        user_data);

static GtkTextTagTable *create_tag_table (GbfBuildInfo *info);

static gboolean motion_notify_cb (GtkWidget      *widget,
				  GdkEventMotion *event,
				  gpointer        user_data);

static gboolean warning_link_cb (GtkTextTag  *texttag,
				 GObject     *obj,
				 GdkEvent    *event,
				 GtkTextIter *iter,
				 gpointer     data);
static gboolean error_link_cb (GtkTextTag  *texttag,
			       GObject     *obj,
			       GdkEvent    *event,
			       GtkTextIter *iter,
			       gpointer     data);

GNOME_CLASS_BOILERPLATE (GbfBuildInfo, gbf_build_info, GtkVBox, GTK_TYPE_VBOX);

static void 
gbf_build_info_class_init (GbfBuildInfoClass *klass)
{
	GObjectClass   *g_object_class;

	g_object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	g_object_class->dispose = dispose;
	g_object_class->get_property = get_property;
	g_object_class->set_property = set_property;

	g_object_class_install_property 
		(g_object_class, PROP_PROJECT,
		 g_param_spec_pointer ("project", 
				       _("Project"),
				       _("Project Object"),
				       G_PARAM_READWRITE));

	info_signals [WARNING_SELECTED] = 
		g_signal_new ("warning_selected", 
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GbfBuildInfoClass, warning_selected),
			      NULL, /* accumulator */
			      NULL, /* accu_data */
			      gbf_marshal_VOID__STRING_INT,
			      G_TYPE_NONE, /* return type */
			      2,
			      G_TYPE_STRING,
			      G_TYPE_INT);
	info_signals [ERROR_SELECTED] = 
		g_signal_new ("error_selected", 
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GbfBuildInfoClass, error_selected),
			      NULL, /* accumulator */
			      NULL, /* accu_data */
			      gbf_marshal_VOID__STRING_INT,
			      G_TYPE_NONE, /* return type */
			      2,
			      G_TYPE_STRING,
			      G_TYPE_INT);
}

static void
gbf_build_info_instance_init (GbfBuildInfo *info)
{
	GbfBuildInfoPrivate *priv;
	const char *msg = _("No build started");
	PangoFontDescription *font_desc = NULL;

	priv = g_new0 (GbfBuildInfoPrivate, 1);

	info->priv = priv;

	priv->warnings = g_hash_table_new (g_direct_hash, g_direct_equal);
	priv->errors = g_hash_table_new (g_direct_hash, g_direct_equal);

	priv->buffer = gtk_text_buffer_new (create_tag_table (info));
	priv->text_view = gtk_text_view_new_with_buffer (priv->buffer);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (priv->text_view), FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (priv->text_view), FALSE);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (priv->text_view), GTK_WRAP_CHAR);
	gtk_text_buffer_insert_at_cursor (priv->buffer, msg, strlen (msg));
	g_signal_connect (priv->text_view, "motion-notify-event",
			  G_CALLBACK (motion_notify_cb), info);

	font_desc = pango_font_description_from_string ("mono 9");
	gtk_widget_modify_font (priv->text_view, font_desc);
	pango_font_description_free (font_desc);

	priv->text_sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (priv->text_sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (priv->text_sw),
					     GTK_SHADOW_IN);

	gtk_container_add (GTK_CONTAINER (priv->text_sw), priv->text_view);
	gtk_container_add (GTK_CONTAINER (info), priv->text_sw);
}

static void
get_property (GObject    *object,
	      guint       prop_id,
	      GValue     *value,
	      GParamSpec *pspec)
{
	GbfBuildInfo *info = GBF_BUILD_INFO (object);

	switch (prop_id) {
	case PROP_PROJECT:
		g_value_set_pointer (value, info->priv->prj);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
set_property (GObject      *object,
	      guint         prop_id,
	      const GValue *value,
	      GParamSpec   *pspec)
{
	GbfBuildInfo *info = GBF_BUILD_INFO (object);

	switch (prop_id) {
	case PROP_PROJECT:
		if (info->priv->prj) {
			gbf_project_remove_build_callback (info->priv->prj,
							   build_callback,
							   NULL);
			g_object_unref (info->priv->prj);
		}

		info->priv->prj = g_value_get_pointer (value);
		if (info->priv->prj) {
			g_object_ref (info->priv->prj);
			gbf_project_add_build_callback (info->priv->prj,
							build_callback,
							info,
							NULL);
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}   
}

static void
dispose (GObject *object) 
{
	GbfBuildInfo *info;
	GbfBuildInfoPrivate *priv;

	info = GBF_BUILD_INFO (object);
	priv = info->priv;

	if (priv) {
		if (priv->prj) {
			gbf_project_remove_build_callback (priv->prj,
							   build_callback,
							   NULL);
			g_object_unref (priv->prj);
		}
		
		g_hash_table_destroy (priv->warnings);
		g_hash_table_destroy (priv->errors);

		g_free (priv);
		info->priv = NULL;
	}
}

static gboolean
foreach_warning_remove (gpointer key,
			gpointer value,
			gpointer user_data)
{
	GbfBuildWarning *warn = value;
	g_free (warn->filename);
	g_free (warn->warning);
	g_free (warn->output);
	g_free (warn);
	return TRUE;
}

static gboolean
foreach_error_remove (gpointer key,
		      gpointer value,
		      gpointer user_data)
{
	GbfBuildError *err = value;
	g_free (err->filename);
	g_free (err->error);
	g_free (err->output);
	g_free (err);
	return TRUE;
}

static void
build_callback (GbfProject     *project,
		GbfBuildMessage msg,
		gpointer        data,
		gpointer        user_data)
{
	GbfBuildInfo *info = GBF_BUILD_INFO (user_data);
	GbfBuildInfoPrivate *priv = info->priv;
	GtkTextIter start, end;
	GtkTextMark *mark;
	int offset;
	GbfBuildWarning *warn;
	GbfBuildError *err;

	switch (msg) {
	case GBF_BUILD_START:
		gtk_text_buffer_get_start_iter (priv->buffer, &start);
		gtk_text_buffer_get_end_iter (priv->buffer, &end);
		gtk_text_buffer_delete (priv->buffer, &start, &end);

		gtk_text_buffer_get_end_iter (priv->buffer, &start);
		gtk_text_buffer_insert_with_tags_by_name (priv->buffer,
							  &start,
							  data, strlen (data),
							  "status", NULL);

		g_hash_table_foreach_remove (priv->warnings, foreach_warning_remove, NULL);
		g_hash_table_foreach_remove (priv->errors, foreach_error_remove, NULL);
		break;
	case GBF_BUILD_END:
		gtk_text_buffer_get_end_iter (priv->buffer, &start);
		gtk_text_buffer_insert_with_tags_by_name (priv->buffer,
							  &start,
							  data, strlen (data),
							  "status", NULL);
		break;
	case GBF_BUILD_OUTPUT:
		gtk_text_buffer_get_end_iter (priv->buffer, &start);
		gtk_text_buffer_insert_with_tags_by_name (priv->buffer,
							  &start,
							  data, strlen (data),
							  "output", NULL);
		break;
	case GBF_BUILD_WARNING:
		warn = data;
		gtk_text_buffer_get_end_iter (priv->buffer, &start);
		offset = gtk_text_iter_get_offset (&start);
		g_hash_table_insert (priv->warnings, GINT_TO_POINTER (offset), warn);
		gtk_text_buffer_insert_with_tags_by_name (priv->buffer,
							  &start,
							  warn->output, strlen (warn->output),
							  "warning", NULL);
		gtk_text_buffer_get_end_iter (priv->buffer, &start);
		/* Non-breaking zero-width space (aka byte order mark) to
		   separate warnings */
		gtk_text_buffer_insert(priv->buffer, &start, "\xEF\xBB\xBF",3);
		break;
	case GBF_BUILD_ERROR:
		err = data;
		gtk_text_buffer_get_end_iter (priv->buffer, &start);
		offset = gtk_text_iter_get_offset (&start);
		g_hash_table_insert (priv->errors, GINT_TO_POINTER (offset), err);
		gtk_text_buffer_insert_with_tags_by_name (priv->buffer,
							  &start,
							  err->output, strlen (err->output),
							  "error", NULL);
		gtk_text_buffer_get_end_iter (priv->buffer, &start);
		/* Non-breaking zero-width space (aka byte order mark) to
		   separate errors */
		gtk_text_buffer_insert(priv->buffer, &start, "\xEF\xBB\xBF",3);
		break;
	}

	/* Scroll to end. */
	mark = gtk_text_buffer_get_insert (priv->buffer);
	gtk_text_view_scroll_to_mark (GTK_TEXT_VIEW (priv->text_view),
				      mark, 0.0, TRUE, 0.0, 1.0);
}

static GtkTextTagTable *
create_tag_table (GbfBuildInfo *info)
{
	GtkTextTagTable *table;
	GtkTextTag *tag;
	GdkColor color;

	table = gtk_text_tag_table_new ();

	/* Standard build output. */
	tag = gtk_text_tag_new ("output");
	gtk_text_tag_table_add (table, tag);
	gdk_color_parse ("#9F9F9F", &color);
	g_object_set (G_OBJECT (tag), "foreground-gdk", &color, NULL);

	/* Build warning. */
	tag = gtk_text_tag_new ("warning");
	gtk_text_tag_table_add (table, tag);
	g_object_set (G_OBJECT (tag), "foreground", "yellow", NULL);
	g_object_set (G_OBJECT (tag), "underline", PANGO_UNDERLINE_SINGLE, NULL);
	g_signal_connect (G_OBJECT (tag), "event", G_CALLBACK (warning_link_cb), info);

	/* Build error. */
	tag = gtk_text_tag_new ("error");
	gtk_text_tag_table_add (table, tag);
	g_object_set (G_OBJECT (tag), "foreground", "red", NULL);
	g_object_set (G_OBJECT (tag), "underline", PANGO_UNDERLINE_SINGLE, NULL);
	g_signal_connect (G_OBJECT (tag), "event", G_CALLBACK (error_link_cb), info);

	/* Build status. */
	tag = gtk_text_tag_new ("status");
	gtk_text_tag_table_add (table, tag);
	g_object_set (G_OBJECT (tag), "foreground", "blue", NULL);

	return table;
}


static gboolean
motion_notify_cb (GtkWidget      *widget,
		  GdkEventMotion *event,
		  gpointer        user_data)
{
	static GdkCursor *hand_cursor = NULL;
	static GdkCursor *ibar_cursor = NULL;

	GtkTextView *view = GTK_TEXT_VIEW (widget);
	GtkTextTag *warn_tag, *err_tag;
	GtkTextTagTable *table;
	GtkTextWindowType type;
	GtkTextIter iter;
	GdkWindow *win;
	int x, y, buf_x, buf_y;

	type = gtk_text_view_get_window_type (view, event->window);

	win = gtk_text_view_get_window (view, type);
	gdk_window_get_pointer (win, &x, &y, NULL);

	if (type != GTK_TEXT_WINDOW_TEXT)
		return FALSE;

	gtk_text_view_window_to_buffer_coords (view, type, x, y, &buf_x, &buf_y);

	gtk_text_view_get_iter_at_location (view, &iter, buf_x, buf_y);

	table = gtk_text_buffer_get_tag_table (gtk_text_view_get_buffer (view));
	warn_tag = gtk_text_tag_table_lookup (table, "warning");
	err_tag = gtk_text_tag_table_lookup (table, "error");

	if (!hand_cursor)
		hand_cursor = gdk_cursor_new (GDK_HAND2);

	if (!ibar_cursor)
		ibar_cursor = gdk_cursor_new (GDK_XTERM);

	if (gtk_text_iter_has_tag (&iter, warn_tag) ||
	    gtk_text_iter_has_tag (&iter, err_tag)) {
		gdk_window_set_cursor (win, hand_cursor);
	} else {
		gdk_window_set_cursor (win, ibar_cursor);
	}

	return FALSE;
}

static gboolean
warning_link_cb (GtkTextTag  *texttag,
		 GObject     *obj,
		 GdkEvent    *event,
		 GtkTextIter *iter,
		 gpointer     data)
{
	GbfBuildInfo *info = GBF_BUILD_INFO (data);
	GtkTextBuffer *buffer;
	GtkTextTagTable *table;
	GtkTextTag *tag;
	GtkTextIter *start_iter;
	int offset;
	GbfBuildWarning *warn;

	if (event->type == GDK_BUTTON_PRESS) {
		start_iter = gtk_text_iter_copy (iter);

		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (obj));
		table = gtk_text_buffer_get_tag_table (buffer);
		tag = gtk_text_tag_table_lookup (table, "warning");

		gtk_text_iter_backward_to_tag_toggle (start_iter, tag);

		offset = gtk_text_iter_get_offset (start_iter);
		warn = g_hash_table_lookup (info->priv->warnings, GINT_TO_POINTER (offset));
		g_assert (warn != NULL);
		g_signal_emit (G_OBJECT (info), info_signals [WARNING_SELECTED],
			       0, warn->filename, warn->line, NULL);
		gtk_text_iter_free (start_iter);
	}

	return FALSE;
}

static gboolean
error_link_cb (GtkTextTag  *texttag,
	       GObject     *obj,
	       GdkEvent    *event,
	       GtkTextIter *iter,
	       gpointer     data)
{
	GbfBuildInfo *info = GBF_BUILD_INFO (data);
	GtkTextBuffer *buffer;
	GtkTextTagTable *table;
	GtkTextTag *tag;
	GtkTextIter *start_iter;
	int offset;
	GbfBuildWarning *err;

	if (event->type == GDK_BUTTON_PRESS) {
		start_iter = gtk_text_iter_copy (iter);

		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (obj));
		table = gtk_text_buffer_get_tag_table (buffer);
		tag = gtk_text_tag_table_lookup (table, "error");

		gtk_text_iter_backward_to_tag_toggle (start_iter, tag);

		offset = gtk_text_iter_get_offset (start_iter);
		err = g_hash_table_lookup (info->priv->errors, GINT_TO_POINTER (offset));
		g_assert (err != NULL);
		g_signal_emit (G_OBJECT (info), info_signals [ERROR_SELECTED],
			       0, err->filename, err->line, NULL);
		gtk_text_iter_free (start_iter);
	}

	return FALSE;
}

/* ----------------------------------------------------------------------
 * Public interface 
 * ---------------------------------------------------------------------- */

GtkWidget *
gbf_build_info_new (void)
{
	return GTK_WIDGET (g_object_new (GBF_TYPE_BUILD_INFO, NULL));
}
