/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gbf_project-tree.c
 *
 * Copyright (C) 2000  JP Rosevear.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <gal/e-table/e-table.h>
#include <gal/e-table/e-tree-scrolled.h>
#include <gal/e-table/e-tree-memory-callbacks.h>
#include <gal/e-table/e-tree-table-adapter.h>
#include <gal/e-table/e-cell-text.h>
#include <gal/e-table/e-cell-tree.h>
#include "images.h"
#include "gbf-target-tree.h"


enum {
	ARG_PROJECT = 1,
};

static GtkVBoxClass *parent_class = NULL;

struct _GbfTargetTreePrivate 
{	
	GbfProjectClient *prj;
	
	GtkWidget *etree_sc;
	GtkWidget *etree;
	
	ETreeModel *etree_model;
	ETreePath etree_root;
	
	GBF_TargetList *targets;
	GHashTable *dirs;
	GBF_Target *current_target;

	Bonobo_EventSource_ListenerId listener_id;

	BonoboUIComponent *uic;	
	BonoboEventSource *event_source;
};

typedef enum {
	GBF_TARGET_TREE_TYPE,
	GBF_TARGET_TREE_PROGRAM,
	GBF_TARGET_TREE_SHAREDLIB,
	GBF_TARGET_TREE_STATICLIB,
	GBF_TARGET_TREE_FILE,
} GbfTargetTreeNodeType;

typedef struct 
{
	GbfTargetTreeNodeType type;
	gpointer data;
} GbfTargetTreeNodeData;

typedef enum {
	GBF_TARGET_TREE_TEXT,
	GBF_TARGET_TREE_CURRENT,
	GBF_TARGET_TREE_NUM_COLUMNS
} GbfTargetTreeCols;

static const char *types[4] = {"Programs", "Shared Libraries", "Static Libraries", NULL};
static const GNOME_Development_TargetType type_num[4] =
              {GNOME_Development_PROGRAM, 
	       GNOME_Development_SHAREDLIB, 
	       GNOME_Development_STATICLIB,
	       0};

#define TARGET_TREE_SPEC  "<ETableSpecification cursor-mode=\"line\" selection-mode=\"browse\"> \
  <ETableColumn model_col=\"0\" _title=\"Targets\" expansion=\"1.0\" minimum_width=\"20\" resizable=\"true\" cell=\"tree-string\" compare=\"string\"/> \
        <ETableState> \
	        <column source=\"0\"/> \
	        <grouping></grouping>                                         \
        </ETableState>
</ETableSpecification>"

static void class_init (GbfTargetTreeClass *klass);
static void init (GbfTargetTree *tree);
static void get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void set_arg (GtkObject *object, GtkArg *arg, guint arg_id);

/* Memory Callbacks */
static GdkPixbuf *gbf_tt_icon_at (ETreeModel *model, ETreePath path, void *data);
static gint gbf_tt_col_count (ETreeModel *model, void *data);
static void *gbf_tt_dup_val (ETreeModel *model, int col, const void *val, void *data);
static void gbf_tt_free_val (ETreeModel *model, int col, void *val, void *data);
static void *gbf_tt_init_val (ETreeModel *model, int col, void *data);
static gboolean gbf_tt_val_is_empty (ETreeModel *model, int col, const void *val, void *data);
static char *gbf_tt_val_to_string (ETreeModel *model, int col, const void *val, void *data);
static void *gbf_tt_val_at (ETreeModel *model, ETreePath path, int col, void *data);
static void gbf_tt_set_val_at (ETreeModel *model, ETreePath path, int col, const void *val, void *data);
static gboolean gbf_tt_is_editable (ETreeModel *model, ETreePath path, int col, void *data);

static void gbf_tt_load_be_data (GbfTargetTree *tree);

static void event_cb (BonoboListener *listener, const char *event,
		      CORBA_any *any, CORBA_Environment *ev,
		      gpointer user_data);
static void double_click_cb (ETree *et, int row, ETreePath path, int col, GdkEvent *event, gpointer data);

GtkType
gbf_target_tree_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "GbfTargetTree",
        sizeof (GbfTargetTree),
        sizeof (GbfTargetTreeClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_vbox_get_type (), &info);
    }

  return type;
}

static void
class_init (GbfTargetTreeClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	gtk_object_add_arg_type ("GbfTargetTree::project",
				 GTK_TYPE_OBJECT,
				 GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
				 ARG_PROJECT);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	object_class->get_arg = get_arg;
	object_class->set_arg = set_arg;

	object_class->destroy = gbf_target_tree_destroy;
}


static void 
init (GbfTargetTree *tree)
{
	GbfTargetTreePrivate *priv;
	ECell *cell;
	ETableExtras *extras;

	priv = g_new0 (GbfTargetTreePrivate, 1);

	tree->priv = priv;

	priv->etree_model = e_tree_memory_callbacks_new (gbf_tt_icon_at,
							 gbf_tt_col_count,
							 NULL,
							 NULL,
							 NULL,
							 NULL,
							 gbf_tt_val_at,
							 gbf_tt_set_val_at,
							 gbf_tt_is_editable,
							 gbf_tt_dup_val,
							 gbf_tt_free_val,
							 gbf_tt_init_val,
							 gbf_tt_val_is_empty,
							 gbf_tt_val_to_string,
							 tree);
	
	e_tree_memory_set_expanded_default (E_TREE_MEMORY (priv->etree_model), FALSE);

	/* Bold the current target */
	extras = e_table_extras_new();
	cell = e_cell_text_new (NULL, GTK_JUSTIFY_LEFT);

	gtk_object_set (GTK_OBJECT (cell), "bold_column", 
			GBF_TARGET_TREE_CURRENT,
			NULL);

	cell = e_cell_tree_new (NULL, NULL, TRUE, cell);

	e_table_extras_add_cell (extras, "tree-string", cell);

	priv->etree_sc = e_tree_scrolled_new (priv->etree_model, extras, TARGET_TREE_SPEC, NULL);
	priv->etree = GTK_WIDGET (e_tree_scrolled_get_tree (E_TREE_SCROLLED (priv->etree_sc)));

	gtk_object_sink (GTK_OBJECT (extras));

	gtk_signal_connect (GTK_OBJECT (priv->etree), "double_click",
			    GTK_SIGNAL_FUNC (double_click_cb), tree);

	priv->dirs = g_hash_table_new (g_str_hash, g_str_equal);

        gtk_box_pack_start (GTK_BOX (tree), priv->etree_sc, TRUE, TRUE, 5);
	gtk_widget_show (priv->etree_sc);

	priv->event_source = bonobo_event_source_new ();
	priv->etree_root = NULL;
}

void 
get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
    GbfTargetTree *tree = GBF_TARGET_TREE (object);
    
    switch (arg_id) {
    case ARG_PROJECT:
	    GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (gbf_target_tree_get_project (tree));
	    break;
    default :
        arg->type = GTK_TYPE_INVALID;
    }   
}

void 
set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
    GbfTargetTree *tree = GBF_TARGET_TREE (object);
    
    switch (arg_id) {
    case ARG_PROJECT:
        gbf_target_tree_set_project (tree, GTK_VALUE_OBJECT (*arg) ? GBF_PROJECT_CLIENT (GTK_VALUE_OBJECT (*arg)) : NULL);
        break;
    default :
        break;
    }   
}




GtkWidget *
gbf_target_tree_new      (void)
{
	return GTK_WIDGET (gtk_type_new (GBF_TYPE_TARGET_TREE));
}

static GbfTargetTreeNodeData *
gbf_tt_node_data_new (GbfTargetTreeNodeType type, gpointer data) 
{
	GbfTargetTreeNodeData *node_data = g_new0 (GbfTargetTreeNodeData, 1);

	node_data->type = type;
	node_data->data = data;

	return node_data;
}

static ETreePath
gbf_tt_find_parent (GbfTargetTree *tree, GNOME_Development_TargetType type) 
{
	GbfTargetTreePrivate *priv;
	int i;
	
	priv = tree->priv;

	for (i=0; types[i] != NULL; i++) {
		if (type == type_num[i])
			return g_hash_table_lookup (priv->dirs, types[i]);
	}

	return NULL;
}

static gboolean
gbf_tt_destroy_node (gpointer key, gpointer value, gpointer user_data) 
{
	g_free (key);
	
	return TRUE;
}

static void
gbf_tt_load_be_data (GbfTargetTree *tree)
{
	GbfTargetTreePrivate *priv;
	GbfProjectClientResult res;
	GbfTargetTreeNodeData *node_data;
	CORBA_char *tmp, *tmp2;
	int i, j;
	
	priv = tree->priv;

	e_tree_memory_freeze (E_TREE_MEMORY (priv->etree_model));
	
	if (priv->etree_root) {
		e_tree_memory_node_remove (E_TREE_MEMORY (priv->etree_model), priv->etree_root);
		priv->etree_root = NULL;
		g_hash_table_foreach_remove (priv->dirs, gbf_tt_destroy_node, NULL);
	}

	if (!priv->prj)
		goto cleanup;

	res = gbf_project_client_get_project_root (priv->prj, &tmp);	
	if (res != GBF_PROJECT_CLIENT_OK)
		goto cleanup;

	tmp2 = strrchr (tmp, G_DIR_SEPARATOR);
	if (tmp2 == NULL) 
		tmp2 = tmp;
	else
		tmp2++;

	node_data = gbf_tt_node_data_new (GBF_TARGET_TREE_TYPE, g_strdup (tmp2));
	priv->etree_root = e_tree_memory_node_insert (E_TREE_MEMORY (priv->etree_model), NULL, 0, node_data);
	g_hash_table_insert (priv->dirs, g_strdup (""), priv->etree_root);
	CORBA_free (tmp);

	for (i = 0; types[i] != NULL; i++) {
		ETreePath etp;
		
		node_data = gbf_tt_node_data_new (GBF_TARGET_TREE_TYPE, g_strdup (types[i]));
		etp = e_tree_memory_node_insert (E_TREE_MEMORY (priv->etree_model), priv->etree_root, 0, node_data);
		g_hash_table_insert (priv->dirs, g_strdup (types[i]), etp);
	}

	res = gbf_project_client_get_targets (priv->prj, &priv->targets);
	if (res != GBF_PROJECT_CLIENT_OK)
		goto cleanup;

	for (i = 0; i < priv->targets->_length; i++) {
		GBF_Target *target = &priv->targets->_buffer[i];
		ETreePath etp, parent = gbf_tt_find_parent (tree, target->type);
		GBF_SourceList *sources;
		
		switch (target->type) {
		case GNOME_Development_SHAREDLIB :
			node_data = gbf_tt_node_data_new (GBF_TARGET_TREE_SHAREDLIB, target);
			break;
		case GNOME_Development_STATICLIB :
			node_data = gbf_tt_node_data_new (GBF_TARGET_TREE_STATICLIB, target);
			break;
		case GNOME_Development_PROGRAM :
			node_data = gbf_tt_node_data_new (GBF_TARGET_TREE_PROGRAM, target);
			break;			
		default :
		}
			
		etp = e_tree_memory_node_insert (E_TREE_MEMORY (priv->etree_model), parent, -1, node_data);

		res = gbf_project_client_get_target_sources (priv->prj, target, &sources);
		if (res != GBF_PROJECT_CLIENT_OK)
			goto cleanup;

		for (j = 0; j < sources->_length; j++) {
			GBF_Source *source = &sources->_buffer[j];

			node_data = gbf_tt_node_data_new (GBF_TARGET_TREE_FILE, source);
			e_tree_memory_node_insert (E_TREE_MEMORY (priv->etree_model), etp, -1, node_data);
		}
	}

 cleanup:
	e_tree_memory_thaw (E_TREE_MEMORY (priv->etree_model));
}

static gint 
gbf_tt_col_count (ETreeModel *model, void *data)
{
	return GBF_TARGET_TREE_NUM_COLUMNS;
}

static void *
gbf_tt_dup_val (ETreeModel *model, int col, const void *val, void *data)
{
	void *new_val = NULL;
	
	g_print ("dup_val\n");

	switch (col) {
	case 0:
		new_val = g_strdup ((char *)val);
		break;
	}

	return new_val;
}

static void 
gbf_tt_free_val (ETreeModel *model, int col, void *val, void *data)
{
	g_print ("free_val\n");

	switch (col) {
	case 0:
		g_free (val);
		break;
	}	
}

static void *
gbf_tt_init_val (ETreeModel *model, int col, void *data)
{
	void *init_val = NULL;

	g_print ("init_val %d\n", col);
	
	switch (col) {
	case GBF_TARGET_TREE_TEXT:
		init_val = NULL;
		break;
	case GBF_TARGET_TREE_CURRENT:
		break;
	}
	
	return init_val;
}

static gboolean
gbf_tt_val_is_empty (ETreeModel *model, int col, const void *val, void *data)
{
	g_print ("val_is_empty %d\n", col);

	switch (col) {
	case GBF_TARGET_TREE_TEXT:
		return val == NULL;
		break;
	case GBF_TARGET_TREE_CURRENT:
		break;
	}

	return FALSE;
}

static char *
gbf_tt_val_to_string (ETreeModel *model, int col, const void *val, void *data)
{
	switch (col) {
	case GBF_TARGET_TREE_TEXT:
		return g_strdup (val);
		break;
	case GBF_TARGET_TREE_CURRENT:
		break;
	}

	return NULL;
}

static GdkPixbuf *
gbf_tt_icon_at  (ETreeModel *model, ETreePath path, void *data)
{
	GbfTargetTreeNodeData *node_data = e_tree_memory_node_get_data (E_TREE_MEMORY (model), path);
	
	switch (node_data->type) {
	case GBF_TARGET_TREE_FILE:
		return image_for_filename (((GBF_Source*)node_data->data)->name);
	case GBF_TARGET_TREE_PROGRAM:
		return image_for_type ("program");
	case GBF_TARGET_TREE_SHAREDLIB :
		return image_for_type ("shared");
	case GBF_TARGET_TREE_STATICLIB :
		return image_for_type ("static");
	case GBF_TARGET_TREE_TYPE:
		return image_for_folder ();
		break;
	}

	return NULL;
}

static void *
gbf_tt_val_at  (ETreeModel *model, ETreePath path, int col, void *data)
{
	GbfTargetTree *tree = data;
	GbfTargetTreeNodeData *node_data = e_tree_memory_node_get_data (E_TREE_MEMORY (model), path);
	GBF_Target *target = NULL;
	GBF_Source *source = NULL;
	void *val_at = NULL;

	g_print ("Val at col %d\n", col);
	
	if (col == GBF_TARGET_TREE_CURRENT) {
		target = node_data->data;

		if (target == tree->priv->current_target)
			return GINT_TO_POINTER (TRUE);

		return GINT_TO_POINTER (FALSE);
	}
		
	switch (node_data->type) {
	case GBF_TARGET_TREE_FILE:
		source = node_data->data;
		val_at = g_strdup (source->name);
		break;
	case GBF_TARGET_TREE_PROGRAM:
	case GBF_TARGET_TREE_SHAREDLIB:
	case GBF_TARGET_TREE_STATICLIB:
		target = node_data->data;
		val_at = g_strdup (target->name);
		break;
	case GBF_TARGET_TREE_TYPE:
		val_at = g_strdup (node_data->data);
		break;
	}
	
	return val_at;
}

static void 
gbf_tt_set_val_at (ETreeModel *model, ETreePath path, int col, const void *val, void *data)
{
	switch (col) {
	case GBF_TARGET_TREE_TEXT:
		break;
	case GBF_TARGET_TREE_CURRENT:
		break;
	}
}

static gboolean 
gbf_tt_is_editable (ETreeModel *model, ETreePath path, int col, void *data)
{
	gboolean editable = FALSE;
	
	switch (col) {
	case GBF_TARGET_TREE_TEXT:
		editable = FALSE;
		break;
	case GBF_TARGET_TREE_CURRENT:
		editable = FALSE;
		break;
	}

	return editable;
}

GbfProjectClient *
gbf_target_tree_get_project (GbfTargetTree *tree)
{
	g_return_val_if_fail (tree != NULL, NULL);
	g_return_val_if_fail (GBF_IS_TARGET_TREE (tree), NULL);

	return tree->priv->prj;
}

void 
gbf_target_tree_set_project (GbfTargetTree *tree, GbfProjectClient *prj)
{
	GbfTargetTreePrivate *priv;
	BonoboListener *listener;
	CORBA_Object source;
	CORBA_Environment ev;
	
	g_return_if_fail (tree != NULL);
	g_return_if_fail (GBF_IS_TARGET_TREE (tree));

	priv = tree->priv;

	CORBA_exception_init (&ev);
	if (priv->prj) {
		source = Bonobo_Unknown_queryInterface (priv->prj->objref, 
							"IDL:Bonobo/EventSource:1.0", 
							&ev);
		Bonobo_EventSource_removeListener (source, priv->listener_id, &ev);
		gtk_object_unref (GTK_OBJECT (priv->prj));
	}

	priv->prj = prj;

	if (prj) {
		gtk_object_ref (GTK_OBJECT (prj));
		if (GTK_OBJECT_FLOATING (GTK_OBJECT (prj)))
			gtk_object_sink (GTK_OBJECT (prj));

		listener = bonobo_listener_new (NULL, NULL);
		gtk_signal_connect (GTK_OBJECT (listener), "event_notify",
				    GTK_SIGNAL_FUNC (event_cb), tree);

		source = Bonobo_Unknown_queryInterface (prj->objref, "IDL:Bonobo/EventSource:1.0", &ev);

		if (!CORBA_Object_is_nil (source, &ev) && ev._major == CORBA_NO_EXCEPTION) {
			priv->listener_id = Bonobo_EventSource_addListener (source, BONOBO_OBJREF (listener), &ev);
		} else {
			g_error ("couldn't get event source for project");
		}

	}

	gbf_tt_load_be_data (tree);

	CORBA_exception_free (&ev);
}

GBF_Target *
gbf_target_tree_get_current_target (GbfTargetTree *tree) 
{
	g_return_val_if_fail (tree != NULL, NULL);
	g_return_val_if_fail (GBF_IS_TARGET_TREE (tree), NULL);

	return tree->priv->current_target;
}

void
gbf_target_tree_set_current_target (GbfTargetTree *tree, GBF_Target *new_target)
{
	GbfTargetTreePrivate *priv;
	int i;
	
	g_return_if_fail (tree != NULL);
	g_return_if_fail (GBF_IS_TARGET_TREE (tree));

	priv = tree->priv;
	
	for (i = 0; i < priv->targets->_length; i++) {
		GBF_Target *target = &priv->targets->_buffer[i];

		if (target->type == new_target->type 
		    && !strcmp (target->name, new_target->name)
		    && !strcmp (target->path, new_target->path)) {
			priv->current_target = target;
		}
	}
}

BonoboEventSource *
gbf_target_tree_get_event_source (GbfTargetTree *tree)
{
	return tree->priv->event_source;
}

void
gbf_target_tree_destroy (GtkObject *obj) 
{
	GTK_OBJECT_CLASS (parent_class)->destroy (obj);
}

static void
event_cb (BonoboListener *listener, const char *event,
	  CORBA_any *any, CORBA_Environment *ev,
	  gpointer user_data)
{
	GbfTargetTree *tree = GBF_TARGET_TREE (user_data);
	GbfTargetTreePrivate *priv;
	
	priv = tree->priv;

	g_print ("Event: %s\n", event);
	if (!strcmp (event, "project-changed"))
		gbf_tt_load_be_data (tree);
}

void 
double_click_cb (ETree *et, int row, ETreePath path, int col, GdkEvent *event, gpointer data)
{
	GbfTargetTree *tree = GBF_TARGET_TREE (data);
	GbfTargetTreeNodeData *node_data;
	GBF_Source *source = NULL;
	GBF_Target *target = NULL;
	BonoboArg *arg;

	node_data = e_tree_memory_node_get_data (E_TREE_MEMORY (tree->priv->etree_model), path);

	switch (node_data->type) {
	case GBF_TARGET_TREE_FILE:
		arg = bonobo_arg_new (BONOBO_ARG_STRING);
		source = node_data->data;
		arg->_type = TC_GNOME_Development_Source;
		arg->_value = source;
		CORBA_any_set_release (arg, FALSE);
		
		bonobo_event_source_notify_listeners (tree->priv->event_source,
						      "file-selected",
						      arg, NULL);
		bonobo_arg_release (arg);
		break;
	case GBF_TARGET_TREE_PROGRAM:
	case GBF_TARGET_TREE_SHAREDLIB:
	case GBF_TARGET_TREE_STATICLIB:
		arg = bonobo_arg_new (BONOBO_ARG_STRING);
		target = node_data->data;
		arg->_type = TC_GNOME_Development_Target;
		arg->_value = target;
		CORBA_any_set_release (arg, FALSE);

		gbf_target_tree_set_current_target (tree, target);
		
		bonobo_event_source_notify_listeners (tree->priv->event_source,
						      "target-selected",
						      arg, NULL);
		bonobo_arg_release (arg);
		break;
	default:
		break;
	}
	
}

