/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gbf-build-info.c
 *
 * Copyright (C) 2001  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <gtkhtml/gtkhtml.h>

#include "gbf-build-info.h"

enum {
	ARG_PROJECT = 1,
};

static GtkVBoxClass *parent_class = NULL;

struct _GbfBuildInfoPrivate
{
	GbfProjectClient *prj;

	GtkWidget *html;
	GtkHTMLStream *html_stream;

	Bonobo_EventSource_ListenerId listener_id;

	BonoboEventSource *event_source;
};

static void class_init (GbfBuildInfoClass *klass);
static void init (GbfBuildInfo *info);
static void get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void set_arg (GtkObject *object, GtkArg *arg, guint arg_id);

static void event_cb (BonoboListener *listener, const char *event,
		      CORBA_any *any, CORBA_Environment *ev,
		      gpointer user_data);
static void link_clicked_cb (GtkHTML *html, const gchar *url, gpointer user_data);

GtkType
gbf_build_info_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "GbfBuildInfo",
        sizeof (GbfBuildInfo),
        sizeof (GbfBuildInfoClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_vbox_get_type (), &info);
    }

  return type;
}

static void
class_init (GbfBuildInfoClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	gtk_object_add_arg_type ("GbfBuildInfo::project",
				 GTK_TYPE_OBJECT,
				 GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
				 ARG_PROJECT);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	object_class->get_arg = get_arg;
	object_class->set_arg = set_arg;

	object_class->destroy = gbf_build_info_destroy;
}


static void
init (GbfBuildInfo *info)
{
	GbfBuildInfoPrivate *priv;
	GtkWidget *scrolled_window;
	char *html;
	
	priv = g_new0 (GbfBuildInfoPrivate, 1);

	info->priv = priv;

	priv->html = gtk_html_new ();

	priv->html_stream = gtk_html_begin (GTK_HTML (priv->html));
        html = g_strdup(
            "<html><body bgcolor=#ffffff text=#000000>"
	    "No build in progress"
            "</body></html>");
        gtk_html_write (GTK_HTML (priv->html), priv->html_stream, html, strlen(html));
	g_free (html);
	gtk_html_end (GTK_HTML (priv->html), priv->html_stream, GTK_HTML_STREAM_OK);

	gtk_signal_connect (GTK_OBJECT (priv->html), "link_clicked",
			    GTK_SIGNAL_FUNC (link_clicked_cb), info);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);

	gtk_container_add (GTK_CONTAINER (scrolled_window), priv->html);
	gtk_container_add (GTK_CONTAINER (info), scrolled_window);
	
	priv->event_source = bonobo_event_source_new ();
}

void 
get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
    GbfBuildInfo *info = GBF_BUILD_INFO (object);
    
    switch (arg_id) {
    case ARG_PROJECT:
	    GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (gbf_build_info_get_project (info));
	    break;
    default :
        arg->type = GTK_TYPE_INVALID;
    }   
}

void 
set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
    GbfBuildInfo *info = GBF_BUILD_INFO (object);
    
    switch (arg_id) {
    case ARG_PROJECT:
        gbf_build_info_set_project (info, GTK_VALUE_OBJECT (*arg) ? GBF_PROJECT_CLIENT (GTK_VALUE_OBJECT (*arg)) : NULL);
        break;
    default :
        break;
    }   
}




GtkWidget *
gbf_build_info_new (void)
{
	GtkWidget *widget;
	
	widget = gtk_type_new (GBF_TYPE_BUILD_INFO);

	return widget;
}

GbfProjectClient *
gbf_build_info_get_project (GbfBuildInfo *info)
{
	g_return_val_if_fail (info != NULL, NULL);
	g_return_val_if_fail (GBF_IS_BUILD_INFO (info), NULL);

	return info->priv->prj;
}

void 
gbf_build_info_set_project (GbfBuildInfo *info, GbfProjectClient *prj)
{
	GbfBuildInfoPrivate *priv;
	BonoboListener *listener;
	CORBA_Object source;
	CORBA_Environment ev;

	g_return_if_fail (info != NULL);
	g_return_if_fail (GBF_IS_BUILD_INFO (info));

	priv = info->priv;

	CORBA_exception_init (&ev);
	if (priv->prj) {
		source = Bonobo_Unknown_queryInterface (priv->prj->objref, 
							"IDL:Bonobo/EventSource:1.0", 
							&ev);
		Bonobo_EventSource_removeListener (source, priv->listener_id, &ev);
		gtk_object_unref (GTK_OBJECT (priv->prj));
	}
    
	priv->prj = prj;

	if (prj) {
		gtk_object_ref (GTK_OBJECT (prj));
		if (GTK_OBJECT_FLOATING (GTK_OBJECT (prj)))
			gtk_object_sink (GTK_OBJECT (prj));

		listener = bonobo_listener_new (NULL, NULL);
		gtk_signal_connect (GTK_OBJECT (listener), "event_notify",
				    GTK_SIGNAL_FUNC (event_cb), info);

		source = Bonobo_Unknown_queryInterface (prj->objref, "IDL:Bonobo/EventSource:1.0", &ev);

		if (!CORBA_Object_is_nil (source, &ev) && ev._major == CORBA_NO_EXCEPTION) {
			priv->listener_id = Bonobo_EventSource_addListener (source, BONOBO_OBJREF (listener), &ev);
		} else {
			g_error ("couldn't get event source for project");
		}

	}
	CORBA_exception_free (&ev);
}

BonoboEventSource *
gbf_build_info_get_event_source (GbfBuildInfo *info)
{
	return info->priv->event_source;
}

static void
gbi_build_start_msg (GbfBuildInfo *info) 
{
	GbfBuildInfoPrivate *priv;
	char *html;
	
	priv = info->priv;
	
	priv->html_stream = gtk_html_begin (GTK_HTML (priv->html));

        html = g_strdup(
            "<html><body bgcolor=#ffffff text=#000000>"
	    "<p>Building...</p>");
        gtk_html_write (GTK_HTML (priv->html), priv->html_stream, html, strlen(html));
	g_free (html);
}

static void
gbi_build_file_msg (GbfBuildInfo *info, CORBA_any *any)
{
	GbfBuildInfoPrivate *priv;
	GBF_Source *source;
	char *html;
	
	priv = info->priv;

	source = any->_value;

        html = g_strdup_printf ("<p>Compiling <a href=\"%s/%s\">%s</a></p>",
				source->path, source->name, source->name);
        gtk_html_write (GTK_HTML (priv->html), priv->html_stream, html, strlen(html));
	g_free (html);
}

static void
gbi_build_error_msg (GbfBuildInfo *info, CORBA_any *any)
{
	GbfBuildInfoPrivate *priv;
	GBF_BuildError *build_err;
	gboolean warn;
	char *html;
	
	priv = info->priv;

	build_err = any->_value;
	warn = 	build_err->type == GNOME_Development_WARNING;

        html = g_strdup_printf ("<p>Error "
				"<a href=\"%s/%s?%d\">%s: %d%s</a> %s</p>",
				build_err->source.path,
				build_err->source.name,
				build_err->line,
				build_err->source.name,
				build_err->line,
				warn ? " (warning)" : "",
				build_err->error);
        gtk_html_write (GTK_HTML (priv->html), priv->html_stream, html, strlen(html));
	g_free (html);
}

static void
gbi_build_end_msg (GbfBuildInfo *info) 
{
	GbfBuildInfoPrivate *priv;
	char *html;
	
	priv = info->priv;

        html = g_strdup(
	    "<p>Finished.</p>"
            "</body></html>");
        gtk_html_write (GTK_HTML (priv->html), priv->html_stream, html, strlen(html));
	g_free (html);

	gtk_html_end (GTK_HTML (priv->html), priv->html_stream, GTK_HTML_STREAM_OK);
}

static void
event_cb (BonoboListener *listener, const char *event,
	  CORBA_any *any, CORBA_Environment *ev,
	  gpointer user_data)
{
	GbfBuildInfo *info = GBF_BUILD_INFO (user_data);
	GbfBuildInfoPrivate *priv;
	
	priv = info->priv;

	g_print ("Event: %s\n", event);
	if (!strcmp (event, "build-started")) {
		gbi_build_start_msg (info);
	} else if (!strcmp (event, "build-ended")) {
		gbi_build_end_msg (info);
	} else if (!strcmp (event, "build-file")) {
		gbi_build_file_msg (info, any);
	} else if (!strcmp (event, "build-error")) {
		gbi_build_error_msg (info, any);
	}

}

static void
name_path_split (const char *full, GBF_Source *source) 
{
	char *d;
	
	source->name = g_strdup (g_basename (full));
	d = g_dirname (full);
	if (strcmp (d, "."))
		source->path = CORBA_string_dup (d);
	else
		source->path = CORBA_string_dup ("");
	g_free (d);
}

static void
link_clicked_cb (GtkHTML *html, const gchar *url, gpointer user_data)
{
	GbfBuildInfo *info = GBF_BUILD_INFO (user_data);
	GbfBuildInfoPrivate *priv;
	GBF_Source source;
	GBF_BuildError error;
	BonoboArg *arg;
	const char *type;
	
	priv = info->priv;
	
	g_print ("Link clicked: %s\n", url);

	if (strchr (url, '?')) {
		char *end, *full = g_strdup (url);
		
		end = strrchr (full, '?');
		*end = '\0';
		end++;
		
		arg = bonobo_arg_new (BONOBO_ARG_NULL);
		name_path_split (full, &error.source);
		error.line = strtol (end, NULL, 0);
		arg->_type = TC_GNOME_Development_BuildError;
		arg->_value = &error;
		type = "error-selected";
		g_free (full);
	} else {
		arg = bonobo_arg_new (BONOBO_ARG_STRING);
		name_path_split (url, &source);
		arg->_type = TC_GNOME_Development_Source;
		arg->_value = &source;
		type = "file-selected";
	}
	CORBA_any_set_release (arg, FALSE);
	
	bonobo_event_source_notify_listeners (priv->event_source,
					      type, arg, NULL);
	bonobo_arg_release (arg);
}

void
gbf_build_info_destroy (GtkObject *obj) 
{
	GTK_OBJECT_CLASS (parent_class)->destroy (obj);
}
