/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Build Framework.
 * 
 * Copyright (C) 2000 JP Rosevear <jpr@helixcode.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <config.h>
#include <bonobo.h>
#include <liboaf/liboaf.h>

#include <gdl/gdl.h>
#include "gbf-project-tree.h"
#include "gbf-target-tree.h"
#include "gbf-build-info.h"
#include "commands.h"
#include "menus.h"
#include "../lib/gbf.h"

enum {
    PROP_PROJECT_IOR = 1,
};

static BonoboUIVerb pt_verbs [] = {
	BONOBO_UI_UNSAFE_VERB ("BuildAll", build_all_cmd), 
	BONOBO_UI_UNSAFE_VERB ("BuildInstall", build_install_cmd),
	BONOBO_UI_UNSAFE_VERB ("BuildRun", build_run_cmd),
	BONOBO_UI_UNSAFE_VERB ("BuildParameters", build_params_cmd),

	BONOBO_UI_VERB_END
};

static BonoboUIVerb tt_verbs [] = {
	BONOBO_UI_UNSAFE_VERB ("BuildAll", build_all_cmd), 
	BONOBO_UI_UNSAFE_VERB ("BuildInstall", build_install_cmd),
	BONOBO_UI_UNSAFE_VERB ("BuildRun", build_run_cmd),
	BONOBO_UI_UNSAFE_VERB ("BuildParameters", build_params_cmd),

	BONOBO_UI_VERB_END
};

static GbfProjectClient *
get_project (gchar *ior)
{
	CORBA_Object objref;
	GbfProjectClient *prj;
	CORBA_Environment ev;

    static GHashTable *active_projects = NULL;
	
    if (ior[0] == '\0')
        return NULL;

	if (!active_projects) 
		active_projects = g_hash_table_new (g_str_hash, g_str_equal);

	prj = (GbfProjectClient *)g_hash_table_lookup (active_projects, ior);
	
	if (!prj) {
		CORBA_exception_init (&ev);
		objref = CORBA_ORB_string_to_object (oaf_orb_get (), ior, &ev);
		/* FIXME: Check for exceptions */
		CORBA_exception_free (&ev);
	
		prj = gbf_project_client_new_from_corba (objref);
		g_hash_table_insert (active_projects, (gpointer)ior, (gpointer)prj);
	} else {
        gtk_object_ref (GTK_OBJECT (prj));
    }
	
	return prj;
}

static void
get_prop (BonoboPropertyBag *bag,
          BonoboArg *arg,
          guint arg_id,
	  CORBA_Environment *ev,
          gpointer user_data)
{
    GtkObject *obj = user_data;
    
    switch (arg_id) {
    case PROP_PROJECT_IOR:
    {
        GbfProjectClient *prj;
        char *ior;
        GtkArg args[1];
        args[0].name = "project";
        gtk_object_getv (obj, 1, args);
        g_assert (args[0].type == GTK_TYPE_OBJECT);
        if (GTK_VALUE_OBJECT (args[0]))
            prj = GBF_PROJECT_CLIENT (GTK_VALUE_OBJECT (args[0]));
        else 
            prj = NULL;
        
        if (prj) {
            CORBA_Environment ev;
            CORBA_exception_init (&ev);
            ior = CORBA_ORB_object_to_string (oaf_orb_get (), prj->objref, &ev);
            CORBA_exception_free (&ev);
        } else {
            ior = CORBA_string_dup ("");
        }
        
        BONOBO_ARG_SET_STRING (arg, ior);
        CORBA_free (ior);
        break;
    }
    }
}

static void
set_prop (BonoboPropertyBag *bag,
          const BonoboArg *arg,
          guint arg_id,
          CORBA_Environment *ev,
          gpointer user_data)
{
    GtkObject *obj = user_data;

    switch (arg_id) {
    case PROP_PROJECT_IOR :
    {
        GbfProjectClient *prj;
        char *s;
        s = BONOBO_ARG_GET_STRING (arg);
        
        if (strcmp (s, ""))
            prj = get_project (s);
        else
            prj = NULL;
        
        gtk_object_set (obj, "project", GTK_OBJECT (prj), NULL);

        if (!GBF_IS_BUILD_INFO (obj))
            menus_sensitize (obj);
        
        if (prj)
            gtk_object_unref (GTK_OBJECT (prj));
        break;
    }
    }
}

static void
project_tree_control_set_frame_cb (BonoboControl *control, gpointer data)
{
    menus_add (control, data, "gbf-project-tree", pt_verbs);
}

static BonoboObject *
project_tree_factory (BonoboGenericFactory *fact,
            void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	BonoboEventSource *event_source;
    GtkWidget *pt;

	/* Create the control */
	pt = GTK_WIDGET (gbf_project_tree_new ());
	gtk_widget_show (GTK_WIDGET (pt));
	gdl_server_manager_register_object (GTK_OBJECT (pt));
	
	control = bonobo_control_new (GTK_WIDGET (pt));

	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, pt);
	bonobo_control_set_properties (control, pb);
	bonobo_property_bag_add (pb, "project-ior", PROP_PROJECT_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the project backend", 0);

    /* Listen to insert menus */
	gtk_signal_connect (GTK_OBJECT (control), "set_frame",
			    GTK_SIGNAL_FUNC (project_tree_control_set_frame_cb), pt);

    event_source = gbf_project_tree_get_event_source (GBF_PROJECT_TREE (pt));
    bonobo_object_add_interface (BONOBO_OBJECT (control), 
                                 BONOBO_OBJECT (event_source));

	return BONOBO_OBJECT (control);
}

static void
target_tree_control_set_frame_cb (BonoboControl *control, gpointer data)
{
    menus_add (control, data, "gbf-target-tree", tt_verbs);
}

static BonoboObject *
target_tree_factory (BonoboGenericFactory *fact,
                     void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	BonoboEventSource *event_source;
    GtkWidget *tt;

	/* Create the control */
	tt = GTK_WIDGET (gbf_target_tree_new ());
	gtk_widget_show (GTK_WIDGET (tt));
	gdl_server_manager_register_object (GTK_OBJECT (tt));
	
	control = bonobo_control_new (GTK_WIDGET (tt));

	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, tt);
	bonobo_control_set_properties (control, pb);
	bonobo_property_bag_add (pb, "project-ior", PROP_PROJECT_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the project backend", 0);

    /* Listen to insert menus */
	gtk_signal_connect (GTK_OBJECT (control), "set_frame",
                GTK_SIGNAL_FUNC (target_tree_control_set_frame_cb), tt);
    

    event_source = gbf_target_tree_get_event_source (GBF_TARGET_TREE (tt));
    bonobo_object_add_interface (BONOBO_OBJECT (control), 
                                 BONOBO_OBJECT (event_source));

	return BONOBO_OBJECT (control);
}

static BonoboObject *
build_info_factory (BonoboGenericFactory *fact,
                      void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	BonoboEventSource *event_source;
    GtkWidget *bd;

	/* Create the control */
	bd = gbf_build_info_new ();
	gtk_widget_show_all (bd);
	gdl_server_manager_register_object (GTK_OBJECT (bd));
	
	control = bonobo_control_new (bd);

	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, bd);
	bonobo_control_set_properties (control, pb);
	bonobo_property_bag_add (pb, "project-ior", PROP_PROJECT_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the project backend", 0);

    event_source = gbf_build_info_get_event_source (GBF_BUILD_INFO (bd));
    bonobo_object_add_interface (BONOBO_OBJECT (control), 
                                 BONOBO_OBJECT (event_source));

	return BONOBO_OBJECT (control);
}

static BonoboObject *
control_factory (BonoboGenericFactory *factory, 
                 const char *component_id,
                 gpointer closure)
{
    if (!strcmp (component_id, "OAFIID:GNOME_Development_ProjectTree"))
        return project_tree_factory (factory, closure);
    if (!strcmp (component_id, "OAFIID:GNOME_Development_TargetTree"))
        return target_tree_factory (factory, closure);
    if (!strcmp (component_id, "OAFIID:GNOME_Development_BuildInfo"))
        return build_info_factory (factory, closure);
    return NULL;
}

BONOBO_OAF_FACTORY_MULTI ("OAFIID:GNOME_Development_BuildControlFactory",
                          "Factory for the gnome-build controls.",
                          VERSION, 
                          control_factory, NULL);

