/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gbf-am-project.c
 *
 * Copyright (C) 2000  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <bonobo/bonobo-event-source.h>
#include <fcntl.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include "../../lib/gnome-build.h"
#include "gbf-am-project.h"

#define SCRIPT_NAME "gbf-am-parse"
#define BUILD_SCRIPT_NAME "gbf-am-build"

typedef GNOME_Development_Project    GBF_Project;
typedef GNOME_Development_Source     GBF_Source;
typedef GNOME_Development_SourceList GBF_SourceList;
typedef GNOME_Development_Target     GBF_Target;
typedef GNOME_Development_TargetList GBF_TargetList;
typedef GNOME_Development_BuildError GBF_BuildError;

#define GBF_ex_DoesntExist ex_GNOME_Development_Project_DoesntExist
#define GBF_ex_Malformed   ex_GNOME_Development_Project_Malformed

typedef enum {
	BUILD
} GbfAmProjectOpType;

typedef struct 
{
	GbfAmProject *prj;
	GbfAmProjectOpType type;

	GNOME_Development_BuildType build_type;
} GbfAmProjectOp;

struct _GbfAmProjectPrivate {
	char *project_dir;
	
	GList *sources;
	GList *targets;

	char *buffer;
	
	GList *ops;
        gint op_handler;

	BonoboEventSource *event_source;
};

typedef struct 
{
	char *name;
	char *type;
	GList *sources;
} GbfAmProjectTargetData;

static BonoboObjectClass *parent_class = NULL;
static POA_GNOME_Development_Project__epv am_project_epv;
static POA_GNOME_Development_Project__vepv am_project_vepv;

static void impl_load_project (PortableServer_Servant servant, 
			       const CORBA_char *file_name, 
			       CORBA_Environment *ev);
static CORBA_char* impl_get_project_root (PortableServer_Servant servant, 
					  CORBA_Environment *ev);

static void build_project_op (GbfAmProject *prj, 
			      GNOME_Development_BuildType type);
static void impl_build_project (PortableServer_Servant servant, 
				GNOME_Development_BuildType type, 
				CORBA_Environment *ev);

static GBF_TargetList *impl_get_targets (PortableServer_Servant servant, 
					 CORBA_Environment *ev);
static GBF_TargetList * impl_get_targets_of_source (PortableServer_Servant servant,
						    const GBF_Source *source,
						    CORBA_Environment *ev);

static void impl_add_target (PortableServer_Servant servant, 
			     const GBF_Target *target, 
			     CORBA_Environment *ev);
static void impl_remove_target (PortableServer_Servant servant, 
				const GBF_Target *target, 
				CORBA_Environment *ev);

static GBF_SourceList *impl_get_sources (PortableServer_Servant servant, 
					 CORBA_Environment *ev);
static GBF_SourceList *impl_get_target_sources (PortableServer_Servant servant, 
						const GBF_Target *target, 
						CORBA_Environment *ev);

static void impl_add_target_source (PortableServer_Servant servant, 
				    const GBF_Target *target,
				    const GBF_Source *source, 
				    CORBA_Environment *ev);
static void impl_remove_target_source (PortableServer_Servant servant, 
				       const GBF_Target *target, 
				       const GBF_Source *source, 
				       CORBA_Environment *ev);

static void
init_corba_class (void) 
{
	/* EPV */
	am_project_epv.load_project = impl_load_project;
	am_project_epv.build_project = impl_build_project;
	am_project_epv.get_project_root = impl_get_project_root;
	am_project_epv.get_targets = impl_get_targets;
	am_project_epv.get_targets_of_source = impl_get_targets_of_source;
	am_project_epv.add_target = impl_add_target;
	am_project_epv.remove_target = impl_remove_target;
	am_project_epv.get_sources = impl_get_sources;
	am_project_epv.get_target_sources = impl_get_target_sources;
	am_project_epv.add_target_source = impl_add_target_source;
	am_project_epv.remove_target_source = impl_remove_target_source;
   
	/* VEPV */
	am_project_vepv.Bonobo_Unknown_epv = bonobo_object_get_epv ();
	am_project_vepv.GNOME_Development_Project_epv = &am_project_epv;
}

static CORBA_Object
init_corba_object (BonoboObject *object) 
{
    POA_GNOME_Development_Project *servant;
    CORBA_Environment ev;
    CORBA_exception_init (&ev);
    
    servant = (POA_GNOME_Development_Project*)g_new0 (BonoboObjectServant, 1);
    servant->vepv = &am_project_vepv;
    
    POA_GNOME_Development_Project__init ((PortableServer_Servant) servant, &ev);
    if (ev._major != CORBA_NO_EXCEPTION) {
        g_free (servant);
        CORBA_exception_free (&ev);
        return CORBA_OBJECT_NIL;
    }
    CORBA_exception_free (&ev);

    return bonobo_object_activate_servant (object, servant);
}

static void
class_init (GbfAmProjectClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (bonobo_object_get_type ());

	init_corba_class ();
}


static void
init (GbfAmProject *prj)
{
	GbfAmProjectPrivate *priv;

	priv = g_new0 (GbfAmProjectPrivate, 1);

	prj->priv = priv;

	priv->ops = NULL;
	priv->op_handler = -1;
	
	priv->event_source = bonobo_event_source_new ();
	bonobo_object_add_interface (BONOBO_OBJECT (prj), 
				     BONOBO_OBJECT (priv->event_source));
}

GtkType
gbf_am_project_get_type (void)
{
	static GtkType type = 0;

	if (type == 0) {
		static const GtkTypeInfo info =
		{
			"GbfAmProject",
			sizeof (GbfAmProject),
			sizeof (GbfAmProjectClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		type = gtk_type_unique (bonobo_object_get_type (), &info);
	}
	
	return type;
}



GbfAmProject *
gbf_am_project_new (void)
{
    GbfAmProject *prj;
    GBF_Project objref;

    prj = gtk_type_new (gbf_am_project_get_type ());

    objref = init_corba_object (BONOBO_OBJECT (prj));
    if (objref == CORBA_OBJECT_NIL) {
	gtk_object_destroy (GTK_OBJECT (prj));
	return NULL;
    }
    
    bonobo_object_construct (BONOBO_OBJECT (prj), objref);
    
    return prj;
}

static gboolean
gbf_am_project_check_queue (GbfAmProject *prj)
{
	GbfAmProjectPrivate *priv;
	GbfAmProjectOp *op;
	
	priv = prj->priv;
	
	if (!priv->ops) {
		priv->op_handler = -1;
		return FALSE;
	}

	op = priv->ops->data;
	priv->ops = g_list_remove (priv->ops, op);

	switch (op->type) {
	case BUILD:
		build_project_op (op->prj, op->type);
		break;
	}
	
	if (!priv->ops) {
		priv->op_handler = -1;
		return FALSE;
	}
	
	return TRUE;
}

static void
gbf_am_project_queue_op (GbfAmProject *prj, GbfAmProjectOp *op)
{
	GbfAmProjectPrivate *priv;

	priv = prj->priv;
	
	priv->ops = g_list_append (priv->ops, op);

	if (priv->op_handler == -1)
		g_idle_add ((GSourceFunc)gbf_am_project_check_queue, prj);
}

static void
write_source_xml (xmlDocPtr doc, xmlNodePtr parent, GList *sources)
{
	GList *l;
	
	for (l = sources; l != NULL; l = l->next) {
		xmlNodePtr cur;
		char *src = l->data;
		
		cur = xmlNewDocNode (doc, NULL, "source", NULL);
		xmlSetProp (cur, "name", src);

		xmlAddChild (parent, cur);
	}	
}

static void
write_target_xml (xmlDocPtr doc, xmlNodePtr parent, GList *targets)
{
	GList *l;
	
	for (l = targets; l != NULL; l = l->next) {
		GbfAmProjectTargetData *data = l->data;
		xmlNodePtr cur;
		
		cur = xmlNewDocNode (doc, NULL, "target", NULL);
		xmlSetProp (cur, "name", data->name);
		xmlSetProp (cur, "type", data->type);

		xmlAddChild (parent, cur);
		write_source_xml (doc, cur, data->sources);
	}
}

static xmlDocPtr
gbf_am_project_create_xml_doc (GbfAmProject *prj)
{
	GbfAmProjectPrivate *priv;
	xmlDocPtr doc;
	
	priv = prj->priv;
	
	doc = xmlNewDoc ("1.0");
	if (doc == NULL)
		return NULL;

	doc->root = xmlNewDocNode (doc, NULL, "project", NULL);
	xmlSetProp (doc->root, "name", priv->project_dir);
	write_target_xml (doc, doc->root, priv->targets);


//	xmlSetDocCompressMode (doc, 0);
//	ret = xmlSaveFile ("/tmp/prj.xml", doc);

	return doc;
	
}

static GbfAmProject *
project_from_servant (PortableServer_Servant servant)
{
	return GBF_AM_PROJECT (bonobo_object_from_servant (servant));
}

static GNOME_Development_TargetType
string_to_type (char *type)
{
	if (!strcmp (type, "program"))
		return GNOME_Development_PROGRAM;
	else if (!strcmp (type, "shared_lib"))
		return GNOME_Development_SHAREDLIB;
	else if (!strcmp (type, "static_lib"))
		return GNOME_Development_STATICLIB;
	else if (!strcmp (type, "extra"))
		return GNOME_Development_EXTRA;

	return GNOME_Development_PROGRAM;
}

static char *
target_as_string (const GBF_Target *target) 
{
	return g_strconcat (target->path, "/", target->name, NULL); 
}

static char *
source_as_string (const GBF_Source *source)
{
	return g_strconcat (source->path, "/", source->name, NULL);
}

static GList *
target_to_list_elem (GList *targets, const GBF_Target *target)
{
	char *trg;
	GList *l;

	trg = target_as_string (target);
	
	for (l = targets; l != NULL; l = l->next) {
		GbfAmProjectTargetData *data = l->data;
		
		if (target->type == string_to_type (data->type) 
		    && !strcmp (trg, data->name))
			break;
	}
	g_free (trg);
	
	return l;
}

static GList *
source_to_list_elem (GList *sources, const GBF_Source *source)
{
	char *src;
	GList *l;

	src = source_as_string (source);
	
	for (l = sources; l != NULL; l = l->next) {
		if (!strcmp (src, (char *)l->data))
			break;
	}
	g_free (src);
	
	return l;
}

static gchar *
make_script_path (gchar *name)
{
	gchar *path = g_strjoin ("/", SCRIPTS_DIR, name, NULL);

	return path;
}

static int
load_project (char *argv[], xmlSAXHandler *handler, gpointer data) 
{
	int fd[2];
	int t, len;
	char *p;
	char *script_path;
	
	xmlSubstituteEntitiesDefault (TRUE);

	pipe (fd);

	t = fork ();
	if (t < 0) {
		g_error ("Unable to fork.");
	} else if (t) {
		/* Parent */

		close (fd[1]);	/* Close writing end */

		/* LibXML support for parsing from memory is good, but parsing from
		 * opened filehandles is not supported unless you write your own feed
		 * mechanism. Let's just load it all into memory, then. Also, refusing
		 * enormous documents can be considered a plus. </dystopic> */

		p = malloc (102400);
		fcntl(fd[0], F_SETFL, 0);  /* Let's block */
		for (len = 0; (t = read (fd[0], p + len, 102399 - len)); )
			len += t;

		if (len < 1 || len == 102399) {
			free (p);
			g_warning ("Empty project returned");
			return -1;
		}

		p = realloc (p, len + 1);
		*(p + len) = 0;

		if (xmlSAXUserParseMemory (handler, data, p, len) < 0)
			return -1;

		free (p);
		close (fd[0]);
	} else {
		/* Child */

		close (fd[0]);	/* Close reading end */
		dup2 (fd[1], STDOUT_FILENO);

		script_path = make_script_path (argv[0]);
		execve (script_path, argv, __environ);
		g_error ("Unable to run backend: %s", script_path);
	}
	
	return 0;
}

static int
save_project (char *argv[], xmlDocPtr doc, gpointer data) 
{
	int fd[2];
	FILE *f;
	int t;
	char *script_path;
	
	xmlSubstituteEntitiesDefault (TRUE);

	pipe (fd);

	t = fork ();
	if (t < 0) {
		g_error ("Unable to fork.");
	} else if (t) {
		/* Parent */

		close (fd [0]);	/* Close reading end of pipe */

		f = fdopen (fd [1], "w");
		xmlDocDump (f, doc);
		fclose (f);
	} else {
		/* Child */

		close (fd[1]);	/* Close reading end */
		dup2 (fd[0], STDIN_FILENO);

		script_path = make_script_path (argv[0]);
		execve (script_path, argv, __environ);
		g_error ("Unable to run backend: %s", script_path);
	}
	
	return 0;
}

static void
get_targets_sax_start_element (GbfAmProject *prj, const xmlChar *name, const xmlChar **attrs)
{
	GbfAmProjectPrivate *priv;
	
	priv = prj->priv;
	
	if (!strcmp (name, "target")) {
		GbfAmProjectTargetData *tdata = g_new0 (GbfAmProjectTargetData, 1);

		while (attrs && *attrs != NULL) {
			const xmlChar **val = attrs;
			
			val++;
			if (!strcmp (*attrs, "name"))
				tdata->name = g_strdup (*val);
			else if (!strcmp (*attrs, "type"))
				tdata->type = g_strdup (*val);

			attrs = ++val;
		}
		priv->targets = g_list_prepend (priv->targets, tdata);
	}

	if (!strcmp (name, "source")) {
		GbfAmProjectTargetData *tdata = priv->targets->data;
		
		while (attrs && *attrs != NULL) {
			const xmlChar **val = attrs;
			
			val++;
			if (!strcmp (*attrs, "name")) {
  				tdata->sources = g_list_prepend (tdata->sources, g_strdup (*val));
				priv->sources = g_list_prepend (priv->sources, g_strdup (*val));
				break;
			}	
			attrs = ++val;
		}
	}
}

static int
get_targets (GbfAmProject *prj)
{
	char *argv[] = { SCRIPT_NAME, "--get", prj->priv->project_dir, 0 };
	
	xmlSAXHandler handler;
	
	memset (&handler, 0, sizeof (xmlSAXHandler));
	handler.startElement = (startElementSAXFunc)get_targets_sax_start_element;

	return load_project (argv, &handler, prj);
}

static void
impl_load_project (PortableServer_Servant servant, const CORBA_char *path, CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GbfAmProjectPrivate *priv;
	int res;
	
	priv = prj->priv;
	
	priv->project_dir = g_strdup (path);
	
	if (!g_file_test (priv->project_dir, G_FILE_TEST_ISDIR))
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, GBF_ex_DoesntExist, NULL);

	res = get_targets (prj);
	if (res != 0)
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, GBF_ex_Malformed, NULL);
}

static CORBA_char *
impl_get_project_root (PortableServer_Servant servant, CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);

	return CORBA_string_dup (prj->priv->project_dir);
}

static gboolean
parse_error_buffer (GbfAmProject *prj)
{
	BonoboArg *arg;
	GbfAmProjectPrivate *priv;
	gchar **tokens = NULL;
	char *end, *line, *text, *tmp;
	int len;
	
	priv = prj->priv;

	if (priv->buffer == NULL)
		return FALSE;
	end = strchr (priv->buffer, '\n');
	if (end == NULL)
		return FALSE;
		
	len = end - priv->buffer;
	line = g_new0 (char, len + 1);
	strncpy (line, priv->buffer, len);
	line[len] = '\0';
		
	if (!strncmp (line, "FILE: ", 6)) {
		GBF_Source source;
		
		text = strchr (line, ' ');
		text++;

		tokens = g_strsplit (text, " ", 2);
		arg = bonobo_arg_new (BONOBO_ARG_NULL);
		arg->_type = TC_GNOME_Development_Source;
		source.name = tokens[1];
		source.path = tokens[0];
		arg->_value = &source;
		CORBA_any_set_release (arg, FALSE);
		bonobo_event_source_notify_listeners (priv->event_source,
						      "build-file",
						      arg, NULL);
		bonobo_arg_release (arg);
	} else if (!strncmp (line, "ERROR: ", 7)) {
		GBF_BuildError build_err;
		int i, max = 3;
		g_print ("Line: %s\n", line);
		
		text = strchr (line, ' ');
		text++;

		arg = bonobo_arg_new (BONOBO_ARG_NULL);
		arg->_type = TC_GNOME_Development_BuildError;

		tokens = g_strsplit (text, " ", 4);
		build_err.source.name = tokens[1];
		build_err.source.path = tokens[0];
		build_err.line = strtol (tokens[2], NULL, 0);

		if (!strcmp (tokens[3], "warning:")) {
			max++;
			build_err.type = GNOME_Development_WARNING;
		} else {
			build_err.type = GNOME_Development_CRITICAL;
		}

		for (i = 0; i < max; i++) {
			text = strchr (text, ' ');
			text++;
		}
		build_err.error = text;
		arg->_value = &build_err;
		CORBA_any_set_release (arg, FALSE);
		bonobo_event_source_notify_listeners (priv->event_source,
						      "build-error",
						      arg, NULL);
		bonobo_arg_release (arg);
	}
	if (tokens != NULL)
		g_strfreev (tokens);
	g_free (line);

	tmp = priv->buffer;
	priv->buffer = g_strdup (++end);
	g_free (tmp);
	
	return TRUE;
}

static gboolean
build_input_cb (GIOChannel *chan, GIOCondition cond, gpointer user_data)
{
	GbfAmProject *prj = user_data;
	GbfAmProjectPrivate *priv;
	char in_buf[1024];
	char *tmp;
	int read;
	
	priv = prj->priv;
	
	g_io_channel_read (chan, in_buf, 1024, &read);
	if (read > 0) {
		if (priv->buffer) {
			tmp = priv->buffer;
			priv->buffer = g_strconcat (priv->buffer, in_buf, NULL);
			g_free (tmp);
		} else {
			priv->buffer = g_strdup (in_buf);
		}
		parse_error_buffer (prj);
	}
	
	return TRUE;
}

static gboolean
build_huperr_cb (GIOChannel *chan, GIOCondition cond, gpointer user_data)
{
	BonoboArg *arg;
	GbfAmProject *prj = user_data;
	
	while (parse_error_buffer (prj));

	arg = bonobo_arg_new (BONOBO_ARG_NULL);
	CORBA_any_set_release (arg, FALSE);
	bonobo_event_source_notify_listeners (prj->priv->event_source,
					      "build-ended",
					      arg, NULL);
	bonobo_arg_release (arg);

	return FALSE;
}

static int
launch_build_script (char *argv[], gpointer data) 
{
	GIOChannel *chan;
	int fd[2];
	int t;
	char *script_path;
	
	pipe (fd);

	t = fork ();
	if (t < 0) {
		g_error ("Unable to fork.");
	} else if (t) {
		/* Parent */
		close (fd[1]);	/* Close writing end */

		chan = g_io_channel_unix_new (fd[0]);
		g_io_add_watch (chan, G_IO_IN, build_input_cb, data);
		g_io_add_watch (chan, G_IO_ERR | G_IO_HUP, build_huperr_cb, data);
	} else {
		/* Child */
		close (fd[0]);	/* Close reading end */
		dup2 (fd[1], STDOUT_FILENO);

		script_path = make_script_path (argv[0]);
		execve (script_path, argv, __environ);
		g_error ("Unable to run backend: %s", script_path);
	}
	
	return 0;
}

static void
build_project_op (GbfAmProject *prj, 
		  GNOME_Development_BuildType type) 
{
	BonoboArg *arg;
	char *argv[] = { BUILD_SCRIPT_NAME, "--build", "", prj->priv->project_dir, 0 };

	switch (type) {
	case GNOME_Development_ALL:
		argv[2] = "all";
		break;
	case GNOME_Development_CURRENT:
		argv[2] = "all";
		break;
	case GNOME_Development_INSTALL:
		argv[2] = "install";
		break;
	}
	
	if (launch_build_script (argv, prj) != 0) {
		g_print ("Can't launch build script!");
		return;
	}

	arg = bonobo_arg_new (BONOBO_ARG_NULL);
	CORBA_any_set_release (arg, FALSE);
	bonobo_event_source_notify_listeners (prj->priv->event_source,
					      "build-started",
					      arg, NULL);
	bonobo_arg_release (arg);
}

static void
impl_build_project (PortableServer_Servant servant, 
		    GNOME_Development_BuildType type, 
		    CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GbfAmProjectOp *op = g_new0 (GbfAmProjectOp, 1);

	op->prj = prj;
	op->type = BUILD;
	op->build_type = type;
	
	gbf_am_project_queue_op (prj, op);
}

static GBF_TargetList *
impl_get_targets (PortableServer_Servant servant, 
		  CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GBF_TargetList *tl;
	GList *l;
	int i, n;
	
	n = g_list_length (prj->priv->targets);
	tl = GNOME_Development_TargetList__alloc ();
	tl->_length = n;
	tl->_maximum = n;
	tl->_buffer = CORBA_sequence_GNOME_Development_Target_allocbuf (n);
	CORBA_sequence_set_release (tl, TRUE);
	
	for (l = prj->priv->targets, i = 0; l != NULL; l = l->next, i++) {
		GBF_Target *corba_t;
		GbfAmProjectTargetData *tdata = l->data;
		char *d;
		
		corba_t = &tl->_buffer[i];
		corba_t->name = CORBA_string_dup (g_basename (tdata->name));
		d = g_dirname (tdata->name);
		if (strcmp (d, "."))
			corba_t->path = CORBA_string_dup (d);
		else
			corba_t->path = CORBA_string_dup ("");
		g_free (d);
		corba_t->type = string_to_type (tdata->type);
	}

	return tl;
}

static GBF_TargetList *
impl_get_targets_of_source (PortableServer_Servant servant,
			    const GBF_Source *source,
			    CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GbfAmProjectPrivate *priv;
	GList *targets = NULL, *elem, *l;
	GBF_TargetList *tl;
	int i, n;

	priv = prj->priv;

	/* Find all the targets containing the source */
	for (l = priv->targets; l != NULL; l = l->next) {
		GbfAmProjectTargetData *tdata = l->data;

		elem = source_to_list_elem (tdata->sources, source);
		if (elem != NULL)
			targets = g_list_prepend (targets, tdata);
	}

	/* Build the actual corba sequence */
	n = g_list_length (targets);
	tl = GNOME_Development_TargetList__alloc ();
	tl->_length = n;
	tl->_maximum = n;
	tl->_buffer = CORBA_sequence_GNOME_Development_Target_allocbuf (n);
	CORBA_sequence_set_release (tl, TRUE);

	for (l = targets, i = 0; l != NULL; l = l->next, i++) {
		GBF_Target *corba_t;
		GbfAmProjectTargetData *tdata = l->data;
		char *d;
		
		corba_t = &tl->_buffer[i];
		corba_t->name = CORBA_string_dup (g_basename (tdata->name));
		d = g_dirname (tdata->name);
		if (strcmp (d, "."))
			corba_t->path = CORBA_string_dup (d);
		else
			corba_t->path = CORBA_string_dup ("");
		g_free (d);
		corba_t->type = string_to_type (tdata->type);
	}
	g_list_free (targets);
	
	return tl;
}

static void
impl_add_target (PortableServer_Servant servant, 
		 const GBF_Target *target, 
		 CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);

}

static void
impl_remove_target (PortableServer_Servant servant, 
		    const GBF_Target *target, 
		    CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
}

static GBF_SourceList *
impl_get_sources (PortableServer_Servant servant, 
		  CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GBF_SourceList *sl;
	GList *l;
	int i, n;

	n = g_list_length (prj->priv->sources);
	sl = GNOME_Development_SourceList__alloc ();
	sl->_length = n;
	sl->_maximum = n;
	sl->_buffer = CORBA_sequence_GNOME_Development_Source_allocbuf (n);
	CORBA_sequence_set_release (sl, TRUE);
	
	for (l = prj->priv->sources, i = 0; l != NULL; l = l->next, i++) {
		GBF_Source *corba_s;
		char *s = l->data;
		char *d;
		
		corba_s = &sl->_buffer[i];
		corba_s->name = CORBA_string_dup (g_basename (s));
		d = g_dirname (s);
		if (strcmp (d, "."))
			corba_s->path = CORBA_string_dup (d);
		else
			corba_s->path = CORBA_string_dup ("");
		g_free (d);
	}

	return sl;
}

static GBF_SourceList *
impl_get_target_sources (PortableServer_Servant servant, 
			 const GBF_Target *target, 
			 CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GBF_SourceList *sl;
	GList *l;
	int i, n;

	sl = GNOME_Development_SourceList__alloc ();	
	CORBA_sequence_set_release (sl, TRUE);

	for (l = prj->priv->targets; l != NULL; l = l->next) {
		GbfAmProjectTargetData *tdata = l->data;
		char *tmp;

		if (target->path && *target->path)
			tmp = g_strdup_printf ("%s/%s", target->path, target->name);
		else
			tmp = g_strdup (target->name);
		
		if (!strcmp (tdata->name, tmp)) {
			GList *m;

			n = g_list_length (tdata->sources);
			sl->_length = n;
			sl->_maximum = n;
			sl->_buffer = CORBA_sequence_GNOME_Development_Source_allocbuf (n);

			for (m = tdata->sources, i = 0; m != NULL; m = m->next, i++) {
				GBF_Source *corba_s;
				char *s = m->data;
				char *d;
				
				corba_s = &sl->_buffer[i];
				corba_s->name = CORBA_string_dup (g_basename (s));
				d = g_dirname (s);
				if (strcmp (d, "."))
					corba_s->path = CORBA_string_dup (d);
				else
					corba_s->path = CORBA_string_dup ("");
				g_free (d);
			}
			g_free (tmp);
			
			return sl;
		}
		g_free (tmp);
	}

	CORBA_exception_set (ev, CORBA_USER_EXCEPTION, 
			     ex_GNOME_Development_Project_DoesntExist, NULL);
	
	return sl;
}

static void
impl_add_target_source (PortableServer_Servant servant, 
			const GBF_Target *target, 
			const GBF_Source *source, 
			CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GbfAmProjectPrivate *priv;
	GbfAmProjectTargetData *tdata;
	GList *elem;
	BonoboArg *arg;

	char *argv[] = { SCRIPT_NAME, "--set", prj->priv->project_dir, 0 };
	xmlDocPtr doc;
	
	priv = prj->priv;

	elem = target_to_list_elem (priv->targets, target);
	if (elem == NULL) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, GBF_ex_DoesntExist, NULL);
		return;
	}
	tdata = elem->data;

	tdata->sources = g_list_prepend (tdata->sources, source_as_string (source));
	priv->sources = g_list_prepend (priv->sources, source_as_string (source));

	doc = gbf_am_project_create_xml_doc (prj);
	save_project (argv, doc, prj);
	xmlFreeDoc (doc);
	
	arg = bonobo_arg_new (BONOBO_ARG_NULL);
	bonobo_event_source_notify_listeners (priv->event_source,
					      "project-changed",
					      arg, NULL);
	bonobo_arg_release (arg);
}

static void
impl_remove_target_source (PortableServer_Servant servant, 
			   const GBF_Target *target, 
			   const GBF_Source *source, 
			   CORBA_Environment *ev)
{
	GbfAmProject *prj = project_from_servant (servant);
	GbfAmProjectPrivate *priv;
	GList *elem, *elem2;
	BonoboArg *arg;

	char *argv[] = { SCRIPT_NAME, "--set", prj->priv->project_dir, 0 };
	xmlDocPtr doc;
	
	priv = prj->priv;

	elem = target_to_list_elem (priv->targets, target);
	if (elem != NULL) {
		GbfAmProjectTargetData *tdata = elem->data;
		
		elem2 = source_to_list_elem (tdata->sources, source);
		if (elem2 != NULL) {
			tdata->sources = g_list_remove (tdata->sources, elem2->data);
		} else {
			CORBA_exception_set (ev, CORBA_USER_EXCEPTION, GBF_ex_DoesntExist, NULL);
			return;
		}		
	} else {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, GBF_ex_DoesntExist, NULL);
		return;
	}

	elem = source_to_list_elem (priv->sources, source);
	if (elem != NULL) {
		priv->sources = g_list_remove (priv->sources, elem->data);
	} else {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION, GBF_ex_DoesntExist, NULL);
		return;
	}

	doc = gbf_am_project_create_xml_doc (prj);
	save_project (argv, doc, prj);
	xmlFreeDoc (doc);
	
	arg = bonobo_arg_new (BONOBO_ARG_NULL);
	bonobo_event_source_notify_listeners (priv->event_source,
					      "project-changed",
					      arg, NULL);
	bonobo_arg_release (arg);
}
