/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include "config.h"
#include <gmf.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

static void GMF_Pipe_Info__copy(GMF_Pipe_Info *out, GMF_Pipe_Info *in);
static void GMF_Pipe_ConnectionInfo__copy(GMF_Pipe_ConnectionInfo *out,
					  GMF_Pipe_ConnectionInfo *in);
static void GMF_Transport_Info__copy(GMF_Transport_Info *out,
				     GMF_Transport_Info *in);
static void GMF_MediaTypeInfo__copy(GMF_MediaTypeInfo *out,
				    GMF_MediaTypeInfo *in);

static void gmf_pipe_class_init (GMFPipeClass * klass);
static void gmf_pipe_init (GMFPipe * obj);
static void gmf_pipe_destroy (GtkObject * obj);
static void gmf_pipe_get_arg (GtkObject * obj, GtkArg *arg, guint arg_id);
static void gmf_pipe_shmem_send_sample(GMFPipe *apipe, GMF_Sample *sample);
static void gmf_pipe_notify_send_sample(GMFPipe *apipe, GMF_Sample *sample);
static void gmf_pipe_fd_send_sample(GMFPipe *apipe, GMF_Sample *sample);
static void gmf_pipe_corba_send_sample(GMFPipe *apipe, GMF_Sample *sample);
static CORBA_char *gmf_pipe_get_address(GMFPipe *apipe, GMF_Transport_Type ttype);
static gint gmf_pipe_connect_transport(GMFPipe *apipe);
static gint gmf_pipe_setup_fd(GMFPipe *apipe, gint fd, gboolean is_server);
static void gmf_pipe_setup_server(GMFPipe *apipe);

typedef struct {
  POA_GMF_Pipe servant;
  PortableServer_POA poa;
  PortableServer_ObjectId *objid;
  
  GMF_Pipe_Info attr_pipeInfo;

  GMF_Pipe_ConnectionInfo *attr_pConnectionInfo;

  GMFPipe *gtkobj;

  GMF_Transport_Type acceptable_transports;

  gboolean in_flush;

  /* These are for UNIX_SOCKET & TCPIP transports */
  int usock_serv_fd, usock_serv_fd_id;
  int tcp_serv_fd, tcp_serv_fd_id;
  int fd, fd_id;
  union {
    struct sockaddr_un usock_addr; /* Our local addresses */
    struct sockaddr_in tcpsock_addr; /* Our local addresses */
  } u;
  long shmem_key;

  CORBA_long pipeID;
} impl_POA_GMF_Pipe;

#define PIPE_DATA(x) ((impl_POA_GMF_Pipe *)(GMF_PIPE(x)->servant))

static void impl_GMF_Pipe__destroy(impl_POA_GMF_Pipe * servant,
				   CORBA_Environment * ev);

static GMF_Pipe_Info *
impl_GMF_Pipe__get_pipeInfo(impl_POA_GMF_Pipe * servant,
			    CORBA_Environment * ev);

static GMF_Pipe_ConnectionInfo *
impl_GMF_Pipe__get_pConnectionInfo(impl_POA_GMF_Pipe * servant,
				   CORBA_Environment * ev);

static CORBA_boolean
impl_GMF_Pipe_CanProcessType(impl_POA_GMF_Pipe * servant,
			     GMF_MediaTypeInfo * typeInfo,
			     CORBA_Environment * ev);

static GMF_MediaTypeInfoList *
impl_GMF_Pipe__get_processableTypes(impl_POA_GMF_Pipe * servant,
				    CORBA_Environment * ev);

static void
impl_GMF_Pipe_NewSegment(impl_POA_GMF_Pipe * servant,
			 GMF_TimeVal * tStart,
			 GMF_TimeVal * tStop,
			 CORBA_double playRate,
			 CORBA_Environment * ev);

static void
impl_GMF_Pipe_BeginFlush(impl_POA_GMF_Pipe * servant,
			 CORBA_Environment * ev);

static void
impl_GMF_Pipe_EndFlush(impl_POA_GMF_Pipe * servant,
		       CORBA_Environment * ev);

static void
impl_GMF_Pipe_EndOfStream(impl_POA_GMF_Pipe * servant,
			  CORBA_Environment * ev);

static void
impl_GMF_Pipe_HandleSample(impl_POA_GMF_Pipe * servant,
			   GMF_Sample * sdata,
			   CORBA_Environment * ev);

static void
impl_GMF_Pipe_NotifySample(impl_POA_GMF_Pipe * servant,
			   CORBA_Environment * ev);

static void
impl_GMF_Pipe_ConnectTo(impl_POA_GMF_Pipe * servant,
			GMF_Pipe other_end,
			CORBA_Environment *ev);

static void
impl_GMF_Pipe_AcceptConnection(impl_POA_GMF_Pipe * servant,
			       GMF_Pipe_ConnectionInfo * cinfo,
			       CORBA_Environment * ev);

static void
impl_GMF_Pipe_CloseConnection(impl_POA_GMF_Pipe * servant,
			      CORBA_Environment * ev);

static CORBA_long
impl_GMF_Pipe__get_pipeID(impl_POA_GMF_Pipe * servant,
			  CORBA_Environment * ev);

static void
impl_GMF_Pipe__set_pipeID(impl_POA_GMF_Pipe * servant,
			  CORBA_long value,
			  CORBA_Environment * ev);

/* module vars */

enum {
  RECEIVE_SAMPLE,
  CAN_PROCESS_TYPE,
  GET_PROCESSABLE_TYPES,
  NEW_SEGMENT,
  BEGIN_FLUSH,
  END_FLUSH,
  END_OF_STREAM,
  LAST_SIGNAL
};

static guint pipe_signals[LAST_SIGNAL] = { 0 };

static CORBA_Environment modev;

static PortableServer_ServantBase__epv impl_GMF_Pipe_base_epv =
{
  NULL,			/* _private data */
  NULL,                 /* finalize routine */
  NULL,			/* default_POA routine */
};

static POA_GMF_Pipe__epv impl_GMF_Pipe_epv =
{
  NULL,			/* _private */
  (gpointer) & impl_GMF_Pipe__get_pipeInfo,
  (gpointer) & impl_GMF_Pipe__get_pConnectionInfo,
  (gpointer) & impl_GMF_Pipe_CanProcessType,
  (gpointer) & impl_GMF_Pipe__get_processableTypes,
  (gpointer) & impl_GMF_Pipe_NewSegment,
  (gpointer) & impl_GMF_Pipe_BeginFlush,
  (gpointer) & impl_GMF_Pipe_EndFlush,
  (gpointer) & impl_GMF_Pipe_EndOfStream,
  (gpointer) & impl_GMF_Pipe_HandleSample,
  (gpointer) & impl_GMF_Pipe_NotifySample,
  (gpointer) & impl_GMF_Pipe_ConnectTo,
  (gpointer) & impl_GMF_Pipe_AcceptConnection,
  (gpointer) & impl_GMF_Pipe_CloseConnection,
  (gpointer) & impl_GMF_Pipe__get_pipeID,
  (gpointer) & impl_GMF_Pipe__set_pipeID
};

static POA_GMF_Pipe__vepv impl_GMF_Pipe_vepv =
{
  &impl_GMF_Pipe_base_epv,
  &impl_GMF_Pipe_epv,
};

static GMF_Pipe
impl_GMF_Pipe__create(PortableServer_POA poa, gpointer *servantptr,
		      CORBA_Environment * ev)
{
  GMF_Pipe retval;
  impl_POA_GMF_Pipe *newservant;
  PortableServer_ObjectId *objid;

  newservant = *servantptr = g_new0(impl_POA_GMF_Pipe, 1);
  newservant->servant.vepv = &impl_GMF_Pipe_vepv;

  POA_GMF_Pipe__init((PortableServer_Servant) newservant, ev);
  objid = PortableServer_POA_activate_object(poa, newservant, ev);
  CORBA_free(objid);

  newservant->poa = poa;
  newservant->pipeID = -1; /* No ID to start with */

  newservant->usock_serv_fd =
    newservant->usock_serv_fd_id =
    newservant->tcp_serv_fd =
    newservant->tcp_serv_fd_id =
    newservant->fd =
    newservant->fd_id = -1;

  retval = PortableServer_POA_servant_to_reference(poa, newservant, ev);

  return retval;
}

/* You shouldn't call this routine directly without first deactivating
   the servant... */
static void
impl_GMF_Pipe__destroy(impl_POA_GMF_Pipe * servant, CORBA_Environment * ev)
{
  PortableServer_POA_deactivate_object(servant->poa, servant->objid, ev);
  POA_GMF_Pipe__fini((PortableServer_Servant) servant, ev);
  CORBA_Object_release(servant->gtkobj->corba_object, ev);
  CORBA_free(servant->objid);
  g_free(servant);
  
  g_warning("gmf_pipe_destroy not yet finished...");
}

static GMF_Pipe_Info *
impl_GMF_Pipe__get_pipeInfo(impl_POA_GMF_Pipe * servant,
			    CORBA_Environment * ev)
{
  GMF_Pipe_Info *retval;

  retval = GMF_Pipe_Info__alloc();
  GMF_Pipe_Info__copy(retval, &servant->attr_pipeInfo);

  return retval;
}

static GMF_Pipe_ConnectionInfo *
impl_GMF_Pipe__get_pConnectionInfo(impl_POA_GMF_Pipe * servant,
				   CORBA_Environment * ev)
{
  GMF_Pipe_ConnectionInfo *retval;

  retval = GMF_Pipe_ConnectionInfo__alloc();
  GMF_Pipe_ConnectionInfo__copy(retval, servant->attr_pConnectionInfo);

  return retval;
}


static void
GMF_Pipe_Info__copy(GMF_Pipe_Info *out, GMF_Pipe_Info *in)
{
  out->pDirection = in->pDirection;
  out->pFilter = CORBA_Object_duplicate(in->pFilter, &modev);
  out->acceptedTypes = in->acceptedTypes;
}

static void
GMF_Pipe_ConnectionInfo__copy(GMF_Pipe_ConnectionInfo *out,
			      GMF_Pipe_ConnectionInfo *in)
{
  GMF_MediaTypeInfo__copy(&out->connectionMediaInfo, &in->connectionMediaInfo);
  out->connectedTo = CORBA_Object_duplicate(in->connectedTo, &modev);
  GMF_Transport_Info__copy(&out->connectionInfo, &in->connectionInfo);
}

static void
GMF_Transport_Info__copy(GMF_Transport_Info *out, GMF_Transport_Info *in)
{
  out->_d = in->_d;
  switch(out->_d) {
  case GMF_Transport_UNIX_SOCKETS:
  case GMF_Transport_TCPIP:
    out->_u.address = CORBA_string_dup(in->_u.address);
    break;
  case GMF_Transport_CUSTOM:
    CORBA_any__copy(&out->_u.tdata, &in->_u.tdata);
    break;
  case GMF_Transport_SHARED_MEM:
    out->_u.shmem_key = in->_u.shmem_key;
    break;
  default:
    break;
  }
}

static void
GMF_MediaTypeInfo__copy(GMF_MediaTypeInfo *out, GMF_MediaTypeInfo *in)
{
  out->majorType = in->majorType;
  out->minorType = CORBA_string_dup(in->minorType);
  out->fixedSizeSamples = in->fixedSizeSamples;
  out->temporalCompression = in->temporalCompression;
  out->sampleSize = in->sampleSize;
  CORBA_any__copy(&out->typeData, &in->typeData);
  
  out->format = in->format;
  CORBA_any__copy(&out->formatData, &in->formatData);
}

static CORBA_boolean
impl_GMF_Pipe_CanProcessType(impl_POA_GMF_Pipe * servant,
			     GMF_MediaTypeInfo * typeInfo,
			     CORBA_Environment * ev)
{
  gboolean retval = FALSE;

  gtk_signal_emit(GTK_OBJECT(servant->gtkobj), pipe_signals[CAN_PROCESS_TYPE],
		  typeInfo, &retval);

  return (CORBA_boolean)retval;
}

static GMF_MediaTypeInfoList *
impl_GMF_Pipe__get_processableTypes(impl_POA_GMF_Pipe * servant,
				    CORBA_Environment * ev)
{
  GMF_MediaTypeInfoList *retval = NULL;

  gtk_signal_emit(GTK_OBJECT(servant->gtkobj),
		  pipe_signals[GET_PROCESSABLE_TYPES],
		  &retval);

  if(!retval) {
    retval = GMF_MediaTypeInfoList__alloc();
    retval->_length = 0;
    retval->_buffer = NULL;
  }

  return retval;
}

static void
impl_GMF_Pipe_NewSegment(impl_POA_GMF_Pipe * servant,
			 GMF_TimeVal * tStart,
			 GMF_TimeVal * tStop,
			 CORBA_double playRate,
			 CORBA_Environment * ev)
{
  gtk_signal_emit(GTK_OBJECT(servant->gtkobj),
		  pipe_signals[NEW_SEGMENT], tStart, tStop,
		  playRate);
}

static void
impl_GMF_Pipe_BeginFlush(impl_POA_GMF_Pipe * servant,
			 CORBA_Environment * ev)
{
  if(servant->in_flush) {
    g_return_if_fail(!servant->in_flush);
    return;
  }

  servant->in_flush = TRUE;

  switch(servant->attr_pipeInfo.pDirection) {
  case GMF_OUT:
    /* If we are on the upstream end of a connection, propogate the flush
       on to the downstream pin */
    GMF_Pipe_BeginFlush(servant->attr_pConnectionInfo->connectedTo, ev);
    break;
  case GMF_IN:
    /* If we are on the downstream end of a connection, just tell the filter
       about the flush (the filter will handle it) */
    gtk_signal_emit_by_name(GTK_OBJECT(servant->gtkobj), "begin_flush");
    break;
  }
}

static void
impl_GMF_Pipe_EndFlush(impl_POA_GMF_Pipe * servant,
		       CORBA_Environment * ev)
{
  if(!servant->in_flush) {
    g_return_if_fail(servant->in_flush);
    return;
  }

  servant->in_flush = FALSE;

  switch(servant->attr_pipeInfo.pDirection) {
  case GMF_OUT:
    /* If we are on the upstream end of a connection, propogate the flush
       on to the downstream pin */
    GMF_Pipe_EndFlush(servant->attr_pConnectionInfo->connectedTo, ev);
    break;
  case GMF_IN:
    /* If we are on the downstream end of a connection, just tell the filter
       about the flush (the filter will handle it) */
    gtk_signal_emit_by_name(GTK_OBJECT(servant->gtkobj), "end_flush");
    break;
  }
}

static void
impl_GMF_Pipe_EndOfStream(impl_POA_GMF_Pipe * servant,
			  CORBA_Environment * ev)
{
  gtk_signal_emit(GTK_OBJECT(servant->gtkobj), pipe_signals[END_OF_STREAM]);
}

static void
impl_GMF_Pipe_HandleSample(impl_POA_GMF_Pipe * servant,
			   GMF_Sample * sdata,
			   CORBA_Environment * ev)
{
  gboolean foo;

  /* CORBA data transport */
  gtk_signal_emit(GTK_OBJECT(servant->gtkobj),
		  pipe_signals[RECEIVE_SAMPLE], sdata, TRUE, &foo);
}

static void
impl_GMF_Pipe_NotifySample(impl_POA_GMF_Pipe * servant,
			   CORBA_Environment * ev)
{
  gboolean foo;

  g_return_if_fail(servant->attr_pConnectionInfo);

  switch(servant->attr_pConnectionInfo->connectionInfo._d)
    {
    case GMF_Transport_SHARED_MEM:
      g_error("GMF_Transport_SHARED_MEM NYI");
      break;
    case GMF_Transport_CUSTOM:
      gtk_signal_emit(GTK_OBJECT(servant->gtkobj),
		      pipe_signals[RECEIVE_SAMPLE], NULL, FALSE,
		      &foo);
      break;
    default:
      g_error("Transport %d does not support NotifyData",
	      servant->attr_pConnectionInfo->connectionInfo._d);
      break;
    }

  /* CORBA data transport */
}

static gint
gmf_pipe_connect_transport(GMFPipe *apipe)
{
  GMF_Pipe_ConnectionInfo *pcnx;
  struct sockaddr_un remote;
  impl_POA_GMF_Pipe *servant;

  servant = PIPE_DATA(apipe);
  pcnx = servant->attr_pConnectionInfo;
  
  switch(pcnx->connectionInfo._d) {
  case GMF_Transport_UNIX_SOCKETS:
    servant->fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(servant->fd < 0) return -1;
    remote.sun_family = AF_UNIX;
    strcpy(remote.sun_path, pcnx->connectionInfo._u.address);
    if(connect(servant->fd, &remote, SUN_LEN(&remote)) < 0) {
      close(servant->fd); servant->fd = -1;
      return -1;
    }
    if(servant->attr_pipeInfo.pDirection == GMF_IN)
      servant->fd_id = gmf_pipe_setup_fd(apipe, servant->fd, FALSE);
    break;

  case GMF_Transport_CORBA:
    break;

  default:
    g_error("Transports beside UNIX sockets & CORBA NYI");
    break;
  }

  return 0;
}

static void
impl_GMF_Pipe_ConnectTo(impl_POA_GMF_Pipe * servant,
			GMF_Pipe other_end,
			CORBA_Environment *ev)
{
  /* Wherein the magic occurs */

  GMF_MediaTypeInfoList *our_types = NULL;
  GMF_Pipe_Info *remote_pinfo = NULL;
  GMF_Pipe_ConnectionInfo *pcnx = NULL;
  GMF_Transport_Type negotiated_transport;
  int i, n;

  if(CORBA_Object_is_nil(other_end, ev))
    goto out;

  /* steps to negotiating a connection:
   *  1. (a) Get pipe info from the remote end.
   *     (b) Figure out a transport to use.
   *  2. Negotiate a media type.
   *  3. Set up our pConnectionInfo structure.
   *  4. Tell the remote pipe to connect.
   */

  /* 1. */
  remote_pinfo = GMF_Pipe__get_pipeInfo(other_end, ev);
  if(modev._major != CORBA_NO_EXCEPTION) {
    remote_pinfo = NULL;
    goto out;
  }

  /* 1. (a) */
  /* For now, just check for UNIX sockets, and if we don't have that use CORBA */
  /* XXX fixme - we need to allow fallbacks if the other end isn't running on
     the same system */
  if(remote_pinfo->acceptedTypes & GMF_Transport_UNIX_SOCKETS)
    negotiated_transport = GMF_Transport_UNIX_SOCKETS;
  else
    negotiated_transport = GMF_Transport_CORBA;

  /* 2. */
  our_types = impl_GMF_Pipe__get_processableTypes(servant,
						  ev);
  if(!our_types || modev._major != CORBA_NO_EXCEPTION) {
    our_types = NULL;
    goto out;
  }

  for(i = 0, n = our_types->_length; i < n; i++) {
    CORBA_boolean rv;

    rv = GMF_Pipe_CanProcessType(other_end, &our_types->_buffer[i], ev);

    if(modev._major != CORBA_NO_EXCEPTION) goto out;

    if(rv) break; /* Go with this type */
  }

  if(i == n) /* nothing found */
    goto out;

  /* 3. */
  pcnx = GMF_Pipe_ConnectionInfo__alloc();
  pcnx->connectionInfo._d = negotiated_transport;
  switch(negotiated_transport) {
  case GMF_Transport_UNIX_SOCKETS:
  case GMF_Transport_TCPIP:
    pcnx->connectionInfo._u.address = gmf_pipe_get_address(servant->gtkobj,
							   negotiated_transport);
    break;
  case GMF_Transport_SHARED_MEM:
    pcnx->connectionInfo._u.shmem_key = servant->shmem_key;
    break;
  case GMF_Transport_CUSTOM:
    g_error("Custom transport NYI");
    break;
  default:
    break;
  }

  GMF_MediaTypeInfo__copy(&pcnx->connectionMediaInfo, &our_types->_buffer[i]);
  CORBA_free(our_types); our_types = NULL;

  pcnx->connectedTo = CORBA_Object_duplicate(servant->gtkobj->corba_object, ev);

  /* 4. */
  GMF_Pipe_AcceptConnection(other_end, pcnx, ev);
  if(modev._major != CORBA_NO_EXCEPTION) goto out;

  /* 5. */ /* We are now connected - do misc maintainance & clean up */

  /* this has been modified by the remote end to something we can use */
  servant->attr_pConnectionInfo = pcnx;

  if(gmf_pipe_connect_transport(servant->gtkobj)) {
    GMF_Pipe_CloseConnection(other_end, ev);

    CORBA_free(servant->attr_pConnectionInfo);
    servant->attr_pConnectionInfo = NULL;

    goto out;
  }

  return;

 out:
  if(our_types)
    CORBA_free(our_types);

  if(remote_pinfo)
    CORBA_free(remote_pinfo);

  if(pcnx)
    CORBA_free(pcnx);

  CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
		      ex_GMF_Pipe_ConnectionRefused, NULL);
}

static void
impl_GMF_Pipe_AcceptConnection(impl_POA_GMF_Pipe * servant,
			       GMF_Pipe_ConnectionInfo * cinfo,
			       CORBA_Environment * ev)
{
  GMF_Pipe_ConnectionInfo *my_cinfo;

  if(servant->attr_pConnectionInfo) {
    CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
			ex_GMF_Pipe_ConnectionRefused, NULL);
    return;
  }

  my_cinfo = GMF_Pipe_ConnectionInfo__alloc();
  GMF_Pipe_ConnectionInfo__copy(my_cinfo, cinfo);

  CORBA_Object_release(cinfo->connectedTo, ev);
  cinfo->connectedTo = CORBA_Object_duplicate(servant->gtkobj->corba_object, ev);
  switch(cinfo->connectionInfo._d) {
  case GMF_Transport_CUSTOM:
    if(CORBA_any_get_release(&cinfo->connectionInfo._u.tdata))
      CORBA_free(cinfo->connectionInfo._u.tdata._value);
    CORBA_Object_release((CORBA_Object)cinfo->connectionInfo._u.tdata._type, ev);
    g_error("CUSTOM transport not yet fully implemented");
    break;
  case GMF_Transport_UNIX_SOCKETS:
  case GMF_Transport_TCPIP:
    CORBA_free(cinfo->connectionInfo._u.address);
    cinfo->connectionInfo._u.address =
      gmf_pipe_get_address(servant->gtkobj, cinfo->connectionInfo._d);
    break;
  }

  servant->attr_pConnectionInfo = my_cinfo;

  /* Connection accepted... */
}

static void
impl_GMF_Pipe_CloseConnection(impl_POA_GMF_Pipe * servant,
			      CORBA_Environment * ev)
{
  if(servant->attr_pConnectionInfo) {
    CORBA_free(servant->attr_pConnectionInfo);
    servant->attr_pConnectionInfo = NULL;
  } else
    g_warning("How can they close the connection when we're not connected? Explain this to me!");
}

static CORBA_long
impl_GMF_Pipe__get_pipeID(impl_POA_GMF_Pipe * servant,
			  CORBA_Environment * ev)
{
  return servant->pipeID;
}

static void
impl_GMF_Pipe__set_pipeID(impl_POA_GMF_Pipe * servant,
			  CORBA_long value,
			  CORBA_Environment * ev)
{
  servant->pipeID = value;
}

typedef void (*GtkSignal_NONE__POINTER_BOOL) (GtkObject *object, 
					      gpointer arg1,
					      gboolean arg2,
					      gpointer user_data);
typedef void (*GtkSignal_NONE__POINTER_POINTER) (GtkObject *object, 
						 gpointer arg1,
						 gpointer arg2,
						 gpointer user_data);
typedef void (*GtkSignal_NONE__POINTER_POINTER_DOUBLE)(GtkObject *object,
						       gpointer arg1,
						       gpointer arg2,
						       gdouble arg3,
						       gpointer user_data);

static void
gtk_marshal_NONE__POINTER_BOOL (GtkObject    *object, 
				GtkSignalFunc func, 
				gpointer      func_data, 
				GtkArg       *args)
{
  GtkSignal_NONE__POINTER_BOOL rfunc;

  rfunc = (GtkSignal_NONE__POINTER_BOOL) func;
  (* rfunc) (object,
	     GTK_VALUE_POINTER(args[0]),
	     GTK_VALUE_BOOL(args[1]),
	     func_data);
}

static void
gtk_marshal_NONE__POINTER_POINTER_DOUBLE (GtkObject    *object, 
					  GtkSignalFunc func, 
					  gpointer      func_data, 
					  GtkArg       *args)
{
  GtkSignal_NONE__POINTER_POINTER_DOUBLE rfunc;

  rfunc = (GtkSignal_NONE__POINTER_POINTER_DOUBLE) func;
  (* rfunc) (object,
	     GTK_VALUE_POINTER(args[0]),
	     GTK_VALUE_POINTER(args[1]),
	     GTK_VALUE_DOUBLE(args[2]),
	     func_data);
}


GtkType
gmf_pipe_get_type (void)
{
  static GtkType gmf_pipe_type = 0;
  if (!gmf_pipe_type)
    {
      static const GtkTypeInfo gmf_pipe_info =
      {
	"GMFPipe",
	sizeof (GMFPipe),
	sizeof (GMFPipeClass),
	(GtkClassInitFunc) gmf_pipe_class_init,
	(GtkObjectInitFunc) gmf_pipe_init,
	NULL, NULL,		/* reserved 1 & 2 */
	NULL
      };

      gmf_pipe_type = gtk_type_unique (gtk_object_get_type(), &gmf_pipe_info);
    }

  return gmf_pipe_type;
}

enum { ARG_0, ARG_DIRECTION, ARG_INUSE };

static void 
gmf_pipe_class_init (GMFPipeClass * klass)
{
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (klass);

  CORBA_exception_init(&modev);

  object_class->destroy = &gmf_pipe_destroy;
  object_class->get_arg = &gmf_pipe_get_arg;

  klass->receive_sample = NULL;

  pipe_signals[RECEIVE_SAMPLE] =
    gtk_signal_new("receive_sample", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFPipeClass, receive_sample),
		   gtk_marshal_NONE__POINTER_BOOL,
		   GTK_TYPE_NONE, 2, GTK_TYPE_POINTER,
		   GTK_TYPE_BOOL);

  pipe_signals[CAN_PROCESS_TYPE] =
    gtk_signal_new("can_process_type", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFPipeClass, can_process_type),
		   gtk_marshal_NONE__POINTER_POINTER,
		   GTK_TYPE_NONE, 2, GTK_TYPE_POINTER,
		   GTK_TYPE_POINTER);

  pipe_signals[GET_PROCESSABLE_TYPES] =
    gtk_signal_new("get_processable_types", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFPipeClass, get_processable_types),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  pipe_signals[NEW_SEGMENT] =
    gtk_signal_new("new_segment", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFPipeClass, new_segment),
		   gtk_marshal_NONE__POINTER_POINTER_DOUBLE,
		   GTK_TYPE_NONE, 3, GTK_TYPE_POINTER,
		   GTK_TYPE_POINTER, GTK_TYPE_DOUBLE);
  pipe_signals[BEGIN_FLUSH] =
    gtk_signal_new("begin_flush", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFPipeClass, begin_flush),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);
  pipe_signals[END_FLUSH] =
    gtk_signal_new("end_flush", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFPipeClass, end_flush),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);
  pipe_signals[END_OF_STREAM] =
    gtk_signal_new("end_of_stream", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFPipeClass, end_of_stream),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, pipe_signals, LAST_SIGNAL);

  gtk_object_add_arg_type("GMFPipe::direction", GTK_TYPE_ENUM,
                          GTK_ARG_READABLE, ARG_DIRECTION);
  gtk_object_add_arg_type("GMFPipe::inuse", GTK_TYPE_BOOL,
                          GTK_ARG_READABLE, ARG_INUSE);
}

static void
gmf_pipe_get_arg (GtkObject * obj, GtkArg *arg, guint arg_id)
{
  switch(arg_id) {
  case ARG_DIRECTION:
    GTK_VALUE_ENUM(*arg) = PIPE_DATA(GMF_PIPE(obj))->attr_pipeInfo.pDirection;
    break;
  case ARG_INUSE:
    GTK_VALUE_BOOL(*arg) =
      PIPE_DATA(GMF_PIPE(obj))->attr_pConnectionInfo?TRUE:FALSE;
    break;
  }
}

static void 
gmf_pipe_init (GMFPipe * obj)
{
  PortableServer_POA poa;

  poa = (PortableServer_POA)
    CORBA_ORB_resolve_initial_references((CORBA_ORB)gnome_CORBA_ORB(), "RootPOA", &modev);
  obj->corba_object = impl_GMF_Pipe__create(poa, &obj->servant, &modev);
  CORBA_Object_release((CORBA_Object)poa, &modev);
}

static void
gmf_pipe_destroy (GtkObject * obj)
{
  impl_GMF_Pipe__destroy(GMF_PIPE(obj)->servant, &modev);
  CORBA_Object_release(GMF_PIPE(obj)->corba_object, &modev);
}

GtkObject *gmf_pipe_new (GMFFilter *filter,
			 GMF_Direction pipedir,
			 GMF_Transport_Type acceptable_transports)
{
  GtkObject *retval;
  GMFPipe *apipe;

  retval = gtk_type_new(gmf_pipe_get_type());

  apipe = GMF_PIPE(retval);

  apipe->filter_object = GTK_OBJECT(filter);
  PIPE_DATA(apipe)->gtkobj = apipe;
  PIPE_DATA(apipe)->attr_pipeInfo.pDirection = pipedir;
  PIPE_DATA(apipe)->attr_pipeInfo.pFilter =
    CORBA_Object_duplicate(GMF_FILTER(filter)->corba_object, &modev);

  PIPE_DATA(apipe)->acceptable_transports = acceptable_transports;

  PIPE_DATA(apipe)->acceptable_transports &=
    (GMF_Transport_CORBA|GMF_Transport_UNIX_SOCKETS);
  if(PIPE_DATA(apipe)->acceptable_transports != acceptable_transports) {
    g_warning("Trimming your choice of acceptable transports down to CORBA & UNIX sockets - nothing more to offer at present.");
  }

  gmf_pipe_setup_server(apipe);

  return retval;
}

static gboolean
gmf_pipe_handle_fd(GMFPipe *apipe, gint fd, GdkInputCondition condition)
{
  GMF_Sample samp;
  int nread, ntoread, n;

  impl_POA_GMF_Pipe *servant;

  servant = PIPE_DATA(apipe);

  g_return_val_if_fail(fd != servant->fd, FALSE);

  if(condition & GDK_INPUT_EXCEPTION) goto errout;

  ntoread = sizeof(GMF_SampleInfo);
  for(nread = 0, n = 1; (n > 0) && (nread < ntoread); nread += n)
    n = read(fd, &samp.sInfo, ntoread - nread);
    
  if(n <= 0) goto errout;

  CORBA_sequence_set_release(&samp.sData, CORBA_FALSE);
  samp.sData._length = samp.sInfo.actualDataLength;
  ntoread = samp.sData._length;
  samp.sData._buffer = alloca(ntoread);
  for(nread = 0, n = 1; (n > 0) && (nread < ntoread); nread += n)
    n = read(fd, samp.sData._buffer, ntoread - nread);
    
  if(n <= 0) goto errout;

  impl_GMF_Pipe_HandleSample(servant, &samp, &modev);
  return TRUE;

 errout:
  close(fd);
  servant->fd = servant->fd_id = -1;
  return FALSE;
}

static gboolean
gmf_pipe_handle_server_fd(GMFPipe *apipe, gint fd, GdkInputCondition condition)
{
  impl_POA_GMF_Pipe *servant;
  struct sockaddr_un usock_addr;
  struct sockaddr_in tcp_addr;
  int n;
  int newfd;
  gpointer addr_ptr;

  g_return_val_if_fail(condition & GDK_INPUT_READ, FALSE);

  servant = PIPE_DATA(apipe);

  if(fd == servant->usock_serv_fd) { /* server thing */
    usock_addr.sun_family = AF_UNIX;
    n = sizeof(usock_addr);
    addr_ptr = &usock_addr;

  } else if(fd == servant->tcp_serv_fd) {

    tcp_addr.sin_family = AF_INET;
    n = sizeof(tcp_addr);
    addr_ptr = &tcp_addr;
  } else
    addr_ptr = NULL;

  newfd = accept(fd, addr_ptr, &n);

  if(servant->fd >= 0) {
    /* Instead of doing this, learn how to reject a connection */
    close(newfd);
    g_warning("Double connection");
  } else {
    servant->fd = newfd;
    servant->fd_id = gmf_pipe_setup_fd(apipe, newfd, FALSE);
  }

  return TRUE;
}

static gint
gmf_pipe_setup_fd(GMFPipe *apipe, gint fd, gboolean is_server)
{
  impl_POA_GMF_Pipe *servant;
  gpointer routine = NULL;

  servant = PIPE_DATA(apipe);

  if(is_server)
    routine = gmf_pipe_handle_server_fd;
  else if(servant->attr_pipeInfo.pDirection == GMF_IN)
    routine = gmf_pipe_handle_fd;

  if(routine)
    return gdk_input_add(fd, GDK_INPUT_READ|GDK_INPUT_EXCEPTION, routine, apipe);
  else
    return -1;
}

static void
gmf_pipe_setup_usock_server(GMFPipe *apipe)
{
  impl_POA_GMF_Pipe *servant;
  int n;
  static int didrand = 0;

  servant = PIPE_DATA(apipe);
  servant->usock_serv_fd = socket(AF_UNIX, SOCK_STREAM, 0);

  g_return_if_fail(servant->usock_serv_fd > 0);

  if(!didrand) {
    srand(time(NULL) % getpid());
    didrand = 1;
  }

  snprintf(servant->u.usock_addr.sun_path,
	   sizeof(servant->u.usock_addr.sun_path),
	   "/tmp/orbit-%s/gmf-%d%d", g_get_user_name(), rand(), rand());

  servant->u.usock_addr.sun_family = AF_UNIX;

  n = bind(servant->usock_serv_fd, &servant->u.usock_addr,
	   SUN_LEN(&servant->u.usock_addr));

  g_return_if_fail(n == 0);

  n = listen(servant->usock_serv_fd, 5);

  g_return_if_fail(n == 0);

  servant->usock_serv_fd_id = gmf_pipe_setup_fd(apipe,
						servant->usock_serv_fd, TRUE);
}

static void
gmf_pipe_setup_tcpip_server(GMFPipe *apipe)
{
  g_error("TCP/IP transport NYI");
}

static void
gmf_pipe_setup_shmem_server(GMFPipe *apipe)
{
  g_error("Shmem transport NYI");
}

static void
gmf_pipe_setup_custom_server(GMFPipe *apipe)
{
  g_error("Custom transport NYI");
}

static void
gmf_pipe_setup_server(GMFPipe *apipe)
{
  impl_POA_GMF_Pipe *servant;

  servant = PIPE_DATA(apipe);

  if(servant->acceptable_transports & GMF_Transport_UNIX_SOCKETS)
    gmf_pipe_setup_usock_server(apipe);

  if(servant->acceptable_transports & GMF_Transport_TCPIP)
    gmf_pipe_setup_tcpip_server(apipe);

  if(servant->acceptable_transports & GMF_Transport_SHARED_MEM)
    gmf_pipe_setup_shmem_server(apipe);

  if(servant->acceptable_transports & GMF_Transport_CUSTOM) 
    gmf_pipe_setup_custom_server(apipe);

  /* CORBA requires no extra work */
}

static CORBA_char *
gmf_pipe_get_address(GMFPipe *apipe, GMF_Transport_Type ttype)
{
  CORBA_char *retval = NULL;
  char *ctmp;

  switch(ttype) {
  case GMF_Transport_UNIX_SOCKETS:
    retval = CORBA_string_dup(PIPE_DATA(apipe)->u.usock_addr.sun_path);
    break;
  case GMF_Transport_TCPIP:
    ctmp = inet_ntoa(PIPE_DATA(apipe)->u.tcpsock_addr.sin_addr);
    retval = CORBA_string_alloc(strlen(ctmp)
				+ 1 /* ':' */ +  5 /* port */ + 1 /* '\0' */);
    sprintf(retval, "%s:%d", ctmp,
	    ntohs(PIPE_DATA(apipe)->u.tcpsock_addr.sin_port));
    break;
  default:
    g_error("gmf_pipe_get_address(): This should not happen.");
    break;
  }

  return retval;
}

void
gmf_pipe_send_sample(GMFPipe *apipe, GMF_Sample *sample)
{
  impl_POA_GMF_Pipe *servant;

  g_return_if_fail(apipe);

  servant = (impl_POA_GMF_Pipe *)apipe->servant;

  if(!servant->attr_pConnectionInfo)
    return;

  switch(servant->attr_pConnectionInfo->connectionInfo._d) {
  case GMF_Transport_SHARED_MEM:
    gmf_pipe_shmem_send_sample(apipe, sample);
    gmf_pipe_notify_send_sample(apipe, sample);
    break;

  case GMF_Transport_UNIX_SOCKETS:
  case GMF_Transport_TCPIP:
    gmf_pipe_fd_send_sample(apipe, sample);
    break;

  case GMF_Transport_CORBA:
    gmf_pipe_corba_send_sample(apipe, sample);
    break;

  case GMF_Transport_CUSTOM:
    gmf_pipe_notify_send_sample(apipe, sample);
    break;
  }
}

static void
gmf_pipe_corba_send_sample(GMFPipe *apipe, GMF_Sample *sample)
{
  GMF_Pipe_HandleSample(PIPE_DATA(apipe)->attr_pConnectionInfo->connectedTo,
			sample, &modev);
}

static void
gmf_pipe_shmem_send_sample(GMFPipe *apipe, GMF_Sample *sample)
{
  g_error("SHMEM transport NYI");
}

static void
gmf_pipe_notify_send_sample(GMFPipe *apipe, GMF_Sample *sample)
{
  impl_POA_GMF_Pipe *servant;

  servant = (impl_POA_GMF_Pipe *)apipe->servant;

  GMF_Pipe_NotifySample(servant->attr_pConnectionInfo->connectedTo, &modev);
}

static void
gmf_pipe_fd_send_sample(GMFPipe *apipe, GMF_Sample *sample)
{
  impl_POA_GMF_Pipe *servant;

  servant = (impl_POA_GMF_Pipe *)apipe->servant;

  if(servant->attr_pConnectionInfo->connectionInfo._d == GMF_Transport_TCPIP)
    g_error("TCP/IP transport sucks at present (no packing/endianness considerations). Use CORBA transport instead.");

#if 0
  if(!servant->attr_pConnectionInfo->connectionMediaInfo.fixedSizeSamples)
    {
      CORBA_unsigned_long blen;

      blen = htonl(sample->sData._length);
      write(servant->fd, &blen, sizeof(blen));
    }
  else
#endif

  if(servant->attr_pConnectionInfo->connectionMediaInfo.fixedSizeSamples)
    {
      g_assert(sample->sData._length
	       == servant->attr_pConnectionInfo->connectionMediaInfo.sampleSize);
    }

  g_assert(sample->sData._length == sample->sInfo.actualDataLength);

  write(servant->fd, &sample->sInfo, sizeof(sample->sInfo));
  write(servant->fd, sample->sData._buffer, sample->sData._length);
}
