/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include "config.h"
#include <gmf.h>

typedef struct {
  char *name;
  GMF_Filter obj;
  GMF_Filter_Type ftype;
} FG_FilterInfo;

typedef struct {
  POA_GMF_FilterGraph servant;
  PortableServer_POA poa;

  /* notused */ /* GMF_TimeReference attr_syncSource; */

  GHashTable *filters_byname;
  GList *source_filters, *render_filters;

  gint end_of_stream_recv;

  GMFFilterGraph *gtkobj;
} impl_POA_GMF_FilterGraph;

#define FG_DATA(x) ((impl_POA_GMF_FilterGraph *)(GMF_FILTER_GRAPH(x)->servant))

static void impl_GNOME_Obj_ref(impl_POA_GMF_FilterGraph * servant,
			       CORBA_Environment * ev);
static void impl_GNOME_Obj_unref(impl_POA_GMF_FilterGraph * servant,
				 CORBA_Environment * ev);
static CORBA_Object impl_GNOME_Obj_query_interface(impl_POA_GMF_FilterGraph * servant,
						   CORBA_char * repo_id,
						   CORBA_Environment * ev);

static void impl_GMF_FilterGraph__destroy(impl_POA_GMF_FilterGraph * servant,
					  CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_SetDefaultSyncSource(impl_POA_GMF_FilterGraph * servant,
					  CORBA_Environment * ev);
static GMF_TimeReference
impl_GMF_FilterGraph__get_syncSource(impl_POA_GMF_FilterGraph * servant,
				     CORBA_Environment * ev);
static void
impl_GMF_FilterGraph__set_syncSource(impl_POA_GMF_FilterGraph * servant,
				     GMF_TimeReference value,
				     CORBA_Environment * ev);
static GMF_Filter
impl_GMF_FilterGraph_FindFilterByName(impl_POA_GMF_FilterGraph * servant,
				      CORBA_char * name,
				      CORBA_Environment * ev);
static GMF_Filter_RecordList *
impl_GMF_FilterGraph__get_graphFilters(impl_POA_GMF_FilterGraph * servant,
				       CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_BuildGraph(impl_POA_GMF_FilterGraph * servant,
				GMF_GraphPoint renderFrom,
				GMF_GraphPoint renderTo,
				CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_SaveGraph(impl_POA_GMF_FilterGraph * servant,
			       CORBA_char * outfile,
			       CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_LoadGraph(impl_POA_GMF_FilterGraph * servant,
			       CORBA_char * infile,
			       CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_AddFilter(impl_POA_GMF_FilterGraph * servant,
			       CORBA_char * name,
			       GMF_Filter filter,
			       CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_RemoveFilter(impl_POA_GMF_FilterGraph * servant,
				  CORBA_char * name,
				  CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_EndOfStream(impl_POA_GMF_FilterGraph * servant,
				 CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_Stop(impl_POA_GMF_FilterGraph * servant,
				 CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_Pause(impl_POA_GMF_FilterGraph * servant,
				 CORBA_Environment * ev);
static void
impl_GMF_FilterGraph_Run(impl_POA_GMF_FilterGraph * servant,
			 GMF_TimeVal *timeBase,
			 CORBA_Environment * ev);


static void gmf_filter_graph_class_init (GMFFilterGraphClass * klass);
static void gmf_filter_graph_init       (GMFFilterGraph * obj);
static void gmf_filter_graph_destroy    (GMFFilterGraph * obj);

enum {
  END_OF_STREAM,
  STOP,
  PAUSE,
  RUN,
  LAST_SIGNAL
};

static guint fg_signals[LAST_SIGNAL] = { 0 };
static CORBA_Environment modev;

static PortableServer_ServantBase__epv impl_GMF_FilterGraph_base_epv =
{
   NULL,			/* _private data */
   NULL,                	/* finalize routine...? */
   NULL,			/* default_POA routine */
};

static POA_GNOME_obj__epv gmf_impl_GNOME_Obj_epv =
{
  NULL, /* _private */
  (gpointer) & impl_GNOME_Obj_ref,
  (gpointer) & impl_GNOME_Obj_unref,
  (gpointer) & impl_GNOME_Obj_query_interface
};

static POA_GMF_FilterGraph__epv impl_GMF_FilterGraph_epv =
{
   NULL,			/* _private */
   (gpointer) & impl_GMF_FilterGraph_SetDefaultSyncSource,
   (gpointer) & impl_GMF_FilterGraph__get_syncSource,
   (gpointer) & impl_GMF_FilterGraph__set_syncSource,
   (gpointer) & impl_GMF_FilterGraph_FindFilterByName,
   (gpointer) & impl_GMF_FilterGraph__get_graphFilters,
   (gpointer) & impl_GMF_FilterGraph_BuildGraph,
   (gpointer) & impl_GMF_FilterGraph_SaveGraph,
   (gpointer) & impl_GMF_FilterGraph_LoadGraph,
   (gpointer) & impl_GMF_FilterGraph_AddFilter,
   (gpointer) & impl_GMF_FilterGraph_RemoveFilter,
   (gpointer) & impl_GMF_FilterGraph_EndOfStream,
   (gpointer) & impl_GMF_FilterGraph_Stop,
   (gpointer) & impl_GMF_FilterGraph_Pause,
   (gpointer) & impl_GMF_FilterGraph_Run
};

static POA_GMF_FilterGraph__vepv impl_GMF_FilterGraph_vepv =
{
   &impl_GMF_FilterGraph_base_epv,
   &gmf_impl_GNOME_Obj_epv,
   &impl_GMF_FilterGraph_epv
};

static void impl_GNOME_Obj_ref(impl_POA_GMF_FilterGraph * servant,
			       CORBA_Environment * ev)
{
  gtk_object_ref(GTK_OBJECT(servant->gtkobj));
  gtk_object_sink(GTK_OBJECT(servant->gtkobj));
}

static void impl_GNOME_Obj_unref(impl_POA_GMF_FilterGraph * servant,
				 CORBA_Environment * ev)
{
  gtk_object_unref(GTK_OBJECT(servant->gtkobj));
}

static CORBA_Object impl_GNOME_Obj_query_interface(impl_POA_GMF_FilterGraph * servant,
						   CORBA_char * repo_id,
						   CORBA_Environment * ev)
{
  return CORBA_OBJECT_NIL;
}

static GMF_FilterGraph 
impl_GMF_FilterGraph__create(PortableServer_POA poa,
			     impl_POA_GMF_FilterGraph **servant_ret,
			     CORBA_Environment * ev)
{
   GMF_FilterGraph retval;
   impl_POA_GMF_FilterGraph *newservant;
   PortableServer_ObjectId *objid;

   newservant = g_new0(impl_POA_GMF_FilterGraph, 1);
   if(servant_ret)
     *servant_ret = newservant;
   newservant->servant.vepv = &impl_GMF_FilterGraph_vepv;
   newservant->poa = poa;
   POA_GMF_FilterGraph__init((PortableServer_Servant) newservant, ev);

   objid = PortableServer_POA_activate_object(poa, newservant, ev);
   CORBA_free(objid);

   newservant->filters_byname = g_hash_table_new(g_str_hash,
						 g_str_equal);

   retval = PortableServer_POA_servant_to_reference(poa, newservant, ev);

   return retval;
}

/* You shouldn't call this routine directly without first deactivating the servant... */
static void
impl_GMF_FilterGraph__destroy(impl_POA_GMF_FilterGraph * servant, CORBA_Environment * ev)
{
  g_warning("Finish up on the filtergraph destruction");
  POA_GMF_FilterGraph__fini((PortableServer_Servant) servant, ev);
  g_free(servant);
}

static void
impl_GMF_FilterGraph_SetDefaultSyncSource(impl_POA_GMF_FilterGraph * servant,
					  CORBA_Environment * ev)
{
  g_warning("SetDefaultSyncSource NYI");
}

static GMF_TimeReference
impl_GMF_FilterGraph__get_syncSource(impl_POA_GMF_FilterGraph * servant,
				     CORBA_Environment * ev)
{
   GMF_TimeReference retval = CORBA_OBJECT_NIL;

   g_warning("get_syncSource NYI");

   return retval;
}

static void
impl_GMF_FilterGraph__set_syncSource(impl_POA_GMF_FilterGraph * servant,
				     GMF_TimeReference value,
				     CORBA_Environment * ev)
{
   g_warning("set_syncSource NYI");
}

static GMF_Filter
impl_GMF_FilterGraph_FindFilterByName(impl_POA_GMF_FilterGraph * servant,
				      CORBA_char * name,
				      CORBA_Environment * ev)
{
   GMF_Filter retval = CORBA_OBJECT_NIL;
   FG_FilterInfo *finfo;

   finfo = g_hash_table_lookup(servant->filters_byname, name);
   if(finfo)
     retval = CORBA_Object_duplicate(finfo->obj, ev);

   return retval;
}

static void
put_fginfo_on_list(char *name, FG_FilterInfo *fi,
		   GMF_Filter_RecordList *retval)
{
  int i;

  i = retval->_maximum;
  retval->_buffer[i].name = CORBA_string_dup(name);
  retval->_buffer[i].filterType = fi->ftype;
  retval->_buffer[i].obj = CORBA_Object_duplicate(fi->obj, &modev);
  retval->_maximum++;
}

static GMF_Filter_RecordList *
impl_GMF_FilterGraph__get_graphFilters(impl_POA_GMF_FilterGraph * servant,
				       CORBA_Environment * ev)
{
   GMF_Filter_RecordList *retval;

   retval = GMF_Filter_RecordList__alloc();
   retval->_length = g_hash_table_size(servant->filters_byname);
   retval->_maximum = 0;
   retval->_buffer = CORBA_sequence_GMF_Filter_Record_allocbuf(retval->_length);
   CORBA_sequence_set_release(retval, CORBA_TRUE);
   g_hash_table_foreach(servant->filters_byname,
			(GHFunc)put_fginfo_on_list, retval);

   return retval;
}

static void
impl_GMF_FilterGraph_BuildGraph(impl_POA_GMF_FilterGraph * servant,
				GMF_GraphPoint renderFrom,
				GMF_GraphPoint renderTo,
				CORBA_Environment * ev)
{
  g_warning("BuildGraph NYI");
}

static void
impl_GMF_FilterGraph_SaveGraph(impl_POA_GMF_FilterGraph * servant,
			       CORBA_char * outfile,
			       CORBA_Environment * ev)
{
  g_warning("SaveGraph NYI");
}

static void
impl_GMF_FilterGraph_LoadGraph(impl_POA_GMF_FilterGraph * servant,
			       CORBA_char * infile,
			       CORBA_Environment * ev)
{
  g_warning("LoadGraph NYI");
}

static void
impl_GMF_FilterGraph_AddFilter(impl_POA_GMF_FilterGraph * servant,
			       CORBA_char * name,
			       GMF_Filter filter,
			       CORBA_Environment * ev)
{
  FG_FilterInfo *fi;

  fi = g_hash_table_lookup(servant->filters_byname, name);

  if(fi)
    goto errout;

  GMF_Filter_JoinFilterGraph(filter, name, servant->gtkobj->corba_object, ev);
  if(ev->_major != CORBA_NO_EXCEPTION) goto errout;

  fi = g_new0(FG_FilterInfo, 1);
  fi->ftype = GMF_Filter__get_filterType(filter, ev);

  fi->name = g_strdup(name);

  fi->obj = CORBA_Object_duplicate(filter, ev);

  if(fi->ftype == GMF_Filter_RENDERER)
    servant->render_filters = g_list_prepend(servant->render_filters, fi);
  else if(fi->ftype == GMF_Filter_SOURCE)
    servant->source_filters = g_list_prepend(servant->source_filters, fi);

  g_hash_table_insert(servant->filters_byname, fi->name, fi);
  
  return;

 errout:
  CORBA_exception_set(ev, CORBA_USER_EXCEPTION, ex_GMF_Filter_JoinRefused, NULL);
  return;
}

static void
impl_GMF_FilterGraph_RemoveFilter(impl_POA_GMF_FilterGraph * servant,
				  CORBA_char * name,
				  CORBA_Environment * ev)
{
  FG_FilterInfo *fi;

  fi = g_hash_table_lookup(servant->filters_byname, name);

  if(!fi)
    return;

  GMF_Filter_JoinFilterGraph(fi->obj, "", CORBA_OBJECT_NIL, ev);
  g_hash_table_remove(servant->filters_byname, name);

  CORBA_Object_release(fi->obj, ev);
  g_free(fi->name);
  g_free(fi);
}

static void
impl_GMF_FilterGraph_EndOfStream(impl_POA_GMF_FilterGraph * servant,
				 CORBA_Environment * ev)
{
  if(servant->end_of_stream_recv < g_list_length(servant->render_filters))
    servant->end_of_stream_recv++;
  else
    gtk_signal_emit(GTK_OBJECT(servant->gtkobj), fg_signals[END_OF_STREAM]);
}

static void
fg_stop_to_all(guchar *key, FG_FilterInfo *fi,
	       CORBA_Environment *ev)
{
  GMF_Filter_Stop(fi->obj, ev);
}

static void
impl_GMF_FilterGraph_Stop(impl_POA_GMF_FilterGraph * servant,
			  CORBA_Environment * ev)
{
  g_hash_table_foreach(servant->filters_byname, (GHFunc)fg_stop_to_all, ev);
  gtk_signal_emit(GTK_OBJECT(servant->gtkobj), fg_signals[STOP]);
}

static void
fg_pause_to_all(guchar *key, FG_FilterInfo *fi,
		CORBA_Environment *ev)
{
  GMF_Filter_Pause(fi->obj, ev);
}

static void
impl_GMF_FilterGraph_Pause(impl_POA_GMF_FilterGraph * servant,
			   CORBA_Environment * ev)
{
  g_hash_table_foreach(servant->filters_byname, (GHFunc)fg_pause_to_all, ev);
  gtk_signal_emit(GTK_OBJECT(servant->gtkobj), fg_signals[PAUSE]);
}

static void
fg_run_to_all(guchar *key, FG_FilterInfo *fi,
	      GMF_TimeVal *timeBase)
{
  GMF_Filter_Run(fi->obj, timeBase, &modev);
}

static void
impl_GMF_FilterGraph_Run(impl_POA_GMF_FilterGraph * servant,
			 GMF_TimeVal *timeBase,
			 CORBA_Environment * ev)
{
  g_hash_table_foreach(servant->filters_byname, (GHFunc)fg_run_to_all, timeBase);
  gtk_signal_emit(GTK_OBJECT(servant->gtkobj), fg_signals[RUN], timeBase);
}

GtkType
gmf_filter_graph_get_type (void)
{
  static GtkType gmf_filter_graph_type = 0;
  if (!gmf_filter_graph_type)
    {
      static const GtkTypeInfo gmf_filter_graph_info =
      {
	"GMFFilterGraph",
	sizeof (GMFFilterGraph),
	sizeof (GMFFilterGraphClass),
	(GtkClassInitFunc) gmf_filter_graph_class_init,
	(GtkObjectInitFunc) gmf_filter_graph_init,
	NULL, NULL,		/* reserved 1 & 2 */
	NULL
      };

      gmf_filter_graph_type = gtk_type_unique (gtk_object_get_type(), &gmf_filter_graph_info);
    }

  return gmf_filter_graph_type;
}

static void 
gmf_filter_graph_class_init (GMFFilterGraphClass * klass)
{
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (klass);

  CORBA_exception_init(&modev);

  object_class->destroy = (void (*)(GtkObject *))gmf_filter_graph_destroy;
  fg_signals[END_OF_STREAM] =
    gtk_signal_new("end_of_stream", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFFilterGraphClass, end_of_stream),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);
  fg_signals[STOP] =
    gtk_signal_new("stop", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFFilterGraphClass, stop),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);
  fg_signals[PAUSE] =
    gtk_signal_new("pause", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFFilterGraphClass, pause),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);
  fg_signals[RUN] =
    gtk_signal_new("run", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GMFFilterGraphClass, run),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
}

static void 
gmf_filter_graph_init (GMFFilterGraph * obj)
{
  PortableServer_POA poa;

  poa = (PortableServer_POA)
    CORBA_ORB_resolve_initial_references((CORBA_ORB)gnome_CORBA_ORB(), "RootPOA", &modev);
  obj->corba_object = impl_GMF_FilterGraph__create(poa,
						   (impl_POA_GMF_FilterGraph **)
						   &obj->servant, &modev);
  CORBA_Object_release((CORBA_Object)poa, &modev);
}

static void
gmf_filter_graph_destroy (GMFFilterGraph * obj)
{
  impl_GMF_FilterGraph__destroy(obj->servant, &modev);
  CORBA_Object_release(obj->corba_object, &modev);
}

GtkObject *
gmf_filter_graph_new (void)
{
  GtkObject *retval;
  GMFFilterGraph *fg;
  impl_POA_GMF_FilterGraph *servant;

  retval = gtk_type_new(gmf_filter_graph_get_type());
  fg = GMF_FILTER_GRAPH(retval);

  servant = FG_DATA(fg);
  servant->gtkobj = fg;

  return retval;
}

void
gmf_filter_graph_render_file (GMFFilterGraph *filter_graph,
			      const char *filename)
{
  GMF_GraphPoint p1;
  GMF_GraphPoint p2;
  char *curdir = NULL;

  p1._d = GMF_POINT_URL;

  if(*filename != '/') {
    curdir = g_get_current_dir();
    p1._u.url = alloca(strlen(filename) + sizeof("file:///")
		       + strlen(curdir) + 1);
    sprintf(p1._u.url, "file:///%s/%s", curdir, filename);
  } else {
    p1._u.url = alloca(strlen(filename) + sizeof("file://"));
    sprintf(p1._u.url, "file:///%s", filename);
  }

  p2._d = GMF_POINT_AUTO;

  GMF_FilterGraph_BuildGraph(filter_graph->corba_object, &p1, &p2, &modev);

  g_free(curdir);
}
